//
// MODULE NAME:  NAME.C.
//
// FUNCTIONAL DESCRIPTION.
//	This module manages the DLC name database for the analyzers.
//
// MODIFICATION HISTORY.
//	S. E. Jones	92/12/29.	2.0, original.
//
// NOTICE:  Copyright (C) 1992-1993 General Software, Inc.  All rights reserved.
//

#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <dos.h>
#include "..\inc\system.h"		// DOS operating system defns.
#include "..\cow\cow.h"                 // character-oriented windows.
#include "..\inc\ktypes.h"		// commonly-used types.
#include "analyzer.h"			// common stuff for all modules.

extern VOID MakeHexAddress (UCHAR *Address, UCHAR *HexString); // in ANALYZER.C.
extern UCHAR *MakeHexString (UCHAR *Address); // in ANALYZER.C.

static UCHAR far *NameMenuHelpText [] = {
    "                      Help for Names Menu",
    " ",
    "The Names menu allows you to assign human-readable names to known",
    "6-octet Ethernet addresses.  Any number of names may be defined in",
    "the name table.  These names are shown as appropriate on real-time",
    "capture screens and in the traffic display mode.",
    " ",
    "----- Viewing and Editing the Name Table -----",
    " ",
    "To view the names in the table, select Edit Names option.  This",
    "causes the entire list of known names to be displayed in a list",
    "box.  Any of these names may be deleted by positioning the scroll",
    "bar over the name to be deleted, and pressing the DEL key.  A new",
    "name may be inserted into the list by pressing the INS key.",
    " ",
    "----- Saving the Name Table to a Disk File -----",
    " ",
    "You can save the name table to disk by selecting the Save Names",
    "option.  The filename you specify will be relative to the names",
    "directory as defined in the main menu's 'Options and Setup' menu.",
    " ",
    "----- Loading a Name Table from a Disk File -----",
    " ",
    "You can load a name table previously saved to disk by selecting",
    "the Load Names option.  The filename you specify will be relative",
    "to the names directory as with the Save Names option.  All names",
    "in the file overwrite any existing names in the active name table.",
    NULL				// end of help text symbol.
}; // NameMenuHelpText

typedef struct _NAME {
    UCHAR AsciizName [14];		// 12-character (plus 0-byte) name.
    UCHAR Address [6];			// 6-octet binary address.
    struct _NAME *Fwdlink;		// next name in database.
} NAME, *PNAME;

static PNAME NameTable=NULL;		// ptr to list of defined names.

static VOID NameMenuHelpRtn (List)
    PLIST List;
{
    PopupHelp (NameMenuHelpText);
} // NameMenuHelpRtn

UCHAR *TranslateName (Buffer)
    UCHAR *Buffer;
{
    PNAME p;
    USHORT i;

    for (p=NameTable; p != NULL; p=p->Fwdlink) {
	for (i=0; i<6; i++) {
	    if (p->Address [i] != Buffer [i]) {
		break;			// doesn't match.
	    }
	}
	if (i == 6) {			// there was a match!
	    return p->AsciizName;	// return pointer to the name.
	}
    }
    return NULL;			// we didn't find anything.
} // TranslateName

static BOOLEAN EditName (Name)
    PNAME Name;
{
    ACTION a;
    PSCREEN Screen;
    PFIELD f_asc, f_hex;

    Screen = ScreenCreate ("Edit Name", 40, 12, 50, 9);
    if (Screen == NULL) {
	return FALSE;
    }

    f_asc = FieldCreate ("Name: ", FIELD_TYPE_STRING,
			 1, 1, 6+12+1, FIELD_FLAGS_EDITABLE);
    strcpy (f_asc->Data.String, Name->AsciizName);
    ScreenAddField (Screen, f_asc);

    f_hex = FieldCreate ("Ethernet Address: ", FIELD_TYPE_HEXSTRING,
			 1, 3, 18+12+1, FIELD_FLAGS_EDITABLE);
    strcpy (f_hex->Data.String, MakeHexString (Name->Address));
    ScreenAddField (Screen, f_hex);

    a = ScreenEdit (Screen);
    ScreenDestroy (Screen);

    if (a == ACTION_SAVE) {
	if (strlen (f_hex->Data.String) != 12) {
	    PopupWarning ("Address field must contain 12 hex digits.");
	    return FALSE;
	}

	MakeHexAddress (Name->Address, f_hex->Data.String);
	strcpy (Name->AsciizName, f_asc->Data.String);
	return TRUE;
    }
    return FALSE;
} // EditName

static VOID NameSelectRoutine (ListElement)
    PLISTE ListElement;
{
    PNAME n;

    n = (PNAME)ListElement->Value;
    EditName (n);
    strcpy (ListElement->Title, n->AsciizName);
} // NameSelectRoutine

static PLISTE NameInsertRoutine ()
{
    PLISTE p;
    PNAME n;
    USHORT i;

    n = malloc (sizeof (NAME));
    if (n == NULL) {
	return NULL;
    }

    for (i=0; i<6; i++) {
	n->Address [i] = 0;
    }
    n->AsciizName [0] = 0;

    if (!EditName (n)) {
	free (n);
	return NULL;
    }

    p = ListElementCreate (n->AsciizName, (ULONG)n);
    if (p != NULL) {
	n->Fwdlink = NameTable;
	NameTable = n;
    } else {
	free (n);
    }
    return p;
} // NameInsertRoutine

static BOOLEAN NameDeleteRoutine (ListElement)
    PLISTE ListElement;
{
    PNAME p, q, n;

    n = (PNAME)(ListElement->Value);

    //
    // If it's the first one on the list, pop off the front of the list.
    //

    if (n == NameTable) {
	NameTable = n->Fwdlink;
	free (n);
	return TRUE;
    }

    //
    // Splice it out of the list.
    //

    for (p=NameTable; p!=NULL; p=p->Fwdlink) {
	if (p->Fwdlink == n) {
	    p->Fwdlink = n->Fwdlink;	// splice it out of the list.
	    free (n);
	    return TRUE;
	}
    }
    return FALSE;			// it wasn't on the list.
} // NameDeleteRoutine

static VOID EditNameTable ()
{
    PNAME n;
    PLIST p;

    p = ListCreate ("Edit Name Table  [Help=F2]", 40, 12, 50, 8,
		    LIST_FLAGS_INSERT | LIST_FLAGS_DELETE | LIST_FLAGS_SELECT);
    p->DeleteRtn = NameDeleteRoutine;
    p->InsertRtn = NameInsertRoutine;
    p->SelectRtn = NameSelectRoutine;

    for (n=NameTable; n!=NULL; n=n->Fwdlink) {
	sprintf (ScratchBuf, "%-13s", n->AsciizName);
	ListInsert (p, ScratchBuf, (ULONG)n);
    }

    ListEdit (p);			// updates names w/no cancellation.
    ListDestroy (p);
} // EditNameTable

static VOID SaveNameTable ()
{
    PSCREEN Screen;
    PFIELD f_fn;
    PWINDOW w;
    ACTION a;
    FILE *DosFile;
    PNAME p;

    Screen = ScreenCreate ("Save Name Table to File", 40, 12, 40, 5);
    if (Screen == NULL) {
	return;
    }

    f_fn = FieldCreate ("Save Filename: ", FIELD_TYPE_STRING,
			 1, 0, 15+9, FIELD_FLAGS_EDITABLE);
    strcpy (f_fn->Data.String, "");
    ScreenAddField (Screen, f_fn);

    a = ScreenEdit (Screen);
    if (a != ACTION_SAVE) {
	ScreenDestroy (Screen);
	return;
    }
    strcpy (TmpStr, NameDirectory);
    if (strlen (TmpStr) > 0) {
	if (TmpStr [strlen (TmpStr)-1] != '\\') {
	    strcat (TmpStr, "\\");
	}
    }
    strcat (TmpStr, f_fn->Data.String);
    if (strchr (f_fn->Data.String, '.') == NULL) {
	strcat (TmpStr, ".NAM");
    }

    if ((DosFile=fopen (TmpStr, "wb")) == NULL) {
	PopupWarning ("Unable to create name file.");
	ScreenDestroy (Screen);
	return;
    }

    w = WinCreate ("Saving Name Table", 40, 12, 50, 5,
		   WINDOW_BORDER_SINGLE, NORMAL_PALETTE);
    if (w == NULL) {
	fclose (DosFile);
	ScreenDestroy (Screen);
	return;
    }

    //
    // Write out the name records.
    //

    for (p=NameTable; p!=NULL; p=p->Fwdlink) {

	//
	// Keep user informed about our status.
	//

	sprintf (TmpStr, "Saving Name %s", p->AsciizName);
	WinWrite (w, TmpStr, 15, 0, 999); // update window.

	//
	// Write out the name record in full.  It includes a
	// pointer, but we won't care about that because on
	// loading, we will ignore the pointer.
	//

	if (fwrite (p, sizeof (NAME), 1, DosFile) != 1) {
	    PopupWarning ("Unable to write to names file.");
	    break;
	}
    }

    fclose (DosFile);
    WinDestroy (w);			// we're done with the window.
    ScreenDestroy (Screen);		// we're done with the screen.
} // SaveNameTable

static VOID LoadNameTable ()
{
    struct find_t DirEntry;
    UCHAR FileName [20];
    FILE *DosFile;
    PWINDOW w;
    PLISTE le;
    PMENU m;
    USHORT i;
    UCHAR ch;
    PNAME n;
    NAME Name;

    strcpy (TmpStr, NameDirectory);
    if (strlen (TmpStr) > 0) {
	if (TmpStr [strlen (TmpStr)-1] != '\\') {
	    strcat (TmpStr, "\\");
	}
    }
    strcat (TmpStr, "*.NAM");

    if (_dos_findfirst (TmpStr, _A_NORMAL, &DirEntry) != 0) {
	PopupWarning ("No name table files present.");
	return;
    }
    m = MenuCreate ("Select Name Table", 40, 12, 30, 10);
    if (m == NULL) {
	return;
    }

    while (TRUE) {
	MenuItem (m, DirEntry.name, 0L);
	if (_dos_findnext (&DirEntry) != 0) {
	    break;
	}
    }
    MenuSort (m);			// sort the menu alphabetically.
    le = MenuSelect (m);		// select from the menu.
    if (le == NULL) {
	MenuDestroy (m);
	return;
    }

    strcpy (TmpStr, NameDirectory);
    if (strlen (TmpStr) > 0) {
	if (TmpStr [strlen (TmpStr)-1] != '\\') {
	    strcat (TmpStr, "\\");
	}
    }
    strcat (TmpStr, le->Title);
    if ((DosFile=fopen (TmpStr, "rb")) == NULL) {
	PopupWarning ("Unable to read name table file.");
	MenuDestroy (m);
	return;
    }

    //
    // Delete our old name table.
    //

    while (NameTable != NULL) {
	n = NameTable;
	NameTable = n->Fwdlink;
	free (n);
    }
    NameTable = NULL;

    //
    // Now read-in our new name table.
    //

    w = WinCreate ("Loading Name Table", 40, 12, 50, 5,
		   WINDOW_BORDER_SINGLE, NORMAL_PALETTE);
    if (w == NULL) {
	fclose (DosFile);
	MenuDestroy (m);
	return;
    }

    while (TRUE) {

	//
	// Read a name from the name table file.
	//

	if (fread (&Name, sizeof (NAME), 1, DosFile) != 1) {
	    break;			// there are no more.
	}

	sprintf (TmpStr, "Reading Name %s", Name.AsciizName);
	WinWrite (w, TmpStr, 15, 0, 999); // update window.

	//
	// We got a record, so allocate a name structure.
	//

	n = (PNAME)malloc (sizeof (NAME));
	if (n == NULL) {
	    PopupWarning ("Name table size exceeds available memory.");
	    break;
	}

	//
	// Copy the name into the allocated structure.
	//

	*n = Name;			// structure copy.

	//
	// Link-in this record with our name database.
	//

	n->Fwdlink = NameTable;
	NameTable = n;
    }

    fclose (DosFile);
    WinDestroy (w);			// we're done with the window.
    MenuDestroy (m);			// we're done with the menu.
} // LoadNameTable

VOID NameMenuSelectRoutine (Selection)
    PLISTE Selection;
{
    switch ((USHORT)Selection->Value) {
	case 1: EditNameTable (); break;
	case 2: SaveNameTable (); break;
	case 3: LoadNameTable (); break;
    }
} // NameMenuSelectRoutine

VOID NameMenu ()
{
    PLIST p;
    ACTION a;

    p = ListCreate ("Name Management", 40, 12, 30, 7, LIST_FLAGS_SELECT);
    p->SelectRtn = NameMenuSelectRoutine;
    p->HelpRtn = NameMenuHelpRtn;

    ListInsert (p, "Edit Name Table", 1L);
    ListInsert (p, "Save Names to File", 2L);
    ListInsert (p, "Load Names from File", 3L);
    while (TRUE) {
	a = ListEdit (p);
	if (a == ACTION_ABORT) {
	    break;
	}
    }
    ListDestroy (p);
} // NameMenu
