//
// MODULE NAME:  LLC.C.
//
// FUNCTIONAL DESCRIPTION.
//	This module handles packet summary and protocol decoding for LLC
//	(Logical Link Control) Class 1 (connectionless) and Class 2 (connection
//	oriented).
//
//	There are two main entrypoints to this module.	PacketSummaryLlc
//	returns a pointer to an ASCIIZ string allocated with malloc that
//	best describes the packet. PacketDetailLlc returns a pointer to
//	a list of detail records (PDETREC) that describes the frame in detail.
//	If either function returns NULL, then no decoding is possible.
//
// MODIFICATION HISTORY.
//	S. E. Jones	92/03/19.	Original.
//	S. E. Jones	92/11/24.	#1.9, added LlcTdi at SAP 55h.
//	S. E. Jones	92/11/24.	#2.5, moved LlcTdi to SAP 54h.
//
// NOTICE:  Copyright (C) 1992-1993 General Software, Inc.  All rights reserved.
//

#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <dos.h>
#include "..\inc\system.h"		// DOS operating system defns.
#include "..\cow\cow.h"                 // character-oriented windows.
#include "..\inc\ktypes.h"		// commonly-used types.
#include "analyzer.h"			// common stuff for all modules.

//
// Service Access Points for use in LLC C1 and C2.
//

#define SAP_NULL		0x00	// the NULL sap.
#define SAP_SNA_1		0x04	// SNA sap.
#define SAP_SNA_2		0x05	// another SNA sap.
#define SAP_SNA_3		0x08	// another SNA sap.
#define SAP_SNA_4		0x0c	// another SNA sap.
#define SAP_LLCTDI		0x54	// General Software LLC TDI protocol.
#define SAP_XNS                 0x80	// XNS protocol.
#define SAP_SNAP		0xaa	// Bluebook (ethertype) inside LLC.
#define SAP_NOVELL		0xe0	// Novell.
#define SAP_NETBIOS		0xf0	// the NETBEUI sap.
#define SAP_UNGERMANN_BASS	0xfa	// U-B sap.
#define SAP_ISO                 0xfe	// ISO OSI protocols.
#define SAP_GLOBAL		0xff	// the global SAP.
#define SAP_TR_ROUTING_INFO	0xc2	// NOT A SAP AT ALL; TOKEN RING ROUTING INFO.

//
// LLC packet structures.
//

typedef struct _PDU {
    UCHAR Dsap;                 // destination sap.
    UCHAR Ssap;                 // source sap.

#define SSAP_RESPONSE	0x01	// set if response, else command.

    UCHAR Command;		// command byte.

#define DLCS_RR         0x01	// receive-ready.
#define DLCS_RNR	0x05	// receive-not-ready.
#define DLCS_REJ	0x09	// reject.

#define DLCU_SABME	0x6f	// set asynchronous balanced mode extended.
#define DLCU_DISC	0x43	// disconnect.
#define DLCU_UA         0x63	// unnumbered acknowlegement.
#define DLCU_DM         0x0f	// disconnected mode.
#define DLCU_FRMR	0x87	// frame reject.
#define DLCU_UI         0x03	// unnumbered information.
#define DLCU_XID	0xaf	// exchange ID.
#define DLCU_TEST	0xe3	// test frame.
#define DLCU_PF         0x10	// poll/final bit for DLCU frames.

    UCHAR Data1;		// data byte.
#define DLCS_PF         0x01	// poll/final bit for DLCS frames.

    UCHAR Misc [5];		// other bytes, as necessary.
} PDU, *PPDU;

//
// Routines in other modules.
//

extern BOOLEAN PacketReqTypeLlcTdi (PVOID Buffer, USHORT BufferLength);

//
// Routines in this module.
//

BOOLEAN PacketReqTypeAboveLlc (UCHAR Sap, PVOID Buffer, USHORT BufferLength)
{
    switch (Sap) {
	case SAP_NETBIOS:
	    return PacketReqTypeNetbeui (Buffer, BufferLength);

	case SAP_LLCTDI:
	    return PacketReqTypeLlcTdi (Buffer, BufferLength);

	case SAP_SNA_1:
	case SAP_SNA_2:
	case SAP_SNA_3:
	case SAP_SNA_4:
	case SAP_ISO:
	    return FALSE;
    }
} // PacketReqTypeAboveLlc

BOOLEAN PacketReqTypeLlc (PPDU Buffer, USHORT BufferLength)
{
    UCHAR *t, dsap, cmd;

    if (BufferLength < 3) {
	return FALSE;			// this is a fragment.
    }

    //
    // Handle Madge Networks' bogus routing information over Ethernet.
    // This information is present in Token ring but definitely not
    // allowed on Ethernet.  Unfortunately, these guys didn't care and
    // dumped the routing information on Ethernet anyway, clobbering
    // the 0xC2 SAP.  Fortunately, nobody uses it anyway.
    //

    if (Buffer->Dsap == SAP_TR_ROUTING_INFO) {
	t = (UCHAR *)Buffer;
	t += 2;                         // skip over this junk.
	Buffer = (PPDU)t;
    }

    dsap = Buffer->Dsap;
    cmd = Buffer->Command;

    //
    // We're not selecting raw LLC packets, but if it carries something
    // we are selecting, then we should still accept this packet.
    //

    if ((cmd & 0x01) == 0) {		// we have an I-frame.
	return PacketReqTypeAboveLlc (dsap, &(Buffer->Data1)+1, BufferLength-4);
    }
    if (cmd == DLCU_UI) {		// if this is a UI-frame.
	return PacketReqTypeAboveLlc (dsap, &Buffer->Data1, BufferLength-3);
    }
    return FALSE;			// we aren't accepting random LLC.
} // PacketReqTypeLlc

BOOLEAN PacketCountLlc (PPDU Buffer, USHORT BufferLength)
{
    UCHAR *t, dsap, cmd;

    if (BufferLength < 3) {
	return FALSE;			// this is a fragment.
    }

    CountProtocolType (INDEX_LLC);

    //
    // Handle Madge Networks' bogus routing information over Ethernet.
    // This information is present in Token ring but definitely not
    // allowed on Ethernet.  Unfortunately, these guys didn't care and
    // dumped the routing information on Ethernet anyway, clobbering
    // the 0xC2 SAP.  Fortunately, nobody uses it anyway.
    //

    if (Buffer->Dsap == SAP_TR_ROUTING_INFO) {
	t = (UCHAR *)Buffer;
	t += 2;                         // skip over this junk.
	Buffer = (PPDU)t;
    }

    dsap = Buffer->Dsap;
    cmd = Buffer->Command;

    switch (dsap) {
	case SAP_SNA_1:
	case SAP_SNA_2:
	case SAP_SNA_3:
	case SAP_SNA_4:
	    CountProtocolType (INDEX_SNA);
	    return TRUE;
	case SAP_NETBIOS:
	    CountProtocolType (INDEX_NETBEUI);
	    CountProtocolType (INDEX_SMB);
	    return TRUE;
	case SAP_ISO:
	    CountProtocolType (INDEX_ISO);
	    return TRUE;

	case SAP_LLCTDI:
	case SAP_UNGERMANN_BASS:
	case SAP_NOVELL:
	case SAP_SNAP:
	case SAP_XNS:
	case SAP_NULL:
	case SAP_GLOBAL:
	default:
	    return FALSE;
    }
} // PacketCountLlc

BOOLEAN PacketFilterAboveLlc (UCHAR Sap, PVOID Buffer, USHORT BufferLength)
{
    switch (Sap) {
	case SAP_NETBIOS:
	    return PacketFilterNetbeui (Buffer, BufferLength);
	case SAP_LLCTDI:
	    return PacketFilterLlcTdi (Buffer, BufferLength);
	case SAP_SNA_1:
	case SAP_SNA_2:
	case SAP_SNA_3:
	case SAP_SNA_4:
	    return (CaptureFilter & FILTER_SNA) == FILTER_SNA;
	case SAP_ISO:
	    return (CaptureFilter & FILTER_ISO) == FILTER_ISO;
    }
    return FALSE;
} // PacketFilterAboveLlc

BOOLEAN PacketFilterLlc (PPDU Buffer, USHORT BufferLength)
{
    UCHAR *t, dsap, cmd;

    if (BufferLength < 3) {
	return FALSE;			// this is a fragment.
    }

    //
    // Handle Madge Networks' bogus routing information over Ethernet.
    // This information is present in Token ring but definitely not
    // allowed on Ethernet.  Unfortunately, these guys didn't care and
    // dumped the routing information on Ethernet anyway, clobbering
    // the 0xC2 SAP.  Fortunately, nobody uses it anyway.
    //

    if (Buffer->Dsap == SAP_TR_ROUTING_INFO) {
	t = (UCHAR *)Buffer;
	t += 2;                         // skip over this junk.
	Buffer = (PPDU)t;
    }

    dsap = Buffer->Dsap;
    cmd = Buffer->Command;

    //
    // If LLC is selected, then only capture if this is a known LLC SAP.
    // We can't allow the GLOBAL SAP because NetWare dumps 0xffff into
    // both SAPs.
    //

    if (CaptureFilter & FILTER_LLC) {
	switch (dsap) {
	    case SAP_SNA_1:
	    case SAP_SNA_2:
	    case SAP_SNA_3:
	    case SAP_SNA_4:
	    case SAP_NULL:
	    case SAP_XNS:
	    case SAP_SNAP:
	    case SAP_NOVELL:
	    case SAP_NETBIOS:
	    case SAP_LLCTDI:
	    case SAP_UNGERMANN_BASS:
	    case SAP_ISO:
		return TRUE;

	    case SAP_GLOBAL:
	    default:
		return FALSE;
	}
    }

    //
    // We're not selecting raw LLC packets, but if it carries something
    // we are selecting, then we should still accept this packet.
    //

    if ((cmd & 0x01) == 0) {		// we have an I-frame.
	return PacketFilterAboveLlc (dsap, &(Buffer->Data1)+1, BufferLength-4);
    }
    if (cmd == DLCU_UI) {		// if this is a UI-frame.
	return PacketFilterAboveLlc (dsap, &Buffer->Data1, BufferLength-3);
    }
    return FALSE;			// we aren't accepting random LLC.
} // PacketFilterLlc

PUCHAR LlcSapName (UCHAR Sap)
{
    UCHAR *t;
    switch (Sap) {
	case SAP_NULL:		t = "Null SAP"; break;
	case SAP_SNA_1:
	case SAP_SNA_2:
	case SAP_SNA_3:
	case SAP_SNA_4:         t = "IBM SNA"; break;
	case SAP_XNS:		t = "XNS over LLC"; break;
	case SAP_SNAP:		t = "SNAP"; break;
	case SAP_NOVELL:	t = "Novell"; break;
	case SAP_NETBIOS:	t = "NETBEUI"; break;
	case SAP_UNGERMANN_BASS:t = "U-B"; break;
	case SAP_ISO:		t = "ISO"; break;
	case SAP_GLOBAL:	t = "GLOBAL"; break;
	case SAP_LLCTDI:	t = "LLC TDI"; break;
	default:		t = "Undecoded";
    }
    return t;
} // LlcSapName

PUCHAR PacketSummaryAboveLlc (UCHAR Sap, PVOID Buffer, USHORT BufferLength)
{
    switch (Sap) {
	case SAP_NETBIOS:		// IBM NETBEUI/SJ JETBEUI.
	    return PacketSummaryNetbeui (Buffer, BufferLength);
	    break;
	case SAP_LLCTDI:		// General Software LLC TDI transport.
	    return PacketSummaryLlcTdi (Buffer, BufferLength);
	    break;
    }
    return NULL;
} // PacketSummaryAboveLlc

PDETREC PacketDetailAboveLlc (UCHAR Sap, PVOID Buffer, USHORT BufferLength)
{
    switch (Sap) {
	case SAP_NETBIOS:		// IBM NETBEUI/SJ JETBEUI.
	    return PacketDetailNetbeui (Buffer, BufferLength);
	    break;
	case SAP_LLCTDI:	       // General Software LLC TDI transport.
	    return PacketDetailLlcTdi (Buffer, BufferLength);
	    break;
    }
    return NULL;
} // PacketDetailAboveLlc

PUCHAR PacketSummaryLlc (PPDU Buffer, USHORT BufferLength)
{
    UCHAR *p, cmd, *t;
    UCHAR ssap, dsap;
    BOOLEAN CmdFlag, PfFlag;

    if (!(DisplayFilter & FILTER_LLC)) {
	return NULL;			// we aren't decoding LLC.
    }
    if (BufferLength < 3) {
	return NULL;			// this is a fragment.
    }

    p = malloc (80);
    if (p == NULL) {
	return NULL;
    }

    //
    // Handle Madge Networks' bogus routing information over Ethernet.
    // This information is present in Token ring but definitely not
    // allowed on Ethernet.  Unfortunately, these guys didn't care and
    // dumped the routing information on Ethernet anyway, clobbering
    // the 0xC2 SAP.  Fortunately, nobody uses it anyway.
    //

    if (Buffer->Dsap == SAP_TR_ROUTING_INFO) {
	t = (UCHAR *)Buffer;
	t += 2;                         // skip over this junk.
	Buffer = (PPDU)t;
    }

    ssap = Buffer->Ssap & ~(SSAP_RESPONSE);
    dsap = Buffer->Dsap;
    CmdFlag = (Buffer->Ssap & SSAP_RESPONSE) == 0;

    cmd = Buffer->Command;
    if ((cmd & 0xf3) == 0x01) {         // if this is an S-frame.
	switch (cmd & 0xf3) {
	    case DLCS_RR: t = "RR"; break;
	    case DLCS_RNR: t = "RNR"; break;
	    case DLCS_REJ: t = "REJ"; break;
	    default: t = "Invalid S-frame"; break;
	}
	PfFlag = (Buffer->Data1 & DLCS_PF) != 0;
	sprintf (p, "LLC %c S=%02x D=%02x %s%s NR=%u",
		 CmdFlag ? 'C' : 'R',
		 ssap, dsap,
		 t,
		 CmdFlag ? (PfFlag ? "-p" : "") : (PfFlag ? "-f" : ""),
		 Buffer->Data1 & 0xfe);
    } else if ((cmd & 0x03) == 0x03) {	// if this is a U-frame.
	if (cmd == DLCU_UI) {
	    if ((t=PacketSummaryAboveLlc (dsap, &Buffer->Data1, BufferLength-3)) != NULL) {
		strcpy (p, t);
		free (t);
		return p;
	    }
	}
	switch (cmd & ~DLCU_PF) {	// remove POLL/FINAL bit.
	    case DLCU_SABME: t = "SABME"; break;
	    case DLCU_DISC: t = "DISC"; break;
	    case DLCU_UA: t = "UA"; break;
	    case DLCU_DM: t = "DM"; break;
	    case DLCU_FRMR: t = "FRMR"; break;
	    case DLCU_UI: t = "UI"; break;
	    case DLCU_XID: t = "XID"; break;
	    case DLCU_TEST: t = "TEST"; break;
	    default: t = "Invalid U-frame";
	}
	PfFlag = (cmd & DLCU_PF) != 0;
	sprintf (p, "LLC %c S=%02x D=%02x %s%s",
		 CmdFlag ? 'C' : 'R',
		 ssap, dsap,
		 t,
		 CmdFlag ? (PfFlag ? "-p" : "") : (PfFlag ? "-f" : ""));
    } else if ((cmd & 0x01) == 0) {	// we have an I-frame.
	if ((t=PacketSummaryAboveLlc (dsap, &(Buffer->Data1)+1, BufferLength-4)) != NULL) {
	    strcpy (p, t);
	    free (t);
	    return p;
	}
	PfFlag = (Buffer->Data1 & 0x01) != 0;
	sprintf (p, "LLC %c S=%02x D=%02x I%s  NR=%u NS=%u",
		 (CmdFlag ? 'C' : 'R'),
		 ssap, dsap,
		 CmdFlag ? (PfFlag ? "-p" : "") : (PfFlag ? "-f" : ""),
		 Buffer->Data1 >> 1,
		 Buffer->Command >> 1);
    } else {
	free (p);
	return NULL;
    }
    return p;
} // PacketSummaryLlc

VOID LlcTrailer (PDETREC *Ptr)
{
    AppendDetRec (Ptr, "LLC:");
} // LlcTrailer

PDETREC PacketDetailLlc (PPDU Buffer, USHORT BufferLength)
{
    PDETREC r=NULL, q;
    UCHAR *p, cmd, *t;
    UCHAR ssap, dsap;
    BOOLEAN CmdFlag, PfFlag;

    if (!(DisplayFilter & FILTER_LLC)) {
	return NULL;			// we aren't decoding LLC.
    }
    if (BufferLength < 3) {
	return NULL;			// this is a fragment.
    }

    AppendHeader (&r, "LLC:  ----- Logical Link Control header -----");

    //
    // Handle Madge Networks' bogus routing information over Ethernet.
    // This information is present in Token ring but definitely not
    // allowed on Ethernet.  Unfortunately, these guys didn't care and
    // dumped the routing information on Ethernet anyway, clobbering
    // the 0xC2 SAP.  Fortunately, nobody uses it anyway.
    //

    if (Buffer->Dsap == SAP_TR_ROUTING_INFO) {
	AppendDetRec (&r, "LLC:  Illegal Token Ring routing information skipped");
	t = (UCHAR *)Buffer;
	t += 2;                         // skip over this junk.
	Buffer = (PPDU)t;
    }

    ssap = Buffer->Ssap & ~(SSAP_RESPONSE);
    dsap = Buffer->Dsap;
    sprintf (ScratchBuf, "LLC:  Source SAP = 0x%02x (%s), Dest SAP = 0x%02x (%s)",
	     ssap, LlcSapName (ssap),
	     dsap, LlcSapName (dsap));
    AppendDetRec (&r, ScratchBuf);

    CmdFlag = (Buffer->Ssap & SSAP_RESPONSE) == 0;
    cmd = Buffer->Command;
    if ((cmd & 0xf3) == 0x01) {         // if this is an S-frame.
	switch (cmd & 0xf3) {
	    case DLCS_RR: t = "RR"; break;
	    case DLCS_RNR: t = "RNR"; break;
	    case DLCS_REJ: t = "REJ"; break;
	    default: t = "Invalid S-frame"; break;
	}
	PfFlag = (Buffer->Data1 & DLCS_PF) != 0;
	sprintf (ScratchBuf, "LLC:  Command = %s-%c%s",
		 t,
		 CmdFlag ? 'c' : 'r',
		 CmdFlag ? (PfFlag ? "-p" : "") : (PfFlag ? "-f" : ""));
	AppendDetRec (&r, ScratchBuf);
	sprintf (ScratchBuf, "LLC:  N(R) = %u (Acknowlegement Sequence Number)",
		 Buffer->Data1 & 0xfe);
	AppendDetRec (&r, ScratchBuf);
    } else if ((cmd & 0x03) == 0x03) {	// if this is a U-frame.
	switch (cmd & ~DLCU_PF) {	// remove POLL/FINAL bit.
	    case DLCU_SABME: t = "SABME"; break;
	    case DLCU_DISC: t = "DISC"; break;
	    case DLCU_UA: t = "UA"; break;
	    case DLCU_DM: t = "DM"; break;
	    case DLCU_FRMR: t = "FRMR"; break;
	    case DLCU_UI: t = "UI"; break;
	    case DLCU_XID: t = "XID"; break;
	    case DLCU_TEST: t = "TEST"; break;
	    default: t = "Invalid U-frame";
	}
	PfFlag = (Buffer->Command & DLCU_PF) != 0;
	sprintf (ScratchBuf, "LLC:  Command = %s-%c%s",
		 t,
		 CmdFlag ? 'c' : 'r',
		 CmdFlag ? (PfFlag ? "-p" : "") : (PfFlag ? "-f" : ""));
	AppendDetRec (&r, ScratchBuf);
	if (cmd == DLCU_UI) {
	    LlcTrailer (&r);
	    if ((q=PacketDetailAboveLlc (dsap, &(Buffer->Data1), BufferLength-3)) != NULL) {
		JoinDetRec (&r, q);
		return r;
	    }
	    sprintf (ScratchBuf,
		     "LLC:  Encapsulated data, size = %u (0x%04x hex) bytes.",
		     BufferLength-3, BufferLength-3);
	    AppendDetRec (&r, ScratchBuf);
	}
    } else if ((cmd & 0x01) == 0) {	// we have an I-frame.
	PfFlag = (Buffer->Data1 & 0x01) != 0;
	sprintf (ScratchBuf, "LLC:  Command = I-%c%s (Sequenced Information)",
		 CmdFlag ? 'c' : 'r',
		 CmdFlag ? (PfFlag ? "-p" : "") : (PfFlag ? "-f" : ""));
	AppendDetRec (&r, ScratchBuf);
	sprintf (ScratchBuf, "LLC:  N(R) = %u (Acknowlegement Sequence Number)",
		 Buffer->Data1 >> 1);
	AppendDetRec (&r, ScratchBuf);
	sprintf (ScratchBuf, "LLC:  N(S) = %u (Send Sequence Number)",
		 Buffer->Command >> 1);
	AppendDetRec (&r, ScratchBuf);
	LlcTrailer (&r);

	if ((q=PacketDetailAboveLlc (dsap, &(Buffer->Data1)+1, BufferLength-4)) != NULL) {
	    JoinDetRec (&r, q);
	    return r;
	}

	sprintf (ScratchBuf,
		 "LLC:  Encapsulated data, size = %u (0x%04x hex) bytes.",
		 BufferLength-4, BufferLength-4);
	AppendDetRec (&r, ScratchBuf);
    } else {				// we have nothing.
	AppendDetRec (&r, "LLC:  Unrecognizable LLC frame");
    }
    return r;
} // PacketDetailLlc
