//
// MODULE NAME:  IP.C.
//
// FUNCTIONAL DESCRIPTION.
//	This module handles packet summary and protocol decoding for IP,
//	the basis for the TCP/IP family of protocols.
//
//	There are two main entrypoints to this module.	PacketSummaryIp
//	returns a pointer to an ASCIIZ string allocated with malloc that
//	best describes the packet. PacketDetailIp returns a pointer to
//	a list of detail records (PDETREC) that describes the frame in detail.
//	If either function returns NULL, then no decoding is possible.
//
// MODIFICATION HISTORY.
//	S. E. Jones	92/04/03.	Original.
//	S. E. Jones	93/02/02.	#2.1, fixed swap of vers/hlen.
//	S. E. Jones	93/02/02.	#2.1, corrected header size calc.
//	S. E. Jones	93/02/02.	#2.1, correctly decoded TCP code bits.
//
// NOTICE:  Copyright (C) 1992-1993 General Software, Inc.  All rights reserved.
//

#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <dos.h>
#include "..\inc\system.h"		// DOS operating system defns.
#include "..\cow\cow.h"                 // character-oriented windows.
#include "..\inc\ktypes.h"		// commonly-used types.
#include "analyzer.h"			// common stuff for all modules.

typedef struct _IP {
    UCHAR HeaderAdmin;			// 7-4: vers, 3-0: headerlen.
    UCHAR TypeOfService;		// type of service.
    USHORT TotalIpLength;		// total pkt length, including header.
    USHORT PacketId;			// for fragmentation.
    USHORT FlagsNFrag;			// 0-2: flags, 3-15: fragment offset.
    UCHAR TimeToLive;			// in seconds.
    UCHAR Protocol;			// protocol type.

#define IP_PROTOCOL_ICMP	1
#define IP_PROTOCOL_GGP         3
#define IP_PROTOCOL_TCP         6
#define IP_PROTOCOL_EGP         8
#define IP_PROTOCOL_IGP         9
#define IP_PROTOCOL_UDP         17
#define IP_PROTOCOL_RVD         66
#define IP_PROTOCOL_SUNFS	77
#define IP_PROTOCOL_NETBLT	255

    USHORT CheckSum;			// header checksum.
    UCHAR SrcAddr [4];			// source internet address.
    UCHAR DestAddr [4];                 // destination internet address.
    UCHAR Options;			// option octet.
    UCHAR Padding [3];			// not used.
    UCHAR Buffer [1];			// start of encapsulated data.
} IP, *PIP;

typedef struct _UDP {
    USHORT SrcPort;			// source port number.
    USHORT DestPort;			// destination port number.

#define UDP_RJE                 5
#define UDP_ECHO		7
#define UDP_DISCARD		9
#define UDP_USERS		11
#define UDP_DAYTIME		13
#define UDP_NETSTAT		15
#define UDP_QUOTE		17
#define UDP_CHARGEN		19
#define UDP_TIME		37
#define UDP_RLP                 39
#define UDP_NAMESERVER		42
#define UDP_NICNAME		43
#define UDP_DOMAIN		53
#define UDP_BOOTPS		67
#define UDP_BOOTPC		68
#define UDP_TFTP		69
#define UDP_FINGER		79
#define UDP_NTP                 123

    USHORT Length;			// length of entire UDP pkt, including hdr.
    USHORT Checksum;			// checksum (0 if none).
    UCHAR Buffer [1];			// where the buffer starts.
} UDP, *PUDP;

typedef struct _TCP {
    USHORT SrcPort;			// source port number.
    USHORT DestPort;			// destination port number.

#define TCP_RJE                 5
#define TCP_ECHO		7
#define TCP_DISCARD		9
#define TCP_USERS		11
#define TCP_DAYTIME		13
#define TCP_NETSTAT		15
#define TCP_QUOTE		17
#define TCP_CHARGEN		19
#define TCP_FTPDATA		20
#define TCP_FTP                 21
#define TCP_TELNET		23
#define TCP_SMTP		25
#define TCP_TIME		37
#define TCP_RLP                 39
#define TCP_NAMESERVER		42
#define TCP_NICNAME		43
#define TCP_DOMAIN		53
#define TCP_BOOTPS		67
#define TCP_BOOTPC		68
#define TCP_TFTP		69
#define TCP_PRIVATE_DIALOUT	75
#define TCP_PRIVATE_RJE         77
#define TCP_FINGER		79
#define TCP_SUPDUP		95
#define TCP_HOSTNAME		101
#define TCP_ISO_TSAP		102
#define TCP_AUTH		113
#define TCP_UCCP_PATH		117
#define TCP_NTP                 123

    ULONG SequenceNumber;
    ULONG AckNumber;
    UCHAR OfsRes;
    UCHAR Code;

#define TCP_CODE_URG	0x20		// urgent data.
#define TCP_CODE_ACK	0x10		// acknowlegement field is valid.
#define TCP_CODE_PSH	0x08		// segment requests a push.
#define TCP_CODE_RST	0x04		// reset the connection.
#define TCP_CODE_SYN	0x02		// synchronize sequence numbers.
#define TCP_CODE_FIN	0x01		// sender has reached end of stream.

    USHORT Window;
    USHORT Checksum;
    USHORT UrgentPtr;
    UCHAR Options;
    UCHAR Padding [3];
} TCP, *PTCP;

static UCHAR IpAddrBuf [20];		// IP address buffer.

//
// Routines in other modules.
//

//
// Routines in this module.
//

BOOLEAN PacketReqTypeIp (PIP Buffer, USHORT BufferLength)
{
    return TRUE;
} // PacketReqTypeIp

BOOLEAN PacketCountIp (PIP Buffer, USHORT BufferLength)
{
    CountProtocolType (INDEX_IP);

    switch (Buffer->Protocol) {
	case IP_PROTOCOL_TCP:
	    CountProtocolType (INDEX_TCP);
	    return TRUE;
	    break;

	case IP_PROTOCOL_UDP:
	    CountProtocolType (INDEX_UDP);
	    return TRUE;

	default:
	    return FALSE;
    }
} // PacketCountIp

BOOLEAN PacketFilterIp (PIP Buffer, USHORT BufferLength)
{
    if (CaptureFilter & FILTER_IP) {
	return TRUE;			// we're accepting IP.
    }

    switch (Buffer->Protocol) {
	case IP_PROTOCOL_ICMP:		// currently we don't do anything here.
	case IP_PROTOCOL_GGP:
	case IP_PROTOCOL_EGP:
	case IP_PROTOCOL_IGP:
	case IP_PROTOCOL_RVD:
	case IP_PROTOCOL_SUNFS:
	case IP_PROTOCOL_NETBLT:
	    break;

	case IP_PROTOCOL_TCP:
	    return (CaptureFilter & FILTER_TCP) == FILTER_TCP;

	case IP_PROTOCOL_UDP:
	    return (CaptureFilter & FILTER_UDP) == FILTER_UDP;
    }
    return FALSE;			// don't accept this.
} // PacketFilterIp

PUCHAR IpAddress (InternetAddress)
    UCHAR *InternetAddress;
{
    sprintf (IpAddrBuf, "[%u.%u.%u.%u]",
	     InternetAddress [0], InternetAddress [1],
	     InternetAddress [2], InternetAddress [3]);
    return IpAddrBuf;
} // IpAddress

PUCHAR IpProtocolName (UCHAR Protocol)
{
    switch (Protocol) {
	case IP_PROTOCOL_ICMP: return "ICMP";
	case IP_PROTOCOL_GGP: return "GGP";
	case IP_PROTOCOL_TCP: return "TCP";
	case IP_PROTOCOL_EGP: return "EGP";
	case IP_PROTOCOL_IGP: return "IGP";
	case IP_PROTOCOL_UDP: return "UDP";
	case IP_PROTOCOL_RVD: return "RVD";
	case IP_PROTOCOL_SUNFS: return "SUNFS";
	case IP_PROTOCOL_NETBLT: return "NETBLT";
	default: return "Undecoded";
    }
} // IpProtocolName

PUCHAR UdpProtocolName (USHORT Protocol)
{
    switch (Protocol) {
	case UDP_RJE: return "RJE Service";
	case UDP_ECHO: return "ECHO Service";
	case UDP_DISCARD: return "Discard";
	case UDP_USERS: return "Active Users Service";
	case UDP_DAYTIME: return "Daytime Service";
	case UDP_NETSTAT: return "NetStat Service";
	case UDP_QUOTE: return "Quote Service";
	case UDP_CHARGEN: return "CHARGEN";
	case UDP_TIME: return "Time Service (TIME)";
	case UDP_RLP: return "Resource Location Protocol";
	case UDP_NAMESERVER: return "Host Name Service";
	case UDP_NICNAME: return "Whois Service";
	case UDP_DOMAIN: return "Domain Name Service";
	case UDP_BOOTPS: return "BOOTP Server";
	case UDP_BOOTPC: return "BOOTP Client";
	case UDP_TFTP: return "TFTP File Transfer";
	case UDP_FINGER: return "Finger Service";
	case UDP_NTP: return "Network Time Protocol (NTP)";
	default: return "Data";
    }
} // UdpProtocolName

PUCHAR TcpProtocolName (USHORT Protocol)
{
    switch (Protocol) {
	case TCP_RJE: return "RJE Service";
	case TCP_ECHO: return "Echo Service";
	case TCP_DISCARD: return "Discard";
	case TCP_USERS: return "Active Users Service";
	case TCP_DAYTIME: return "Daytime Service";
	case TCP_NETSTAT: return "NetStat Service";
	case TCP_QUOTE: return "Quote Service";
	case TCP_CHARGEN: return "CHARGEN";
	case TCP_FTPDATA: return "FTP Data";
	case TCP_FTP: return "FTP Control";
	case TCP_TELNET: return "Telnet";
	case TCP_SMTP: return "SMTP Mail";
	case TCP_TIME: return "Time Service (TIME)";
	case TCP_RLP: return "Resource Location Protocol";
	case TCP_NAMESERVER: return "Host Name Service";
	case TCP_NICNAME: return "Whois Service";
	case TCP_DOMAIN: return "Domain Name Service";
	case TCP_BOOTPS: return "BOOTP Server";
	case TCP_BOOTPC: return "BOOTP Client";
	case TCP_TFTP: return "TFTP File Transfer";
	case TCP_PRIVATE_DIALOUT: return "Private Dialout Service";
	case TCP_PRIVATE_RJE: return "Private RJE Service";
	case TCP_FINGER: return "Finger Service";
	case TCP_SUPDUP: return "SUPDUP Protocol";
	case TCP_HOSTNAME: return "NIC Host Name Service";
	case TCP_ISO_TSAP: return "ISO Protocol";
	case TCP_AUTH: return "Authentication Service";
	case TCP_UCCP_PATH: return "UUCP Path Service";
	case TCP_NTP: return "Network Time Protocol (NTP)";
	default: return "Data";
    }
} // TcpProtocolName

PUCHAR PacketSummaryUdp (PUDP Buffer, USHORT BufferLength)
{
    UCHAR *p, *t;

    if (!(DisplayFilter & FILTER_UDP)) {
	return NULL;			// we aren't decoding UDP.
    }

    p = malloc (80);
    if (p == NULL) {
	return NULL;
    }

    t = UdpProtocolName (Swap (Buffer->DestPort));
    if (!strcmp (t, "Data")) {
	t = UdpProtocolName (Swap (Buffer->SrcPort));
    }
    sprintf (p, "UDP %s", t);
    return p;
} // PacketSummaryUdp

PUCHAR PacketSummaryTcp (PTCP Buffer, USHORT BufferLength)
{
    UCHAR *p, *t;

    if (!(DisplayFilter & FILTER_TCP)) {
	return NULL;			// we aren't decoding TCP.
    }

    p = malloc (80);
    if (p == NULL) {
	return NULL;
    }

    t = TcpProtocolName (Swap (Buffer->DestPort));
    if (!strcmp (t, "Data")) {
	t = TcpProtocolName (Swap (Buffer->SrcPort));
    }
    sprintf (p, "TCP %s", t);
    return p;
} // PacketSummaryTcp

PUCHAR PacketSummaryIp (PIP Buffer, USHORT BufferLength)
{
    UCHAR *p, *t;

    if (!(DisplayFilter & FILTER_IP)) {
	return NULL;			// we aren't decoding UDP.
    }

    p = malloc (80);
    if (p == NULL) {
	return NULL;
    }

    //
    // Try to decode the higher-layer protocols.
    //

    switch (Buffer->Protocol) {
	case IP_PROTOCOL_ICMP:
	    strcpy (p, "ICMP Protocol");
	    return p;

	case IP_PROTOCOL_GGP:
	    strcpy (p, "Gateway to Gateway Protocol");
	    return p;

	case IP_PROTOCOL_EGP:
	    strcpy (p, "Exterior Gateway Protocol");
	    return p;

	case IP_PROTOCOL_IGP:
	    strcpy (p, "Interior Gateway Protocol");
	    return p;

	case IP_PROTOCOL_RVD:
	    strcpy (p, "Gateway to Gateway Protocol");
	    return p;

	case IP_PROTOCOL_SUNFS:
	    strcpy (p, "SUN NFS Protocol");
	    return p;

	case IP_PROTOCOL_NETBLT:
	    strcpy (p, "NetBlt Protocol");
	    return p;

	case IP_PROTOCOL_TCP:
	    t = PacketSummaryTcp ((PTCP)Buffer->Buffer, BufferLength-(sizeof (IP)-1));
	    if (t != NULL) {
		free (p);
		return t;
	    }
	    break;

	case IP_PROTOCOL_UDP:
	    t = PacketSummaryUdp ((PUDP)Buffer->Buffer, BufferLength-(sizeof (IP)-1));
	    if (t != NULL) {
		free (p);
		return t;
	    }
	    break;
    }
    sprintf (p, "IP  Encapsulated %s protocol", IpProtocolName (Buffer->Protocol));
    return p;
} // PacketSummaryIp

VOID IpTrailer (PDETREC *Ptr)
{
    AppendDetRec (Ptr, "IP:");
} // IpTrailer

PDETREC PacketDetailUdp (PUDP Buffer, USHORT BufferLength)
{
    PDETREC r=NULL, q;

    if (!(DisplayFilter & FILTER_UDP)) {
	return NULL;			// we aren't decoding UDP.
    }

    AppendHeader (&r, "UDP:  ----- User Datagram Protocol header -----");
    sprintf (ScratchBuf, "UDP:  Source port = 0x%04x  (%s)",
	     Swap (Buffer->SrcPort), UdpProtocolName (Swap (Buffer->SrcPort)));
    AppendDetRec (&r, ScratchBuf);
    sprintf (ScratchBuf, "UDP:  Destination port = 0x%04x  (%s)",
	     Swap (Buffer->DestPort), UdpProtocolName (Swap (Buffer->DestPort)));
    AppendDetRec (&r, ScratchBuf);
    sprintf (ScratchBuf, "UDP:  User datagram length = %u bytes",
	     Swap (Buffer->Length));
    AppendDetRec (&r, ScratchBuf);
    sprintf (ScratchBuf, "UDP:  Checksum = 0x%04x", Swap (Buffer->Checksum));
    AppendDetRec (&r, ScratchBuf);
    sprintf (ScratchBuf, "UDP:  Encapsulated data, size = %u (0x%04x hex) bytes.",
	     BufferLength-(sizeof (UDP)-1), BufferLength-(sizeof (UDP)-1));
    AppendDetRec (&r, ScratchBuf);
    return r;
} // PacketDetailUdp

PDETREC PacketDetailTcp (PTCP Buffer, USHORT BufferLength)
{
    PDETREC r=NULL, q;

    if (!(DisplayFilter & FILTER_TCP)) {
	return NULL;			// we aren't decoding TCP.
    }

    AppendHeader (&r, "TCP:  ----- Transmission Control Protocol header -----");
    sprintf (ScratchBuf, "TCP:  Source port = 0x%04x  (%s)",
	     Swap (Buffer->SrcPort), TcpProtocolName (Swap (Buffer->SrcPort)));
    AppendDetRec (&r, ScratchBuf);
    sprintf (ScratchBuf, "TCP:  Destination port = 0x%04x  (%s)",
	     Swap (Buffer->DestPort), TcpProtocolName (Swap (Buffer->DestPort)));
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf, "TCP:  Sequence Number = %08lx  Ack Number = %08lx",
	     SwapDword (Buffer->SequenceNumber),
	     SwapDword (Buffer->AckNumber));
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf, "TCP:  Offset to data in this datagram = %u",
	     (Buffer->OfsRes & 0x000f));
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf, "TCP:  Code = %02x", Buffer->Code);
    if (Buffer->Code & TCP_CODE_URG) {
	strcat (ScratchBuf, " URG");
    }
    if (Buffer->Code & TCP_CODE_ACK) {
	strcat (ScratchBuf, " ACK");
    }
    if (Buffer->Code & TCP_CODE_PSH) {
	strcat (ScratchBuf, " PSH");
    }
    if (Buffer->Code & TCP_CODE_RST) {
	strcat (ScratchBuf, " RST");
    }
    if (Buffer->Code & TCP_CODE_SYN) {
	strcat (ScratchBuf, " SYN");
    }
    if (Buffer->Code & TCP_CODE_FIN) {
	strcat (ScratchBuf, " FIN");
    }
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf, "TCP:  Advertized window = %u", Swap (Buffer->Window));
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf, "TCP:  Checksum = 0x%04x", Swap (Buffer->Checksum));
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf, "TCP:  Urgent data pointer = %u", Swap (Buffer->UrgentPtr));
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf, "TCP:  Encapsulated data, size = %u (0x%04x hex) bytes.",
	     BufferLength-(sizeof (TCP)-1), BufferLength-(sizeof (TCP)-1));
    AppendDetRec (&r, ScratchBuf);
    return r;
} // PacketDetailTcp

PDETREC PacketDetailIp (PIP Buffer, USHORT BufferLength)
{
    PDETREC r=NULL, q;
    USHORT tmp, HeaderLength, Version;
    UCHAR *ptr;

    AppendHeader (&r, "IP:  ----- Internet Protocol header -----");

    Version = (USHORT)(Buffer->HeaderAdmin >> 4);
    HeaderLength = (USHORT)(Buffer->HeaderAdmin & 0x0f);
    sprintf (ScratchBuf, "IP:  Header length = %u (0x%02x hex) 32-bit words, Version = %u",
	     HeaderLength, HeaderLength, Version);
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf, "IP:  Type of service = %u", Buffer->TypeOfService);
    AppendDetRec (&r, ScratchBuf);
    sprintf (ScratchBuf, "IP:  Total packet length = %u", Swap (Buffer->TotalIpLength));
    AppendDetRec (&r, ScratchBuf);

    tmp = Swap (Buffer->FlagsNFrag);
    sprintf (ScratchBuf, "IP:  Frag Ctrl:  Packet ID = %u, Offset = %u, Flags = 0x%02x",
	     Buffer->PacketId, (tmp >> 3), (tmp & 0x0007));
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf, "IP:  Time to live (seconds) = %u", Buffer->TimeToLive);
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf, "IP:  Encapsulated protocol = %u  (%s)",
	     Buffer->Protocol, IpProtocolName (Buffer->Protocol));
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf, "IP:  Checksum = %04x", Swap (Buffer->CheckSum));
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf, "IP:  Source IP address = %s", IpAddress (Buffer->SrcAddr));
    AppendDetRec (&r, ScratchBuf);

    sprintf (ScratchBuf, "IP:  Dest IP address = %s", IpAddress (Buffer->DestAddr));
    AppendDetRec (&r, ScratchBuf);
    IpTrailer (&r);

    //
    // Decode higher-layer protocols.
    //

    ptr = (UCHAR *)Buffer;
    ptr += (HeaderLength * 4);		// advance over options and padding.
    switch (Buffer->Protocol) {
	case IP_PROTOCOL_UDP:
	    q = PacketDetailUdp ((PUDP)ptr, BufferLength-(HeaderLength*4));
	    if (q != NULL) {
		JoinDetRec (&r, q);
	    }
	    break;

	case IP_PROTOCOL_TCP:
	    q = PacketDetailTcp ((PTCP)ptr, BufferLength-(HeaderLength*4));
	    if (q != NULL) {
		JoinDetRec (&r, q);
	    }
	    break;

	case IP_PROTOCOL_ICMP:
	case IP_PROTOCOL_GGP:
	case IP_PROTOCOL_EGP:
	case IP_PROTOCOL_IGP:
	case IP_PROTOCOL_RVD:
	case IP_PROTOCOL_SUNFS:
	case IP_PROTOCOL_NETBLT:
	    sprintf (ScratchBuf, "IP:  Encapsulated %s protocol fields not decoded",
		     IpProtocolName (Buffer->Protocol));
	    AppendDetRec (&r, ScratchBuf);
	    break;
	default:
	    sprintf (ScratchBuf, "IP:  Encapsulated data, size = %u (0x%04x hex) bytes.",
		     BufferLength-(HeaderLength*4), BufferLength-(HeaderLength*4));
	    AppendDetRec (&r, ScratchBuf);
    }
    return r;
} // PacketDetailIp
