//
// MODULE NAME:  GSIPC.C.
//
// FUNCTIONAL DESCRIPTION.
//	This module handles packet summary and protocol decoding for
//	General Software's IPC Protocol for Embedded LAN.
//
//	There are two main entrypoints to this module.	PacketSummaryGsIpc
//	returns a pointer to an ASCIIZ string allocated with malloc that
//	best describes the packet. PacketDetailGsIpc returns a pointer to
//	a list of detail records (PDETREC) that describes the frame in detail.
//	If either function returns NULL, then no decoding is possible.
//
// MODIFICATION HISTORY.
//	S. E. Jones	92/11/24.	Original for 1.9.
//	S. E. Jones	93/02/05.	#2.2, cloned from GSLLCTDI.C.
//	S. E. Jones	93/02/10.	#2.3, added GSFSP I/O request counts.
//	S. E. Jones	93/03/16.	#2.5, cloned for GSIPC.
//
// NOTICE:  Copyright (C) 1992-1993 General Software, Inc.  All rights reserved.
//

#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <dos.h>
#include "..\inc\system.h"		// DOS operating system defns.
#include "..\cow\cow.h"                 // character-oriented windows.
#include "..\inc\ktypes.h"		// commonly-used types.
#include "analyzer.h"			// common stuff for all modules.
#include "ipcp.h"			// interprocess communication protocol.

//
// Routines in other modules.
//

//
// Routines in this module.
//

BOOLEAN PacketReqTypeGsIpc (PVOID Buffer, USHORT BufferLength)
{
    PIPC_FRAME pheader;

    pheader = (PIPC_FRAME)Buffer;

    if (BufferLength < 6) {             // need signature & cmd as a minimum.
	return FALSE;			// partial record not decoded.
    }

    if (pheader->Signature != IPC_FRAME_SIGNATURE) {
	return FALSE;
    }
    return TRUE;		       // found one of ours.
} // PacketReqTypeGsIpc

BOOLEAN PacketFilterGsIpc (PVOID Buffer, USHORT BufferLength)
{
    PIPC_FRAME pheader;

    pheader = (PIPC_FRAME)Buffer;

    if (BufferLength < 6) {
	return FALSE;			// partial record not decoded.
    }

    if (pheader->Signature != IPC_FRAME_SIGNATURE) {
	return FALSE;
    }

    if (CaptureFilter & FILTER_GSIPC) {
	return TRUE;			// we're accepting GSIPC.
    }

    return FALSE;
} // PacketFilterGsIpc

PUCHAR GsIpcCmdName (USHORT Cmd)
{
    switch (Cmd) {
	case IPC_FRAME_REQ_NULL:		return "C Null";
	case IPC_FRAME_RSP_NULL:		return "R Null";
	case IPC_FRAME_REQ_CREATE_Q:		return "C Create Queue";
	case IPC_FRAME_RSP_CREATE_Q:		return "R Create Queue";
	case IPC_FRAME_REQ_OPEN_Q:		return "C Open Queue";
	case IPC_FRAME_RSP_OPEN_Q:		return "R Open Queue";
	case IPC_FRAME_REQ_CLOSE_Q:		return "C Close Queue";
	case IPC_FRAME_RSP_CLOSE_Q:		return "R Close Queue";
	case IPC_FRAME_REQ_PUSH_Q:		return "C Push Queue";
	case IPC_FRAME_RSP_PUSH_Q:		return "R Push Queue";
	case IPC_FRAME_REQ_APPEND_Q:		return "C Append Queue";
	case IPC_FRAME_RSP_APPEND_Q:		return "R Append Queue";
	case IPC_FRAME_REQ_POP_Q:		return "C Pop Queue";
	case IPC_FRAME_RSP_POP_Q:		return "R Pop Queue";
	case IPC_FRAME_REQ_CREATE_MSGPORT:	return "C Create Message Port";
	case IPC_FRAME_RSP_CREATE_MSGPORT:	return "R Create Message Port";
	case IPC_FRAME_REQ_OPEN_MSGPORT:	return "C Open Message Port";
	case IPC_FRAME_RSP_OPEN_MSGPORT:	return "R Open Message Port";
	case IPC_FRAME_REQ_CLOSE_MSGPORT:	return "C Close Message Port";
	case IPC_FRAME_RSP_CLOSE_MSGPORT:	return "R Close Message Port";
	case IPC_FRAME_REQ_SEND_MSGPORT:	return "C Send Message";
	case IPC_FRAME_RSP_SEND_MSGPORT:	return "R Send Message";
	case IPC_FRAME_REQ_RECEIVE_MSGPORT:	return "C Receive Message";
	case IPC_FRAME_RSP_RECEIVE_MSGPORT:	return "R Receive Message";
	default:				return "Undecoded";
    }
} // GsIpcCmdName

PUCHAR PacketSummaryGsIpc (PVOID Buffer, USHORT BufferLength)
{
    PIPC_REQ_CREATE_Q IpcCreateQueueReq;
    PIPC_RSP_CREATE_Q IpcCreateQueueRsp;
    PIPC_REQ_OPEN_Q IpcOpenQueueReq;
    PIPC_RSP_OPEN_Q IpcOpenQueueRsp;
    PIPC_REQ_CLOSE_Q IpcCloseQueueReq;
    PIPC_RSP_CLOSE_Q IpcCloseQueueRsp;
    PIPC_REQ_PUSH_Q IpcPushQueueReq;
    PIPC_RSP_PUSH_Q IpcPushQueueRsp;
    PIPC_REQ_APPEND_Q IpcAppendQueueReq;
    PIPC_RSP_APPEND_Q IpcAppendQueueRsp;
    PIPC_REQ_POP_Q IpcPopQueueReq;
    PIPC_RSP_POP_Q IpcPopQueueRsp;
    PIPC_REQ_CREATE_MSGPORT IpcCreateMsgPortReq;
    PIPC_RSP_CREATE_MSGPORT IpcCreateMsgPortRsp;
    PIPC_REQ_OPEN_MSGPORT IpcOpenMsgPortReq;
    PIPC_RSP_OPEN_MSGPORT IpcOpenMsgPortRsp;
    PIPC_REQ_CLOSE_MSGPORT IpcCloseMsgPortReq;
    PIPC_RSP_CLOSE_MSGPORT IpcCloseMsgPortRsp;
    PIPC_REQ_SEND_MSGPORT IpcSendMsgPortReq;
    PIPC_RSP_SEND_MSGPORT IpcSendMsgPortRsp;
    PIPC_REQ_RECEIVE_MSGPORT IpcReceiveMsgPortReq;
    PIPC_RSP_RECEIVE_MSGPORT IpcReceiveMsgPortRsp;
    PIPC_FRAME pheader;
    UCHAR *p, *t;

    pheader = (PIPC_FRAME)Buffer;

    if (!(DisplayFilter & FILTER_GSIPC)) {
	return NULL;			// we're not decoding GSIPC.
    }

    p = malloc (80);
    if (p == NULL) {
	return NULL;
    }

    if ((pheader->Signature != IPC_FRAME_SIGNATURE) || (BufferLength < 6)) {
	free (p);
	return NULL;
    }

    switch (pheader->Cmd) {
	case IPC_FRAME_REQ_CREATE_Q:
	    IpcCreateQueueReq = (PIPC_REQ_CREATE_Q)&(pheader->Operands);
	    sprintf (p, "IPC %s '%s'",
		     GsIpcCmdName (pheader->Cmd),
		     IpcCreateQueueReq->QueueName);
	    break;

	case IPC_FRAME_RSP_CREATE_Q:
	    IpcCreateQueueRsp = (PIPC_RSP_CREATE_Q)&(pheader->Operands);
	    sprintf (p, "IPC %s (%u)  Hndl=%u",
		     GsIpcCmdName (pheader->Cmd),
		     IpcCreateQueueRsp->Status,
		     IpcCreateQueueRsp->Handle);
	    break;

	case IPC_FRAME_REQ_OPEN_Q:
	    IpcOpenQueueReq = (PIPC_REQ_OPEN_Q)&(pheader->Operands);
	    sprintf (p, "IPC %s '%s'",
		     GsIpcCmdName (pheader->Cmd),
		     IpcOpenQueueReq->QueueName);
	    break;

	case IPC_FRAME_RSP_OPEN_Q:
	    IpcOpenQueueRsp = (PIPC_RSP_OPEN_Q)&(pheader->Operands);
	    sprintf (p, "IPC %s (%u)  Hndl=%u",
		     GsIpcCmdName (pheader->Cmd),
		     IpcOpenQueueRsp->Status,
		     IpcOpenQueueRsp->Handle);
	    break;

	case IPC_FRAME_REQ_CLOSE_Q:
	    IpcCloseQueueReq = (PIPC_REQ_CLOSE_Q)&(pheader->Operands);
	    sprintf (p, "IPC %s  Hndl=%u",
		     GsIpcCmdName (pheader->Cmd),
		     IpcCloseQueueReq->Handle);
	    break;

	case IPC_FRAME_RSP_CLOSE_Q:
	    IpcCloseQueueRsp = (PIPC_RSP_CLOSE_Q)&(pheader->Operands);
	    sprintf (p, "IPC %s (%u)",
		     GsIpcCmdName (pheader->Cmd),
		     IpcCloseQueueRsp->Status);
	    break;

	case IPC_FRAME_REQ_PUSH_Q:
	    IpcPushQueueReq = (PIPC_REQ_PUSH_Q)&(pheader->Operands);
	    sprintf (p, "IPC %s  Hndl=%u, Datum=0x%08lx",
		     GsIpcCmdName (pheader->Cmd),
		     IpcPushQueueReq->Handle,
		     IpcPushQueueReq->Datum);
	    break;

	case IPC_FRAME_RSP_PUSH_Q:
	    IpcPushQueueRsp = (PIPC_RSP_PUSH_Q)&(pheader->Operands);
	    sprintf (p, "IPC %s (%u)",
		     GsIpcCmdName (pheader->Cmd),
		     IpcPushQueueRsp->Status);
	    break;

	case IPC_FRAME_REQ_APPEND_Q:
	    IpcAppendQueueReq = (PIPC_REQ_APPEND_Q)&(pheader->Operands);
	    sprintf (p, "IPC %s  Hndl=%u, Datum=0x%08lx",
		     GsIpcCmdName (pheader->Cmd),
		     IpcAppendQueueReq->Handle,
		     IpcAppendQueueReq->Datum);
	    break;

	case IPC_FRAME_RSP_APPEND_Q:
	    IpcAppendQueueRsp = (PIPC_RSP_APPEND_Q)&(pheader->Operands);
	    sprintf (p, "IPC %s (%u)",
		     GsIpcCmdName (pheader->Cmd),
		     IpcAppendQueueRsp->Status);
	    break;

	case IPC_FRAME_REQ_POP_Q:
	    IpcPopQueueReq = (PIPC_REQ_POP_Q)&(pheader->Operands);
	    sprintf (p, "IPC %s  Hndl=%u",
		     GsIpcCmdName (pheader->Cmd),
		     IpcPopQueueReq->Handle);
	    break;

	case IPC_FRAME_RSP_POP_Q:
	    IpcPopQueueRsp = (PIPC_RSP_POP_Q)&(pheader->Operands);
	    sprintf (p, "IPC %s (%u)",
		     GsIpcCmdName (pheader->Cmd),
		     IpcPopQueueRsp->Status);
	    break;

	case IPC_FRAME_REQ_CREATE_MSGPORT:
	    IpcCreateMsgPortReq = (PIPC_REQ_CREATE_MSGPORT)&(pheader->Operands);
	    sprintf (p, "IPC %s '%s'",
		     GsIpcCmdName (pheader->Cmd),
		     IpcCreateMsgPortReq->MsgPortName);
	    break;

	case IPC_FRAME_RSP_CREATE_MSGPORT:
	    IpcCreateMsgPortRsp = (PIPC_RSP_CREATE_MSGPORT)&(pheader->Operands);
	    sprintf (p, "IPC %s (%u)  Hndl=%u",
		     GsIpcCmdName (pheader->Cmd),
		     IpcCreateMsgPortRsp->Status,
		     IpcCreateMsgPortRsp->Handle);
	    break;

	case IPC_FRAME_REQ_OPEN_MSGPORT:
	    IpcOpenMsgPortReq = (PIPC_REQ_OPEN_MSGPORT)&(pheader->Operands);
	    sprintf (p, "IPC %s '%s'",
		     GsIpcCmdName (pheader->Cmd),
		     IpcOpenMsgPortReq->MsgPortName);
	    break;

	case IPC_FRAME_RSP_OPEN_MSGPORT:
	    IpcOpenMsgPortRsp = (PIPC_RSP_OPEN_MSGPORT)&(pheader->Operands);
	    sprintf (p, "IPC %s (%u)  Hndl=%u",
		     GsIpcCmdName (pheader->Cmd),
		     IpcOpenMsgPortRsp->Status,
		     IpcOpenMsgPortRsp->Handle);
	    break;

	case IPC_FRAME_REQ_CLOSE_MSGPORT:
	    IpcCloseMsgPortReq = (PIPC_REQ_CLOSE_MSGPORT)&(pheader->Operands);
	    sprintf (p, "IPC %s  Hndl=%u",
		     GsIpcCmdName (pheader->Cmd),
		     IpcCloseMsgPortReq->Handle);
	    break;

	case IPC_FRAME_RSP_CLOSE_MSGPORT:
	    IpcCloseMsgPortRsp = (PIPC_RSP_CLOSE_MSGPORT)&(pheader->Operands);
	    sprintf (p, "IPC %s (%u)",
		     GsIpcCmdName (pheader->Cmd),
		     IpcCloseMsgPortRsp->Status);
	    break;

	case IPC_FRAME_REQ_SEND_MSGPORT:
	    IpcSendMsgPortReq = (PIPC_REQ_SEND_MSGPORT)&(pheader->Operands);
	    sprintf (p, "IPC %s  Hndl=%u  Len=%u  Flags=0x%04x",
		     GsIpcCmdName (pheader->Cmd),
		     IpcSendMsgPortReq->Handle,
		     IpcSendMsgPortReq->MsgLength,
		     IpcSendMsgPortReq->Flags);
	    break;

	case IPC_FRAME_RSP_SEND_MSGPORT:
	    IpcSendMsgPortRsp = (PIPC_RSP_SEND_MSGPORT)&(pheader->Operands);
	    sprintf (p, "IPC %s (%u)",
		     GsIpcCmdName (pheader->Cmd),
		     IpcSendMsgPortRsp->Status);
	    break;

	case IPC_FRAME_REQ_RECEIVE_MSGPORT:
	    IpcReceiveMsgPortReq = (PIPC_REQ_RECEIVE_MSGPORT)&(pheader->Operands);
	    sprintf (p, "IPC %s  Hndl=%u  Len=%u  Flags=0x%04x",
		     GsIpcCmdName (pheader->Cmd),
		     IpcReceiveMsgPortReq->Handle,
		     IpcReceiveMsgPortReq->BufferLength,
		     IpcReceiveMsgPortReq->Flags);
	    break;

	case IPC_FRAME_RSP_RECEIVE_MSGPORT:
	    IpcReceiveMsgPortRsp = (PIPC_RSP_RECEIVE_MSGPORT)&(pheader->Operands);
	    sprintf (p, "IPC %s (%u)",
		     GsIpcCmdName (pheader->Cmd),
		     IpcReceiveMsgPortRsp->Status,
		     IpcReceiveMsgPortRsp->BytesTransferred);
	    break;

	case IPC_FRAME_REQ_NULL:
	case IPC_FRAME_RSP_NULL:
	    sprintf (p, "IPC %s", GsIpcCmdName (pheader->Cmd));
	    break;

	default:
	    free (p);
	    return NULL;
    }
    return p;
} // PacketSummaryGsIpc

VOID GsIpcTrailer (PDETREC *Ptr)
{
    AppendDetRec (Ptr, "IPC:");
} // GsIpcTrailer

PDETREC PacketDetailGsIpc (PVOID Buffer, USHORT BufferLength)
{
    PIPC_REQ_CREATE_Q IpcCreateQueueReq;
    PIPC_RSP_CREATE_Q IpcCreateQueueRsp;
    PIPC_REQ_OPEN_Q IpcOpenQueueReq;
    PIPC_RSP_OPEN_Q IpcOpenQueueRsp;
    PIPC_REQ_CLOSE_Q IpcCloseQueueReq;
    PIPC_RSP_CLOSE_Q IpcCloseQueueRsp;
    PIPC_REQ_PUSH_Q IpcPushQueueReq;
    PIPC_RSP_PUSH_Q IpcPushQueueRsp;
    PIPC_REQ_APPEND_Q IpcAppendQueueReq;
    PIPC_RSP_APPEND_Q IpcAppendQueueRsp;
    PIPC_REQ_POP_Q IpcPopQueueReq;
    PIPC_RSP_POP_Q IpcPopQueueRsp;
    PIPC_REQ_CREATE_MSGPORT IpcCreateMsgPortReq;
    PIPC_RSP_CREATE_MSGPORT IpcCreateMsgPortRsp;
    PIPC_REQ_OPEN_MSGPORT IpcOpenMsgPortReq;
    PIPC_RSP_OPEN_MSGPORT IpcOpenMsgPortRsp;
    PIPC_REQ_CLOSE_MSGPORT IpcCloseMsgPortReq;
    PIPC_RSP_CLOSE_MSGPORT IpcCloseMsgPortRsp;
    PIPC_REQ_SEND_MSGPORT IpcSendMsgPortReq;
    PIPC_RSP_SEND_MSGPORT IpcSendMsgPortRsp;
    PIPC_REQ_RECEIVE_MSGPORT IpcReceiveMsgPortReq;
    PIPC_RSP_RECEIVE_MSGPORT IpcReceiveMsgPortRsp;
    PIPC_FRAME pheader;
    PDETREC r=NULL, q;

    pheader = (PIPC_FRAME)Buffer;

    if (!(DisplayFilter & FILTER_GSIPC)) {
	return NULL;			// we're not decoding GSIPC.
    }

    sprintf (ScratchBuf, "IPC:  ----- General Software IPC %s -----",
	     GsIpcCmdName (pheader->Cmd));
    AppendHeader (&r, ScratchBuf);

    if ((pheader->Signature != IPC_FRAME_SIGNATURE) || (BufferLength < 6)) {
	return NULL;
    }

    sprintf (ScratchBuf, "IPC:  Command = 0x%04x  (%s)",
	     pheader->Cmd, GsIpcCmdName (pheader->Cmd));
    AppendDetRec (&r, ScratchBuf);

    switch (pheader->Cmd) {
	case IPC_FRAME_REQ_CREATE_Q:
	    IpcCreateQueueReq = (PIPC_REQ_CREATE_Q)&(pheader->Operands);
	    sprintf (ScratchBuf, "IPC:  Queue Name = '%s'", IpcCreateQueueReq->QueueName);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case IPC_FRAME_RSP_CREATE_Q:
	    IpcCreateQueueRsp = (PIPC_RSP_CREATE_Q)&(pheader->Operands);
	    sprintf (ScratchBuf, "IPC:  Status = %u (0x%04x)",
		     IpcCreateQueueRsp->Status, IpcCreateQueueRsp->Status);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "IPC:  Handle = %u", IpcCreateQueueRsp->Handle);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case IPC_FRAME_REQ_OPEN_Q:
	    IpcOpenQueueReq = (PIPC_REQ_OPEN_Q)&(pheader->Operands);
	    sprintf (ScratchBuf, "IPC:  Queue Name = '%s'", IpcOpenQueueReq->QueueName);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case IPC_FRAME_RSP_OPEN_Q:
	    IpcOpenQueueRsp = (PIPC_RSP_OPEN_Q)&(pheader->Operands);
	    sprintf (ScratchBuf, "IPC:  Status = %u (0x%04x)",
		     IpcOpenQueueRsp->Status, IpcOpenQueueRsp->Status);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "IPC:  Handle = %u", IpcOpenQueueRsp->Handle);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case IPC_FRAME_REQ_CLOSE_Q:
	    IpcCloseQueueReq = (PIPC_REQ_CLOSE_Q)&(pheader->Operands);
	    sprintf (ScratchBuf, "IPC:  Handle = %u", IpcCloseQueueReq->Handle);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case IPC_FRAME_RSP_CLOSE_Q:
	    IpcCloseQueueRsp = (PIPC_RSP_CLOSE_Q)&(pheader->Operands);
	    sprintf (ScratchBuf, "IPC:  Status = %u (0x%04x)",
		     IpcCloseQueueRsp->Status, IpcCloseQueueRsp->Status);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case IPC_FRAME_REQ_PUSH_Q:
	    IpcPushQueueReq = (PIPC_REQ_PUSH_Q)&(pheader->Operands);
	    sprintf (ScratchBuf, "IPC:  Handle = %u", IpcPushQueueReq->Handle);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "IPC:  Datum = 0x%08lx", IpcPushQueueReq->Datum);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case IPC_FRAME_RSP_PUSH_Q:
	    IpcPushQueueRsp = (PIPC_RSP_PUSH_Q)&(pheader->Operands);
	    sprintf (ScratchBuf, "IPC:  Status = %u (0x%04x)",
		     IpcPushQueueRsp->Status, IpcPushQueueRsp->Status);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case IPC_FRAME_REQ_APPEND_Q:
	    IpcAppendQueueReq = (PIPC_REQ_APPEND_Q)&(pheader->Operands);
	    sprintf (ScratchBuf, "IPC:  Handle = %u", IpcAppendQueueReq->Handle);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "IPC:  Datum = 0x%08lx", IpcAppendQueueReq->Datum);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case IPC_FRAME_RSP_APPEND_Q:
	    IpcAppendQueueRsp = (PIPC_RSP_APPEND_Q)&(pheader->Operands);
	    sprintf (ScratchBuf, "IPC:  Status = %u (0x%04x)",
		     IpcAppendQueueRsp->Status, IpcAppendQueueRsp->Status);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case IPC_FRAME_REQ_POP_Q:
	    IpcPopQueueReq = (PIPC_REQ_POP_Q)&(pheader->Operands);
	    sprintf (ScratchBuf, "IPC:  Handle = %u", IpcPopQueueReq->Handle);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case IPC_FRAME_RSP_POP_Q:
	    IpcPopQueueRsp = (PIPC_RSP_POP_Q)&(pheader->Operands);
	    sprintf (ScratchBuf, "IPC:  Status = %u (0x%04x)",
		     IpcPopQueueRsp->Status, IpcPopQueueRsp->Status);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "IPC:  Datum = 0x%08lx",
		     IpcPopQueueRsp->Datum);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case IPC_FRAME_REQ_CREATE_MSGPORT:
	    IpcCreateMsgPortReq = (PIPC_REQ_CREATE_MSGPORT)&(pheader->Operands);
	    sprintf (ScratchBuf, "IPC:  Message Port Name = '%s'", IpcCreateMsgPortReq->MsgPortName);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case IPC_FRAME_RSP_CREATE_MSGPORT:
	    IpcCreateMsgPortRsp = (PIPC_RSP_CREATE_MSGPORT)&(pheader->Operands);
	    sprintf (ScratchBuf, "IPC:  Status = %u (0x%04x)",
		     IpcCreateMsgPortRsp->Status, IpcCreateMsgPortRsp->Status);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "IPC:  Handle = %u", IpcCreateMsgPortRsp->Handle);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case IPC_FRAME_REQ_OPEN_MSGPORT:
	    IpcOpenMsgPortReq = (PIPC_REQ_OPEN_MSGPORT)&(pheader->Operands);
	    sprintf (ScratchBuf, "IPC:  Message Port Name = '%s'", IpcOpenMsgPortReq->MsgPortName);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case IPC_FRAME_RSP_OPEN_MSGPORT:
	    IpcOpenMsgPortRsp = (PIPC_RSP_OPEN_MSGPORT)&(pheader->Operands);
	    sprintf (ScratchBuf, "IPC:  Status = %u (0x%04x)",
		     IpcOpenMsgPortRsp->Status, IpcOpenMsgPortRsp->Status);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "IPC:  Handle = %u", IpcOpenMsgPortRsp->Handle);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case IPC_FRAME_REQ_CLOSE_MSGPORT:
	    IpcCloseMsgPortReq = (PIPC_REQ_CLOSE_MSGPORT)&(pheader->Operands);
	    sprintf (ScratchBuf, "IPC:  Handle = %u", IpcCloseMsgPortReq->Handle);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case IPC_FRAME_RSP_CLOSE_MSGPORT:
	    IpcCloseMsgPortRsp = (PIPC_RSP_CLOSE_MSGPORT)&(pheader->Operands);
	    sprintf (ScratchBuf, "IPC:  Status = %u (0x%04x)",
		     IpcCloseMsgPortRsp->Status, IpcCloseMsgPortRsp->Status);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case IPC_FRAME_REQ_SEND_MSGPORT:
	    IpcSendMsgPortReq = (PIPC_REQ_SEND_MSGPORT)&(pheader->Operands);
	    sprintf (ScratchBuf, "IPC:  Handle = %u", IpcSendMsgPortReq->Handle);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "IPC:  Message Length = %u", IpcSendMsgPortReq->MsgLength);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case IPC_FRAME_RSP_SEND_MSGPORT:
	    IpcSendMsgPortRsp = (PIPC_RSP_SEND_MSGPORT)&(pheader->Operands);
	    sprintf (ScratchBuf, "IPC:  Status = %u (0x%04x)",
		     IpcSendMsgPortRsp->Status, IpcSendMsgPortRsp->Status);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case IPC_FRAME_REQ_RECEIVE_MSGPORT:
	    IpcReceiveMsgPortReq = (PIPC_REQ_RECEIVE_MSGPORT)&(pheader->Operands);
	    sprintf (ScratchBuf, "IPC:  Handle = %u", IpcReceiveMsgPortReq->Handle);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "IPC:  Buffer Length = %u", IpcReceiveMsgPortReq->BufferLength);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case IPC_FRAME_RSP_RECEIVE_MSGPORT:
	    IpcReceiveMsgPortRsp = (PIPC_RSP_RECEIVE_MSGPORT)&(pheader->Operands);
	    sprintf (ScratchBuf, "IPC:  Status = %u (0x%04x)",
		     IpcReceiveMsgPortRsp->Status, IpcReceiveMsgPortRsp->Status);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "IPC:  Message Length = %u", IpcReceiveMsgPortRsp->BytesTransferred);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case IPC_FRAME_REQ_NULL:
	case IPC_FRAME_RSP_NULL:
	default:			// nothing else to print.
	    return r;			// DO NOTHING.
    }
    return r;
} // PacketDetailGsIpc
