//
// MODULE NAME:  GSFSP.C.
//
// FUNCTIONAL DESCRIPTION.
//	This module handles packet summary and protocol decoding for
//	General Software's File Sharing Protocol for Embedded LAN.
//
//	There are two main entrypoints to this module.	PacketSummaryGsFsp
//	returns a pointer to an ASCIIZ string allocated with malloc that
//	best describes the packet. PacketDetailGsFsp returns a pointer to
//	a list of detail records (PDETREC) that describes the frame in detail.
//	If either function returns NULL, then no decoding is possible.
//
// MODIFICATION HISTORY.
//	S. E. Jones	92/11/24.	Original for 1.9.
//	S. E. Jones	93/02/05.	#2.2, cloned from GSLLCTDI.C.
//	S. E. Jones	93/02/10.	#2.3, added GSFSP I/O request counts.
//
// NOTICE:  Copyright (C) 1992-1993 General Software, Inc.  All rights reserved.
//

#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <dos.h>
#include "..\inc\system.h"		// DOS operating system defns.
#include "..\cow\cow.h"                 // character-oriented windows.
#include "..\inc\ktypes.h"		// commonly-used types.
#include "analyzer.h"			// common stuff for all modules.
#include "fsp.h"			// file sharing protocol formats.

//
// Routines in other modules.
//

//
// Routines in this module.
//

BOOLEAN PacketReqTypeGsFsp (PVOID Buffer, USHORT BufferLength)
{
    PFSP_FRAME pheader;

    pheader = (PFSP_FRAME)Buffer;

    if (BufferLength < 6) {             // need signature & cmd as a minimum.
	return FALSE;			// partial record not decoded.
    }

    if (pheader->Signature != FSP_FRAME_SIGNATURE) {
	return FALSE;
    }

    switch (pheader->Cmd) {
	case FSP_FRAME_REQ_OPEN:
	    CountFileIoType (FILEIO_OPEN);
	    break;

	case FSP_FRAME_REQ_CLOSE:
	    CountFileIoType (FILEIO_CLOSE);
	    break;

	case FSP_FRAME_REQ_CREATE:
	    CountFileIoType (FILEIO_CREATE);
	    break;

	case FSP_FRAME_REQ_DELETE:
	    CountFileIoType (FILEIO_DELETE);
	    break;

	case FSP_FRAME_REQ_READ:
	    CountFileIoType (FILEIO_READ);
	    break;

	case FSP_FRAME_REQ_WRITE:
	    CountFileIoType (FILEIO_WRITE);
	    break;

	case FSP_FRAME_REQ_RENAME:
	    CountFileIoType (FILEIO_RENAME);
	    break;

	case FSP_FRAME_REQ_FINDFIRST:
	case FSP_FRAME_REQ_FINDNEXT:
	    CountFileIoType (FILEIO_SEARCH);
	    break;

	case FSP_FRAME_REQ_LOCK:
	    CountFileIoType (FILEIO_LOCK);
	    break;

	case FSP_FRAME_REQ_UNLOCK:
	    CountFileIoType (FILEIO_UNLOCK);
	    break;

    }
    return TRUE;		       // found one of ours.
} // PacketReqTypeGsFsp

BOOLEAN PacketFilterGsFsp (PVOID Buffer, USHORT BufferLength)
{
    PFSP_FRAME pheader;

    pheader = (PFSP_FRAME)Buffer;

    if (BufferLength < 6) {
	return FALSE;			// partial record not decoded.
    }

    if (pheader->Signature != FSP_FRAME_SIGNATURE) {
	return FALSE;
    }

    if (CaptureFilter & FILTER_GSFSP) {
	return TRUE;			// we're accepting GSFSP.
    }

    return FALSE;
} // PacketFilterGsFsp

PUCHAR GsFspCmdName (USHORT Cmd)
{
    switch (Cmd) {
	case FSP_FRAME_REQ_NULL:	return "C Null";
	case FSP_FRAME_RSP_NULL:	return "R Null";
	case FSP_FRAME_REQ_OPEN:	return "C Open";
	case FSP_FRAME_RSP_OPEN:	return "R Open";
	case FSP_FRAME_REQ_CLOSE:	return "C Close";
	case FSP_FRAME_RSP_CLOSE:	return "R Close";
	case FSP_FRAME_REQ_CREATE:	return "C Create";
	case FSP_FRAME_RSP_CREATE:	return "R Create";
	case FSP_FRAME_REQ_DELETE:	return "C Delete";
	case FSP_FRAME_RSP_DELETE:	return "R Delete";
	case FSP_FRAME_REQ_READ:	return "C Read";
	case FSP_FRAME_RSP_READ:	return "R Read";
	case FSP_FRAME_REQ_WRITE:	return "C Write";
	case FSP_FRAME_RSP_WRITE:	return "R Write";
	case FSP_FRAME_REQ_RENAME:	return "C Rename";
	case FSP_FRAME_RSP_RENAME:	return "R Rename";
	case FSP_FRAME_REQ_FINDFIRST:	return "C Find First";
	case FSP_FRAME_RSP_FINDFIRST:	return "R Find First";
	case FSP_FRAME_REQ_FINDNEXT:	return "C Find Next";
	case FSP_FRAME_RSP_FINDNEXT:	return "R Find Next";
	case FSP_FRAME_REQ_COMMIT:	return "C Commit File";
	case FSP_FRAME_RSP_COMMIT:	return "R Commit File";
	case FSP_FRAME_REQ_FILEINFO:	return "C File Information";
	case FSP_FRAME_RSP_FILEINFO:	return "R File Information";
	case FSP_FRAME_REQ_FILETIME:	return "C File Time";
	case FSP_FRAME_RSP_FILETIME:	return "R File Time";
	case FSP_FRAME_REQ_FSINFO:	return "C File System Information";
	case FSP_FRAME_RSP_FSINFO:	return "R File System Information";
	case FSP_FRAME_REQ_LOCK:	return "C Lock Record";
	case FSP_FRAME_RSP_LOCK:	return "R Lock Record";
	case FSP_FRAME_REQ_UNLOCK:	return "C Unlock Record";
	case FSP_FRAME_RSP_UNLOCK:	return "R Unlock Record";
	default:			return "Undecoded";
    }
} // GsFspCmdName

PUCHAR PacketSummaryGsFsp (PVOID Buffer, USHORT BufferLength)
{
    PFSP_REQ_OPEN FspOpenReq;
    PFSP_RSP_OPEN FspOpenRsp;
    PFSP_REQ_CLOSE FspCloseReq;
    PFSP_RSP_CLOSE FspCloseRsp;
    PFSP_REQ_CREATE FspCreateReq;
    PFSP_RSP_CREATE FspCreateRsp;
    PFSP_REQ_DELETE FspDeleteReq;
    PFSP_RSP_DELETE FspDeleteRsp;
    PFSP_REQ_READ FspReadReq;
    PFSP_RSP_READ FspReadRsp;
    PFSP_REQ_WRITE FspWriteReq;
    PFSP_RSP_WRITE FspWriteRsp;
    PFSP_REQ_RENAME FspRenameReq;
    PFSP_RSP_RENAME FspRenameRsp;
    PFSP_REQ_FINDFIRST FspFindFirstReq;
    PFSP_RSP_FINDFIRST FspFindFirstRsp;
    PFSP_REQ_FINDNEXT FspFindNextReq;
    PFSP_RSP_FINDNEXT FspFindNextRsp;
    PFSP_REQ_COMMIT FspCommitReq;
    PFSP_RSP_COMMIT FspCommitRsp;
    PFSP_REQ_FILEINFO FspFileInfoReq;
    PFSP_RSP_FILEINFO FspFileInfoRsp;
    PFSP_REQ_FILETIME FspFileTimeReq;
    PFSP_RSP_FILETIME FspFileTimeRsp;
    PFSP_REQ_FSINFO FspFsInfoReq;
    PFSP_RSP_FSINFO FspFsInfoRsp;
    PFSP_REQ_LOCK FspLockReq;
    PFSP_RSP_LOCK FspLockRsp;
    PFSP_REQ_UNLOCK FspUnlockReq;
    PFSP_RSP_UNLOCK FspUnlockRsp;
    PFSP_FRAME pheader;
    UCHAR *p, *t;

    pheader = (PFSP_FRAME)Buffer;

    if (!(DisplayFilter & FILTER_GSFSP)) {
	return NULL;			// we're not decoding GSFSP.
    }

    p = malloc (80);
    if (p == NULL) {
	return NULL;
    }

    if ((pheader->Signature != FSP_FRAME_SIGNATURE) || (BufferLength < 6)) {
	free (p);
	return NULL;
    }

    switch (pheader->Cmd) {
	case FSP_FRAME_REQ_OPEN:
	    FspOpenReq = (PFSP_REQ_OPEN)&(pheader->Operands);
	    sprintf (p, "FSP %s '%s'",
		     GsFspCmdName (pheader->Cmd),
		     FspOpenReq->FileName);
	    break;

	case FSP_FRAME_RSP_OPEN:
	    FspOpenRsp = (PFSP_RSP_OPEN)&(pheader->Operands);
	    sprintf (p, "FSP %s (%u)  Hndl=%u",
		     GsFspCmdName (pheader->Cmd),
		     FspOpenRsp->Status,
		     FspOpenRsp->Handle);
	    break;

	case FSP_FRAME_REQ_CREATE:
	    FspCreateReq = (PFSP_REQ_CREATE)&(pheader->Operands);
	    sprintf (p, "FSP %s '%s'",
		     GsFspCmdName (pheader->Cmd),
		     FspCreateReq->FileName);
	    break;

	case FSP_FRAME_RSP_CREATE:
	    FspCreateRsp = (PFSP_RSP_CREATE)&(pheader->Operands);
	    sprintf (p, "FSP %s (%u)  Hndl=%u",
		     GsFspCmdName (pheader->Cmd),
		     FspCreateRsp->Status,
		     FspCreateRsp->Handle);
	    break;

	case FSP_FRAME_REQ_CLOSE:
	    FspCloseReq = (PFSP_REQ_CLOSE)&(pheader->Operands);
	    sprintf (p, "FSP %s  Hndl=%u",
		     GsFspCmdName (pheader->Cmd),
		     FspCloseReq->Handle);
	    break;

	case FSP_FRAME_RSP_CLOSE:
	    FspCloseRsp = (PFSP_RSP_CLOSE)&(pheader->Operands);
	    sprintf (p, "FSP %s (%u)",
		     GsFspCmdName (pheader->Cmd),
		     FspCloseRsp->Status);
	    break;

	case FSP_FRAME_REQ_DELETE:
	    FspDeleteReq = (PFSP_REQ_DELETE)&(pheader->Operands);
	    sprintf (p, "FSP %s%s'%s'",
		     GsFspCmdName (pheader->Cmd),
		     (FspDeleteReq->IsDirectory ? " Directory " : " "),
		     FspDeleteReq->FileName);
	    break;

	case FSP_FRAME_RSP_DELETE:
	    FspDeleteRsp = (PFSP_RSP_DELETE)&(pheader->Operands);
	    sprintf (p, "FSP %s (%u)",
		     GsFspCmdName (pheader->Cmd),
		     FspDeleteRsp->Status);
	    break;

	case FSP_FRAME_REQ_READ:
	    FspReadReq = (PFSP_REQ_READ)&(pheader->Operands);
	    sprintf (p, "FSP %s %u bytes at %lu, Hndl=%u",
		     GsFspCmdName (pheader->Cmd),
		     FspReadReq->BytesToRead,
		     FspReadReq->Offset,
		     FspReadReq->Handle);
	    break;

	case FSP_FRAME_RSP_READ:
	    FspReadRsp = (PFSP_RSP_READ)&(pheader->Operands);
	    sprintf (p, "FSP %s (%u), %u bytes",
		     GsFspCmdName (pheader->Cmd),
		     FspReadRsp->Status,
		     FspReadRsp->BytesRead);
	    break;

	case FSP_FRAME_REQ_WRITE:
	    FspWriteReq = (PFSP_REQ_WRITE)&(pheader->Operands);
	    sprintf (p, "FSP %s %u bytes at %lu, Hndl=%u",
		     GsFspCmdName (pheader->Cmd),
		     FspWriteReq->BytesToWrite,
		     FspWriteReq->Offset,
		     FspWriteReq->Handle);
	    break;

	case FSP_FRAME_RSP_WRITE:
	    FspWriteRsp = (PFSP_RSP_WRITE)&(pheader->Operands);
	    sprintf (p, "FSP %s (%u), %u bytes",
		     GsFspCmdName (pheader->Cmd),
		     FspWriteRsp->Status,
		     FspWriteRsp->BytesWritten);
	    break;

	case FSP_FRAME_REQ_FINDFIRST:
	    FspFindFirstReq = (PFSP_REQ_FINDFIRST)&(pheader->Operands);
	    sprintf (p, "FSP %s '%s'",
		     GsFspCmdName (pheader->Cmd),
		     FspFindFirstReq->FileName);
	    break;

	case FSP_FRAME_RSP_FINDFIRST:
	    FspFindFirstRsp = (PFSP_RSP_FINDFIRST)&(pheader->Operands);
	    sprintf (p, "FSP %s (%u)  Hndl=%u",
		     GsFspCmdName (pheader->Cmd),
		     FspFindFirstRsp->Status,
		     FspFindFirstRsp->Handle);
	    break;

	case FSP_FRAME_REQ_FINDNEXT:
	    FspFindNextReq = (PFSP_REQ_FINDNEXT)&(pheader->Operands);
	    sprintf (p, "FSP %s  Hndl=%u",
		     GsFspCmdName (pheader->Cmd),
		     FspFindNextReq->Handle);
	    break;

	case FSP_FRAME_RSP_FINDNEXT:
	    FspFindNextRsp = (PFSP_RSP_FINDNEXT)&(pheader->Operands);
	    sprintf (p, "FSP %s (%u)",
		     GsFspCmdName (pheader->Cmd),
		     FspFindNextRsp->Status);
	    break;

	case FSP_FRAME_REQ_RENAME:
	    FspRenameReq = (PFSP_REQ_RENAME)&(pheader->Operands);
	case FSP_FRAME_RSP_RENAME:
	    FspRenameRsp = (PFSP_RSP_RENAME)&(pheader->Operands);
	case FSP_FRAME_REQ_COMMIT:
	    FspCommitReq = (PFSP_REQ_COMMIT)&(pheader->Operands);
	case FSP_FRAME_RSP_COMMIT:
	    FspCommitRsp = (PFSP_RSP_COMMIT)&(pheader->Operands);
	case FSP_FRAME_REQ_FILEINFO:
	    FspFileInfoReq = (PFSP_REQ_FILEINFO)&(pheader->Operands);
	case FSP_FRAME_RSP_FILEINFO:
	    FspFileInfoRsp = (PFSP_RSP_FILEINFO)&(pheader->Operands);
	case FSP_FRAME_REQ_FILETIME:
	    FspFileTimeReq = (PFSP_REQ_FILETIME)&(pheader->Operands);
	case FSP_FRAME_RSP_FILETIME:
	    FspFileTimeRsp = (PFSP_RSP_FILETIME)&(pheader->Operands);
	case FSP_FRAME_REQ_FSINFO:
	    FspFsInfoReq = (PFSP_REQ_FSINFO)&(pheader->Operands);
	case FSP_FRAME_RSP_FSINFO:
	    FspFsInfoRsp = (PFSP_RSP_FSINFO)&(pheader->Operands);
	case FSP_FRAME_REQ_LOCK:
	    FspLockReq = (PFSP_REQ_LOCK)&(pheader->Operands);
	case FSP_FRAME_RSP_LOCK:
	    FspLockRsp = (PFSP_RSP_LOCK)&(pheader->Operands);
	case FSP_FRAME_REQ_UNLOCK:
	    FspUnlockReq = (PFSP_REQ_UNLOCK)&(pheader->Operands);
	case FSP_FRAME_RSP_UNLOCK:
	    FspUnlockRsp = (PFSP_RSP_UNLOCK)&(pheader->Operands);
	case FSP_FRAME_REQ_NULL:
	case FSP_FRAME_RSP_NULL:
	    sprintf (p, "FSP %s", GsFspCmdName (pheader->Cmd));
	    break;

	default:
	    free (p);
	    return NULL;
    }
    return p;
} // PacketSummaryGsFsp

VOID GsFspTrailer (PDETREC *Ptr)
{
    AppendDetRec (Ptr, "FSP:");
} // GsFspTrailer

PDETREC PacketDetailGsFsp (PVOID Buffer, USHORT BufferLength)
{
    PFSP_REQ_OPEN FspOpenReq;
    PFSP_RSP_OPEN FspOpenRsp;
    PFSP_REQ_CLOSE FspCloseReq;
    PFSP_RSP_CLOSE FspCloseRsp;
    PFSP_REQ_CREATE FspCreateReq;
    PFSP_RSP_CREATE FspCreateRsp;
    PFSP_REQ_DELETE FspDeleteReq;
    PFSP_RSP_DELETE FspDeleteRsp;
    PFSP_REQ_READ FspReadReq;
    PFSP_RSP_READ FspReadRsp;
    PFSP_REQ_WRITE FspWriteReq;
    PFSP_RSP_WRITE FspWriteRsp;
    PFSP_REQ_RENAME FspRenameReq;
    PFSP_RSP_RENAME FspRenameRsp;
    PFSP_REQ_FINDFIRST FspFindFirstReq;
    PFSP_RSP_FINDFIRST FspFindFirstRsp;
    PFSP_REQ_FINDNEXT FspFindNextReq;
    PFSP_RSP_FINDNEXT FspFindNextRsp;
    PFSP_REQ_COMMIT FspCommitReq;
    PFSP_RSP_COMMIT FspCommitRsp;
    PFSP_REQ_FILEINFO FspFileInfoReq;
    PFSP_RSP_FILEINFO FspFileInfoRsp;
    PFSP_REQ_FILETIME FspFileTimeReq;
    PFSP_RSP_FILETIME FspFileTimeRsp;
    PFSP_REQ_FSINFO FspFsInfoReq;
    PFSP_RSP_FSINFO FspFsInfoRsp;
    PFSP_REQ_LOCK FspLockReq;
    PFSP_RSP_LOCK FspLockRsp;
    PFSP_REQ_UNLOCK FspUnlockReq;
    PFSP_RSP_UNLOCK FspUnlockRsp;
    PFSP_FRAME pheader;
    PDETREC r=NULL, q;

    pheader = (PFSP_FRAME)Buffer;

    if (!(DisplayFilter & FILTER_GSFSP)) {
	return NULL;			// we're not decoding GSFSP.
    }

    sprintf (ScratchBuf, "FSP:  ----- General Software FSP %s -----",
	     GsFspCmdName (pheader->Cmd));
    AppendHeader (&r, ScratchBuf);

    if ((pheader->Signature != FSP_FRAME_SIGNATURE) || (BufferLength < 6)) {
	return NULL;
    }

    sprintf (ScratchBuf, "FSP:  Command = 0x%04x  (%s)",
	     pheader->Cmd, GsFspCmdName (pheader->Cmd));
    AppendDetRec (&r, ScratchBuf);

    switch (pheader->Cmd) {
	case FSP_FRAME_REQ_OPEN:
	    FspOpenReq = (PFSP_REQ_OPEN)&(pheader->Operands);
	    sprintf (ScratchBuf, "FSP:  Filename = '%s'", FspOpenReq->FileName);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "FSP:  Flags = %04x", FspOpenReq->Flags);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case FSP_FRAME_RSP_OPEN:
	    FspOpenRsp = (PFSP_RSP_OPEN)&(pheader->Operands);
	    sprintf (ScratchBuf, "FSP:  Status = %u (0x%04x)",
		     FspOpenRsp->Status, FspOpenRsp->Status);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "FSP:  Handle = %u", FspOpenRsp->Handle);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case FSP_FRAME_REQ_CREATE:
	    FspCreateReq = (PFSP_REQ_CREATE)&(pheader->Operands);
	    if (FspCreateReq->IsDirectory) {
		sprintf (ScratchBuf, "FSP:  Directory = '%s'", FspCreateReq->FileName);
	    } else {
		sprintf (ScratchBuf, "FSP:  Filename = '%s'", FspCreateReq->FileName);
	    }
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "FSP:  Flags = %04x", FspCreateReq->Flags);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case FSP_FRAME_RSP_CREATE:
	    FspCreateRsp = (PFSP_RSP_CREATE)&(pheader->Operands);
	    sprintf (ScratchBuf, "FSP:  Status = %u (0x%04x)",
		     FspCreateRsp->Status, FspCreateRsp->Status);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "FSP:  Handle = %u", FspCreateRsp->Handle);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case FSP_FRAME_REQ_CLOSE:
	    FspCloseReq = (PFSP_REQ_CLOSE)&(pheader->Operands);
	    sprintf (ScratchBuf, "FSP:  Handle = %u", FspCloseReq->Handle);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case FSP_FRAME_RSP_CLOSE:
	    FspCloseRsp = (PFSP_RSP_CLOSE)&(pheader->Operands);
	    sprintf (ScratchBuf, "FSP:  Status = %u (0x%04x)",
		     FspCloseRsp->Status, FspCloseRsp->Status);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case FSP_FRAME_REQ_DELETE:
	    FspDeleteReq = (PFSP_REQ_DELETE)&(pheader->Operands);
	    if (FspDeleteReq->IsDirectory) {
		sprintf (ScratchBuf, "FSP:  Directory = '%s'", FspDeleteReq->FileName);
	    } else {
		sprintf (ScratchBuf, "FSP:  Filename = '%s'", FspDeleteReq->FileName);
	    }
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case FSP_FRAME_RSP_DELETE:
	    FspDeleteRsp = (PFSP_RSP_DELETE)&(pheader->Operands);
	    sprintf (ScratchBuf, "FSP:  Status = %u (0x%04x)",
		     FspDeleteRsp->Status, FspDeleteRsp->Status);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case FSP_FRAME_REQ_READ:
	    FspReadReq = (PFSP_REQ_READ)&(pheader->Operands);
	    sprintf (ScratchBuf, "FSP:  Handle = %u", FspReadReq->Handle);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "FSP:  Offset = %lu", FspReadReq->Offset);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "FSP:  Bytes to read = %u", FspReadReq->BytesToRead);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case FSP_FRAME_RSP_READ:
	    FspReadRsp = (PFSP_RSP_READ)&(pheader->Operands);
	    sprintf (ScratchBuf, "FSP:  Status = %u (0x%04x)",
		     FspReadRsp->Status, FspReadRsp->Status);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "FSP:  Bytes read = %u", FspReadRsp->BytesRead);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case FSP_FRAME_REQ_WRITE:
	    FspWriteReq = (PFSP_REQ_WRITE)&(pheader->Operands);
	    sprintf (ScratchBuf, "FSP:  Handle = %u", FspWriteReq->Handle);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "FSP:  Offset = %lu", FspWriteReq->Offset);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "FSP:  Bytes to write = %u", FspWriteReq->BytesToWrite);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case FSP_FRAME_RSP_WRITE:
	    FspWriteRsp = (PFSP_RSP_WRITE)&(pheader->Operands);
	    sprintf (ScratchBuf, "FSP:  Status = %u (0x%04x)",
		     FspWriteRsp->Status, FspWriteRsp->Status);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "FSP:  Bytes written = %u", FspWriteRsp->BytesWritten);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case FSP_FRAME_REQ_FINDFIRST:
	    FspFindFirstReq = (PFSP_REQ_FINDFIRST)&(pheader->Operands);
	    sprintf (ScratchBuf, "FSP:  Filename = '%s'", FspFindFirstReq->FileName);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "FSP:  Search flags = %04x", FspFindFirstReq->SearchAttributeFlags);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case FSP_FRAME_RSP_FINDFIRST:
	    FspFindFirstRsp = (PFSP_RSP_FINDFIRST)&(pheader->Operands);
	    sprintf (ScratchBuf, "FSP:  Status = %u (0x%04x)",
		     FspFindFirstRsp->Status, FspFindFirstRsp->Status);
	    AppendDetRec (&r, ScratchBuf);
	    sprintf (ScratchBuf, "FSP:  Handle = %u", FspFindFirstRsp->Handle);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case FSP_FRAME_REQ_FINDNEXT:
	    FspFindNextReq = (PFSP_REQ_FINDNEXT)&(pheader->Operands);
	    sprintf (ScratchBuf, "FSP:  Handle = %u", FspFindNextReq->Handle);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case FSP_FRAME_RSP_FINDNEXT:
	    FspFindNextRsp = (PFSP_RSP_FINDNEXT)&(pheader->Operands);
	    sprintf (ScratchBuf, "FSP:  Status = %u (0x%04x)",
		     FspFindNextRsp->Status, FspFindNextRsp->Status);
	    AppendDetRec (&r, ScratchBuf);
	    break;

	case FSP_FRAME_REQ_RENAME:
	    FspRenameReq = (PFSP_REQ_RENAME)&(pheader->Operands);
	case FSP_FRAME_RSP_RENAME:
	    FspRenameRsp = (PFSP_RSP_RENAME)&(pheader->Operands);
	case FSP_FRAME_REQ_COMMIT:
	    FspCommitReq = (PFSP_REQ_COMMIT)&(pheader->Operands);
	case FSP_FRAME_RSP_COMMIT:
	    FspCommitRsp = (PFSP_RSP_COMMIT)&(pheader->Operands);
	case FSP_FRAME_REQ_FILEINFO:
	    FspFileInfoReq = (PFSP_REQ_FILEINFO)&(pheader->Operands);
	case FSP_FRAME_RSP_FILEINFO:
	    FspFileInfoRsp = (PFSP_RSP_FILEINFO)&(pheader->Operands);
	case FSP_FRAME_REQ_FILETIME:
	    FspFileTimeReq = (PFSP_REQ_FILETIME)&(pheader->Operands);
	case FSP_FRAME_RSP_FILETIME:
	    FspFileTimeRsp = (PFSP_RSP_FILETIME)&(pheader->Operands);
	case FSP_FRAME_REQ_FSINFO:
	    FspFsInfoReq = (PFSP_REQ_FSINFO)&(pheader->Operands);
	case FSP_FRAME_RSP_FSINFO:
	    FspFsInfoRsp = (PFSP_RSP_FSINFO)&(pheader->Operands);
	case FSP_FRAME_REQ_LOCK:
	    FspLockReq = (PFSP_REQ_LOCK)&(pheader->Operands);
	case FSP_FRAME_RSP_LOCK:
	    FspLockRsp = (PFSP_RSP_LOCK)&(pheader->Operands);
	case FSP_FRAME_REQ_UNLOCK:
	    FspUnlockReq = (PFSP_REQ_UNLOCK)&(pheader->Operands);
	case FSP_FRAME_RSP_UNLOCK:
	    FspUnlockRsp = (PFSP_RSP_UNLOCK)&(pheader->Operands);
	case FSP_FRAME_REQ_NULL:
	case FSP_FRAME_RSP_NULL:
	default:			// nothing else to print.
	    return r;			// DO NOTHING.
    }
    return r;
} // PacketDetailGsFsp
