//
// MODULE NAME:  FORMAT.C.
//
// FUNCTIONAL DESCRIPTION.
//	This module handles output formatting for the protocol decoding
//	modules.  If you have general output formatting stuff, put it here.
//	Also, any address resolution stuff goes here.
//
// MODIFICATION HISTORY.
//	S. E. Jones	92/04/03.	Original.
//
// NOTICE:  Copyright (C) 1992-1993 General Software, Inc.  All rights reserved.
//

#define FORMAT_MODULE

#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <dos.h>
#include "..\inc\system.h"		// DOS operating system defns.
#include "..\cow\cow.h"                 // character-oriented windows.
#include "..\inc\ktypes.h"		// commonly-used types.
#include "analyzer.h"			// common stuff for all modules.

UCHAR far TempBuf [50];

static PDETREC LastDetRecGenerated;

ULONG far DlcVendorIds [] = {
    0x00000022L, 0x00000065L, 0x00000093L, 0x0000009fL, 0x000000a2L,
    0x000000a7L, 0x000000a9L, 0x000000b0L, 0x000000b3L, 0x000000c0L,
    0x000000c6L, 0x0010005aL, 0x00020701L, 0x00080017L, 0x00080036L,
    0x00080049L, 0x0008005aL, 0x00080067L, 0x000000aaL, 0x0000007aL,
    0x0000000cL, 0x0000001bL, 0x000000e2L, 0x0000aa00L, 0x0002608cL,
    0x00080003L, 0x00080008L, 0x0008000bL, 0x00080014L, 0x0008001eL,
    0x00080025L, 0x00080047L, 0x00080068L, 0x0008006eL, 0x00080089L,
    0x00080090L, 0x000000b7L, 0x00000089L, 0x0000000fL, 0x000000c8L,
    0x000000efL, 0x0000dd00L, 0x0002cf1fL, 0x00080005L, 0x00080009L,
    0x00080010L, 0x0008001aL, 0x00080020L, 0x00080028L, 0x00080039L,
    0x0008004cL, 0x00080069L, 0x0008004cL, 0x0008008bL, 0x00aa0003L,

    0x0000006bL, 0x0000002aL, 0x00000010L, 0x000000ddL, 0x0000805cL,
    0x0000dd01L, 0x00080002L, 0x00080007L, 0x0008000aL, 0x00080011L,
    0x0008001bL, 0x00080022L, 0x0008002bL, 0x00080041L, 0x0008004eL,
    0x0008006aL, 0x00080087L, 0x0008008dL, 0x00aa0004L,
    0L					// end of the table.
}; // DlcVendorIds

UCHAR * far DlcVendorNames [] = {
    "VisTec",				// Visual Technology.
    "NwkGnl",				// Network General.
    "Prteon",				// Proteon.
    "Amrstr",				// Ameristar Technology.
    "Wllflt",				// Wellfleet.
    "NCD",				// Network Computing Devices.
    "NSC",				// Network Systems Corp.
    "RND",				// RAD Network Devices.
    "Cimlin",				// CIMlink.
    "WstDig",				// Western Digital.
    "HP-Int",				// HP Intelligent Networks.
    "IBM",				// big blue.
    "Intrln",				// Interlan.
    "NSC",				// Network Systems Corp, again.
    "Intrgr",				// Intergraph.
    "Univtn",				// Univation.
    "IBM",				// big blue, only reversed.
    "ComDes",				// COM Design.

    "Xerox", "Ardent", "Cisco", "Novell", "Acer",
    "Intel", "3Com", "ACC", "BBN", "Unisys",
    "Exceln", "Apollo", "CDC", "MetPhr", "Sequnt",
    "Ridge", "Exceln", "Kinetx", "Retix", "Dove",
    "Cayman", "NeXT", "Altos", "Alantc", "U-B",
    "CMC", "Symblx", "H-P", "AT&T", "DG",
    "Sun", "TI", "Spider", "Encore", "SilGrf",
    "Vtalnk", "Pyramd", "DEC", "MIPS", "TRW",
    "Sytek", "Gould", "Agilis", "U-B", "Bridge",
    "Apple", "Nestar", "Tktrnx", "DG", "NBI",
    "DEC", "DCA", "BICC", "AT&T", "Xyplex",
    "Xyvisn", "DECnet",
    NULL
}; // DlcVendorNames

static UCHAR TrNetBios [6] = { 0xc0, 0x00, 0x00, 0x00, 0x00, 0x80 };
static UCHAR EnNetBios [6] = { 0x03, 0x00, 0x00, 0x00, 0x00, 0x01 };

//
// Routines in other modules.
//

extern UCHAR *TranslateName (UCHAR *SrcAddress);	// in NAME.C.

//
// Routines in this module.
//

USHORT Swap (USHORT Value)
{
    USHORT temp;
    temp = Value / 256;
    temp |= (Value % 256) * 256;
    return temp;
} // Swap

VOID MakeAsciiz (PUCHAR Dest, PUCHAR Src, USHORT Len)
{
    USHORT i;
    for (i=0; i<Len; i++) {
	*(Dest++) = *(Src++) & 0x7f;	// remove netware high bits.
    }
    *Dest = 0;				// add ASCIIZ terminator.
} // MakeAsciiz

ULONG SwapDword (ULONG Src)
{
    return ((Src & 0xff000000L) >> 24) |
	   ((Src & 0x00ff0000L) >> 8) |
	   ((Src & 0x0000ff00L) << 8) |
	   ((Src & 0x000000ffL) << 24);
} // SwapDword

USHORT NibbleToBinary (UCHAR Value)
{
    USHORT Result=0;

    //
    // This ugly routine transforms a 4-bit quantity into a 16-bit hex
    // number that contains only 1's and 0's.  It is intended to be used
    // to convert nibbles to a value that can be sprintf'd with %04x.
    //

    if (Value & 0x01) {
	Result |= 0x0001;
    }
    if (Value & 0x02) {
	Result |= 0x0010;
    }
    if (Value & 0x04) {
	Result |= 0x0100;
    }
    if (Value & 0x08) {
	Result |= 0x1000;
    }
    return Result;
} // NibbleToBinary

PUCHAR ResolveDlcAddress (PUCHAR DestBuf, PUCHAR SrcBuf)
{
    UCHAR *s, *p;
    USHORT i;
    ULONG Addr;
    BOOLEAN IsBcst=TRUE, IsNull=TRUE, IsTrNetBios=TRUE, IsEnNetBios=TRUE;

    s = SrcBuf;

    if (XlateAddresses) {

	//
	// If we can resolve this address in our name table, do that first.
	// This lets the user redefine BROADCAST, NETBIOS, etc.
	//

	p = TranslateName (SrcBuf);
	if (p != NULL) {
	    sprintf (DestBuf, "%-12s", p);
	    return DestBuf;
	}

	//
	// It isn't a user-defined name, so attempt to decode it here.
	//

	Addr = ((ULONG)*(s+0) << 16) |
	       ((ULONG)*(s+1) << 8) |
	       ((ULONG)*(s+2) << 0);

	for (i=0; i<6; i++) {
	    if (*(s+i) != 0xff) {
		IsBcst = FALSE;
	    }
	    if (*(s+i) != 0) {
		IsNull = FALSE;
	    }
	    if (*(s+i) != TrNetBios [i]) {
		IsTrNetBios = FALSE;
	    }
	    if (*(s+i) != EnNetBios [i]) {
		IsEnNetBios = FALSE;
	    }
	}
	if (IsBcst) {
	    strcpy (DestBuf, "BROADCAST   ");
	    return DestBuf;
	} else if (IsNull) {
	    strcpy (DestBuf, "NULL        ");
	    return DestBuf;
	} else if (IsEnNetBios || IsTrNetBios) {
	    strcpy (DestBuf, "NETBIOS     ");
	    return DestBuf;
	}

	for (i=0; i<999; i++) {
	    if (DlcVendorIds [i] == 0L) {
		break;
	    }
	    if (DlcVendorIds [i] == Addr) {
		sprintf (DestBuf, "%s%02x%02x%02x",
			 DlcVendorNames [i], *(s+3), *(s+4), *(s+5));
		strcat (DestBuf, "      "); // pad with blanks.
		DestBuf [12] = 0;	    // and truncate to exactly 12 chars.
		return DestBuf;
	    }
	}
    }

    //
    // Either we're not translating addresses, or the address we're
    // attempting to resolve is not translatable.  Use raw hex form.
    //

    sprintf (DestBuf, "%02x%02x%02x%02x%02x%02x",
	     *(s+0), *(s+1), *(s+2), *(s+3), *(s+4), *(s+5));
    return DestBuf;
} // ResolveDlcAddress

VOID DeallocateDetRecChain (Win)
    PDISPLAY_WINDOW Win;
{
    PDETREC p, q;

    if ((Win == NULL) || (Win->Win == NULL)) {
	return;
    }
    p = Win->Head;
    while (p != NULL) {
	q = p->Fwdlink;
	free (p);
	p = q;
    }
    Win->Head = NULL;
    Win->Top = NULL;
} // DeallocateDetRecChain

PDETREC AllocateDetRec (Buffer)
    UCHAR *Buffer;
{
    PDETREC p;

    p = (PDETREC)malloc (strlen (Buffer) + 1 + sizeof (DETREC));
    if (p == NULL) {
	return NULL;
    }
    strcpy (p->DescBuf, Buffer);
    p->Fwdlink = NULL;
    LastDetRecGenerated = p;		// save for "highest level only".
    return p;
} // AllocateDetRec

BOOLEAN AppendDetRec (Ptr, Buffer)
    PDETREC *Ptr;
    PUCHAR Buffer;
{
    PDETREC p, q;

    p = AllocateDetRec (Buffer);
    if (p == NULL) {
	return FALSE;
    }
    if (*Ptr == NULL) {
	*Ptr = p;
    } else {
	for (q=*Ptr; q->Fwdlink!=NULL; q=q->Fwdlink) ;
	q->Fwdlink = p;
    }
    return TRUE;
} // AppendDetRec

VOID AppendBlankDetRec (Ptr)
    PDETREC *Ptr;
{
    AppendDetRec (Ptr, " ");
} // AppendBlankDetRec

VOID AppendHeader (Ptr, Buffer)
    PDETREC *Ptr;
    PUCHAR Buffer;
{
    USHORT i;

    for (i=0; i<40; i++) {                      // copy component name.
	TempBuf [i] = Buffer [i];
	if (TempBuf [i] == ':') {
	    i++;				// keep colon.
	    break;
	}
    }
    TempBuf [i] = 0;				// zero byte terminate it.
    AppendDetRec (Ptr, Buffer);
    LastHeaderDetRec = LastDetRecGenerated;	// save for "highest level only".
    AppendDetRec (Ptr, TempBuf);
} // AppendHeader

VOID JoinDetRec (Ptr, Tail)
    PDETREC *Ptr;
    PDETREC Tail;
{
    PDETREC p;

    if (*Ptr == NULL) {
	*Ptr = Tail;
    } else {
	for (p=*Ptr; p->Fwdlink!=NULL; p=p->Fwdlink) ;
	p->Fwdlink = Tail;
    }
} // JoinDetRec

VOID AppendBitRecord (
    PDETREC *Ptr,
    UCHAR *Prefix,
    UCHAR *BitName,
    USHORT BitMask,
    USHORT Value)
{
    UCHAR *t;

    if (BitName != NULL) {
	switch (BitMask) {
	    case 0x0001: t = "%s      .... ...%c = %s"; break;
	    case 0x0002: t = "%s      .... ..%c. = %s"; break;
	    case 0x0004: t = "%s      .... .%c.. = %s"; break;
	    case 0x0008: t = "%s      .... %c... = %s"; break;
	    case 0x0010: t = "%s      ...%c .... = %s"; break;
	    case 0x0020: t = "%s      ..%c. .... = %s"; break;
	    case 0x0040: t = "%s      .%c.. .... = %s"; break;
	    case 0x0080: t = "%s      %c... .... = %s"; break;
	}
	sprintf (ScratchBuf, t, Prefix, (Value & BitMask) ? '1' : '0', BitName);
	AppendDetRec (Ptr, ScratchBuf);
    }
} // AppendBitRecord

VOID Append8BitMask (
    PDETREC *Ptr,
    UCHAR *Prefix,
    USHORT Value,
    UCHAR *FieldName,
    UCHAR *B0Name,
    UCHAR *B1Name,
    UCHAR *B2Name,
    UCHAR *B3Name,
    UCHAR *B4Name,
    UCHAR *B5Name,
    UCHAR *B6Name,
    UCHAR *B7Name)
{
    sprintf (ScratchBuf, "%s  %s = 0x%02x", Prefix, FieldName, Value);
    AppendDetRec (Ptr, ScratchBuf);

    AppendBitRecord (Ptr, Prefix, B0Name, 0x0001, Value);
    AppendBitRecord (Ptr, Prefix, B1Name, 0x0002, Value);
    AppendBitRecord (Ptr, Prefix, B2Name, 0x0004, Value);
    AppendBitRecord (Ptr, Prefix, B3Name, 0x0008, Value);
    AppendBitRecord (Ptr, Prefix, B4Name, 0x0010, Value);
    AppendBitRecord (Ptr, Prefix, B5Name, 0x0020, Value);
    AppendBitRecord (Ptr, Prefix, B6Name, 0x0040, Value);
    AppendBitRecord (Ptr, Prefix, B7Name, 0x0080, Value);
} // Append8BitMask
