//
// MODULE NAME:  FILESUPP.C.
//
// FUNCTIONAL DESCRIPTION.
//	This module handles file support for the analyzers.
//
// MODIFICATION HISTORY.
//	S. E. Jones	92/12/29.	2.0, original.
//
// NOTICE:  Copyright (C) 1992-1993 General Software, Inc.  All rights reserved.
//

#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <dos.h>
#include "..\inc\system.h"		// DOS operating system defns.
#include "..\cow\cow.h"                 // character-oriented windows.
#include "..\inc\ktypes.h"		// commonly-used types.
#include "analyzer.h"			// common stuff for all modules.

extern PCAPREC AllocateCapRec (USHORT BufferLength);
extern VOID ResetCaptureBuffer ();

VOID LoadFile ()
{
    struct find_t DirEntry;
    UCHAR FileName [20];
    FILE *DosFile;
    PSNIFHDR snifhdrp;
    PCAPREC cp;
    PWINDOW w;
    PLISTE le;
    PMENU m;
    USHORT i;
    UCHAR ch;

    strcpy (TmpStr, CaptureDirectory);
    if (strlen (TmpStr) > 0) {
	if (TmpStr [strlen (TmpStr)-1] != '\\') {
	    strcat (TmpStr, "\\");
	}
    }
    strcat (TmpStr, "*.ENC");

    if (_dos_findfirst (TmpStr, _A_NORMAL, &DirEntry) != 0) {
	PopupWarning ("No capture files present.");
	return;
    }
    m = MenuCreate ("Select Capture File", 40, 12, 30, 10);
    if (m == NULL) {
	return;
    }

    while (TRUE) {
	MenuItem (m, DirEntry.name, 0L);
	if (_dos_findnext (&DirEntry) != 0) {
	    break;
	}
    }
    MenuSort (m);			// sort the menu alphabetically.
    le = MenuSelect (m);		// select from the menu.
    if (le == NULL) {
	MenuDestroy (m);
	return;
    }

    strcpy (TmpStr, CaptureDirectory);
    if (strlen (TmpStr) > 0) {
	if (TmpStr [strlen (TmpStr)-1] != '\\') {
	    strcat (TmpStr, "\\");
	}
    }
    strcat (TmpStr, le->Title);
    if ((DosFile=fopen (TmpStr, "rb")) == NULL) {
	PopupWarning ("Unable to read capture file.");
	MenuDestroy (m);
	return;
    }

    ResetCaptureBuffer ();		// clear-out the capture buffer.
    CaptureBufferFull = FALSE;		// this buffer isn't full yet.
    CurrentSerialNo = 1;		// start with packet number 1.
    ReceivedPackets = 0;		// count # packets arrived.

    snifhdrp = malloc (sizeof (SNIFHDR));
    if (snifhdrp == NULL) {
	fclose (DosFile);
	MenuDestroy (m);
	return;
    }

    w = WinCreate ("Loading Capture File", 40, 12, 50, 5,
		   WINDOW_BORDER_SINGLE, NORMAL_PALETTE);
    if (w == NULL) {
	free (snifhdrp);
	fclose (DosFile);
	MenuDestroy (m);
	return;
    }

    //
    // Read a junk message up to the first 0x1a (^Z character) in the file.
    //

    while (fread (&ch, 1, 1, DosFile) == 1) {
	if (ch == 0x1a) {
	    break;
	}
    }

    while (TRUE) {

	//
	// Read the "Sniffer"-compatible record header that identifies
	// this record's type and length.
	//

	if (fread (snifhdrp, sizeof (SNIFHDR), 1, DosFile) != 1) {
	    break;
	}

	//
	// Based on the record type, process the record.
	//

	if (snifhdrp->Type == SNIFHDR_TYPE_EOF) {
	    WinWrite (w, "End of file", 15, 0, 999); // update window.
	    break;			// found EOF mark.
	} else if (snifhdrp->Type == SNIFHDR_TYPE_VERSION) {
	    for (i=0; i<snifhdrp->Length; i++) {
		fread (&ch, 1, 1, DosFile);
	    }
	    WinWrite (w, "Version", 15, 0, 999); // update window.
	} else if (snifhdrp->Type == SNIFHDR_TYPE_DATA) {
	    ReceivedPackets++;		// count incoming packets.
	    sprintf (TmpStr, "Reading Packet %u", ReceivedPackets);
	    WinWrite (w, TmpStr, 15, 0, 999); // update window.

	    fread (&SnifDataRec, sizeof (SnifDataRec), 1, DosFile);
	    cp = AllocateCapRec (SnifDataRec.Size);
	    if (cp == NULL) {		// if we couldn't get memory.
		PopupWarning ("Insufficient memory; some packets discarded.");
		break;
	    }
	    fread (cp->Buffer, SnifDataRec.Size, 1, DosFile);
	    cp->SerialNo = (USHORT)(ReceivedPackets & 0x0000ffffL);

	    cp->Timestamp = (((ULONG)SnifDataRec.LowTime * 100L) / 681L) +
			    ((ULONG)SnifDataRec.MidTime * 10000L) +
			    ((ULONG)SnifDataRec.HighTime * 36000000L);

	    cp->Flags = CAPREC_FLAGS_CAPTURED;
	    if (SnifDataRec.Size < SnifDataRec.TrueSize) {
		cp->Flags |= CAPREC_FLAGS_TRUNCATED;
	    }
	    cp->Length = SnifDataRec.Size;
	} else {			// unknown record type.
	    WinWrite (w, "Unknown record", 15, 0, 999); // update window.
	}
    }

    free (snifhdrp);
    fclose (DosFile);
    WinDestroy (w);			// we're done with the window.
    MenuDestroy (m);			// we're done with the menu.
} // LoadFile

VOID SaveFile ()
{
    PSCREEN Screen;
    PFIELD f_fn;
    PWINDOW w;
    ACTION a;
    FILE *DosFile;
    UCHAR *t, ch;
    PCAPREC cp;
    ULONG l1, l2;

    Screen = ScreenCreate ("Save Capture File", 40, 12, 40, 5);
    if (Screen == NULL) {
	return;
    }

    f_fn = FieldCreate ("Save Filename: ", FIELD_TYPE_STRING,
			 1, 0, 15+9, FIELD_FLAGS_EDITABLE);
    strcpy (f_fn->Data.String, "");
    ScreenAddField (Screen, f_fn);

    a = ScreenEdit (Screen);
    if (a != ACTION_SAVE) {
	ScreenDestroy (Screen);
	return;
    }
    strcpy (TmpStr, CaptureDirectory);
    if (strlen (TmpStr) > 0) {
	if (TmpStr [strlen (TmpStr)-1] != '\\') {
	    strcat (TmpStr, "\\");
	}
    }
    strcat (TmpStr, f_fn->Data.String);
    strupr (TmpStr);
    if (strchr (TmpStr, '.') == NULL) {
	strcat (TmpStr, ".ENC");
    }

    if ((DosFile=fopen (TmpStr, "wb")) == NULL) {
	PopupWarning ("Unable to create capture file.");
	ScreenDestroy (Screen);
	return;
    }

    w = WinCreate ("Saving Capture File", 40, 12, 50, 5,
		   WINDOW_BORDER_SINGLE, NORMAL_PALETTE);
    if (w == NULL) {
	fclose (DosFile);
	ScreenDestroy (Screen);
	return;
    }

    //
    // Write the standard junk ASCII message to the file first.
    // Terminate the message with a ^Z character.
    //

    t = "TRSNIFF data    ";
    fwrite (t, strlen (t), 1, DosFile);
    ch = 0x1a;				// terminate with ^Z (MSDOS EOF).
    fwrite (&ch, 1, 1, DosFile);

    //
    // Write-out a VERSION record to make those sniffers happy.
    //

    SnifHeader.Type = SNIFHDR_TYPE_VERSION; // version record.
    SnifHeader.Length = sizeof (SNIFVER);
    SnifHeader.Reserved = 0;
    fwrite (&SnifHeader, sizeof (SNIFHDR), 1, DosFile);
    SnifVersion.MajVer = 1;
    SnifVersion.MinVer = 0;		// typically 16.
    SnifVersion.Time = 0;
    SnifVersion.Date = 0;
    SnifVersion.Type = 4;
    SnifVersion.Network = 1;
    SnifVersion.Timeunit = 1;
    SnifVersion.Reserved [0] = 0;
    SnifVersion.Reserved [1] = 0;
    SnifVersion.Reserved [2] = 0;
    SnifVersion.ReservedByte = 0;
    fwrite (&SnifVersion, sizeof (SNIFVER), 1, DosFile);

    //
    // Write out the body of the sniffer file.
    //

    cp = CaptureBuffer;
    ReceivedPackets = 1;
    while (cp != NULL) {
	if (cp->Flags & CAPREC_FLAGS_CAPTURED) {

	    //
	    // Keep user informed about our status.
	    //

	    sprintf (TmpStr, "Saving Packet %u", ReceivedPackets++);
	    WinWrite (w, TmpStr, 15, 0, 999); // update window.

	    //
	    // Write-out sniffer-compatible record header.
	    //

	    SnifHeader.Type = SNIFHDR_TYPE_DATA; // frame data.
	    SnifHeader.Length = sizeof (SNIFDATA) + cp->Length;
	    SnifHeader.Reserved = 0;
	    fwrite (&SnifHeader, sizeof (SNIFHDR), 1, DosFile);

	    //
	    // This is a "frame data" record.
	    //

	    SnifDataRec.HighTime = (USHORT)(cp->Timestamp / 36000000L);
	    l1 = cp->Timestamp % 36000000L;
	    SnifDataRec.MidTime = (USHORT)(l1 / 10000L);
	    l2 = l1 % 10000L;
	    SnifDataRec.LowTime = (USHORT)((l2 * 681L) / 100L);

	    SnifDataRec.Size = cp->Length;
	    SnifDataRec.FrameStatusBits = 0;
	    SnifDataRec.Flags = 0;
	    SnifDataRec.TrueSize = cp->Length;
	    SnifDataRec.Reserved = 0;
	    if (fwrite (&SnifDataRec, sizeof (SNIFDATA), 1, DosFile) != 1) {
		PopupWarning ("Unable to save entire capture buffer.");
		break;
	    }

	    //
	    // Now for the data itself.
	    //

	    if (fwrite (cp->Buffer, cp->Length, 1, DosFile) != 1) {
		PopupWarning ("Unable to save entire capture buffer.");
		break;
	    }

	}
	cp = cp->Fwdlink;
    }

    //
    // Write-out an END-OF-FILE record to make those sniffers happy.
    //

    SnifHeader.Type = SNIFHDR_TYPE_EOF	; // EOF record.
    SnifHeader.Length = 0;
    SnifHeader.Reserved = 0;
    fwrite (&SnifHeader, sizeof (SNIFHDR), 1, DosFile);

    fclose (DosFile);
    WinDestroy (w);			// we're done with the window.
    ScreenDestroy (Screen);		// we're done with the screen.
} // SaveFile

VOID PrintData ()
{
    PSCREEN Screen;
    PFIELD f_fn;
    PWINDOW w;
    ACTION a;
    FILE *DosFile;
    PCAPREC cp;
    ULONG lasttime=0L, deltasec, deltams;
    UCHAR *s;

    Screen = ScreenCreate ("Print Capture Buffer", 40, 12, 40, 5);
    if (Screen == NULL) {
	return;
    }

    f_fn = FieldCreate ("Printer Name: ", FIELD_TYPE_STRING,
			 1, 0, 14+9, FIELD_FLAGS_EDITABLE);
    strcpy (f_fn->Data.String, Printer);
    ScreenAddField (Screen, f_fn);

    a = ScreenEdit (Screen);
    if (a != ACTION_SAVE) {
	ScreenDestroy (Screen);
	return;
    }
    strcpy (TmpStr, f_fn->Data.String);
    strupr (TmpStr);
    if (strchr (TmpStr, '.') == NULL) {
	strcat (TmpStr, ".PRN");
    }

    if ((DosFile=fopen (TmpStr, "wt")) == NULL) {
	PopupWarning ("Unable to open printer or output file.");
	ScreenDestroy (Screen);
	return;
    }

    w = WinCreate ("Printing Capture Buffer", 40, 12, 50, 5,
		   WINDOW_BORDER_SINGLE, NORMAL_PALETTE);
    if (w == NULL) {
	fclose (DosFile);
	ScreenDestroy (Screen);
	return;
    }

    cp = CaptureBuffer;
    lasttime = cp->Timestamp;
    ReceivedPackets = 1;

    while (cp != NULL) {
	if (cp->Flags & CAPREC_FLAGS_CAPTURED) {
	    sprintf (TmpStr, "Printing Packet %u", ReceivedPackets++);
	    WinWrite (w, TmpStr, 15, 0, 999); // update window.

	    //
	    // Print-out the summary line here.
	    //

	    deltasec = (cp->Timestamp-lasttime) / 10000L;
	    deltams = (cp->Timestamp-lasttime) % 10000L;
	    lasttime = cp->Timestamp;

	    sprintf (TmpStr, "%5u %4lu.%04lu  ", cp->SerialNo, deltasec, deltams);
	    CurrentCapRec = cp;      // for protocol interpreters.

	    s = PacketSummaryEthernet (cp->Buffer, cp->Length);
	    if (s != NULL) {
		strcat (TmpStr, s);
		free (s);
	    }
	    fprintf (DosFile, "%s\n", TmpStr);
	}
	cp = cp->Fwdlink;
    }

    fclose (DosFile);
    WinDestroy (w);			// we're done with the window.
    ScreenDestroy (Screen);		// we're done with the screen.
} // PrintData
