//
// MODULE NAME:  CAPTURE.C.
//
// FUNCTIONAL DESCRIPTION.
//	This module handles capture support for the analyzers.
//
// MODIFICATION HISTORY.
//	S. E. Jones	92/12/29.	2.0, original.
//
// NOTICE:  Copyright (C) 1992-1993 General Software, Inc.  All rights reserved.
//

#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <dos.h>
#include "..\inc\system.h"		// DOS operating system defns.
#include "..\cow\cow.h"                 // character-oriented windows.
#include "..\inc\ktypes.h"		// commonly-used types.
#include "analyzer.h"			// common stuff for all modules.

PWINDOW CaptureWin;

#define BARGRAPH_TOP		4	// first displayable line.
#define MAX_CAPTURE_TIME (5L)		// five seconds for demos.

//
// Externs in other modules.  Note that GetBufRtn and CopyCompleteRtn
// are incorrectly prototyped to allow correct compilation.  The actual
// call to these routines is from an assembly language routine in the
// packet drivers, so it doesn't matter anyway.
//

extern VOID GetBufRtn ();		// in ETHERPRO.C.
extern VOID CopyCompleteRtn ();         // in ETHERPRO.C.
extern VOID ResetCaptureBuffer ();
extern VOID DosGetTime (
    USHORT *StartHour,
    USHORT *StartMin,
    USHORT *StartSec,
    USHORT *Hundredths);

#ifdef ALARMS
extern VOID CheckAlarms ();		// in ALARMS.C.
#endif

#ifdef TRIGGERS
extern VOID CheckTimeTrigger (USHORT Hours, USHORT Minutes, USHORT Seconds);
#endif

static UCHAR * far CaptureMenuHelpText [] = {
    "                      Help for Capture Menu",
    " ",
    "The Capture menu provides several ways to capture data from your",
    "LAN.  All of the methods provide basic data capture, subject to",
    "the options you select in the 'Options and Setup' screen in the",
    "main menu.  However, each capture method provides a different",
    "screen that shows LAN activity in real-time.",
    " ",
    "To begin capturing data, position the scroll bar over the type",
    "of capture you wish to perform, and then press ENTER.  Capturing",
    "of LAN traffic will begin immediately.  If selected in the main",
    "options menu, the speaker will click faintly each time a packet",
    "arrives, so you can easily hear the traffic density on the LAN",
    "concurrently with your real-time display.",
    " ",
#ifdef TRIGGERS
    "If you have defined a capture trigger in the 'Trigger' menu, then",
    "the real-time display will start immediately, but traffic will",
    "not be captured into the buffer until one of the triggers is",
    "activated (either the time-of-day trigger, the data trigger, or",
    "an alarm trigger).",
    " ",
#endif
    "----- Network Utilization Screen -----",
    " ",
    "Shows basic network usage in packets per second and kilobytes",
    "per second, both in numeric and bargraph form.  This screen is",
    "a good, basic, display that shows whether a LAN is being under-",
    "or over-utilized.",
    " ",
    "----- Protocol Suites Screen -----",
    " ",
    "Breaks-down incoming packets by protocol family, such as TCP/IP,",
    "SNA, ISO, LLC, IPX, SMB, NCP, and so on, and displays real-time",
    "bargraphs for each protocol family showing number of requests/sec.",
    " ",
    "----- File Service Request Types Screen -----",
    " ",
    "Decodes incoming file-sharing requests from multiple file sharing",
    "protocols simultaneously, and displays the number of requests/sec",
    "by request type, such as open file, close file, read, write, etc.",
    " ",
    "----- DLC Packet Types Screen -----",
    " ",
    "A very basic display that shows the number of unicast (directed),",
    "multicast, and broadcast packets arriving per second in bargraphs.",
    " ",
#ifdef NODE_PAIR_CAPTURE
    "----- Traffic Between Nodes Screen -----",
    " ",
    "Sorts-out incoming traffic into a list of source/destination pairs",
    "so that each conversation on the LAN can be analyzed.",
    " ",
#endif
#ifdef FILESHARINGERROR_CAPTURE
    "----- File Sharing Errors Screen -----",
    " ",
    "Similar to the file sharing requests screen, this display analyzes",
    "the return codes in file sharing response packets to determine",
    "the number of requests that are failing.  This display makes it",
    "easy to see when a server is having difficulty servicing requests,",
    "often before it goes down completely.",
    " ",
#endif
#ifdef SKYLINE_CAPTURE
    "----- Utilization Skyline Screen -----",
    " ",
    "Displays network utilization in kilobytes per second in skyline",
    "format so that a scrolling history can be seen on the display.",
    " ",
#endif
#ifdef HISTOGRAM_CAPTURE
    "----- 24hr Kb/Sec Histogram Screen -----",
    " ",
    "A continuous, 24-hour period histogram showing average traffic",
    "density at half-hour intervals.  This capture mode is useful for",
    "determining when your LAN is least- and most-used during the day.",
    " ",
    "----- 24hr Login Histogram Screen -----",
    " ",
    "A continuous, 24-hour period histogram showing the number of",
    "logins and logouts at half-hour intervals.  This capture mode",
    "shows on the average when users gain access to your LAN.",
    " ",
    "----- 24hr #users Histogram Screen -----",
    " ",
    "A continuous, 24-hour period histogram showing the average",
    "number of source addresses present during half-hour intervals.",
    "This capture mode shows on the average the number of stations",
    "actually transmitting data on the LAN.",
    " ",
#endif
    NULL				// end of help text symbol.
}; // CaptureMenuHelpText

//
// Routines in this module.
//

static VOID CaptureMenuHelpRtn (List)
    PLIST List;
{
    PopupHelp (CaptureMenuHelpText);
} // CaptureMenuHelpRtn

VOID CountProtocolType (USHORT ProtocolIndex)
{
    ProtocolTypes [ProtocolIndex]++;
    ProtocolTypesPerSec [ProtocolIndex]++;
} // CountProtocolType

VOID CountFileIoType (USHORT FileIoIndex)
{
    FileIoTypes [FileIoIndex]++;
    FileIoTypesPerSec [FileIoIndex]++;
} // CountFileIoType

USHORT LongToPerf (perf)
    ULONG perf;
{
    USHORT len;

    if (perf > 50000L) {
	len = 50;
    } else if (perf > 20000L) {
	len = 45 + (USHORT)((perf-20000L) / ((50000L-20000L)/5L));
    } else if (perf > 10000L) {
	len = 40 + (USHORT)((perf-10000L) / ((20000L-10000L)/5L));
    } else if (perf > 5000L) {
	len = 35 + (USHORT)((perf-5000L) / ((10000L-5000L)/5L));
    } else if (perf > 2000L) {
	len = 30 + (USHORT)((perf-2000L) / ((5000L-2000L)/5L));
    } else if (perf > 1000L) {
	len = 25 + (USHORT)((perf-1000L) / ((2000L-1000L)/5L));
    } else if (perf > 500L) {
	len = 20 + (USHORT)((perf-500L) / ((1000L-500L)/5L));
    } else if (perf > 200L) {
	len = 15 + (USHORT)((perf-200L) / ((500L-200L)/5L));
    } else if (perf > 100L) {
	len = 10 + (USHORT)((perf-100L) / ((200L-100L)/5L));
    } else if (perf > 50L) {
	len = 5 + (USHORT)((perf-50L) / ((100L-50L)/5L));
    } else {
	len = (USHORT)(perf / 10L);
    }
    return len;
} // LongToPerf

BOOLEAN InitWinGeneral ()
{

    CaptureWin = WinCreate ("Real-Time Packet Capture (Press ESC to stop)",
			    40, 16, 76, 16,
			    WINDOW_BORDER_SINGLE, NORMAL_PALETTE);
    if (CaptureWin == NULL) {
	return FALSE;
    }
    WinWrite (CaptureWin,
	      "NETWORK PERFORMANCE  0   50   100  200  500  1K   2K   5K   10K  20K  50K",
	      1, BARGRAPH_TOP-2, 999);
    WinWrite (CaptureWin,
	      "Statistic    Count   +----+----+----+----+----+----+----+----+----+----+",
	      1, BARGRAPH_TOP-1, 999);

    WinWrite (CaptureWin, "Pkts/sec", 1, 0+BARGRAPH_TOP, 10);
    WinWrite (CaptureWin, "KB/sec", 1, 1+BARGRAPH_TOP, 10);
    WinWrite (CaptureWin, "Coll/sec", 1, 2+BARGRAPH_TOP, 10);
    WinWrite (CaptureWin, "Trunc/sec", 1, 3+BARGRAPH_TOP, 10);
    WinWrite (CaptureWin, "Drop/sec", 1, 4+BARGRAPH_TOP, 10);
    WinWrite (CaptureWin, "Broadcast", 1, 5+BARGRAPH_TOP, 10);
    WinWrite (CaptureWin, "Captured", 1, 6+BARGRAPH_TOP, 10);
    WinWrite (CaptureWin, "Network", 1, 7+BARGRAPH_TOP, 10);
    return TRUE;
} // InitWinGeneral

BOOLEAN InitWinByAddrType ()
{

    CaptureWin = WinCreate ("Real-Time Packet Capture (Press ESC to stop)",
			    40, 12, 80, 25,
			    WINDOW_BORDER_SINGLE, NORMAL_PALETTE);
    if (CaptureWin == NULL) {
	return FALSE;
    }
    WinWrite (CaptureWin,
	      "PACKET ADDRESS TYPE  0   50   100  200  500  1K   2K   5K   10K  20K  50K",
	      1, BARGRAPH_TOP-2, 999);
    WinWrite (CaptureWin,
	      "STATISTICS   Count   +----+----+----+----+----+----+----+----+----+----+",
	      1, BARGRAPH_TOP-1, 999);

    WinWrite (CaptureWin, "UNICAST", 1, 0+BARGRAPH_TOP, 10);
    WinWrite (CaptureWin, "  Pkts/sec", 1, 1+BARGRAPH_TOP, 10);
    WinWrite (CaptureWin, "  Kb/sec", 1, 2+BARGRAPH_TOP, 10);

    WinWrite (CaptureWin, "MULTICAST", 1, 4+BARGRAPH_TOP, 10);
    WinWrite (CaptureWin, "  Pkts/sec", 1, 5+BARGRAPH_TOP, 10);
    WinWrite (CaptureWin, "  Kb/sec", 1, 6+BARGRAPH_TOP, 10);

    WinWrite (CaptureWin, "BROADCAST", 1, 8+BARGRAPH_TOP, 10);
    WinWrite (CaptureWin, "  Pkts/sec", 1, 9+BARGRAPH_TOP, 10);
    WinWrite (CaptureWin, "  Kb/sec", 1, 10+BARGRAPH_TOP, 10);
    return TRUE;
} // InitWinByAddrType

BOOLEAN InitWinByProtocol ()
{
    USHORT i, limit;
    CaptureWin = WinCreate ("Real-Time Packet Capture (Press ESC to stop)",
			    40, 12, 80, 25,
			    WINDOW_BORDER_SINGLE, NORMAL_PALETTE);
    if (CaptureWin == NULL) {
	return FALSE;
    }
    WinWrite (CaptureWin,
	      "PACKETS BY PROTOCOL  0   50   100  200  500  1K   2K   5K   10K  20K  50K",
	      1, BARGRAPH_TOP-2, 999);
    WinWrite (CaptureWin,
	      "STATISTICS   Count   +----+----+----+----+----+----+----+----+----+----+",
	      1, BARGRAPH_TOP-1, 999);

    limit = (MAX_FILTERS-1);
    if (limit > 17) {
	limit = 17;			// all the room we have on the screen.
    }
    for (i=0; i<limit; i++) {
	WinWrite (CaptureWin, FilterName [i], 1, i+BARGRAPH_TOP, 12);
    }
    return TRUE;
} // InitWinByProtocol

BOOLEAN InitWinByReqType ()
{
    USHORT i;
    CaptureWin = WinCreate ("Real-Time Packet Capture (Press ESC to stop)",
			    40, 12, 80, 25,
			    WINDOW_BORDER_SINGLE, NORMAL_PALETTE);
    if (CaptureWin == NULL) {
	return FALSE;
    }
    WinWrite (CaptureWin,
	      "PACKETS BY REQ.TYPE  0   50   100  200  500  1K   2K   5K   10K  20K  50K",
	      1, BARGRAPH_TOP-2, 999);
    WinWrite (CaptureWin,
	      "STATISTICS   Count   +----+----+----+----+----+----+----+----+----+----+",
	      1, BARGRAPH_TOP-1, 999);

    for (i=0; i<MAX_FILEIO_STATISTICS; i++) {
	WinWrite (CaptureWin, FileIoName [i], 1, i+BARGRAPH_TOP, 12);
    }
    return TRUE;
} // InitWinByReqType

VOID UpdateBarGraph (Value, PeakValueStorage, Lineno)
    ULONG Value;
    ULONG *PeakValueStorage;
    USHORT Lineno;
{
    USHORT len1, len2;

    len1 = LongToPerf (Value);
    if (*PeakValueStorage > Value) {
	len2 = LongToPerf (*PeakValueStorage) - len1;
    } else {
	*PeakValueStorage = Value;
	len2 = 0;
    }

    WinWrite (CaptureWin, SolidBlockBuffer, 22, Lineno+BARGRAPH_TOP, 1);
    WinWrite (CaptureWin, SolidBlockBuffer, 23, Lineno+BARGRAPH_TOP, len1);
    WinWrite (CaptureWin, LightBlockBuffer, 23+len1, Lineno+BARGRAPH_TOP, len2);
    WinWrite (CaptureWin, " ", 23+len1+len2, Lineno+BARGRAPH_TOP, 999);
} // UpdateBarGraph

VOID UpdateBarGraphPercent (Percent, Lineno)
    ULONG Percent;
    USHORT Lineno;
{
    USHORT len;

    len = (USHORT)Percent / 2;
    if (len > 50) {
	len = 50;
    }
    WinWrite (CaptureWin, SolidBlockBuffer, 22, Lineno+BARGRAPH_TOP, 1);
    WinWrite (CaptureWin, SolidBlockBuffer, 23, Lineno+BARGRAPH_TOP, len);
    WinWrite (CaptureWin, " ", 23+len, Lineno+BARGRAPH_TOP, 999);
} // UpdateBarGraphPercent

VOID DisplayGeneral (BOOLEAN Fast)
{
    if (Fast) {

	//
	// Update some counters in real time; they're inexpensive to display.
	//

	ltoa (ReceivedPackets, TmpStr, 10);	  // total packet count.
	WinWrite (CaptureWin, TmpStr, 14, 6+BARGRAPH_TOP, 7);

	ltoa (UnmatchedPackets, TmpStr, 10);	 // total packet count.
	WinWrite (CaptureWin, TmpStr, 14, 7+BARGRAPH_TOP, 7);

    } else {

	ltoa (ReceivedPacketsPerSecond, TmpStr, 10); // packets/second.
	WinWrite (CaptureWin, TmpStr, 14, 0+BARGRAPH_TOP, 7);

	ltoa (KbPerSecond, TmpStr, 10);             // kb/second.
	WinWrite (CaptureWin, TmpStr, 14, 1+BARGRAPH_TOP, 7);

	ltoa (UndersizedPackets, TmpStr, 10);	     // runts.
	WinWrite (CaptureWin, TmpStr, 14, 2+BARGRAPH_TOP, 7);

	ltoa (OversizedPackets, TmpStr, 10);	    // huge packets.
	WinWrite (CaptureWin, TmpStr, 14, 3+BARGRAPH_TOP, 7);

	PdGetStatistics (&StatBuffer, sizeof (STATISTICS_BUFFER));
	ltoa (StatBuffer.DroppedPackets, TmpStr, 10);
	WinWrite (CaptureWin, TmpStr, 14, 4+BARGRAPH_TOP, 7);

	ltoa (BroadcastPackets, TmpStr, 10);
	WinWrite (CaptureWin, TmpStr, 14, 5+BARGRAPH_TOP, 7);

	UpdateBarGraph (ReceivedPacketsPerSecond, &PeakReceivedPacketsPerSecond, 0);
	UpdateBarGraph (KbPerSecond, &PeakKbPerSecond, 1);
	UpdateBarGraph (UndersizedPackets, &PeakUndersizedPackets, 2);
	UpdateBarGraph (OversizedPackets, &PeakOversizedPackets, 3);
	UpdateBarGraph (StatBuffer.DroppedPackets, &PeakDroppedPackets, 4);
    }
} // DisplayGeneral

VOID DisplayByProtocol (BOOLEAN Fast)
{
    USHORT i, limit;

    limit = (MAX_FILTERS-1);
    if (limit > 17) {
	 limit = 17;		// all the room we have on the screen.
    }

    if (Fast) {

	//
	// Update packet arrival counters in real time.
	//

	for (i=0; i<limit; i++) {
	    ltoa (ProtocolTypes [i], TmpStr, 10); // total packet count.
	    WinWrite (CaptureWin, TmpStr, 14, i+BARGRAPH_TOP, 7);
	}

    } else {

	for (i=0; i<limit; i++) {
	    UpdateBarGraph (ProtocolTypesPerSec [i], &PeakProtocolTypesPerSec [i], i);
	    ProtocolTypesPerSec [i] = 0L;
	}
    }
} // DisplayByProtocol

VOID DisplayByAddrType (BOOLEAN Fast)
{
    if (!Fast) {
	ltoa (UnicastPackets, TmpStr, 10);
	WinWrite (CaptureWin, TmpStr, 14, 1+BARGRAPH_TOP, 7);
	UpdateBarGraph (UnicastPackets, &PeakUnicastPackets, 1);

	ltoa (MulticastPackets, TmpStr, 10);
	WinWrite (CaptureWin, TmpStr, 14, 5+BARGRAPH_TOP, 7);
	UpdateBarGraph (MulticastPackets, &PeakMulticastPackets, 5);

	ltoa (BroadcastPackets, TmpStr, 10);
	WinWrite (CaptureWin, TmpStr, 14, 9+BARGRAPH_TOP, 7);
	UpdateBarGraph (BroadcastPackets, &PeakBroadcastPackets, 9);

	ltoa ((UnicastKb+1023L)/1024L, TmpStr, 10);
	WinWrite (CaptureWin, TmpStr, 14, 2+BARGRAPH_TOP, 7);
	UpdateBarGraph ((UnicastKb+1023L)/1024L, &PeakUnicastKb, 2);

	ltoa ((MulticastKb+1023L)/1024L, TmpStr, 10);
	WinWrite (CaptureWin, TmpStr, 14, 6+BARGRAPH_TOP, 7);
	UpdateBarGraph ((MulticastKb+1023L)/1024L, &PeakMulticastKb, 6);

	ltoa ((BroadcastKb+1023L)/1024L, TmpStr, 10);
	WinWrite (CaptureWin, TmpStr, 14, 10+BARGRAPH_TOP, 7);
	UpdateBarGraph ((BroadcastKb+1023L)/1024L, &PeakBroadcastKb, 10);
    }
} // DisplayByAddrType

VOID DisplayByReqType (BOOLEAN Fast)
{
    USHORT i;

    if (Fast) {

	//
	// Update packet arrival counters in real time.
	//

	for (i=0; i<MAX_FILEIO_STATISTICS; i++) {
	    ltoa (FileIoTypes [i], TmpStr, 10); // total packet count.
	    WinWrite (CaptureWin, TmpStr, 14, i+BARGRAPH_TOP, 7);
	}

    } else {

	for (i=0; i<MAX_FILEIO_STATISTICS; i++) {
	    UpdateBarGraph (FileIoTypesPerSec [i], &PeakFileIoTypesPerSec [i], i);
	    FileIoTypesPerSec [i] = 0L;
	}
    }
} // DisplayByReqType

VOID CaptureTraffic ()
{
    USHORT i, rc, Handle;
    USHORT Hours, Minutes, Seconds, Hundredths, OldSecs;
    USHORT StartHour, StartMin, StartSec;
    ULONG StartTime, ElapsedTime, TotHours, TotMinutes, TotSecs;

#ifdef ALARMS
    BOOLEAN LogFileOpened=FALSE;
#endif

    ResetCaptureBuffer ();		// clear-out the capture buffer.
    CaptureBufferFull = FALSE;		// this buffer isn't full yet.
    CurrentSerialNo = 1;		// start with packet number 1.
    ReusedBuffer = FALSE;

#ifdef TRIGGERS
    if (TriggerOnData || TriggerOnTime || TriggerOnAlarm ||
	TriggerOnCollision || TriggerOnOversize) {
	CaptureSuspended = TRUE;
    } else {
	CaptureSuspended = FALSE;
    }
#endif

    //
    // Open the first network interface we can find for now.
    //

    PdSetEventHandler (0, GetBufRtn);		// register getbuf rtn.
    PdSetEventHandler (1, CopyCompleteRtn);	// register copy complete rtn.
    PdResetStatistics ();			// clear dropped pkts, etc.
    for (i=0; i<32; i++) {
	if ((rc=PdOpen (i, &Handle)) == 0) {
	    break;
	}
    }
    if (rc != 0) {
	PopupWarning ("A network packet driver could not be opened.");
	return;
    }

    //
    // Open up an event log to write to.
    //

#ifdef ALARMS
    if (LoggingEvents) {
	if ((LogFile=fopen (LogFileName, "wb")) == NULL) {
	    PopupWarning ("Unable to create log file.");
	    LoggingEvents = FALSE;
	} else {
	    LogFileOpened = TRUE;
	}
    }
#endif

    //
    // Put up real-time display.
    //

    UnmatchedPackets = 0L;
    ReceivedPackets = 0L;
    ReceivedPacketsPerSecond = 0L;
    UndersizedPackets = 0L;
    OversizedPackets = 0L;
    UnicastPackets = 0L;
    MulticastPackets = 0L;
    BroadcastPackets = 0L;
    UnicastKb = 0L;
    MulticastKb = 0L;
    BroadcastKb = 0L;
    KbPerSecond = 0L;
    ByteCount = 0;
    PeakReceivedPackets = 0L;
    PeakReceivedPacketsPerSecond = 0L;
    PeakUndersizedPackets = 0L;
    PeakOversizedPackets = 0L;
    PeakUnicastPackets = 0L;
    PeakMulticastPackets = 0L;
    PeakBroadcastPackets = 0L;
    PeakUnicastKb = 0L;
    PeakMulticastKb = 0L;
    PeakBroadcastKb = 0L;
    PeakKbPerSecond = 0L;
    for (i=0; i<MAX_FILTERS; i++) {
	ProtocolTypes [i] = 0L;         // packet counts by protocol types.
	ProtocolTypesPerSec [i] = 0L;	// packet counts/sec by protocol types.
	PeakProtocolTypesPerSec [i] = 0L;
    }
    for (i=0; i<MAX_FILEIO_STATISTICS; i++) {
	FileIoTypes [i] = 0L;		// packet counts by request type.
	FileIoTypesPerSec [i] = 0L;	// packet counts/sec by request type.
	PeakFileIoTypesPerSec [i] = 0L;
    }

    switch (CaptureDisplayType) {
	case DISPLAYTYPE_BY_PROTOCOL:
	    InitWinByProtocol ();
	    break;

	case DISPLAYTYPE_BY_ADDRTYPE:
	    InitWinByAddrType ();
	    break;

	case DISPLAYTYPE_BY_REQTYPE:
	    InitWinByReqType ();
	    break;

	case DISPLAYTYPE_GENERAL:
	default:
	    InitWinGeneral ();
    }

    //
    // Go promiscuous.
    //

    if (PdSetRcvMode (Handle, SETRCVMODE_PROMISCUOUS) != 0) {
	PopupWarning ("Promiscuous mode not supported; some packets may be lost.");
	if (PdSetRcvMode (Handle, SETRCVMODE_BROADCAST) != 0) {
	    PdSetRcvMode (Handle, SETRCVMODE_DIRECTED);
	}
    }

    DosGetTime (&StartHour, &StartMin, &StartSec, &Hundredths);
    StartTime = (ULONG)StartHour * 3600L +
		(ULONG)StartMin * 60L +
		(ULONG)StartSec;

    //
    // Spin in a loop, waiting for the user to press a key while we're
    // collecting packets.
    //

    while (!CaptureBufferFull) {
	if (KeyReady ()) {
	    break;
	}

	//
	// Handle alarms by calling the alarm module.  If an alarm is
	// raised, then either a popup is displayed, or capturing into
	// the buffer may begin.
	//

#ifdef ALARMS
	CheckAlarms ();
#endif

	//
	// Handle fast, real-time counters quickly.
	//

	switch (CaptureDisplayType) {
	    case DISPLAYTYPE_BY_PROTOCOL:
		DisplayByProtocol (TRUE);
		break;

	    case DISPLAYTYPE_BY_ADDRTYPE:
		DisplayByAddrType (TRUE);
		break;

	    case DISPLAYTYPE_BY_REQTYPE:
		DisplayByReqType (TRUE);
		break;

	    case DISPLAYTYPE_GENERAL:
	    default:
		DisplayGeneral (TRUE);
	}

	//
	// Update bar graphs every second as well.
	//

	if (!(i++ & 0x0003)) {
	    DosGetTime (&Hours, &Minutes, &OldSecs, &Hundredths);

	    if (OldSecs == Seconds) {
		continue;
	    }
	    Seconds = OldSecs;

	    //
	    // Compute elapsed time and display in window.
	    //

	    ElapsedTime = (ULONG)Hours * 3600L +
			  (ULONG)Minutes * 60L +
			  (ULONG)Seconds - StartTime;

	    //
	    // A demo times-out after a few seconds of interaction
	    // with the network.
	    //

#ifdef DEMO
	    if (ElapsedTime > MAX_CAPTURE_TIME) { // a few seconds.
		break;				// break-out now.
	    }
#endif

	    TotSecs = ElapsedTime % 60L;
	    ElapsedTime /= 60L;
	    TotMinutes = ElapsedTime % 60L;
	    TotHours = ElapsedTime / 60L;

	    sprintf (TmpStr, "  Elapsed Capture Time %02ld:%02ld:%02ld  ",
		     TotHours, TotMinutes, TotSecs);
	    WinWrite (CaptureWin, TmpStr, (74/2)-(strlen (TmpStr)/2), 0, 50);

	    //
	    // Check the time trigger to see if it is going off.
	    //

#ifdef TRIGGERS
	    if (TriggerOnTime) {
		CheckTimeTrigger (Hours, Minutes, Seconds);
	    }
#endif

	    //
	    // Output once-per-second counters and bargraphs.
	    //

	    switch (CaptureDisplayType) {
		case DISPLAYTYPE_BY_PROTOCOL:
		    DisplayByProtocol (FALSE);
		    break;

		case DISPLAYTYPE_BY_ADDRTYPE:
		    DisplayByAddrType (FALSE);
		    break;

		case DISPLAYTYPE_BY_REQTYPE:
		    DisplayByReqType (FALSE);
		    break;

		case DISPLAYTYPE_GENERAL:
		default:
		    DisplayGeneral (FALSE);
	    }

	    //
	    // Reset per-second counters.
	    //

	    ReceivedPacketsPerSecond = 0L;
	    KbPerSecond = 0L;
	    UndersizedPackets = 0L;
	    OversizedPackets = 0L;
	    StatBuffer.DroppedPackets = 0L;
	    UnicastPackets = 0L;
	    MulticastPackets = 0L;
	    BroadcastPackets = 0L;
	    UnicastKb = 0L;
	    MulticastKb = 0L;
	    BroadcastKb = 0L;
	    ByteCount = 0;
	    for (i=0; i<MAX_FILTERS-1; i++) {
		ProtocolTypesPerSec [i] = 0L;
	    }
	    for (i=0; i<MAX_FILEIO_STATISTICS; i++) {
		FileIoTypesPerSec [i] = 0L;
	    }
	}
    }

#ifdef DEMO
    if (ElapsedTime > MAX_CAPTURE_TIME) {
	PopupWarning ("Demo capture time limit reached.  Try again for more.");
    }
#endif

    //
    // Close the logfile, if it exists.
    //

#ifdef ALARMS
    if (LogFileOpened) {
	fclose (LogFile);
    }
#endif

    //
    // Close our MAC handle.
    //

    PdSetRcvMode (Handle, SETRCVMODE_RCVR_OFF); // don't rcv any more packets.
    PdSetEventHandler (0, NULL);	// turn off GETBUF indications.
    PdSetEventHandler (1, NULL);	// turn off COPY COMPLETE indications.
    PdClose (Handle);			// close the MAC handle.

    WinDestroy (CaptureWin);
} // CaptureTraffic

VOID CaptureSimple ()
{
    CaptureDisplayType = DISPLAYTYPE_GENERAL;
    CaptureTraffic ();
} // CaptureSimple

VOID CaptureByProtocol ()
{
    CaptureDisplayType = DISPLAYTYPE_BY_PROTOCOL;
    CaptureTraffic ();
} // CaptureByProtocol

VOID CaptureByReqType ()
{
    CaptureDisplayType = DISPLAYTYPE_BY_REQTYPE;
    CaptureTraffic ();
} // CaptureByReqType

VOID CaptureByPacketType ()
{
    CaptureDisplayType = DISPLAYTYPE_BY_ADDRTYPE;
    CaptureTraffic ();
} // CaptureByPacketType

VOID CaptureMenuSelectRoutine (Selection)
    PLISTE Selection;
{
    switch ((USHORT)Selection->Value) {
	case 1: CaptureSimple (); break;
	case 2: CaptureByProtocol (); break;
	case 3: CaptureByReqType (); break;
	case 4: CaptureByPacketType (); break;
    }
} // CaptureMenuSelectRoutine

VOID CaptureMenu ()
{
    PLIST p;
    ACTION a;
    USHORT nlines=8;

#ifdef NODE_PAIR_CAPTURE
    nlines++;
#endif

#ifdef SKYLINE_CAPTURE
    nlines++;
#endif

#ifdef HISTOGRAM_CAPTURE
    nlines += 3;
#endif

#ifdef FILESHARINGERROR_CAPTURE
    nlines++;
#endif

    p = ListCreate ("Capture Display Options", 40, 13, 30, nlines, LIST_FLAGS_SELECT);
    p->SelectRtn = CaptureMenuSelectRoutine;
    p->HelpRtn = CaptureMenuHelpRtn;

    ListInsert (p, "Network Utilization", 1L);
    ListInsert (p, "Protocol Suites", 2L);
    ListInsert (p, "File Service Request Types", 3L);
    ListInsert (p, "DLC Packet Types", 4L);

#ifdef NODE_PAIR_CAPTURE
    ListInsert (p, "Traffic Between Nodes", 5L);
#endif

#ifdef SKYLINE_CAPTURE
    ListInsert (p, "Utilization Skyline", 6L);
#endif

#ifdef HISTOGRAM_CAPTURE
    ListInsert (p, "24hr Kb/Sec Histogram", 7L);
    ListInsert (p, "24hr Login Histogram", 8L);
    ListInsert (p, "24hr #users Histogram", 9L);
#endif

#ifdef FILESHARINGERROR_CAPTURE
    ListInsert (p, "File Sharing Errors", 10L);
#endif

    while (TRUE) {
	a = ListEdit (p);
	if (a == ACTION_ABORT) {
	    break;
	}
    }
    ListDestroy (p);
} // CaptureMenu
