//
// PROGRAM NAME:  ANALYZER.C.
//
// FUNCTIONAL DESCRIPTION.
//	This program is a network capture/analysis tool for managing
//	networks and debugging PC-based protocol stacks.
//
// MODIFICATION HISTORY.
//	S. E. Jones	92/03/16.	Original for The Snooper.
//	S. E. Jones	92/03/29.	Added load/save file.
//	S. E. Jones	92/04/21.	Snooper #1.7, added broadcast packet count.
//	S. E. Jones	92/05/09.	#1.2, added LANprobe structure.
//	S. E. Jones	92/07/03.	#1.4, fixed readtime & timewarps.
//	S. E. Jones	92/07/13.	#1.5, name change (LANprobe->Snooper).
//	S. E. Jones	92/07/20.	#1.6, fixed COW screen detection, did demo.
//	S. E. Jones	92/07/29.	#1.7, fixed MASM600A complaints.
//	S. E. Jones	92/11/24.	#1.9, added LLCTDI.
//	S. E. Jones	92/12/03.	#1.9, fixed PROTOCOL screen size limit.
//	S. E. Jones	93/01/01.	#2.0, new release.
//
// NOTICE:  Copyright (C) 1992-1993 General Software, Inc.  All rights reserved.
//
#define MAIN_PROGRAM	1		// for analyzer.h definitions.

#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <dos.h>
#include "..\inc\system.h"		// DOS operating system defns.
#include "..\cow\cow.h"                 // character-oriented windows.
#include "..\inc\ktypes.h"		// commonly-used types.
#include "analyzer.h"			// common stuff for all modules.

extern VOID LoadConfiguration ();	// in CONFIG.C.
extern VOID SaveConfiguration ();	// in CONFIG.C.
extern VOID DisplayOptions ();		// in CONFIG.C.
extern VOID DisplayData ();		// in DISPLAY.C.
extern VOID LoadFile ();		// in FILESUPP.C.
extern VOID SaveFile ();		// in FILESUPP.C.
extern VOID PrintData ();		// in FILESUPP.C.
extern VOID NameMenu ();		// in NAME.C.
extern VOID CaptureMenu ();		// in CAPTURE.C.

#ifdef ALARMS
extern VOID AlarmMenu ();		// in ALARMS.C.
#endif

#ifdef GENERATE_TRAFFIC
extern VOID GenMenu ();                 // in GENERATE.C.
#endif

#ifdef TRIGGERS
extern VOID TriggerMenu ();		// in TRIGGER.C.
extern VOID CheckDataTrigger (PVOID Buffer, USHORT Length); // in TRIGGER.C.
#endif

//
// Text for the "About this program" screen.
//

#ifdef ETHERPROBE
static UCHAR * far AboutText [] = {
    "        EtherProbe - The Real-Time Ethernet Network Analyzer",
    " ",
    "  GENERAL KEYBOARD FUNCTIONS:",
    " ",
    "  Help is available for all important menus and selections at any",
    "  time.  Simply press F2 to view help about any menu.  When editing",
    "  an alarm, trigger, or filter list, use F1 to save your edits, or",
    "  ESC to discard them.  Pressing ESC always exits the current menu",
    "  or mode and returns to the previous one.  Pressing ENTER selects",
    "  highlighted menu items.",
    " ",
    "  INSTALLING PACKET DRIVERS (REQUIRED FOR CAPTURE MODE):",
    " ",
    "  TO CAPTURE FROM THE NETWORK, you must load a PACKET DRIVER that",
    "  matches the Ethernet NIC you are using.  Packet drivers are supplied",
    "  with EtherProbe's software; however, you must determine DIP switch",
    "  settings for the NIC you will be using and initialize the packet",
    "  driver with these settings.  Once a packet driver is loaded, it",
    "  does not have to be reloaded until the machine is rebooted.  The",
    "  following are some examples for NE1000 and NE2000 card users.  The",
    "  '0x7e' field must be present and instructs the packet driver to",
    "  provide services on that interrupt (allowing it to be moved away",
    "  from any TSRs that might be using a special interrupt.)",
    " ",
    "      C> NE1000 0x7e 3 0x300       { IRQ=3, I/O port=300 hex }",
    "      C> NE2000 0x7e 4 0x330       { IRQ=4, I/O port=300 hex }",
    " ",
    "  You should be careful to NOT load a packet driver for a NIC that",
    "  you are using with other network software.  If you will be using",
    "  your computer as a network workstation AND ALSO with EtherProbe,",
    "  you must install a second NIC and load a packet driver for it alone.",
    NULL				// end of text marker.
}; // AboutText
#endif

#ifdef SNOOPER
static UCHAR * far AboutText [] = {
    "       The Snooper - The Real-Time Ethernet Protocol Analyzer",
    " ",
    "  GENERAL KEYBOARD FUNCTIONS:",
    " ",
    "  Help is available for all important menus and selections at any",
    "  time.  Simply press F2 to view help about any menu.  When editing",
    "  an alarm, trigger, or filter list, use F1 to save your edits, or",
    "  ESC to discard them.  Pressing ESC always exits the current menu",
    "  or mode and returns to the previous one.  Pressing ENTER selects",
    "  highlighted menu items.",
    " ",
    "  INSTALLING PACKET DRIVERS (REQUIRED FOR CAPTURE MODE):",
    " ",
    "  TO CAPTURE FROM THE NETWORK, you must load a PACKET DRIVER that",
    "  matches the Ethernet NIC you are using.  Packet drivers are supplied",
    "  with The Snooper's software; however, you must determine DIP switch",
    "  settings for the NIC you will be using and initialize the packet",
    "  driver with these settings.  Once a packet driver is loaded, it",
    "  does not have to be reloaded until the machine is rebooted.  The",
    "  following are some examples for NE1000 and NE2000 card users.  The",
    "  '0x7e' field must be present and instructs the packet driver to",
    "  provide services on that interrupt (allowing it to be moved away",
    "  from any TSRs that might be using a special interrupt.)",
    " ",
    "      C> NE1000 0x7e 3 0x300       { IRQ=3, I/O port=300 hex }",
    "      C> NE2000 0x7e 4 0x330       { IRQ=4, I/O port=300 hex }",
    " ",
    "  You should be careful to NOT load a packet driver for a NIC that",
    "  you are using with other network software.  If you will be using",
    "  your computer as a network workstation AND ALSO with The Snooper,",
    "  you must install a second NIC and load a packet driver for it alone.",
    NULL				// end of text marker.
}; // AboutText
#endif

static UCHAR * far MainMenuHelpText [] = {
    "                      Help for Main Menu",
    " ",
    "The Main menu allows you to control the analyzer through a set",
    "of various submenus.  For information about how the menu system",
    "works, select the 'Help' option by pressing 'H' to move the",
    "scroll bar (highlight) over the Help option, and then press ENTER.",
    "Once inside any submenu, you may press F2 for additional help.",
    " ",
    "                       MAIN MENU TOPICS",
    " ",
    "From the main menu, you may perform any analyzer function.",
    "The following is a list of the major functional categories.",
    " ",
    "----- Capture Network Traffic -----",
    " ",
    "To capture live traffic from your network, select this option.",
    "Even though they all save traffic to memory in the same way, there",
    "are several real-time capture screens that can show you what kind",
    "of traffic is on the LAN as it is being captured.  The Capture",
    "submenu has help that explains each of these screens.",
    " ",
    "Once traffic is captured, it is saved in memory.  The memory buffer",
    "(up to several thousand packets) may be displayed on the screen in",
    "any of three ways, or saved to a disk file for later use, or even",
    "printed in a report to your printer.",
    " ",
    "In order for the capture screens to work, you MUST load a packet",
    "driver as a TSR (Terminate and Stay Resident) program.  Packet",
    "drivers are supplied with the analyzer (source and binary forms)",
    "for commonly-used Ethernet adapters.  You may also use packet",
    "drivers from FTP Software or packet drivers that come with new",
    "adapters (see your dealer for available drivers).",
    " ",
    "----- Display Capture Buffer -----",
    " ",
    "The display mode offers three screens (controlled in the Options",
    "menu) that show captured traffic in summary, detailed, and hex",
    "forms.  When displaying, the analyzer automatically decodes recorded",
    "packets in memory using built-in protocol interpreters.  You can 4-way",
    "scroll through the packets in time order (as they were received from",
    "the LAN and examine every layer of the protocols, from physical to",
    "file-sharing protocol layers, and everything in-between.  If a packet",
    "is found that cannot be recognized, a hexadecimal display shows the",
    "packet contents.  Normally, such packets are at least identified by",
    "protocol family even if they are not fully decoded at the detail level.",
    " ",
    "----- Filter Traffic -----",
    " ",
    "The analyzer offers three ways to filter traffic; by station address",
    "and protocol type during capture, and by protocol type during display.",
    "This allows you to analyze traffic in various logical 'slices' on a",
    "busy LAN that contains packets that are not a part of the problem.",
    " ",
#ifdef GENERATE_TRAFFIC
    "----- Generate Traffic -----",
    " ",
    "You can cause the analyzer to automatically generate packets on the",
    "LAN according to your specifications, from one to thousands, and",
    "in rapid succession or slowly.  You can also dictate their target",
    "address, and even destination address, as well as packet contents.",
    "Caution: This feature must be used with care, as it is possible to",
    "confuse file servers and other workstations alike.",
    " ",
#endif
#ifdef ALARMS
    "----- Alarms -----",
    " ",
    "You can program multiple conditions, or alarms, to be constantly",
    "checked during capture mode, so that if some parameter moves out",
    "of your specified variances, it will trigger capturing, issue a",
    "popup, produce a tone on the console, or even log the event to",
    "a disk file for later inspection.  Alarms can automatically rearm",
    "themselves, or they can be programmed in a one-shot mode where",
    "they leave themselves disengaged after being triggered.  All of",
    "the programmed alarms are automatically saved to disk in the main",
    "Options and Setup menu.",
    " ",
#endif
#ifdef TRIGGERS
    "----- Triggers -----",
    " ",
    "The analyzer can be programmed to delay capture when in capture",
    "mode until any of three triggers are activated; at a certain time",
    "of day, or by any combination of alarms, or by packet contents.",
    "This allows the analyzer to be placed in capture mode around the",
    "clock waiting for unexpected or problematic events to occur.  When",
    "they do, an accurate capture of the events is kept.",
    " ",
#endif
    "----- Names -----",
    " ",
    "To provide more easily-readable displays, you can assign names up to",
    "12 characters in length to any Ethernet address, including station",
    "addresses, multicast addresses, and the broadcast address.  These",
    "names are automatically used by all display components that print",
    "an Ethernet address.  The name table can be saved and loaded to and",
    "from disk files, and an unlimited number of name tables can be used.",
    " ",
    "----- Options and Setup -----",
    " ",
    "This screen manages the options applicable to capture and display",
    "modes, and some additional options.  The selected options can be",
    "saved to a configuration file so that the user-defined options become",
    "the defaults when the analyzer runs again.",
    " ",
    "----- Load Capture File -----",
    " ",
    "The analyzer can load files containing previously-captured network",
    "traffic.  All capture files have an '.ENC' (EtherNet Capture)",
    "extension and are scanned for in a user-specified directory.",
    " ",
    "----- Save Capture File -----",
    " ",
    "You can save captured traffic to a disk file with this menu.  This",
    "allows later analysis or archival of captured traffic.",
    " ",
    "----- Print Capture Buffer -----",
    " ",
    "Captured traffic in the capture buffer can be formatted for output",
    "to your printer in this menu.  The default printer type is stored",
    "in the Options and Setup screen",
    " ",
    "----- Help -----",
    " ",
    "This screen provides information about the help system itself, and",
    "about how to use the windowing system.",
    " ",
    "----- Exit to DOS -----",
    " ",
    "To return to the operating system, use this main menu option.",
    NULL				// end of help text symbol.
}; // MainMenuHelpText

static UCHAR * far FilterMenuHelpText [] = {
    "                      Help for Filter Menu",
    " ",
    "The Filter menu provides three ways to selectively capture and",
    "display captured traffic on your LAN.  These filters may be used",
    "singly, in combination, or may be left unused.",
    " ",
    "To learn about each type of filter, select a menu item and press",
    "F2 for help once in the menu.",
    NULL				// end of help text symbol.
}; // FilterMenuHelpText

static UCHAR * far CapFilterMenuHelpText [] = {
    "              Help for Capture Filter Selection List",
    " ",
    "The Capture Filters list lets you select which protocols are",
    "to be captured when in capture mode.  This filter does NOT affect",
    "your display, only which packets will be captured when received.",
    "Only those packets matching one of the protocols in the current",
    "capture filters list will be accepted.  If you wish to include",
    "other protocols in the list, press the INS key when in this menu",
    "and select (from a list of available protocols) another protocol.",
    "If you wish to exclude protocols on the list, simply press DEL",
    "after positioning the scroll bar over the protocol to exclude.",
    "Excluded protocols are moved to the INS list, so that they can",
    "be restored with INS as necessary.",
    " ",
    "Once you have edited the capture filter list, it is necessary to",
    "press the F1 key to make your changes effective, or ESC to discard",
    "your changes.",
    NULL				// end of help text symbol.
}; // CapFilterMenuHelpText

static UCHAR * far ProtFilterMenuHelpText [] = {
    "                  Help for Display Filter List",
    " ",
    "While Capture Protocol Filters allow you to selectively capture",
    "packets of a specified type, the Display Filters allow you to",
    "CONTROL THE DEGREE TO WHICH PACKETS ARE DECODED DURING DISPLAY.",
    "For example, once you have captured traffic in capture mode and",
    "and have entered display mode on the analyzer, the formatting",
    "filters selected from this menu control whether decoding will",
    "stop at the IPX layer, or at the NCP layer.  Excluding NCP from",
    "the list would disable NCP decoding in the display, allowing you",
    "to see IPX information in the summary and detail areas.",
    " ",
    "Once you have edited the display filter list, it is necessary to",
    "press the F1 key to make your changes effective, or ESC to discard",
    "your changes.",
    NULL				// end of help text symbol.
}; // ProtFilterMenuHelpText

static UCHAR * far AddrMenuHelpText [] = {
    "                  Help for Address Filter List",
    " ",
    "Just as Capture Protocol Filters make the analyzer capture those",
    "packets matching a specific protocol, Address Filters make the",
    "analyzer choose only packets sent (1) from a specified Ethernet",
    "address; (2) to a specified Ethernet address; (3) from one address",
    "to another address; or (4) bidirectional traffic between two",
    "addresses.  If no address filters are defined, then address",
    "filtering is disabled.  If you specify one or more address filters,",
    "then only those packets matching one or more of those filters will",
    "be accepted.",
    " ",
    "To define a new filter, press the INS key, and a new filter",
    "definition record will be defined.  You may fill-in the source",
    "or destination or both addresses, and specify whether the filter",
    "should allow bidirectional traffic, or just unidirectional traffic.",
    "When you have fully defined your filter, press F1 to save it.",
    " ",
    "To delete an existing filter, position the scroll bar over the",
    "filter you wish to delete, and press the DEL key.",
    NULL				// end of help text symbol.
}; // AddrMenuHelpText

//
// Routines in this module.
//

static VOID FilterMenuHelpRtn (List)
    PLIST List;
{
    PopupHelp (FilterMenuHelpText);
} // FilterMenuHelpRtn

static VOID MainMenuHelpRtn (List)
    PLIST List;
{
    PopupHelp (MainMenuHelpText);
} // MainMenuHelpRtn

static VOID CapFilterMenuHelpRtn (List)
    PLIST List;
{
    PopupHelp (CapFilterMenuHelpText);
} // CapFilterMenuHelpRtn

static VOID ProtFilterMenuHelpRtn (List)
    PLIST List;
{
    PopupHelp (ProtFilterMenuHelpText);
} // ProtFilterMenuHelpRtn

static VOID AddrMenuHelpRtn (List)
    PLIST List;
{
    PopupHelp (AddrMenuHelpText);
} // AddrMenuHelpRtn

BOOLEAN IsStarlite ()
{
    union REGS regs;

    regs.h.ah = DOSGETVERSION;		// (AH) = get version function code.
    regs.h.al = 0;			// MS-DOS requires this.
    regs.h.bh = 0;			// for wild O/S's that don't know about this.
    intdos (&regs, &regs);		// (BH) = STARLITE_OEM.
    if (regs.h.al != 3) {
	return FALSE;			// not on 3.x system.
    }

    //
    // Note to OEM:  Your OEM account number (assigned by General
    // Software, Inc.) is returned by this function at this point
    // ONLY if BH=STARLITE_OEM.  In this case, your account number
    // is returned in register (CX).  Register (BL) is zero.
    //
    // It is your responsibility to change SYSINIT.ASM in your
    // adaptation so that it will return the correct OEM number.
    // If you do not change the code, it will return the value 0,
    // which is reserved for General Software, Inc.
    //

    return (regs.h.bh == STARLITE_OEM); // TRUE if running on STARLITE.
} // IsStarlite

STATUS DosGetTime (Hours, Minutes, Seconds, Hundredths)
    USHORT *Hours;
    USHORT *Minutes;
    USHORT *Seconds;
    USHORT *Hundredths;
{
    union REGS regs;

    regs.h.ah = DOSGETTIME;		// (AH) = get time function code.
    intdos (&regs, &regs);		// get the time.
    *Hours = regs.h.ch;                 // Hours = (0-23).
    *Minutes = regs.h.cl;		// Minutes=(0-59).
    *Seconds = regs.h.dh;		// Seconds=(0-59).
    *Hundredths = regs.h.dl;		// Hundredths=(0-99).
    return DOSERR_SUCCESS;
} // DosGetTime

STATUS DosAllocSeg (Paragraphs, BufPtr)
    USHORT Paragraphs;
    PVOID *BufPtr;
{
    ULONG Pointer;
    UCHAR *p;
    union REGS regs;
    struct SREGS sregs;

    regs.h.ah = DOSALLOCSEG;		// (AH) = allocate segment function code.
    regs.x.bx = Paragraphs;		// (BX) = # paragraphs to allocate.
    intdosx (&regs, &regs, &sregs);	// (AX) = newly-allocated segment.
    if (regs.x.cflag) {
	return regs.x.ax;		// probably no memory...
    }
    Pointer = (ULONG)(regs.x.ax) << 16; // tricky casting to satisfy MSC.
    p = (UCHAR *)Pointer;
    *BufPtr = p;
    return DOSERR_SUCCESS;
} /* DosAllocSeg */

STATUS DosFreeSeg (BufPtr)
    PVOID BufPtr;
{
    union REGS regs;
    struct SREGS sregs;

    regs.h.ah = DOSFREESEG;		// (AH) = free segment function code.
    sregs.es = FP_SEG (BufPtr);         // (ES) = segment address.
    intdosx (&regs, &regs, &sregs);	// destroy segment.
    return regs.x.ax;			// return status code.
} /* DosFreeSeg */

VOID DeallocateCaptureBuffer ()
{
    PBIGBUF b;

    while (BufferChain != NULL) {
	b = BufferChain;
	BufferChain = b->Fwdlink;
	DosFreeSeg (b);
    }
    BufferChain = NULL;
    CurrBuf = NULL;
    CaptureBuffer = NULL;
    LastPacket = NULL;
} // DeallocateCaptureBuffer

VOID AllocateCaptureBuffer ()
{
    USHORT npara;
    PBIGBUF b;
    UCHAR *save;

    //
    // Immediately free any capture buffer we had previously allocated.
    //

    DeallocateCaptureBuffer ();

    //
    // Allocate a 48kb area just to reserve an area immediately following
    // this program for MALLOC to work with the window system.	We release
    // it immediately after we're done doing the allocation.
    //

#if 0
    if (DosAllocSeg (((32768+16384) / 16), &save) != DOSERR_SUCCESS) {
	return;                         // unable to allocate.
    }
#else
    if ((save=malloc (32767)) == NULL) {
	return;
    }
#endif

    //
    // The secret to our success in truncating packets is to always let
    // the packet driver transfer up to MaxPacketSize number of bytes at
    // packet reception time.  AllocateCapRec will return an area that
    // can always hold the defined maximum packet size, even though the
    // next CAPREC may be allocated over the end of the last one.  This
    // scheme counts on multiple receive indications NOT happening at the
    // same time.
    //

    npara = (65530 / 16);		// allocate nearly-64K segments.
    while (DosAllocSeg (npara, &b) == DOSERR_SUCCESS) {
	if (b == NULL) {
	    break;
	}
	b->Fwdlink = BufferChain;	// link BIGBUFs together.
	BufferChain = b;
	b->NextFill = b->CaptureRecords;// point to 1st free slot/this BIGBUF.
	b->BigBufSize = (npara*16) - (sizeof (BIGBUF) + MaxPacketSize);
	b->BytesRemaining = b->BigBufSize;
    }

#if 0
    DosFreeSeg (save);			// release our save buffer.
#else
    free (save);
#endif

    CurrBuf = BufferChain;		// 1st buffer is one to start using.
} // AllocateCaptureBuffer

VOID ResetCaptureBuffer ()
{
    PBIGBUF p;

    CaptureBuffer = NULL;
    LastPacket = NULL;
    CurrBuf = BufferChain;		// 1st buffer is one to start using.

    for (p=CurrBuf; p!=NULL; p=p->Fwdlink) {
	p->BytesRemaining = p->BigBufSize;
	p->NextFill = p->CaptureRecords;
   }
} // ResetCaptureBuffer

PCAPREC AllocateCapRec (BufferLength)
    USHORT BufferLength;
{
    PCAPREC cp;
    PBIGBUF bp, bp1;

    if (BufferLength > LargestDlcPacket) {
	return NULL;			// can't store this packet in a BIGBUF.
    }

    if (CurrBuf == NULL) {
	return NULL;
    }

    while (TRUE) {
	// if (CurrBuf->BytesRemaining > BufferLength + sizeof (CAPREC)) {
	if (CurrBuf->BytesRemaining > LargestDlcPacket + sizeof (CAPREC)) {

	    //
	    // We can fit this packet into the current BIGBUF.
	    //

	    cp = (PCAPREC)CurrBuf->NextFill;
	    CurrBuf->NextFill += (sizeof (CAPREC) + BufferLength);
	    CurrBuf->BytesRemaining -= (sizeof (CAPREC) + BufferLength);

	    cp->Fwdlink = NULL;         // we have no successor yet.
	    cp->Baklink = LastPacket;	// point to our predecessor.
	    if (LastPacket == NULL) {	// was there a last packet to chain to?
		CaptureBuffer = cp;	// remember we're the first one.
	    } else {
		LastPacket->Fwdlink = cp; // if so, we are his successor.
	    }
	    LastPacket = cp;		// now we're the last-allocated packet.
	    return cp;			// we're done.
	}

	//
	// The current BIGBUF is full.	Try to chain to the next one.
	//

	CurrBuf = CurrBuf->Fwdlink;	// start using the next BIGBUF.
	if (CurrBuf != NULL) {		// if there is one.
	    CurrBuf->NextFill = CurrBuf->CaptureRecords;
	    CurrBuf->BytesRemaining = CurrBuf->BigBufSize;
	    continue;			// go try allocation in this BIGBUF.
	}

	//
	// We've exhausted the capture buffer.  If the user has set the
	// option to stop capturing when the buffer is full, deny access.
	//

	if (StopOnBufferFull) {
	    CaptureBufferFull = TRUE;	// tell main about this condition.
	    return NULL;
	}

	//
	// We ran off the end of the BIGBUF chain.  Reuse the first BIGBUF.
	// We have to be very tricky and remove the first BIGBUF from the
	// head of the chain, and append him to the END of the chain.  This
	// lets us recycle them in round-robin order.  Also don't forget to
	// make CaptureBuffer point to the first CAPREC in the BIGBUF that
	// BECOMES the first one in the chain.
	//

	bp = BufferChain;		// point to the 1st bigbuf in line.
	BufferChain = bp->Fwdlink;	// delink the 1st from the chain.
	for (bp1=BufferChain; bp1->Fwdlink!=NULL; bp1=bp1->Fwdlink) ;
	bp1->Fwdlink = bp;		// append us to the end of the chain.
	bp->Fwdlink = NULL;		// we have no successor.
	bp->NextFill = bp->CaptureRecords;
	bp->BytesRemaining = bp->BigBufSize;
	CurrBuf = bp;			// carve our CAPREC from this bigbuf.
	bp->Fwdlink = NULL;		// ending BIGBUF has no successor.

	//
	// Reset the CaptureBuffer pointer to point to the first CAPREC
	// in the second bigbuf (which just became the first bigbuf).
	//

	CaptureBuffer = (PCAPREC)BufferChain->CaptureRecords;
	CaptureBuffer->Baklink = NULL;	// he has no predecessor.
	continue;			// try to carve from this buffer.
    }
} // AllocateCapRec

PVOID GetBufRtn (RequestedBufferLength)
    USHORT RequestedBufferLength;
{
    UCHAR *p;
    PCAPREC cp;
    USHORT PktLen;
    ULONG TimeInTicks;

    UnmatchedPackets++;                 // count all network packets.

    //
    // Allocate a packet buffer.
    //

    PktLen = (RequestedBufferLength > MaxPacketSize) ?
	      MaxPacketSize : RequestedBufferLength;

    if ((cp=AllocateCapRec (PktLen)) == NULL) {
	return NULL;
    }

    //
    // Initialize the packet's flags.
    //

    cp->Flags = CAPREC_FLAGS_CAPTURED;
    if (RequestedBufferLength > MaxPacketSize) {
	cp->Flags |= CAPREC_FLAGS_TRUNCATED;
    }
    cp->Length = PktLen;
    cp->SerialNo = CurrentSerialNo++;

    //
    // This code handles a bug in the PC's 8253 timer where time can
    // actually flow backward for a small amount of time.  This is due
    // to interrupt latency on ISR 8 (IRQ0).
    //

    cp->Timestamp = readtime ();	// this is in 100us since midnite.
    if (cp->Baklink != NULL) {
	if (cp->Baklink->Timestamp > cp->Timestamp) {
	    cp->Timestamp = cp->Baklink->Timestamp + 10;
	}
    }

    p = cp->Buffer;			// point to data buffer area.
    return p;				// return pointer to buffer for copy.
} // GetBufRtn;

BOOLEAN PacketFilterProtocol (UCHAR *Buffer)
{
    //
    // We return TRUE if the packet should be accepted, and FALSE if
    // not.  The call-tree we initiate below follows the same rule.
    //

    if (CaptureFilter & FILTER_OTHER) {
	return TRUE;		// we're accepting all other packets.
    }
    return PacketFilterEthernet (Buffer, LastPacket->Length);
} // PacketFilterProtocol

VOID DumpCurrentPacket ()
{
    PCAPREC cp;

    UnmatchedPackets++;                 // count uncollected packets.

    //
    // Throw away our current CAPREC, repositioning the current BIGBUF's
    // internal pointers and values.  Then make the current CAPREC's
    // predecessor the current one.  We don't have to backup CurrBuf because
    // the AllocateCapRec routine simply starts allocating from whatever
    // the current buffer is, and we always deallocate the last packet
    // from the current buffer.  Hence, it will be fine for the next allocation.
    //

    if (LastPacket->Baklink == NULL) {	// we're the first pkt in the buffer.
	CurrBuf->NextFill = CurrBuf->CaptureRecords;
	CurrBuf->BytesRemaining = CurrBuf->BigBufSize;
	LastPacket = NULL;
	CaptureBuffer = NULL;
	return;                         // done.
    }

    cp = LastPacket->Baklink;		// our predecessor.
    cp->Fwdlink = NULL;                 // we are no longer his successor.
    CurrBuf->BytesRemaining += (sizeof (CAPREC) + LastPacket->Length);
    CurrBuf->NextFill -= (sizeof (CAPREC) + LastPacket->Length);
    LastPacket = cp;			// our predecessor is now the last one.
} // DumpCurrentPacket

BOOLEAN MatchAddr (UCHAR *Buffer1, UCHAR *Buffer2)
{
    USHORT i;

    for (i=0; i<6; i++) {
	if (Buffer1 [i] != Buffer2 [i]) {
	    return FALSE;		// compare failed, so no match.
	}
    }
    return TRUE;			// addresses match perfectly.
} // MatchAddr

VOID CopyCompleteRtn (PVOID Buffer)
{
    PADDRESS_FILTER f;
    BOOLEAN Match;
    UCHAR *p=Buffer;

    if (LastPacket->Length < 60) {
	UndersizedPackets++;		// count undersized packets (runts).
	if (TriggerOnCollision) {
	    CaptureSuspended = FALSE;	// enable capture now.
	}
    }

    //
    // If we are filtering on addresses, then only pass this packet if
    // its source and destination addresses are what we really want.
    //

    if (AddressFilterList != NULL) {
	Match = FALSE;
	for (f=AddressFilterList; f!=NULL; f=f->Fwdlink) {
	    if (f->Flags & ADDRESS_FILTER_LINK) {
		if (MatchAddr ((p+0), f->SrcAddress)) {
		   if (MatchAddr ((p+6), f->DestAddress)) {
		       Match = TRUE;
		       break;
		   }
		} else if (MatchAddr ((p+6), f->SrcAddress)) {
		   if (MatchAddr ((p+0), f->DestAddress)) {
		       Match = TRUE;
		       break;
		   }
		}
	    } else if (f->Flags & ADDRESS_FILTER_SOURCE) {
		if (MatchAddr ((p+6), f->SrcAddress)) {
		    Match = TRUE;
		    break;
		}
	    } else if (f->Flags & ADDRESS_FILTER_DEST) {
		if (MatchAddr ((p+0), f->DestAddress)) {
		    Match = TRUE;
		    break;
		}
	    }
	}
	if (!Match) {			// if we didn't find an address match,
	    DumpCurrentPacket ();	// then throw this packet away.
	    return;
	}
    }

    //
    // Filter on protocols.  If the frame we just copied cannot be
    // scanned by selected trigger protocols, then we must BACKUP
    // in the capture buffer and free the current buffer.  This is
    // not easy, but we simply don't have an option to do this in GetBufRtn
    // as the broken packet driver spec we're using doesn't have a
    // lookahead function.
    //

    if (PacketFilterProtocol (p)) {
	if (LastPacket->Flags & CAPREC_FLAGS_TRUNCATED) {
	    if (TriggerOnOversize) {
		CaptureSuspended = FALSE; // enable capture now.
	    }
	    OversizedPackets++;         // count oversized packets.
	}

	if (MatchAddr ((p+0), BcstAddr)) {
	    BroadcastPackets++;         // count broadcast packets.
	    BroadcastKb += LastPacket->Length;
	} else if (*(p+0) & 0x01) {
	    MulticastPackets++;         // count multicast packets.
	    MulticastKb += LastPacket->Length;
	} else {
	    UnicastPackets++;		// count unicast packets.
	    UnicastKb += LastPacket->Length;
	}

	//
	// If we're doing the "BY PROTOCOL" display, then we need
	// real-time statistics on the protocol families.  Classify
	// this packet by protocol.  Also count request types, but
	// only if we are displaying them in real time.
	//

	PacketCountEthernet (p, LastPacket->Length);
	PacketReqTypeEthernet (p, LastPacket->Length);

	//
	// Call the data trigger routine, because this may be a
	// packet that contains our data trigger.
	//

#ifdef TRIGGERS
	CheckDataTrigger (p, LastPacket->Length);
#endif

	//
	// If we are triggering on an alarm or other trigger, then don't
	// capture yet.  When the trigger is released, we begin capturing.
	//

	if (CaptureSuspended) {
	    DumpCurrentPacket ();
	    return;
	}

	//
	// Count this packet in received traffic statistics.
	//

	ReceivedPackets++;		// count received packets.
	ReceivedPacketsPerSecond++;	// count received pkts/second.
	ByteCount += LastPacket->Length;
	if (ByteCount > 1024) {
	    KbPerSecond += (ULONG)(ByteCount / 1024);
	    ByteCount %= 1024;
	}

	if (SpeakerOn) {
	    tick ();			// tell user that a packet has arrived.
	}
	return;
    }

    //
    // The protocol didn't match.  Throw away this packet.
    //

    DumpCurrentPacket ();
} // CopyCompleteRtn

PLISTE FilterInsertRoutine ()
{
    PMENU m;
    PLISTE e, f;
    USHORT i;

    m = MenuCreate ("Select Filter  [Help=F2]", 22, 15, 30, 11);

    for (i=0; i<MAX_FILTERS; i++) {
	if (!(TempFilter & FilterMask [i])) {
	    MenuItem (m, FilterName [i], FilterMask [i]);
	}
    }
    e = MenuSelect (m);
    if (e == NULL) {
	MenuDestroy (m);
	return NULL;
    }
    f = ListElementCreate (e->Title, e->Value);
    MenuDestroy (m);
    TempFilter |= f->Value;
    return f;				// insert this into the list.
} // FilterInsertRoutine

BOOLEAN FilterDeleteRoutine (ListElement)
    struct _LISTE *ListElement;
{
    TempFilter &= ~ListElement->Value;
    return TRUE;
} // FilterDeleteRoutine

VOID CaptureFilters ()
{
    PLIST p;
    USHORT i;

    TempFilter = CaptureFilter;

    p = ListCreate ("Edit Filters  [Help=F2]", 20, 13, 32, 11,
		    LIST_FLAGS_INSERT | LIST_FLAGS_DELETE);
    p->DeleteRtn = FilterDeleteRoutine;
    p->InsertRtn = FilterInsertRoutine;
    p->HelpRtn = CapFilterMenuHelpRtn;

    for (i=0; i<MAX_FILTERS; i++) {
	if (TempFilter & FilterMask [i]) {
	    ListInsert (p, FilterName [i], FilterMask [i]);
	}
    }

    if (ListEdit (p) == ACTION_SAVE) {
	CaptureFilter = TempFilter;
    }

    ListDestroy (p);
} // CaptureFilters

VOID DisplayFilters ()
{
    PLIST p;
    USHORT i;

    TempFilter = DisplayFilter;

    p = ListCreate ("Edit Filters  [Help=F2]", 20, 13, 32, 11,
		    LIST_FLAGS_INSERT | LIST_FLAGS_DELETE);
    p->DeleteRtn = FilterDeleteRoutine;
    p->InsertRtn = FilterInsertRoutine;
    p->HelpRtn = ProtFilterMenuHelpRtn;

    for (i=0; i<MAX_FILTERS; i++) {
	if (TempFilter & FilterMask [i]) {
	    ListInsert (p, FilterName [i], FilterMask [i]);
	}
    }

    if (ListEdit (p) == ACTION_SAVE) {
	DisplayFilter = TempFilter;
    }

    ListDestroy (p);
} // DisplayFilters

UCHAR *AddrFilterName (AddrFilter)
    PADDRESS_FILTER AddrFilter;
{
    UCHAR *s, *d;
    USHORT i, len;

    s = AddrFilter->SrcAddress;
    d = AddrFilter->DestAddress;
    if (AddrFilter->Flags & ADDRESS_FILTER_LINK) {
	sprintf (TmpStr,
		 "Link between %02x%02x%02x%02x%02x%02x and %02x%02x%02x%02x%02x%02x",
		 *(s+0), *(s+1), *(s+2), *(s+3), *(s+4), *(s+5),
		 *(d+0), *(d+1), *(d+2), *(d+3), *(d+4), *(d+5));
    } else if ((AddrFilter->Flags & ADDRESS_FILTER_SOURCE) &&
	       (AddrFilter->Flags & ADDRESS_FILTER_DEST)) {
	sprintf (TmpStr,
		 "From %02x%02x%02x%02x%02x%02x to %02x%02x%02x%02x%02x%02x",
		 *(s+0), *(s+1), *(s+2), *(s+3), *(s+4), *(s+5),
		 *(d+0), *(d+1), *(d+2), *(d+3), *(d+4), *(d+5));
    } else if (AddrFilter->Flags & ADDRESS_FILTER_SOURCE) {
	sprintf (TmpStr,
		 "From %02x%02x%02x%02x%02x%02x",
		 *(s+0), *(s+1), *(s+2), *(s+3), *(s+4), *(s+5));
    } else if (AddrFilter->Flags & ADDRESS_FILTER_DEST) {
	sprintf (TmpStr,
		 "To %02x%02x%02x%02x%02x%02x",
		 *(d+0), *(d+1), *(d+2), *(d+3), *(d+4), *(d+5));
    } else {
	strcpy (TmpStr, "Uninitialized filter");
    }

    //
    // Make each name the same length.
    //

    len = strlen (TmpStr);
    for (i=len; i<50; i++) {
	TmpStr [i] = ' ';
    }
    TmpStr [50] = 0;			// install trailing zero byte.
    return TmpStr;
} // AddrFilterName

UCHAR HexDigitToBinary (HexDigit)
    UCHAR HexDigit;
{
    if ((HexDigit >= (UCHAR)'0') && (HexDigit <= (UCHAR)'9')) {
	return (HexDigit - (UCHAR)'0');
    } else if ((HexDigit >= (UCHAR)'A') && (HexDigit <= (UCHAR)'F')) {
	return (HexDigit - (UCHAR)'A' + 10);
    } else if ((HexDigit >= (UCHAR)'a') && (HexDigit <= (UCHAR)'f')) {
	return (HexDigit - (UCHAR)'a' + 10);
    }
    return 0;
} // HexDigitToBinary

UCHAR *MakeHexString (UCHAR *HexAddress)
{
    UCHAR *d=HexAddress;
    sprintf (TmpStr, "%02x%02x%02x%02x%02x%02x",
	     *(d+0), *(d+1), *(d+2), *(d+3), *(d+4), *(d+5));
    return TmpStr;
} // MakeHexString

UCHAR NibbleToBin (UCHAR Ch)
{
    UCHAR v;
    v = toupper (Ch);
    if ((v >= '0') && (v <= '9')) {
	return (v - '0');
    } else {
	return (v - 'A' + 10);
    }
} // NibbleToBin

VOID MakeHexAddress (UCHAR *AddressBuffer, UCHAR *HexString)
{
    UCHAR *d=AddressBuffer;
    USHORT i;

    for (i=0; i<12; i+=2) {
	*(d++) = NibbleToBin (HexString [i])*16 + NibbleToBin (HexString [i+1]);
    }
} // MakeHexAddress

BOOLEAN EditAddrFilter (AddrFilter)
    PADDRESS_FILTER AddrFilter;
{
    ACTION a;
    PSCREEN Screen;
    PFIELD f_src, f_dest, f_link;

    Screen = ScreenCreate ("Edit Address Filter", 40, 12, 50, 7);
    if (Screen == NULL) {
	return FALSE;
    }

    f_dest = FieldCreate ("Destination Address: ", FIELD_TYPE_HEXSTRING,
			 2, 0, 34, FIELD_FLAGS_EDITABLE);
    if (AddrFilter->Flags & ADDRESS_FILTER_DEST) {
	strcpy (f_dest->Data.String, MakeHexString (AddrFilter->DestAddress));
    } else {
	strcpy (f_dest->Data.String, "");
    }
    ScreenAddField (Screen, f_dest);

    f_src = FieldCreate ("Source Address:      ", FIELD_TYPE_HEXSTRING,
			 2, 1, 34, FIELD_FLAGS_EDITABLE);
    if (AddrFilter->Flags & ADDRESS_FILTER_SOURCE) {
	strcpy (f_src->Data.String, MakeHexString (AddrFilter->SrcAddress));
    } else {
	strcpy (f_src->Data.String, "");
    }
    ScreenAddField (Screen, f_src);

    f_link = FieldCreate ("Bidirectional Link:  ", FIELD_TYPE_BOOLEAN,
			  2, 2, 24, FIELD_FLAGS_EDITABLE);
    f_link->Data.Boolean = (AddrFilter->Flags & ADDRESS_FILTER_LINK) ? TRUE : FALSE;
    ScreenAddField (Screen, f_link);

    a = ScreenEdit (Screen);

    if (a == ACTION_SAVE) {
	AddrFilter->Flags = 0;
	if (strlen (f_src->Data.String) == 12) {
	    AddrFilter->Flags |= ADDRESS_FILTER_SOURCE;
	    MakeHexAddress (AddrFilter->SrcAddress, f_src->Data.String);
	}
	if (strlen (f_dest->Data.String) == 12) {
	    AddrFilter->Flags |= ADDRESS_FILTER_DEST;
	    MakeHexAddress (AddrFilter->DestAddress, f_dest->Data.String);
	}

	if (f_link->Data.Boolean) {
	    AddrFilter->Flags |= ADDRESS_FILTER_LINK;
	}
    }
    ScreenDestroy (Screen);
    return (a==ACTION_SAVE) ? TRUE : FALSE;
} // EditAddrFilter

VOID AddrSelectRoutine (ListElement)
    PLISTE ListElement;
{
    PADDRESS_FILTER f;
    f = (PADDRESS_FILTER)ListElement->Value;
    EditAddrFilter (f);
    strcpy (ListElement->Title, AddrFilterName (f));
} // AddrSelectRoutine

PLISTE AddrInsertRoutine ()
{
    PLISTE p;
    USHORT i;
    UCHAR *Title;
    PADDRESS_FILTER f;

    f = malloc (sizeof (ADDRESS_FILTER));
    if (f == NULL) {
	return NULL;
    }

    f->Flags = 0;
    for (i=0; i<6; i++) {
	f->SrcAddress [i] = 0;
	f->DestAddress [i] = 0;
    }

    if (!EditAddrFilter (f)) {
	free (f);
	return NULL;
    }

    p = ListElementCreate (AddrFilterName (f), (ULONG)f);
    if (p != NULL) {
	f->Fwdlink = AddressFilterList;
	AddressFilterList = f;
    } else {
	free (f);
    }
    return p;
} // AddrInsertRoutine

BOOLEAN AddrDeleteRoutine (ListElement)
    PLISTE ListElement;
{
    PADDRESS_FILTER p, q, f;

    f = (PADDRESS_FILTER)(ListElement->Value);

    //
    // If it's the first one on the list, pop off the front of the list.
    //

    if (f == AddressFilterList) {
	AddressFilterList = f->Fwdlink;
	free (f);
	return TRUE;
    }

    //
    // Splice it out of the list.
    //

    for (p=AddressFilterList; p!=NULL; p=p->Fwdlink) {
	if (p->Fwdlink == f) {
	    p->Fwdlink = f->Fwdlink;	// splice it out of the list.
	    free (f);
	    return TRUE;
	}
    }
    return FALSE;			// it wasn't on the list.
} // AddrDeleteRoutine

VOID AddressFilters ()
{
    PADDRESS_FILTER f;
    PLIST p;
    USHORT i;

    TempFilter = CaptureFilter;

    p = ListCreate ("Edit Address Filters  [Help=F2]", 40, 12, 50, 8,
		    LIST_FLAGS_INSERT | LIST_FLAGS_DELETE | LIST_FLAGS_SELECT);
    p->DeleteRtn = AddrDeleteRoutine;
    p->InsertRtn = AddrInsertRoutine;
    p->SelectRtn = AddrSelectRoutine;
    p->HelpRtn = AddrMenuHelpRtn;

    for (f=AddressFilterList; f!=NULL; f=f->Fwdlink) {
	ListInsert (p, AddrFilterName (f), (ULONG)f);
    }

    ListEdit (p);			// updates filters w/no cancellation.
    ListDestroy (p);
} // AddressFilters

VOID ExitProgram ()
{
    UtilEnd ();
    exit (0);
} // ExitProgram

VOID AboutThisProgram ()
{
    PopupHelp (AboutText);
} // AboutThisProgram

VOID FilterMenuSelectRoutine (Selection)
    PLISTE Selection;
{
    switch ((USHORT)Selection->Value) {
	case 1: CaptureFilters (); break;
	case 2: DisplayFilters (); break;
	case 3: AddressFilters (); break;
    }
} // FilterMenuSelectRoutine

VOID FilterMenu ()
{
    PLIST p;
    ACTION a;

    p = ListCreate ("Filtering Options", 40, 12, 30, 7, LIST_FLAGS_SELECT);
    p->SelectRtn = FilterMenuSelectRoutine;
    p->HelpRtn = FilterMenuHelpRtn;

    ListInsert (p, "Capture Filters", 1L);
    ListInsert (p, "Formatting Filters", 2L);
    ListInsert (p, "Address Filters", 3L);
    while (TRUE) {
	a = ListEdit (p);
	if (a == ACTION_ABORT) {
	    break;
	}
    }
    ListDestroy (p);
} // FilterMenu

VOID MainMenuSelectRoutine (Selection)
    PLISTE Selection;
{
    switch ((USHORT)Selection->Value) {
	case 1: CaptureMenu (); break;
	case 2: DisplayData (); break;
	case 3: FilterMenu (); break;
#ifdef GENERATE_TRAFFIC
	case 4: GenMenu (); break;
#endif
#ifdef ALARMS
	case 5: AlarmMenu (); break;
#endif
#ifdef TRIGGERS
	case 6: TriggerMenu (); break;
#endif
	case 7: NameMenu (); break;
	case 8: DisplayOptions (); break;
	case 9: LoadFile (); break;
	case 10: SaveFile (); break;
	case 11: PrintData (); break;
	case 12: AboutThisProgram (); break;
	case 13: ExitProgram (); break;
    }
} // MainMenuSelectRoutine

VOID MainMenu ()
{
    PLIST p;
    ACTION a;
    USHORT nlines=14;

#ifdef GENERATE_TRAFFIC
    nlines++;
#endif

#ifdef ALARMS
    nlines++;
#endif

#ifdef TRIGGERS
    nlines++;
#endif

    p = ListCreate ("Main Options", 16, 13, 30, nlines, LIST_FLAGS_SELECT);
    p->SelectRtn = MainMenuSelectRoutine;
    p->HelpRtn = MainMenuHelpRtn;

    ListInsert (p, "Capture Network Traffic", 1L);
    ListInsert (p, "Display Capture Buffer",  2L);
    ListInsert (p, "Filtering Options", 3L);
#ifdef GENERATE_TRAFFIC
    ListInsert (p, "Generate Traffic", 4L);
#endif
#ifdef ALARMS
    ListInsert (p, "Alarms", 5L);
#endif
#ifdef TRIGGERS
    ListInsert (p, "Triggering Options", 6L);
#endif
    ListInsert (p, "Name Management", 7L);
    ListInsert (p, "Options & Setup", 8L);
    ListInsert (p, "Load Capture File", 9L);
    ListInsert (p, "Save Capture File", 10L);
    ListInsert (p, "Print Capture Buffer", 11L);
    ListInsert (p, "Help", 12L);
    ListInsert (p, "Exit to DOS", 13L);
    while (TRUE) {
	a = ListEdit (p);
	if (a == ACTION_ABORT) {
	    sprintf (ScratchBuf, "Exit %s", PRODUCT_NAME);
	    if (PopupYesOrNo (ScratchBuf, "Are you sure?")) {
		break;
	    }
	}
    }
    ListDestroy (p);
} // MainMenu

VOID InitData ()
{
    USHORT i;

    AllocateCaptureBuffer ();

    for (i=0; i<BLOCKLEN-1; i++) {
	SolidBlockBuffer [i] = 219;
    }
    SolidBlockBuffer [BLOCKLEN-1] = 0;

    for (i=0; i<BLOCKLEN-1; i++) {
	LightBlockBuffer [i] = 176;
    }
    LightBlockBuffer [BLOCKLEN-1] = 0;

    for (i=0; i<BLOCKLEN-1; i++) {
	DarkBlockBuffer [i] = 178;
    }
    DarkBlockBuffer [BLOCKLEN-1] = 0;
} // InitData

VOID main (argc, argv)
    USHORT argc;
    UCHAR *argv [];
{
    sprintf (ScratchBuf, "%s (tm) Ethernet Network Analyzer V%u.%u   General Software, Inc.",
	     PRODUCT_NAME,		// as defined in ANALYZER.H.
	     PRODUCT_MAJVER,		// as defined in MAKEFILE.
	     PRODUCT_MINVER);		// as defined in MAKEFILE.

    UtilStart (ScratchBuf);
    InitData ();
    LoadConfiguration ();

    //
    // Call the main menu.
    //

    MainMenu ();
    ExitProgram ();
} /* analyzer.c */
