with bios;
with bit_ops;           use bit_ops;
with com_port;
with hw_int;
with iio;
with interrupt;
with machine_code;
with port;
with rtkernel;
with system;
with task_control;
with text_io;
with tty;

procedure serial1 is

  subtype buffer is string(1 .. 256);

  stop         : exception;

  master       : boolean := true;
  time_out     : boolean := false;
  num_msgs     : integer;
  cur_msg      : integer := 0;
  done         : boolean := false;
  line_arrived : boolean := false;
  in_buf       : buffer;
  in_len       : integer := in_buf'first - 1;
  pr_buf       : buffer;
  pr_len       : integer;
  out_msg      : string(1 .. 80) := "111111111-111111111-111111111-" &
				    "111111111-111111111-111111111-" &
				    "111111111-111111111-";

  procedure putch(ch: character) is
  begin
    task_control.pre_emption_off;
    tty.put(ch);
    task_control.pre_emption_on;
  end putch;

  procedure clear_interrupt is
  begin
    port.out_byte(hw_int.INC1_OCR, hw_int.OCR_EOI_com_port1);
  end clear_interrupt;

  function wait_for_status(comreg : integer;
			   status : integer) return boolean is
    curstat : integer;
  begin
    for i in 1 .. 32000 loop
      curstat := port.in_byte(comreg);
      if (curstat and status) = status then
	return true;
      end if;
    end loop;
    return false;
  end wait_for_status;

  procedure send_data(ch: character) is
  begin
--  if not wait_for_status(com_port.COM1_MSR,
--                         com_port.MSR_CTS + com_port.MSR_DSR) then
--    putch('C');
--  end if;

    if not wait_for_status(com_port.COM1_LSR, com_port.LSR_THR_empty) then
      putch('E');               -- This is probably an error
    end if;
				-- Transmit character
    port.out_byte(com_port.COM1_THR, character'pos(ch));
  end send_data;

  procedure send_line(line : string) is
  begin
  --putch('S');
    for i in line'range loop
      send_data(line(i));
    end loop;
    send_data(ascii.cr);
    send_data(ascii.lf);
  end send_line;

  procedure receive_data is
    char_in : integer;
  begin
    time_out := true;
    char_in := port.in_byte(com_port.COM1_RDR);
    if in_len < in_buf'last then
      in_len := in_len + 1;
      in_buf(in_len) := character'val(char_in);
    end if;
    if character'val(char_in) = ascii.lf then
      line_arrived := true;
    end if;
  end receive_data;

  procedure receive_line is
    t   : long_integer := 0;
    len : integer;
  begin
  --putch('R');
    len := in_len;
    while not line_arrived loop
      if rtkernel."/="(rtkernel.rtk_get_kernel_type, rtkernel.rtk_dos) then
	delay 0.0;              -- Allow scheduler to run
      end if;
      if time_out then
	if len /= in_len then
	  t := 0;
	  len := in_len;
	end if;
	t := t + 1;
	if t > 1_000_000 then
	  text_io.put_line("Receive_line timed out!");
	  raise stop;
	end if;
      end if;
    end loop;
    line_arrived := false;
  end receive_line;

  task com_port_handler is
    pragma priority(system.priority'last);

    entry serial_interrupt;     -- Take over the COM1 serial port interrupt
    for serial_interrupt use at hw_int.int_com_port1;
  end com_port_handler;

  task body com_port_handler is
    int_id : integer;
  begin
    loop
      accept serial_interrupt do
	int_id := port.in_byte(com_port.COM1_IIR);
	case int_id and com_port.IIR_interrupt_mask is
	  when com_port.IIR_modem_status  =>  putch('M');
	  when com_port.IIR_transmit_data =>  putch('T');
	  when com_port.IIR_receive_data  =>  receive_data;
	  when com_port.IIR_line_status   =>  putch('L');
	  when others                     =>  putch('O');
	end case;
	clear_interrupt;
      end serial_interrupt;
    end loop;
  end com_port_handler;

  procedure setup is
    modem_control_values : constant := com_port.MCR_set_DTR +
				       com_port.MCR_set_RTS +
				       com_port.MCR_set_out2;
    intmask  : integer;
    baud     : integer;
    comparm  : integer;
    reg_file : interrupt.registers := (0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
    resp     : string(1 .. 80);
    last     : natural;
  begin
    text_io.put_line("Start up the Slave side first, then the Master.");
    text_io.new_line;
    text_io.put("Is this side the Master or the Slave? (m or s) [m] ");
    text_io.get_line(resp, last);
    if last >= resp'first and then resp(resp'first) /= 'm' then
      master := false;
    end if;
    text_io.put("This side is the ");
    if master then
      text_io.put_line("Master.");
    else
      text_io.put_line("Slave.");
    end if;

    text_io.new_line;
    text_io.put_line("Enter the same baud rate for both sides.");
    text_io.new_line;
    loop
      text_io.put("Baud rate (110, 150, 300, 600, 1200, " &
			     "2400, 4800, 9600)? [1200] ");
      text_io.get_line(resp, last);
      if last < resp'first then
	baud := 1200;
      else
	baud := integer'value(resp(1 .. last));
      end if;
      case baud is
	when  110 => comparm := bios.inp_110_baud;   exit;
	when  150 => comparm := bios.inp_150_baud;   exit;
	when  300 => comparm := bios.inp_300_baud;   exit;
	when  600 => comparm := bios.inp_600_baud;   exit;
	when 1200 => comparm := bios.inp_1200_baud;  exit;
	when 2400 => comparm := bios.inp_2400_baud;  exit;
	when 4800 => comparm := bios.inp_4800_baud;  exit;
	when 9600 => comparm := bios.inp_9600_baud;  exit;
	when others =>
	  text_io.put_line("  Illegal baud rate!");
      end case;
    end loop;

    if master then
      text_io.new_line;
      text_io.put("How many messages should be sent? [50] ");
      text_io.get_line(resp, last);
      if last < resp'first then
	num_msgs := 50;
      else
	num_msgs := integer'value(resp(1 .. last));
      end if;
    end if;

    rtkernel.rtk_disable_interrupts;
    clear_interrupt;            -- Clear any pending interrupt

				-- Initialize the COM1 port
    reg_file.ax := bios.intf_com_initialize +
		   bios.inp_8_bits          +
		   bios.inp_1_stop_bit      +
		   bios.inp_no_parity       +
		   comparm;     -- baud rate
    reg_file.dx := bios.com_port1;
    interrupt.vector(on => bios.int_com_port_io, register_block => reg_file);

				-- Enable the receive data interrupt
    port.out_byte(com_port.COM1_IER, com_port.IER_receive_data);

				-- Enable COM1 interrupts
    intmask := port.in_byte(hw_int.INC1_IMR);
    intmask := intmask and hw_int.IMR1_enable_com_port1;

    port.out_byte(hw_int.INC1_IMR, integer'pos(intmask));

				-- Set modem signals like Data Terminal Ready
    port.out_byte(com_port.COM1_MCR, modem_control_values);

				-- Toss any pending input or status
    intmask := port.in_byte(com_port.COM1_RDR);
    intmask := port.in_byte(com_port.COM1_IIR);
    intmask := port.in_byte(com_port.COM1_LSR);
    intmask := port.in_byte(com_port.COM1_MSR);
    intmask := port.in_byte(com_port.COM1_MSR);

    rtkernel.rtk_enable_interrupts;
  end setup;

  procedure shutdown is
    modem_control_values : constant := com_port.MCR_set_DTR +
				       com_port.MCR_set_RTS;
    intmask : integer;
  begin
    rtkernel.rtk_disable_interrupts;
				-- Reset modem signals and interrupts
    port.out_byte(com_port.COM1_MCR, modem_control_values);

				-- Disable COM1 interrupts
    intmask := port.in_byte(hw_int.INC1_IMR);
    intmask := intmask or hw_int.IMR1_disable_com_port1;

    port.out_byte(hw_int.INC1_IMR, integer'pos(intmask));

				-- Disable the receive data interrupt
    port.out_byte(com_port.COM1_IER, com_port.IER_disable_all);

    clear_interrupt;            -- Clear any pending interrupt

    rtkernel.rtk_enable_interrupts;

    abort com_port_handler;
  end shutdown;

  procedure master_loop is
    ch : character;
  begin
    time_out := true;
    for cur_msg in 1 .. num_msgs loop
      ch := character'val(character'pos('0') + (cur_msg mod 10));
      for ind in 1 .. (cur_msg mod out_msg'last) loop
	if ind mod 10 /= 0 then
	  out_msg(ind) := ch;
	end if;
      end loop;

      text_io.put_line("Sending msg" & integer'image(cur_msg));
      send_line(out_msg(1 .. (cur_msg mod out_msg'last)));
      receive_line;             -- Wait for acknowledgement
    end loop;

    text_io.put_line("Sending last msg");
    send_line("Thats ALL folks!");
    receive_line;               -- Wait for acknowledgement
  end master_loop;

  procedure slave_loop is
  begin
    text_io.put_line("Now start the Master side.");
    while not done loop
      receive_line;             -- Get next line

      pr_len := in_len;         -- Save data
      pr_buf := in_buf;
      in_len := in_buf'first - 1;

				-- Display line
      for ind in pr_buf'first .. pr_len loop
	putch(pr_buf(ind));
	if pr_buf(ind) = 'A' then
	  done := true;
	end if;
      end loop;

      send_line("");            -- Send acknowledgement
    end loop;
  end slave_loop;

begin  -- serial1
  setup;

  if master then
    master_loop;
  else
    slave_loop;
  end if;

  shutdown;

exception
  when others => shutdown;
end serial1;
