title   LOGGER.ASM
page    52,132
;
; logger.asm 12/8/92 sat
; revised 3/31/93
; Copyright 1992,1993 Nu-Mega Technologies, Inc. 
;
; LOGGER is a T&SR for recording Soft-ICE break-points into a buffer without
; popping up into Soft-ICE. It uses the ACTION command to transfer control to
; an INT handler. LOGGER may also be run from the command line (when it has
; previously been installed as a T&SR) to display the log file, to write the
; log file to disk as ASCII text, to print it on LPT1, or to clear the log
; buffer. The syntax of how to install LOGGER in AUTOEXEC.BAT can be displayed
; by the command "LOGGER /?" or "LOGGER /H". LOGGER.DOC contains a detailed 
; explanation of how to use LOGGER.
;
; Valid installation switches: (may be upper or lower case)
;       I       INT # to use
;       B       Buffer size: bytes (rounded down to next lower # of entries)
;       E       Buffer size: # of entries in buffer (16 bytes per entry)
;       S       Single record (not circular buffer)
;
; Valid command line switches: (may be upper or lower case)
;       ? or H  Help message
;       D       Display buffer contents on CRT
;       P       Print buffer contents on LPT1
;       F       Write buffer contents to disk file (ASCII)
;       C       Clear buffer
;       R       Remove LOGGER (if last T&SR installed)
;
;
ENTRY_SIZE      EQU 16
MAX_ENTRIES     EQU 4096
;
; FORMAT OF BREAKPOINT STORAGE:
;
; BYTE #   BPIO    BPINT    BPM    BPR 
;
;   0      BP_IO  BP_INT  BP_MEM  BP_RNG
;   1     IO_PORT  INT #   BP #    BP # 
;  2-3     VALUE    AX      AX    MEM_SEG 
;  4-5     COUNT    CX      CX    MEM_OFF 
;  6-7      DX      DX      DX   MEM_CONT
;  8-9      DI      DI      DI      DI
; 10-11     SI      SI      SI      SI
; 12-13     IP      IP      IP      IP
; 14-15     CS      CS      CS      CS
;
; OFFSETS INTO BREAKPOINT STORAGE:
;
BP_TYPE         EQU 0
IO_PORT         EQU 1
INT_NUM         EQU 1
BPT_NUM         EQU 1
IO_VALUE        EQU 2
REG_AX          EQU 2
MEM_SEG         EQU 2
IO_COUNT        EQU 4
REG_CX          EQU 4
MEM_OFF         EQU 4
REG_DX          EQU 6
MEM_CONT        EQU 6
REG_DI          EQU 8
REG_SI          EQU 10
BPT_IP          EQU 12
BPT_CS          EQU 14
;
;
; BP_TYPE FLAG BITS
; Breakpoint types
BP_IO           EQU 1
BP_INT          EQU 2
BP_MEM          EQU 4
BP_RNG          EQU 8
BP_NUM          EQU 0F0H
; I/O breakpoint flags
IO_WORD         EQU 10H
IO_OUT          EQU 20H
IO_STRING       EQU 40H
IO_DX           EQU 80H
;
;
; FLAG BITS IN run_flags
;
FIRST_ONLY      EQU 1
SI_25           EQU 2
WRAP_AROUND     EQU 4
CLEAR_WRAP      EQU 0FBH
;
;
; INDEX INTO STACK USING BP
;
FS_VALUE        EQU -10
ES_VALUE        EQU -8
DS_VALUE        EQU -6
;
; NOTE: ALL GENERAL REGISTERS SAVED ARE 32-BIT
; EBX stored at -4
;
AX_VALUE        EQU 0
CX_VALUE        EQU 4
DX_VALUE        EQU 8
DI_VALUE        EQU 12
SI_VALUE        EQU 16
; EBP stored at 20
IP_VALUE        EQU 24
CS_VALUE        EQU 26
FLAGS_VALUE     EQU 28
OLD_IP_VALUE    EQU 30
OLD_CS_VALUE    EQU 32
;
;
; INDEX INTO PRINT LINE
;
RA_LOC          EQU 5
IO_LOC          EQU 16
WORD_LOC        EQU 19
PORT_LOC        EQU 29
MMSG_LOC        EQU 33
AX_LOC          EQU 38
MDAT_LOC        EQU 42
MOFF_LOC        EQU 43
CT_LOC          EQU 44
CX_LOC          EQU 47
DX_LOC          EQU 56
DI_LOC          EQU 65
SI_LOC          EQU 74
;
;
;
.386
code    segment Public USE16 'code'
        assume cs:code, ds:code, es:code, ss:NOTHING
        org 100H
;
; Resident part of LOGGER.COM: INT handler & associated routines
;
;
; Data area used by interrupt handler
;
; The JMP instruction below is overwritten by data.
;
buffer_pointer label dword     ; Current data pointer
buffer_offset label word
start:  jmp entry               ; 3-byte JMP
        db 0                    ; Make it 4 bytes
;
run_flags       db 0            ; Default to circular buffer
int_number      db 60h          ; Default to INT 60
buffer_size     dw 8192         ; Default to 8K buffer

;
; Jump table to breakpoint handler routines
;
bp_handler dw bpm_handler       ;0
        dw bpio_handler         ;1
        dw bpint_handler        ;2
        dw int_handler_exit     ;3
        dw int_handler_exit     ;4
        dw bpr_handler          ;5
;
;
; This string is used to locate & identify this program.
; It MUST be just before the handler entry point.
;
logger_name db 'logger..'
;
;
; INT handler routine
;
; Entry point for INT handler
int_handler:
; Save all registers
        push    ebp
        push    esi
        push    edi
        push    edx
        push    ecx
        push    eax
; Set BP to stack index
        mov     bp,sp
        push    ebx
        push    ds
        push    es
        push    fs
; Set DS to here
        push    cs
        pop     ds
        cld
;
; Check if buffer full
        mov     ax,buffer_offset
        cmp     ax,buffer_size
        jb      short not_buffer_full
; If buffer full & single shot, exit
        test    run_flags,FIRST_ONLY
        jne     short int_handler_exit
; If not single shot, wrap buffer around
        mov     buffer_offset,0
        or      run_flags,WRAP_AROUND
;
; Save registers in data buffer
not_buffer_full:
        les     di,buffer_pointer
        add     di,REG_AX
        mov     ax,[bp + AX_VALUE]
        stosw
        mov     ax,[bp + CX_VALUE]
        stosw
        mov     ax,[bp + DX_VALUE]
        stosw
        mov     ax,[bp + DI_VALUE]
        stosw
        mov     ax,[bp + SI_VALUE]
        stosw
        mov     ax,[bp + IP_VALUE]
        stosw
        mov     ax,[bp + CS_VALUE]
        stosw
;
; Get breakpoint type
        mov     ax,912h
        mov     si,'FG'
        mov     di,'JM'
        int     3
;        
; Set up data buffer pointer
        les     di,buffer_pointer
; Get return address
        lfs     si,[bp + IP_VALUE]
; Save Breakpoint Number
        mov     es:[di + BPT_NUM],dh
; Go to appropriate BP handler routine
        and     dx,7
        mov     bx,dx
        shl     bx,1
        call    bp_handler[bx]
;
;
; Exit routine
;
; Set to next block in buffer
int_handler_exit_ok:
        add     buffer_offset,ENTRY_SIZE
;
; Exit here if I/O string repeat
int_handler_exit:
        pop     fs
        pop     es
        pop     ds
        pop     ebx
        pop     eax
        pop     ecx
        pop     edx
        pop     edi
        pop     esi
        pop     ebp
        iret
;
;
; Breakpoint storage routines
;
; BPIO handler
;
; First check if String I/O Repeat
bpio_handler:
; Go back to start of last block
        test    di,di
        jne     bpio_not_start
        mov     di,buffer_size
bpio_not_start:
        sub     di,ENTRY_SIZE
; Was last breakpoint a string I/O?
        test    byte ptr es:[di],IO_STRING
        je      short io_not_string
; If so, check if this is the same instruction
        mov     eax,[bp + IP_VALUE]
        cmp     eax,es:[di + BPT_IP]
        jne     io_not_string
; If same instruction, is count = FFFF?
        cmp     word ptr es:[di + IO_COUNT],0ffffh
        je      io_not_string
; If not, just increment count & exit
        inc     word ptr es:[di + IO_COUNT]
        ret
;
; Not a repeat string
io_not_string:
; Reset pointer to start of this block
        mov     di,buffer_offset
        mov     byte ptr es:[di],BP_IO
; Get last two instruction bytes
        mov     ax,fs:[si - 2]
; Is IP < 2 ?
        cmp     si,2
        jb      short io_not_port
; Check if it could be a port type
        cmp     al,0e7h
        ja      short io_not_port
        cmp     al,0e4h
        jb      short io_not_port
; Port type, correct instruction address
        dec     word ptr es:[di + BPT_IP]
        xchg    ah,al
        jmp     short bpio_handler_exit
;
; Check if it could be a DX type
io_not_port:
        xor     al,al
        cmp     ah,0ech
        jb      short not_dx_inst
        cmp     ah,0efh
        jbe     short bpio_handler_exit
;
; Must be a string type, so flag it
not_dx_inst:
        or      byte ptr es:[di],IO_STRING
; Make the count = 1
        mov     word ptr es:[di + IO_COUNT],1
;
bpio_handler_exit:
; Get  & set in/out, byte/word and port/dx bits
        and     ah,0bh
        shl     ah,4
        or      es:[di],ah
; If port type, store port, else zero
        mov     es:[di + IO_PORT],al
; & set address to point to the instruction
        dec     word ptr es:[di + BPT_IP]
        ret
;
;
; BPINT handler
;
bpint_handler:
; Set flag bit
        mov     byte ptr es:[di],BP_INT
;
; Get return address
        mov     eax,[bp + IP_VALUE]
; Set up to search vector table for it
        xor     di,di
        mov     es,di
; Check from INT 0 on up
        mov     ecx,100h
        mov     bx,cx
        dec     bx
; Look for matching interrupt vector
        repne   scasd
; Calculate corresponding INT #
int_match_found:
        sub     bx,cx
        les     di,buffer_pointer
        mov     es:[di + INT_NUM],bl
; Get actual return address & store that
        mov     eax,[bp + OLD_IP_VALUE]
        mov     es:[di + BPT_IP],eax
        ret
;
;
; Store BPM type
bpm_handler:
        mov     al,BP_MEM
        mov     es:[di],al
        ret
;
;
; Store BPR data
bpr_handler:
        mov     al,BP_RNG
        mov     es:[di],al
;
; If Soft-ICE 2.60+, get memory address & contents
        test    run_flags,SI_25
        jne     short bpr_common_exit
;
; Get flat address into esi
        mov     eax,cr2
        mov     esi,eax
; Find if it matches any of the segments
        mov     ax,[bp + DS_VALUE]              ; try DS
        call    check_if_this_segment
        jnc     short bpr_get_contents
        mov     ax,[bp + ES_VALUE]              ; try ES
        call    check_if_this_segment
        jnc     short bpr_get_contents
        mov     ax,[bp + CS_VALUE]              ; try CS
        call    check_if_this_segment
        jnc     short bpr_get_contents
        mov     ax,ss                           ; try SS
        call    check_if_this_segment
        jnc     short bpr_get_contents
        mov     ax,[bp + FS_VALUE]              ; try FS
        call    check_if_this_segment
        jnc     short bpr_get_contents
        mov     ax,gs                           ; try GS
        call    check_if_this_segment
        jnc     short bpr_get_contents
; No match, make it simple
        mov     eax,esi
        and     eax,0ffff0h
        call    set_segment
; Get word contents of address
bpr_get_contents:
        mov     bx,fs:[si]
; Store address & contents (over AX, CX & DX)
        mov     es:[di + MEM_OFF],si
        mov     es:[di + MEM_SEG],fs
        mov     es:[di + MEM_CONT],bx
bpr_common_exit:
        ret
;
;
; Check if flat address is within range of segment register
; Upon entry, ESI = flat address read from CR2
;              AX = a current segment (DS, ES ...)
; Upon exit, if address is not in this segment, Carry set
;            if in this segment, Carry clear
;                                SI set to offset, FS set to segment
;
check_if_this_segment:
; Make segment start a flat address
        and     eax,0ffffh
        shl     eax,4
; Set EBX to top of this segment
        mov     ebx,eax
        add     ebx,0ffffh
; Check if address is in here
        cmp     esi,eax
        jb      short not_in_this_segment
        cmp     esi,ebx
        ja      short not_in_this_segment
; Calculate offset
set_segment:
        sub     esi,eax
; Calculate segment
        shr     eax,4
        mov     fs,ax
        clc
        ret
;
; Set Carry flag to try another
not_in_this_segment:
        stc
        ret
;
;
; END OF RESIDENT CODE (BUFFER BEGINS HERE)
;
;
; Non-resident section of LOGGER.ASM: Installation code
;
; Data block used by installation
;
jmp_vector label dword
jmp_offset  dw 0
jmp_segment dw 0                                ; Segment will be stored here
;
;
; Install logger as a T&SR
;
;
; Process command line switches
logger_install_entry:
        mov     si,81h
logger_install:
        call    look_for_switch
        je      short default_install           ; No more switches
;
; INT # ?
        cmp     al,'i'
        jne     short not_int
        call    hex_digit                       ; Get first digit of INT #
        jc      short logger_install            ; Error, use default INT #
        shl     ax,4
        mov     bx,ax
        call    hex_digit                       ; Get second digit
        jc      short logger_install            ; Error, use default INT #
        add     ax,bx
        cmp     ax,100H                         ; Can't be over FFH
        jae     short logger_install
        cmp     ax,60H                          ; Or under 60H
        jb      short logger_install
        mov     int_number,al
        jmp     short logger_install
;
; Buffer size ?
not_int:
        cmp     al,'b'
        jne     short not_buffer_size
        call    decimal_number                  ; Get buffer size
        je      short logger_install            ; If zero, use default size
        and     bx,0fff0h                       ; Round down to next 16
        mov     buffer_size,bx
        jmp     short logger_install
;
; Number of entries?
not_buffer_size:
        cmp     al,'e'
        jne     short not_entries
        call    decimal_number                  ; Get number of entries
        je      short logger_install            ; If zero, use default size
        cmp     bx,MAX_ENTRIES
        jbe     short logger_install_ok         ; If too large,
        mov     bx,MAX_ENTRIES                  ; Make it maximum
logger_install_ok:
        mov     ax,ENTRY_SIZE                   ; # bytes per entry
        mul     bx
        mov     buffer_size,ax
        jmp     short logger_install
;
; First time only?
not_entries:
        cmp     al,'s'
        jne     short logger_install
        or      run_flags,FIRST_ONLY            ; Set flag for first time
        jmp     short logger_install
;
;
; Entry point if no parameters specified
;
default_install:
        call    find_if_installed               ; If LOGGER already installed,
        jnc     logger_already_installed        ; exit
;
; Get Soft-ICE version
        call    get_sice_version
        jge     sice_version_260
; Version 2.5x
        or      run_flags,SI_25
sice_version_260:
; Check if Soft-ICE installed
        cmp     si,'FG'
        je      soft_ice_not_installed
;
; Set new pointer into interrupt handler
        xor     ax,ax
        mov     es,ax
; Int # x 4
        mov     al,int_number
        shl     ax,2
        mov     di,ax
        lea     ax,int_handler
        cli
        stosw
        mov     ax,cs
        stosw
        sti
;
; Calculate data buffer segment
        lea     si,exit_code                    ; Offset
        dec     si                              ; Round up to next paragraph
        or      si,0fh
        inc     si
        shr     si,4
        mov     ax,cs                           ; Add to code segment
        add     ax,si
        mov     buffer_offset+2,ax
        mov     buffer_offset,0                 ; Set start offset = 0
;
; Calculate where T&SR will end
        mov     si,buffer_size                  ; Size of Data buffer
        shr     si,4
        add     ax,si
        mov     jmp_segment,ax
;
; Move the termination code there & go do it
        mov     es,ax
        xor     di,di
        lea     si,exit_code
        mov     cx,EXIT_CODE_SIZE
        rep     movsb
        mov     es,buffer_offset+2
        jmp     [jmp_vector]
;
;
; Routines called by installation to process the switches
;
; Reads a hexadecimal digit from the command line
;  Returns: If not a digit, Carry Flag set
;           If hexadecimal digit, Carry Flag clear
;                                 AX = digit
;
hex_digit:
        xor     ax,ax
        lodsb
        sub     al,'0'
        jb      short hex_digit_exit
        cmp     al,'9'
        jbe     short hex_digit_exit_ok
        sub     al,7
        jb      short hex_digit_exit
        cmp     al,15
        ja      short hex_digit_exit
hex_digit_exit_ok:
        clc
        ret
hex_digit_exit:
        stc
        ret
;
;
; Reads a decimal number from the command line
;  Returns: BX = number
;
decimal_number:
        xor     bx,bx
        xor     ax,ax
decimal_digit:
        lodsb
        sub     al,'0'
        jb      short decimal_number_exit
        cmp     al,'9'
        ja      short decimal_number_exit
        shl     bx,1
        add     ax,bx
        shl     bx,2
        add     bx,ax
        jmp     short decimal_digit
decimal_number_exit:
        test    bx,bx
        ret
;
;
; Termination code: before this code is executed,
; it is relocated to be after the buffer so that
; the buffer is part of the memory-resident T&SR.
;
exit_code:
; Zero the buffer
        mov     cx,buffer_size
        xor     ax,ax
        xor     di,di
        rep     stosb
; Calculate length in paragraphs
        mov     dx,cs
        mov     ax,ds
        sub     dx,ax
; Terminate
        mov     ax,3100h
        int     21h
;
EXIT_CODE_SIZE EQU $ - exit_code
;        
;
; Non-resident section of LOGGER.ASM: Command line error routines
;
; Data used by error routines
;
zlai    db 13,10,'Error - LOGGER already installed',13,10,'$'
zsni    db 13,10,'Error - Soft-ICE not installed',13,10,'$'
;
;
get_sice_version:
; Get Soft-ICE version
;
        xor     ax,ax
        mov     si,'FG'
        mov     di,'JM'
        int     3
; Check if version 2.60 +
        cmp     si,260h
        ret
;
;
; Error routines
;
; Error: tried to install LOGGER when already installed
;
logger_already_installed:
        lea     dx,zlai                         ; Tell LOGGER already installed
        mov     ah,9
        int     21h
        mov     ax,4c02h                        ; Exit errorlevel 2
        int     21h
;
;
; Error: tried to install LOGGER when Soft-ICE not installed
;
soft_ice_not_installed:
        lea     dx,zsni                         ; Tell Soft-ICE not installed
        mov     ah,9
        int     21h
        mov     ax,4c03h                        ; Exit errorlevel 3
        int     21h
;
;
; Non-resident section of LOGGER.ASM: Command line routines to translate
; buffer data to text for printing or storing.
;
; Data block used for display, print & file routines
;
; Data used for getting file name
zfn     db 13,10,'File name? $'
file_handle label word
name_buffer db 78
name_length db 0
text_line db 79 dup(0)
        db 13,10
;
;
; Text strings used to convert buffer data to ASCII
;
zline   db 'CSIP=    :                         AX=  '
        db '    CX=      DX=      DI=      SI=     ',0
zio     db ' IN BYTE  PT=      DA=',0
zcount  db 'CT=',0
zint    db ' INTERRUPT #=',0
zmem    db '  MEMORY BP#=',0
zrng    db '   RANGE BP#=',0
zout    db 'OUT',0
zword   db 'WORD',0
zadd    db 'MADD=',0
zdat    db ':        MDAT=',0
;
;
; Routines for converting different breakpoint types
decode_routine dw decode_io
        dw decode_int
        dw decode_mem
        dw decode_rng
;
;
; Converts data from buffer into a line of ASCII text
;
decode_into_line:
; Entry: FS:BX points to start of block in buffer
;        "file_handle" is the handle of the output device
; Exit:  FS:BX points to start of next block in buffer
;        "text_line" contains the block data in ASCII text form
;        "text_line" has been written to the output device
;
; Load the basic line
        push    cs
        pop     es
        lea     di,text_line
        lea     si,zline
        mov     cx,79
        rep     movsb
;
; Store RA
        lea     di,text_line
        add     di,RA_LOC
        mov     cx,fs:[bx + BPT_CS]
        call    make_4_digit_hex
        mov     al,':'
        stosb
        mov     cx,fs:[bx + BPT_IP]
        call    make_4_digit_hex
;
; Store AX (data if I/O)
        lea     di,text_line
        add     di,AX_LOC
        mov     cx,fs:[bx + REG_AX]
        call    make_4_digit_hex
;
; Store CX (count if I/O string)
        lea     di,text_line
        add     di,CX_LOC
        mov     cx,fs:[bx + REG_CX]
        call    make_4_digit_hex
;
; Store DX (mem contents if BPM/BPR & SI 2.6+)
        lea     di,text_line
        add     di,DX_LOC
        mov     cx,fs:[bx + REG_DX]
        call    make_4_digit_hex
;
; Store DI
        lea     di,text_line
        add     di,DI_LOC
        mov     cx,fs:[bx + REG_DI]
        call    make_4_digit_hex
;
; Store SI
        lea     di,text_line
        add     di,SI_LOC
        mov     cx,fs:[bx + REG_SI]
        call    make_4_digit_hex
;
; Get BP type flags into DL
        mov     dl,fs:[bx]
; Set pointer to start of identification text
        lea     di,text_line
        add     di,IO_LOC
; Make BP_TYPE flag bit into an index
        bsf     si,dx
        shl     si,1
; Go to routine to process breakpoint type
        call    decode_routine[si]
; Write line to output device
        push    bx
        mov     bx,file_handle
        mov     cx,81
        lea     dx,text_line
        mov     ah,40h
        int     21h
        pop     bx
; If end of circular buffer, wrap to start
        add     bx,ENTRY_SIZE
        cmp     bx,buffer_size
        jb      decode_no_wrap
        test    run_flags,FIRST_ONLY
        jne     decode_no_wrap
        xor     bx,bx
decode_no_wrap:
        ret
;
;
; Routines to process specific breakpoint types
;
; Converts I/O data to text
;
decode_io:
; Copy the basic I/O string
        lea     si,zio
        call    write_null_string

; out?
        test    dl,IO_OUT
        je      short not_io_out
        lea     di,text_line
        add     di,IO_LOC
        lea     si,zout
        call    write_null_string
not_io_out:
;
; word?
        test    dl,IO_WORD
        je      short not_io_word
        lea     di,text_line
        add     di,WORD_LOC
        lea     si,zword
        call    write_null_string
        jmp     short not_io_byte
;
; If I/O byte, fix data to byte size
not_io_word:
        lea     di,text_line
        add     di,AX_LOC
        mov     ax,es:[di + 2]
        stosw
        mov     word ptr es:[di],'  '
;
; Print port #
not_io_byte:
        xor     cx,cx
        mov     cl,fs:[bx + IO_PORT]
        test    dl,IO_DX
        je      short not_dx_port
        mov     cx,fs:[bx + REG_DX]
not_dx_port:
        lea     di,text_line
        add     di,PORT_LOC
        call    make_4_hex_digits
;
; If string I/O, make it CT (count) instead of CX
        test    dl,IO_STRING
        je      decode_io_exit
        lea     di,text_line
        add     di,CT_LOC
        lea     si,zcount
        call    write_null_string
;
decode_io_exit:
        ret
;
; Converts INT data to text
;
decode_int:
; Tell it's INTERRUPT breakpoint
        lea     si,zint
        call    write_null_string
;
; Print INT #
        xor     cx,cx
        mov     cl,fs:[bx + INT_NUM]
        lea     di,text_line
        add     di,PORT_LOC
        call    make_2_digit_hex
        ret
;
;
; Converts BPM data into text
;
decode_mem:
; Tell it's MEMORY breakpoint
        lea     si,zmem
        call    write_null_string
;
; Print Breakpoint #
        call    print_bp_number
        ret
;
;
; Converts BPR data into text
;
decode_rng:
; Tell it's RANGE breakpoint
        lea     si,zrng
        call    write_null_string
;
; Print Breakpoint #
        call    print_bp_number
;
; Is it Soft-ICE 2.6?
        call    get_sice_version
        jb      short decode_r_common_exit
;
; If Soft-ICE 2.6, overwrite headers
        lea     di,text_line
        add     di,MMSG_LOC
        lea     si,zadd
        call    write_null_string
;
        lea     di,text_line
        add     di,MDAT_LOC
        lea     si,zdat
        call    write_null_string
;
; Print referenced offset (segment & contents already done)
        mov     cx,fs:[bx + MEM_OFF]
        lea     di,text_line
        add     di,MOFF_LOC
        call    make_4_digit_hex
;
decode_r_common_exit:
        ret
;
; Routines to digitize numbers into text
;
make_4_hex_digits:
; Clear flag for no leading zero
        xor     ah,ah
        jmp     short make_4_digits
;
make_4_digit_hex:
; Set flag for leading zero
        mov     ah,1
;
make_4_digits:
; Do higher 2 digits
        xchg    ch,cl
        call    make_2_hex_digits
; Do lower 2 digits
        xchg    ch,cl
; Third digit
        mov     al,cl
        shr     al,4
        call    make_hex_digit
; Fourth digit
        inc     ah
        mov     al,cl
        call    make_hex_digit
        ret
;
make_2_digit_hex:
        mov     ah,1
make_2_hex_digits:
; First digit
        mov     al,cl
        shr     al,4
        call    make_hex_digit
; Second digit
        mov     al,cl
        call    make_hex_digit
        ret
;
;
make_hex_digit:
        and     al,0fh
        jne     short make_hex_digit_not_zero
; If leading zero, ignore
        test    ah,ah
        je     short make_hex_digit_exit
make_hex_digit_not_zero:
        inc     ah
        or      al,'0'
        cmp     al,'9'
        jle     make_hex_digit_numeric
        add     al,7
make_hex_digit_numeric:
        stosb
make_hex_digit_exit:
        ret
;
;
write_null_string_loop:
        stosb
write_null_string:
        lodsb
        test    al,al
        jne     short write_null_string_loop
        ret
;
;
; Print Breakpoint #
print_bp_number:
        xor     cx,cx
        mov     cl,fs:[bx + BPT_NUM]
        lea     di,text_line
        add     di,PORT_LOC
        call    make_2_digit_hex
        ret
;
;
; Non-resident section of LOGGER.ASM: Command line switch processing
;
; Data block used by command line switch routines
;
zlnlt:  db 13,10,'Error - LOGGER not last TSR',13,10,'$'
zlbe    db 13,10,'Error - LOGGER buffer empty',13,10,'$'
zlni    db 13,10,'Error - LOGGER not installed',13,10,'$'
;
;
; Help message
;
zh      db 13,10,'   To install logger in AUTOEXEC, use '
        db 'the following syntax. All the switches'
        db 13,10,'are optional. Either "-" or "/" may be'
        db ' used as the switch character.',13,10
        db 13,10,'        LOGGER /Inn /Bnnnn /Ennn /S',13,10
        db 13,10,'   Inn specifies the INT number (hex) '
        db 'where the handler will be installed. If'
        db 13,10,'no I switch is used, it will be installed at INT 60H.'
        db 13,10,'   Bnnnn specifies the buffer size to '
        db 'be used (decimal, max 65536). If no B or'
        db 13,10,'E switch is used, the buffer size will '
        db 'be 8192 bytes (512 entries).'
        db 13,10,'   Ennn specifies the number of entries '
        db '(decimal, max 4096) in the buffer. This'
        db 13,10,'is an alternative to the /B switch: do not use both.'
        db 13,10,'   S specifies that only the first '
        db 'occurrences will be logged (the buffer does '
        db 13,10,'not wrap around). The default is a circular buffer.',13,10
        db 13,10,'   Valid command line switches are as follows:'
        db 13,10,'        LOGGER /H  or LOGGER /? will '
        db 'display this help screen.'
        db 13,10,'        LOGGER /D  will display the '
        db 'contents of the log buffer.'
        db 13,10,'        LOGGER /P  will print the '
        db 'contents of the buffer on LPT1.'
        db 13,10,'        LOGGER /F  will write the '
        db 'contents of the buffer to a disk file as'
        db 13,10,'                   ASCII text. You will '
        db 'be prompted for the name of the file.'
        db 13,10,'        LOGGER /C  will clear the '
        db 'log buffer.'
        db 13,10,'        LOGGER /R  will remove LOGGER '
        db 'if it is the last T&SR.',13,10,'$'
;
;
get_buffer_start:
; Returns : If buffer empty, Carry Flag set
;           If buffer not empty, Carry Flag clear
;                                FS:BX = pointer to oldest data
;
; Set to start at current_pointer
        lfs     bx,buffer_pointer
; If not wrapped, start at buffer start
        test    run_flags,WRAP_AROUND
        jne     short get_buffer_start_exit
        xor     bx,bx
        cmp     buffer_offset,bx
        jne     short get_buffer_start_exit
; If buffer empty, set carry & exit
        stc
        ret
get_buffer_start_exit:
        clc
        ret
;
;
; Find if LOGGER is installed
;  Returns: If not installed, Carry flag set
;           If installed, Carry flag clear
;                         data block copied to here from TSR
;                         GS = code segment of TSR
;       
find_if_installed:
        mov     bx,180h                         ; Start at INT 60H
;
find_if_installed_loop:
        lea     si,int_handler                  ; Get offset for handler
        xor     ax,ax                           ; Set ES to Interrupts
        mov     es,ax
        les     di,es:[bx]                      ; Address of an INT handler
        cmp     di,si
        jne     find_if_installed_next          ; Can't be this one
        sub     di,8
        lea     si,logger_name                  ; Check if "logger.."
        mov     cx,8                            ; is just before it
        repe    cmpsb
        je      short find_if_installed_exit    ; If so, return with pointer
;
find_if_installed_next:
        add     bx,4                            ; If not, try next INT
        cmp     bx,400h
        jl      short find_if_installed_loop
;
; If LOGGER not installed, set Carry flag
        stc
        ret
;
find_if_installed_exit:
; If LOGGER installed, copy data & clear Carry
; Set up offsets
        mov     di,100h
        mov     si,di
; Get length to write
        lea     cx,logger_name
        sub     cx,di
; Save code segment
        push    es
        pop     gs
; Set up segments
        push    es
        pop     ds
        push    cs
        pop     es
; Copy the actual data to here
        rep     movsb
; Restore DS
        push    cs
        pop     ds
        clc
        ret
;
;
; Error routines
;
logger_not_last_tsr:
        lea     dx,zlnlt
        jmp     short log_error_common
;
log_buffer_empty:
        lea     dx,zlbe
        jmp     short log_error_common
;
logger_not_installed:
;
        lea     dx,zlni                         ; Tell LOGGER not installed
log_error_common:
        mov     ah,9
        int     21h
        mov     ax,4c01h                        ; Exit errorlevel 1
        int     21h
;
;
; Process /P switch
;
logger_print:
;
; If LOGGER not installed, exit
        call    find_if_installed
        jc      logger_not_installed
; Set pointer to start of data
        call    get_buffer_start
        jc      short log_buffer_empty
; Set file handle to printer
        mov     file_handle,4
;
; Print a line at a time
logger_print_line:
        call    decode_into_line
        cmp     bx,buffer_offset
        jne     logger_print_line
        jmp     short terminate_logger
;
;
; Process /D switch
;
logger_display:
;
; If LOGGER not installed, exit
        call    find_if_installed
        jc      short logger_not_installed
; Set pointer to start of data
        call    get_buffer_start
        jc      short log_buffer_empty

; Set file handle to standard output
        mov     file_handle,1
;
; Do one page
logger_display_page:
        mov     cx,23
; Decode and display line on screen
logger_display_line:
        push    cx
        call    decode_into_line
        pop     cx
; If end of stored data, quit
        cmp     bx,buffer_offset
        je      short logger_display_exit
        loop    logger_display_line
; Wait for key
logger_display_wait:
        xor     ah,ah
        int     16h
        jmp     short logger_display_page
;
logger_display_exit:
        jmp     short terminate_logger
;
;
; Process /F switch
;
logger_file:
;
; If LOGGER not installed, exit
        call    find_if_installed
        jc      logger_not_installed
; Get name of file
logger_file_name:
        lea     dx,zfn
        mov     ah,9
        int     21h
        lea     dx,name_buffer
        mov     ah,0ah
        int     21h
; Make it null terminated
        xor     ax,ax
        mov     al,name_length
        lea     di,text_line
        add     di,ax
        mov     byte ptr [di],0
; Create file
        lea     dx,text_line
        xor     cx,cx
        mov     ah,3ch
        int     21h
        jc      logger_file_name
        mov     file_handle,ax
;
        call    get_buffer_start
        jnc     logger_file_line
; Buffer empty
        mov     bx,file_handle
        mov     ah,3eh
        int     21h
        jmp     log_buffer_empty
;
; Write a line at a time
logger_file_line:
        call    decode_into_line
        cmp     bx,buffer_offset
        jne     logger_file_line
; Close the file
        mov     bx,file_handle
        mov     ah,3eh
        int     21h
terminate_logger:
        mov     ax,4c00h
        int     21h
;
;
; Process /C switch
;
logger_clear:
; If LOGGER not installed, exit
        call    find_if_installed
        jc      logger_not_installed
; Clear the wrap-around flag
        and     gs:run_flags,CLEAR_WRAP
; Reset buffer pointer to start
        xor     ax,ax
        mov     gs:buffer_offset,ax
; Get start address of buffer
        xor     di,di
        mov     es,buffer_offset+2
; Clear the buffer
        mov     cx,buffer_size
        rep     stosb
        jmp     short terminate_logger
;
;
; Process /H switch
;
logger_help:
;
        lea     dx,zh
        mov     ah,9
        int     21h
        jmp     short terminate_logger
;
;
; Process /R switch
;
logger_remove:
; If LOGGER not installed, exit
        call    find_if_installed
        jc      logger_not_installed
;
; Check if LOGGER is last TSR
        mov     ax,gs:[2]
        inc     ax
        cmp     ax,ds:[2ch]
        jb      logger_not_last_tsr
;
; Release the INT vector
        xor     bx,bx
        mov     es,bx
        mov     bl,int_number
        shl     bx,2
        mov     dword ptr es:[bx],0
;
; Fix PSP to point to environment of old LOGGER
        mov     ax,gs:[2ch]
        mov     ds:[2ch],ax
;
; Release environment of old LOGGER
        push    ax
        pop     es
        mov     ah,49h
        int     21h
;
; Release memory used by old LOGGER
        push    gs
        pop     es
        mov     ah,49h
        int     21h
;
        jmp     short terminate_logger
;
;
; Command line routines: Program entry point
;
; Data block for program entry routines
;
; List of valid switches and associated routines
;       ?           help
;       h or H      help
;       p or P      print contents of buffer
;       f or F      write contents of buffer to file in ASCII text
;       d or D      display contents of buffer
;       c or C      clear the buffer
;       r or R      remove LOGGER
;       other       install LOGGER
;
;
switch_list db '?hpdfcr',0
;
routine_list dw logger_help
        dw logger_help
        dw logger_print
        dw logger_display
        dw logger_file
        dw logger_clear
        dw logger_remove
        dw logger_install_entry
;
;
; Program entry point
;  Examines switches and branches to appropriate routine
;
entry:
        mov     si,81H                          ; Get pointer to arguments
        mov     cx,80
; Look for "-" or "/"
search_argument_list:
        call    look_for_switch
        je      default_install                 ; No switches, assume install
;
        lea     di,switch_list                  ; Table of valid switches
        xor     bx,bx                           ; Index into table
        dec     bx
;
check_switch_loop:                              ; Look for match
        inc     bx
        cmp     byte ptr [bx+di],0
        je      short go_to_routine             ; Match found, go to routine
        cmp     al,[bx+di]
        jne     short check_switch_loop
;
go_to_routine:
        shl     bx,1                            ; If no match, try install
        jmp     routine_list[bx]

;
; Searches argument list for switch character
;  Returns: If no switch found, Zero Flag set
;           If switch found, Zero Flag clear
;                            AL = next character (set to lower case)
;                            DS:SI = pointer to next non-space
;
look_for_switch:
        lodsb
        cmp     al,0                            ; If null or Cr, no arguments
        je      look_for_switch_exit
        cmp     al,13
        je      look_for_switch_exit
        cmp     al,"-"
        je      short switch_found
        cmp     al,"/"
        jne     short look_for_switch
;
switch_found:
        lodsb                                   ; Get the next character,
        or      al,20H                          ; make it lower case
look_for_switch_exit:
        ret
;
;
code    ends
        end     start
