//========================================================================================
//
//     File:			ToolPal.cp
//     Creation Date:	4/14/93
//
//     COPYRIGHT 1993 SYMANTEC CORPORATION. ALL RIGHTS RESERVED. UNPUBLISHED -- RIGHTS
//     RESERVED UNDER THE COPYRIGHT LAWS OF THE UNITED STATES. USE OF COPYRIGHT NOTICE IS
//     PRECAUTIONARY ONLY AND DOES NOT IMPLY PUBLICATION OR DISCLOSURE.
//
//     THIS SOFTWARE CONTAINS PROPRIETARY AND CONFIDENTIAL INFORMATION OF SYMANTEC
//     CORPORATION. USE, DISCLOSURE, OR REPRODUCTION IS PROHIBITED WITHOUT THE PRIOR
//     EXPRESS WRITTEN PERMISSION OF SYMANTEC CORPORATION.
//
//     RESTRICTED RIGHTS LEGEND
//     Use, duplication, or disclosure by the Government is subject to restrictions as set
//     forth in subparagraph (c)(l)(ii) of the Rights in Technical Data and Computer
//     Software clause at DFARS 252.227-7013. Symantec Corporation, 10201 Torre Avenue,
//     Cupertino, CA 95014.
//
//========================================================================================

#ifndef BRBITMAP_H
#include <BRBitmap.h>
#endif

#ifndef BRPEN_H
#include <BRPen.h>
#endif

#ifndef BRGRHELP_H
#include <BRGrHelp.h>
#endif

#ifndef BRAPPLIC_H
#include <BRApplic.h>
#endif

#ifndef TOOLPAL_H
#include "ToolPal.h"
#endif

#ifndef DRWTOOLS_H
#include "DrwTools.h"
#endif

#ifndef BNDTOOLS_H
#include "BndTools.h"
#endif

#ifndef PAINT_H
#include "Paint.h"
#endif

#ifndef DRWSTATE_H
#include "DrwState.H"
#endif


//----------------------------------------------------------------------------------------
// cToolPalette::cToolPalette
//----------------------------------------------------------------------------------------

cToolPalette::cToolPalette(BR_CView* enclosure,
						   BR_CBureaucrat* supervisor,
						   cDrawState* State) :
	BR_CView(enclosure, supervisor, kDefaultViewId, kPixelUnits, 0, 0, 0, 0, 0, kFixedLeft, kFixedTop, BR_CColor(kRGBBlack)),
	fDrawState(State),
	fCurrentToolNumber(0)
{
	// Create the collections
	fToolCollection = new BC_TUnboundedCollection < cPaintTool * , BC_CUnmanaged>;
	fBitmapCollection = new BC_TUnboundedCollection < BR_CBitmap * , BC_CUnmanaged>;

	//
	//	Add the tools
	//
	AddTool(new cPenTool(State));
	AddTool(new cEraserTool(State));
	AddTool(new cSprayTool(State));

	AddTool(new cLineTool(State, IDB_LINE_UNSELECT));

	AddTool(new cSquareTool(State, FALSE, IDB_RECT_UNSELECT));
	AddTool(new cSquareTool(State, TRUE, IDB_FILLRECT_UNSELECT));

	AddTool(new cEllipseTool(State, FALSE, IDB_ELLIPSE_UNSELECT));
	AddTool(new cEllipseTool(State, TRUE, IDB_FILLELLIPSE_UNSELECT));

	AddTool(new cRoundRectangleTool(State, FALSE, IDB_ROUNDRECT_UNSELECT));
	AddTool(new cRoundRectangleTool(State, TRUE, IDB_FILLROUNDRECT_UNSELECT));

	//
	// Assign current tool to the first tool added to the collection
	//
	fDrawState->fCurrentTool = (cPaintTool *)((*fToolCollection)[fCurrentToolNumber]);

	//
	//	Calculate and set size of this pane
	//
	long NumberOfTools = fToolCollection->Length();
	long Width = (NumberOfCols) * (ToolWidth + LineWidth);
	long Length = ((NumberOfTools + 1) / NumberOfCols) * (ToolHeight + LineWidth);

	//
	//	Adjust the size of the Pane
	//
	BR_CRectangle PaneRect(0, 0, Width, Length);
	ViewToEnclosure(PaneRect);

	SetFrame(PaneRect, FALSE);

	SetBackgroundColor(kRGBBlack);
}

//----------------------------------------------------------------------------------------
// cToolPalette::~cToolPalette
//----------------------------------------------------------------------------------------

cToolPalette::~cToolPalette()
{
	//
	//	Delete items in tool collection and the tool collection itself
	//
	if (fToolCollection)
	{
		BC_TCollectionActiveIterator<cPaintTool *> iter(*fToolCollection);
		while (!iter.IsDone())
		{
			cPaintTool *paintTool = *iter.Item();
			delete paintTool;
			iter.Next();
		}
		
		delete fToolCollection;
		fToolCollection = NULL;
	}	
	
	//
	//	Delete items in bitmap collection and the bitmap collection itself
	//
	if (fBitmapCollection)
	{
		BC_TCollectionActiveIterator<BR_CBitmap *> iter(*fBitmapCollection);
		while (!iter.IsDone())
		{
			BR_CBitmap *bitmap = *iter.Item();
			delete bitmap;
			iter.Next();
		}
		
		delete fBitmapCollection;
		fBitmapCollection = NULL;
	}	
}

//----------------------------------------------------------------------------------------
// cToolPalette::AddTool
//----------------------------------------------------------------------------------------

void cToolPalette::AddTool(cPaintTool *tool)
{
	fToolCollection->Append(tool);
	
	//
	//	Load the bitmap and add it to the list
	// 
	BR_CBitmap *bitmap = new BR_CBitmap(*GetApplication(), tool->GetBitmapId());
	fBitmapCollection->Append(bitmap);
}

//----------------------------------------------------------------------------------------
// cToolPalette::Draw
//	
// Draw all the tools.  If the tool is selected draw it selected.
//----------------------------------------------------------------------------------------

void cToolPalette::Draw(BR_CCanvas &canvas, const BR_CRectangle &area)//Override
{
	BR_CView::Draw (canvas, area);
	long NumTools = fBitmapCollection->Length();
	
	for (int i=0; i < NumTools; ++i)
	{
		BR_CRectangle Rect = GetToolRectangle(i);
		if (i == fCurrentToolNumber)
			DrawSelected(i, Rect);
		else
			DrawUnselected(i, Rect);
	}

}

//----------------------------------------------------------------------------------------
// cToolPalette::DoClick
//
// Find the tool the user clicked in and set the new current tool. Call to refresh the
// pane so that the selected tool is drawn properly.
//----------------------------------------------------------------------------------------

void  cToolPalette::DoClick(const BR_CPoint& ptMouse, KeyModifiers keys)
{
	BR_CView::DoClick(ptMouse, keys);
	
	int numberOfBitmaps = fBitmapCollection->Length();
	for (int i=0; i < numberOfBitmaps; ++i)
	{
		BR_CRectangle Rect = GetToolRectangle(i);
		if (Rect.IsPointIn(ptMouse))
		{
			RefreshRect(GetToolRectangle(fCurrentToolNumber));
			
			// Assign the new tool to the draw state current tool
			fCurrentToolNumber = i;
			fDrawState->fCurrentTool = (cPaintTool*) ((*fToolCollection)[fCurrentToolNumber]);
			
			RefreshRect(Rect);
		}
	}
}

//----------------------------------------------------------------------------------------
// cToolPalette::GetToolRectangle
//
// Calculate the rectangle for the given tool number.
//----------------------------------------------------------------------------------------

BR_CRectangle cToolPalette::GetToolRectangle(int ToolNumber)
{
	BR_CRectangle ToolRect;
	long Left	= (long)(((ToolNumber % NumberOfCols)) * (ToolWidth + LineWidth))+1;
	long Top	= (long)(((ToolNumber)/NumberOfCols) * (ToolHeight+ LineWidth))+1;

	ToolRect.Set(Left, Top, Left + ToolWidth-1,	Top	+ ToolHeight-1);
	return ToolRect;
}

//----------------------------------------------------------------------------------------
// cToolPalette::DrawSelected
//
// Warning: this method assumes that the drawing environment is set up.
//----------------------------------------------------------------------------------------

void cToolPalette::DrawSelected(int ToolNumber, BR_CRectangle &Location)
{
	BR_CBitmap *pBitmap = (BR_CBitmap *) (*fBitmapCollection)[ToolNumber];
	
	if (pBitmap == NULL)
		return;		// If there is no bitmap return!
	
	BR_CRectangle BitRect(0,0, ToolWidth-2, ToolHeight-2);
	GetCanvas()->DrawBitmap( Location.fLeft+2, Location.fTop+2, BitRect, *pBitmap, BEDROP_SRCCOPY);
	GetCanvas()->MovePenTo(Location.GetTopLeft());
	DrawRect(Location, kRGBGray);
}

//----------------------------------------------------------------------------------------
// cToolPalette::DrawUnselected
//
// Warning: this method assumes that the drawing environment is set up.
//----------------------------------------------------------------------------------------

void cToolPalette::DrawUnselected(int ToolNumber, BR_CRectangle &Location)
{
	BR_CBitmap *pBitmap = (BR_CBitmap *) (*fBitmapCollection)[ToolNumber];
	
	if (pBitmap == NULL)
		return;
	
	BR_CRectangle BitRect(0,0, ToolWidth-1, ToolHeight-1);
	GetCanvas()->DrawBitmap( Location.fLeft, Location.fTop, BitRect, *pBitmap, BEDROP_SRCCOPY);

	GetCanvas()->MovePenTo(Location.fLeft, Location.fBottom);
	BR_CColor color(kRGBWhite);
	BR_CPen pen(2, color);

	{
		BR_CGraphicsToolSelector GTSTemp(this , &pen);
		GetCanvas()->DrawLineTo(Location.fLeft, Location.fTop);
		GetCanvas()->DrawLineTo(Location.fRight, Location.fTop);
	}

	{
		color = kRGBGray;
		pen.SetDashColor(color);
		BR_CGraphicsToolSelector GTSTemp(this , &pen);
		GetCanvas()->DrawLineTo(Location.fRight, Location.fBottom);
		GetCanvas()->DrawLineTo(Location.fLeft, Location.fBottom);

		GetCanvas()->MovePenTo(Location.fLeft, Location.fBottom);
		GetCanvas()->DrawLineTo(Location.GetBottomRight());
	}
}

//----------------------------------------------------------------------------------------
// cToolPalette::DrawRect
//
// Draw a rectangle with the given color with line thickness 2.
//----------------------------------------------------------------------------------------

void cToolPalette::DrawRect(BR_CRectangle &r, BR_CColor::RGBValue anRGBValue)
{
	BR_CColor color(anRGBValue);
	BR_CPen pen(2, color);
	BR_CGraphicsToolSelector GTSTemp(this , &pen);
	GetCanvas()->DrawLineTo(r.fRight, r.fTop);
	GetCanvas()->DrawLineTo(r.fRight, r.fBottom);
	GetCanvas()->DrawLineTo(r.fLeft, r.fBottom);
	GetCanvas()->DrawLineTo(r.fLeft, r.fTop);
}
