//========================================================================================
//
//     File:			myDrwObj.cp
//     Release Version:	$Revision$
//
//     COPYRIGHT 1993 SYMANTEC CORPORATION. ALL RIGHTS RESERVED. UNPUBLISHED -- RIGHTS
//     RESERVED UNDER THE COPYRIGHT LAWS OF THE UNITED STATES. USE OF COPYRIGHT NOTICE IS
//     PRECAUTIONARY ONLY AND DOES NOT IMPLY PUBLICATION OR DISCLOSURE.
//
//     THIS SOFTWARE CONTAINS PROPRIETARY AND CONFIDENTIAL INFORMATION OF SYMANTEC
//     CORPORATION. USE, DISCLOSURE, OR REPRODUCTION IS PROHIBITED WITHOUT THE PRIOR
//     EXPRESS WRITTEN PERMISSION OF SYMANTEC CORPORATION.
//
//     RESTRICTED RIGHTS LEGEND
//     Use, duplication, or disclosure by the Government is subject to restrictions as set
//     forth in subparagraph (c)(l)(ii) of the Rights in Technical Data and Computer
//     Software clause at DFARS 252.227-7013. Symantec Corporation, 10201 Torre Avenue,
//     Cupertino, CA 95014.
//
//========================================================================================

#ifndef   MYDRWOBJ_H
#include "myDrwObj.h"
#endif

#ifndef   MYDRWVW_H
#include "myDrwVw.h"
#endif

#ifndef   __MATH__
#include <Math.h>
#endif

//========================================================================================
// CLASS CMyDrawObject
//========================================================================================

BR_Distance CMyDrawObject::fLeaway = 3;

//----------------------------------------------------------------------------------------
// CMyDrawObject::CMyDrawObject
//----------------------------------------------------------------------------------------
CMyDrawObject::CMyDrawObject(const BR_CPen& itsPen, BR_Boolean select):
	fPen(itsPen.GetThickness(), itsPen.GetDashColor(), itsPen.GetStyle())
{
	fSelected = select;
}

//----------------------------------------------------------------------------------------
// CMyDrawObject::~CMyDrawObject
//----------------------------------------------------------------------------------------

CMyDrawObject::~CMyDrawObject()
{
}

//----------------------------------------------------------------------------------------
// CMyDrawObject::DrawAKnob
//----------------------------------------------------------------------------------------

void CMyDrawObject::DrawAKnob(CMyDrawView* theView, const BR_CPoint& atPoint)
{
	BR_CRectangle aRect(atPoint, atPoint);
	aRect.Inflate(fLeaway, fLeaway);
	theView->Hilite(aRect);
}

//----------------------------------------------------------------------------------------
// CMyDrawObject::HighlightIfSelected
//----------------------------------------------------------------------------------------

void CMyDrawObject::HighlightIfSelected(CMyDrawView* theView)
{
    if (fSelected)
		DrawKnobs(theView);
}

//----------------------------------------------------------------------------------------
// CMyDrawObject::IsPointInShape
//----------------------------------------------------------------------------------------

BR_Boolean CMyDrawObject::IsPointInShape(const BR_CPoint& ptMouse) 
{
	BR_CRectangle testArea(fExtent);
	
	testArea.Inflate(fLeaway, fLeaway);					// leaway on each side

	return (testArea.IsPointIn(ptMouse));
}

//----------------------------------------------------------------------------------------
// CMyDrawObject::SetSelect
//----------------------------------------------------------------------------------------

void CMyDrawObject::SetSelect(BR_Boolean value)
{
	fSelected = value;
}

//----------------------------------------------------------------------------------------
// CMyDrawObject::SetSelectAndDraw
//----------------------------------------------------------------------------------------

void CMyDrawObject::SetSelectAndDraw(BR_Boolean value, CMyDrawView* itsView)
{
	if (value != fSelected)
		DrawKnobs(itsView);		// relies on the fact that knobs are drawn in XOR-style mode.
	SetSelect(value);
}

//========================================================================================
// CLASS CMyDrawLine
//========================================================================================

//----------------------------------------------------------------------------------------
// CMyDrawLine::CMyDrawLine
//----------------------------------------------------------------------------------------

CMyDrawLine::CMyDrawLine(const BR_CPoint& point1, const BR_CPoint& point2,
		const BR_CPen& itsPen, BR_Boolean select):
	CMyDrawObject(itsPen, select),
	fPoint1(point1), fPoint2(point2)
{
	// The extent is expected to be a "valid" rectangle, 
	// regardless of the direction the line was originally drawn.
	// Note, though, that the original direction is preserved, 
	// i.e. fPoint1 is the starting point, and fPoint2 is the ending point.
	
	fExtent.Set(point1, point2);	// fExtent was default rectangle; now we know what it really is.
	fExtent.Sort();
}

//----------------------------------------------------------------------------------------
// CMyDrawLine::~CMyDrawLine
//----------------------------------------------------------------------------------------

CMyDrawLine::~CMyDrawLine()
{
}

//----------------------------------------------------------------------------------------
// CMyDrawLine::Draw
//----------------------------------------------------------------------------------------

void CMyDrawLine::Draw(CMyDrawView* theView)
{
    BR_CGraphicsToolSelector gts(theView, &fPen);

    theView->MovePenTo(fPoint1);
    theView->DrawLineTo(fPoint2);
}

//----------------------------------------------------------------------------------------
// CMyDrawLine::DrawKnobs
//----------------------------------------------------------------------------------------

void CMyDrawLine::DrawKnobs(CMyDrawView* theView)
{
	DrawAKnob(theView, fPoint1);
	DrawAKnob(theView, fPoint2);
}

//----------------------------------------------------------------------------------------
// CMyDrawLine::IsPointInShape
//----------------------------------------------------------------------------------------

BR_Boolean CMyDrawLine::IsPointInShape(const BR_CPoint& theMouse)
{
	BR_CPoint ptMouse(theMouse);
	if (CMyDrawObject::IsPointInShape(ptMouse))
	{
		// This calculation is derived from the formula of a line,
		// given two points (fPoint1 & fPoint2),
		// combined with the formula for the distance of a point (theMouse) from a line.
		
		BR_CPoint delta = fPoint2 - fPoint1;

		float numerator = fabs(delta.fX * theMouse.fY - delta.fY * theMouse.fX
					+ fPoint2.fY * fPoint1.fX - fPoint1.fY * fPoint2.fX);

		float denominator = sqrt(pow(delta.fY, 2) + pow(delta.fX, 2));
		
		return (numerator / denominator <= fLeaway + fPen.GetThickness());	// account for pen thickness
	}
	else
		return (FALSE);
}

//----------------------------------------------------------------------------------------
// CMyDrawObject::Refresh
//----------------------------------------------------------------------------------------

void CMyDrawObject::Refresh(CMyDrawView* theView)
{
	BR_Distance slop = max(fLeaway, fPen.GetThickness());		 // account for pen thickness
	BR_CRectangle aRectangle(fExtent);
	aRectangle.Inflate(slop, slop);

	theView->RefreshRect(aRectangle);
}

//========================================================================================
// CLASS CMyDrawArea
//========================================================================================

//----------------------------------------------------------------------------------------
// CMyDrawArea::CMyDrawArea
//----------------------------------------------------------------------------------------

CMyDrawArea::CMyDrawArea(const BR_CRectangle& newBounds,
		const BR_CPen& itsPen, BR_Boolean select):
	CMyDrawObject(itsPen, select)
{
	fExtent = newBounds;	// fExtent was default rectangle; now we know what it really is.
	fExtent.Sort();			// make sure extent is a valid rectangle.
}

//----------------------------------------------------------------------------------------
// CMyDrawArea::~CMyDrawArea
//----------------------------------------------------------------------------------------

CMyDrawArea::~CMyDrawArea()
{
}

//----------------------------------------------------------------------------------------
// CMyDrawArea::DrawKnobs
//----------------------------------------------------------------------------------------

void CMyDrawArea::DrawKnobs(CMyDrawView* theView)
{
	// Draw knobs at the four corners, in clockwise order.
	
	DrawAKnob(theView, fExtent.GetTopLeft());
	DrawAKnob(theView, fExtent.GetTopRight());
	DrawAKnob(theView, fExtent.GetBottomRight());
	DrawAKnob(theView, fExtent.GetBottomLeft());
}

//========================================================================================
// CLASS CMyDrawRect
//========================================================================================

//----------------------------------------------------------------------------------------
// CMyDrawRect::CMyDrawRect
//----------------------------------------------------------------------------------------

CMyDrawRect::CMyDrawRect(const BR_CRectangle& newBounds,
		const BR_CPen& itsPen, BR_Boolean select):
	CMyDrawArea(newBounds, itsPen, select)
{
}

//----------------------------------------------------------------------------------------
// CMyDrawRect::~CMyDrawRect
//----------------------------------------------------------------------------------------

CMyDrawRect::~CMyDrawRect()
{
}

//----------------------------------------------------------------------------------------
// CMyDrawRect::Draw
//----------------------------------------------------------------------------------------

void CMyDrawRect::Draw(CMyDrawView* theView)
{
    BR_CGraphicsToolSelector gts(theView, &fPen);
	theView->DrawRectangle(fExtent);
}

//========================================================================================
// CLASS CMyDrawEllipse
//========================================================================================

//----------------------------------------------------------------------------------------
// CMyDrawEllipse::CMyDrawEllipse
//----------------------------------------------------------------------------------------

CMyDrawEllipse::CMyDrawEllipse(const BR_CRectangle& newBounds,
		const BR_CPen& itsPen, BR_Boolean select):
	CMyDrawArea(newBounds, itsPen, select)
{
}

//----------------------------------------------------------------------------------------
// CMyDrawEllipse::~CMyDrawEllipse
//----------------------------------------------------------------------------------------

CMyDrawEllipse::~CMyDrawEllipse()
{
}

//----------------------------------------------------------------------------------------
// CMyDrawEllipse::Draw
//----------------------------------------------------------------------------------------

void CMyDrawEllipse::Draw(CMyDrawView* theView)
{
    BR_CGraphicsToolSelector gts(theView, &fPen);
	theView->DrawEllipse(fExtent);
 }

//========================================================================================
// CLASS CMyDrawRRect
//========================================================================================

//----------------------------------------------------------------------------------------
// CMyDrawRRect::CMyDrawRRect
//----------------------------------------------------------------------------------------

CMyDrawRRect::CMyDrawRRect(const BR_CRectangle& newBounds,
		const BR_Distance itsWidth, const BR_Distance itsHeight,
		const BR_CPen& itsPen, BR_Boolean select):
	CMyDrawArea(newBounds, itsPen, select)
{
	fWidth	= itsWidth;
	fHeight	= itsHeight;
}

//----------------------------------------------------------------------------------------
// CMyDrawRRect::~CMyDrawRRect
//----------------------------------------------------------------------------------------

CMyDrawRRect::~CMyDrawRRect()
{
}

//----------------------------------------------------------------------------------------
// CMyDrawRRect::Draw
//----------------------------------------------------------------------------------------

void CMyDrawRRect::Draw(CMyDrawView* theView)
{
    BR_CGraphicsToolSelector gts(theView, &fPen);
    theView->DrawRoundRect(fExtent, fWidth, fHeight);
}

