//========================================================================================
//
//     File:			myDrwMou.cp
//     Release Version:	$Revision$
//
//     COPYRIGHT 1993 SYMANTEC CORPORATION. ALL RIGHTS RESERVED. UNPUBLISHED -- RIGHTS
//     RESERVED UNDER THE COPYRIGHT LAWS OF THE UNITED STATES. USE OF COPYRIGHT NOTICE IS
//     PRECAUTIONARY ONLY AND DOES NOT IMPLY PUBLICATION OR DISCLOSURE.
//
//     THIS SOFTWARE CONTAINS PROPRIETARY AND CONFIDENTIAL INFORMATION OF SYMANTEC
//     CORPORATION. USE, DISCLOSURE, OR REPRODUCTION IS PROHIBITED WITHOUT THE PRIOR
//     EXPRESS WRITTEN PERMISSION OF SYMANTEC CORPORATION.
//
//     RESTRICTED RIGHTS LEGEND
//     Use, duplication, or disclosure by the Government is subject to restrictions as set
//     forth in subparagraph (c)(l)(ii) of the Rights in Technical Data and Computer
//     Software clause at DFARS 252.227-7013. Symantec Corporation, 10201 Torre Avenue,
//     Cupertino, CA 95014.
//
//========================================================================================

#ifndef   MYDRWMOU_H
#include "myDrwMou.h"
#endif

#ifndef   MYDRWDOC_H
#include "myDrwDoc.h"
#endif

#ifndef   MYDRWOBJ_H
#include "myDrwObj.h"
#endif

#ifndef   MYDRAW_K
#include "myDraw.k"
#endif

#ifndef   BRDEBUG_H
#include <BRDebug.h>
#endif

#ifndef   BRGRHELP_H
#include <BRGrHelp.h>
#endif

#ifndef   BRMINMAX_H
#include <BRMinMax.h>
#endif

#ifndef   __MATH__
#include <Math.h>
#endif

//========================================================================================
// fRunTime type information
//========================================================================================

__EXCEPTION_IMPLEMENT(XDrawObjectTooSmall, BR_XBedException)

//========================================================================================
// CLASS CMyMouseAction
//========================================================================================

//----------------------------------------------------------------------------------------
// CMyMouseAction::CMyMouseAction
//----------------------------------------------------------------------------------------

CMyMouseAction::CMyMouseAction(BR_CView* paneToTrackIn):
	BR_CMouseAction(paneToTrackIn, NULL, 0, TRUE /*notifyDocument*/, FALSE /*deleteWhenDone*/)
{
	fPen					= GetMyDrawDocument()->GetPen();
	fUndoStringModule		= paneToTrackIn->GetApplication();
	fUndoStringResourceId	= UNDO_STRINGS;
	fLastDrawObject			= NULL;
	fOwnLastDrawObject		= FALSE;	
}

//----------------------------------------------------------------------------------------
// CMyMouseAction::~CMyMouseAction
//----------------------------------------------------------------------------------------

CMyMouseAction::~CMyMouseAction()
{
	if (fOwnLastDrawObject)
		delete fLastDrawObject;
}

//----------------------------------------------------------------------------------------
// CMyMouseAction::BeginTracking
//----------------------------------------------------------------------------------------

void CMyMouseAction::BeginTracking(const BR_CPoint& startPoint)
{
	BR_CMouseAction::BeginTracking(startPoint);

	ShowFeedback();									// turn on feedback
	/*
		Note: the previous call probably won't draw anything because the mouse hasn't moved yet,
		but it's included here for completeness.
	*/
}

//----------------------------------------------------------------------------------------
// CMyMouseAction::Do
//----------------------------------------------------------------------------------------

BR_Boolean CMyMouseAction::Do()
{
	fLastDrawObject = CreateDrawObject();
	
	RestoreLastObject();
	return (TRUE);
}

//----------------------------------------------------------------------------------------
// CMyMouseAction::Draw
//----------------------------------------------------------------------------------------

void CMyMouseAction::Draw(BR_CCanvas &canvas, const BR_CRectangle& updateArea)
{
	// This assumes that the drawing will be clipped to the area that does not yet show feedback.
	// Otherwise, parts of existing feedback would be erased since feedback is drawn using XOR mode.
	
	ShowFeedback();
	BR_CMouseAction::Draw(canvas, updateArea);
}

//----------------------------------------------------------------------------------------
// CMyMouseAction::EndTracking
//----------------------------------------------------------------------------------------

void CMyMouseAction::EndTracking(const BR_CPoint& currentPoint)
{
	ShowFeedback();									// turn off feedback

	BR_CPoint constrainedPoint(currentPoint);		// Process the point where the mouse button was released
	
	if (BR_CSystem::IsShiftDown())
		DoConstrain(constrainedPoint);

	BR_CMouseAction::EndTracking(constrainedPoint);

	if (GetDistance() <= 3 * CMyDrawObject::fLeaway)
		BR_THROW(XDrawObjectTooSmall());

	Do();
}

//----------------------------------------------------------------------------------------
// CMyMouseAction::GetDescription
//----------------------------------------------------------------------------------------

void CMyMouseAction::GetDescription(BR_CString& description) const
{
	fUndoStringModule->LoadString(fUndoStringResourceId, GetDescriptionIndex(), description);  
}

//----------------------------------------------------------------------------------------
// CMyMouseAction::GetDistance
//----------------------------------------------------------------------------------------

float CMyMouseAction::GetDistance()
{
	BR_CPoint delta = GetEndingPoint() - GetStartingPoint();
	return (sqrt(pow(delta.fX, 2) + pow(delta.fY, 2)));
}

//----------------------------------------------------------------------------------------
// CMyMouseAction::GetEndingPoint
//----------------------------------------------------------------------------------------

BR_CPoint CMyMouseAction::GetEndingPoint()
{
	// Note: this method returns a stack-based copy of the point.
	// This simplifies the code.
	BR_CPoint result;
	GetPreviousPoint(result);
	
	return (result);
}

//----------------------------------------------------------------------------------------
// CMyMouseAction::GetEndingPoint
//----------------------------------------------------------------------------------------

BR_CPoint CMyMouseAction::GetStartingPoint()
{
	// Note: this method returns a stack-based copy of the point.
	// This simplifies the code.
	BR_CPoint result;
	GetStartPoint(result);
	
	return (result);
}

//----------------------------------------------------------------------------------------
// CMyMouseAction::KeepTracking
//----------------------------------------------------------------------------------------

void CMyMouseAction::KeepTracking(const BR_CPoint& currentPoint)
{
	BR_CPoint constrainedPoint(currentPoint);			// "currentPoint" is CONST, so constrain a copy.
	
	if (BR_CSystem::IsShiftDown())
		DoConstrain(constrainedPoint);

	if (GetEndingPoint() != constrainedPoint)
	{
		ShowFeedback();									// turn off old feedback
		BR_CMouseAction::KeepTracking(constrainedPoint);
		ShowFeedback();									// turn on new feedback
	}
}

//----------------------------------------------------------------------------------------
// CMyMouseAction::Redo
//----------------------------------------------------------------------------------------

BR_Boolean CMyMouseAction::Redo()
{
	RemoveLastObject();		// undo/redo meaning is inverted in this version of Bedrock.
	return (TRUE);
}

//----------------------------------------------------------------------------------------
// CMyMouseAction::RemoveLastObject
//----------------------------------------------------------------------------------------

void CMyMouseAction::RemoveLastObject()
{
	Assert(fLastDrawObject);
	fLastDrawObject->Refresh((CMyDrawView*)GetPane());
	GetMyDrawDocument()->RemoveDrawObject(fLastDrawObject);
	fOwnLastDrawObject = TRUE;
	
}

//----------------------------------------------------------------------------------------
// CMyMouseAction::RestoreLastObject
//----------------------------------------------------------------------------------------

void CMyMouseAction::RestoreLastObject()
{
	Assert(fLastDrawObject);

	GetMyDrawDocument()->DeselectAllObjects();
	fLastDrawObject->SetSelect(TRUE);				// make sure object being restored is selected

	GetMyDrawDocument()->AddDrawObject(fLastDrawObject);
	fLastDrawObject->Refresh((CMyDrawView*)GetPane());
	fOwnLastDrawObject = FALSE;
}

//----------------------------------------------------------------------------------------
// CMyMouseAction::ShowFeedback
//----------------------------------------------------------------------------------------

void CMyMouseAction::ShowFeedback()
{
	if (GetStartingPoint() != GetEndingPoint())
	{
		BR_CDrawInitiator			di(GetPane());
		BR_CDrawModeSetRestore		dmsr(GetPane(), BEDDM_XORPEN);
		BR_CGraphicsToolSelector	gts(GetPane(), &fPen);

		DoShowFeedback();
	}
}

//----------------------------------------------------------------------------------------
// CMyMouseAction::Undo
//----------------------------------------------------------------------------------------

BR_Boolean CMyMouseAction::Undo()
{
	RestoreLastObject();		// undo/redo meaning is inverted in this version of Bedrock.
	return (TRUE);
}

//========================================================================================
// CLASS CMyLineAction
//========================================================================================

//----------------------------------------------------------------------------------------
// CMyLineAction::CMyLineAction
//----------------------------------------------------------------------------------------

CMyLineAction::CMyLineAction(BR_CView* paneToTrackIn):
	CMyMouseAction(paneToTrackIn)
{
}

//----------------------------------------------------------------------------------------
// CMyLineAction::~CMyLineAction
//----------------------------------------------------------------------------------------

CMyLineAction::~CMyLineAction()
{
}

//----------------------------------------------------------------------------------------
// CMyLineAction::CreateDrawObject
//----------------------------------------------------------------------------------------

CMyDrawObject* CMyLineAction::CreateDrawObject()
{
	return (new CMyDrawLine(GetStartingPoint(), GetEndingPoint(), fPen));
}

//----------------------------------------------------------------------------------------
// CMyLineAction::DoConstrain
//----------------------------------------------------------------------------------------

void CMyLineAction::DoConstrain(BR_CPoint& currentPoint)
{
	BR_CPoint delta = currentPoint - GetStartingPoint();
	BR_CPoint deltaAbsolute;
	
	for (BR_CPoint::XYSelector sel = BR_CPoint::kX; sel <= BR_CPoint::kY; sel++)
		deltaAbsolute[sel] = abs(delta[sel]);
		
	BR_Distance maxDelta = max(deltaAbsolute.fX, deltaAbsolute.fY);

	BR_Distance halfMaxDelta = maxDelta / 2;
	
	for (sel = BR_CPoint::kX; sel <= BR_CPoint::kY; sel++)
		if (deltaAbsolute[sel] < halfMaxDelta)		// if delta is less than half of max,
			delta[sel] = 0;							// collapse in this direction
		else if (delta[sel] < 0)					// else, expand to maxDelta, preserving sign.
			delta[sel] = -maxDelta;
		else
			delta[sel] = maxDelta;
		
	currentPoint = GetStartingPoint() + delta;		// Compute constrained point
}

//----------------------------------------------------------------------------------------
// CMyLineAction::DoShowFeedback
//----------------------------------------------------------------------------------------

void CMyLineAction::DoShowFeedback()
{
	GetPane()->MovePenTo(GetStartingPoint());
	GetPane()->DrawLineTo(GetEndingPoint());
}

//----------------------------------------------------------------------------------------
// CMyLineAction::GetDescriptionIndex
//----------------------------------------------------------------------------------------

BR_CMultiStringList::StringId CMyLineAction::GetDescriptionIndex() const
{
	return (UNDO_LINE);  
}

//========================================================================================
// CLASS CMyAreaAction
//========================================================================================

//----------------------------------------------------------------------------------------
// CMyAreaAction::CMyAreaAction
//----------------------------------------------------------------------------------------

CMyAreaAction::CMyAreaAction(BR_CView* paneToTrackIn):
	CMyMouseAction(paneToTrackIn)
{
}

//----------------------------------------------------------------------------------------
// CMyAreaAction::~CMyAreaAction
//----------------------------------------------------------------------------------------

CMyAreaAction::~CMyAreaAction()
{
}

//----------------------------------------------------------------------------------------
// CMyAreaAction::DoConstrain
//----------------------------------------------------------------------------------------

void CMyAreaAction::DoConstrain(BR_CPoint& currentPoint)
{
	BR_CPoint delta = currentPoint - GetStartingPoint();
	BR_CPoint deltaAbsolute;
	
	for (BR_CPoint::XYSelector sel = BR_CPoint::kX; sel <= BR_CPoint::kY; sel++)
		deltaAbsolute[sel] = abs(delta[sel]);
		
	BR_Distance maxDelta = max(deltaAbsolute.fX, deltaAbsolute.fY); 

	// Adjust the deltas to be equal; retain the +/- sign.
	for (sel = BR_CPoint::kX; sel <= BR_CPoint::kY; sel++)
		delta[sel] = delta[sel] < 0 ? -maxDelta : maxDelta;
		
	currentPoint = GetStartingPoint() + delta;				// Compute constrained point
}

//========================================================================================
// CLASS CMyRectAction
//========================================================================================

//----------------------------------------------------------------------------------------
// CMyRectAction::CMyRectAction
//----------------------------------------------------------------------------------------

CMyRectAction::CMyRectAction(BR_CView* paneToTrackIn):
	CMyAreaAction(paneToTrackIn)
{
}

//----------------------------------------------------------------------------------------
// CMyRectAction::~CMyRectAction
//----------------------------------------------------------------------------------------

CMyRectAction::~CMyRectAction()
{
}

//----------------------------------------------------------------------------------------
// CMyRectAction::CreateDrawObject
//----------------------------------------------------------------------------------------

CMyDrawObject* CMyRectAction::CreateDrawObject()
{
	return (new CMyDrawRect(BR_CRectangle(GetStartingPoint(), GetEndingPoint()), fPen));
}

//----------------------------------------------------------------------------------------
// CMyRectAction::DoShowFeedback
//----------------------------------------------------------------------------------------

void CMyRectAction::DoShowFeedback()
{
	GetPane()->DrawRectangle(BR_CRectangle(GetStartingPoint(), GetEndingPoint()));
}

//----------------------------------------------------------------------------------------
// CMyRectAction::GetDescriptionIndex
//----------------------------------------------------------------------------------------

BR_CMultiStringList::StringId CMyRectAction::GetDescriptionIndex() const
{
	return (UNDO_RECT);  
}

//========================================================================================
// CLASS CMyEllipseAction
//========================================================================================

//----------------------------------------------------------------------------------------
// CMyEllipseAction::CMyEllipseAction
//----------------------------------------------------------------------------------------

CMyEllipseAction::CMyEllipseAction(BR_CView* paneToTrackIn):
	CMyAreaAction(paneToTrackIn)
{
}

//----------------------------------------------------------------------------------------
// CMyEllipseAction::~CMyEllipseAction
//----------------------------------------------------------------------------------------

CMyEllipseAction::~CMyEllipseAction()
{
}

//----------------------------------------------------------------------------------------
// CMyEllipseAction::CreateDrawObject
//----------------------------------------------------------------------------------------

CMyDrawObject* CMyEllipseAction::CreateDrawObject()
{
	return (new CMyDrawEllipse(BR_CRectangle(GetStartingPoint(), GetEndingPoint()), fPen));
}

//----------------------------------------------------------------------------------------
// CMyEllipseAction::DoShowFeedback
//----------------------------------------------------------------------------------------

void CMyEllipseAction::DoShowFeedback()
{
	GetPane()->DrawEllipse(BR_CRectangle(GetStartingPoint(), GetEndingPoint()));
}

//----------------------------------------------------------------------------------------
// CMyEllipseAction::GetDescriptionIndex
//----------------------------------------------------------------------------------------

BR_CMultiStringList::StringId CMyEllipseAction::GetDescriptionIndex() const
{
	return (UNDO_ELLIPSE);  
}

//========================================================================================
// CLASS CMyRRectAction
//========================================================================================

//----------------------------------------------------------------------------------------
// CMyRRectAction::CMyRRectAction
//----------------------------------------------------------------------------------------

CMyRRectAction::CMyRRectAction(BR_CView* paneToTrackIn, 
		const BR_Distance itsWidth, const BR_Distance itsHeight):
	CMyRectAction(paneToTrackIn)
{
	fWidth	= itsWidth;
	fHeight	= itsHeight;
}

//----------------------------------------------------------------------------------------
// CMyRRectAction::~CMyRRectAction
//----------------------------------------------------------------------------------------

CMyRRectAction::~CMyRRectAction()
{
}

//----------------------------------------------------------------------------------------
// CMyRRectAction::CreateDrawObject
//----------------------------------------------------------------------------------------

CMyDrawObject* CMyRRectAction::CreateDrawObject()
{
	return (new CMyDrawRRect(BR_CRectangle(GetStartingPoint(), GetEndingPoint()),
			fWidth, fHeight, fPen));
}

//----------------------------------------------------------------------------------------
// CMyRRectAction::DoShowFeedback
//----------------------------------------------------------------------------------------

void CMyRRectAction::DoShowFeedback()
{
	GetPane()->DrawRoundRect(BR_CRectangle(GetStartingPoint(), GetEndingPoint()),
				fWidth, fHeight);
}

//----------------------------------------------------------------------------------------
// CMyRRectAction::GetDescriptionIndex
//----------------------------------------------------------------------------------------

BR_CMultiStringList::StringId CMyRRectAction::GetDescriptionIndex() const
{
	return (UNDO_RRECT);  
}

//========================================================================================
// CLASS XDrawObjectTooSmall
//========================================================================================

//----------------------------------------------------------------------------------------
// XDrawObjectTooSmall::XDrawObjectTooSmall
//----------------------------------------------------------------------------------------

XDrawObjectTooSmall::XDrawObjectTooSmall()
{
}

//----------------------------------------------------------------------------------------
// XDrawObjectTooSmall::~XDrawObjectTooSmall
//----------------------------------------------------------------------------------------

XDrawObjectTooSmall::~XDrawObjectTooSmall()
{
}

