#ifndef BRSTRTOO_H
#define BRSTRTOO_H
//========================================================================================
//
//     File:		BRStrToo.h
//     Release Version:	$ 1.0d1 $
//
//     Creation Date:	4/15/93
//
//     COPYRIGHT 1993 SYMANTEC CORPORATION. ALL RIGHTS RESERVED. UNPUBLISHED -- RIGHTS
//     RESERVED UNDER THE COPYRIGHT LAWS OF THE UNITED STATES. USE OF COPYRIGHT NOTICE IS
//     PRECAUTIONARY ONLY AND DOES NOT IMPLY PUBLICATION OR DISCLOSURE.
//
//     THIS SOFTWARE CONTAINS PROPRIETARY AND CONFIDENTIAL INFORMATION OF SYMANTEC
//     CORPORATION. USE, DISCLOSURE, OR REPRODUCTION IS PROHIBITED WITHOUT THE PRIOR
//     EXPRESS WRITTEN PERMISSION OF SYMANTEC CORPORATION.
//
//     RESTRICTED RIGHTS LEGEND
//     Use, duplication, or disclosure by the Government is subject to restrictions as set
//     forth in subparagraph (c)(l)(ii) of the Rights in Technical Data and Computer
//     Software clause at DFARS 252.227-7013. Symantec Corporation, 10201 Torre Avenue,
//     Cupertino, CA 95014.
//
//========================================================================================

#ifndef   BRAUTODE_H
#include "BRAutoDe.h"
#endif

#ifndef BRTSTRS_H
#include "BRTStrs.h"
#endif

enum BR_StringCompareResult
{
	kStringOneLess = -1, kStringsEqual = 0, kStringOneGreater = 1
} ;

//========================================================================================
//	CLASS BR_TStringTool
//
//		A tool for string operations.  StringTools are used for operations that don't fit
//		well as methods of strings.  The principle reason for placing an operation in a 
//		tool instead of a string is when the operation may have multiple algorithm 
//		implementations.  We make this distinction because it is undesirable to force
//		a user to subclass one or more string classes just to use a different set of
//		tool algorithms.
//
//		The BR_TStringTool class is an abstract base class.
//========================================================================================

template <class tCharacter>
class BR_TStringTool : public _BR_CAutoDestructObject
{
public:

	virtual ~BR_TStringTool();
	BR_TStringTool();
	
	virtual BR_StringCompareResult CompareStrings(BR_TTextReader<tCharacter> &reader1,
						 				 BR_TTextReader<tCharacter> &reader2) = 0;
										 
	virtual BR_Boolean FindSubString(const BR_TTextReader<tCharacter> &reader,
									 const BR_TString<tCharacter> &subString,
									 BR_CharacterPosition &foundPosition) = 0;
									 
	virtual BR_Boolean FindCharacter(BR_TTextReader<tCharacter> &reader,
									tCharacter character,
									BR_CharacterPosition &foundPosition) = 0;
								 
	virtual BR_Boolean FindWhiteSpace(BR_TTextReader<tCharacter> &reader,
										BR_CharacterPosition &foundPosition) = 0;
								 
	virtual BR_Boolean FindNonWhiteSpace(BR_TTextReader<tCharacter> &reader,
										BR_CharacterPosition &foundPosition) = 0;

	virtual void Substitute(const BR_TTextReader<tCharacter> &reader,
							const BR_TString<tCharacter> &searchString,
							const BR_TString<tCharacter> &substitutionString) = 0;
	
	virtual BR_Boolean IsWhiteSpace(tCharacter character) = 0;
								 
	virtual void ToUpper(BR_TTextFilter<tCharacter> &string) = 0;
	
	virtual void ToLower(BR_TTextFilter<tCharacter> &string) = 0;

	virtual void ConvertToChar(const tCharacter* source, BR_Char* destination, BR_CharacterCount items) = 0;
	
	virtual void ConvertToWideChar(const tCharacter* source, BR_WideChar* destination, BR_CharacterCount items) = 0;

	// Non-virtual functions implemented using above virtual functions.

	BR_StringCompareResult CompareStrings(const BR_TString<tCharacter> &string1,
						 				 const BR_TString<tCharacter> &string2);
	
	BR_Boolean FindSubString(const BR_TString<tCharacter> &string,
							const BR_TString<tCharacter> &subString,
							BR_CharacterPosition &foundPosition,
							BR_CharacterPosition startPosition=0);
									 
	BR_Boolean FindCharacter(const BR_TString<tCharacter> &string,
							tCharacter character,
							BR_CharacterPosition &foundPosition,
							BR_CharacterPosition startPosition=0);
									
	BR_Boolean FindWhiteSpace(const BR_TString<tCharacter> &string,
							BR_CharacterPosition &foundPosition,
							BR_CharacterPosition startPosition=0);
										 
	BR_Boolean FindNonWhiteSpace(const BR_TString<tCharacter> &string,
								 BR_CharacterPosition &foundPosition,
								 BR_CharacterPosition startPosition=0);
										 
	void Substitute(BR_TString<tCharacter> &string,
					const BR_TString<tCharacter> &searchString,
					const BR_TString<tCharacter> &substitutionString);
	
	void ToUpper(BR_TString<tCharacter> &string);
	
	void ToLower(BR_TString<tCharacter> &string);
									 
	// Static functions for accessing default and current string tools.

	static BR_TStringTool<tCharacter>* GetCurrentStringTool();
		// Return a pointer to the current string tool.
		
	static BR_TStringTool<tCharacter>* GetDefaultStringTool();
		// Return a pointer to the default string tool.
		
#ifdef qTemplateClassGlobalBugIsFixed
	static BR_TStringTool<tCharacter>* SetCurrentStringTool(BR_TStringTool<tCharacter>* tool);
		// Make tool the current string tool.
		// Returns the prior string tool.

	static void AdoptDefaultStringTool(BR_TStringTool<tCharacter>* tool);
		// Make tool the default string tool.
		// Returns the prior default string tool.
#endif

private:

#ifdef qTemplateClassGlobalBugIsFixed
	static BR_TStringTool<tCharacter>*	gCurrentStringTool;
	static BR_TStringTool<tCharacter>*	gDefaultStringTool;
#endif

};

//========================================================================================
//	CLASS BR_TMinimalStringTool
//
//		A minimalist, braindead string tool.  Operations are done without any knowledge
//		of language, locale, character set, case sensitivity, etc.  This tool is provided
//		for quick and dirty use where lack of such knowledge is not considered a problem.
//		Real applications will need to use more sophisticated string tools, provided in
//		other, higher level components.
//========================================================================================

template <class tCharacter>
class BR_TMinimalStringTool : public BR_TStringTool<tCharacter>
{
public:

	virtual ~BR_TMinimalStringTool();
	BR_TMinimalStringTool();

	virtual void ConvertToChar(const tCharacter* source, BR_Char* destination, BR_CharacterCount items);
	
	virtual void ConvertToWideChar(const tCharacter* source, BR_WideChar* destination, BR_CharacterCount items);

	virtual BR_StringCompareResult CompareStrings(BR_TTextReader<tCharacter> &reader1,
						 				 BR_TTextReader<tCharacter> &reader2);
										 
	virtual BR_Boolean FindSubString(const BR_TTextReader<tCharacter> &reader,
									 const BR_TString<tCharacter> &subString,
									 BR_CharacterPosition &foundPosition);
									 
	virtual BR_Boolean FindCharacter(BR_TTextReader<tCharacter> &reader,
									tCharacter character,
									BR_CharacterPosition &foundPosition);
								 
	virtual BR_Boolean FindWhiteSpace(BR_TTextReader<tCharacter> &reader,
										BR_CharacterPosition &foundPosition);
								 
	virtual BR_Boolean FindNonWhiteSpace(BR_TTextReader<tCharacter> &reader,
										BR_CharacterPosition &foundPosition);
	
	virtual void Substitute(const BR_TTextReader<tCharacter> &reader,
							const BR_TString<tCharacter> &searchString,
							const BR_TString<tCharacter> &substitutionString);
	
	virtual BR_Boolean IsWhiteSpace(tCharacter character);
								 
	virtual void ToUpper(BR_TTextFilter<tCharacter> &string);
	
	virtual void ToLower(BR_TTextFilter<tCharacter> &string);
	
	// Non-virtual functions implemented using above virtual functions.

	BR_StringCompareResult CompareStrings(const BR_TString<tCharacter> &string1,
						 				 const BR_TString<tCharacter> &string2);
	
	BR_Boolean FindSubString(const BR_TString<tCharacter> &string,
							const BR_TString<tCharacter> &subString,
							BR_CharacterPosition &foundPosition,
							BR_CharacterPosition startPosition=0);
									 
	BR_Boolean FindCharacter(const BR_TString<tCharacter> &string,
							tCharacter character,
							BR_CharacterPosition &foundPosition,
							BR_CharacterPosition startPosition=0);
									
	BR_Boolean FindWhiteSpace(const BR_TString<tCharacter> &string,
							BR_CharacterPosition &foundPosition,
							BR_CharacterPosition startPosition=0);
										 
	BR_Boolean FindNonWhiteSpace(const BR_TString<tCharacter> &string,
								BR_CharacterPosition &foundPosition,
								BR_CharacterPosition startPosition=0);
										 
	void Substitute(BR_TString<tCharacter> &string,
					const BR_TString<tCharacter> &searchString,
					const BR_TString<tCharacter> &substitutionString);
	
	void ToUpper(BR_TString<tCharacter> &string);
	
	void ToLower(BR_TString<tCharacter> &string);
									 
};

//----------------------------------------------------------------------------------------
//	BR_TMinimalStringTool<tCharacter>::BR_TMinimalStringTool
//----------------------------------------------------------------------------------------

template <class tCharacter>
inline BR_StringCompareResult BR_TMinimalStringTool<tCharacter>::CompareStrings(
									const BR_TString< tCharacter>& string1,
									const BR_TString< tCharacter>& string2)
{
	return BR_TStringTool<tCharacter>::CompareStrings(string1, string2);
}

//----------------------------------------------------------------------------------------
//	BR_TMinimalStringTool<tCharacter>::FindSubString
//----------------------------------------------------------------------------------------

template <class tCharacter>
inline BR_Boolean BR_TMinimalStringTool<tCharacter>::FindSubString(
						const BR_TString<tCharacter>& string,
						 const BR_TString< tCharacter>& subString,
						 BR_CharacterPosition& foundPosition,
						 BR_CharacterPosition startPosition)
{
	return BR_TStringTool<tCharacter>::FindSubString(string, subString, 
													foundPosition, startPosition);
}

//----------------------------------------------------------------------------------------
//	BR_TMinimalStringTool<tCharacter>::FindCharacter
//----------------------------------------------------------------------------------------

template <class tCharacter>
inline BR_Boolean BR_TMinimalStringTool<tCharacter>::FindCharacter(
						const BR_TString<tCharacter>& string,
						 tCharacter character,
						 BR_CharacterPosition& foundPosition,
						 BR_CharacterPosition startPosition)
{
	return BR_TStringTool<tCharacter>::FindCharacter(string, character, 
													foundPosition, startPosition);
}

//----------------------------------------------------------------------------------------
//	BR_TMinimalStringTool<tCharacter>::FindWhiteSpace
//----------------------------------------------------------------------------------------

template <class tCharacter>
inline BR_Boolean BR_TMinimalStringTool<tCharacter>::FindWhiteSpace(
						const BR_TString<tCharacter>& string,
						  BR_CharacterPosition& foundPosition,
						  BR_CharacterPosition startPosition)
{
	return BR_TStringTool<tCharacter>::FindWhiteSpace(string, foundPosition, startPosition);
}

//----------------------------------------------------------------------------------------
//	BR_TMinimalStringTool<tCharacter>::FindNonWhiteSpace
//----------------------------------------------------------------------------------------

template <class tCharacter>
inline BR_Boolean BR_TMinimalStringTool<tCharacter>::FindNonWhiteSpace(
							const BR_TString<tCharacter>& string,
							 BR_CharacterPosition& foundPosition,
							 BR_CharacterPosition startPosition)
{
	return BR_TStringTool<tCharacter>::FindNonWhiteSpace(string, 
														foundPosition, 
														startPosition);
}

#endif
