#if !defined(BRREGION_H) && !defined(__BEDRC__)
#define BRREGION_H
//========================================================================================
//
//     File:			BRRegion.h
//     Release Version:	$ 1.0d1 $
//
//     Creation Date:	11/24/90
//
//     COPYRIGHT 1990-1993 SYMANTEC CORPORATION. ALL RIGHTS RESERVED. UNPUBLISHED -- RIGHTS
//     RESERVED UNDER THE COPYRIGHT LAWS OF THE UNITED STATES. USE OF COPYRIGHT NOTICE IS
//     PRECAUTIONARY ONLY AND DOES NOT IMPLY PUBLICATION OR DISCLOSURE.
//
//     THIS SOFTWARE CONTAINS PROPRIETARY AND CONFIDENTIAL INFORMATION OF SYMANTEC
//     CORPORATION. USE, DISCLOSURE, OR REPRODUCTION IS PROHIBITED WITHOUT THE PRIOR
//     EXPRESS WRITTEN PERMISSION OF SYMANTEC CORPORATION.
//
//     RESTRICTED RIGHTS LEGEND
//     Use, duplication, or disclosure by the Government is subject to restrictions as set
//     forth in subparagraph (c)(l)(ii) of the Rights in Technical Data and Computer
//     Software clause at DFARS 252.227-7013. Symantec Corporation, 10201 Torre Avenue,
//     Cupertino, CA 95014.
//
//========================================================================================

#ifndef BCCOLL_H
#include "BCColl.h"
#endif

#ifndef BRGRAPHT_H
#include "BRGraphT.h"
#endif

#ifndef BRRECT_H
#include "BRRect.h"
#endif

#if defined(BR_BUILD_MAC) && !defined(__QUICKDRAW__)
#include <QuickDraw.h>
#endif

//========================================================================================
//	Forward class declaration
//========================================================================================

class BR_CView;


//========================================================================================
// BR_CRegion
//
// An abstract class that implements regions (a region is an abstract
// area, within a window, that can be filled with graphical output).
//========================================================================================

class BR_CRegion : public BR_CGraphicsTool
{
private:
	friend BR_CGraphicsToolSelector;
	friend BR_CCanvas;

public:
	virtual~ BR_CRegion();
	BR_CView* GetPane() const;

	void GetBounds(BR_CRectangle& rect);

	BR_Boolean IsPointIn(const BR_CPoint& point);
	BR_Boolean IsRectIn(const BR_CRectangle& rect);
	void Offset(BR_Distance X,
				BR_Distance Y);
	void Union(const BR_CRegion& region);
	void Intersect(const BR_CRegion& region);
	void Xor(const BR_CRegion& region);
	void Difference(const BR_CRegion& region);
	void Copy(const BR_CRegion& region);

	BR_CRegion& operator=(const BR_CRegion& region);
	BR_CRegion& operator|=(const BR_CRegion& region);
	BR_CRegion& operator-=(const BR_CRegion& region);
	BR_CRegion& operator^=(const BR_CRegion& region);
	BR_CRegion& operator&=(const BR_CRegion& region);

	BR_Boolean operator==(const BR_CRegion& region);
	BR_Boolean operator!=(const BR_CRegion& region);

protected:
	BR_CRegion(BR_CView *pane);
	virtual void SetupTool(BR_CView& pane,
						   GraphicsToolSelectionOptions selectionOptions,
						   BR_SGraphicsToolSuspendResumeState* state);
	virtual void RestoreNullTool(BR_CView& pane);
	virtual BR_CGraphicsTool* GetCurrentTool(const BR_CCanvas& canvas) const;
	virtual void SetCurrentTool(BR_CCanvas& canvas,
								BR_CGraphicsTool* graphicsTool) const;

private:
	BR_CView* fPane;							// The pane that the region is in

	//----------------------------------------------------------------------------------------
	// Macintosh
	//----------------------------------------------------------------------------------------

#ifdef BR_BUILD_MAC
public:
	void SetHandle(RgnHandle hNew);
	RgnHandle GetHandle() const;

private:
	RgnHandle fRegion;							// the handle for Get/Set Handle
#endif

	//----------------------------------------------------------------------------------------
	// Windows
	//----------------------------------------------------------------------------------------

#ifdef BR_BUILD_WIN
public:
	void SetHandle(HANDLE hNew);
	HANDLE GetHandle() const;
	virtual BR_Boolean BedWinIsRectRegion() const;

protected:
	BR_CRegion& Combine(const BR_CRegion& Region,
						const iMode);
	void BedWinSetGDIHandle(HANDLE region);		// Throws if region handle is NULL

private:
	HANDLE fRegion;								// the handle for Get/Set Handle

#endif

	BR_SETCLASSNAME(BR_CRegion);
};

//----------------------------------------------------------------------------------------
// BR_CRegion::GetPane
//----------------------------------------------------------------------------------------
inline BR_CView* BR_CRegion::GetPane() const
{
	return fPane;
}

//----------------------------------------------------------------------------------------
// BR_CRegion::operator!=
//----------------------------------------------------------------------------------------
inline BR_Boolean BR_CRegion::operator!=(const BR_CRegion& Region)
{
	return (!(*this == Region));
}

#ifdef BR_BUILD_WIN
//----------------------------------------------------------------------------------------
// BR_CRegion::SetHandle
//----------------------------------------------------------------------------------------
inline void BR_CRegion::SetHandle(HANDLE hNew)
{
	fRegion = hNew;
}
#endif

#ifdef BR_BUILD_WIN
//----------------------------------------------------------------------------------------
// BR_CRegion::SetHandle
//----------------------------------------------------------------------------------------
inline HANDLE BR_CRegion::GetHandle() const
{
	return fRegion;
}
#endif

#ifdef BR_BUILD_MAC
//----------------------------------------------------------------------------------------
// BR_CRegion::SetHandle
//----------------------------------------------------------------------------------------
inline void BR_CRegion::SetHandle(RgnHandle hNew)
{
	fRegion = hNew;
}
#endif

#ifdef BR_BUILD_MAC
//----------------------------------------------------------------------------------------
// BR_CRegion::GetHandle
//----------------------------------------------------------------------------------------
inline RgnHandle BR_CRegion::GetHandle() const
{
	return fRegion;
}
#endif

//----------------------------------------------------------------------------------------
// BR_CRegion::operator=
//----------------------------------------------------------------------------------------
inline BR_CRegion& BR_CRegion::operator=(const BR_CRegion& Region)
{
#ifdef BR_BUILD_MAC
	::CopyRgn(Region.GetHandle(), GetHandle());
	return *this;
#endif

#ifdef BR_BUILD_WIN
	return (Combine(Region, RGN_COPY));
#endif

}

//----------------------------------------------------------------------------------------
// BR_CRegion::operator|=
//----------------------------------------------------------------------------------------
inline BR_CRegion& BR_CRegion::operator|=(const BR_CRegion& Region)
{
#ifdef BR_BUILD_MAC
	::UnionRgn(GetHandle(), Region.GetHandle(), GetHandle());
	return *this;
#endif

#ifdef BR_BUILD_WIN
	return (Combine(Region, RGN_OR));
#endif

}

//----------------------------------------------------------------------------------------
// BR_CRegion::operator-=
//----------------------------------------------------------------------------------------
inline BR_CRegion& BR_CRegion::operator-=(const BR_CRegion& Region)
{
#ifdef BR_BUILD_MAC
	::DiffRgn(GetHandle(), Region.GetHandle(), GetHandle());
	return *this;
#endif

#ifdef BR_BUILD_WIN
	return (Combine(Region, RGN_DIFF));
#endif

}

//----------------------------------------------------------------------------------------
// BR_CRegion::operator^=
//----------------------------------------------------------------------------------------
inline BR_CRegion& BR_CRegion::operator^=(const BR_CRegion& Region)
{
#ifdef BR_BUILD_MAC
	::XorRgn(GetHandle(), Region.GetHandle(), GetHandle());
	return *this;
#endif

#ifdef BR_BUILD_WIN
	return (Combine(Region, RGN_XOR));
#endif

}

//----------------------------------------------------------------------------------------
// BR_CRegion::operator&=
//----------------------------------------------------------------------------------------
inline BR_CRegion& BR_CRegion::operator&=(const BR_CRegion& Region)
{
#ifdef BR_BUILD_MAC
	::SectRgn(GetHandle(), Region.GetHandle(), GetHandle());
	return *this;
#endif

#ifdef BR_BUILD_WIN
	return (Combine(Region, RGN_AND));
#endif

}

// these are inline because they are defined in terms of the
// inlines above

//----------------------------------------------------------------------------------------
// BR_CRegion::Union
//----------------------------------------------------------------------------------------
inline void BR_CRegion::Union(const BR_CRegion& Region)
{
	*this |= Region;
}

//----------------------------------------------------------------------------------------
// BR_CRegion::Intersect
//----------------------------------------------------------------------------------------
inline void BR_CRegion::Intersect(const BR_CRegion& Region)
{
	*this &= Region;
}

//----------------------------------------------------------------------------------------
// BR_CRegion::Xor
//----------------------------------------------------------------------------------------
inline void BR_CRegion::Xor(const BR_CRegion& Region)
{
	*this ^= Region;
}

//----------------------------------------------------------------------------------------
// BR_CRegion::Difference
//----------------------------------------------------------------------------------------
inline void BR_CRegion::Difference(const BR_CRegion& Region)
{
	*this -= Region;
}

//----------------------------------------------------------------------------------------
// BR_CRegion::Copy
//----------------------------------------------------------------------------------------
inline void BR_CRegion::Copy(const BR_CRegion& Region)
{
	*this = Region;
}

//========================================================================================
// BR_CRectRegion
//
// A class used to define a rectangular region
//========================================================================================

class BR_CRectRegion : public BR_CRegion
{
public:
	BR_CRectRegion(BR_CView *pane,
					const BR_CRectangle& aRect);
	BR_CRectRegion(BR_CView *pane,
				   BR_Coordinate Left,
				   BR_Coordinate Top,
				   BR_Coordinate Right,
				   BR_Coordinate Bottom);
	virtual~ BR_CRectRegion();

	BR_SETCLASSNAME(BR_CRectRegion);

#ifdef BR_BUILD_WIN	
	virtual BR_Boolean BedWinIsRectRegion() const;
#endif

};

//========================================================================================
// BR_CRoundRectRegion
//
// A class used to define a rectangular region with rounded corners
///========================================================================================
class BR_CRoundRectRegion : public BR_CRegion
{
public:
	BR_CRoundRectRegion(BR_CView *pane,
						 const BR_CRectangle& aRect,
						 BR_Distance fWidth,
						 BR_Distance fHeight);
	BR_CRoundRectRegion(BR_CView *pane,
						 BR_Coordinate Left,
						 BR_Coordinate Top,
						 BR_Coordinate Right,
						 BR_Coordinate Bottom,
						 BR_Distance fWidth,
						 BR_Distance fHeight);
	virtual~ BR_CRoundRectRegion();

public:
#ifdef BR_BUILD_MAC
	void MakeRegion(const Rect& aRect,
					short h,
					short v);
#endif

	BR_SETCLASSNAME(BR_CRoundRectRegion);
};

//========================================================================================
// BR_CEllipseRegion
//
// A class used to define an elliptical region
//========================================================================================
class BR_CEllipseRegion : public BR_CRegion
{
public:
	BR_CEllipseRegion(BR_CView *pane,
						const BR_CRectangle& aRect);
	BR_CEllipseRegion(BR_CView *pane,
						BR_Coordinate Left,
						BR_Coordinate Top,
						BR_Coordinate Right,
						BR_Coordinate Bottom);
	virtual~ BR_CEllipseRegion();

public:											// Implementation
#ifdef BR_BUILD_MAC
	void MakeRegion(const Rect& aRect);
#endif

	BR_SETCLASSNAME(BR_CEllipseRegion);
};

//========================================================================================
// BR_CPolygonRegion
//
// A class used to define a polygon region
//========================================================================================
class BR_CPolygonRegion : public BR_CRegion
{
public:
	BR_CPolygonRegion(BR_CView *pane,
					  const BC_TCollection<BR_CPoint>& points);
	virtual~ BR_CPolygonRegion();

	BR_SETCLASSNAME(BR_CPolygonRegion);
};

#endif


