#if !defined(BRFILE_H) && !defined(__BEDRC__)
#define BRFILE_H
//========================================================================================
//
//     File:		BRFile.h
//     Release Version:	$ 1.0d1 $
//
//     Creation Date:	10/22/90
//
//     COPYRIGHT 1990-93 SYMANTEC CORPORATION. ALL RIGHTS RESERVED. UNPUBLISHED -- RIGHTS
//     RESERVED UNDER THE COPYRIGHT LAWS OF THE UNITED STATES. USE OF COPYRIGHT NOTICE IS
//     PRECAUTIONARY ONLY AND DOES NOT IMPLY PUBLICATION OR DISCLOSURE.
//
//     THIS SOFTWARE CONTAINS PROPRIETARY AND CONFIDENTIAL INFORMATION OF SYMANTEC
//     CORPORATION. USE, DISCLOSURE, OR REPRODUCTION IS PROHIBITED WITHOUT THE PRIOR
//     EXPRESS WRITTEN PERMISSION OF SYMANTEC CORPORATION.
//
//     RESTRICTED RIGHTS LEGEND
//     Use, duplication, or disclosure by the Government is subject to restrictions as set
//     forth in subparagraph (c)(l)(ii) of the Rights in Technical Data and Computer
//     Software clause at DFARS 252.227-7013. Symantec Corporation, 10201 Torre Avenue,
//     Cupertino, CA 95014.
//
//========================================================================================

#ifndef BRAUTODE_H
#include "BRAutoDe.h"
#endif

#ifndef BRBEDEXC_H
#include "BRBedExc.h"
#endif

#ifndef BRSTREAM_H
#include "BRStream.h"
#endif

#if defined(BR_BUILD_MAC) && !defined(__FILES__)
#include "Files.h"
#endif

#ifdef BR_BUILD_WIN
#include "windows.h"
#endif


//========================================================================================
//	Forward Class Declarations
//========================================================================================
class BR_CFileSpecification;


//========================================================================================
//  Constants
//========================================================================================

#ifdef BR_BUILD_MAC
const short int kFileClosed = -1;
#endif
#ifdef BR_BUILD_WIN
const HFILE kFileClosed = -1;
#endif



//========================================================================================
//	BR_CAccessPermission
//
//	Static class which defines some common file operations.  Should be nested inside of
//    BR_CFile, but CFront was having troubles.
//========================================================================================

class BR_CAccessPermission
{
public:
	enum Access
	{
		kRead, kWrite, kReadWrite
	};


	enum Deny
	{
		kDenyNone, kDenyRead, kDenyWrite, kDenyReadWrite
	};


	BR_CAccessPermission(Access access = kReadWrite,
					 Deny deny = kDenyReadWrite);
	void GetPermission(Access& access,
				 	   Deny& deny) const;

private:
	Access fAccess;
	Deny fDeny;
};



//========================================================================================
//	BR_CFile
//
//	Static class which defines some common file operations.
//========================================================================================

class BR_CFile
{
public:	
	enum FileAttributes
	{
		kReadOnly   = 0x0001,
		kHidden     = 0x0002,
		kSystem     = 0x0004,
		kDirectory  = 0x0010,
		kArchive    = 0x0020
	};
	
typedef unsigned long int FileMarkPosition;

#ifdef BR_BUILD_WIN
	enum MoveMethods
	{
		kFromStart = 0,
		kFromCurrent = 1,
		kFromEnd = 2
	};
#endif

#ifdef BR_BUILD_MAC
	enum MoveMethods
	{
		kFromStart = fsFromStart,
		kFromCurrent = fsAtMark,
		kFromEnd = fsFromLEOF,
		kFromMark = fsFromMark,			// Sets the mark relative to the current mark.
		kVerifyRead = rdVerify
	};
#endif

	static void Create(BR_CFileSpecification& fileSpec);
	static void Delete(BR_CFileSpecification& fileSpec);
};


//========================================================================================
//	BR_CAbstractFileAccess
//
//	Abstract class which defines a protocol for getting information from a file.  
//========================================================================================

class BR_CAbstractFileAccess : public _BR_CAutoDestructObject
{
public:
	typedef unsigned long BufferLength;

	virtual~ BR_CAbstractFileAccess();

	virtual void Read(void* buffer,
					    BufferLength& length) = 0;
	// Read length bytes into buffer from file.

	virtual void Write(void* buffer,
					     BufferLength& length) = 0;
	// Write length bytes from buffer into file.

	virtual void GetLength(BufferLength& length) = 0;
	// Return the logical size of file

	virtual void SetLength(BufferLength length) = 0;
	// Set the logical size of file

	virtual void GetCurrentPosition(BufferLength& length) = 0;
	// Return current position

	virtual void SetCurrentPosition(BufferLength length) = 0;
	// Set the position for furture reads or writes

protected:
	BR_CAbstractFileAccess() {}
};


//========================================================================================
//	BR_CFileAccess
//
//	Concrete class which accesses information within a disk-based file.  The file is
//    automatically opened in the constructor with the specified permissions.  The file
//    is closed again in the destructor.
//========================================================================================
class BR_CFileAccess : public BR_CAbstractFileAccess
{
public:

	BR_CFileAccess(const BR_CFileSpecification& fileSpecification);
	// Open with default permissions: exclusive read/write

	BR_CFileAccess(const BR_CFileSpecification& fileSpecification,
				   const BR_CAccessPermission& permission);
	// Open with specified permissions
	
	virtual ~BR_CFileAccess();
	// Close the file.

	virtual void Read(void* buffer,
					    BufferLength& length);
	// Read length bytes into buffer from file.

	virtual void Write(void* buffer,
					     BufferLength& length);
	// Write length bytes from buffer into file.

	virtual void GetLength(BufferLength& length);
	// Return the logical size of file

	virtual void SetLength(BufferLength length);
	// Set the logical size of file

	virtual void GetCurrentPosition(BufferLength& length);
	// Return current position

	virtual void SetCurrentPosition(BufferLength length);
	// Set the position for furture reads or writes

	BR_Boolean BR_CFileAccess::operator== (const BR_CFileAccess& theOtherFile);
	BR_CFileAccess& BR_CFileAccess::operator= (const BR_CFileAccess& theOtherFile);
	
private:
	void BedOpenFile(const BR_CAccessPermission& permission);
	void BedCloseFile();
	
	BR_CFileAccess(const BR_CFileAccess&);

	BR_CFileSpecification* fFileSpec;

#ifdef BR_BUILD_MAC
public:
	void MacGetFileReferenceNumber(short& fileReferenceNumber) const;

private:
	short fFileReference;
#endif 

#ifdef BR_BUILD_WIN
public:
	void WinGetFileHandle(HFILE& fileHandle) const;

private:
	HFILE fFileReference;
#endif

};


//========================================================================================
//  CLASS BR_CFileAccess
//========================================================================================

#ifdef BR_BUILD_MAC
//----------------------------------------------------------------------------------------
//	BR_CFileAccess::MacGetFileReferenceNumber
//
//	<comments>
//----------------------------------------------------------------------------------------

inline void BR_CFileAccess::MacGetFileReferenceNumber(short& fileReferenceNumber) const
{
	fileReferenceNumber = fFileReference;
}
#endif


#ifdef BR_BUILD_WIN
//----------------------------------------------------------------------------------------
//	BR_CFileAccess::WinGetFileHandle
//
//	<comments>
//----------------------------------------------------------------------------------------

inline void BR_CFileAccess::WinGetFileHandle(HFILE& fileHandle) const
{
	fileHandle = fFileReference;
}
#endif


//========================================================================================
//  CLASS BR_CFileStream
//
//  This class is intended as a transition object into the new stream model.  This class
//    implements a stream derived from a disk-based file. 
//  NOTE: This class will not be supported in future versions of Bedrock.
//========================================================================================

class BR_CFileStream : public BR_CStream
{
public:
		BR_CFileStream::BR_CFileStream(const BR_CFileSpecification& fileSpec);
		virtual ~BR_CFileStream();
		
		virtual BR_Boolean Read(void* buffer, StreamBufferSize& nBytes);
		virtual BR_Boolean Write(void* buffer, StreamBufferSize& nBytes);

		virtual StreamSize GetLength() const;
		virtual BR_Boolean SetLength(StreamSize newLength);
		virtual StreamIndex GetPosition() const;
		virtual StreamIndex SetPosition(StreamIndex newPosition);
		
		BR_CFileSpecification* GetFileSpec() const;
		
private:
		BR_CFileSpecification* fFileSpec;
		
		BR_CFileStream(const BR_CFileStream&);

		BR_SETCLASSNAME(BR_CFileStream);
};


//----------------------------------------------------------------------------------------
//	BR_CFileStream::GetFileSpec
//----------------------------------------------------------------------------------------

inline BR_CFileSpecification* BR_CFileStream::GetFileSpec() const
{
	return (fFileSpec);
}


//========================================================================================
//  CLASS BR_XPlatformError
//========================================================================================

class BR_XPlatformError : public BR_XBedException
{
public:
	BR_XPlatformError(PlatformError theError);
	virtual ~BR_XPlatformError();
	
	_BR_EXCEPTION_DEFINE(BR_XPlatformError);
};

#endif


