/*
 * $Header:   J:/22vcs/srccmd/samples.win/uecho.c_v   1.4   20 Jul 1992 19:52:48   paul  $
 */

/*
 * Copyright (C) 1991-1992 by FTP Software, Inc.
 * 
 * This software is furnished under a license and may be used and copied
 * only in accordance with the terms of such license and with the
 * inclusion of the above copyright notice. This software or any other
 * copies thereof may not be provided or otherwise made available to any
 * other person. No title to and ownership of the software is hereby
 * transferred.
 * 
 * The information in this software is subject to change without notice
 * and should not be construed as a commitment by FTP Software, Inc.
 *
 * 
 * EDIT HISTORY:
 * 05-Sep-90  msd@ayuda	Original author.
 * 04-Oct-90  msd@ayuda	First pre-release.
 * 01-Apr-91  msd@ayuda	To FTP for 2.05 pl 2 beta.
 * 06-May-92  ftp	DevKit 2.1 beta.
 */

/* "uecho.c" -- Simple test application for PC/TCP under Windows 3.x.
   Implements a crude UDP ECHO client. */

/* Disclaimer: FTP Software provides these sample Windows applications as 
   simple examples of how the PC/TCP PCTCPAPI.DLL can be used. We cannot 
   ensure that all coding conventions used herein are appropriate for all 
   Windows applications. Please consult an official Windows programming
   reference (i.e. Microsoft SDK) for recommended coding conventions which
   best suit your needs.
*/
   
#include <stdio.h>
#include <dos.h>
#include <memory.h>
#include <string.h>
#include <windows.h>
#include <pctcp/winapp.h>
#include <pctcp/dns_lib.h>
#include <pctcp/asynch.h>
#include <pctcp/ipconfig.h>		/* struct opt_port */
#include <pctcp/config.h>
#include "cgarf.h"
#include "fmters.h"
#include "plaints.h"
#include "uecho.h"

#ifndef	OURPORT
#define	OURPORT		7	/* default is standard ECHO service port # */
#endif

#ifndef	MSGLEN
#define	MSGLEN		666	/* default size of echo datagrams */
#endif
#define	PINGLEN		176	/* (default) size of PINGs */

#define MSGTIMER	2	/* # secs to allow datagram replies */
#define	PAINTTIMER	10	/* # secs between repaints and PING retries */

/* phases of application execution */
#define	INIT2		1	/* completing hostname resolution */
#define	DATA		2	/* rcv/send UDP echo packets & PINGs */
#define	DESTROYED	3	/* under renovation */


/* local function prototypes */

int InitNet1();
int InitNet2();
void PaintLine(char *, HDC, TEXTMETRIC *, int *, int *);
void PaintWnd(HWND);
int ParseCmdLine(LPSTR);
void PingRcv();
int ProcessDatagrams();
void Repaint();
void Send1Datagram();


/* global / local variables */

HANDLE hInst;			/* current application instance */
HWND hOurWnd = NULL;		/* application's root window handle */

char aname[128];		/* argument peer hostname / address */
char cname[128];		/* peer host's canonical name */
unsigned dllVsn;		/* PCTCPAPI.DLL version # */
struct whostent DLLFAR *hostP = 0; /* hostname control structure (in the DLL) */
in_name ipAddr;			/* net descriptor's local IP address */
struct addr msgAddr;		/* addresses */
int msgCheck;			/* need to check for datagram read */
int msgLen = MSGLEN;		/* length to send / receive */
int msgOutstanding;		/* is a datagram outstanding now? */
int nd = -1;			/* datagram net descriptor */
int netPhase;			/* how far network initialization has gone */
unsigned netVsn;		/* PC/TCP version # */
int pingAgain;			/* want to send another PING soon? */
int pingCheck;			/* need to check for PING reply */
char FAR *pingBufP;		/* buffer to use to send next PING */
int pingLen = PINGLEN;		/* length of next PING to send / receive */
unsigned long pingRcvError;	/* # of mangled PING replies */
unsigned long pingRcvOK;	/* # of proper PING replies */
unsigned long pingRcvTimeout;	/* # of missing PING replies */
unsigned long pingSendError;	/* # of PINGs not sent due to errors */
unsigned long pingSendOK;	/* # of PINGs send properly */
int pingOutstanding;		/* is a PING outstanding right now? */
struct wping DLLFAR *pingP = 0;	/* PING control structure (in the DLL) */
char pingStat[140] = {"outstanding"};	/* PING status */
char prbuf[256];		/* message assembly buffer */
unsigned long recvCounter;	/* # of datagrams received */
unsigned long recvGiant;	/* # of oversized datagrams received */
unsigned long recvMismatch;	/* # of mangled / o-o-o datagrams received */
unsigned long recvOK;		/* # of well-formed datagrams received */
unsigned long recvRunt;		/* # of undersized datagrams received */
unsigned long recvTimeout;	/* # of timeouts on received datagrams */
unsigned long retryCounter;	/* # of unsuccessful tries to send last dg */
char *rMsgBufP;			/* buffer for receiving a datagram */
unsigned long sendCounter;	/* # of datagrams sent */
unsigned long sendError;	/* # of datagrams not sent due to errors */
unsigned long sendOK;		/* # of datagrams sent properly */
unsigned long sendRunt;		/* # of datagrams sent with write truncation */
int timer = 9999;		/* arrival (count-up) timer */
int timePainter;		/* repaint (count-down) timer */
int udpPort = OURPORT;		/* UDP port number for the test */
unsigned vxdVsn;		/* VxD version # */
char *wMsgBufP;			/* buffer for sending a datagram */


/* function bodies */

int PASCAL
WinMain (hInstance, hPrevInstance, lpCmdLine, nCmdShow)
HANDLE hInstance;
HANDLE hPrevInstance;
LPSTR lpCmdLine;
int nCmdShow;
{
	MSG msg;

	if (!hPrevInstance)
		if (!InitApplication(hInstance))
			return FALSE;

	if (!InitInstance(hInstance, lpCmdLine, nCmdShow))
		return FALSE;

	/*********************************************************************
	 Message sending is done here and here alone.  Ensures that it is
	 only within the context of the application main loop and not in
	 the MainWndProc (which can be invoked directly, causing *deep*
	 stack nesting explosions). 
	 ********************************************************************/
	
	while (GetMessage(&msg, NULL, NULL, NULL)) {
		TranslateMessage(&msg);
		DispatchMessage(&msg);

		if (netPhase == DESTROYED)
			continue;

		/* Completion of hostname resolution is checked for here. */
		if (hostP)
			InitNet2();

		/* A PING is checked for here. */
		if (pingCheck)
			PingRcv();

		/* A UDP echo datagrams are checked for here. */
		if (msgCheck)
			ProcessDatagrams();

		/* A PING is emitted here. */
		if (pingAgain  &&  netPhase == DATA) {
			/* Alternate zeroes of length PINGLEN and more
			   'interesting' stuff. */
			if (pingBufP) {
				pingBufP = 0;
				pingLen = PINGLEN;
			} else {
				pingBufP = &wMsgBufP[1];
				pingLen = msgLen - 1;
			}
			if (icmp_echo(pingP, msgAddr.fhost, pingBufP, pingLen
			  , 0, 0)) {
				++pingSendOK;
				pingAgain = 0;
				pingOutstanding = 1;
			} else {
				++pingSendError;
				if (neterrno != NET_ERR_NOMEM)
					perrorTypePlaint("icmp_echo", 1);
			}
		}
		/* A UDP echo datagram is sent here. */
		if (!msgOutstanding)
			Send1Datagram();
	}
	return msg.wParam;
}

BOOL
InitApplication (hInstance)
HANDLE hInstance;
{
	WNDCLASS wc;

	wc.style = NULL;
	wc.lpfnWndProc = MainWndProc;
	wc.cbClsExtra = 0;
	wc.cbWndExtra = 0;
	wc.hInstance = hInstance;
	wc.hIcon = LoadIcon(NULL, IDI_APPLICATION);
	wc.hCursor = LoadCursor(NULL, IDC_ARROW);
	wc.hbrBackground = GetStockObject(WHITE_BRUSH);
	wc.lpszMenuName = "UECHOMenu";
	wc.lpszClassName = "UECHOWClass";

	return RegisterClass(&wc);

}

BOOL
InitInstance (hInstance, lpCmdLine, nCmdShow)
HANDLE hInstance;
LPSTR lpCmdLine;
int nCmdShow;
{
	HWND hWnd;		/* main window */

	hInst = hInstance;

	hWnd = CreateWindow("UECHOWClass", "UDP ECHO Client"
	  , WS_OVERLAPPEDWINDOW
	  , CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT
	  , NULL, NULL, hInstance, NULL);

	if (!hWnd)
		return FALSE;

	hOurWnd = hWnd;			/* export for global use */
	if (!ParseCmdLine(lpCmdLine)	/* Parse the command line. */
	  ||  InitNet1() < 0) {		/* Init the network -- must be last. */
		PostQuitMessage(0);
		return FALSE;
	}

	ShowWindow(hWnd, nCmdShow);
	UpdateWindow(hWnd);	/* sends WM_PAINT message */
	return TRUE;
}

long FAR PASCAL
MainWndProc (hWnd, message, wParam, lParam)
HWND hWnd;
unsigned message;
WORD wParam;
LONG lParam;
{
	FARPROC lpProcAbout;
	char prbuf[128];

	switch (message) {
        case WM_COMMAND:    /* message: command from application menu */
		if (wParam == IDM_ABOUT) {
			lpProcAbout = MakeProcInstance(About, hInst);
			DialogBox(hInst, "UECHOAboutBox", hWnd, lpProcAbout);
			FreeProcInstance(lpProcAbout);
			break;
		} else
			goto default_proc;
	case WM_DESTROY:
		netPhase = DESTROYED;
		/* Terminate access to PCTCPAPI.DLL. */
		net_taskDestroy();
		PostQuitMessage(0);
		break;
	case WM_PAINT:
		PaintWnd(hWnd);
		break;
	case WM_TIMER:
		/* Periodic retries of the reception phase of outstanding
		   network replies (of various kinds).  This forestalls
		   permanent death, in the event that an asynch event
		   fails to be delivered. 
		*/
		switch (netPhase) {
		case INIT2:
			/* Main message loop will poll hostname resolution. */
			return NULL;
		case DATA:
			if (++timer >= MSGTIMER  &&  msgOutstanding) {
				/* 2 seconds has elapsed since send */
				++recvTimeout;
				msgOutstanding = 0;	/* forget it! */
			}
			if (--timePainter <= 0) {
				/* periodic repaint / re-PING intervals */
				if (!pingOutstanding)
					pingAgain = 1;	/* re-PING too */
				timePainter = PAINTTIMER;  /* restart timer */
				Repaint();
			}
			/* Poll datagram waiting. */
			msgCheck = 1;
			/* Poll PING reply. */
			pingCheck = pingOutstanding;
		}
		break;
	default:
		if (message != net_msgType)
			goto default_proc;
		/* This is an asynch notice from the PCTCPAPI.DLL. */
		switch (netPhase) {
		default:
			/* What's this one? */
			goto bogus_asynch;
		case INIT2:
			/* Is it a reply about the hostname resolution? */
			if (net_msgND(wParam) != hostP->h_nd)
				goto bogus_asynch;
			/* Main message loop will check. */
			break;
		case DATA:
			/* Is it a UDP datagram? */
			if (net_msgND(wParam) == nd)
				msgCheck = 1;
			/* Or, is it a PING reply? */
			else if (net_msgND(wParam) == pingP->p_nd)
				pingCheck = 1;
			else
				goto bogus_asynch;
			break;
		case DESTROYED:
			/* Eat come-lately's. */
			break;
		}
		break;
	}
	return NULL;

bogus_asynch:;
	sprintf(prbuf, "unknown asynch delivery %u on ND %u in netPhase %d"
	  , net_msgEvent(wParam), net_msgND(wParam), netPhase);
	otherPlaint(prbuf, 0);
default_proc:;
	return DefWindowProc(hWnd, message, wParam, lParam);
}

BOOL FAR PASCAL
About (hDlg, message, wParam, lParam)
HWND hDlg;
unsigned message;
WORD wParam;
LONG lParam;
{
	switch (message) {
	case WM_INITDIALOG:
		return TRUE;
	case WM_COMMAND:	/* message: received a command */
		if (wParam == IDOK	      /* "OK" box selected? */
		   ||  wParam == IDCANCEL) {  /* System menu close command? */
			Repaint();
			EndDialog(hDlg, TRUE);
			return TRUE;
		}
		break;
	}
	return FALSE;		/* Didn't process a message. */
}

/* First phase to real network initialization.  Starts the process of
   hostname <-> address resolution.  Called only once. */
int
InitNet1 () {
	int i;
	int errsave;
	char *errP;
	auto int asize;

	/* Allocate receive and send buffers. */
	if (!(rMsgBufP = my_calloc(msgLen))
	  ||  !(wMsgBufP = my_calloc(msgLen + 1)))
		return otherPlaint("Unable to allocate buffers.", 0);

	/* Initialize access to PCTCPAPI.DLL. */
	if (net_taskInit("uecho") == -1)
		return perrorTypePlaint("Unable to register w/ PC/TCP DLL.", 0);

	/* Get the PC/TCP version ... */
	netVsn = (unsigned) get_netversion();
	/* ... and the VxD version ... */
	asize = sizeof vxdVsn;
	if (get_kernel_info(0, TAG_VxD_0, 0, &vxdVsn, &asize) < 0) {
		errP = "get_kernel_info TAG_VxD_0";
		goto plaintDeregNQuit;
	}
	/* and the PCTCPAPI.DLL version. */
	asize = sizeof vxdVsn;
	if (get_kernel_info(0, TAG_VxD_1, 0, &dllVsn, &asize) < 0) {
		errP = "get_kernel_info TAG_VxD_1";
		goto plaintDeregNQuit;
	}

	/* Create a network descriptor for the hostname translation. */
	if ((i = net_getdesc()) < 0) {
		errP = "net_getdesc HOST";
		goto plaintDeregNQuit;
	}
	/* Register for asynchronous delivery of host resolution reply. */
	if (net_asynchw(i, NET_AS_RCV, hOurWnd, NET_ASWM_POST) == -1L) {
		errP = "net_asynchw HOST";
		goto plaintDeregNQuit;
	}
	if (!host_nm_query2(aname, 0, i, &hostP)) {
		errP = "host_nm_query";
		goto plaintDeregNQuit;
	}
	/* Set a one-second timer. */
	if (SetTimer(hOurWnd, 0, 1000, 0) == NULL) {
		otherPlaint("SetTimer failed", 0);
		goto deregNQuit;
	}
	netPhase = INIT2;
	return 0;

	/* Complain, deregister with PCTCPAPI.DLL and quit. */
plaintDeregNQuit:;
	perrorTypePlaint(errP, 0);
deregNQuit:;
	net_taskDestroy();
	return -1;
}

/***************************************************************************
 Second phase of network initialization.  Attempts to complete the
 hostname resolution phase (begun by 'InitNet1').  If that succeeds,
 allocates PING and datagram send / receive paraphernalia.
 Note: this routine may be called multiple times, until the hostname
 resolution completes or times out. 
*****************************************************************************/
int
InitNet2 () {
	register char *cP;
	register char c;
	struct whostent DLLFAR *hP;
	int i;

	/* Try to complete hostname resolution. */
	if ((i = host_adnm_rcv(hostP)) == 0)
		return 0;	/* spin out resolution */
	if (i < 0) 
		return perrorTypePlaint("host_adnm_rcv", 0);
	_fstrcpy(cname, hostP->h_name);
	msgAddr.fhost = hostP->h_addrs[0];
	host_free((hP = hostP, hostP = 0, hP));

	/* Get a network descriptor for the datagrams. */
	if ((nd = net_getdesc()) == -1)
		return perrorTypePlaint("net_getdesc", 0);
	/* 'Connect' to the peer on the datagram echo socket. */
	msgAddr.fsocket = udpPort;
	msgAddr.protocol = DGRAM;
	if (net_connect(nd, DGRAM, &msgAddr) == -1)
		return perrorTypePlaint("net_connect", 0);
	/* Get our IP address. */
	if ((ipAddr = get_addr(nd)) == 0L)
		return perrorTypePlaint("get_addr", 0);
	/* Register for asynchronous delivery of data arrival notification. */
	if (net_asynchw(nd, NET_AS_RCV, hOurWnd, NET_ASWM_POST) == -1L)
		return perrorTypePlaint("net_asynchw UDP", 0);
	/* Initialize the send buffer w/ interesting stuff. */
	for (cP = wMsgBufP, c = ' '; cP < &wMsgBufP[msgLen]; )
		if ((*cP++ = c++) > '~')
			c = ' ';

	/* Create a network descriptor for the PING. */
	if ((i = net_getdesc()) < 0)
		return perrorTypePlaint("net_getdesc PING", 0);
	/* Register for asynchronous delivery of PING reply. */
	if (net_asynchw(i, NET_AS_RCV, hOurWnd, NET_ASWM_POST) == -1L)
		return perrorTypePlaint("net_asynchw PING", 0);
	/* Initial PING to the designated host. */
	pingLen = PINGLEN;
	if (!(pingP = icmp_echo(i, msgAddr.fhost, pingBufP, pingLen, 0, 0)))
		return perrorTypePlaint("icmp_echo INITIAL", 0);
	++pingSendOK;
	pingOutstanding = 1;
	netPhase = DATA;
	Repaint();
	return 0;
}

/* Parse out options and arguments on the command line. */
int
ParseCmdLine (lpCmdLine)
LPSTR lpCmdLine;
{
	register char c;
	char *errP;
	LPSTR fcP, beginfcP;
	auto LPSTR afcP;
	char errBuf[64];

	/* Parse the command line.  Look for "/P<portNo>" and/or "/L<msgLen>"
	   options. */
	for (fcP = lpCmdLine; c = *fcP; ++fcP) {
		switch (c) {
		case '/':	/* option delimiter -- MS-DOS style */
			switch (*++fcP) {
			case 'p':  case 'P':	/* UDP service port number */
				afcP = ++fcP;
				udpPort = (unsigned) wstrtol(fcP, &afcP, 10);
				if (afcP == fcP)
					goto usage_err_out;
				fcP = afcP - 1;	/* at end of number parsed */
				break;
			case 'l':  case 'L':	/* datagram size */
				afcP = ++fcP;
				msgLen = (unsigned) wstrtol(fcP, &afcP, 10);
				if (afcP == fcP)
					goto usage_err_out;
				if (msgLen < 1)
					msgLen = 1;
				fcP = afcP - 1;	/* at end of number parsed */
				break;
			default:
				goto usage_err_out;
			}
			break;
		default:
			if (!isspace(c))
				goto endArgs;
			break;
		}
	}
endArgs:;
	/* Look now for the hostname (or address) argument. */
	if (c == '\0') {
		errP = "no host argument";
		goto err_out;
	}
	beginfcP = fcP;		/* remember where it starts */
	while ((c = *fcP)  &&  !isspace(c))
		++fcP;		/* find the end */
	if (c)
		*fcP = '\0';	/* terminate name token */
	_fstrcpy(aname, beginfcP);
	return TRUE;

usage_err_out:;
	errP = "options are /P<portNo> /L<msgLen>";
err_out:;
	sprintf(errBuf, "Invalid command line: %s", errP);
	otherPlaint(errBuf, 0);
	return FALSE;
}

/* Polls for reply to a PING. */
void
PingRcv () {
	int i;

	/* Poll PING completion.  Alternate PINGs have a null buffer
	   pointer (and hence will send only zeroes). */
	if ((i = icmp_echo_rcv(pingP, pingBufP, pingLen, 0, 0)) < 0) {
		if (neterrno == NET_ERR_TIMEOUT) {
			++pingRcvTimeout;
			strcpy(pingStat, "timed-out");
			pingCheck = pingOutstanding = 0;
		} else {
			++pingRcvError;
			perrorTypePlaint("icmp_echo_rcv", 1);
		}
	} else if (i > 0) {		/* proper reply was seen */
		sprintf(pingStat, "reply in %lu msec", PingRTT(pingP));
		++pingRcvOK;
		pingCheck = pingOutstanding = 0;
	} else				/* no reply was seen */
		pingCheck = 0;
	if (i < 0  ||  pingRcvOK == 1)
		Repaint();
}

/* Polls for reply UDP datagrams. */
int
ProcessDatagrams () {
	register char *rcP, *wcP;
	int n;

	do {
		if (inError  ||  netPhase == DESTROYED)
			break;		/* pretend we got it */
		if ((n = net_read(nd, rMsgBufP, msgLen+1, &msgAddr, 0)) == -1){
			if (net_errno == NET_ERR_WOULD_BLOCK)
				msgCheck = 0;
			else
				perrorTypePlaint("net_read", 1);
			return 0;	/* try again later */
		}
		++recvCounter;
		if (n != msgLen) {
			if (n < msgLen)
				++recvRunt;
			else
				++recvGiant;
			continue;	/* keep looping */
		}
		for (rcP = &rMsgBufP[msgLen], wcP = &wMsgBufP[msgLen]
		  ; rcP > rMsgBufP; )
			if (*--rcP != *--wcP) {
				++recvMismatch;
				goto no_match;	/* keep looping */
			}
		++recvOK;
		break;
no_match:;
	} while (Yield(), 1);	/* Yield so we don't pig-out. */
	msgCheck = msgOutstanding = 0;	/* Send another now. */
}

/* 
   Send a single UDP datagram to the peer.  Serialize each (in the first byte)
   so that we can match replies. 
*/
void
Send1Datagram () {
	int n;

	if (inError  ||  netPhase == DESTROYED  ||  nd == -1)
		return;
	++wMsgBufP[0];		/* 'uniquified' */
	if ((n = net_write(nd, wMsgBufP, msgLen, 0)) != msgLen) {
		++sendError;
		if (n > 0) {
			sprintf(prbuf, "net_write length mismatch %u vs %u", n
			  , msgLen);
			otherPlaint(prbuf, 1);
		} else if (net_errno != NET_ERR_NOMEM)
			perrorTypePlaint("net_write", 1);
		return;		/* wait for timer to try again later */
	}
	timer = 0;	/* restart the timer */
	if (n != msgLen)
		++sendRunt;
	else {
		msgOutstanding = 1;
		++sendOK;
	}
	++sendCounter;
}

/* Send characters to the screen.  After displaying each line of text,
   advance the vertical position for the next line of text.  The pixel
   distance between the top of each line of text is equal to the standard
   height of the font characters (tmHeight), plus the standard amount of
   spacing (tmExternalLeading) between adjacent lines. 
*/
void
PaintLine (txtP, hDC, tmP, nDrawXP, nDrawYP)
char *txtP;
HDC hDC;
TEXTMETRIC *tmP;
int *nDrawXP, *nDrawYP;
{
	unsigned len = strlen(txtP);

	TextOut(hDC, *nDrawXP, *nDrawYP, txtP, len);
	*nDrawYP += tmP->tmExternalLeading + tmP->tmHeight;
}

/* Paint the entire contents of the main window rectangle. */
void
PaintWnd (hWnd)
HWND hWnd;
{
	HDC hDC;
	PAINTSTRUCT ps;
	TEXTMETRIC textmetric;
	int nDrawX, nDrawY;

	hDC = BeginPaint(hWnd, &ps);
	/* Get the size characteristics of the current font.
	   This information will be used for determining the
	   vertical spacing of text on the screen. */
        GetTextMetrics (hDC, &textmetric);
	/* Initialize drawing position to 1/4 inch from the
	   top and from the left of the top, left corner of
	   the client area of the main windows. */
	nDrawX = GetDeviceCaps (hDC, LOGPIXELSX) / 4;
	nDrawY = GetDeviceCaps (hDC, LOGPIXELSY) / 4;

	/* paint out basic app stuff */
	sprintf(prbuf
	  , "PC/TCP Version %u.%02u -- VxD Version %u.%02u -- PCTCPAPI Version %u.%02u"
	  , netVsn / 256, netVsn % 256, vxdVsn / 256, vxdVsn % 256
	  , dllVsn / 256, dllVsn % 256);
	PaintLine(prbuf, hDC, &textmetric, &nDrawX, &nDrawY);

	sprintf(prbuf, "Peer is %s(%s) port %d -- sending length %u"
	  , cname, pr_in_name(msgAddr.fhost), udpPort, msgLen);
	PaintLine(prbuf, hDC, &textmetric, &nDrawX, &nDrawY);

	sprintf(prbuf
	  , " #recvs %lu -- #runts %lu -- #giants %lu -- #mismatch %lu -- #timeouts %lu -- #OK %lu"
	  , recvCounter, recvRunt, recvGiant, recvMismatch, recvTimeout
	  , recvOK);
	PaintLine(prbuf, hDC, &textmetric, &nDrawX, &nDrawY);

	sprintf(prbuf, " #sends %lu -- #runts %lu -- #OK %lu -- #errors %lu"
	  , sendCounter, sendRunt, sendOK, sendError);
	PaintLine(prbuf, hDC, &textmetric, &nDrawX, &nDrawY);

	sprintf(prbuf, "PING last sent: %s", pingStat);
	PaintLine(prbuf, hDC, &textmetric, &nDrawX, &nDrawY);

	sprintf(prbuf, " recvs -- #timeouts %lu -- #errors %lu -- #OK %lu"
	  , pingRcvTimeout, pingRcvError, pingRcvOK);
	PaintLine(prbuf, hDC, &textmetric, &nDrawX, &nDrawY);

	sprintf(prbuf, " sends -- #errors %lu -- #OK %lu"
	  , pingSendError, pingSendOK);
	PaintLine(prbuf, hDC, &textmetric, &nDrawX, &nDrawY);

	EndPaint(hWnd, &ps);
}

/* Force a full repaint of the main window. */
void
Repaint () {
	if (netPhase < DESTROYED)
		InvalidateRect(hOurWnd, NULL, TRUE);
}

/* eof */

/*
 * $Log:   J:/22vcs/srccmd/samples.win/uecho.c_v  $
 * 
 *    Rev 1.4   20 Jul 1992 19:52:48   paul
 * added pctcp/ipconfig.h
 * 
 *    Rev 1.3   06 May 1992 16:04:34   arnoff
 *  * 06-May-92  ftp	DevKit 2.1 beta.
 * 
 *    Rev 1.2   03 Feb 1992 22:01:38   arnoff
 * pre beta-2 testing freeze
 * 
 *    Rev 1.1   29 Jan 1992 22:47:34   arnoff
 *  
 */
