/*
 * SRCCMD\SOCKTST\USHOUT.C - 2.0 UDP shout example program.
 * This is the 4BSD version of srcccmd\tctest\shout.c
 *
 * Copyright (C) 1986,1990 by FTP Software, Inc.
 *
 * This software is furnished under a license and may be used and copied
 * only in accordance with the terms of such license and with the
 * inclusion of the above copyright notice. This software or any other
 * copies thereof may not be provided or otherwise made available to any
 * other person. No title to and ownership of the software is hereby
 * transferred.
 *
 * The information in this software is subject to change without notice
 * and should not be construed as a commitment by FTP Software, Inc.
 *
 * Edit History
 * 10-Apr-90	daveb 	Ported from jbvb's 2.04pl1 prod tctest\shout.c.
 * 25-May-90	jog	and over to OS/2, cleaned up a bit, print how
 *			much have written if get error when still writing
 * 29-May-90	jog	change default # of bytes to write to 1024
 * 04-Jun-90	jog	make it listen for reply after sending everything..
 *			NOTE: most hosts will not send a reply when sent
 *			data one the discard port. The included examples
 *			do so to try and illustrate how one would send &
 *			receive data using the socket library.
 * 11-Jun-90	jog	arg parsing from switch to if/else, clean up a bit..
 *			prototype main() declaration, use source port
 * 13-Jun-90	jog	use discard socket
 * 14-Jun-90	jog	AF_INET -> PF_INET
 * 06-Jun-90	jog	added stdlib.h, string.h
 * 18-Apr-91	jog	take out extern int errno for dlls
 * 15-May-91	towfiq	Added #include <io.h>
 * 28-Apr-93	rcq	fixed usage to display when queried 
 */

/* Compiler Include files: */
#include <io.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

/*#include <pctcp.h>*/

/* PC/TCP 4BSD Socket Includes: */
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <errno.h>

#define	SOCK_DISCARD	9	/* use the UDP ttytst source port for test */

static	char data_buffer[2048];	/* holds generated data, 2k most can write */
static  char reply_buf[512];	/* Buffer to hold any reply rcvd           */

main(int argc, char *argv[])
{
	struct	sockaddr_in dest;  /* Destination machine address structure */
	struct	hostent *host_ptr; /* Structure describing the hostname     */
	int 	sd;  		   /* Connection socket descriptor */
	int 	ret;		   /* General purpose return code  */

	int	len	  = 1024;	  /* # of bytes to write at a time  */
	long	total_len = 1024L*1024L;  /* Total # of bytes to generate   */
	long	bytes_sent= 0L;		  /* Counter of bytes on connection */
	int	tmp = 0;		  /* holds count for bytes written  */
	long	write_count = 0L;	  /* number of times		    */

	time_t	start, end;		  /* variables to hold read timing  */
	long	total_time;		  /* variable to hold delta t       */

	dest.sin_family = PF_INET;	  /* Internet Family addressing */
	/* default port to connect to.
	 * Must be in network byte order! (Opposite PC)
	 */
	dest.sin_port = htons( (int) SOCK_DISCARD );

	if ((argc == 1) || (argv[1][1] == '?')) {
	    printf("usage: ushout host [bytes per write [port]]\n");
	    exit(1);
	}
	
	if (strcmp(argv[1], "-version") == 0) {
		printf("FTP Software, Inc, UDP \"shout\" example program\n");
		exit(0);
	}	

	if (argc == 4) {	  /* Allow use of Arbitrary TCP port */
		dest.sin_port = htons(atoi(argv[3]));
		if (dest.sin_port == 0) {   /* Port 0 no good w/ BSD */
			printf("usage: ushout host [bytes per write [port]]\n");
			exit(1);
		}
	}
	if (argc >= 3) {          /* Use different write-block sizes */
		len = atoi(argv[2]);
		if ((len < 1) || (len > 2048)) {
			printf("usage: ushout host [bytes per write [port]]\n");
			exit(1);
		}
	}
	else if ((argc < 2) || (argc > 4)) { /* Invalid command line */
		printf("usage: ushout host [bytes per write [port]]\n");
		exit(1);
	}

	/* Now, command line's been parsed, get down to meat of program. */
	/* First, resolve the host name provided on the command line:    */
	host_ptr = gethostbyname(argv[1]);
	if(host_ptr == NULL) {
		perror("gethostbyname");
		exit(1);
	}

	/* Patch host address into structure describing connection: dest */
	bcopy(host_ptr->h_addr, &dest.sin_addr, host_ptr->h_length);	

/* What makes shout unique is that it generates data in memory (as opposed 
 * to accessing the disk).  First, generate the data and place it into an 
 * array, data_buffer:
 */
	for (tmp = 0; tmp < sizeof(data_buffer); tmp++)
		data_buffer[tmp] = tmp;

	/* Now stage a network connection.  1st step is to allocate a network
	 * (socket) descriptor: 
	 */
	sd = socket(PF_INET, SOCK_DGRAM, 0);
	if (sd == -1) {
		perror("socket");
		exit(1);
	}

	/* connect on this descriptor to host described in 'dest' struct. */
	/* connect not actually necessary over udp (SOCK_DGRAM); if you did
	 * not use connect however, you would have to use sendto() and supply
	 * destination address. We use it to show the similarity between 
         * sockets of udp and over tcp.
	 */
	ret = connect(sd, (struct sockaddr *) &dest, sizeof(dest));
	if (ret == -1) {
		perror("connect");
		exit(1);
	}

	printf ("Writing to %s. Writing %ld bytes %d at a time.\n",
		host_ptr->h_name, total_len, len);

	/* Write data on the descriptor like a banshee, careful to time the
	 * writes and count data transmitted: 
	 */
	time( &start );	
	while ( bytes_sent < total_len) {  /* while still bytes to send... */
		tmp = write(sd, data_buffer, len);       /* send them out. */
		if (tmp == -1) {
			perror("write");
			printf("Shout: error on attempt to write data.\n\n");

			/* Calc. time elapsed & stats about any data sent */
			time(&end);
			total_time = difftime(end, start);
			if (total_time) {
			         printf("Shout: sent %ld bytes in %ld writes, in %ld sec.\n",
				         bytes_sent, write_count, total_time);
			         printf("%ld writes/sec, %ld bytes/sec, %ld bits/sec\n",
				         (long) ((write_count)/total_time),
				         (long) ((bytes_sent)/total_time),
				         (long) ((bytes_sent*8)/total_time));
				}
				exit(1);
			}
		write_count++;	    /* incr. counter of times written  */
		bytes_sent += tmp;  /* # of bytes placed on connection */
	}

	/* Get the end time on write() loop: */
	time(&end);

	/* Calculate total time elapsed in write loop: */
	total_time = difftime(end, start);

	/* Print out the statistics calculated about this transfer: */
	printf("Shout: sent %ld bytes in %ld writes, in %ld sec.\n",
		bytes_sent, write_count, total_time);
	printf("%ld writes/sec, %ld bytes/sec, %ld bits/sec\n",
		(long) ((write_count)/total_time),
		(long) ((bytes_sent)/total_time),
		(long) ((bytes_sent * 8)/total_time));
	
	/* look for a reply... NOTE: most hosts won't give 
	 * a 'reply', done to illustrate communication bwteeen sockets,
	 * our ulisten example will give a reply though.
	 */
	printf("\nWaiting for reply from server..\n");
	while (1) {
		tmp = sizeof(dest);
		len = recvfrom(sd, reply_buf, sizeof(reply_buf), 0, (struct sockaddr *) &dest, &tmp);
		if (len == -1) {
			if (errno == EWOULDBLOCK) /* if no data, read again */
				continue;
			/* any error beside would block. just punt */
			perror("recvfrom");
			exit(1);
		}
		/* else got a reply ...*/
		printf("Server's reply: %s\n", reply_buf);
		break;
	}
	
	/* close down the socket */
	close(sd);

	exit(0);
}
