/*
 * $Header:   E:/22vcs/srccmd/trans/trans.c_v   1.6   01 Jan 1993 16:05:28   snyders  $
 */

/*
 * TRANS.C - Translate in-memory i?cust.sys drivers to pctcp.ini format.
 *
 * Copyright (C) 1991, 1992 by FTP Software, Inc.  All rights reserved.
 * 
 * This software is furnished under a license and may be used and copied
 * only in accordance with the terms of such license and with the
 * inclusion of the above copyright notice. This software or any other
 * copies thereof may not be provided or otherwise made available to any
 * other person. No title to and ownership of the software is hereby
 * transferred.
 * 
 * The information in this software is subject to change without notice
 * and should not be construed as a commitment by FTP Software, Inc.
 *
 * Edit History
 * 22-jun-91	arnoff	Created.  Redirects to stdout (for now).  Does most
 *			drivers, but needs work for PPP, serial and scope.
 * 24-jun-91	arnoff	precedence, basic and extended security added.
 * 25-Jun-91	gordon	Changed hostname to host-name
 * 25-Jun-91	gordon	Changed hosttable to host-table
 * 25-Jun-91	gordon	Changed lowwindow to low-window
 * 25-Jun-91	gordon	Moved host-table to [pctcp addresses]
 * 25-Jun-91	gordon	Provided prototypes to eliminate warnings
 * 11-Jul-91	Ben	Added -a[utoconfig] (with a little help from rehmi's
 *			 config library).
 * 12-Jul-91	Ben	Drop broadcast address.
 * 12-Jul-91	gordon	Move host-table to the kernel section.  Don't add
 *			 security and precedence if default.
 * 12-Jul-91	Ben	Added dma=on/off.  Don't add mail-relay, if not set.
 *			 Added -f copy old ftp_config file.
 * 14-Jul-91	Ben	Make trans skip old driver sections.
 * 15-Jul-91	Ben	Fiddled with usage message.
 * 15-Jul-91	arnoff	Fixed multiple '.' problem in search for ".sys" files
 *			 with -a option.  Screwed with format for readability.
 *			 Munged usage messege into a mini man page.
 * 16-jul-91	arnoff	added "use-old-init-scheme=no" for all cases.
 * 17-Jul-91	Ben	Worked a bit on the vmail split routine.
 * 19-Jul-91	Ben	Use includes in pctcp where available.
 * 24-Jul-91	Ben	Added "debug-flags".
 * 02-Aug-91	Ben	Ignore ip-security section.  Added direct check for 
 *			 version.  Let users know if we skip an old section.
 * 14-Aug-91	Ben	Check for old (2.03) ethdrv.sys
 * 15-Aug-91	Ben	Enable check for vmail section.
 * 03-Sep-91	Ben	Copy sect_name to client_sect in convert_vmail_sect.
 * 24-Sep-91	Ben	Check c: then a: for \config.sys.
 * 24-Sep-91	gordon	Make SLIP work, handle baud->divisor translation
 *			 handle IFF_DOES_SERIAL flag
 * 10-Oct-91	Ben	Replace exit's with return's.
 * 25-Dec-91	Ben	Renamed AUTH_* to new names
 * 29-Dec-91	Arnoff	Now copies the full sections of tn, lpr and vt.
 *			 Created trans_section().
 * 08-Jan-92	Ben	Added slip and ppp to driver list.
 * 18-Jun-92	paul	set stack size for Turbo/Borland version
 * 21-jul-92	rcq	qualified to pctcp\ subdirectory rwconf.h, 
 */

#include <stdio.h>
#include <io.h>		/* for isatty() */
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <pctcp/types.h>

#include <pctcp/rwconf.h>
#include <pctcp/ipconfig.h>
#include <pctcp/ifconfig.h>
#include <pctcp/pctcp.h>
#include <pctcp/options.h>
#include <pctcp/interfac.h>
#include <pctcp/if_info.h>

/* external function declarations */
extern check_driver (char *drv_name);

/* local function declarations */
void usage(void);
void ipconfig(char*);
int ifconfig(char*);
int copy_old_config (void);
int convert_vmail_sect (char * sect_name);
void write_variables (int write_vmail_p);
static void trans_section(char *sect);

struct drv_list 	*get_devices (void);

int getopt(int, char**, char*);		/* iterator and getopt() ret*/

in_name router = 0;

/* structure for ifcust list */
struct	drv_list {
    char 	name[10];
    struct drv_list	*next;
};

/* This stuff is for IP Precedence
*/
struct	pr_tab {
	int	val;
	char	*name;
};

/* Name of driver.sys files */
struct	pr_tab	drv_names [] = {
	1, 	"3c500",
	1, 	"3c503",
	1, 	"3c505",
	1, 	"attnau",
	1, 	"bdnint",
	1, 	"dp839eb",
	1, 	"ethdrv",
	1, 	"ifcust",
	1, 	"ibmtr",
	1, 	"intel",
	1, 	"ni5010",
	1, 	"ni5210",
	1, 	"p1300",
	1, 	"p1340",
	1, 	"protint",
	1, 	"ppp",
	1, 	"slip",
	1, 	"scope",
	1, 	"ubnic",
	1, 	"wd8003",
	-1, 0
};


/* Name of section added by trans to pctcp.ini */
struct	pr_tab	trans_names [] = {
	1, 	"general",
	1, 	"kernel",
	1, 	"addresses",
	1, 	"lpr",
	1, 	"tn",
	1, 	"vt",
	1, 	"ip-security",		/* not need by new section */
	-1, 0
};
	    


/* Names of vmail and client sections */
struct	pr_tab	client_names [] = {
	1, "vmail",
	1, "nntp",
	1, "pcmail",
	1, "pop3",
	1, "pop2",
	1, "usm",
	-1, 0
};


/* Names (truncated) of variables used by vmail (truncated at 8 letters) */
struct	pr_tab	vmail_vars [] = {
 	1, "client-s",
	1, "ignore-f",
	1, "highligh",
	1, "hilist-f",
	1, "indent-r",
	1, "folder",
	1, "reply-to",
	1, "ask-over",
	1, "print-no",
	1, "new-mbox",
	1, "post",
	1, "signatur",
	1, "aliases",
	1, "fullname",
	1, "editor",
	1, "printer",
	1, "record",
	1, "50-lines",
	1, "43-lines",
	1, "25-lines",
	1, "high-res",
	1, "low-res",
	-1, 0
};


#define UNAMBIGUOUS	3	/* Use this many chars to identify cmds */

struct	pr_tab	prec[] = {
	IP_PREC_ROUTINE,	"Routine",
	IP_PREC_PRIORITY,	"Priority",
	IP_PREC_IMMEDIATE,	"Immediate",
	IP_PREC_FLASH,		"Flash",
	IP_PREC_FLASH_OVERRIDE,	"Override",
	IP_PREC_CRITICECP,	"Critical/ECP",
	IP_PREC_INET_CONTROL,	"Internet Control",
	IP_PREC_NET_CONTROL,	"Network Control",
	-1, 0
};

/* This stuff is for IP Security
*/

struct	pr_tab	ip_bsec[] = {		/* Basic Security class levels */
	IP_TOPS,	"Top_secret",
	IP_SECR,	"Secret",
	IP_CONFI,	"Confidential",
	IP_UNCLA,	"Unclassified",
	-1, 0
};

struct	pr_tab	ip_auth[] = {
	AUTH_GENSER,	"GENSER",
	AUTH_SIOP,	"SIOP",
	AUTH_SCI,	"SCI",
	AUTH_NSA,	"NSA",
	-1, 0
	
};

/*
 * table of baud rate divisors
 */
struct baud_entry {
    unsigned rate;
    unsigned divisor;
};

struct baud_entry baud_map[] = {
	{	56000,	0x0002	},
	{	38400,	0x0003	},
	{	19200,	0x0006	},
	{	9600,	0x000c	},
	{	7200,	0x0010	},
	{	4800,	0x0018	},
	{	3600,	0x0020	},
	{	2400,	0x0030	},
	{	2000,	0x003a	},
	{	1800,	0x0040	},
	{	1200,	0x0060	},
	{	600,	0x00c0	},
	{	300,	0x0180	},
	{	150,	0x0300	},
	{	110,	0x0417	},
	{	75,	0x0600	},
	{	50,	0x0900	},
	{	0,	0	}
};


extern char		*optarg;		/* getopt() extern 	    */
extern int 		optind;			/* getopt() extern 	    */

#ifdef __TURBOC__
extern unsigned _stklen = 10000;
#endif

main(int argc, char *argv[])
{
    char		*optlist = "fah?";/* getopt() switch list */
    int			i;			/* iterator and getopt() ret*/
    char		auto_config = FALSE;	/* auto_config boolean	*/
    char		old_config = FALSE;	/* old_config boolean	*/
    char driver_name[50];			/* driver name		*/
    char driver_inmem[50];			/* in-memory driver name*/
    struct drv_list	*driver;		/* list of drivers */
    
/****************************************** CMD LINE PARSING ****************/

    if (stricmp (argv[1], "-version") == 0) {
	pr_notice ("Trans");
	exit (0);
    }

    if (stricmp (argv[1], "-?") == 0) {
	usage();
	exit (0);
    }

    while ((i = getopt(argc, argv, optlist)) != EOF) {
	switch (i) {
	    case 'a':
		auto_config = TRUE;
		break;
	    case 'f':
		old_config = TRUE;
		break;
	    case 'h':				/* usage and bad switch	    */
	    case '?':
	    default:
		usage();
		return 0;
		break;
	}
    }
    
    if (!auto_config)
    {
	if (optind < argc) {
	    strcpy(driver_name, argv[optind++]);	/* driver name	    */
	    strcpy(driver_inmem, driver_name);
	    strcat(driver_inmem, "0");
	} 
	else {
	    usage();				/* usage message	    */
	    return 0;				/* abort		    */
	}
    }

    if (auto_config)
    {
	/* get list */
	driver = get_devices ();
	if (driver == (struct drv_list *) 0)   
	{
	    fprintf(stdout, ";; Auto-configure failed.  Could not find \
device drivers in \"config.sys\"\n");
	    if (!isatty(fileno(stdout))) {
		fprintf(stderr, ";; Auto-configure failed.  Could not find \
device drivers in \"config.sys\"\n");
	    }
	    exit (1);
	}

	/* do ipconfig stuff						    */
	if (stricmp (driver->name, "ethdrv") == 0)
	    ipconfig("ifcust");		/* there are still some 2.03 users  */
	else
	    ipconfig(driver->name);
	/* do ifconfig stuff						    */

	/* loop on each name */
	while (driver != (struct drv_list *) 0) {
	    /* append 0 to the name */
	    strcat(driver->name, "0");
	    ifconfig(driver->name);
	    driver = driver->next;
	}
    }
    else {
	ipconfig(driver_name);
	ifconfig(driver_inmem);
    }

    if (old_config)
	copy_old_config();

} /* end main() */

void ipconfig(char *driver_name)
{
    char *name = INMEMORY_IPCONFIG;		/* ipconfig driver name	*/
    struct ipconfig conf;			/* configuration struct */
    char tmp[128];				/* temporary storage	*/
    int i;					/* iterator		*/
    char ip_tos;				/* flags / ip precedence*/
    struct pr_tab *p;				/* ip security/preceden.*/
    
    if (get_ipconfig(name, &conf) == FALSE) {	/* get configuration	*/
	fprintf(stdout,
	    "\n;;Could not read in-memory $ipcust, is it loaded?\n");
	if (!isatty(fileno(stdout))) {
	    fprintf(stderr,
		"\n;;Could not read in-memory $ipcust, is it loaded?\n");
	}
	return;					/* abort this function	*/
    }
    
    strcpy (tmp, getenv("FTP_CONFIG"));		/* old config file	    */
    if (config_open(tmp, 0) == FALSE) printf("can't open config file.\n");
    
    /* PCTCP GENERAL **********						*/
    fprintf(stdout,"\n[pctcp general]\n");

    /* host name							*/
    fprintf(stdout,"host-name=%s\n",conf.c_hostname);
    
    /* domain name							*/
    fprintf(stdout,"domain=%s\n",conf.c_domain);

    /* user id								*/
    fprintf(stdout,"user=%s\n",conf.c_userid);

    /* full name							*/
    fprintf(stdout,"full-name=%s\n",conf.c_username);

    /* office								*/
    fprintf(stdout,"office=%s\n",conf.c_office);

    /* phone								*/
    fprintf(stdout,"office-phone=%s\n",conf.c_phone);

    /* time zone							*/
    fprintf(stdout,"time-zone=%s\n",conf.c_tmlabel);

    /* offset								*/
    fprintf(stdout,"time-zone-offset=%s\n",itoa(conf.c_tmoffset,tmp,10));
    
    /* USE OLD INIT SCHEME IS SET TO OFF IN ALL CASES			*/
    fprintf(stdout,"use-old-init-scheme=no\n");

    /* PCTCP KERNEL **********						*/
    fprintf(stdout,"\n[pctcp kernel]\n");
    
    /* interface							*/
    fprintf(stdout,"interface=%s 0\n",driver_name);

    /* window								*/
    fprintf(stdout,"window=%s\n",itoa(conf.c_window,tmp,10));

    /* low window							*/
    fprintf(stdout,"low-window=%s\n",itoa(conf.c_lowwindow,tmp,10));

    /* add commented out serial-number and key 				*/
    fprintf (stdout, "; serial-number = xxxx-xxxx-xxxx\n");
    fprintf (stdout, "; authentication-key = yyyy-yyyy-yyyy\n");

    /* hosttable							*/
    fprintf(stdout,"host-table=%s\n",conf.c_hosttable);

    /* debugging */
    if (conf.c_debug) 
	fprintf(stdout,"debug-flags=%02x\n",conf.c_debug);
	
    /* 
     *  IP Precedence is only dealt with if the $ipcust file specifies
     *  something other than "lax" / "Routine"
     */
    if ((conf.c_ip_opt_flag & C_IP_STRICT) ||
	(conf.c_ip_tos & ((unsbyte)0xE0)) != IP_PREC_ROUTINE) {
	    
	/* precedence matching						*/
	fprintf(stdout,"ip-precedence-matching=%s\n",
		(conf.c_ip_opt_flag & C_IP_STRICT) ? "strict" : "lax");

	/* precedence value						*/
	ip_tos = conf.c_ip_tos & ((unsbyte)0xE0);
	for (p = prec; p->val != -1; p++) {
	    if ((unsigned char)ip_tos == (unsigned char)p->val)
		fprintf(stdout,"ip-precedence=%s\n",p->name);
	}
    }
    
    /* 
     *  IP Security is only dealt with if the $ipcust file specifies
     *  something other than "lax" / "none"
     */
    if (conf.c_ip_opt_flag & (C_IP_BAS|C_IP_EXT|C_IP_SECURE)) {

	/* ip security matching						*/
	fprintf(stdout,"ip-security-matching=%s\n",
		(conf.c_ip_opt_flag & C_IP_SECURE) ? "strict" : "lax");

	/* ip security type						*/
	fprintf(stdout,"ip-security=");
	if (conf.c_ip_opt_flag & C_IP_BAS) {
	    fprintf(stdout,"basic\n");
	    /* basic ip security					*/
	    fprintf(stdout,"ip-security-basic=");
	    for (i = 0, p = ip_bsec; p->val != -1; p++, i++) {
		if (ip_bsec[i].val==conf.c_ip_options[conf.c_bas_security+2])
		    fprintf(stdout,"%s ",ip_bsec[i].name);
	    }
	    for (i = 0, p = ip_auth; p->val != -1; p++, i++) {
		if (ip_auth[i].val==conf.c_ip_options[conf.c_bas_security+3])
		    fprintf(stdout,"%s",ip_auth[i].name);
	    }
	    fprintf(stdout,"\n");
	} else if (conf.c_ip_opt_flag & C_IP_EXT) {
	    fprintf(stdout,"extended\n");
	    /* extended ip security					*/
	    fprintf(stdout,"ip-security-extended=");
	    for (i = 2; i < conf.c_ip_opt_len; i++)
		fprintf(stdout,"0x%X ",conf.c_ip_options[i]);
	    fprintf(stdout,"\n");
	} else
	    fprintf(stdout,"none\n");

    }

    /* PCTCP ADDRESSES **********					*/
    fprintf(stdout,"\n[pctcp addresses]\n");

    /* default gateway							*/
    if (conf.c_defgw != (in_name)0)
	router = conf.c_defgw;			/* save it for later	*/

    /* domain-name-server						*/
    for (i = 0; i < MAXDNAMES; i++) {
	if (conf.c_dm_servers[i] != (in_name)0)
	    fprintf(stdout,"domain-name-server=%s\n",
		inet_ntoa(conf.c_dm_servers[i]));
    }

    /* cookie-server							*/
    if (conf.c_servers[SRV_COOKIE] != (in_name)0)
	fprintf(stdout,"cookie-server=%s\n",
	    inet_ntoa(conf.c_servers[SRV_COOKIE]));

    /* imagen-print-server						*/
    if (conf.c_servers[SRV_IPRINT] != (in_name)0)
	fprintf(stdout,"imagen-print-server=%s\n",
	    inet_ntoa(conf.c_servers[SRV_IPRINT]));

    /* log-server							*/
    if (conf.c_servers[SRV_LOG] != (in_name)0)
	fprintf(stdout,"log-server=%s\n",
	    inet_ntoa(conf.c_servers[SRV_LOG]));

    /* mail-relay							*/
    if (strlen (conf.c_mailrelay))
	fprintf(stdout,"mail-relay=%s\n",conf.c_mailrelay);

    /* time-server							*/
    for (i = 0; i < MAXTIMES; i++) {
	if (conf.c_time[i] != (in_name)0)
	    fprintf(stdout,"time-server=%s\n",
		inet_ntoa(conf.c_time[i]));
    }

    /* PCTCP LPR **********						*/
    fprintf(stdout,"\n[pctcp lpr]\n");

    /* server								*/
    if (conf.c_servers[SRV_LPR] != (in_name)0)
	fprintf(stdout,"server=%s\n",
	    inet_ntoa(conf.c_servers[SRV_LPR]));
    trans_section("lpr");

    /* PCTCP TN **********						*/
    fprintf(stdout,"\n[pctcp tn]\n");
    
    /* bs (back space)							*/
    fprintf(stdout,"back-arrow-key=%s\n",
	((conf.c_emulators & EM_BS) ? "bs" : "del"));
    trans_section("tn");

    /* PCTCP VT **********						*/
    fprintf(stdout,"\n[pctcp vt]\n");

    /* wrap								*/
    fprintf(stdout,"wrap-line=%s\n",
	((conf.c_emulators & EM_WRAP) ? "on" : "off"));
    trans_section("vt");

    config_close(0);

    return;
} /* end ipconfig() */


ifconfig(char *driver_inmem)
{
    int i;					/* iterator		*/
    struct if_info *ifinfo, *find_if();		/* interface information*/
    struct ifconfig conf;

    /* fetch the ifconfig info */
    if (get_ifconfig(driver_inmem, &conf) == FALSE) {
	fprintf(stdout,"\n;;Could not read in-memory %s, is it loaded?\n",
	    driver_inmem);
	if (!isatty(fileno(stdout))) {
	    fprintf(stderr,"\n;;Could not read in-memory %s, is it loaded?\n",
		driver_inmem);
	}
	return (FALSE);	/* give up */
    }
    
    /* ethdrv0 has not been used since PC/TCP 2.04 and it has been removed
     * from the if_info library, but some people are still using it so use 
     * the number for ifcust0 instead.
     */
    
    if (stricmp (driver_inmem, "ethdrv0") == 0)
	conf.c_driver = 0x0A01;			/* cheat 		*/

    if ((ifinfo = find_if(conf.c_driver)) == NULL) {
	fprintf(stdout,"\n;;Could not find %s (%u) in info-base.\n",
	    driver_inmem, conf.c_driver);
	if (!isatty(fileno(stdout))) {
	    fprintf(stderr,"\n;;Could not find %s (%u) in info-base.\n",
		driver_inmem, conf.c_driver);
	}
	return (FALSE);	/* give up */
    }
    
    /****** DUMP DRIVER SPECIFIC STUFF ********/

    fprintf(stdout, "\n[pctcp %s %d]\n", ifinfo->if_sname, conf.c_number);
    
    if (conf.c_me)					/* ip addr	*/
	fprintf(stdout,"ip-address=%s\n",
	    inet_ntoa(conf.c_me));

    if (conf.c_net_mask)				/* subnet mask	*/
	fprintf(stdout,"subnet-mask=%s\n",
	    inet_ntoa(conf.c_net_mask));

    if (router)						/* def gateway	*/
	    fprintf(stdout,"router=%s\n",inet_ntoa(router));

    if ((ifinfo->if_flags & IFF_DOES_SERIAL) ||
        (!strcmp(ifinfo->if_sname,"slip")))		/* slip interface */
	    fprintf(stdout,"serial=0\n\n[pctcp serial 0]\n");

    if (ifinfo->if_flags & IFF_DOES_INT) {		/* interrupt	*/
	if (conf.c_intvec)
	    fprintf(stdout,"irq=%u\n",conf.c_intvec);
    }

    if (ifinfo->if_flags & IFF_DOES_DMA) { 		/* dma 		*/
	if (conf.c_flags & FL_DMA) { 			/*  dma is set	*/
	    fprintf(stdout,"dma=on\n");
	    fprintf(stdout,"rcv-dma=%u\n",conf.c_rcv_dma);
	    fprintf(stdout,"xmt-dma=%u\n",conf.c_tx_dma);
	}
	else {						/*  dma is not set */
	    fprintf(stdout,"dma=off\n");
	}
    }

    if (ifinfo->if_flags & IFF_DOES_BASE) {		/* base io addr	*/
	if (conf.c_base)
	    fprintf(stdout,"io-addr=0x%03X\n",conf.c_base);
    }

    if (ifinfo->if_flags & IFF_DOES_BAUD) {		/* baud		*/
	if (conf.c_baud) {
	    struct baud_entry *bmp;
	    for (bmp = baud_map; bmp->divisor; ++bmp)
		if (bmp->divisor == conf.c_baud)
		    break;
	    if (bmp->divisor)
		fprintf(stdout,"baud=%u\n",bmp->rate);
	}
    }

    if (ifinfo->if_flags & IFF_DOES_MEMORY) {		/* base memory	*/
	if (conf.c_memory)
	    fprintf(stdout,"ram-addr=0x%04X\n",conf.c_memory);
    }

    if (ifinfo->if_flags & IFF_DOES_ADDR) {		/* mac address	*/
	fprintf(stdout,"mac-address=");
	for (i = 0; i < 6; i++)
	    fprintf(stdout,"%2X ",conf.c_addr[i]);
	fprintf(stdout,"\n");
    }

    if (ifinfo->if_flags & IFF_DOES_MSIZE) {		/* memory size	*/
	if (conf.c_msize)
	    fprintf(stdout,"ram-size=0x%04X\n",conf.c_msize);
    }

    if (ifinfo->if_flags & IFF_DOES_PORT) {		/* port		*/
	if (conf.c_port == 0)
	    fprintf(stdout,"transceiver=thick\n");
	else
	    fprintf(stdout,"transceiver=thin\n");
    }

    if (ifinfo->if_flags & IFF_DOES_X25standard && 
	    conf.c_flags & FL_X25standard)
	fprintf(stdout,"x25=standard\n");

    if (ifinfo->if_flags & IFF_DOES_X25basic && conf.c_flags & FL_X25basic)
	fprintf(stdout,"x25=basic\n");

    if (ifinfo->if_flags & IFF_DOES_DTE_DCE) {
	if (conf.c_flags & FL_DCE)
	    fprintf(stdout,"rs232-role=dce\n");
	else
	    fprintf(stdout,"rs232-role=dte\n");
    }

    return (TRUE);
} /* end ifconfig() */


void usage()
{
  fprintf(stderr,"\
usage: trans [-f] [-a | <driver-name>]\n\
       trans [[-? | -h] | [-version]]\n\
\t-a\tauto-configure: search config.sys for driver names\n\
\t-f\tparse and append old FTP_CONFIG file\n\n\
    Translate in-memory ifcust and ipcust drivers to pctcp.ini file format.\n\
    Directs output to stdout;  Use redirection to put into a file.\n\n\
\tExamples:\n\
\t\ttrans -a > pctcp.ini\n\
\t\ttrans -af > pctcp.ini\n\
\t\ttrans -f ifcust > pctcp.ini\n\
\t\ttrans wd8003 > pctcp.ini\n\
\n\
    Add the following line to your AUTOEXEC.BAT file:\n\
\n\
\t\tSET PCTCP=c:\\pctcp.ini\n\
\n\
     (Where \"c:\\pctcp.ini\" can be replaced by the directory path and file\n\
      name you are redirecting output to.  Be careful not to overwrite an\n\
      existing pctcp.ini or ftp_config.txt file when you redirect with the\n\
      DOS '>' operator.  If any error occurs it will be put in the file as\n\
      a comment.)");
} /* end usage() */


struct drv_list *get_devices (void)
{
    char	buf[128];
    char	*tmp_ptr, *t1, *t2;
    struct drv_list *list_ptr, *new_ptr;
        
    list_ptr = (struct drv_list *) 0;
    
    /* open "config.sys" */
    if (config_open ("c:/config.sys", 0) == FALSE)	
	if (config_open ("a:/config.sys", 0) == FALSE)	/* try again */
	    return (list_ptr);

    /* check for ifconfig devices */
    while (config_read ("device", CONF_ANY_INSTANCE, buf, sizeof(buf), 0))
    {
	/* strip off ".sys" */
	tmp_ptr = strrchr (buf, '.');
	if (tmp_ptr == NULL)
	    continue;
	*tmp_ptr = '\0';

 	/* find name of driver */
	if ((tmp_ptr = strchr (buf, ':')) != NULL)
	    tmp_ptr++;	/* skip the drive */
	else
	    tmp_ptr = buf;

	t1 = strrchr (tmp_ptr, '\\');
	t2 = strrchr (tmp_ptr, '/');

	if (t1 > t2)
	    tmp_ptr = ++t1;
	if (t1 < t2)
	    tmp_ptr = ++t2;
	    
	if (!check_driver (tmp_ptr))
	    continue;	/* it's not a driver */

	/* add driver to list */
	if ((tmp_ptr == NULL) || !strlen (tmp_ptr))
	    continue;
	
	new_ptr = (struct drv_list *) malloc (sizeof (struct drv_list ));
	if (new_ptr == (struct drv_list *) 0)
	{
	    fprintf(stdout,
		";; Could not allocate memory for device driver\n");
	    if (!isatty(fileno(stdout))) {
		fprintf(stderr,
		    ";; Could not allocate memory for device driver\n");
	    }
	    goto close_up;
	}
	strcpy (new_ptr->name, tmp_ptr);

	if (list_ptr == (struct drv_list *) 0)
	{
	    /* set up new list */
	    list_ptr = new_ptr;
	    list_ptr->next = (struct drv_list *) 0;
	}
	else
	{
	    /* insert new ptr into list */
	    new_ptr->next = list_ptr->next;
	    list_ptr->next = new_ptr;
	}
    }

close_up:
    config_close (0);  /* close up file */
    
    /* return list ptr */
    return (list_ptr);
}  /* end of get_devices */


int copy_old_config (void) 
{
    char	buffer [256];
    char	package [MAX_LABEL_LEN], sect [MAX_LABEL_LEN];
    char	subsect [MAX_LABEL_LEN], var_name [MAX_LABEL_LEN];
    struct pr_tab *p;				/* trans names*/
    
    /* get old ftp_config file and open it */
    strcpy (buffer, getenv ("FTP_CONFIG"));
    if (!strlen (buffer) || !config_open (buffer, 0)) {
	fprintf(stdout, ";; Copying old ftp_config failed.  \
Could not find old file\n");
	if (!isatty(fileno(stdout))) {
	    fprintf(stderr, ";; Copying old ftp_config failed.  \
Could not find old file\n");
	}
	return (FALSE);	/* give up */
    }
    
    fprintf(stdout,"\n");		/* whitespace between prev section */

    /* these need to be empty */
    package[0] = sect[0] = subsect[0] = '\0';

    /* loop through ftp_config file */
    while (config_seek (package, sect, subsect, CONF_NEXT_SECTION))
    {
	/* check if section_name was used by trans */
	for (p = trans_names; p->val != -1; p++) {
	    if (stricmp (sect, p->name) == 0) 	/* do we have a match... */
		goto loop_again;		/*   skip this section */
	}
	
	/* try checking for a driver section */
	if (check_driver (sect)) {
	    fprintf(stderr,"\nSkipping old %s section\n", sect);
	    goto loop_again;		/* don't copy driver sections */
	}

        /* check if it is a vmail section */
        for (p = client_names; p->val != -1; p++)
	    if (stricmp (sect, p->name) == 0) {
		convert_vmail_sect (sect);
		goto loop_again;
	    }

	/* append section name */
	if (strlen (subsect))
	    fprintf (stdout, "[%s %s %s]\n", package, sect, subsect);
	else
	    fprintf (stdout, "[%s %s]\n", package, sect);

	var_name[0] = '\0';
	/* copy rest of section */
	while (config_read (var_name, CONF_ANY_INSTANCE, buffer,
	    sizeof(buffer), 0))
	{
	    /* append entry */
	    if (strlen (buffer))
		fprintf (stdout, "%s=%s\n", var_name, buffer); 
	    else
		fprintf (stdout, "%s\n", var_name);

	    var_name[0] = '\0';
	}

	/* add a blank line */
	fprintf (stdout, "\n");

loop_again:
	/* these need to be empty */
	package[0] = sect[0] = subsect[0] = '\0';
    }

    config_close (0);
    return (TRUE);
} /* end of copy_old_config() */

#if 0		/* now in config library */
/* rcq 7/21/92 THIS ROUTINE IS NOW DEFINED IN GET_CUST.C */
int check_driver (char *drv_name)
{
    struct pr_tab *p;

    /* check if driver is one of ours */
    for (p = drv_names; p->val != -1; p++)
	if (stricmp (drv_name, p->name) == 0)
	    return (TRUE);
    return (FALSE);
}	
#endif
	

/* This section is incomplete */

/* function for dealing with vmail and its clients 
 *
 * It will only work if the name of the section is vmail, nntp, 
 *  pcmail, pop2 or pop3
 */

int convert_vmail_sect (char * sect_name)
{
    static int	first_vmail_p = TRUE;
    char 	buffer [64], client_sect [64];
    
    /* figure out section name */
    if (stricmp (sect_name, "vmail") != 0)
	strcpy (client_sect, sect_name);
    else    /* if vmail, change name to PROTOCOL */
    {
	/* get protocol name */

	/* check section for protocol */
	if (config_read ("protocol", CONF_ANY_INSTANCE, buffer, sizeof(buffer), 0)) {
	    strcpy (client_sect, buffer);
	}
	else {
	    /* if no protocol, search for port number */
	    if (config_read ("repository-port", CONF_ANY_INSTANCE, buffer, sizeof(buffer), 0))
		strcpy (client_sect, "pcmail");
	    else if (config_read ("pop2-port", CONF_ANY_INSTANCE, buffer, sizeof(buffer), 0))
		strcpy (client_sect, "pop2");
	    else if (config_read ("pop3-port", CONF_ANY_INSTANCE, buffer, sizeof(buffer), 0))
		strcpy (client_sect, "pop3");
	    else if (config_read ("nntp-port", CONF_ANY_INSTANCE, buffer, sizeof(buffer), 0))
		strcpy (client_sect, "nntp");
	    else {
		fprintf (stdout, ";; Couldn't split vmail section\n", sect_name);
		if (!isatty(fileno(stdout))) {

		    fprintf (stderr, "Couldn't split vmail section\n", sect_name);
		}
		return (FALSE);
	    }
	}

	/* rewind to begining of section */
	config_seek ("pctcp", sect_name, NULL, CONF_SAME_SECTION);
    }

    /* write out client name */
    fprintf (stdout, "[pctcp %s]\n", client_sect);

    /* write out section, skiping vmail variables */
    write_variables (FALSE);

    fprintf (stdout, "\n");

    /* have we already printed a vmail section? */
    if (first_vmail_p == TRUE)
	first_vmail_p = FALSE;
    else {
	fprintf(stderr,"\nSkipping old vmail section of %s\n", sect_name);
	return (TRUE);		/* we already have a vmail section */
    }

    /* rewind to begining of vmail section */
    config_seek ("pctcp", sect_name, NULL, CONF_SAME_SECTION);
    
    fprintf (stdout, "\n[pctcp vmail]\n");
    /* add "client_section" */
    fprintf (stdout, "client-section=%s\n", client_sect);

    /* write out section, printing vmail variables */
    write_variables (TRUE);

    fprintf (stdout, "\n");
    return (TRUE);
}  /* end of convert_vmail_sect () */


void write_variables (int write_vmail_p)
{
    char	buffer [256], var_name [MAX_LABEL_LEN];
    struct pr_tab *p;				/* trans names*/

    var_name[0] = buffer[0] = '\0';

    while (config_read (var_name, CONF_ANY_INSTANCE, buffer, sizeof(buffer), 0))	
    {
	/* check if a vmail variable */
	for (p = vmail_vars; p->val != -1; p++) {
	    if (strnicmp (var_name, p->name, 8) == 0) {
		break;
	    }
	}

	if ((write_vmail_p  && (p->val == -1)) ||
	    (!write_vmail_p && (p->val != -1))) {
	    var_name[0] = '\0';		/* clear old name */
	    continue;
	}

	/* append entry */
	if (strlen (buffer))
	    fprintf (stdout, "%s=%s\n", var_name, buffer); 
	else
	    fprintf (stdout, "%s\n", var_name);

	var_name[0] = '\0';		/* clear old name */
    }
} /* end of write_variable () */


static void				/* 29-dec-91 created arnoff	    */
trans_section(char *sect)		/* copy old fulll section to output */
{
    char var_name[128];
    char buffer[128];
    var_name[0] = '\0';

    if (config_seek ("pctcp", sect, CONF_ANY, 0) == TRUE) {
	/* copy rest of section */
	while (config_read (var_name, CONF_ANY_INSTANCE, buffer,
	    sizeof(buffer), 0))
	{
	    /* append entry */
	    if (strlen (buffer))
		fprintf (stdout, "%s=%s\n", var_name, buffer); 
	    else
		fprintf (stdout, "%s\n", var_name);
	    var_name[0] = '\0';
	}
    }
} /* end trans_section() */

/*
 * $Log:   E:/22vcs/srccmd/trans/trans.c_v  $
 * 
 *    Rev 1.6   01 Jan 1993 16:05:28   snyders
 * Fixed includes to come from the right place (round 2).
 * 
 *    Rev 1.5   05 Oct 1992 14:11:26   rcq
 * qualified rwconf as pctcp/rwconf.h
 * 
 *    Rev 1.4   26 Jul 1992 11:09:04   natalia
 * removed check_driver (now in config library)
 * 
 *    Rev 1.3   20 Jul 1992 20:25:44   paul
 * set stack size for Turbo/Borland version
 * 
 *    Rev 1.2   03 Feb 1992 22:15:36   arnoff
 * pre beta-2 testing freeze
 * 
 *    Rev 1.1   29 Jan 1992 23:07:18   arnoff
 *  
 */
