/*
 * $Header:   J:/22vcs/srclib/socket/send.c_v   1.5.1.1   21 Apr 1993 11:27:48   rcq  $
 */

/*
 * SEND.C - Emulate 4BSD functions send() & sendto() for PC/TCP.
 *
 * Copyright (C) 1987-1992 by FTP Software, Inc.  All rights reserved.
 *
 * This software is furnished under a license and may be used and copied
 * only in accordance with the terms of such license and with the
 * inclusion of the above copyright notice. This software or any other
 * copies thereof may not be provided or otherwise made available to any
 * other person. No title to and ownership of the software is hereby
 * transferred.
 *
 * The information in this software is subject to change without notice
 * and should not be construed as a commitment by FTP Software, Inc.
 *
 * Edit History
 * 14-Oct-87	jbvb	Minor coding error corrected.
 * 18-Nov-87	jbvb	Don't allow MSG_OOB on either send() or sendto()
 *			 unless socket is STREAM.
 * 10-Dec-87	jbvb	Better parameter checking.
 *			Allow sendto() on STREAM sockets, but just pass the
 *			 call to send().
 *			Don't allow INADDR_ANY on STREAM sockets.
 *			Map NET_ERR_NOTESTAB to EWOULDBLOCK for HP #860.
 * 07-Jan-88	jbvb	Allow sendto() on connected DGRAM socket if 'to'
 *			 argument is NULL (HP #907).  Don't allow sendto()
 *			 with a 'to' argument on connected DGRAM (HP #807).
 * 14-Jan-88	jbvb	Don't allow length argument bigger than 32767.  Fixes
 *			 HP #804.
 * 15-Jan-88	jbvb	In sendto(), return the right error for "to" arg set
 *			 on stream socket.  Fixes HP #943.
 * 21-Jan-88	jbvb	For sendto() on STREAM, ignore 'to' (per B. Votipka)
 *			 Should finally fix HP #943.
 * 28-Jan-88	jbvb	Convert NET_ERR_NOTESTAB into ENOTCONN (HP #976)
 *			Convert NET_ERR_RESET to ENETRESET (TCP gave up)
 *			Return EMSGSIZE if the kernel wouldn't send all of
 *			 our DGRAM data in one packet (HP #880).
 * 11-Feb-88	jbvb	Kernel can return NET_ERR_NOMEM on buffer deadlock,
 *			 convert it into ENOBUFS.  Also, turn NET_ERR_EOF
 *			 into ESHUTDOWN.
 * 23-Mar-88	jbvb	Pass zeroed addresses to kernel on net_writeto() that
 *			 wants to go to default address (instead of NULL).
 * 26-Aug-88	jbvb	Don't require INADDR_ANY for SO_BROADCAST send() or
 *			 sendto() (kernel sends to configued IP broadcast
 *			 address regardless).
 * 23 AUG 89	stev	fixed compiler warning about types mismatch on 
 *			 routines using sockaddr_in here but sockaddr 
 *			 in the header file.
 * 07-Nov-91	paul	changed to new-style function declarators,
 *			added function return types,
 *			changed forever loops from while(1) to for(;;)
 * 15-May-92	Ben	Dropped MSG_DONT_SEND, per jbvb's request.
 * 18-May-92	Ben	Moved above #ifdefs, so case won't return an error.
 * 14-Aug-92    rcq     updated the copyright in comments
 * 09-Sep-92	rcq	printf() only when DEBUGging (not in default)
 * 12-Nov-92	rcq	pneterror() only when DEBUG manifest defined
 * 03-Mar-93	rcq	changed SO_BROADCAST so it will only do link layer
 *			 bcast if IP addr is legit bcast addr (like BSD does)
 * 03-Mar-93	rcq	removed requirement in sendto() for socket to be bound
 *			 (its implicitly bound automatically).
 * 17-Mar-93	rcq	added support for TCP_NODELAY option (do net_flush())
 * 19-Mar-93	rcq	added include of ipconfig.h to get opt_port structure
 *			 definition for config.h, so BCC doesn't choke.
 * 21-Apr-93	rcq	need to preserve net_write() return before net_flush()
 */

#define XDEBUG

#include <stdio.h>
#include <stdlib.h>

#include <pctcp/error.h>
#include <pctcp/types.h>
#include <pctcp/pctcp.h>
#include <pctcp/ipconfig.h>
#include <pctcp/config.h>

#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>

#include <debug.h>

#include "4bsd.h"

int check_bcast (long);  /* local function */

int
send(int s, char *buf, int len, int flags)
{
	SOCKET	*sp = NSOCK(s);          /* ptr to socket structure */
	int	iflags;                  /* work variables */
	u_short	mask;
	int	n;

#ifdef	DEBUG
	printf("send(s = %d, buf = x%Np, len = %d, flags = x%x)\n",
		s, buf, len, flags);
#endif
	if(sp == NULL || is_netnd(sp->nd))
		bomb(ENOTSOCK);

	if(! (STATE(sp) & SI_CONNECTED))
		bomb(ENOTCONN)			/* PAP's macro */
	else if(! (STATE(sp) & SI_CAN_SEND))
		bomb(ESHUTDOWN)
	else if (len < 0)			/* HP #804 */
		bomb(EINVAL);

	/* convert flags from 4bsd */
	for(iflags = 0, mask = 1; flags && mask != 0; mask <<= 1) {
		switch(mask & flags) {
		case 0:
			continue;
		case MSG_OOB:
			if (sp->type != STREAM)
				bomb(EOPNOTSUPP);
			iflags |= NET_FLG_OOB;
			break;
		case MSG_DONTROUTE:
#ifdef undef	/* James say's we don't need this option. */
			iflags |= NET_FLG_DONTROUTE;
#endif
			break;
		default:
#ifdef DEBUG		    
			printf("send: discarding option x%x\n",
				mask & flags);
#endif
			break;
		}
		flags ^= mask;
	}
	flags = iflags | SOPTIONS(sp);

	/* If SO_BROADCAST option set to allow broadcasts, we will do
         *  a Link Layer broadcast IFF a broadcast address is passed! 
	 */
	if(SFLAGS(sp) & SO_BROADCAST) {    /* broadcast allowed? */
		n = check_bcast((long)SADDR(sp)); /* check for bcast addr */ 
		if (n == -1) {
			bomb(-1);         /* exit if error */
#ifdef DEBUG		    
			pneterror("bsd_send: check_bcast()");
#endif		    
		} else {
			flags |= n;        /* or set bit flag (may be 0 */
		}			   /*  if dest addr not bcast */
	}
		

	if((n = net_write(sp->nd, buf, len, flags)) == -1)
		switch(neterrno) {
		case NET_ERR_EOF:
			bomb(ESHUTDOWN);
		case NET_ERR_RESET:
			bomb(ENETRESET);	/* Kernel dropped it */
		case NET_ERR_NOTESTAB:
			bomb(ENOTCONN);		/* HP #976 */
		case NET_ERR_WOULD_BLOCK:
			bomb(EWOULDBLOCK);
		case NET_ERR_NOMEM:		/* Buffer deadlock */
			bomb(ENOBUFS);
		default:
#ifdef DEBUG		    
			pneterror("bsd_send: net_write");
#endif		    
			bomb(-1);
		}

	if ((sp->type == DGRAM) && (n != len))	/* Much easier to return */
		bomb(EMSGSIZE);			/*  this after the fact... */
	
	if (sp->flags & TCP_NODELAY)  { /* wouldn't be set if DGRAM */
#ifndef XDEBUG
		net_flush (sp->nd);
#else	
		if ((net_flush (sp->nd)) < 0)
			pneterror("send():net_flush()");
#endif	
	}

	dreturn(" = %ld\n", n);
}

int
sendto(int s, char *buf, int len, int flags, 
	struct sockaddr *to_passed, int tolen)
{
	struct sockaddr_in	*to;
	SOCKET	*sp = NSOCK(s);
	ADDR	na;			/* PC/TCP addr structure */
	int	iflags;
	u_short	mask;
	int	n;

	to = (struct sockaddr_in *) to_passed;
#ifdef	DEBUG
	printf("sendto(s=%d, buf=x%Np, len=%d, flags=x%x, to=%s, \
tolen = %d)\n",
		s, buf, len, flags, inet_ntoa(to->sin_addr.s_addr), tolen);
#endif

	if(sp == NULL || is_netnd(sp->nd))
		bomb(ENOTSOCK);
/*
 * HP says sendto() is allowed on STREAM sockets, but PC/TCP won't allow
 * net_writeto() on TCP, so we just treat it as a time-wasting method of
 * calling send().
 */
	if (sp->type == STREAM)
		return(send(s, buf, len, flags));
#if 0
	if(! (STATE(sp) & SI_BOUND)) /* Sun doesn't require it to be bound. */
		bomb(EINVAL);
#endif
	if (len < 0)			/* HP #804 */
		bomb(EINVAL);

	/* convert flags from 4bsd */
	for(iflags = 0, mask = 1; flags && mask != 0; mask <<= 1) {
		switch(mask & flags) {
		case 0:
			continue;
		case MSG_OOB:
			bomb(EOPNOTSUPP);
		case MSG_DONTROUTE:
#ifdef undef	/* James say's we don't need this option. */
			iflags |= NET_FLG_DONTROUTE;
#endif
			break;
		default:
#ifdef DEBUG		    
			printf("sendto: discarding option x%x\n",
				mask & flags);
#endif
			break;
		}
		flags ^= mask;
	}
	flags = iflags | SOPTIONS(sp);

/*
 * HP says sendto() on a connected DGRAM socket is allowed if the to
 * structure pointer is NULL.  However, if the 'to' argument is present,
 * I can't allow it if connected.  Here we go....
 */
	na.protocol = sp->protocol;		/* This is always set */

	if (tolen != sizeof(*to)) {		/* HP says must be equal */
		if (to == NULL) {			/* Is it there? */
		    if (!(STATE(sp) & SI_CONNECTED))	/* No: connected? */
			bomb(ENOTCONN);			/* No: flush it */
		    na.fhost =			/* Not there, but connected */
		    na.fsocket =		/* Address set earlier by */
		    na.lsocket = 0;		/*  net_connect() call. */
		} else
			bomb(EFAULT);		/* Passed bad parameters */
	} else {
		if(to->sin_family != AF_INET)
			bomb(EAFNOSUPPORT);
		if ((STATE(sp) & SI_CONNECTED))	/* Connected? 'to' illegal */
			bomb(EISCONN);		/* PAP's macro */

		na.fhost = to->sin_addr.s_addr;
		na.fsocket = ntohs(to->sin_port);
		na.lsocket = sp->lsocket;
	}

	/* If socket is not bound yet, we need to bind it since the
	 *  PC/TCP kernel will return NET_ERR_UNBOUND otherwise!
	 */
	if(! (STATE(sp) & SI_CAN_SEND)) {
		if (net_connect(sp->nd, DGRAM, &na) < 1)
			bomb(ESHUTDOWN);
		sp->state |= SI_CAN_SEND;  /* change state now */
	}

	/* If SO_BROADCAST option set to allow broadcasts, we will do
         *  a Link Layer broadcast IFF a broadcast address is passed! 
	 */
	if (SFLAGS(sp) & SO_BROADCAST)
	if(SFLAGS(sp) & SO_BROADCAST) {    /* broadcast allowed? */
		n = check_bcast((long)na.fhost); /* check for bcast addr */ 
		if (n == -1) {
			bomb(-1);          /* exit if error */
#ifdef DEBUG		    
			pneterror("sendto: check_bcast");
#endif		    
		} else {
			flags |= n;        /* or set bit flag (may be 0 */
		}			   /*  if dest addr not bcast */
	}

	if((n = net_writeto(sp->nd, buf, len, &na, flags)) == -1)
		switch(neterrno) {
		case NET_ERR_EOF:
			bomb(ESHUTDOWN);
		case NET_ERR_NOHOST:
			bomb(EADDRNOTAVAIL);
		default:
#ifdef DEBUG		    
			printf ("sendto (addr = %s)\n",inet_ntoa(na.fhost));
			pneterror("sendto: net_writeto");
#endif		    
			bomb(-1);
		}

	if (n != len)			/* Noone here but us DGRAMS... */
		bomb(EMSGSIZE);		/*  so this must be what happened */

	dreturn(" = %ld\n", n);
}

/* function: check_bcast()
 *
 * description: checks the destination ip address passed and returns a
 *   broadcast bit flag if its a legitimate broadcast address, otherwise
 *   0 or -1 if in error.
 */
int check_bcast (long dest_ip_addr) {
	static  long my_subcast_addr;  /* these needed for bcasts (static */
	static  long my_bcast_addr=0L; /*  so we only get them once) */

	long my_ip_addr;		/* work variables */
	long my_subnet_mask;
	int len = 4;

#ifdef DEBUG
	printf("send()/sendto(): checking broadcast address: %s\n",
		inet_ntoa(dest_ip_addr));
#endif
	/* Init broadcast addrs if not done yet */
	if (!my_bcast_addr) {
		if (get_kernel_info(0,TAG_IP_ADDR,0,
			            (void far *) &my_ip_addr,
			            (int far *)  &len)) {
			bomb(-1);
		}
#ifdef DEBUG
	printf("send()/sendto(): kernel local IP addr: %s\n",
		inet_ntoa(my_ip_addr));
#endif
		if (get_kernel_info(0,TAG_SUBNET_MSK,0,
				    (void far *) &my_subnet_mask,
				    (int far *)  &len)) {
			bomb(-1);
		}
			
#ifdef DEBUG
	printf("send()/sendto(): kernel subnet mask: %s\n",
		inet_ntoa(my_subnet_mask));
#endif
		/* make subnet broadcast address */
		my_subcast_addr = ((my_ip_addr & my_subnet_mask) |
					((u_long)~my_subnet_mask));
						    
#ifdef DEBUG
	printf("send()/sendto(): subnet broadcast addr created: %s\n",
		inet_ntoa(my_subcast_addr));
#endif
		if (get_kernel_info(0,TAG_BCAST_ADDR,0,
				    (void far *) &my_bcast_addr,
				    (int far *)  &len)) {
			bomb(-1);
		}
#ifdef DEBUG
	printf("send()/sendto(): kernel broadcast addr: %s\n",
		inet_ntoa(my_bcast_addr));
#endif
	}
		
	/* Do Link Layer broadcast subnet or config'd broadcast
	 *  address is the destination address
	 */
	if ((dest_ip_addr == my_subcast_addr) ||
	    (dest_ip_addr == my_bcast_addr)) {
		return (NET_FLG_BROADCAST);
	} else {
		return (0);
	}
} /* end check_bcast() */

/*
 * $Log:   J:/22vcs/srclib/socket/send.c_v  $
 * 
 *    Rev 1.5.1.1   21 Apr 1993 11:27:48   rcq
 * changed call to net_flush() so return from net_write() is preserved
 * in the support of TCP_NODELAY
 * 
 *    Rev 1.5.1.0   09 Apr 1993 15:12:34   rcq
 * added TCP_NODELAY support, allow sendto before bound
 * 
 *    Rev 1.5   11 Nov 1992 19:54:34   rcq
 * pneterror() only if DEBUG manifest defined
 * 
 *    Rev 1.4   02 Oct 1992 18:58:38   rcq
 * 
 *    Rev 1.4   27 Aug 1992 16:00:20   arnoff
 *  * 14-Aug-92    rcq     updated the copyright in comments
 * 
 *    Rev 1.3   18 May 1992 18:15:04   arnoff
 * Moved above #ifdefs, so case won't return an error.   ---Ben
 * 
 * 
 *    Rev 1.2   15 May 1992 11:10:14   arnoff
 * Dropped MSG_DONT_SEND, per jbvb's request.   ---Ben
 * 
 *    Rev 1.1   30 Jan 1992 00:51:52   arnoff
 *  
 */
