/*
 * $Header:   J:/22vcs/srclib/socket/listen.c_v   1.3   02 Oct 1992 18:56:50   rcq  $
 */

/*
 * LISTEN.C - Emulate 4BSD listen() call on PC/TCP 2.0.
 *
 * Copyright (C) 1987-1992 by FTP Software, Inc.  All rights reserved.
 *
 * This software is furnished under a license and may be used and copied
 * only in accordance with the terms of such license and with the
 * inclusion of the above copyright notice. This software or any other
 * copies thereof may not be provided or otherwise made available to any
 * other person. No title to and ownership of the software is hereby
 * transferred.
 *
 * The information in this software is subject to change without notice
 * and should not be construed as a commitment by FTP Software, Inc.
 *
 * Edit History
 * 13-Oct-87	jbvb	Fix it so illegal backlog value gets nearest legal
 *			 value (per HP defect CNOdm00609).
 * 30-Oct-87	jbvb	Better debugging messages.
 * 18-Dec-87	jbvb	Philip was using NET_OPT_NONBLOCKING where he meant
 *			 NET_FLG_NONBLOCKING (in SOPTIONS).
 * 18-Jan-88	jbvb	In incoming(), don't relisten() if there aren't any
 *			 more descriptors.
 * 21-Jan-88	jbvb	Copy l_intrvl for incoming connections.
 * 23-Jan-88	jbvb	Allow listen() to be called a second time, to re-fill
 *			 the queue or change the backlog (only up, just now).
 *			 Add new routine bsd_fill_con_q() for use by accept().
 *			Rename relisten() to bsd_relisten, global for accept()
 * 26-Jan-88	jbvb	Correct backlog range-limiting logic.
 * 03-Feb-88	jbvb	Fix arguments to calloc().
 * 04-Feb-88	jbvb	Terminate queue properly if net_getdesc() fails in
 *			 bsd_fill_con_q().
 * 18-Feb-88	jbvb	Handle case where listen() is called first, with a
 *			 backlog of 1, and a connection comes in, but isn't
 *			 accepted, so incoming doesn't re-listen.  Then,
 *			 listen() is called again, with backlog = 2, so we
 *			 should listen again.  Fixes HP #940.
 * 03-May-88	jbvb	Error returns from net_listen() were faulty.
 * 06-Jun-88	jbvb	Add extra cast on incoming() for ANSI/MSC braindamage
 * 03-Aug-89	jbvb	Clear SI_LISTEN_ERR when bsd_listen() works.
 * 23 AUG 89	stev	fix cast in call to net_asynch() (unsigned long)
 * 13-Dec-89	jbvb	ditto.
 * 31-Jan-91	wcl	Fixed warning message in net_asynch() call - had to
 *			 cast incoming() to (int).
 * 07-Nov-91	paul	changed to new-style function declarators,
 *			added function return types,
 *			changed forever loops from while(1) to for(;;)
 * 10-Mar-92	paul	bogus cast in bsd_relisten ( (u_long)(int)incoming )
 *			 was preventing large-model code from connecting
 * 14-Aug-92    rcq     updated the copyright in comments
 * 15-Sep-92	rcq	eliminated set_option call if NET_FLG_NONBLOCKING
 *			since ioctl() and fcntl() do it now.
 */

#include <stdio.h>
#include <stdlib.h>

#include <pctcp/error.h>
#include <pctcp/types.h>
#include <pctcp/pctcp.h>
#include <pctcp/asynch.h>
#include <pctcp/options.h>

#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include <debug.h>

#include "4bsd.h"

extern int bsd_relisten(SOCKET *sp);
extern int bsd_fill_con_q(SOCKET *sp);
extern void ipanic(char *s, ...);

typedef	int	(far *pff)();
int	asynch_stub();

int
listen(int s, int backlog)
/* s		socket					*/
/* backlog	Number of pending connections to allow	*/ 
{
	SOCKET	*sp = NSOCK(s);
	int	i;
	int	quit = 0;
	NCONN	*nc;

#ifdef	DEBUG
	printf("listen(s = %d, backlog = %d)", s, backlog);
#endif

	if(sp == NULL || is_netnd(sp->nd))
		bomb(ENOTSOCK);

	if(sp->type != STREAM)
		bomb(EOPNOTSUPP);

	if (STATE(sp) & SI_CONNECTED)	/* We can't be connected... */
		bomb(EISCONN)

	if (backlog < 1)		/* HP says this is how it works */
		backlog = 1;
	else if (backlog > MAXLISTEN)	/* Peak-limit the backlog */
		backlog = MAXLISTEN;

	/* If we haven't been bound, we make opimistic
	 * assumptions (wildcard local address).
	 */
	if(! (STATE(sp) & SI_BOUND)) {
		sp->state |= SI_BOUND;	/* do an impromtu bind */
		sp->lsocket = 0;
		sp->lhost = 0L;
	}

	/* search connection queue for unconnected descriptor
	 * (marked by being negative).
	 */
	disable_asynch();		/* No race conditions here */
	for(nc = sp->nconn_q; nc && nc->nd >= 0; nc = nc->next)
		;
	i = (int) nc;			/* Save old queue state */

	sp->max_q_len = backlog;	/* Load the length to use */

	quit = bsd_fill_con_q(sp);	/* Fill (or re-fill) the queue */

#ifdef DEBUG
	printf("error = %d, real backlog is %d\n", quit, sp->max_q_len);
#endif

	if ((!(STATE(sp) & SI_LISTENING)) &&	/* If first time through */
	    (quit != 0))			/*  & couldn't fill queue */
		goto flushit;			/*  go to error return. */

	if (i == 0) {				/* If not listening before */
		for(nc = sp->nconn_q; nc && nc->nd >= 0; nc = nc->next)
			;			/* Scan for unused descr. */
		if (nc != NULL)			/* If descriptors available */
			quit = bsd_relisten(sp);/*  start listening now. */
	}
	if (quit) {				/* If relisten failed... */
flushit:	while (nc = sp->nconn_q) {	/* Clean out pending queue */
		    if (nc->nd < 0)		/* If descriptor unclaimed */
			nc->nd *= -1;		/*  fix it up for kernel */
		    net_release(nc->nd);	/* Free kernel resources */
		    sp->nconn_q = nc->next;	/* Follow the link */
		    free(nc);			/* Free the memory */
		}
		enable_asynch();
		bomb(quit);			/* Return error to caller. */
	}

	enable_asynch();			/* Now, allow interrupts */

	if(i == 0)				/* If actually listening... */
		sp->state |= SI_LISTENING;	/*  indicate it */

	return(0);				/* Return success */
}

/*
 * listen - asynchronous handler for incoming connection requests on
 * listening socket.  net_swap() the connection onto a valid descriptor
 * from the queue, which accept() will return later.  After the descriptor
 * is net_swap()'d, we need to do a new net_listen() on that socket, but
 * only if there is a queued descriptor for it to use.
 */

static int
incoming(int nd, int event, struct addr far *from)
{
	SOCKET	*sp = NSOCK(nd);
	int	i = 0;
	NCONN	*nc;

#ifdef	AS_DEBUG
	dbg("incoming(nd = %d, event = x%02x, from = x%Fp)\n",nd,event,from);
#endif
	
	if(event != NET_AS_OPEN)
		ipanic("incoming: (funky event %d)\n", event);

	if(sp == NULL)
		ipanic("incoming: %d (funky handle)\n", nd);

	/* search connection queue for unconnected descriptor
	 * (marked by being negative).
	 */
	for(nc = sp->nconn_q; nc && nc->nd >= 0; nc = nc->next)
		;

	/* Further on, we refuse to re-listen if there isn't a descriptor
	 * on which to accept the next connection.  Thus, this should never
	 * happen......
	 */
	if (nc == NULL) {
		ipanic("incoming: no connection.");
		return(0);
	}
		
	/* grab pre-alloc'd descriptor, which we will shortly swap with
	 * connected descriptor.  then we will place previously listening
	 * descriptor back into the listening state.
	 */
	sp->cd = (nc->nd *= -1);

	if(net_swap(sp->nd, sp->cd))
		ipanic("incoming: net_swap: %d\n", neterrno);

#ifdef	AS_DEBUG
	dbg("new conn = x%Np, nd = %d\n", nc, nc->nd);
#endif

	/* save rest of association...
	 */
	nc->fhost = from->fhost;
	nc->fsocket = from->fsocket;
	nc->lsocket = from->lsocket;
	nc->protocol = from->protocol;
	nc->l_intrvl = sp->l_intrvl;

	/* this should be part of the addr structure, but...
	 */
	if((nc->lhost = get_addr(nc->nd)) == 0L)
		ipanic("incoming: get_addr: %d, (handle %d)\n",
			neterrno, nc->nd);

#ifdef	AS_DEBUG
	dbg("fhost %08lx:%d  lhost %08lx:%d\n",
	  nc->fhost, nc->fsocket, nc->lhost, nc->lsocket);
#endif

	++(sp->cur_q_len);

#ifdef	AS_DEBUG
	dbg("calling bsd_relisten\n");
#endif
	if (nc->next)			/* If it is going to work, */
		i = bsd_relisten(sp);	/*  do the next net_listen() */

#ifdef	AS_DEBUG
	dbg("bsd_relisten returns %d\n", i);
#endif
	return(i);
}

/* bsd_relisten - put the given descriptor into a listening, nonblocking
 * state with address previously specified.  set up handler for the
 * incoming connection requests (first).  This is called by accept(), too.
 */

int
bsd_relisten(SOCKET *sp)
{

	/* set up handler to catch connects on the fly.  see above. */
	if (net_asynch(sp->nd, NET_AS_OPEN, asynch_stub, (u_long)incoming)
	    == (pff) -1)
		return EFAULT;

	/* so the following net_listen() doesn't block indefinitely...
	 */
	if (set_option(sp->nd, sp->type, NET_OPT_NONBLOCKING,(char far *)1L,0)
		== -1)
		return EFAULT;

	/* this could fail if the foreign socket/host pair is specific,
	 * because it will already be in use on subsequent calls.
	 */
#ifdef AS_DEBUG
	printf("re-listening %s: %d on local %d", inet_ntoa(SADDR(sp)->fhost),
		SADDR(sp)->fsocket, SADDR(sp)->lsocket);
#endif
	if(net_listen(sp->nd, sp->type, SADDR(sp)) == -1) {
		switch (neterrno) {
			case NET_ERR_NOMEM:
				return (ENOBUFS);
			case NET_ERR_INUSE:
				return (EADDRINUSE);
			default:
				return (EFAULT);
		}
	}
#ifdef AS_DEBUG
	printf("re-listening %s: local port: %d", inet_ntoa(SADDR(sp)->fhost),
		SADDR(sp)->lsocket);
#endif

	if(! (SOPTIONS(sp) & NET_FLG_NONBLOCKING))
		if(set_option(sp->nd, sp->type, NET_OPT_NONBLOCKING, 0L, 0)
		    == -1)
			return EFAULT;

	sp->state &= ~SI_LISTEN_ERR;		/* TCP listening ok */
	return(0);
}


/*
 * bsd_fill_con_q() - common routine to fill or re-fill the queue of pending
 * connections on socket sp to backlog.  Returns 0 if success, error code
 * if it can't hack it.  This is called by accept, too.
 */

bsd_fill_con_q(SOCKET *sp)
/* sp		Socket to re-fill queue */
{
	int	quit	= 0;	/* Return code */
	int	i	= 0;	/* Number of elements in queue */
	NCONN	**nc;		/* use me to pre-alloc nconn's */
	int	backlog;	/* Desired # pending connections (try) */

	backlog = sp->max_q_len;	/* Figure out how many to try for */
	
	/*
	 * Munch down the queue of pending connections, adding connections
	 * if the queue is shorter than 'backlog' connections.
	 * If we aren't able to add one, fail gracefully.
	 */
	for (nc = &(sp->nconn_q); i < backlog; ++i, nc = &((*nc)->next)) {
	    if (*nc == NULL) {
		if ((*nc = (NCONN *) calloc(1, sizeof(NCONN)))
			== NULL) {
		    quit = ENOBUFS;
		    break;
		} else if (((*nc)->nd = net_getdesc() * -1) == 1) {
		    free(*nc);
		    *nc = NULL;		/* Re-terminate queue */
		    quit = EMFILE;
		    break;
		}
	    }
	}		/* List is already terminated (by use of calloc()) */

#ifdef	DEBUG
	{	
		NCONN	*tc;
		for(tc = sp->nconn_q; tc != NULL; tc = tc->next)
			printf("tc = x%Np, nd = %d\n", tc, tc->nd);
	}
#endif
	return (quit);
}

/*
 * $Log:   J:/22vcs/srclib/socket/listen.c_v  $
 * 
 *    Rev 1.3   02 Oct 1992 18:56:50   rcq
 * merged changes done in 2.1
 * 
 *    Rev 1.3   27 Aug 1992 15:56:12   arnoff
 *  * 14-Aug-92    rcq     updated the copyright in comments
 * 
 *    Rev 1.2   10 Mar 1992 12:29:34   arnoff
 * Fixed a bogus cast in bsd_relisten ( (u_long)(int)incoming )
 * that was preventing large-model code from connecting.
 * 
 *    Rev 1.1   30 Jan 1992 00:51:34   arnoff
 *  
 */
