/*
 * $Header:   J:/22vcs/srclib/socket/getsocko.c_v   1.4.1.0   13 Apr 1993 18:38:46   rcq  $
 */

/*
 * SOCKET\GETSOCKO.C - Contains 4bsd functions setsockopt() and getsockopt().
 *
 * Copyright (C) 1987-1992 by FTP Software, Inc.  All rights reserved.
 *
 * This software is furnished under a license and may be used and copied
 * only in accordance with the terms of such license and with the
 * inclusion of the above copyright notice. This software or any other
 * copies thereof may not be provided or otherwise made available to any
 * other person. No title to and ownership of the software is hereby
 * transferred.
 *
 * The information in this software is subject to change without notice
 * and should not be construed as a commitment by FTP Software, Inc.
 *
 * Edit History
 * 14-Oct-87	jbvb	Correct the format of the set_option() call according
 *			 to the option being set.
 * 07-Jan-88	jbvb	Don't allow LINGER on DGRAM, or BROADCAST on STREAM.
 *			Accept SO_REUSEADDR for both getsockopt() and
 *			 setsockopt() (part of HP #s 752 & 885).
 *			Re-write getsockopt() to comform to HP-UX manual
 *			 (return 0 if set).  Fixes HP #849, maybe others.
 * 13-Jan-88	jbvb	Fix a minor typo.
 * 14-Jan-88	jbvb	Comment out code that processes SO_DEBUG (the kernel
 *			 doesn't yet handle NET_OPT_DEBUG).  Fixes HP #802.
 * 15-Jan-88	jbvb	If user passes getsockopt() optval & optlen arguments,
 *			 set the optlen to 0 (no options have values yet).
 * 21-Jan-88	jbvb	SO_LINGER has a value (for 4.2 linger implementation).
 * 22-Jan-88	jbvb	Options don't have values, except for SO_LINGER, and
 *			 can't be cleared (except for linger/dontlinger).
 *			 This should finally fix HP #752.
 * 02-May-90	jbvb	Add SO_ACCEPTCONN, SO_SNDLOWAT, SO_RCVLOWAT,
 *			 SO_SNDTIMEO and SO_RCVTIMEO to list of socket options
 *			 accepted but ignored.
 * 31-Jan-91	wcl	fixed 'parameter declaration different' error
 *			 in set sockopt (union *optval -> char *optval)
 * 07-Nov-91	paul	changed to new-style function declarators,
 *			added function return types,
 *			changed forever loops from while(1) to for(;;)
 * 26-Feb-92	paul	added include pctcp/error.h for pneterror
 * 14-Aug-92    rcq     updated the copyright in comments
 * 09-Sep-92	rcq	printf() only with DEBUG manifest defined
 * 12-Nov-92	rcq	pneterror() only with DEBUG manifest defined
 * 17-Mar-93	rcq	added support for TCP_NODELAY
 * 17-Mar-93	rcq	fixed default so rather than just setting an option,
 *			 it checks for optval (on/off switch) and uses it if
 *			 a pointer was passed to enable or disable option!
 */

#include <stdio.h>
#include <stdlib.h>

#include <pctcp/types.h>
#include <pctcp/pctcp.h>
#include <pctcp/options.h>
#include <pctcp/error.h>	/* pneterror */

#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>

#include <debug.h>

#include "4bsd.h"

int
setsockopt(int s, int level, int optname, char *optval, int optlen)
/* s		socket to check					*/
/* level	Level (only SOL_SOCKET at the moment)		*/
/* optname	Number of the option to check			*/
/* optval	Place to put value				*/
/* optlen	Length of place - length of value on ret	*/
{
	SOCKET	*sp;
	int	option;
	int	arglen;
	union {
		long	num;
		char far *ptr;
	} arg;

	union {
		char		c;
		short		s;
		long		l;
		struct linger	plinger;
	} *optval_union;


	(char *)(optval_union) = optval;

/*	static char	dbgfile[] = "con"; */

#ifdef	DEBUG
	printf("setsockopt(s = %d, level = %d, optname = %d, optval = x%Np, \
optlen = %d)",
		s, level, optname, optval, optlen);
#endif

	if(is_netnd(s) || (sp = NSOCK(s)) == NULL)
		bomb(ENOTSOCK);

	if(level == SOL_SOCKET) {

	  switch(optname) {
	  case SO_DEBUG:
		option = NET_OPT_DEBUG;
/* Kernel support not yet implemented */
/*		if (optval_union->l) {
			arg.ptr = (char far *) dbgfile;
			arglen = sizeof(dbgfile);
		} else
			arg.num = arglen = 0;
		if(set_option(sp->nd, sp->type, option, (char far *) &arg,
				arglen) == -1) {
#ifdef DEBUG				
			pneterror("setsockopt: set_option");
#endif		    
			bomb(-1);
		}
*/
		break;
	  case SO_KEEPALIVE:
		option = NET_OPT_KEEPALIVE;
boolean:	arg.num = TRUE;
		arglen = 0;
		if (set_option(sp->nd, sp->type, option, (char far *)arg.num,
				arglen) == -1) {
#ifdef DEBUG				
			pneterror("setsockopt: set_option");
#endif		    
			bomb(-1);
		}
		SFLAGS(sp) |= SO_KEEPALIVE;		/* add to flags	*/
		break;
	  case SO_DONTROUTE:
		option = NET_OPT_DONTROUTE;
		goto boolean;
	  /* this is an option that doesn't require
	   * immediate kernel interaction...
	   */
	  case SO_LINGER:
		if (sp->type == DGRAM)
			bomb(EOPNOTSUPP);
		if (optval == NULL)		/* This one needs a value */
			bomb(EINVAL);
		if (optval_union->plinger.l_onoff == 0)  {
		    SFLAGS(sp) &= ~SO_LINGER;		/* Turn off DO */
		    SFLAGS(sp) |= SO_DONTLINGER;	/* Turn on DONT */
		}
		else  {
		    SFLAGS(sp) &= ~SO_DONTLINGER;	/* Turn off DONT */
		    SFLAGS(sp) |= SO_LINGER;		/* Turn on DO */
		}
		    /* Set interval */
		sp->l_intrvl = (u_char) optval_union->plinger.l_linger;
		return(0);				/* All done */

	  case SO_DONTLINGER:
		if (sp->type == DGRAM)
			bomb(EOPNOTSUPP);
		SFLAGS(sp) &= ~SO_LINGER;		/* Turn off DO */
		break;					/* to common code */
	  case SO_BROADCAST:
		if (sp->type == STREAM)
			bomb(EOPNOTSUPP);
		break;
	  case SO_TYPE:
		if (sp->type == DGRAM)  {
		   *(short *) optval = SOCK_DGRAM;
		}
		else if (sp->type == STREAM)  {
		    *(short *) optval = SOCK_STREAM;
		}
	  /* These are all accepted but ignored...*/
	  case SO_ACCEPTCONN:
	  case SO_SNDLOWAT:
	  case SO_RCVLOWAT:
	  case SO_SNDTIMEO:
	  case SO_RCVTIMEO:
	  case SO_ERROR:
	  case SO_REUSEADDR:
	  case SO_SNDBUF:
	  case SO_RCVBUF:
	  case SO_OOBINLINE:
		break;
	  default:
#if DEBUG	    
		printf("setsockopt: unknown option x%x\n", optname);
#endif	    
		bomb(EINVAL);
	  }
	} else if (level == IPPROTO_TCP) {

	  if (sp->type == DGRAM)    /* only valid for TCP of course! */
		bomb(EOPNOTSUPP);
	    
	  switch (optname) {
	  case TCP_NODELAY:
		break;
	  default:
#if DEBUG	    
		printf("setsockopt: unknown option x%x\n", optname);
#endif	    
        	bomb(EINVAL);
          }
	} else
	    bomb (EINVAL);    /* unsupported "option level" */

	SFLAGS(sp) |= optname;	/* assume we want to set option ON */
	
	/* But if ptr to switch value passed, then turn off if requested */
	if ((optval) && (!(*optval)))  
		SFLAGS(sp) &= ~optname;		/* switch OFF */

	dreturn(" = %d\n", 0);
}

/*
 * getsockopt() - Per the HP-UX manual, returns 0 if specified option is
 * set, -1 and ENOPROTOOPT if it isn't.  Options that have values (only
 * SO_LINGER) will get them returned in optval, with optlen set to the size.
 */

int
getsockopt(int s, int level, int optname, char *optval, int *optlen)
/* s		socket to check					*/
/* level	Level (only SOL_SOCKET at the moment)		*/
/* optname	Number of the option to check			*/
/* optval	Place to put value				*/
/* optlen	Length of place - length of value on ret	*/
{
	SOCKET	*sp;

#ifdef DEBUG
	printf("setsockopt(s = %d, level = %d, optname = %d, optval = x%Np, \
optlen = %d)",
		s, level, optname, optval, optlen);
#endif

	if(is_netnd(s) || (sp = NSOCK(s)) == NULL)
		bomb(ENOTSOCK);

	if(level == SOL_SOCKET) {
	    switch(optname) {
	    case SO_LINGER:
		if (sp->type == DGRAM)
			bomb(EOPNOTSUPP);	/* Fall through */
		if ((SFLAGS(sp) & SO_LINGER) == 0)	/* Option set? */
			bomb(ENOPROTOOPT);		/* No: return error */
		if ((optval != NULL) &&
		    (optlen != NULL)) {
			if (*optval < sizeof(u_char))	/* Enough room? */
			    bomb(EFAULT);		/* No: error */
			*(long *) optval = sp->l_intrvl;	
			*optlen = sizeof(u_char);
		}
		return (0);			/* All done */
	    case SO_BROADCAST:
		if (sp->type == STREAM)
			bomb(EOPNOTSUPP);
		break;
	    case SO_DONTLINGER:
		if (sp->type == DGRAM)
			bomb(EOPNOTSUPP);	/* Fall through */
	    case SO_ACCEPTCONN:
	    case SO_SNDLOWAT:
	    case SO_RCVLOWAT:
	    case SO_SNDTIMEO:
	    case SO_RCVTIMEO:
	    case SO_ERROR:
	    case SO_KEEPALIVE:
	    case SO_REUSEADDR:
	    case SO_DEBUG:
	    case SO_DONTROUTE:
	    case SO_SNDBUF:
	    case SO_RCVBUF:
	    case SO_OOBINLINE:
		break;
	    default:
		bomb(EINVAL);
	    }
	} else if (level == IPPROTO_TCP) {
	    if (sp->type == DGRAM)
		bomb(EOPNOTSUPP);	/* Fall through */
	    
	    switch (optname) {
	      case TCP_NODELAY:
		break;
	      default:
		bomb(EINVAL);
	    }
	} else
	    bomb (EINVAL);    /* unsupported "option level" */

	if ((optval != NULL) && (optlen != NULL)) {	/* Wants the value */
		*optlen = 0;			/* No value: len set to 0 */
	}

	if ((SFLAGS(sp) & optname) == 0)	/* Is the option set? */
		bomb(ENOPROTOOPT);		/* No: return proper error */
	dreturn(" = %d\n", 0);			/* Yes: return no error */
}

/*
 * $Log:   J:/22vcs/srclib/socket/getsocko.c_v  $
 * 
 *    Rev 1.4.1.0   13 Apr 1993 18:38:46   rcq
 * added TCP_NODELAY & made setsockopt() toggle options on or off
 * 
 *    Rev 1.4.1.0   09 Apr 1993 15:11:42   rcq
 * added TCP_NODELAY support and made setsockopt toggle option
 * 
 *    Rev 1.4   11 Nov 1992 19:54:14   rcq
 * pneterror() only if DEBUG manifest defined
 * 
 *    Rev 1.3   02 Oct 1992 18:54:02   rcq
 * merged changes done in 2.1
 * 
 *    Rev 1.3   27 Aug 1992 15:53:12   arnoff
 *  * 14-Aug-92    rcq     updated the copyright in comments
 * 
 *    Rev 1.2   13 Apr 1992 16:02:12   arnoff
 * Added include pctcp/error.h for pneterror
 * 
 *    Rev 1.1   30 Jan 1992 00:51:18   arnoff
 *  
 */
