/******************************************************************************
      (C) Copyright 1992 by Autodesk, Inc.

      This program is copyrighted by Autodesk, Inc. and is  licensed
      to you under the following conditions.  You may not distribute
      or  publish the source code of this program in any form.   You
      may  incorporate this code in object form in derivative  works
      provided  such  derivative  works  are  (i.) are  designed and 
      intended  to  work  solely  with  Autodesk, Inc. products, and 
      (ii.)  contain  Autodesk's  copyright  notice  "(C)  Copyright  
      1992 by Autodesk, Inc."

      AUTODESK  PROVIDES THIS PROGRAM "AS IS" AND WITH  ALL  FAULTS.
      AUTODESK  SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF  MER-
      CHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK,  INC.
      DOES  NOT  WARRANT THAT THE OPERATION OF THE PROGRAM  WILL  BE
      UNINTERRUPTED OR ERROR FREE.

*******************************************************************************/

/* For the Metaware High C and High C/C++ compilers, turn off an
   unwanted warning message */
#ifdef __HIGHC__
pragma Offwarn(67); /* kills "switch statement has no cases" warning */
#endif

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include "sxp.h"

#define LIM255(c) (((c)>=255)?255:(c))

/*Variable definitions */
#define SIZE 1
#define WIDTH 2
#define COL1 4
#define COL2 5
/* The above definitions are the LineID numbers for the dialog lines in which the variables will be set. 
The integers assigned to the dialog lines containing variables are selected by the IPAS programmer and 
used by 3D Studio to identify the variables. */

/************************/

/* Dialog description */
DlgEntry cdialog[]={
	0,"TITLE=\"3D Marble Function\"",			/* Text to appear in dialog box */
    SIZE, "LFLOAT=\"Size:\",0.0001,10000.0,5",         /*  Setup for 'limited float' type
								variable and associated range 
								values */
    WIDTH, "-LFLOAT=\"Width:\",0.0,10000.0,5",      /* Limited float variable */
	0,"TITLE=\"\"",						/* Blank line for spacing */
	COL1,"COLOR=\"Color 1\"",				/* Setup for standard color triple 	
								slider dialog*/
	COL2,"-COLOR=\"Color 2\"",				/* Setup for standard color triple 	
								slider dialog*/
	0,"TITLE=\"\"",						/* Blank line for spacing */
	0,NULL
	};

/* Version test value */
/* Every external process must have a unique version number  */
#define MARBLE_VERS 0xE001
	
/* State Structure definition */
/* Any variables whose values are set via dialog interface must be declared within the 'State' struct. */	
typedef struct {
	ulong version;
	float size;
	float width;
	Col24 col1,col2;
	} MarbleState;

/* the "state" struct MUST start with a "ulong" which is the version#,
	to prevent using data from old versions of this program.
	This verification is performed automatically. */

static MarbleState init_state = { MARBLE_VERS,60.0,.02,{0.0,0.0,0.0},{230,204,155}};     /*Initial */
static MarbleState state = { MARBLE_VERS,60.0,.02,{0.0,0.0,0.0},{230,204,155}};	/*Default state */
/*----------------------------------------------------------------*/


#define NOISE_DIM 20    
#define FNOISE_DIM 20.0

static float noise_table[NOISE_DIM+1][NOISE_DIM+1][NOISE_DIM+1];

void init_noise() {
	int i,j,k,ii,jj,kk;
	for (i=0; i<=NOISE_DIM; i++)
		for (j=0; j<=NOISE_DIM; j++)
			for (k=0; k<=NOISE_DIM; k++) {
				noise_table[i][j][k] = (float)(rand()&0x7FFF);
				ii = (i==NOISE_DIM)?0:i; 
				jj = (j==NOISE_DIM)?0:j; 
				kk = (k==NOISE_DIM)?0:k; 
				noise_table[i][j][k] = noise_table[ii][jj][kk];
				}
	}


float noise(float x,float y,float z) {
	int ix,iy,iz;
	float fx,fy,fz,mx,my,mz;
	float n,n00,n01,n10,n11,n0,n1;

	mx = fmod(x,FNOISE_DIM); if (mx<0) mx += FNOISE_DIM;
	my = fmod(y,FNOISE_DIM); if (my<0) my += FNOISE_DIM;
	mz = fmod(z,FNOISE_DIM); if (mz<0) mz += FNOISE_DIM;
	ix = (int)mx;
	iy = (int)my;
	iz = (int)mz;
	fx = fmod(mx,1.0);
	fy = fmod(my,1.0);
	fz = fmod(mz,1.0);
	n = noise_table[ix][iy][iz];
	n00 = n + fx*(noise_table[ix+1][iy][iz]-n);
	n = noise_table[ix][iy][iz+1];
	n01 = n + fx*(noise_table[ix+1][iy][iz+1]-n);
	n = noise_table[ix][iy+1][iz];
	n10 = n + fx*(noise_table[ix+1][iy+1][iz]-n);
	n = noise_table[ix][iy+1][iz+1];
	n11 = n + fx*(noise_table[ix+1][iy+1][iz+1]-n);
	n0 = n00 + fy*(n10-n00);
	n1 = n01 + fy*(n11-n01);
	return(((float)(n0+fz*(n1-n0)))/32768.0);
	}


static void lerp_color(Col24 *c, Col24 *a, Col24 *b, float f) {
	int alph,ialph;
	if (f>1.0) f = 1.0;
	alph = (int)4096*f;
	ialph = 4096-alph;
	c->r = (ialph*a->r + alph*b->r)>>12;
	c->g = (ialph*a->g + alph*b->g)>>12;
	c->b = (ialph*a->b + alph*b->b)>>12;
	}


#define WD .02
#define FACT 500.0

float marble(float x, float y, float z) {
	int id;
	float d,i;
	x *= FACT/state.size;
	y *= FACT/state.size;
	z *= FACT/state.size;
	d = (x+10000.0)*state.width + 7*noise(x/100,y/200,z/200);
	id = ((int)d)%17;
	if (id<4)
		i = 0.7+0.2*noise(x/70,y/50,z/50);
	else {
		if (id<9 || id>=12) {
			d = fabs(d- ((int)d/17)*17-10.5)*0.1538462;
			i = 0.4 + 0.3*d + 0.2*noise(x/100,y/100,z/100);
			}
		else 
			i = 0.2*(1.0 + noise(x/100,y/100,z/100));
		}
	return(i);
	}

/* The following function is a 'Client' procedure that defines a 3-dimensional function to modify the 
diffuse color and/or the shininess. This function is passed the SXPDATA structure contained in the 
EXPbuf, but has no access to packet commands or to other EXPbuf data structures. */

void Client3DText(Sxpdata *sd) {
	float i;
	i = marble(sd->p[0],sd->p[1],sd->p[2]);
	lerp_color(&sd->col,&state.col1,&state.col2,i);
	}

/* The following function is a 'Client' procedure that defines a 3-dimensional function to compute an 
opacity value. This function is passed the SXPDATA structure contained in the EXPbuf, but has no access 
to packet commands or to other EXPbuf data structures. */

void Client3DOpac(Sxpdata *sd) {
	float i;
	i = marble(sd->p[0],sd->p[1],sd->p[2]);
	lerp_color(&sd->col,&state.col1,&state.col2,i);
	}

/* The following function is a 'Client' procedure that defines a 3-dimensional function to compute a 
normal perturbation value. This function is passed the SXPDATA structure contained in the EXPbuf, but 
has no access to packet commands or to other EXPbuf data structures. */

void Client3DBump(Sxpdata *sd) {
	float del,d,x,y,z;
	x = sd->p[0];
	y = sd->p[1];
	z = sd->p[2];
	d = marble(x,y,z);
	del = .2;
	sd->norm_pert[0] = (marble(x+del,y,z) - d)/del;
	sd->norm_pert[1] = (marble(x,y+del,z) - d)/del;
	sd->norm_pert[2] = (marble(x,y,z+del) - d)/del;
	}

/* Following function sets the appropriate 'State' variables based
   on the input from the dialog */
void ClientSetStateVar(int id, void *ptr) {
	OVL o;				/* Union of possible values that the dialog can return */
	ulong *ul;

	ul=(ulong *)ptr;
	o.ul = *ul;
	switch(id) {
		case SIZE:  state.size = o.f; break;			/* State variables set here based on
								values stored in the fields of the 
								Union type .*/
		case WIDTH: state.width = o.f; break;
		case COL1: 					/* Set each field within the COL 
								variable in RGB format */
			state.col1.r = o.c.r;
			state.col1.g = o.c.g;
			state.col1.b = o.c.b;
			break;
		case COL2: 					/* Set each field within the COL 
								variable in RGB format */
			state.col2.r = o.c.r;
			state.col2.g = o.c.g;
			state.col2.b = o.c.b;
			break;
		}
	}

/* Following function sets the appropriate value of the Union 'OVL' variable
   to return values stored in the 'State' variable */
ulong ClientGetStateVar(int id) {
	OVL o;
	switch(id) {
		case SIZE: o.f = state.size; break;		/* Union variables set to appropriate value
							from 'state' variables */
		case WIDTH: o.f = state.width; break;
		case COL1:
			o.c.r = state.col1.r;
			o.c.g = state.col1.g;
			o.c.b = state.col1.b;
			break;
		case COL2:
			o.c.r = state.col2.r;
			o.c.g = state.col2.g;
			o.c.b = state.col2.b;
			break;
		}
	return(o.ul);					/* the variable requested by 3D Studio is 
							returned to 3D Studio via the 'ulong' field
							of the 'Union' structure 'OVL'. */
	}

/* ---For most applications, the following functions will not change */

/* Following function allows 3D Studio to determine the size of
   the 'State' structure and returns a pointer to the 'State' struct */

char  *ClientGetState(int *size) {
	*size = sizeof(MarbleState);
	return((char *)&state);
	}

/* Following function allows 3D Studio to reset the state of the
   external process */

void ClientResetState(void) {	
	state = init_state;	
	}

/* The following function performs any calculations necessary to the operation of 3DTEXT, 3DBUMP, or 
3DOPAC that cannot be performed within any of these functions. The 'ClientStartUp' procedure does not 
have access to packet-commands. */

void ClientStartup(void) {
	/* allocate data structures, compute lookup tables, etc */
	init_noise();
	}

void ClientTerminate(void) { 
	/* free any data structures, etc. */
	}

/* The following function allows 3D Studio to get the address of a string containing a variable. The IPAS
programmer will not have to call this function or modify the following code, but will have to include it */

DlgEntry *ClientDialog(int n) {	
	return(&cdialog[n]); 
	}

/* Following function allows 3D Studio to determine the size of
   any variable in the dialog */

int ClientVarSize(int id) {
	switch(id) {
		default:
			return(1);
		}
	}
