/******************************************************************************
      (C) Copyright 1992 by Autodesk, Inc.

      This program is copyrighted by Autodesk, Inc. and is  licensed
      to you under the following conditions.  You may not distribute
      or  publish the source code of this program in any form.   You
      may  incorporate this code in object form in derivative  works
      provided  such  derivative  works  are  (i.) are  designed and 
      intended  to  work  solely  with  Autodesk, Inc. products, and 
      (ii.)  contain  Autodesk's  copyright  notice  "(C)  Copyright  
      1992 by Autodesk, Inc."

      AUTODESK  PROVIDES THIS PROGRAM "AS IS" AND WITH  ALL  FAULTS.
      AUTODESK  SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF  MER-
      CHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK,  INC.
      DOES  NOT  WARRANT THAT THE OPERATION OF THE PROGRAM  WILL  BE
      UNINTERRUPTED OR ERROR FREE.

*******************************************************************************/

/*       3D Studio's       */
/*   Wave-object Process   */
/*  by Tom Hudson 11/14/91 */

/* PXP process raw client routine core code */

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include "pxp.h"
#include "trig.h"

/* Types of waves this thing can make... */

#define XY 0
#define XZ 1
#define YZ 2
#define XYZ 3

/* Variable definitions */

#define TYPE 1
#define PERIOD1 2
#define WAVELENGTH1 3
#define AMP1 4
#define PERIOD2 5
#define WAVELENGTH2 6
#define AMP2 7
#define MORPHS 8
#define HIDE 9
#define PREFIX 10

/* Function declarations */
void warp(float *x,float *y,float *z);

/* The above definitions are the LineID numbers for the dialog lines in
   which the variables will be set. The integers assigned to the dialog
   lines containing variables are selected by the IPAS programmer and
   used by 3D Studio to identify the variables. */

/************************/

/* Dialog description */

DlgEntry cdialog[]={
	0,"TITLE=\"Autodesk 3D Studio Object Wave Generator\"",	/* Text to appear in dialog box */
    0,"TITLE=\"by Tom Hudson, Version 1.0\"",               /* Text to appear in dialog box */

    TYPE,"RADIO=\"Amplitude axis:\",\"Top\",\"Front\",\"Right\",\"All\"", /* Setup for radio
                                                                             button and
                                                                             variable */
    0,"TITLE=\"\"",                         /* Blank line for spacing */
	0,"TITLE=\"WAVE 1:\"",					/* Text to appear in dialog box */
    PERIOD1,"LINT=\"Period:\",0,500,3",     /* Setup for 'limited integer type
                                               variable and associated range
                                               values */
	WAVELENGTH1,"-LFLOAT=\"Wavelength:\",0.0,99999,8",	/*  Setup for 'limited float' type 
                                                            variable and associated range
                                                            values */
    AMP1,"-LFLOAT=\" Amplitude:\",0.0,99999,8",  /* Limited float variable */
    0,"TITLE=\"WAVE 2:\"",                       /* Text to appear in dialog box */
    PERIOD2,"LINT=\"Period:\",0,500,3",          /* Limited integer variable */
	WAVELENGTH2,"-LFLOAT=\"Wavelength:\",0.0,99999,8",	/* Limited float variable */
    AMP2,"-LFLOAT=\" Amplitude:\",0.0,99999,8",         /* Limited float variable */
    0,"TITLE=\"\"",                                     /* Blank line for spacing */
    MORPHS,"LINT=\"# of morphs:\",1,999,3",      /* Limited integer variable */
    HIDE,"-YES-NO=\"Hide >0?\"",                 /* Setup for 'Yes-No' button */
    PREFIX,"-STRING=\"Name prefix:\",7,7",       /* Setup for editable text field */
    0,"TITLE=\"\"",                              /* Blank line for spacing */
	0,NULL
	};

/* Version test value */
/* Every external process must have a unique version number  */

#define VERSION 0x1EE0

/* State Structure definition */
/* Any variables whose values are set via dialog interface must be declared
   within the 'State' struct. */
typedef struct {
    ulong version;   /* required to be first field within 'State' struct */
	int type;			
	int period1;
	float wavelength1;
	float amp1;
	int period2;
	float wavelength2;
	float amp2;
	int morphs;
	int hide;
	char prefix[8];
	} State;

/* the "state" struct MUST start with a "ulong" which is the version number,
   to prevent using data from old versions of this program. This verification
   is performed automatically. */

static State init_state = { VERSION,0,10,10.0,1.0,5,10.0,1.0,10,1,"WAVE" };	/* initial state settings */
static State state = { VERSION,0,10,10.0,1.0,5,10.0,1.0,10,1,"WAVE" };      /* default state settings */

static int verts,faces,tverts /**** ,ix */;
static int index,vertex,face;
static int morphnum;

/*----------------------------------------------------------------*/
/* Following function sets the appropriate 'State' variables based
   on the input from the dialog */

void ClientSetStateVar(int id, void *ptr) {
    OVL o;         /* Union of possible values that the dialog can return */
	ulong *ul;
	char *s;

	ul=(ulong *)ptr;
	s=(char *)ptr;
	o.ul = *ul;
	switch(id) {
        case TYPE: state.type=o.i; break; /* State variables set here based on
                                             values stored in the fields of the
                                             Union type .*/
		case PERIOD1: state.period1=o.i; break;
		case WAVELENGTH1: state.wavelength1=o.f; break;
		case AMP1: state.amp1=o.f; break;
		case PERIOD2: state.period2=o.i; break;
		case WAVELENGTH2: state.wavelength2=o.f; break;
		case AMP2: state.amp2=o.f; break;
		case MORPHS: state.morphs=o.i; break;
		case HIDE: state.hide=o.i; break;
		case PREFIX: strcpy(state.prefix,s); break;
		}
	}

/* Following function sets the appropriate value of the Union 'OVL' variable
   to return values stored in the 'State' variable */

ulong ClientGetStateVar(int id) {
	OVL o;
	switch(id) {
        case TYPE: o.i=state.type; break; /* Union variables set to appropriate value
                                             from 'state' variables */
		case PERIOD1: o.i=state.period1; break;
		case WAVELENGTH1: o.f=state.wavelength1; break;
		case AMP1: o.f=state.amp1; break;
		case PERIOD2: o.i=state.period2; break;
		case WAVELENGTH2: o.f=state.wavelength2; break;
		case AMP2: o.f=state.amp2; break;
		case MORPHS: o.i=state.morphs; break;
		case HIDE: o.i=state.hide; break;
		case PREFIX: o.s=state.prefix; break;
		}
    return(o.ul);        /* the variable requested by 3D Studio is
							returned to 3D Studio via the 'ulong' field
							of the 'Union' structure 'OVL'. */
	}

/* Following function allows 3D Studio to determine the size of
   any variable in the dialog */

int ClientVarSize(int id)
	{
	switch(id)
		{
		case PREFIX: return(2);
		default:
			return(1);
		}
	}

/* Following function allows 3D Studio to determine the size of
   the 'State' structure and returns a pointer to the 'State' struct */

char  *ClientGetState(int *size) {
	*size = sizeof(State);
	return((char *)&state);
	}

/* Following function allows 3D Studio to reset the state of the
   external process */

void ClientResetState() {	
	state = init_state;	
	}

/* Following function performs any calculations or packet-commands
   necessary before the EXP can begin */

void ClientStartup(EXPbuf *buf) {
    buf->opcode=EXP_NOP;  /* Packet command sent to 3D Studio
							 maintaining flow of control. */
    buf->usercode=0x0200; /* Next entry point for the
                             ClientUserCode function */
	}

/* Following function contains user codes which process data */
void ClientUserCode(EXPbuf *buf)
	{
	switch(buf->usercode)
		{
		case 0x0200:
            /* Test for possible object name conflicts */

			index=0;

			conflict_loop:
#define ob buf->data.object
			sprintf(ob.name,"%s%03d",state.prefix,index);	/* print object name to 										lower left screen corner */
#undef ob
            buf->opcode=EXP_OBJ_EXIST; /* Signal 3D Studio to check if an
                                          object with the same name already
                                          exists */
            buf->usercode=0x0205;      /* Next entry point for the
                                          ClientUserCode function */
			break;
		case 0x0205:
            if(buf->status) /* Status = 1 if last operation was
                               successful. Satus = 0 (zero) for failure.
                               buf->status is set automatically by 3D
                               Studio */
				{
				strcpy(buf->data.string,"Name conflict -- use a unique prefix");
				buf->opcode=EXP_CONTINUE;	/* Packet command to maintain 
                                               flow of control. */
                buf->usercode=EXP_TERMINATE; /* Next operation in this
                                                case terminate on err */
				break;
				}
			index++;
			if(index<state.morphs)
				goto conflict_loop;

			buf->opcode=EXP_PICK_OBJECT;	/* Packet command sent to 3D 									Studio initiating a 'Pick-Object'
                                               operation. */
			buf->usercode=0x0210;			/* Next entry point for the
                                               ClientUserCode function */
			strcpy(buf->data.string,"Pick a mesh object to wave"); /* Message displayed in 
                                                                      lower left screen corner */
			break;
  
			case 0x0210:
			if(buf->status==0)
				goto terminate;			/* goto exit loop */
#define co buf->data.object
			/* get data from input object to build output object */
		
			verts=co.verts;				/* record number of vertices */
			tverts=co.tverts;			/* record number of textured vertices */
			faces=co.faces;				/* record number of faces */
#undef co	

			morphnum=0;

			morph_loop:

			/* Set up for an exact copy of this object */

            buf->opcode=EXP_READY_OBJ; /* 3D Studio packet command for
                                          opening a new object for output */
            buf->usercode=0x0215;      /* Next entry point for the
                                          ClientUserCode function */
#define ob buf->data.object
            ob.verts=verts;            /* Set number of vertices for output object*/
            ob.faces=faces;            /* Set number of faces for ouput object */
            ob.tverts=tverts;          /* Number of textured vertices for output */
#undef ob
			break;
		case 0x0215:
			if(buf->status==0)	
                goto terminate;        /* If last operation failed, then terminate */
            buf->opcode=EXP_PROMPT;    /* Signal 3D Studio to display
                                          message to user */
			sprintf(buf->data.string,
			"Creating Wave Object %d...",morphnum);	/* User is given information
                                                       about the operation. The
                                                       buf->data.string's contents will
                                                       be printed in the lower left corner
                                                       of the screen. */
			buf->usercode=0x0220;			/* Next entry point for the
                                               ClientUserCode function */
			break;

			/* Copy face structure */

		case 0x0220:
			face=0;

			face_loop:
			buf->opcode=EXP_GET_FACE;		/* 3D Studio packet command for
                                               sequentially getting 'face' data */
			buf->usercode=0x0225;			/* Next entry point for the
                                               ClientUserCode function */
#define fc buf->data.face
            fc.face=face;                   /* Set current face number */
#undef fc
			break;

		case 0x0225:
			buf->opcode=EXP_PUT_FACE;		/* 3D Studio packet command to
                                               put 'face' data to output object */
			buf->usercode=0x0230;			/* Next entry point for the
                                               ClientUserCode function */
#define fc buf->data.face
			fc.face=face;				
#undef fc
			break;

		case 0x0230:
			face++;
			if(face<faces)
				goto face_loop;			

			/* Faces copied, now copy verts with warping */

			vertex=0;

			vert_loop:
			buf->opcode=EXP_GET_VERTEX;	/* 3D Studio packet command for 
                                           getting vertex information from
                                           input object */
            buf->usercode=0x0235;       /* Next entry point for the
                                           ClientUserCode function */
#define vt buf->data.vert
            vt.vertex=vertex;           /* Set current vertex number */
#undef vt
			break;

		case 0x0235:

			/* Got vertex, now modify XYZ based on parameters */

#define vt buf->data.vert
			switch(state.type)
				{
                /* The vertex parameters are sent to the warp function and
                   altered. Therefore, these vertex parameters will not have
                   to be set later */
				case XY:
					warp(&vt.x,&vt.y,&vt.z);
					break;
				case XZ:
					warp(&vt.x,&vt.z,&vt.y);
					break;
				case YZ:
					warp(&vt.y,&vt.z,&vt.x);
					break;
				case XYZ:
					warp(&vt.x,&vt.y,&vt.z);
					warp(&vt.x,&vt.z,&vt.y);
					warp(&vt.y,&vt.z,&vt.x);
					break;
				}

			buf->opcode=EXP_PUT_VERTEX;	/* 3D Studio packet command for 
                                           writing vertex information to
                                           output object */
            buf->usercode=0x0240;       /* Next entry point for the
                                           ClientUserCode function */
            vt.vertex=vertex;           /* Output vertex number set */
			/* Rest of vertex parameters sent to warp function and modified/set there */
#undef vt
			break;

		case 0x0240:
			vertex++;
			if(vertex<verts)
				goto vert_loop;

			buf->opcode=EXP_CREATE_OBJ;	/* 3D Studio packet command to 
                                           signal completion of output object.
                                           Object can now be displayed on
                                           screen and treated as any other
                                           object created in 3D Studio */
            buf->usercode=0x0245;       /* Next entry point for the
                                           ClientUserCode function */
#define ob buf->data.object
			sprintf(ob.name,"%s%03d",state.prefix,morphnum);	/*Print output obj. name*/
			if(morphnum>0 && state.hide)
				ob.flags=EXP_OBJ_HIDDEN;	/* Output object is hidden upon
                                               creation ie. stored in memory,
                                               but not output to the screen */
			else
				ob.flags=0;			/* Otherwise, object is visible */
#undef ob
			break;

		case 0x0245:
			morphnum++;
			if(morphnum<state.morphs)
				goto morph_loop;

			buf->opcode=buf->usercode=EXP_TERMINATE;	/* Signal 3D Studio to 
									terminate */
			buf->status=1;
			break;

		default:
			terminate:					/* termination routine */
			buf->opcode=buf->usercode=EXP_TERMINATE;
			buf->status=0;					/* set status to 'exit' */
			break;
		}
	}

void ClientTerminate(void) { 
	/* free any data structures, etc. */
	}

/* The following function allows 3D Studio to get the address of the string
   containing the dialogue. The IPAS programmer will not have to call this
   function or modify the following code, but will have to include it */

DlgEntry *ClientDialog(int n) {	
	return(&cdialog[n]); 
	}

/* The following functions were written by the IPAS programmer for this
   specific routine, and therefore, are not examples of required 'Client'
   functions */

/* Wave the point based on parameters */

void warp(float *x,float *y,float *z)
{
float sin1,sin2;

sin1=sin((*x/state.wavelength1+(float)morphnum/(float)state.period1)*PI_360);
sin2=sin((*y/state.wavelength2+(float)morphnum/(float)state.period2)*PI_360);
*z+=(sin1*state.amp1+sin2*state.amp2);
}

