/******************************************************************************
      (C) Copyright 1992 by Autodesk, Inc.

      This program is copyrighted by Autodesk, Inc. and is  licensed
      to you under the following conditions.  You may not distribute
      or  publish the source code of this program in any form.   You
      may  incorporate this code in object form in derivative  works
      provided  such  derivative  works  are  (i.) are  designed and 
      intended  to  work  solely  with  Autodesk, Inc. products, and 
      (ii.)  contain  Autodesk's  copyright  notice  "(C)  Copyright  
      1992 by Autodesk, Inc."

      AUTODESK  PROVIDES THIS PROGRAM "AS IS" AND WITH  ALL  FAULTS.
      AUTODESK  SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF  MER-
      CHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK,  INC.
      DOES  NOT  WARRANT THAT THE OPERATION OF THE PROGRAM  WILL  BE
      UNINTERRUPTED OR ERROR FREE.

*******************************************************************************/

/*       3D Studio's       */
/*   Grid-object Process   */
/*  by Tom Hudson 11/13/91 */


/* For the Metaware High C and High C/C++ compilers, turn off an
   unwanted warning message */
#ifdef __HIGHC__
pragma Offwarn(67); /* kills "switch statement has no cases" warning */
#endif

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include "pxp.h"

/* Types of objects this procedure can make... */

#define XY 0
#define XZ 1
#define YZ 2
#define XYZ 3

/* Variable definitions */

#define LENGTH 1
#define LGRIDS 2
#define WIDTH 3
#define WGRIDS 4
#define HEIGHT 5
#define HGRIDS 6

/* Function declarations */

void XYplane(double z, int facing);
void XZplane(double y, int facing);
void YZplane(double x, int facing);
void makegrid(int axis1, int axis2, int ptbase, int smooth);

/* The above definitions are the LineID numbers for the dialog lines in
   which the variables will be set. The integers assigned to the dialog
   lines containing variables are selected by the IPAS programmer and
   used by 3D Studio to identify the variables. */

/************************/
/* Type definitions unique to this PXP */

typedef struct
{
float x, y, z;
} Vertex;

static Vertex *vdata=NULL;

typedef struct
{
unsigned short a, b, c, flags, smooth;
} Face;

static Face *fdata=NULL;
 
/* Variables */

int dialog_ix, verts, faces, first, ix, onedges, offedges, edges, totlights, edge;
int vertex, face, creverts, crefaces, type, lzero, wzero, hzero, xix, yix, zix;
int topstart, midstart, xzpts, yzpts, xypts;
float minx, miny, minz, maxx, maxy, maxz, dx, dy, dz;
float fromx, fromy, fromz, tox, toy, toz, xstep, ystep, zstep, xpos, ypos, zpos, size2;
unsigned char mtls[4];

/* Dialog description */

DlgEntry cdialog[]={
    0, "TITLE=\"Autodesk 3D Studio Grid Object Generator\"", /* Text to appear in dialog box */
    0, "TITLE=\"by Tom Hudson, Version 1.0\"",
    LENGTH, "LFLOAT=\"Length:\", 0.0, 99999, 8",             /*  Setup for 'limited float' type
                                                                 variable and associated range
                                                                 values */
    LGRIDS, "-LINT=\"Grids:\", 0, 500, 3",                   /* Setup for 'limited integer type
                                                                variable and associated range
                                                                values */
    WIDTH, "LFLOAT=\" Width:\", 0.0, 99999, 8",              /* Limited float variable */
    WGRIDS, "-LINT=\"Grids:\", 0, 500, 3",                   /* Limited integer variable */
    HEIGHT, "LFLOAT=\"Height:\", 0.0, 99999, 8",             /* Limited float variable */
    HGRIDS, "-LINT=\"Grids:\", 0, 500, 3",                   /* Limited integer variable */
    0, "TITLE=\"\"",                                         /* Blank line for spacing */
    0, NULL
	};

/* Version test value */
/* Every external process must have a unique version number */

#define VERSION 0x82F0

/* State Structure definition */
/* Any variables whose values are set via dialog interface must be declared within the 'State' struct. */
	
typedef struct {
	ulong version;
	float length;
	int lgrids;
	float width;
	int wgrids;
	float height;
	int hgrids;
	} State;

/* the "state" struct MUST start with a "ulong" which is the version number,
   to prevent using data from old versions of this program. This verification
   is performed automatically. */

static State init_state={ VERSION, 100.0, 10, 100.0, 10, 100.0, 10 }; /* initial state settings */
static State state={ VERSION, 100.0, 10, 100.0, 10, 100.0, 10 };      /* default state settings */

/*----------------------------------------------------------------*/
/* Following function sets the appropriate 'State' variables based
   on the input from the dialog */

void ClientSetStateVar(int id, void *ptr) {
	OVL o;				/* Union of possible values that the dialog can return */
	ulong *ul;

    ul=(ulong *)ptr;
    o.ul=*ul;
	switch(id) {
        case LENGTH: state.length=o.f; break; /* State variables set here based on
                                                 values stored in the fields of the
                                                 Union type .*/
        case LGRIDS: state.lgrids=o.i; break;
        case WIDTH:  state.width=o.f; break;
        case WGRIDS: state.wgrids=o.i; break;
        case HEIGHT: state.height=o.f; break;
        case HGRIDS: state.hgrids=o.i; break;
		}
	}

/* Following function sets the appropriate value of the Union 'OVL' variable
   to return values stored in the 'State' variable */

ulong ClientGetStateVar(int id) {
	OVL o;
	switch(id) {
        case LENGTH: o.f=state.length; break;   /* Union variables set to appropriate value
                                                   from 'state' variables */
        case LGRIDS: o.i=state.lgrids; break;
        case WIDTH: o.f=state.width; break;
        case WGRIDS: o.i=state.wgrids; break;
        case HEIGHT: o.f=state.height; break;
        case HGRIDS: o.i=state.hgrids; break;
		}
    return(o.ul);        /* the variable requested by 3D Studio is
							returned to 3D Studio via the 'ulong' field
							of the 'Union' structure 'OVL'. */
	}

/* Following function allows 3D Studio to determine the size of
   any variable in the dialog */

int ClientVarSize(int id)
	{
	switch(id)
		{
		default:
			return(1);
		}
	}

/* Following function allows 3D Studio to determine the size of
   the 'State' structure and returns a pointer to the 'State' struct */

char  *ClientGetState(int *size) {
    *size=sizeof(State);
	return((char *)&state);
	}

/* Following function allows 3D Studio to reset the state of the
   external process */

void ClientResetState() {	
    state=init_state;
	}

/* Following function performs any calculations or packet-commands
   necessary before the EXP can begin */

void ClientStartup(EXPbuf *buf) {
    lzero=wzero=hzero=0;     /* Initial calculations performed
								within this client procedure */
    if(state.length==0.0 || state.lgrids==0)
        lzero=1;
    if(state.width==0.0 || state.wgrids==0)
        wzero=1;
    if(state.height==0.0 || state.hgrids==0)
        hzero=1;
	if((lzero+wzero+hzero)>1)
		{
        buf->opcode=EXP_CONTINUE; /* Packet command sent to 3D Studio
                                     maintaining flow of control. */
        buf->usercode=EXP_TERMINATE;  /* Next operation in this case termination
                                         upon error  */
        strcpy(buf->data.string, "Must have two nonzero dimensions");
		return;
		}
	if(hzero)
		{
        creverts=(state.lgrids+1) * (state.wgrids+1);
        crefaces=state.lgrids * state.wgrids * 2;
        type=XY;
		}
	else
	if(lzero)
		{
        creverts=(state.wgrids+1) * (state.hgrids+1);
        crefaces=state.wgrids * state.hgrids * 2;
        type=XZ;
		}
	else
	if(wzero)
		{
        creverts=(state.lgrids+1) * (state.hgrids+1);
        crefaces=state.lgrids * state.hgrids * 2;
        type=YZ;
		}
	else
		{
        creverts=(state.lgrids+1)*(state.wgrids+1)*2+
			(state.lgrids+1)*(state.hgrids+1)*2+
			(state.wgrids+1)*(state.hgrids+1)*2;
        crefaces=state.lgrids * state.wgrids * 4 +
			state.lgrids * state.hgrids * 4 + 
			state.wgrids * state.hgrids * 4;
        type=XYZ;
		}

	if(creverts>65500 || crefaces>65500)
		{
        buf->opcode=EXP_CONTINUE;    /* Packet command sent to 3D Studio
                                        maintaining flow of control. */
        buf->usercode=EXP_TERMINATE; /* Next operation in this case is termination
                                        upon error  */
        strcpy(buf->data.string, "Object exceeds vertex/face limits");
                                 /* Error message to be
									displayed at lower left 
									corner of the screen */
		return;
		}
    if((vdata=(Vertex *)malloc(creverts*sizeof(Vertex)))==NULL)
		{
		noram:
        buf->opcode=EXP_CONTINUE;  /* Packet command sent to 3D Studio
                                      maintaining flow of control. */
        buf->usercode=EXP_TERMINATE;  /* Next operation in this case: termination
                                         upon error  */
        strcpy(buf->data.string, "No RAM for object workspace"); /* Error message */
		}
    if((fdata=(Face *)malloc(crefaces*sizeof(Face)))==NULL)
		goto noram;

    buf->opcode=EXP_READY_OBJ;          /* 3D Studio packet command for
                                           opening a new object for output */
    buf->usercode=0x0200;               /* Next entry point for the
                                           ClientUserCode function */
#define ob buf->data.object
    ob.verts=creverts;              /* Set number of vertices in output object */
    ob.faces=crefaces;              /* Set number of faces in output object */
    ob.tverts=0;                    /* No textured vertices included. */
#undef ob
	}

/* Following function contains user codes which process data */
void ClientUserCode(EXPbuf *buf)
	{
	switch(buf->usercode)
		{
		case 0x0200:
            if(buf->status==0)
                goto terminate;         /* If previous operation failed,
                                           then terminate IPAS routine */
            buf->opcode=EXP_PROMPT;     /* Signal 3D Studio to display
                                           message to user */
            sprintf(buf->data.string,
                "Creating Grid Object type %d (%d verts, %d faces)...",
                type, creverts, crefaces); /* User is given information
                                              about the operation. The
                                              buf->data.string's contents will
                                              be printed in the lower left corner
                                              of the screen. */
            buf->usercode=0x0205;  /* Next entry point for the
                                      ClientUserCode function */
			break;

		case 0x0205:
            buf->opcode=buf->usercode=EXP_TERMINATE; /* 3D Studiocommands
                                                     necessary to signal
                                                     termination */
            if(lzero==0)
                ystep=state.length/state.lgrids;
            if(wzero==0)
                xstep=state.width/state.wgrids;
            if(hzero==0)
                zstep=state.height/state.hgrids;
            vertex=0;
			switch(type)
				{
				case XY:
                    XYplane(0.0, 0);
					break;
				case XZ:
                    XZplane(0.0, 0);
					break;
				case YZ:
                    YZplane(0.0, 0);
					break;
				case XYZ:
                    XYplane(0.0, 1);
                    XYplane(state.height, 0);
                    YZplane(0.0, 1);
                    YZplane(state.width, 0);
                    XZplane(0.0, 1);
                    XZplane(state.length, 0);
					break;
				}
            vertex=0;

            vert_loop:                  /* Label for vertex loop */
            buf->opcode=EXP_PUT_VERTEX; /* 3D Studio packet command for
                                           writing vertex information to
                                           output object */
            buf->usercode=0x0210;       /* Next entry point for the
                                           ClientUserCode function */
#define cv buf->data.vert
            cv.x=vdata[vertex].x;       /* Set 'x' value of vertex */
            cv.y=vdata[vertex].y;       /* Set 'y' value of vertex */
            cv.z=vdata[vertex].z;       /* Set 'z' value of vertex */
            cv.vertex=vertex++;         /* Set vertex # (0 - [total -1]) */
#undef cv
			break;

		case 0x0210:
			if(vertex<creverts)
				goto vert_loop;

			/* Now create the faces for the object */

            face=0;
			switch(type)
				{
				case XY:
                    makegrid(state.lgrids, state.wgrids, 0, 1);
					break;
				case XZ:
                    makegrid(state.hgrids, state.wgrids, 0, 2);
					break;
				case YZ:
                    makegrid(state.hgrids, state.lgrids, 0, 4);
					break;
				case XYZ:
                    xypts=(state.wgrids+1)*(state.lgrids+1);
                    yzpts=(state.lgrids+1)*(state.hgrids+1);
                    xzpts=(state.wgrids+1)*(state.hgrids+1);
                    makegrid(state.lgrids, state.wgrids, 0, 1);
                    makegrid(state.lgrids, state.wgrids, xypts, 1);
                    makegrid(state.hgrids, state.lgrids, xypts*2, 2);
                    makegrid(state.hgrids, state.lgrids, xypts*2+yzpts, 2);
                    makegrid(state.hgrids, state.wgrids, xypts*2+yzpts*2, 4);
                    makegrid(state.hgrids, state.wgrids, xypts*2+yzpts*2+xzpts, 4);
					break;
				}

            face=0;

			face_loop:
            buf->opcode=EXP_PUT_FACE;  /* 3D Studio packet command to
                                          put 'face' data to output object */
            buf->usercode=0x0220;      /* Next entry point for the
                                          ClientUserCode function */
#define cf buf->data.face
            cf.a=fdata[face].a;        /* Record face data describing
                                          which vertices are connected to
                                          each other to form faces */
            cf.b=fdata[face].b;
            cf.c=fdata[face].c;
            cf.flags=fdata[face].flags;     /* Set visibility flags */
            cf.material=255;                /* Set face material */
            cf.sm_group=fdata[face].smooth; /* Set face smoothing group */
            cf.face=face++;                 /* Set face number */
#undef cf
			break;

		case 0x0220:
			if(face<crefaces)
				goto face_loop;
            buf->opcode=EXP_CREATE_OBJ; /* 3D Studio packet command to
                                           signal completion of output object.
                                           Object can now be displayed on
                                           screen and treated as any other
                                           object created in 3D Studio */
            buf->usercode=0x0250;       /* Next entry point for the
                                           ClientUserCode function */
#define ob buf->data.object
            strcpy(ob.name, "");
            ob.flags=0;
#undef ob
			break;

		case 0x0250:
            buf->opcode=EXP_PROMPT;      /* Signal 3D Studio to display
                                            message to user */
            strcpy(buf->data.string, "Done!");
            buf->usercode=EXP_TERMINATE; /* Termination command */
			break;

		default:
			terminate:
            buf->opcode=buf->usercode=EXP_TERMINATE; /* Error termination */
            buf->status=0;                           /* Error status */
			break;
		}
	}

void ClientTerminate(void) { 
    /* free any data structures, etc. */
	}

/* The following function allows 3D Studio to get the address of the string
   containing the dialog box definition. The IPAS programmer will not have
   to call this function or modify the following code, but will have to
   include it */

DlgEntry *ClientDialog(int n) {	
	return(&cdialog[n]); 
	}

/* The following functions were written by the IPAS programmer for this
   specific procedure, and therefore, are not examples of required 'Client'
   functions */

void XYplane(double z, int facing)
{
float xs, xd;

if(facing==0)
 {
 xs=0.0;
 xd=xstep;
 }
else
 {
 xs=state.width;
 xd= -xstep;
 }

for(yix=0, ypos=0.0; yix<=state.lgrids; ++yix, ypos+=ystep)
 {
 for(xix=0, xpos=xs; xix<=state.wgrids; ++xix, xpos+=xd)
  {
  vdata[vertex].x=xpos;
  vdata[vertex].y=ypos;
  vdata[vertex++].z=z;
  }
 }
}

void XZplane(double y, int facing)
{
float xs, xd;

if(facing==1)
 {
 xs=0.0;
 xd=xstep;
 }
else
 {
 xs=state.width;
 xd= -xstep;
 }

for(zix=0, zpos=0.0; zix<=state.hgrids; ++zix, zpos+=zstep)
 {
 for(xix=0, xpos=xs; xix<=state.wgrids; ++xix, xpos+=xd)
  {
  vdata[vertex].x=xpos;
  vdata[vertex].y=y;
  vdata[vertex++].z=zpos;
  }
 }
}

void YZplane(double x, int facing)
{
float ys, yd;

if(facing==0)
 {
 ys=0.0;
 yd=ystep;
 }
else
 {
 ys=state.length;
 yd= -ystep;
 }
for(zix=0, zpos=0.0; zix<=state.hgrids; ++zix, zpos+=zstep)
 {
 for(yix=0, ypos=ys; yix<=state.lgrids; ++yix, ypos+=yd)
  {
  vdata[vertex].x=x;
  vdata[vertex].y=ypos;
  vdata[vertex++].z=zpos;
  }
 }
}

void makegrid(int axis1, int axis2, int ptbase, int smooth)
{
for(yix=0; yix<axis1; ++yix)
 {
 for(xix=0; xix<axis2; ++xix)
  {
  fdata[face].a=yix*(axis2+1)+xix+ptbase;
  fdata[face].b=fdata[face].a+1;
  fdata[face].c=fdata[face].a+axis2+2;
  fdata[face].flags=FC_ABLINE | FC_BCLINE; /* Edge visibility flags for AB and BC edges */
  fdata[face++].smooth=smooth;             /* Set smoothing group bits for each face */
  fdata[face].a=yix*(axis2+1)+xix+ptbase;
  fdata[face].b=fdata[face].a+axis2+2;
  fdata[face].c=fdata[face].a+axis2+1;
  fdata[face].flags=FC_BCLINE | FC_CALINE; /* Edge visibility flags for BC and CA edges */
  fdata[face++].smooth=smooth;             /* Set smoothing group bits for each face */
  }
 }
}

