/******************************************************************************
      (C) Copyright 1992 by Autodesk, Inc.

      This program is copyrighted by Autodesk, Inc. and is  licensed
      to you under the following conditions.  You may not distribute
      or  publish the source code of this program in any form.   You
      may  incorporate this code in object form in derivative  works
      provided  such  derivative  works  are  (i.) are  designed and 
      intended  to  work  solely  with  Autodesk, Inc. products, and 
      (ii.)  contain  Autodesk's  copyright  notice  "(C)  Copyright  
      1992 by Autodesk, Inc."

      AUTODESK  PROVIDES THIS PROGRAM "AS IS" AND WITH  ALL  FAULTS.
      AUTODESK  SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF  MER-
      CHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK,  INC.
      DOES  NOT  WARRANT THAT THE OPERATION OF THE PROGRAM  WILL  BE
      UNINTERRUPTED OR ERROR FREE.

*******************************************************************************/

/* BY TOM HUDSON */

/* For the Metaware High C and High C/C++ compilers, turn off an
   unwanted warning message */
#ifdef __HIGHC__
pragma Off(Prototype_conversion_warn); /* turns off warnings about non standard conversions */
#endif

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include "ixp.h"
#include "trig.h"

typedef float FLOAT;
typedef FLOAT TRow[3];
typedef TRow TMatrix[4];

extern void split_fn(char *path,char *file,char *pf);
extern void split_fext(char *fe,char *f,char *e);
extern void xform_point(float x,float y,float z,
	float *rx,float *ry,float *rz);
extern void frect(float x, float y, float w, float h,int brite);

#define RelScaleX(tm, s) RelScaleRow(tm[0],s) 
#define RelScaleY(tm, s) RelScaleRow(tm[1],s) 
#define RelScaleZ(tm, s) RelScaleRow(tm[2],s)

#define PROJ_ORTHOGRAPHIC 0
#define PROJ_PERSPECTIVE 1

typedef struct
{
float x,y,z,tx,ty,tz,bank,focal;
} Camera;
static Camera FC,*c;

/* Frame params */
/* Values for some of the following will be passed in via 'ClientProcImage' function parameters */

static float dev_aspect,ov_zoom,view_hangle,view_vangle,view_tilt;
static int dev_width,dev_height, view_width, view_type,proj_type;
static float xmat[4][3], i_xmat[4][3],basemat[4][3],refmat[4][3];
static float heading,pitch,bank,camzfac,camw2,camh2,vpt_x,vpt_y,vpt_z;
static float scale,yscale,camdist;
 static float dimmest,brightest;
static float ident_mat[12]={1.0,0.0,0.0,0.0,1.0,0.0,0.0,0.0,1.0,0.0,0.0,0.0};
static FrameInfo *fi;						/* Included in 'ixp.h' file */
static FrameParm *fp;						/* Included in 'ixp.h' file */

/* Star database */

typedef struct
{
float x,y,z,fmag;
unsigned char mag;
} Star;

/* Pointer to star database */

static Star *stardata=NULL;

static int starcount=0;

/* Variables */

static int dim,bright,logarithmic;
static char infile[13];

/* Variable definitions */

#define VDIM 1
#define VBRIGHT 2
#define VSCALE 3
#define VFILE 4

/* Function declarations */

void free_alloced();
void process_image();
int find_angle(float deltax,float deltay,float *angle,float *angle90);
void Translate(TMatrix tmat, FLOAT tx, FLOAT ty, FLOAT tz);
void RelScaleRow(TRow r, FLOAT s);
void RotateX(TMatrix tmat, FLOAT alpha);
void RotateY(TMatrix tmat, FLOAT alpha);
void RotateZ(TMatrix tmat, FLOAT alpha);
void TransformPoint(TMatrix tm, FLOAT *p, FLOAT *pt);
void xform_point(float x,float y,float z,float *rx,float *ry,float *rz);
int build_xmatrix(void);
void ch_replace(char *string,char oldch,char newch);
void split_fn(char *path,char *file,char *pf);
void split_fext(char *fe,char *f,char *e);
int striicmp(char *s1,char *s2);
void plot(int x, int y, uchar v );
void frect(float x, float y, float w, float h,int brite);

/* The above definitions are the LineID numbers for the dialog lines in which the variables will be set. The integers assigned to the dialog lines containing variables are selected by the IPAS programmer and used by 3D Studio to identify the variables. */

/**********************/

/* Dialog description */

DlgEntry cdialog[]={
	0,"TITLE=\"Autodesk 3D Studio Starfield Generator\"",	/* Text to appear in dialog box */
	0,"TITLE=\"by Tom Hudson, Version 1.0\"",		/* Text to appear in dialog box */
	VDIM,"SLIDER=\"  Dimmest star value:\",0,255,20",	/* Setup for slider interface */
	VBRIGHT,"SLIDER=\"Brightest star value:\",0,255,20",	/* Setup for slider interface */
	VSCALE,"RADIO=\"Brightness scale:\",\"Linear\",\"Logarithmic\"",	/* Setup for radio-button*/
	VFILE,"FILENAME=\"Star Database:\"",			/* Setup for editable text field */
	0,"TITLE=\"\"",						/* Blank line for spacing */
	0,NULL
	};

/* Version test value */
/* Every external process must have a unique version number  */

#define VERSION 0x8891

/* State Structure definition */
/* Any variables whose values are set via dialog interface must be declared within the 'State' struct. */	

typedef struct {
	ulong version;		/* required to be first field within 'State' struct */
	int dim,bright,scale;
	char file[13],filler[3];
	} State;

/* the "state" struct MUST start with a "ulong" which is the version#,
	to prevent using data from old versions of this program.
	This verification is performed automatically. */

static State init_state = { VERSION,0,255,0,"EARTH.STB","" };	/* Initial state settings */
static State state = { VERSION,0,255,0,"EARTH.STB","" };		/* Default state settings */

/*----------------------------------------------------------------*/

/*Following function contains all the routines particular to the image processing done by the IXP. The passed parameters contain all the data from an image that an IXP can access/modify. */
int ClientProcImage(FrameParm *f,FrameInfo *i,int state,Report *rpt) {
	fi=i;
	fp=f;

	if(fi->viewtype!=VIEW_CAMERA)	/* If current view is not the camera view */
		{
		error("Can't generate stars in orthogonal view",NULL,NULL,NULL,NULL);
		return(0);			/* 0 (zero) is returned by the 'ClientProcImage' 							function to signal an error. */
		}
    view_type=VIEW_CAMERA;
	c=(Camera *)&FC;			/* pointer to the variable declared as a Camera */
	c->x=fi->camera.x;			/* recording of camers data from FrameInfo struct */
	c->y=fi->camera.y;
	c->z=fi->camera.z;
	c->tx=fi->camera.tx;
	c->ty=fi->camera.ty;
	c->tz=fi->camera.tz;
	c->bank=fi->camera.bank;
	c->focal=fi->camera.focal;
	dev_width=fi->dev_width;
	dev_height=fi->dev_height;
	dev_aspect=fi->dev_aspect;
     ov_zoom=fi->ov_zoom;
	process_image();				/* image processing function called once camera 							data is recorded from the passed parameters */

	return(1);				/* after processing image, a 'terminate without error' 							command is sent. */
	}

void ClientSetStateVar(int id, void *ptr) {
 	OVL o;				/*Union of possible values that the dialog can return */
	ulong *ul;
	char *s;

	ul=(ulong *)ptr;
	s=(char *)ptr;
	o.ul = *ul;
	switch(id) {
		case VDIM: state.dim=o.i; break;			/* State variables set here based on
								values stored in the fields of the 
								Union type .*/
		case VBRIGHT: state.bright=o.i; break;
		case VSCALE: state.scale=o.i; break;
		case VFILE: strcpy(state.file,s); break;
		}
	}

/* Following function sets the appropriate value of the Union 'OVL' variable
   to return values stored in the 'State' variable */

ulong ClientGetStateVar(int id) {
	OVL o;
	switch(id) {
		case VDIM: o.i=state.dim; break;			/* Union variables set to 
								appropriate value from 'state'
								variables */
		case VBRIGHT: o.i=state.bright; break;
		case VSCALE: o.i=state.scale; break;
		case VFILE: o.s=state.file; break;
		}
	return(o.ul);					/* the variable requested by 3D Studio is 
							returned to 3D Studio via the 'ulong' field
							of the 'Union' structure 'OVL'. */
	}

/* Following function allows 3D Studio to determine the size of
   any variable in the dialog */

int ClientVarSize(int id) {
	switch(id) {
		case VFILE: return(((sizeof(state.file)+3)/4)*4);	/* Returns size of file string*/
		default:
			return(1);
		}
	}

/* Following function allows 3D Studio to determine the size of
   the 'State' structure and returns a pointer to the 'State' struct */

char  *ClientGetState(int *size) {
	*size = sizeof(State);
	return((char *)&state);
	}

/* Following function allows 3D Studio to reset the state of the
   external process */

void ClientResetState() {	
	state = init_state;	
	}

/* Following function performs any calculations or packet-commands
   necessary before the EXP can begin. Note: IXPs do not have ability to send packet commands as the EXPbuf is not passed in to the 'ClientStartup' procedure, and there is no 'ClientUserCode' function. */

void ClientStartup(void) {
	/* allocate data structures, compute lookup tables, etc */
	}

void ClientTerminate(void) { 
	/* free any data structures, etc. */
	free_alloced();
	}

/* The following function allows 3D Studio to get the address of a string containing a variable. The IPAS
programmer will not have to call this function or modify the following code, but will have to include it */

DlgEntry *ClientDialog(int n) {	
	return(&cdialog[n]); 
	}

/* The following functions were written by the IPAS programmer for this specific routine, and therefore, are not examples of required 'Client' functions */

void free_alloced()
{
if(stardata)
 {
 free(stardata);
 stardata=NULL;
 }
}

/* The following functions are the image processing machine. They are unique to the 'Stars.IXP' program*/

void process_image()
{
   float  mm,mag,hangle,vangle,wy,factor1,factor2,fsx,fsy;
   int line,ix,hh,deg,min,items,pixel,pixels,sx,sy,vis;
   char inf[10],ext[5];
    Camera *c;
   FILE *istream;

   /* Copy variables from Cdata */

   dim=state.dim;
   bright=state.bright;
   logarithmic=state.scale;
   state.file[12]=0;
   strcpy(infile,state.file);

   /* Get stars */

   split_fn(NULL,inf,infile);
   split_fext(inf,NULL,ext);

   if((istream=fopen(infile,"rb"))==NULL)
   {
      error("Can't open star file:",infile,NULL,NULL,NULL);
      return;
   }

   if(fread(&starcount,sizeof(int),1,istream)!=1)
   {
      bad_infile:
      error("Error in data file",NULL,NULL,NULL,NULL);
      fclose(istream);

      cleanup:
      if(stardata)
      {
      free(stardata);
      stardata=NULL;
      }
      return;
   }

   if(starcount==0)
      goto bad_infile;

   /* Allocate room for stars */

   if((stardata=malloc(starcount*sizeof(Star)))==NULL)
   {
      fclose(istream);
      error("No RAM for star database",NULL,NULL,NULL,NULL);
      return;
   }

   if(fread(stardata,sizeof(Star),starcount,istream)!=starcount)
      goto bad_infile;
   fclose(istream);
 
   /* Calculate plotted brightnesses from magnitude */

   factor2=((float)bright-(float)dim)/10.0;
   for(ix=0; ix<starcount; ++ix)
   {
      mag=stardata[ix].fmag;
      if(logarithmic)
         stardata[ix].mag=(int)(pow(10.0-mag,2.407));
      else
         stardata[ix].mag=(int)((10.0-mag)*factor2)+dim;
   }

   /* Now set up transformation matrix */

   build_xmatrix();

   /* Initialize the output image area:
      Copy the required number of pixels from the input map to the output
      map. */

   pixels=dev_width*dev_height;
   far_to_far(fp->outpix,fp->inpix,pixels*3);          /* routine to copy the 24-bit pixel data */

   /* Now transform and plot each star if it's visible */

   vis=0;
   for(ix=0; ix<starcount; ++ix)
   {
      xform_point(stardata[ix].x,stardata[ix].y,stardata[ix].z,
      &stardata[ix].x,&stardata[ix].y,&stardata[ix].z);
      if(stardata[ix].z>100.0)
      {
         fsx=dev_width/2+stardata[ix].x/stardata[ix].z*camzfac*camw2;
         fsy=dev_height/2+stardata[ix].y/stardata[ix].z*camzfac*camh2;
         sx=(int)fsx;
         sy=(int)fsy;
         if(sx>=0 && sx<dev_width && sy>=0 && sy<dev_height)
         {
            frect(fsx,fsy,1.5,1.5,stardata[ix].mag);
            vis++;
         }
      }
   }

   goto cleanup;
}

/* Find angle for given vector, and its 90-degree offset	*/
/* Returns angles in radians				*/
/* Returns 0 if angle indeterminate				*/

int find_angle(float deltax,float deltay,float *angle,float *angle90)
{
   float awk,a90wk;

   if(deltax==0.0)
   {
      if(deltay==0.0)
      {
        *angle= *angle90=0.0;
        return(0);
      }

      if(deltay<0.0)
         awk=PI_270;
      else
         awk=PI_90;
      goto get_90;
   }

   if(deltay==0.0)
   {
      if(deltax<0.0)
         awk=PI_180;
      else
         awk=0.0;
      goto get_90;
   }

   awk=atan(deltay/deltax);
   if(deltax<0)
      awk+=PI_180;
   while(awk<0.0)
      awk+=PI_360;
   while(awk>=PI_360)
      awk-=PI_360;

   get_90:
   a90wk=awk+PI_90;
   while(a90wk>=PI_360)
      a90wk-=PI_360;

   *angle=awk;
   *angle90=a90wk;

return(1);
}

/* The following procedures are used to carry out the matrix math to determine the position of the stars */

void Translate(TMatrix tmat, FLOAT tx, FLOAT ty, FLOAT tz)
{
   FLOAT *t = tmat[3];
   t[0] += tx;
   t[1] += ty;
   t[2] += tz;
}

void RelScaleRow(TRow r, FLOAT s)
{
   r[0] *= s;  r[1] *= s;  r[2] *= s;
}

void RotateX(TMatrix tmat, FLOAT alpha)
{
   FLOAT tmp,s,c;
   int i;
   FLOAT *t;
   s = sin(alpha);
   c = cos(alpha);
   for (i=0; i<4; i++) {
      t = tmat[i];
      tmp = t[1]*c - t[2]*s;
      t[2] = t[1]*s + t[2]*c;
      t[1] = tmp;
   }
}

void RotateY(TMatrix tmat, FLOAT alpha)
{
   FLOAT tmp,s,c;
   int i;
   FLOAT *t;
   s = sin(alpha);
   c = cos(alpha);
   for (i=0; i<4; i++) {
      t = tmat[i];
      tmp = t[0]*c - t[2]*s;
      t[2] = t[0]*s + t[2]*c;
      t[0] = tmp;
   }
}

/*------ RotateZ: */

void RotateZ(TMatrix tmat, FLOAT alpha)
{
   FLOAT tmp,s,c;
   int i;
   FLOAT *t;
   s = sin(alpha);
   c = cos(alpha);
   for (i=0; i<4; i++) {
      t = tmat[i];
      tmp = t[0]*c - t[1]*s;
      t[1] = t[0]*s + t[1]*c;
      t[0] = tmp;
   }
}

void TransformPoint(TMatrix tm, FLOAT *p, FLOAT *pt)
{
   FLOAT tmp[3];
   tmp[0] = p[0]*tm[0][0] + p[1]*tm[1][0] + p[2]*tm[2][0] + tm[3][0];
   tmp[1] = p[0]*tm[0][1] + p[1]*tm[1][1] + p[2]*tm[2][1] + tm[3][1];
   tmp[2] = p[0]*tm[0][2] + p[1]*tm[1][2] + p[2]*tm[2][2] + tm[3][2];
   memcpy(pt,tmp,3*sizeof(FLOAT));
}

/* Transform a 3D point     */

void xform_point(float x,float y,float z,float *rx,float *ry,float *rz)
{
   float inp[3],outp[3];
 
   inp[0]=x;
   inp[1]=z;
   inp[2]=y;
   TransformPoint(xmat,inp,outp);
   *rx=outp[0];
   *ry=outp[1];
   *rz=outp[2];
}

/* Build the transformation matrix for the current view */

int build_xmatrix(void)
{
   float fdum,fov;
   float vx,vy,vz,newx;
   Camera *c;
 
   /* If camera, set up view accordingly */

   if(view_type==VIEW_CAMERA)
   {
       proj_type=PROJ_PERSPECTIVE;
      c=(Camera *)&FC;
 

      vpt_x=c->tx;
      vpt_y=c->ty;
      vpt_z=c->tz;
 
      fov=(15.0/c->focal)*160.0;
      camzfac=tan(((180.0-fov)/2.0)/RAD_DEG);
      camw2=dev_width/2;
      camh2=camw2* -dev_aspect;
 
      vx=c->tx-c->x;
      vy=c->ty-c->y;
      vz=c->tz-c->z;
  camdist=sqrt(vx*vx+vy*vy+vz*vz);
 
      find_angle(vx,vy,&view_hangle,&fdum);
      newx=cos(PI_360-view_hangle)*vx-sin(PI_360-view_hangle)*vy;
      find_angle(newx,vz,&view_vangle,&fdum);
      view_hangle=view_hangle*RAD_DEG-90.0;
      view_vangle= (-view_vangle)*RAD_DEG;
      view_tilt=c->bank;
   }
   else
       proj_type=PROJ_ORTHOGRAPHIC;
 
   /* Convert degree inputs to radians */
 
   heading= -view_hangle/RAD_DEG;
   pitch= -view_vangle/RAD_DEG;
   bank=view_tilt/RAD_DEG;
    scale=(float)dev_width/view_width*ov_zoom;
    yscale= -scale*dev_aspect;
 
   /* Build the matrix */
 
   memcpy(xmat,ident_mat,sizeof(float)*12);
   /*Translate(xmat,-vpt_x,-vpt_z,-vpt_y);*/
   RelScaleZ(xmat,-1.0);
   RotateY(xmat,heading);
   RotateX(xmat,pitch);
   RotateZ(xmat,bank);
   RelScaleZ(xmat,-1.0);
   /*
   if(proj_type==PROJ_PERSPECTIVE)
   Translate(xmat,0.0,0.0,camdist);
   */
 
   return(1);
}


/* Replace one character with another in non-quoted areas of a string	*/
/* If oldch==0, zap everything except what's inside the quotes		*/

void ch_replace(char *string,char oldch,char newch)
{
   int inquote,op,np;
   char *shold;

   shold=string;
   inquote=0;

   loop:
   if(*string=='\"')
   {
      inquote=1-inquote;
   }
   if(inquote==0)
   {
      if(oldch==0 && *string!=0)
         *string='\"';
      else
         if(*string==oldch)
            *string=newch;
   }
   if(*string)
   {
      ++string;
      goto loop;
   }

   /* Clean up if necessary */

   if(oldch==0)
   {
      string=shold;
      op=np=0;

      loop2:
      if(shold[op]==0)
      {
         string[np]=0;
         return;
      }
      if(shold[op]!='\"')
         string[np++]=shold[op];
      op++;
      goto loop2;
   }
}

/* Split path & file into separate strings	*/
/* Ignores NULL pointers			*/

void split_fn(char *path,char *file,char *pf)
{
int ix,jx,bs_loc,fn_loc;

if(strlen(pf)==0)
 {
 if(path)
  *path=0;
 if(file)
  *file=0;
 return;
 }
bs_loc=strlen(pf);
for(ix=bs_loc-1; ix>=0; --ix)
 {
 if(pf[ix]=='\\')
  {
  bs_loc=ix;
  fn_loc=ix+1;
  goto do_split;
  }
 if(pf[ix]==':')
  {
  bs_loc=ix+1;
  fn_loc=ix+1;
  goto do_split;
  }
 }
bs_loc= -1;
fn_loc=0;

do_split:
if(file)
 strcpy(file,&pf[fn_loc]);
if(path)
 {
 if(bs_loc>0)
  {
  for(jx=0; jx<bs_loc; ++jx)
   path[jx]=pf[jx];
  path[jx]=0;
  }
 else
  path[0]=0;
 }
}

/* Split filename.ext into separate filename & ext strings	*/
/* Returns 8-char fname, 4-char ext (.EXT)			*/
/* Ignores null pointers					*/

void split_fext(char *fe,char *f,char *e)
{
int ix;
char ext[13],wkname[13];

strcpy(wkname,fe);
ix=0;
while(wkname[ix]!='.' && wkname[ix]!=0)
 ++ix;
if(wkname[ix]=='.')
 {
 strcpy(ext,&wkname[ix]);
 ext[4]=0;
 }
else
 ext[0]=0;
wkname[ix]=wkname[8]=0;
if(f)
 strcpy(f,wkname);
if(e)
 strcpy(e,ext);
}


/* Case-insensitive string compare */

int striicmp(char *s1,char *s2)
{
for( ; toupper(*s1) == toupper(*s2); s1++,s2++)
 {
 if(*s1==0)
  return(0);
 }
return(toupper(*s1)-toupper(*s2));
}

void plot(int x, int y, uchar v )
{
int pixel,r,g,b;

if(x<0 || y<0 || x>=dev_width || y>=dev_height)
 return;
pixel=dev_width*y+x;
r=fp->outpix[pixel].r+v;				/* Set individual red values for output pixels */
g=fp->outpix[pixel].g+v;				/* Set individual green values for output pixels */
b=fp->outpix[pixel].b+v;				/* Set individual blue values for output pixels */
fp->outpix[pixel].r=(r>255) ? 255:r;
fp->outpix[pixel].g=(g>255) ? 255:g;
fp->outpix[pixel].b=(b>255) ? 255:b;
}

/*------------------------------------------------------------
Dan's Handy-Dandy Floating-Point Rectancle Plotter (DHDFPRP) (TM)...
	
	Render an anti-aliased floating point rectangle on pixel grid.
	x,y,w,h are floating point screen coords. 

	This could be sped up using fixed point.

-------------------------------------------------------------*/

#define frac(z) fmod(z,1.0)

void frect(float x, float y, float w, float h,int brite)
{
	int xa,xb,ya,yb,i,j;
	float x2,y2,lfrac,rfrac,tfrac,bfrac,xfrac,yfrac;
	x2 = x + w;
	y2 = y + w;
  	xa = (int)x;
	xb = (int)x2;
  	ya = (int)y;
	yb = (int)y2;
	lfrac = 1.0-frac(x);	
	rfrac = frac(x2);
	tfrac = 1.0-frac(y);	
	bfrac = frac(y2);
	if (xa==xb) {
		xfrac = lfrac+rfrac-1.0;
		if (ya==yb) { 
			/* --- only one pixel effected */
            plot(xa,ya,xfrac*(tfrac+bfrac-1.0)*brite);
			}
		else {  
			/* --- column of pixels */
			plot(xa,ya,xfrac*tfrac*brite);
			for (j=ya+1; j<yb; j++) plot(xa,j,xfrac*brite);
			plot(xa,yb,xfrac*bfrac*brite);
			}
		}
	else {
		if (ya==yb) { 
			/* --- row of pixels */
			yfrac = tfrac+bfrac-1.0;
			plot(xa,ya,yfrac*lfrac*brite);
			for (i=xa+1; i<xb; i++) plot(i,ya,yfrac*brite);
			plot(xb,ya,yfrac*rfrac*brite);
			}
		else {  
			/* general case */

			/* --- top row */
			plot(xa,ya,tfrac*lfrac*brite);
			for (i=xa+1; i<xb; i++) plot(i,ya,tfrac*brite);
			plot(xb,ya,tfrac*rfrac*brite);

			/* --- middle (whole pixels ) */
			for (j=ya+1; j<yb; j++) {
				plot(xa,j,lfrac*brite);
				for (i=xa+1; i<xb; i++) plot(i,j,brite);
				plot(xb,j,rfrac*brite);
				}

			/* --- bottom row */
			plot(xa,yb,bfrac*lfrac*brite);
			for (i=xa+1; i<xb; i++) plot(i,yb,bfrac*brite);
			plot(xb,yb,bfrac*rfrac*brite);
			}
		}
	}

/* The following function returns the name of the external process to 3D Studio upon request. */
char *ClientName(void)
{
return("STARS.IXP");
}
