/******************************************************************************
      (C) Copyright 1992 by Autodesk, Inc.

      This program is copyrighted by Autodesk, Inc. and is  licensed
      to you under the following conditions.  You may not distribute
      or  publish the source code of this program in any form.   You
      may  incorporate this code in object form in derivative  works
      provided  such  derivative  works  are  (i.) are  designed and 
      intended  to  work  solely  with  Autodesk, Inc. products, and 
      (ii.)  contain  Autodesk's  copyright  notice  "(C)  Copyright  
      1992 by Autodesk, Inc."

      AUTODESK  PROVIDES THIS PROGRAM "AS IS" AND WITH  ALL  FAULTS.
      AUTODESK  SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF  MER-
      CHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK,  INC.
      DOES  NOT  WARRANT THAT THE OPERATION OF THE PROGRAM  WILL  BE
      UNINTERRUPTED OR ERROR FREE.

*******************************************************************************/

/* AXP process raw client routine core code */

/* For the Metaware High C and High C/C++ compilers, turn off an
   unwanted warning message */
#ifdef __HIGHC__
pragma Offwarn(67); /* kills "switch statement has no cases" warning */
#endif

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include "axp.h"
#include "trig.h"

#define Rand01() ((float)(rand() & 0x7fff) / 32767.0)

/* Variable definitions */

#define PARTICLES 1
#define SIZE 2
#define SEED 3
#define INNER 4
#define OUTER 5
#define CHAOS 6
#define ROTATION 7

/* The above definitions are the LineID numbers for the dialog lines in which the variables will be set. The integers assigned to the dialog lines containing variables are selected by the IPAS programmer and used by 3D Studio to identify the variables. */

/************************/

/* Dialog description */

DlgEntry cdialog[]={
	0,"TITLE=\"Autodesk 3D Studio Tornado Generator\"",	/* Text to appear in dialog box */
	0,"TITLE=\"by Tom Hudson, Version 1.0\"",		/* Text to appear in dialog box */
    PARTICLES,"LINT=\"# of Particles:\",1,9999,4",      /* Setup for 'limited integer' type
								variariable and associated range
								values */
	SIZE,"LFLOAT=\"Object/Particle size ratio:\",0.001,0.5,6",	/*  Setup for 'limited float' type 
								variable and associated range 
								values */
	SEED,"INT=\"Random # Seed:\",5",			/* Simple integer variable */
    INNER,"LINT=\"Inner Orbit Period (Frames):\",1,99999,5",     /* Limited integer variable */
    OUTER,"LINT=\"Outer Orbit Period (Frames):\",1,99999,5",     /* Limited integer variable */
	0,"TITLE= \"          Low                High\"",			/* Text to appear in dialog box */
	CHAOS,"SLIDER=\"Chaos:\",0,100,24",			/* Setup for slider interface */
	ROTATION,"RADIO=\"Rotation:\",\"CW\",\"CCW\"",	/*Setup for radio-button interface*/
	0,"TITLE=\"\"",						/* Blank line for spacing */
	0,NULL
	};

/* Version test value */
/* Every external process must have a unique version number  */

#define VERSION 0x4981

/* State Structure definition */
/* Any variables whose values are set via dialog interface must be declared within the 'State' struct. */	
	
typedef struct {
	ulong version;			/* required to be first field within 'State' struct */
	int particles;
	float size;
	int seed,inner,outer,chaos,rotation;
	} State;

/* the "state" struct MUST start with a "ulong" which is the version#,
	to prevent using data from old versions of this program.
	This verification is performed automatically. */

static State init_state = { VERSION,100,0.02,12345,5,15,0,0 };	/* Initial state settings */
static State state = { VERSION,100,0.02,12345,5,15,0,0 };		/* Default state settings */

/* Variables */

static int frame,particle,overts;
static float sizefac,partx,party,partz,px /**** ,py */,pz,radius,orbit_time,position;
static float sinfac,cosfac,xfac,yfac,zfac,fchaos,forder,taper,ybasis/****, ylimit*/;
static int /**** dialog_ix, verts, faces, */first,ix /*** ,onedges,offedges,edges,totlights,edge*/;
static int vertex,face,base;
static float minx,miny,minz,maxx,maxy,maxz,dx,dy,dz;
static unsigned char material;

/*----------------------------------------------------------------*/
/* Following function sets the appropriate 'State' variables based
   on the input from the dialog */

void ClientSetStateVar(int id, void *ptr) {
	OVL o;				/* Union of possible values that the dialog can return */
	ulong *ul;

	ul=(ulong *)ptr;
	o.ul = *ul;
	switch(id) {
		case PARTICLES: state.particles=o.i; break;	/* State variables set here based on
								values stored in the fields of the 
								Union type .*/
		case SIZE: state.size=o.f; break;
		case SEED: state.seed=o.i; break;
		case INNER: state.inner=o.i; break;
		case OUTER: state.outer=o.i; break;
		case CHAOS: state.chaos=o.i; break;
		case ROTATION: state.rotation=o.i; break;
		}
	}

/* Following function sets the appropriate value of the Union 'OVL' variable
   to return values stored in the 'State' variable */

ulong ClientGetStateVar(int id) {
	OVL o;
	switch(id) {
		case PARTICLES: o.i=state.particles; break;	/* Union variables set to 
								appropriate value from 'state'
								variables */
		case SIZE: o.f=state.size; break;
		case SEED: o.i=state.seed; break;
		case INNER: o.i=state.inner; break;
		case OUTER: o.i=state.outer; break;
		case CHAOS: o.i=state.chaos; break;
		case ROTATION: o.i=state.rotation; break;
		}
	return(o.ul);					/* the variable requested by 3D Studio is 
							returned to 3D Studio via the 'ulong' field
							of the 'Union' structure 'OVL'. */
	}

/* Following function allows 3D Studio to determine the size of
   any variable in the dialog */

int ClientVarSize(int id)
	{
	switch(id)
		{
		default:
			return(1);
		}
	}

/* Following function allows 3D Studio to determine the size of
   the 'State' structure and returns a pointer to the 'State' struct */

char  *ClientGetState(int *size) {
	*size = sizeof(State);
	return((char *)&state);
	}

/* Following function allows 3D Studio to reset the state of the
   external process */

void ClientResetState() {	
	state = init_state;	
	}

/* Following function performs any calculations or packet-commands
   necessary before the EXP can begin */

void ClientStartup(EXPbuf *buf) {
#define ob buf->data.object
	overts=ob.verts;			/*Get existing object's vertex count for determining extents */
	frame=ob.curfield;		/* Get current field of animation */
#undef ob
	sizefac=state.size/2.0;

	/* Randomize for this object */
	srand(state.seed);

	first=1;
	ix=0;

					/* Start to find object's extents */

#define cv buf->data.vert
	cv.vertex=ix;
	buf->opcode=EXP_GET_VERTEX;			/* 3D Studio packet command for 
								getting vertex information from 
								input object */
	buf->usercode=0x0200;					/* Next entry point for the
                                          					ClientUserCode function */
	buf->status=1;
#undef cv
	}

/* Following function contains user codes which process data */

void ClientUserCode(EXPbuf *buf)
	{
	switch(buf->usercode)
		{
		case 0x0200:	/* Extents */
#define cv buf->data.vert
			if(first)
				{
				minx=maxx=cv.x;	/* Since AXPs do not have access to the 
							EXP_OBJ_EXTENTS command, the object
							extents must be found manually */
				miny=maxy=cv.y;
				minz=maxz=cv.z;
				first=0;
				}
			else
				{
				if(cv.x<minx)
					minx=cv.x;
				else
				if(cv.x>maxx)
					maxx=cv.x;
				if(cv.y<miny)
					miny=cv.y;
				else
				if(cv.y>maxy)
					maxy=cv.y;
				if(cv.z<minz)
					minz=cv.z;
				else
				if(cv.z>maxz)
					maxz=cv.z;
				}
			ix++;
			if(ix<overts)
				{
				cv.vertex=ix;
				buf->opcode=EXP_GET_VERTEX;/* 3D Studio packet command for 
								getting vertex information from 
								input object */	
				buf->usercode=0x0200;		/* Next entry point for the
                                          					ClientUserCode function */	
				buf->status=1;
				break;
				}
#undef cv
			if(buf->status==0)	/* If previous operation failed, terminate AXP */
				{
				terminate:
				buf->opcode=buf->usercode=EXP_TERMINATE;	/*Exit */
				break;
				}
			dx=maxx-minx;
			dy=maxy-miny;
			dz=maxz-minz;

			/* Find object material */

			buf->opcode=EXP_GET_FACE;		/* 3D Studio packet command for
								sequentially getting 'face' data */
			buf->usercode=0x0210;			/* Next entry point for the
                                          					ClientUserCode function */
#define cf buf->data.face
			cf.face=0;				/* Set initial face number */
			break;
		case 0x0210:
			material=cf.material;			/* Retain given material */
#undef cf
			/* Create the necessary number of verts & faces */

			buf->opcode=EXP_READY_OBJ;		/* 3D Studio packet command for
								opening a new object for output */
			buf->usercode=0x0240;			/* Next entry point for the
                                          					ClientUserCode function */
			buf->status=1;
#define ob buf->data.object
			ob.verts=6*state.particles;			/* Set total number of vertices */
			ob.faces=8*state.particles;			/* Set total number of faces */
			ob.tverts=0;				/* No textured vertices */
#undef ob
			break;
		case 0x0240:
			vertex=face=0;
			if(buf->status==0)
				goto terminate;		/* If previous operation failed, terminate */
			particle=0;
			fchaos=(float)state.chaos/100.0;
			forder=1.0-fchaos;
			part_loop:
			if(particle>=state.particles)
				goto pvert_done;

			ybasis=Rand01();
			party=1.0-ybasis;

			recalc:
			px=Rand01()-0.5;
			pz=Rand01()-0.5;
			radius=sqrt(px*px+pz*pz);
			if(radius>0.5)
				goto recalc;

			/* Scale down, making a cone, depending on chaos setting */

			taper=fchaos+(forder*party);
			if(Rand01()>=fchaos)
				{
				do_taper:
				px*=taper;
				pz*=taper;
				}
			else
				{
				taper+=(fchaos*2);
				if(taper>1.0)
					taper=1.0;
				goto do_taper;
				}

			orbit_time=(float)state.inner+(float)(state.outer-state.inner)*(radius/0.5);
			position=(float)frame/orbit_time*PI_360;
			if(state.rotation)
				position= -position;
			sinfac=sin(position);
			cosfac=cos(position);

			partx=(px*cosfac+pz*sinfac+0.5)*dx+minx;
			party=maxy-party*dy;
			partz=(pz*cosfac-px*sinfac+0.5)*dz+minz;
			xfac=sizefac*dx;
			yfac=sizefac*dy;
			zfac=sizefac*dz;

			/* Now create the appropriate vertices for this particle */

#define cv buf->data.vert
			buf->opcode=EXP_PUT_VERTEX;	/* 3D Studio packet command for 
								writing vertex information to 
								output object */
			buf->usercode=0x0260;			/* Next entry point for the
                                          					ClientUserCode function */
			/* Set actual vertex coordinates */
			cv.x=partx-xfac;				/* Set 'x' coordinates */
			cv.y=party;				/* Set 'y' coordinates */
			cv.z=partz;				/* Set 'z' coordinates */
			cv.vertex=vertex++;
			break;

		case 0x0260:
			buf->opcode=EXP_PUT_VERTEX;	/* 3D Studio packet command for 
								writing vertex information to 
								output object */
			buf->usercode=0x0261;			/* Next entry point for the
                                          					ClientUserCode function */
			/* Set actual vertex coordinates */
			cv.x=partx;						
			cv.y=party-yfac;				
			cv.z=partz;				
			cv.vertex=vertex++;
			break;

		case 0x0261:
			buf->opcode=EXP_PUT_VERTEX;
			buf->usercode=0x0262;
			/* Set actual vertex coordinates */
			cv.x=partx+xfac;
			cv.y=party;
			cv.z=partz;
			cv.vertex=vertex++;
			break;

		case 0x0262:
			buf->opcode=EXP_PUT_VERTEX;
			buf->usercode=0x0263;
			/* Set actual vertex coordinates */
			cv.x=partx;
			cv.y=party+yfac;
			cv.z=partz;
			cv.vertex=vertex++;
			break;

		case 0x0263:
			buf->opcode=EXP_PUT_VERTEX;
			buf->usercode=0x0264;
			/* Set actual vertex coordinates */
			cv.x=partx;
			cv.y=party;
			cv.z=partz-zfac;
			cv.vertex=vertex++;
			break;

		case 0x0264:
			buf->opcode=EXP_PUT_VERTEX;
			buf->usercode=0x0265;
			/* Set actual vertex coordinates */
			cv.x=partx;
			cv.y=party;
			cv.z=partz+zfac;
			cv.vertex=vertex++;
			break;

		case 0x0265:
			particle++;
			goto part_loop;
#undef cv
			/* Now add the appropriate faces */

			pvert_done:
			ix=face=0;

			face_loop:
			if(ix>=state.particles)
			{
			buf->opcode=EXP_CREATE_OBJ;	/* 3D Studio packet command to 
								signal completion of output object.
								Object can now be displayed on 
								screen and treated as any other 
								object created in 3D Studio */
               		buf->usercode=EXP_TERMINATE;	/* Command to terminate 
								routine */
			buf->status=1;
			break;
				}
			base=ix*6;

#define cf buf->data.face
			buf->opcode=EXP_PUT_FACE;		/* 3D Studio packet command to
								put 'face' data to output object */
			buf->usercode=0x0305;			/* Next entry point for the
                                          					ClientUserCode function */
			/* Set face number to correspond to vertices for edge calculation */
			cf.a=base;				
			cf.b=base+1;
			cf.c=base+5;
			cf.flags=FC_ABLINE | FC_BCLINE | FC_CALINE;	/* Visibility attributes */
			cf.sm_group=0;
			cf.material=material;
			cf.face=face++;
            break;

		case 0x0305:
			buf->opcode=EXP_PUT_FACE;
			buf->usercode=0x0310;
			/* Set face number to correspond to vertices for edge calculation */
			cf.a=base+1;
			cf.b=base+2;
			cf.c=base+5;
			cf.flags=FC_ABLINE | FC_BCLINE | FC_CALINE;	/* Visibility attributes */
			cf.sm_group=0;
			cf.material=material;
			cf.face=face++;
			break;

		case 0x0310:
			buf->opcode=EXP_PUT_FACE;
			buf->usercode=0x0315;
			/* Set face number to correspond to vertices for edge calculation */
			cf.a=base+2;
			cf.b=base+3;
			cf.c=base+5;
			cf.flags=FC_ABLINE | FC_BCLINE | FC_CALINE;	/* Visibility attributes */
			cf.sm_group=0;
			cf.material=material;
			cf.face=face++;
			break;

		case 0x0315:
			buf->opcode=EXP_PUT_FACE;
			buf->usercode=0x0320;
			/* Set face number to correspond to vertices for edge calculation */
			cf.a=base+3;
			cf.b=base;
			cf.c=base+5;
			cf.flags=FC_ABLINE | FC_BCLINE | FC_CALINE;	/* Visibility attributes */
			cf.sm_group=0;
			cf.material=material;
			cf.face=face++;
			break;

		case 0x0320:
			buf->opcode=EXP_PUT_FACE;
			buf->usercode=0x0325;
			/* Set face number to correspond to vertices for edge calculation */
			cf.a=base;
			cf.b=base+4;
			cf.c=base+1;
			cf.flags=FC_ABLINE | FC_BCLINE | FC_CALINE;	/* Visibility attributes */
			cf.sm_group=0;
			cf.material=material;
			cf.face=face++;
			break;

		case 0x0325:
			buf->opcode=EXP_PUT_FACE;
			buf->usercode=0x0330;
			/* Set face number to correspond to vertices for edge calculation */
			cf.a=base+1;
			cf.b=base+4;
			cf.c=base+2;
			cf.flags=FC_ABLINE | FC_BCLINE | FC_CALINE;	/* Visibility attributes */
			cf.sm_group=0;
			cf.material=material;
			cf.face=face++;
			break;

		case 0x0330:
			buf->opcode=EXP_PUT_FACE;
			buf->usercode=0x0335;
			/* Set face number to correspond to vertices for edge calculation */
			cf.a=base+2;
			cf.b=base+4;
			cf.c=base+3;
			cf.flags=FC_ABLINE | FC_BCLINE | FC_CALINE;	/* Visibility attributes */
			cf.sm_group=0;
			cf.material=material;
			cf.face=face++;
			break;

		case 0x0335:
			buf->opcode=EXP_PUT_FACE;
			buf->usercode=0x0340;
			/* Set face number to correspond to vertices for edge calculation */
			cf.a=base+3;
			cf.b=base+4;
			cf.c=base;
			cf.flags=FC_ABLINE | FC_BCLINE | FC_CALINE;	/* Visibility attributes */
			cf.sm_group=0;
			cf.material=material;
			cf.face=face++;
			break;

		case 0x0340:
			ix++;
			goto face_loop;

		default:
		error("Invalid user opcode",NULL,NULL,NULL,NULL);
		buf->opcode=buf->usercode=EXP_TERMINATE;
		buf->status=0;
		}
	}

void ClientTerminate(void) { 
	/* free any data structures, etc. */
	}

/* The following function allows 3D Studio to get the address of a string containing a variable. The IPAS
programmer will not have to call this function or modify the following code, but will have to include it */

DlgEntry *ClientDialog(int n) {	
	return(&cdialog[n]); 
	}

/* The following function returns the name of the external process to 3D Studio upon request. */
char *ClientName(void)
	{
	return(".AXP");		/* Put your process name here */
	}
