/*  @(#)usercode.h	5.2 as of 6/10/93 09:50:09  */

/*
 * usercode.h - Macros and prototypes for C language User Code programs.
 *     Also used for compiling the User Code interface routines themselves.
 *
 * Modification history:
 * 18 Feb 93 - Original version for release 5.0 of SPSS
 * 14 Mar 93 - Modified for release 6.0 of SPSS for Windows
 */

#if !defined(USERCODE_H)
#define USERCODE_H 1

#include <stddef.h>             /* To get the definition of size_t */


/* Basic SPSS Data Types */

typedef char*   CHARACTER;      /* Assumes no dope vectors */
typedef long    INTEGER;        /* At least 32 bits */
typedef long    SI;             /* 16 bits in some implementations of SPSS */
typedef long    LOGICAL;        /* Conventions vary - use macros below */
typedef float   REAL;           /* May be the same as LR - generally not */
typedef double  LR;             /* At least 48 bits - normally 64 */
typedef double  OBS;            /* Guaranteed to be the same as LR */


/* Special support for LOGICAL */
#define ISTRUE(LogVar)          /* Test LogVar for value .TRUE. */ \
           ((int)(0xff&(LogVar)))
#define ISFALSE(LogVar)         /* Test LogVar for value .FALSE. */ \
           (!ISTRUE(LogVar))
#define LOG_SET(LogVar,Expr)    /* Set LogVar to value of logical expr */ \
           ((LogVar) = (Expr) ? 1L : 0L)
#define LOG_SET_TRUE(LogVar)    /* Set LogVar to .TRUE. */ \
           ((LogVar) = 1L)
#define LOG_SET_FALSE(LogVar)   /* Set LogVar to .FALSE. */ \
        ((LogVar) = 0L)


/* APIENTRY - attributes related to API calling conventions */
#define APIENTRY _far _fortran


/* IC(n) - number of INTEGER words to contain n characters */
#define IC(n)   (((n)+sizeof(INTEGER)-1)/sizeof(INTEGER))


/* OC(n) - number of OBS words to contain n characters */
#define OC(n)   (((n)+sizeof(OBS)-1)/sizeof(OBS))


/* SUBROUTINE - function attributes comparable to Fortran subroutines */
#define SUBROUTINE  void APIENTRY


/* FUNCTION - function attributes comparable to Fortran functions */
#define FUNCTION(Type) Type APIENTRY


/* SPSS format codes */

#define FMCA         1          /* a */
#define FMCAHEX      2          /* ahex */
#define FMCCOMMA     3          /* comma   */
#define FMCDOLLR     4          /* dollar  */
#define FMCF         5          /* f */
#define FMCIB        6          /* ib */
#define FMCPIBHX     7          /* pibhex */
#define FMCP         8          /* p */
#define FMCPIB       9          /* pib */
#define FMCPK       10          /* pk */
#define FMCRB       11          /* rb */
#define FMCRBHEX    12          /* rbhex */
#define FMCZ        15          /* z */
#define FMCN        16          /* n */
#define FMCE        17          /* e */
#define FMCDATE     20          /* date - international date (dd-mmm-yy) */
#define FMCTIME     21          /* time */
#define FMCDATTM    22          /* datetime */
#define FMCADATE    23          /* adate - American date (mm/dd/yy) */
#define FMCJDATE    24          /* jdate - yyddd date */
#define FMCDTIME    25          /* dtime (days and time) */
#define FMCWKDAY    26          /* wkday - day of the week - 1 = Sunday */
#define FMCMONTH    27          /* month */
#define FMCMOYR     28          /* moyr - mmm yy date */
#define FMCQYR      29          /* qyr - q Q yy date */
#define FMCWKYR     30          /* wkyr - ww WK yy date */
#define FMCPCT      31          /* percent */
#define FMCDOT      32          /* dot */
#define FMCCCA      33          /* cca - output only */
#define FMCCCB      34          /* ccb - output only */
#define FMCCCC      35          /* ccc - output only */
#define FMCCCD      36          /* ccd - output only */
#define FMCCCE      37          /* cce - output only */
#define FMCEDATE    38          /* edate - European date (dd.mm.yy) */
#define FMCSDATE    39          /* sdate - Sortable date (yy/mm/dd) */


/* Display type and severity definitions for uerror */

#define ERDISKP      0          /* Display neither error number nor msg */
#define ERDINO       1          /* Display error number and message */
#define ERDICC       2          /* Display error #, command #, and msg */
#define ERDISYM      3          /* Display error #, cmd #, token, and msg */
#define ERDINOID     4          /* Display message without error number */

#define ERCOMM       1          /* Comment */
#define ERWARN       2          /* Warning */
#define ERSERS       3          /* Serious error - terminates procedure */
#define ERFATL       4          /* Fatal error - no further commands executed */
#define ERCATA       5          /* Catastophic error  - terminates session */


/*
 * Macros defining all the Fortran callable functions in the API.
 * Each User Code function is defined with the first letter in upper
 * case.  The macros convert C function definitions and references
 * to whatever is compatible with Fortran calling conventions on the
 * host system.  This enables us to change the capitalization of the
 * function name, add prefix and suffix characters, and reorder the
 * parameters without changing the definition of the API.
 */

#define Uclock(date,time) \
    UCLOCK(date,time)
#define Uclowr(string,istr,length) \
    UCLOWR(string,istr,length)
#define Ucmove(src,isrc,trgt,itrgt,length) \
    UCMOVE(src,isrc,trgt,itrgt,length)
#define Ucnstr(trgt,itrgt,ival) \
    UCNSTR(trgt,itrgt,ival)
#define Ucuppr(string,istr,length) \
    UCUPPR(string,istr,length)
#define Udicgt(ivar) \
    UDICGT(ivar)
#define Udicnm(nvar,indexs,names) \
    UDICNM(nvar,indexs,names)
#define Udicsc(ivar,string) \
    UDICSC(ivar,string)
#define Uelaps(icpu,ireal) \
    UELAPS(icpu,ireal)
#define Uerror(errnum,errprt,errlev) \
    UERROR(errnum,errprt,errlev)
#define Ufmts(dipfmt,fmtcod,fmtwid,fmtdig) \
    UFMTS(dipfmt,fmtcod,fmtwid,fmtdig)
#define Uheadr \
    UHEADR
#define Uincv(intern,fmtcod,fmtwid,fmtdic,extrep,nerr) \
    UINCV(intern,fmtcod,fmtwid,fmtdic,extrep,nerr)
#define Ulinen(k) \
    ULINEN(k)
#define Umscop(obsvec,qumiss,nmiss,obsnew,nvar,indexs) \
    UMSCOP(obsvec,qumiss,nmiss,obsnew,nvar,indexs)
#define Unewfm(ivar,ipnm,ipwid,ipd,iwnm,iwwid,iwd,qerror) \
    UNEWFM(ivar,ipnm,ipwid,ipd,iwnm,iwwid,iwd,qerror)
#define Unewlb(ivar,string,nchar) \
    UNEWLB(ivar,string,nchar)
#define Unewvr(name,type,ivar,qerr) \
    UNEWVR(name,type,ivar,qerr)
#define Uobadd(obsvec,length) \
    UOBADD(obsvec,length)
#define Uoutcv(intern,fmtcod,fmtwid,fmtdic,extrep,nerr) \
    UOUTCV(intern,fmtcod,fmtwid,fmtdic,extrep,nerr)
#define Upinit \
    UPINIT
#define Uprint(record,length,spaces) \
    UPRINT(record,length,spaces)
#define Uqlmis(qreslt,obsvec,qumiss,nvar,indexs) \
    UQLMIS(qreslt,obsvec,qumiss,nvar,indexs)
#define Urdcmd(comand,cmdlth,qeof) \
    URDCMD(comand,cmdlth,qeof)
#define Uread(obsvec) \
    UREAD(obsvec)
#define Usgchi(prob,chisq,df) \
    USGCHI(prob,chisq,df)
#define Usgfsh(n1,n2,n3,n4,p1tail,p2tail) \
    USGFSH(n1,n2,n3,n4,p1tail,p2tail)
#define Usgnrm(prob,z) \
    USGNRM(prob,z)
#define Usigf(prob,fvalue,df1,df2) \
    USIGF(prob,fvalue,df1,df2)
#define Usigt(prob,t,df) \
    USIGT(prob,t,df)
#define Usnext \
    USNEXT
#define Usrget(wrkspc,ndbles) \
    USRGET(wrkspc,ndbles)
#define Usrpgm(obsvec,wrkspc,ndbles) \
    USRPGM(obsvec,wrkspc,ndbles)
#define Usrset \
    USRSET
#define Usvals(values,maxval,nval,qerr) \
    USVALS(values,maxval,nval,qerr)
#define Usvars(itype,maxvar,indexs,nvar,qerr) \
    USVARS(itype,maxvar,indexs,nvar,qerr)
#define Uvallb(nval,values,nchar,labels) \
    UVALLB(nval,values,nchar,labels)
#define Uvarck(len,string,result) \
    UVARCK(len,string,result)
#define Uvarlb(nchar,string) \
    UVARLB(nchar,string)
#define Uvladd(nlab,lablen,vallab,nvars,varids,qerror) \
    UVLADD(nlab,lablen,vallab,nvars,varids,qerror)


/* Function prototypes per the API with the first letter in upper case */

SUBROUTINE Uclock(              /* Get date and time in character form */
    char* date,                 /* Returned as 9 character (dd-mmm-yy) date */
    char* time);                /* Returned as 8 character (hh:mm:ss) time */

SUBROUTINE Uclowr(              /* Lower case a substring */
    char* string,               /* Containing string */
    INTEGER* istr,              /* Unit origin index of substring */
    INTEGER* length);           /* Length of substring */

SUBROUTINE Ucmove(              /* Copy characters */
    char* src,                  /* Source string */
    INTEGER* isrc,              /* Unit origin index of first source char */
    char* trgt,                 /* Destination string */
    INTEGER* itrgt,             /* Unit origin index of first dest char */
    INTEGER* length);           /* Number of characters to copy */

SUBROUTINE Ucnstr(              /* Store character into a string */
    char* trgt,                 /* Destination string */
    INTEGER* itrgt,             /* Unit origin index of char in dest string */
    INTEGER* ival);             /* Integer value of char to store */

SUBROUTINE Ucuppr(              /* Upper case a substring */
    char* string,               /* Containing string */
    INTEGER* istr,              /* Unit origin index of substring */
    INTEGER* length);           /* Length of substring */

SUBROUTINE Udicgt(              /* Retrieve variable information by index */
    INTEGER* ivar);             /* Index of variable to retrieve */

SUBROUTINE Udicnm(              /* Get names of multiple variables */
    INTEGER* nvar,              /* Number of variables to retrieve */
    SI indexs[],                /* Indexes of variables to retrieve */
    char names[][8]);           /* Returned as names of variables */

SUBROUTINE Udicsc(              /* Retrieve variable information by name */
    INTEGER* ivar,              /* Returned as index of variable if found */
    char* string);              /* 8 character name for which to search */

SUBROUTINE Uelaps(              /* Get CPU and elapsed time in milliseconds */
    INTEGER* icpu,              /* Returned as CPU time in milliseconds */
    INTEGER* ireal);            /* Returned as elapsed time in milliseconds */

SUBROUTINE Uerror(              /* Issue an error */
    INTEGER* errnum,            /* SPSS error number */
    INTEGER* errprt,            /* Type of display - 0 thru 4 */
    INTEGER* errlev);           /* Severity - 1 thru 5 */

SUBROUTINE Ufmts(               /* Decode a packed SPSS format */
    INTEGER* dipfmt,            /* Packed SPSS print or write format */
    INTEGER* fmtcod,            /* Returned as format code, e.g. 5 for "F" */
    INTEGER* fmtwid,            /* Returned as field width */
    INTEGER* fmtdig);           /* Returned as fractional digits */

SUBROUTINE Uheadr(void);        /* Skip to top of page */

SUBROUTINE Uincv(               /* Decode a string into an OBS datum */
    OBS intern[],               /* Returned as decoded datum */
    INTEGER* fmtcod,            /* Format code, e.g. 5 for "F" */
    INTEGER* fmtwid,            /* Field width */
    INTEGER* fmtdic,            /* Number of fractional digits */
    char* extrep,               /* Field to decode */
    INTEGER* nerr);             /* Returned non-zero if an error encountered */

SUBROUTINE Ulinen(              /* Report number of lines to be printed */
    INTEGER* k);                /* Number of lines to be printed */

SUBROUTINE Umscop(              /* Filter data vector for missing values */
    OBS obsvec[],               /* Original data values */
    LOGICAL* qumiss,            /* Set to ignore user missing values */
    INTEGER* nmiss,             /* Returned as number of missing values found */
    OBS obsnew[],               /* Returned as selected data values */
    INTEGER* nvar,              /* Number of variables to select */
    SI indexs[]);               /* Indexes of variables to select */

SUBROUTINE Unewfm(              /* Update print and write formats */
    INTEGER* ivar,              /* Index of variable to which formats apply */
    INTEGER* ipnm,              /* Print format code, e.g. 5 for "F" */
    INTEGER* ipwid,             /* Print format field width */
    INTEGER* ipd,               /* Print format fractional digits */
    INTEGER* iwnm,              /* Write format code */
    INTEGER* iwwid,             /* Write format field width */
    INTEGER* iwd,               /* Write format fractional digits */
    LOGICAL* qerror);           /* Returned set if an error encountered */

SUBROUTINE Unewlb(              /* Update variable label */
    INTEGER* ivar,              /* Index of variable to be labelled */
    char* string,               /* New variable label */
    INTEGER* nchar);            /* Number of characters in label */

SUBROUTINE Unewvr(              /* Define new variable */
    char* name,                 /* Name of the new variable */
    INTEGER* type,              /* SPSS variable type - 0 for numeric */
    INTEGER* ivar,              /* Returned as new variable's index */
    LOGICAL* qerr);             /* Returned set if an error encountered */

SUBROUTINE Uobadd(              /* Add a case of data (user get only) */
    OBS obsvec[],               /* Vector of values of variables */
    INTEGER* length);           /* Number of values */

SUBROUTINE Uoutcv(              /* Encode an OBS datum into a string */
    OBS intern[],               /* Datum to be encoded */
    INTEGER* fmtcod,            /* Format code, e.g. 5 for "F" */
    INTEGER* fmtwid,            /* Field width */
    INTEGER* fmtdic,            /* Number of fractional digits */
    char* extrep,               /* Returned as encoded datum */
    INTEGER* nerr);             /* Returned non-zero if an error encountered */

SUBROUTINE Upinit(void);        /* Data pass initialization (user proc only) */

SUBROUTINE Uprint(              /* Print a line */
    char* record,               /* String to be printed */
    INTEGER* length,            /* Characters to print */
    INTEGER* spaces);           /* Lines to skip before printing */

SUBROUTINE Uqlmis(              /* Check data vector for missing values */
    LOGICAL* qreslt,            /* Returned set if any variable missing */
    OBS obsvec[],               /* Original data values */
    LOGICAL* qumiss,            /* Set to ignore user missing values */
    INTEGER* nvar,              /* Number of variables to inspect */
    SI indexs[]);               /* Indexes of variables to inspect */

SUBROUTINE Urdcmd(              /* Read one line of the user get/proc command */
    char* comand,               /* Returned as command line */
    INTEGER* cmdlth,            /* Capacity of comand in characters */
    LOGICAL* qeof);             /* Set at end of command */

SUBROUTINE Uread(               /* Read one case (user proc only) */
    OBS obsvec[]);              /* Same as obsvec parameter to usrpgm */

SUBROUTINE Usgchi(              /* Get probability of chi-square value */
    REAL* prob,                 /* Returned as probability of chi-square */
    LR* chisq,                  /* Value of chi-square */
    INTEGER* df);               /* Degrees of freedom */

SUBROUTINE Usgfsh(              /* Get probability of 2 x 2 table */
    INTEGER* n1,                /* Number of cases in row 1, column 1 */
    INTEGER* n2,                /* Number of cases in row 1, column 2 */
    INTEGER* n3,                /* Number of cases in row 2, column 1 */
    INTEGER* n4,                /* Number of cases in row 2, column 2 */
    LR* p1tail,                 /* Returned as one-tailed probability */
    LR* p2tail);                /* Returned as two-tailed probability */

SUBROUTINE Usgnrm(              /* Get probability of Z */
    REAL* prob,                 /* Returned as probability of Z */
    LR* z);                     /* Value of Z */

SUBROUTINE Usigf(               /* Get probability of F */
    REAL* prob,                 /* Returned as probability of F */
    LR* fvalue,                 /* Value of F */
    INTEGER* df1,               /* Numerator of degrees of freedom */
    LR* df2);                   /* Denominator of degrees of freedom */

SUBROUTINE Usigt(               /* Get probability of t */
    REAL* prob,                 /* Returned as probability of t */
    LR* t,                      /* Value of t */
    LR* df);                    /* Degrees of freedom */

SUBROUTINE Usnext(void);        /* Get next token from user get/proc command */

SUBROUTINE Usrget(              /* User-written code for user get */
    OBS wrkspc[],               /* Workspace */
    INTEGER* ndbles);           /* Size of workspace */

SUBROUTINE Usrpgm(              /* User-written code for user proc */
    OBS obsvec[],               /* Observation vector */
    OBS wrkspc[],               /* Workspace */
    INTEGER* ndbles);           /* Size of workspace */

SUBROUTINE Usrset(void);        /* User-written code for initialization */

SUBROUTINE Usvals(              /* Process command tokens as values */
    OBS values[],               /* Returned as list of values */
    INTEGER* maxval,            /* Maximum values to accept */
    INTEGER* nval,              /* Returned as number of values found */
    INTEGER* qerr);             /* Returned set if an error encountered */

SUBROUTINE Usvars(              /* Process command tokens as variable names */
    INTEGER* itype,             /* Type of acceptable variables */
    INTEGER* maxvar,            /* Maximum variables to accept */
    SI indexs[],                /* Returned as indexes of variables found */
    INTEGER* nvar,              /* Returned as number of variables found */
    LOGICAL* qerr);             /* Returned set if an error encountered */

SUBROUTINE Uvallb(              /* Retrieve value labels */
    INTEGER* nval,              /* Number of values */
    OBS values[],               /* Values for which labels requested */
    INTEGER* nchar,             /* Label length - multiple of sizeof(OBS) */
    void* labels);              /* Returned as blank-padded labels */

SUBROUTINE Uvarck(              /* Check for valid variable name */
    INTEGER* len,               /* Length of candidate variable name */
    char* string,               /* Candidate variable name */
    INTEGER* result);           /* Returned set to 0 if valid name */

SUBROUTINE Uvarlb(              /* Retrieve variable label */
    INTEGER* nchar,             /* Maximum characters to retrieve */
    char* string);              /* Returned as variable label */

SUBROUTINE Uvladd(              /* Define value labels */
    INTEGER* nlab,              /* Number of value labels */
    INTEGER* lablen,            /* OBS words for each value/label pair */
    void* vallab,               /* Value/Label pairs */
    INTEGER* nvars,             /* Number of variables to be labelled */
    SI varids[],                /* Indexes of variables to be labelled */
    LOGICAL* qerror);           /* Returned set if an error encountered */

void _huge* _far Uhalloc(       /* Allocates a huge memory block */
    long num,                   /* Number of elements */
    size_t size);               /* Length in bytes of each element */

void _far* _far Umalloc(        /* Allocate memory blocks */
    size_t size);               /* Types to allocate */

void _far* _far Urealloc(       /* Reallocate memory blocks */
    void _far* memblock,        /* Pointer to previously allocated block */
    size_t size);               /* New size in bytes */

void _far Ufree(                /* Deallocate a memory block */
    void _far* memblock);       /* Allocated memory block*/


#endif  /* !defined(USERCODE_H) */
