/*
** Math & Fixed Point Functions by Neil C. Obremski           [ Magenta's Maze ]
** 2017 Gibdon Moon Productions                   [ http://magsmaze.gibdon.com ]
**
** .14 FIXED POINT leaves 2 bits in an int and 18 bits in a long for whole.
** Divides shift up by 12 leaving 6 bits in a long (-64 + 64).
**
** This is free and unencumbered software released into the public domain.
**
** Anyone is free to copy, modify, publish, use, compile, sell, or
** distribute this software, either in source code form or as a compiled
** binary, for any purpose, commercial or non-commercial, and by any
** means.
**
** In jurisdictions that recognize copyright laws, the author or authors
** of this software dedicate any and all copyright interest in the
** software to the public domain. We make this dedication for the benefit
** of the public at large and to the detriment of our heirs and
** successors. We intend this dedication to be an overt act of
** relinquishment in perpetuity of all present and future rights to this
** software under copyright law.
**
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
** EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
** MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
** IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR
** OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
** ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
** OTHER DEALINGS IN THE SOFTWARE.
**
** Tested w/ DOSBox 0.74, Microsoft C 5.10
 */
#ifndef __MAG_MATH
#define __MAG_MATH

#include <math.h>
#include <stdarg.h>
#include <string.h>

/* 8-bit char for booleans */
#define TRUE (char)1
#define FALSE (char)0

/* http://stackoverflow.com/questions/5007925/using-m-pi-with-c89-standard */
#ifndef M_PI
#    define M_PI 3.14159265358979323846
#endif

/* 18.14 PI = 51,471 */
#define FX_PI 0xC90FL

/* some 32-bit fixed point numbers */
#define FX_0_5 0x2000L
#define FX_1_0 0x4000L
#define FX_1_5 0x6000L
#define FX_2_0 0x8000L
#define FX_2_5 0xA000L
#define FX_3_0 0xC000L

/* convert DEGREES to RADIANS */
#define DEG2RAD(degrees) ( M_PI*(double)(degrees)/180.0 )

/* ASCII character for degrees */
#define DEGCHAR ((char)248)

/* divide an INTEGER by a FIXED POINT FRACTION */
#define DIV_FRAC(i,fraction) ((long)(i) << 14 / (fraction))

/* divide two FIXED POINT numbers (long is recommended type) */
#define DIV_FX(dividend, divisor) ( (((dividend) << 12) / (divisor)) << 2 )

/* convert mixed number or FRACTION to FIXED POINT */
#define FRAC_FX(n,d) DIV_FX((long)INT_FX(n),(long)INT_FX(d))

/* absolute INT */
#define INT_ABS(i) (i < 0 ? -i : i)

/* is integer even? */
#define INT_EVEN(i) (!(i&1))

/* which integer is bigger? */
#define INT_MAX(a,b) ((a) > (b) ? (a) : (b))

/* which integer is smaller */
#define INT_MIN(a,b) ((a) < (b) ? (a) : (b))

/* is integer odd? */
#define INT_ODD(i) (i&1)

/* convert INTEGER to FIXED POINT (long is recommended type) */
#define INT_FX(i) ((long)(i)<<14)

/* multiply two FIXED POINT numbers (long is recommended type) */
#define MUL_FX(m1, m2) ( ((long)(m1) * (long)(m2)) >> 14 )

/* multiply an INTEGER by a FIXED POINT FRACTION */
#define MUL_FRAC(i,fraction) ((long)(i) * (long)(fraction) >> 14)

/* multiply an INTEGER by SIN(degrees) */
#define MUL_SIN_DEG(i,d) (int)MUL_FRAC(i,SIN_FX[d])

/* multiply an INTEGER by COS(degrees) */
#define MUL_COS_DEG(i,d) (int)MUL_FRAC(i,COS_FX[d])

/* round FIXED POINT number to nearest integer 8192 == 0.5 or 1/2 */
#define ROUND_FX(i) (((i)+8192)>>14)

/* pre-calculated SINE for each of 360 degrees in 2.14 fixed point */
int SIN_FX[] = {
	0, 286, 572, 858, 1143, 1428, 1713, 1997,
	2281, 2564, 2846, 3127, 3407, 3686, 3964, 4241,
	4517, 4791, 5063, 5335, 5604, 5872, 6138, 6402,
	6664, 6925, 7183, 7439, 7692, 7944, 8192, 8439,
	8683, 8924, 9162, 9398, 9631, 9861, 10087, 10311,
	10532, 10749, 10964, 11174, 11382, 11586, 11786, 11983,
	12176, 12366, 12551, 12733, 12911, 13085, 13255, 13421,
	13583, 13741, 13895, 14044, 14189, 14330, 14467, 14599,
	14726, 14849, 14968, 15082, 15191, 15296, 15396, 15492,
	15583, 15669, 15750, 15826, 15898, 15965, 16026, 16083,
	16136, 16183, 16225, 16262, 16295, 16322, 16345, 16362,
	16375, 16382, 16384, 16382, 16375, 16362, 16345, 16322,
	16295, 16262, 16225, 16183, 16136, 16083, 16026, 15965,
	15898, 15826, 15750, 15669, 15583, 15492, 15396, 15296,
	15191, 15082, 14968, 14849, 14726, 14599, 14467, 14330,
	14189, 14044, 13895, 13741, 13583, 13421, 13255, 13085,
	12911, 12733, 12551, 12366, 12176, 11983, 11786, 11586,
	11382, 11174, 10964, 10749, 10532, 10311, 10087, 9861,
	9631, 9398, 9162, 8924, 8683, 8439, 8192, 7944,
	7692, 7439, 7183, 6925, 6664, 6402, 6138, 5872,
	5604, 5335, 5063, 4791, 4517, 4241, 3964, 3686,
	3407, 3127, 2846, 2564, 2281, 1997, 1713, 1428,
	1143, 858, 572, 286, 1, -285, -571, -857,
	-1142, -1427, -1712, -1996, -2280, -2563, -2845, -3126,
	-3406, -3685, -3963, -4240, -4516, -4790, -5062, -5334,
	-5603, -5871, -6137, -6401, -6663, -6924, -7182, -7438,
	-7691, -7943, -8191, -8438, -8682, -8923, -9161, -9397,
	-9630, -9860, -10086, -10310, -10531, -10748, -10963, -11173,
	-11381, -11585, -11785, -11982, -12175, -12365, -12550, -12732,
	-12910, -13084, -13254, -13420, -13582, -13740, -13894, -14043,
	-14188, -14329, -14466, -14598, -14725, -14848, -14967, -15081,
	-15190, -15295, -15395, -15491, -15582, -15668, -15749, -15825,
	-15897, -15964, -16025, -16082, -16135, -16182, -16224, -16261,
	-16294, -16321, -16344, -16361, -16374, -16381, -16384, -16381,
	-16374, -16361, -16344, -16321, -16294, -16261, -16224, -16182,
	-16135, -16082, -16025, -15964, -15897, -15825, -15749, -15668,
	-15582, -15491, -15395, -15295, -15190, -15081, -14967, -14848,
	-14725, -14598, -14466, -14329, -14188, -14043, -13894, -13740,
	-13582, -13420, -13254, -13084, -12910, -12732, -12550, -12365,
	-12175, -11982, -11785, -11585, -11381, -11173, -10963, -10748,
	-10531, -10310, -10086, -9860, -9630, -9397, -9161, -8923,
	-8682, -8438, -8192, -7943, -7691, -7438, -7182, -6924,
	-6663, -6401, -6137, -5871, -5603, -5334, -5062, -4790,
	-4516, -4240, -3963, -3685, -3406, -3126, -2845, -2563,
	-2280, -1996, -1712, -1427, -1142, -857, -571, -285
};

/* pre-calculated COSINE for each of 360 degrees in 2.14 fixed point */
int COS_FX[] = {
	16384, 16382, 16375, 16362, 16345, 16322, 16295, 16262,
	16225, 16183, 16136, 16083, 16026, 15965, 15898, 15826,
	15750, 15669, 15583, 15492, 15396, 15296, 15191, 15082,
	14968, 14849, 14726, 14599, 14467, 14330, 14189, 14044,
	13895, 13741, 13583, 13421, 13255, 13085, 12911, 12733,
	12551, 12366, 12176, 11983, 11786, 11586, 11382, 11174,
	10964, 10749, 10532, 10311, 10087, 9861, 9631, 9398,
	9162, 8924, 8683, 8439, 8193, 7944, 7692, 7439,
	7183, 6925, 6664, 6402, 6138, 5872, 5604, 5335,
	5063, 4791, 4517, 4241, 3964, 3686, 3407, 3127,
	2846, 2564, 2281, 1997, 1713, 1428, 1143, 858,
	572, 286, 1, -285, -571, -857, -1142, -1427,
	-1712, -1996, -2280, -2563, -2845, -3126, -3406, -3685,
	-3963, -4240, -4516, -4790, -5062, -5334, -5603, -5871,
	-6137, -6401, -6663, -6924, -7182, -7438, -7691, -7943,
	-8191, -8438, -8682, -8923, -9161, -9397, -9630, -9860,
	-10086, -10310, -10531, -10748, -10963, -11173, -11381, -11585,
	-11785, -11982, -12175, -12365, -12550, -12732, -12910, -13084,
	-13254, -13420, -13582, -13740, -13894, -14043, -14188, -14329,
	-14466, -14598, -14725, -14848, -14967, -15081, -15190, -15295,
	-15395, -15491, -15582, -15668, -15749, -15825, -15897, -15964,
	-16025, -16082, -16135, -16182, -16224, -16261, -16294, -16321,
	-16344, -16361, -16374, -16381, -16384, -16381, -16374, -16361,
	-16344, -16321, -16294, -16261, -16224, -16182, -16135, -16082,
	-16025, -15964, -15897, -15825, -15749, -15668, -15582, -15491,
	-15395, -15295, -15190, -15081, -14967, -14848, -14725, -14598,
	-14466, -14329, -14188, -14043, -13894, -13740, -13582, -13420,
	-13254, -13084, -12910, -12732, -12550, -12365, -12175, -11982,
	-11785, -11585, -11381, -11173, -10963, -10748, -10531, -10310,
	-10086, -9860, -9630, -9397, -9161, -8923, -8682, -8438,
	-8192, -7943, -7691, -7438, -7182, -6924, -6663, -6401,
	-6137, -5871, -5603, -5334, -5062, -4790, -4516, -4240,
	-3963, -3685, -3406, -3126, -2845, -2563, -2280, -1996,
	-1712, -1427, -1142, -857, -571, -285, 0, 286,
	572, 858, 1143, 1428, 1713, 1997, 2281, 2564,
	2846, 3127, 3407, 3686, 3964, 4241, 4517, 4791,
	5063, 5335, 5604, 5872, 6138, 6402, 6664, 6925,
	7183, 7439, 7692, 7944, 8193, 8439, 8683, 8924,
	9162, 9398, 9631, 9861, 10087, 10311, 10532, 10749,
	10964, 11174, 11382, 11586, 11786, 11983, 12176, 12366,
	12551, 12733, 12911, 13085, 13255, 13421, 13583, 13741,
	13895, 14044, 14189, 14330, 14467, 14599, 14726, 14849,
	14968, 15082, 15191, 15296, 15396, 15492, 15583, 15669,
	15750, 15826, 15898, 15965, 16026, 16083, 16136, 16183,
	16225, 16262, 16295, 16322, 16345, 16362, 16375, 16382
};

#ifndef MAX_TERM
#define MAX_TERM 10
#endif

struct term {
	int coefficient;
	char exponent;
	char constant;		/* Not coefficient? TRUE or FALSE */
};

char addroot(struct term[], int);
char addterm(struct term[], int, char, char);
char clipline(int*, int*, int*, int*, int, int, int, int);
char coinflip(void);
char *polynomial(struct term[]);
struct term *roots(int, ...);

/* multiplies terms by (X - R) where R is the root */
char addroot(struct term terms[], int root)
{
	int i = 0;
	struct term scratch[MAX_TERM];

	if (!terms[i].coefficient) { /* first root becomes first term */
		addterm(terms, 1, 1, FALSE);
		addterm(terms, -root, 0, TRUE);
		return TRUE;
	}

	scratch[0].coefficient = 0;

	for ( ; i < MAX_TERM && terms[i].coefficient; i++) {

		/* multiply each term by X */
		if (!addterm(scratch, terms[i].coefficient,
			terms[i].exponent + 1, FALSE)) {
			return FALSE;
		}

		/* multiply each term by constant */
		if (!addterm(scratch, terms[i].coefficient * -root,
			terms[i].exponent, terms[i].constant)) {
			return FALSE;
		}
	}

	memcpy(terms, scratch, sizeof(scratch));

	return TRUE;
}

/* puts a term in the list or combines it if it is like another */
char addterm(struct term terms[], int coefficient, char exponent, char constant)
{
	int i = 0;

	/* look for like term */
	for ( ; i < MAX_TERM && terms[i].coefficient; i++) {
		if (terms[i].exponent == exponent && terms[i].constant == constant) {
			/* add coefficient when exponent matches or term is constant */
			terms[i].coefficient += coefficient;

			if (!terms[i].coefficient) { /* remove cancelled out term */
				for (i++; terms[i].coefficient; i++) {
					terms[i-1].coefficient = terms[i].coefficient;
					terms[i-1].exponent = terms[i].exponent;
					terms[i-1].constant = terms[i].constant;
				}
				terms[i-1].coefficient = 0;
			}
			return TRUE;
		}
	}

	/* append term to list */
	if (i >= MAX_TERM) {
		return FALSE;
	}

	terms[i].coefficient = coefficient;
	terms[i].exponent = exponent;
	terms[i].constant = constant;

	if (1 + i < MAX_TERM) {
		terms[1+i].coefficient = 0;
	}

	return TRUE;
}

/* Cohen-Sutherland algorithm for line clipping */
char clipline(int *px1, int *py1, int *px2, int *py2,
	int xmin, int ymin, int xmax, int ymax)
{
	char accept = 0, oc1 = 0, oc2 = 0;
	int x, y, x1 = *px1, y1 = *py1, x2 = *px2, y2 = *py2;

	if (x1 < xmin) { oc1 |= 1; } else if (x1 > xmax) { oc1 |= 2; }
	if (y1 < ymin) { oc1 |= 4; } else if (y1 > ymax) { oc1 |= 8; }
	if (x2 < xmin) { oc2 |= 1; } else if (x2 > xmax) { oc2 |= 2; }
	if (y2 < ymin) { oc2 |= 4; } else if (y2 > ymax) { oc2 |= 8; }

	do {
		if (!(oc1 | oc2)) {
			return TRUE;						/* full accept */
		} else if (oc1 & oc2) {
			return FALSE;						/* full reject */
		} else {
			char ocOut = oc1 ? oc1 : oc2;

			if (xmax > 180 || ymax > 180) {		/* use 32-bit for big roots */
				if (ocOut & 8) {				/* TOP (long) */
					y = ymax; x = x1 + (int)(
						(long)(x2 - x1) * (long)(ymax - y1) / (long)(y2 - y1));
				} else if (ocOut & 4) {			/* BOTTOM (long) */
					y = ymin; x = x1 + (int)(
						(long)(x2 - x1) * (long)(ymin - y1) / (long)(y2 - y1));
				} else if (ocOut & 2) {			/* RIGHT (long) */
					x = xmax; y = y1 + (int)(
						(long)(y2 - y1) * (long)(xmax - x1) / (long)(x2 - x1));
				} else if (ocOut & 1) {			/* LEFT (long) */
					x = xmin; y = y1 + (int)(
						(long)(y2 - y1) * (long)(xmin - x1) / (long)(x2 - x1));
				}
			} else {							/* use 16-bit for small roots */
				if (ocOut & 8) {				/* TOP (short) */
					y = ymax; x = x1 + (x2 - x1) * (ymax - y1) / (y2 - y1);
				} else if (ocOut & 4) {			/* BOTTOM (short) */
					y = ymin; x = x1 + (x2 - x1) * (ymin - y1) / (y2 - y1);
				} else if (ocOut & 2) {			/* RIGHT (short) */
					x = xmax; y = y1 + (y2 - y1) * (xmax - x1) / (x2 - x1);
				} else if (ocOut & 1) {			/* LEFT (short) */
					x = xmin; y = y1 + (y2 - y1) * (xmin - x1) / (x2 - x1);
				}
			}

			if (ocOut == oc1) {
				*px1 = x1 = x; *py1 = y1 = y;	oc1 = 0;
				if (x1 < xmin) { oc1 |= 1; } else if (x1 > xmax) { oc1 |= 2; }
				if (y1 < ymin) { oc1 |= 4; } else if (y1 > ymax) { oc1 |= 8; }
			} else {
				*px2 = x2 = x; *py2 = y2 = y;	oc2 = 0;
				if (x2 < xmin) { oc2 |= 1; } else if (x2 > xmax) { oc2 |= 2; }
				if (y2 < ymin) { oc2 |= 4; } else if (y2 > ymax) { oc2 |= 8; }
			}
		}

	} while (++accept < 5); /* prevent endless loop */

	return accept;
}

/* returns TRUE or FALSE randomly */
char coinflip(void)
{
	static int bitsleft = 0;
	static int data = 0;

	if (!bitsleft) {
		data = rand();
		bitsleft = 15;
		return (char)(data & 1);
	}

	bitsleft--;
	data >>= 1;

	return (char)(data & 1);
}

/* outputs a polynomial expression using a static char buffer */
char *polynomial(struct term terms[])
{
	static char buf[MAX_TERM * 10];
	char *p = buf;
	int i = 0;

	for ( ; i < MAX_TERM && terms[i].coefficient; i++) {

		unsigned coefficient;

		/* sign */
		if (p == buf) {
			if (terms[i].coefficient < 0) {
				*p++ = '-';
				coefficient = (unsigned)(-terms[i].coefficient);
			} else {
				coefficient = (unsigned)(terms[i].coefficient);
			}
		} else {
			*p++ = ' ';
			if (terms[i].coefficient < 0) {
				*p++ = '-';
				coefficient = (unsigned)(-terms[i].coefficient);
			} else {
				*p++ = '+';
				coefficient = (unsigned)(terms[i].coefficient);
			}
			*p++ = ' ';
		}

		/* coefficient */
		if (terms[i].constant || 1 != coefficient) {
			sprintf(p, "%u", coefficient);
			while (*p++);
			p--;
		}

		/* variable */
		if (!terms[i].constant) {
			*p++ = 'X';

			/* exponent */
			switch (terms[i].exponent) {
				case 1: /* hide ^1 */
					break;
				case 2:
					*p++ = (char)253;
					break;
				case 3:
					*p++ = (char)252;
					break;
				default:
					sprintf(p, "^%d", terms[i].exponent);
					while (*p++);
					p--;
					break;
			}
		}
	}

	*p = '\0';

	return buf;
}

/* returns terms calculated from a list of roots */
struct term *roots(int numroot, ...)
{
	static struct term terms[MAX_TERM];
	va_list valist;

	memset(terms, 0, sizeof(terms));

	va_start(valist, numroot);
	for ( ; numroot; numroot--) {
		addroot(terms, va_arg(valist, int));
	}
	va_end(valist);

	return terms;
}

/* rotate around an axis by 0-359 degrees */
void rotatefx(int *px, int *py, int deg)
{
	int newx, newy;

	while (deg >= 360) {
		deg -= 360;
	}

	while (deg < 0) {
		deg += 360;
	}

	newx = MUL_SIN_DEG(*py,deg) + MUL_COS_DEG(*px,deg);
	newy = MUL_COS_DEG(*py,deg) - MUL_SIN_DEG(*px,deg);

	*px = newx;
	*py = newy;
}

#endif
