/*
 * Zmiy is a snake-like game for DOS and 8086.
 * Copyright (C) 2013-2015 Mateusz Viste
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 */

#include <dos.h>
#include <stdio.h>
#include <stdlib.h> /* rand() */
#include <string.h>

#include "io.h"
#include "levels.h"
#include "timer.h"


#define PVER "0.85.2"
#define PDATE "2013-2015"


/* Color schemes       BG  FG */
/*                       ||   */
#define COLOR_BAR      0x7000u
#define COLOR_SNAKE1   0xE000u
#define COLOR_WALL     0x6000u
#define COLOR_ITEM     0x1A00u
#define COLOR_MSG_GOOD 0x2F00u
#define COLOR_MSG_BAD  0x4F00u
#define COLOR_MSG_INFO 0x8F00u

#define MONO_BAR      0x7000u
#define MONO_SNAKE1   0x7000u
#define MONO_WALL     0x7000u
#define MONO_ITEM     0x0700u
#define MONO_MSG_GOOD 0x7000u
#define MONO_MSG_BAD  0x7000u
#define MONO_MSG_INFO 0x7000u

#define MSG_GOODJOB  0
#define MSG_TRYAGAIN 1
#define MSG_GAMEOVER 2
#define MSG_PAUSE    3


/* character-plotting code is hidden inside this #define instead of a static
 * function for speed gain (+10% faster, on Turbo C v2.01 at least) */
unsigned short far *vmem = (unsigned short far *)0xB8000000l;
#define printchar(column, row, c, attr) vmem[((row) << 6) + ((row) << 4) + (column)] = (attr) | c


static void PrintScoreBar(struct gamestruct *game) {
  int x = 0;
  char tmpstr[16];
  char *tmpstrptr;
  /* draw the 'SCORE' string */
  for (tmpstrptr = " SCORE: "; *tmpstrptr != 0; tmpstrptr += 1) {
    printchar(x++, 0, *tmpstrptr, game->col_bar);
  }
  /* draw the score itself */
  itoa(game->snakescore[0], tmpstr, 10);
  for (tmpstrptr = tmpstr; *tmpstrptr != 0; tmpstrptr += 1) {
    printchar(x++, 0, *tmpstrptr, game->col_bar);
  }
  if (game->snakescore[0] > 0) printchar(x++, 0, '0', game->col_bar); /* add a '0' to the displayed score */
  /* Draw some background */
  for (; x < 35; x++) {
    printchar(x, 0, ' ', game->col_bar);
  }
  /* draw the 'LEVEL' string */
  for (tmpstrptr = "LEVEL: "; *tmpstrptr != 0; tmpstrptr += 1) {
    printchar(x++, 0, *tmpstrptr, game->col_bar);
  }
  /* draw the level itself */
  itoa(game->level, tmpstr, 10);
  for (tmpstrptr = tmpstr; *tmpstrptr != 0; tmpstrptr += 1) {
    printchar(x++, 0, *tmpstrptr, game->col_bar);
  }
  /* Draw some background */
  for (; x < 71; x++) {
    printchar(x, 0, ' ', game->col_bar);
  }
  /* draw the 'SPEED' string */
  for (tmpstrptr = "SPEED: "; *tmpstrptr != 0; tmpstrptr += 1) {
    printchar(x++, 0, *tmpstrptr, game->col_bar);
  }
  /* draw the speed itself */
  itoa(game->snakespeed[0], tmpstr, 10);
  for (tmpstrptr = tmpstr; *tmpstrptr != 0; tmpstrptr += 1) {
    printchar(x++, 0, *tmpstrptr, game->col_bar);
  }
  /* draw the rest of the background */
  for (; x < 80; x++) {
    printchar(x, 0, ' ', game->col_bar);
  }
}


/* flush the content of the keyboard buffer */
static void keybflush(void) {
  while (getkey_ifany() != -1); /* flush the keyboard buffer */
}


/* Output a message on screen */
static void OutMessage(int msgid, struct gamestruct *game) {
  int x, linenum;
  int msglen;
  int msghoffset, msgvoffset;
  unsigned short msgcolor;
  char *line[5];
  /* define the 'good job' texts */
  char *line1_goodjob[4];
  char *line2_goodjob[4];
  char *line3_goodjob[4];
  char *line4_goodjob[4];
  char *line5_goodjob[4];
  /* define the 'game over' text */
  char *line1_gameover = "  XXX  XX   X X  XXX    XX  X  X XXX XXX   X ";
  char *line2_gameover = " X    X  X X X X X     X  X X  X X   X  X  X ";
  char *line3_gameover = " X XX XXXX X X X XXX   X  X X  X XXX XXX   X ";
  char *line4_gameover = " X  X X  X X X X X     X  X X X  X   X  X    ";
  char *line5_gameover = "  XXX X  X X X X XXX    XX   X   XXX X  X  X ";
  /* define the 'try again' text */
  char *line1_tryagain = " XXXXX XXX  X  X    XX   XXX  XX  XXX X  X  X ";
  char *line2_tryagain = "   X   X  X X  X   X  X X    X  X  X  XX X  X ";
  char *line3_tryagain = "   X   XXX   XXX   XXXX X XX XXXX  X  X XX  X ";
  char *line4_tryagain = "   X   X  X    X   X  X X  X X  X  X  X  X    ";
  char *line5_tryagain = "   X   X  X  XX    X  X  XXX X  X XXX X  X  X ";
  /* define the 'pause' text */
  char *line1_pause = " XXX   XX  X  X  XXX XXX ";
  char *line2_pause = " X  X X  X X  X X    X   ";
  char *line3_pause = " XXX  XXXX X  X  XX  XXX ";
  char *line4_pause = " X    X  X X  X    X X   ";
  char *line5_pause = " X    X  X  XX  XXX  XXX ";
  /* populate the 'good job' messages */
  line1_goodjob[0] = "  XXX  XX   XX  XXX       X  XX  XXX   X ";
  line2_goodjob[0] = " X    X  X X  X X  X      X X  X X  X  X ";
  line3_goodjob[0] = " X XX X  X X  X X  X      X X  X XXX   X ";
  line4_goodjob[0] = " X  X X  X X  X X  X   X  X X  X X  X    ";
  line5_goodjob[0] = "  XXX  XX   XX  XXX     XX   XX  XXX   X ";
  line1_goodjob[1] = " X   X XXX X   X     XXX   XX  X  X XXX  X ";
  line2_goodjob[1] = " X   X X   X   X     X  X X  X XX X X    X ";
  line3_goodjob[1] = " X X X XXX X   X     X  X X  X X XX XXX  X ";
  line4_goodjob[1] = " X X X X   X   X     X  X X  X X  X X      ";
  line5_goodjob[1] = "  X X  XXX XXX XXX   XXX   XX  X  X XXX  X ";
  line1_goodjob[2] = " X   X XXX  XX XXX  X ";
  line2_goodjob[2] = " XX  X  X  X   X    X ";
  line3_goodjob[2] = " X X X  X  X   XXX  X ";
  line4_goodjob[2] = " X  XX  X  X   X      ";
  line5_goodjob[2] = " X   X XXX  XX XXX  X ";
  line1_goodjob[3] = " XXX X   X  XX XXX X   X   XXX X  X XXX  X ";
  line2_goodjob[3] = " X    X X  X   X   X   X   X   XX X  X   X ";
  line3_goodjob[3] = " XXX   X   X   XXX X   X   XXX X XX  X   X ";
  line4_goodjob[3] = " X    X X  X   X   X   X   X   X  X  X     ";
  line5_goodjob[3] = " XXX X   X  XX XXX XXX XXX XXX X  X  X   X ";

  /* link pointers to correct message */
  if (msgid == MSG_GOODJOB) {
      x = rand() & 3;
      line[0] = line1_goodjob[x];
      line[1] = line2_goodjob[x];
      line[2] = line3_goodjob[x];
      line[3] = line4_goodjob[x];
      line[4] = line5_goodjob[x];
      msgcolor = game->col_msg_good;
    } else if (msgid == MSG_TRYAGAIN) {
      line[0] = line1_tryagain;
      line[1] = line2_tryagain;
      line[2] = line3_tryagain;
      line[3] = line4_tryagain;
      line[4] = line5_tryagain;
      msgcolor = game->col_msg_bad;
    } else if (msgid == MSG_GAMEOVER) {
      line[0] = line1_gameover;
      line[1] = line2_gameover;
      line[2] = line3_gameover;
      line[3] = line4_gameover;
      line[4] = line5_gameover;
      msgcolor = game->col_msg_bad;
    } else if (msgid == MSG_PAUSE) {
      line[0] = line1_pause;
      line[1] = line2_pause;
      line[2] = line3_pause;
      line[3] = line4_pause;
      line[4] = line5_pause;
      msgcolor = game->col_msg_info;
    } else {
      return;
  }
  msglen = strlen(line[0]);
  msghoffset = 40 - (msglen >> 1);
  msgvoffset = (game->screenheight >> 1) + game->scrolloffset - 5;
  for (x = 0; line[0][x] != 0; x++) {
    printchar(msghoffset + x, msgvoffset - 1, ' ', msgcolor);
    printchar(msghoffset + x, msgvoffset + 5, ' ', msgcolor);
    for (linenum = 0; linenum < 5; linenum++) {
      if (line[linenum][x] == ' ') {
        printchar(msghoffset + x, msgvoffset + linenum, ' ', msgcolor);
      } else {
        printchar(msghoffset + x, msgvoffset + linenum, ' ', msgcolor << 4);
      }
    }
  }
}


static void PutItem(struct gamestruct *game, int itemnum) {
  int x, y;
  for (;;) {
    x = rand() % game->playfieldwidth;
    y = rand() % game->playfieldheight;
    y += game->playfieldvoffset;
    if (((game->playfield[y][x] & BLOCK_TYPE) == BLOCK_EMPTY) && (game->playfieldval[y][x] == 0)) break;
  }
  game->playfield[y][x] = BLOCK_REFRESH_FLAG | BLOCK_EMPTY;
  game->playfieldval[y][x] = itemnum;
}


static void pausegame(struct gamestruct *game) {
  int x, y;
  OutMessage(MSG_PAUSE, game);
  milisleep(500, game->sleepmeth);
  keybflush();
  getkey();
  milisleep(100, game->sleepmeth);
  /* mark all fields as 'to be refreshed' */
  for (y = 0; y < 50; y++) {
    for (x = 0; x < 80; x++) {
      game->playfield[y][x] |= BLOCK_REFRESH_FLAG;
    }
  }
}


static void refreshscreen(struct gamestruct *game) {
  int x, y, z;
  static char itemsarr[10] = {' ','1','2','3','4','5','6','7','8','9'};
  /* if screen's eight less than 50, then use scrolling when needed */
  if (game->screenheight < 50) {
    int newscrolloff = game->snakeposy[0] - 12;
    if (newscrolloff < 0) {
      newscrolloff = 0;
    } else if (newscrolloff > 25) {
      newscrolloff = 25;
    }
    if (newscrolloff != game->scrolloffset) {
      game->scrolloffset = newscrolloff;
      scrollscreen((newscrolloff << 6) + (newscrolloff << 4));
    }
  }
  /* iterate on fields to update them */
  for (y = game->playfieldvoffset + game->playfieldheight - 1; y >= game->playfieldvoffset; y--) {
    unsigned short *fieldshort = (unsigned short *)(game->playfield[y]);
    for (z = (game->playfieldwidth >> 1) - 1; z >= 0; z--) { /* for speed, I read 2 bytes a time */
      /* skip if both fields contain no snake and no refresh flag */
      if ((fieldshort[z] & DBLREFORSNAKE) == 0) continue;

      for (x = (z << 1) + 1; x >= (z << 1); x--) {
        /* if it's part of a snake, increment it */
        if ((game->playfield[y][x] & BLOCK_TYPE) == BLOCK_SNAKE1) {
          if (game->playfieldval[y][x] >= game->snakelen[0]) {
            game->playfield[y][x] = BLOCK_EMPTY | BLOCK_REFRESH_FLAG;
            game->playfieldval[y][x] = 0;
          } else {
            game->playfieldval[y][x] += 1;
          }
        }
        /* check if it needs refreshing - and if so, display it */
        if ((game->playfield[y][x] & BLOCK_REFRESH_FLAG) == 0) continue;  /* skip if no refresh needed */
        switch (game->playfield[y][x] & BLOCK_TYPE) {
          case BLOCK_EMPTY:
            printchar(x, y, itemsarr[game->playfieldval[y][x]], game->col_item);
            break;
          case BLOCK_SNAKE1:
            printchar(x, y, ' ', game->col_snake1);
            break;
          case BLOCK_WALL:
            printchar(x, y, ' ', game->col_wall);
            break;
        }
        /* mark the block as refreshed */
        game->playfield[y][x] &= ~BLOCK_REFRESH_FLAG;
      } /* for (x = ...) */
    }
  }
}


/* returns the current dos time, as a number of seconds */
static long getdostime(void) {
  long result;
  union REGS regs;
  regs.h.ah = 0x2C; /* get system time */
  int86(0x21, &regs, &regs);
  result = regs.h.ch * 3600l; /* hour (0-23) */
  result += regs.h.cl * 60l;  /* minutes (0-59) */
  result += regs.h.dh;        /* seconds (0-59) */
  return(result);
}


static int updatesnakeposition(struct gamestruct *game) {
  int blocktype, blockvalue;
  if (game->snakedirection[0] == SNAKEDIR_UP) {
      if (game->snakeposy[0] > game->playfieldvoffset) {
          game->snakeposy[0] -= 1;
        } else {
          game->snakeposy[0] = game->playfieldheight + game->playfieldvoffset - 1;
      }
    } else if (game->snakedirection[0] == SNAKEDIR_RIGHT) {
      if (game->snakeposx[0] + 1 < game->playfieldwidth) {
          game->snakeposx[0] += 1;
        } else {
          game->snakeposx[0] = 0;
      }
    } else if (game->snakedirection[0] == SNAKEDIR_DOWN) {
      if (game->snakeposy[0] + 1 < (game->playfieldheight + game->playfieldvoffset)) {
          game->snakeposy[0] += 1;
        } else {
          game->snakeposy[0] = game->playfieldvoffset;
      }
    } else if (game->snakedirection[0] == SNAKEDIR_LEFT) {
      if (game->snakeposx[0] > 0) {
          game->snakeposx[0] -= 1;
        } else {
          game->snakeposx[0] = game->playfieldwidth - 1;
      }
  }
  /* check where we are now */
  blocktype = game->playfield[game->snakeposy[0]][game->snakeposx[0]] & BLOCK_TYPE;
  if (blocktype == BLOCK_EMPTY) {
      blockvalue = game->playfieldval[game->snakeposy[0]][game->snakeposx[0]];
      game->playfield[game->snakeposy[0]][game->snakeposx[0]] = BLOCK_SNAKE1 | BLOCK_REFRESH_FLAG;
      game->playfieldval[game->snakeposy[0]][game->snakeposx[0]] = 1;
      if (blockvalue > 0) {
        game->snakescore[0] += 1;
        PrintScoreBar(game);
        if (blockvalue == 9) return(EXITGAME_SUCCESS);
        game->snakelen[0] = blockvalue * 20;
        blockvalue++;
        PutItem(game, blockvalue);
      }
    } else {
      if (game->snakescore[0] >= 10) {
          game->snakescore[0] -= 10;
          return(EXITGAME_RESTART);
        } else {
          return(EXITGAME_GAMEOVER);
      }
  }
  return(-1);
}


static int processkeys(struct gamestruct *game) {
  int lastkey;
  for (;;) { /* looping used to ignore invalid keys (like RIGHT if snake already running right) */
    lastkey = getkey_ifany();
    if (lastkey == -1) { /* no key pressed */
        return(-1);
      } else if (lastkey == 0x150) { /* DOWN */
        if ((game->snakedirection[0] == SNAKEDIR_LEFT) || (game->snakedirection[0] == SNAKEDIR_RIGHT)) {
          game->snakedirection[0] = SNAKEDIR_DOWN;
          return(-1);
        }
      } else if (lastkey == 0x148) { /* UP */
        if ((game->snakedirection[0] == SNAKEDIR_LEFT) || (game->snakedirection[0] == SNAKEDIR_RIGHT)) {
          game->snakedirection[0] = SNAKEDIR_UP;
          return(-1);
        }
      } else if (lastkey == 0x14B) { /* LEFT */
        if ((game->snakedirection[0] == SNAKEDIR_UP) || (game->snakedirection[0] == SNAKEDIR_DOWN)) {
          game->snakedirection[0] = SNAKEDIR_LEFT;
          return(-1);
        }
      } else if (lastkey == 0x14D) { /* RIGHT */
        if ((game->snakedirection[0] == SNAKEDIR_UP) || (game->snakedirection[0] == SNAKEDIR_DOWN)) {
          game->snakedirection[0] = SNAKEDIR_RIGHT;
          return(-1);
        }
      } else if (lastkey == 0x1B) { /* ESC */
        return(EXITGAME_ESCAPE);
      } else if ((lastkey == 0x0D) || (lastkey == 'p') || (lastkey == 'P')) { /* ENTER or 'P' */
        pausegame(game);
        return(-1);
    }
  }
}


static int PlayGame(struct gamestruct *game) {
  int exitflag = -1;
  PutItem(game, 1); /* put the first item onscreen */
  keybflush(); /* flush the keyboard buffer */
  PrintScoreBar(game); /* draw the score bar */
  refreshscreen(game); /* draw the initial screen */
  milisleep(500, game->sleepmeth); /* wait half a second */
  while (exitflag < 0) {
    /* wait a short moment */
    milisleep(0 - (32 + ((10 - game->snakespeed[0]) * 10)), game->sleepmeth);
    /* process input keys as long as there are keys in the queue, or an action happen */
    exitflag = processkeys(game); /* process input keys */
    if (exitflag >= 0) break;
    /* move the snake */
    exitflag = updatesnakeposition(game);
    /* refresh the screen */
    refreshscreen(game);
  }
  return(exitflag);
}


static void AdjustSpeedToScore(struct gamestruct *game) {
  int x;
  for (x = 0; x < 2; x++) {
    if (game->snakescore[x] >= 90) {
      game->snakespeed[x] = 10;
    } else {
      game->snakespeed[x] = 1 + game->snakescore[x] / 9;
    }
  }
}


static void showhelp(void) {
  puts("Zmiy v" PVER " Copyright (C) Mateusz Viste " PDATE "\n"
       "\n"
       " /timer=dos    use 'int28' DOS idle interrupts (default)\n"
       " /timer=dpmi   use 'int2F AX=1680h' DPMI idle calls\n"
       " /timer=apm    use 'int15 AX=5305h' APM calls for timing\n"
       " /timer=bios   use 'int15 AH=86h' BIOS calls for timing\n"
       " /timer=loop   use a busy loop for timing\n"
       " /cga          force CGA mode, even if a VGA card is available\n"
       " /mono         use the monochrome display scheme\n"
       " /color        use the color scheme (default, unless DOS is in BW mode)\n"
       "");
}


int main(int argc, char **argv) {
  struct gamestruct *game;
  enum gameexitstatus exitflag = EXITGAME_SUCCESS;
  int oldvmode;

  /* allocate memory for the game structure first */
  game = calloc(1, sizeof(struct gamestruct));
  if (game == NULL) {
    puts("Error: out of memory!");
    return(1);
  }

  /* set some default params */
  game->sleepmeth = MILISLEEP_DOS;  /* use DOS idle interrupts by default */
  game->mono = 0;
  game->screenheight = 50;

  /* Parse command line params */
  while (--argc > 0) {
    if (strcmp(argv[argc], "/timer=bios") == 0) {
      game->sleepmeth = MILISLEEP_BIOS;
    } else if (strcmp(argv[argc], "/timer=apm") == 0) {
      game->sleepmeth = MILISLEEP_APM;
    } else if (strcmp(argv[argc], "/timer=dos") == 0) {
      game->sleepmeth = MILISLEEP_DOS;
    } else if (strcmp(argv[argc], "/timer=dpmi") == 0) {
      game->sleepmeth = MILISLEEP_DPMI;
    } else if (strcmp(argv[argc], "/timer=loop") == 0) {
      game->sleepmeth = MILISLEEP_LOOP;
    } else if (strcmp(argv[argc], "/cga") == 0) {
      game->screenheight = 25;
    } else if (strcmp(argv[argc], "/mono") == 0) {
      game->mono = 1;
    } else if (strcmp(argv[argc], "/color") == 0) {
      game->mono = -1;
    } else {
      free(game);
      showhelp();
      return(0);
    }
  }

  /* first of all, save the current video mode to restore it later */
  oldvmode = getcurvideomode();

  /* if the old mode is monochrome, run zmiy to mono if color not forced */
  if (((oldvmode == 0x02) || (oldvmode == 0x00)) && (game->mono == 0)) game->mono = 1;

  /* if no VGA detected, use the standard 80x25 mode + scrolling */
  if (detectvga() == 0) {
    game->screenheight = 25;
  }

  /* load the color scheme */
  if (game->mono == 1) { /* can be 0, 1 or -1 (forbidden) */
    game->col_bar = MONO_BAR;
    game->col_snake1 = MONO_SNAKE1;
    game->col_wall = MONO_WALL;
    game->col_item = MONO_ITEM;
    game->col_msg_good = MONO_MSG_GOOD;
    game->col_msg_bad = MONO_MSG_BAD;
    game->col_msg_info = MONO_MSG_INFO;
  } else {
    game->col_bar = COLOR_BAR;
    game->col_snake1 = COLOR_SNAKE1;
    game->col_wall = COLOR_WALL;
    game->col_item = COLOR_ITEM;
    game->col_msg_good = COLOR_MSG_GOOD;
    game->col_msg_bad = COLOR_MSG_BAD;
    game->col_msg_info = COLOR_MSG_INFO;
  }

  setvideomode_80(game->screenheight);
  cursor_set(0x0F, 0x0E); /* hide the cursor */
  srand((unsigned int)getdostime());   /* feed the random generator with a seed */

  game->snakescore[0] = 0;
  game->snakescore[1] = 0;
  game->level = 1;

  timer_init();

  for (;;) {
    AdjustSpeedToScore(game);
    LoadLevel(game, game->level);
    scrollscreen(0);
    exitflag = PlayGame(game);
    if ((exitflag == EXITGAME_GAMEOVER) || (exitflag == EXITGAME_ESCAPE)) break;
    if (exitflag == EXITGAME_SUCCESS) {
      OutMessage(MSG_GOODJOB, game);
      game->level += 1;
    } else {
      OutMessage(MSG_TRYAGAIN, game);
    }
    milisleep(500, game->sleepmeth);
    keybflush();
    getkey();
  }

  OutMessage(MSG_GAMEOVER, game);
  milisleep(800, game->sleepmeth);
  keybflush();
  if (exitflag == EXITGAME_GAMEOVER) { /* if the game is over for natural reasons, wait for a keypress */
    getkey();                          /* (do NOT wait if the game has been aborted with Escape) */
  }

  free(game);

  timer_stop();

  /* restore the previous video mode */
  setvideomode(oldvmode);

  puts("Zmiy v" PVER " Copyright (C) Mateusz Viste " PDATE "\n"
       "\n"
       "Redistribution and use in source and binary forms, with or without\n"
       "modification, are permitted provided that the following conditions are met:\n"
       "\n"
       " 1. Redistributions of source code must retain the above copyright notice, this\n"
       "    list of conditions and the following disclaimer.\n"
       " 2. Redistributions in binary form must reproduce the above copyright notice,\n"
       "    this list of conditions and the following disclaimer in the documentation\n"
       "    and/or other materials provided with the distribution.\n");

  return(0);
}
