\MATEY.XPL	17-May-2010	Loren Blaney	loren.blaney@idcomm.com
\Chess Playing Program
\Compile with XPLX (version 2.4.6 or better - xjsb.bat).
\
\REVISIONS
\1.0, 29-Jul-2004, Released original version.
\1.1, 10-Feb-2005, Fix uninitialized variable bug when two people play. Sound
\ button turns off dieing king sound.
\1.2, 10-Apr-2010, 3-way Info button also shows legal moves for selected piece,
\ smaller .bmp file uses 16-color format (instead of 256-color), less confusing
\ options for who goes first and second.
\1.3, 25-Apr-2010, When in 2-player mode, allow both players to castle. Only
\ allow legal castling moves. Show hand symbol immediately after clicking on a
\ piece. Run slightly faster.
\1.3pb, 2-May-2010, Info button defaults to show legal moves.
\1.4, 17-May-2010, Timer clocks show time taken by each player.
\
\The 8x8 playing board is surrounded by a border. The border provides an
\ efficient way to detect if a piece is attempting to move off the board.
\ The squares are numbered 0 thru 77 as shown here:
\
\			      black
\			0 1 2 3 4 5 6 7 8 9
\		   00 # # # # # # # # # # #		# = border
\		   10	# # # # # # # # # #
\	Board ->   00	. . . . . . . . # #  8		Note that the address of
\		   10	. . . . . . . . # #  7		 the array is relocated
\		   20	. . . . . . . . # #  6
\		   30	. . . . . . . . # #  5
\		   40	. . . . . . . . # #  4
\		   50	. . . . . . . . # #  3
\		   60	. . . . . . . . # #  2
\		   70	. . . . . . . . # #  1
\		   80	# # # # # # # # # #
\		   90	# # # # # # # # #
\			a b c d e f g h
\			      white
\
inc	C:\CXPL\CODESI;			\intrinsic routine definitions

def	Random=20;			\Randomness in move selection (1..100)
def	IntSize=2;			\number of bytes in an integer (2 or 4)
def	SideLen=48,			\length of side of board square (pixels)
	OffsetX=(640-SideLen*8)-32,	\position of board on screen (pixels)
	OffsetY=(480-SideLen*8)/2;	\ (a wooden border surrounds the board)
def	LineMax=27;	\maximum (bottom) line number for listing moves
def	ButtonWidth=7*8, ButtonHeight=16, \soft button dimensions (pixels)
	ResetX=30*8, ResetY=29*16,	\button screen positions (in pixels, but
	SoundX=40*8, SoundY=29*16,	\ on character cell boundaries)
	InfoX=50*8,  InfoY=29*16,
	UndoX=60*8,  UndoY=29*16,
	ExitX=70*8,  ExitY=29*16;

def	Empty=	   0,			\values of square contents (pieces)
	Pawn=	 100,			\black is positive; white is negative
	Knight=	 300,
	Bishop=	 320,
	Rook=	 500,
	Queen=	 900,
	King=  10000,
	Border=30000;

def	M0=3, M1=7, M2=20;		\values of squares in MobTbl (mobility)

int	CpuReg,		\address of CPU register array (from GetReg)
	CursorX, CursorY, \screen cursor position for listing moves (char cells)
	DataSeg,	\segment address of our data
	DoInfo,		\show move values or legal moves (3-way Info button)
	DoSound,	\flag: play sounds (Sound button)
	Handle,		\output file handle for GAME.TXT
	Human1st,	\flag: human plays white pieces (else Matey plays white)
	Matey2nd,	\flag: Matey plays black pieces (else human plays black)
	MayCastleBlackKingSide,	 \black's rook or king didn't move
	MayCastleBlackQueenSide,
	MayCastleWhiteKingSide,	 \white's rook or king didn't move
	MayCastleWhiteQueenSide,
	MayCastle2BlackKingSide, \backup copies for Undo command
	MayCastle2BlackQueenSide,
	MayCastle2WhiteKingSide,
	MayCastle2WhiteQueenSide,
	Resigned,	\tips over king (Empty, King, -King)
	Resigned2;	\backup copy for Undo command

\Variables used by TryMove procedure:
int	BoardTbl,	\table of Board's square numbers, most likely move first
	Depth,		\current depth of recursive search
	DepthMax,	\normal maximum depth of recursion (= Level)
	DepthCap,	\maximum (extended) depth when captures occur
	Illegal,	\flag: test human's move for legality
	MobTbl,		\mobility table (the center of the board is worth more)
	MoveCount,	\move counter
	MoveCount2,	\backup copy of move counter for Undo command
	MoveSrc,	\move's source square number
	MoveDst,	\move's destination square number
	MoveCap,	\move's captured piece (can be Empty)
	MovedPiece,	\move's piece
	MoveValue,	\value of move
	OffTbl,		\move offset table for rooks, bishops, queens, and kings
	OffTblK,	\move offset table for knights
	OffTblP;	\move offset table for pawns

int	Board(10*12),	\playing board (see diagram above)
	Board2(10*12),	\backup copy of playing board for Undo command
	BoardT(10*12),	\temporary copy of playing board to test for legal moves
	CapWh(6),	\counts for displaying captured white pieces: pawn..king
	CapBl(6),	\ditto for black pieces
	CapWh2(6),	\backup copies for Undo command
	CapBl2(6);

int	Alpha(11);	\cutoff values for alpha-beta tree pruning
			\ (maximum depth = 7 + 7/2 = 10, starts at 0)
\Sprite images:
char	SpPieceWh(6, 2+48*48),	\playing pieces (white and black)
	SpPieceBl(6, 2+48*48),
	SpSmPieceWh(6, 2+12*14),\small (captured) pieces
	SpSmPieceBl(6, 2+12*14),
	SpSmBlank(2+12*14),	\small blank for erasing captured pieces
	SpWoodWh(4, 2+48*48),	\board squares (white/maple and black/walnut)
	SpWoodBl(4, 2+48*48),
	SpBorder(2+216*24);	\board border (walnut)

\Mouse pointer images:
int	PtrArrow(2*16),		\normal arrow
	PtrHour(2*16),		\hourglass
	PtrHand(2*16);		\grasping hand

def	BS=$08, CR=$0D, Esc=$1B; \ASCII control characters

\-------------------------------------------------------------------------------

proc	Exit(Msg);		\Display (error) message and terminate program
char	Msg;
begin
SetVid(3);			\restore normal text mode
Text(0, Msg);
CrLf(0);
exit;				\(output file is closed automatically)
end;	\Exit



func	CallInt(Int, AX, BX, CX, DX, BP, DS, ES); \Call a software interrupt
int	Int, AX, BX, CX, DX, BP, DS, ES; \(unused arguments need not be passed)
begin
CpuReg(0):= AX;
CpuReg(1):= BX;
CpuReg(2):= CX;
CpuReg(3):= DX;
CpuReg(6):= BP;
CpuReg(9):= DS;
CpuReg(11):= ES;
SoftInt(Int);
return CpuReg(0);		\return AX register
end;	\CallInt



proc	AlignPalette;		\Make palette regs correspond to VGA DAC regs
int	N;
for N:= 0, 15 do
	CallInt($10, $1000, N<<8 !N);	\function $10, subfunction $00



proc	Beep;			\A not-too-obnoxious beep
begin
Sound(false, 1, 1000);		\synchronize with system timer to make tone a
Sound(DoSound, 1, 3000);	\ consistent duration and a consistent sound
end;	\Beep



proc	Delay(T);		\Delay T eighteenths of a second (approximately)
int	T;
Sound(false, T, 1);



func	IntLen(N);		\Return number of digits in N, a decimal integer
int	N;			\adds 1 for a minus sign if N is negative
int	I;
for I:= if N>=0 then 1 else 2, 20 do
	[N:= N /10;   if N = 0 then return I];



func	KeyIn;			\One-char buffered input for device 6; provides
int	I, Key, Flash;		\ background color along with a flashing cursor
char	Buf(2);			\ (graphic modes don't provide a cursor)
begin
I:= 0;				\initialize index into Buf
Flash:= true;			\start with cursor flashed on
loop	begin
	repeat	begin
		ChOut(6, if Flash then ^_ else ^ );
		Flash:= ~Flash;	\alternate flashing cursor
		ChOut(1, BS);
		Delay(3);
		end;
	until ChkKey;
	ChOut(6, ^ );		\make sure flashing cursor is off
	ChOut(1, BS);

	Key:= ChIn(1);		\(doesn't echo to display)
	case Key of
	  BS:	if I > 0 then
			[I:= I - 1;   ChOut(1, BS)];
	  Esc:	Exit(" ");
	  CR:	[Buf(I):= Key;   CrLf(6);   quit]
	other	if I = 0 then
			[Buf(I):= Key;   I:= I + 1;   ChOut(6, Key)];
	end;
return Buf(0);
end;	\KeyIn

\=============================== MOUSE ROUTINES ================================

func	OpenMouse;		\Initializes mouse; returns 'false' if it fails
begin				\pointer is set to center of screen but is hidden
CallInt($21, $3533);		\make sure mouse vector ($33) points to something
if CpuReg(1)=0 & CpuReg(11)=0 then return false;
return CallInt($33, $0000);	\reset mouse; return 'false' if failure
end;	\OpenMouse



proc	ShowMouse(On);		\Turn mouse pointer on or off
\The video mode should be set before calling this routine.
\This counts the number of times the pointer is turned on or off. If the
\ pointer is turned on twice, it must be turned off twice before it
\ actually goes off. The pointer should be turned off before drawing over
\ it and before your program exits. Setting the video mode will also turn
\ off the pointer.
int	On;	\Flag: True = pointer on; False = pointer off
CallInt($33, if On then $0001 else $0002);



func	GetMousePosition(N); \Return position of specified mouse coordinate
int	N;	\0 = X coordinate; 1 = Y coordinate
\For video modes $0-$E and $13 the maximum coordinates are 639x199, minus
\ the size of the pointer. For modes $F-$12 the coordinates are the same as
\ the pixels. For 80-column text modes divide the mouse coordinates by 8 to
\ get the character cursor position.
begin
CallInt($33, $0003);
return if N then CpuReg(3) else CpuReg(2);
end;	\GetMousePosition



func	GetMouseButton;		\Returns 'true' if a mouse button is down
begin
if ChkKey then
	if ChIn(1) = Esc then Exit(" ");	\(keys not echoed to display)
CallInt($33, $0003);
return (CpuReg(1) & $03) # 0;
end;	\GetMouseButton

\===============================================================================

char	TimeOfDay(4),		\current time (0=hours, 1=mins, 2=secs, 3=huns)
	TimerWhite(4), TimerBlack(4),
	Time0(4);		\initial time for a player's turn (sort of)
int	SecOld;			\used to only update display when seconds change


proc	GetTimeOfDay;		\Return time of day in TimeOfDay(4)
begin
CpuReg(0):= $2C00;
SoftInt($21);
TimeOfDay(0):= CpuReg(2)>>8;	\hour
TimeOfDay(1):= CpuReg(2);	\minute
TimeOfDay(2):= CpuReg(3)>>8;	\second
TimeOfDay(3):= CpuReg(3);	\hundreths
end;	\GetTimeOfDay



proc	ShowHMS(T, X);		\Display hours : minutes : seconds
char	T;			\time array (0=hours, 1=mins, 2=secs, 3=huns)
int	X;			\horizontal position to display
int	I;
begin
ShowMouse(false);
OpenO(6);			\remove SetWind limits
Attrib($5E);			\dark brown on white
Cursor(X, 0);

for I:= 0, 2 do
	begin
	if T(I) < 10 then ChOut(6, ^0);	\show leading zero
	IntOut(6, T(I));
	if I<2 then ChOut(6, ^:);
	end;

SetWind(1, 2, 22, LineMax, 0, false);	\restore move list window
ShowMouse(true);
Attrib($D7);			\restore white on brown for move list
Cursor(CursorX, CursorY);	\restore cursor position in window
end;	\ShowHMS



proc	ElapsedTime(A, B, C);	\Return C:= A-B [hrs(0), min(1), sec(2), hun(3)]
char	A, B, C;
int	Borrow;
begin
C(3):= A(3) - B(3);		\hundreths
Borrow:= 0;
if Extend(C(3)) < 0 then [Borrow:= 1;  C(3):= C(3)+100];

C(2):= A(2) - B(2) - Borrow;	\seconds
Borrow:= 0;
if Extend(C(2)) < 0 then [Borrow:= 1;  C(2):= C(2)+60];

C(1):= A(1) - B(1) - Borrow;	\minutes
Borrow:= 0;
if Extend(C(1)) < 0 then [Borrow:= 1;  C(1):= C(1)+60];

C(0):= A(0) - B(0) - Borrow;	\hours
if Extend(C(0)) < 0 then C(0):= C(0)+24;
end;	\ElapsedTime



proc	TickTock(White);	\Display seconds ticking away on clocks
int	White;		\set 'true' for white's move; 'false' for black's
begin
GetTimeOfDay;
if White then
	begin
	ElapsedTime(TimeOfDay, Time0, TimerWhite);
	if TimerWhite(2) # SecOld then
		begin
		SecOld:= TimerWhite(2);
		ShowHMS(TimerWhite, 55);
		end;
	end
else	begin
	ElapsedTime(TimeOfDay, Time0, TimerBlack);
	if TimerBlack(2) # SecOld then
		begin
		SecOld:= TimerBlack(2);
		ShowHMS(TimerBlack, 67);
		end;
	end;
end;	\TickTock

\===============================================================================

proc	DrawSprite(X0, Y0, S, Trans, Flip);	\Draw a sprite
int	X0, Y0;		\coordinates of upper-left corner (pixels)
char	S;		\address of sprite data
int	Trans,		\flag: transparent background (versus opaque)
	Flip;		\flag: rotate image about a -45 degree axis
int	X, Y, I, C, W, H;
begin
W:= S(0);		\get width and height (in pixels)
H:= S(1);
I:= 2;
for Y:= 0, H-1 do
    for X:= 0, W-1 do
	begin
	C:= S(I);			\get pixel's color
	I:= I + 1;
	if C ! ~Trans then		\background (0) is transparent
		if Flip then Point(X0+Y, Y0+X, C) else Point(X0+X, Y0+Y, C);
	end;
end;	\DrawSprite

\-------------------------------------------------------------------------------

proc	LoadSprites;	\Load sprite images from .BMP file and set color regs
int	Hdl,		\file handle
	W, H,		\dimensions of entire image in .BMP file (pixels)
	C, R, G, B,	\color, red, green, blue
	X, Y,		\screen coordinates (pixels)
	Y320,		\Y * 320
	I, T;		\indexes and temporary scratch
seg char Screen(1);	\temporary memory area for loading images



proc	LoadPointer(X0, Y0, Ptr);	\Load mouse pointer image into Ptr
int	X0, Y0,		\coordinates on Screen to get pointer from
	Ptr;		\array to load
int	Xor,		\xor portion of Ptr array (the second half)
	S, I, X, Y, C;
begin
Xor:= Ptr + 2*16;		\point to 2nd half of integer array
I:= 0;
for Y:= Y0, Y0+16-1 do
	begin
	Ptr(I):= $FFFF;
	Xor(I):= $0000;
	S:= $8000;
	for X:= X0, X0+16-1 do
		begin
		C:= Screen(0, X+Y*320);
		if C # 0 then
			Ptr(I):= Ptr(I) & ~S;
		if C = $F then
			Xor(I):= Xor(I) ! S;
		S:= S >> 1;
		end;
	I:= I + 1;
	end;
end;	\LoadPointer



proc	LoadSprite(X0, Y0, W, H, Sp);	\Load a sprite data array from Screen
int	X0, Y0,		\coordinates in Screen to get sprite from
	W, H;		\width and height of sprite (pixels)
char	Sp;		\sprite to load
int	I, X, Y;
begin
Sp(0):= W;
Sp(1):= H;
I:= 2;
for Y:= Y0, Y0+H-1 do
    for X:= X0, X0+W-1 do
	begin
	Sp(I):= Screen(0, X+Y*320);
	I:= I + 1;
	end;
end;	\LoadSprite



begin	\LoadSprites
Screen(0):= MAlloc(320/16*200);		\temporary 64000-byte hunk of memory

\Read in a 16-color .BMP file
Trap(false);			\don't trap "file not found" error; we'll do it
Hdl:= FOpen("MATEY.BMP", 0);	\open file for input
if GetErr \#0\ then Exit("MATEY.BMP not found");
Trap(true);
FSet(Hdl, ^I);			\set device 3 to handle
OpenI(3);

for I:= 0, 17 do T:= ChIn(3);	\skip unused header info
W:= ChIn(3) + ChIn(3)<<8;	\0..32764 (ample range)
for I:= 0, 2-1 do T:= ChIn(3);	\skip
H:= ChIn(3) + ChIn(3)<<8;	\0..32767 (ample range)
for I:= 24, 53 do T:= ChIn(3);	\skip

port($3C8):= 0;			\set color registers
for I:= 0, 16-1 do
	begin
	B:= ChIn(3)>>2;
	G:= ChIn(3)>>2;
	R:= ChIn(3)>>2;
	T:= ChIn(3);
	port($3C9):= R;
	port($3C9):= G;
	port($3C9):= B;
	end;

\Load .BMP image into Screen
for Y:= -(H-1), 0 do		\.BMP files are upside down (!)
	begin
	Y320:= 320 * -Y;
	for X:= 0, W-1 do
		begin
		C:= ChIn(3);
		Screen(0, X+Y320):= C>>4;
		X:= X+1;
		Screen(0, X+Y320):= C & $0F;
		end;
	end;
FClose(Hdl);			\close handle so it can be used again

for I:= 0, 5 do
	begin			\load piece images
	LoadSprite(I*48,  0, 48, 48, addr SpPieceWh(I,0));
	LoadSprite(I*48, 48, 48, 48, addr SpPieceBl(I,0));
	LoadSprite(288, I*16+2, 12, 14, addr SpSmPieceWh(I,0));
	LoadSprite(300, I*16+2, 12, 14, addr SpSmPieceBl(I,0));
	end;
LoadSprite(288, 6*16+2, 12, 14, SpSmBlank);

for I:= 0, 3 do
	begin			\load square images
	LoadSprite(I*48,  96, 48, 48, addr SpWoodWh(I,0));
	LoadSprite(I*48, 144, 48, 48, addr SpWoodBl(I,0));
	end;
LoadSprite(0, 160, 216, 24, SpBorder);

LoadPointer(288, 112, PtrArrow);
LoadPointer(304, 112, PtrHour);
LoadPointer(288, 128, PtrHand);

Release(Screen(0));
end;	\LoadSprites

\===============================================================================

proc	ShadeButton(X, Y);	\Make button look 3D
int	X, Y;	\coordinates of upper-left corner of button face (pixels)
begin
Move(X-1, Y+ButtonHeight-2);
Line(X-1, Y-1, 7);
Line(X+ButtonWidth-2, Y-1, 7);
Move(X, Y+ButtonHeight-1);
Line(X+ButtonWidth-1, Y+ButtonHeight-1, 9);
Line(X+ButtonWidth-1, Y, 9);
end;	\ShadeButton



proc	ShowVals(Sq, V1, V2);	\Display values V1 & V2 on square Sq
int	Sq, V1, V2;
int	X, Y, I;
begin
Y:= Sq/10;			\get square coordinates in character cells
X:= rem(0);
X:= (OffsetX + SideLen*X)/8;
Y:= (OffsetY + SideLen*Y)/16;

Cursor(X, Y+1);			\display right-justified value
ShowMouse(false);
for I:= 1, 6-IntLen(V1) do ChOut(0, ^ );
IntOut(0, V1);

Cursor(X, Y+2);
for I:= 1, 6-IntLen(V2) do ChOut(0, ^ );
IntOut(0, V2);
ShowMouse(true);
Delay(18);
end;	\ShowVals



proc	DrawLine(Org, Dst);	\Draw a line between squares
int	Org, Dst;
int	X0, Y0, X1, Y1;
begin
Y0:= Org/10;			\get coordinates of centers of squares
X0:= rem(0);
Y0:= Y0*SideLen + OffsetY + SideLen/2;
X0:= X0*SideLen + OffsetX + SideLen/2;

Y1:= Dst/10;
X1:= rem(0);
Y1:= Y1*SideLen + OffsetY + SideLen/2;
X1:= X1*SideLen + OffsetX + SideLen/2;

if X0#X1 ! Y0#Y1 then				\don't draw single-pixel lines
	begin					\ (it leaves dots behind)
	ShowMouse(false);
	Move(X0, Y0);   Line(X1, Y1, $8E);	\use XOR so line can be erased
	ShowMouse(true);			\ and its background restored
	end;
end;	\DrawLine



proc	DrawRectangle(X, Y, W, H, C, F);	\Draw a filled or unfilled box
int	X, Y,	\coordinates of upper-left corner (pixels)
	W, H,	\width and height
	C, F;	\color and fill flag
int	J;
begin
ShowMouse(false);
if F then					\draw a filled (solid) box
	begin
	for J:= Y, Y+H-1 do
		[Move(X, J);   Line(X+W-1, J, C)];
	end
else	begin				       \draw a non-filled (outlined) box
	Move(X, Y);
	Line(X+W-1, Y, C);
	Line(X+W-1, Y+H-1, C);
	Line(X, Y+H-1, C);
	Line(X, Y, C);
	end;
ShowMouse(true);
end;	\DrawRectangle



proc	OutlineSquare(S);	\Draw an outline around a chessboard square
int	S;	\square number
int	I, J;	\board coordinates (squares)
begin
ShowMouse(false);
J:= S/10;			\get coordinates of squares
I:= rem(0);
DrawRectangle(OffsetX+I*SideLen,   OffsetY+J*SideLen,   SideLen,   SideLen,
		$AA00, false);
DrawRectangle(OffsetX+I*SideLen+1, OffsetY+J*SideLen+1, SideLen-2, SideLen-2,
		0, false);
DrawRectangle(OffsetX+I*SideLen+2, OffsetY+J*SideLen+2, SideLen-4, SideLen-4,
		0, false);
DrawRectangle(OffsetX+I*SideLen+3, OffsetY+J*SideLen+3, SideLen-6, SideLen-6,
		$AA00, false);
ShowMouse(true);
end;	\OutlineSquare



proc	DrawSquare(S);		\Draw a chessboard square
int	S;	\square number
int	I, J;	\board coordinates (squares)
begin
ShowMouse(false);
J:= S/10;			\get coordinates of squares
I:= rem(0);
if (J|I)&1 then	\alternate colors with white in lower-right corner of board
	DrawSprite(OffsetX+I*SideLen, OffsetY+J*SideLen, addr SpWoodBl(I&3,0),
		false, true)
else	DrawSprite(OffsetX+I*SideLen, OffsetY+J*SideLen, addr SpWoodWh(I&3,0),
		false, false);
ShowMouse(true);
end;	\DrawSquare



proc	DrawPiece(S, P);	\Draw piece P on a square S
int	S, P;
int	I, J;	\board coordinates (squares)
int	Tbl, K;
begin
if P = Empty then return;
Tbl:= [Pawn, Bishop, Knight, Rook, Queen, King];

loop for K:= 0, 5 do		\get index (K) of sprite for piece
	if abs(P) = Tbl(K) then quit;

J:= S/10;
I:= rem(0);
ShowMouse(false);
DrawSprite(OffsetX+I*SideLen, OffsetY+J*SideLen,
	if P >= 0 then addr SpPieceBl(K,0) else addr SpPieceWh(K,0), true,
	P=Resigned);			\if equal then tip over king
ShowMouse(true);
end;	\DrawPiece



proc	DrawBoard;		\Draw the chess board and all of its pieces
int	I, J;
begin
for J:= 0, 7 do
    for I:= 0, 7 do
	begin
	DrawSquare(I+J*10);
	DrawPiece(I+J*10, Board(I+J*10));
	end;
end;	\DrawBoard



proc	ShowCaps;		\Show the captured pieces
int	I, J, X, Y, Cap, Sp, P;
begin
ShowMouse(false);
Cap:= CapWh;
Sp:= SpSmPieceWh;
Y:= 28*16+2;			\empirically derived vertical coordinate
for J:= 0, 1 do			\for white and black...
	begin
	X:= 0;			\horizontal coordinate
	for P:= -5, 0 do	\for piece type 5 downto 0...
	    for I:= 1, Cap(-P) do
		begin
		DrawSprite(X, Y, Sp(-P), false, false);
		X:= X + 12;	\squeeze in 16 pieces
		end;
	DrawSprite(X, Y, SpSmBlank, false, false);	\for Undo command
	Cap:= CapBl;
	Sp:= SpSmPieceBl;
	Y:= 29*16+2;
	end;
ShowMouse(true);
end;	\ShowCaps



proc	RecordCap(P);		\Record captured piece P (if any)
int	P;			\ so it can be displayed
int	Tbl;
int	I, J;
begin
if P = Empty then return;
Tbl:= [Pawn, Bishop, Knight, Rook, Queen, King];
for I:= 0, 5 do			\get index of captured piece
	if abs(P) = Tbl(I) then J:= I;
if P < 0 then
	CapWh(J):= CapWh(J) + 1
else	CapBl(J):= CapBl(J) + 1;
end;	\RecordCap



proc	KillKing(K);		\Long live(d) the King!
int	K;	\+King or -King
int	I;
begin
if Resigned # K then				\haven't done it before
	begin
	Resigned:= K;
	for I:= 1, 8 do Sound(DoSound, 1, 4000*I);	\cute sound
	for I:= 0, 77 do
	    if Board(I) = K then
		begin
		DrawSquare(I);			\erase old king
		DrawPiece(I, K);		\draw toppled king
		end;
	Delay(36);				\don't move immediately
	end;
end;	\KillKing

\===============================================================================

func	TryMove(White, StaticValue0, EPDst); \Returns value of the best move
\ (from the standpoint of the current side), which is the largest positive value
int	White,		\set 'true' for white's move; 'false' for black's
	StaticValue0,	\sum of node values, initial value for current Depth
	EPDst;		\destination of opponent's pawn that moved two squares
			\ if non-zero then en-passant move is explored
int	BackupValue,	\value of descendent node
	BestValue,	\best descendent node
	Capture,	\contents of square before moving into it (can be Empty)
	Dir,		\direction to move
	Dst,		\destination (or new location) of a tenative move
	EPSkip,		\en passant: square no. skipped by a 2-square pawn move
	GoDeeper,	\flag: search deeper (if capture or queened pawn)
	I,		\scratch index
	Inx,		\index into Board
	Loc,		\location on Board
	Mobility,	\sum of the value's of squares covered by major pieces
	MY,		\mouse coordinate (pixels)
	Org,		\square number (on Board) where piece starts from
	Piece,		\piece that is moving (also can be a Border or Empty)
	Sq,		\contents of square (usually Empty)
	StaticValue;	\sum of node values leading down to current node


proc	MovePiece(Dir1, Dir2, OneStep);	\Trial move of non-pawn piece
int	Dir1, Dir2,	\beginning and ending pointers into OffTbl (or OffTblK)
	OneStep;	\flag: piece moves once per direction (e.g: King) 
begin
for Dir:= Dir1, Dir2 do		\for the directions in OffTbl...
	begin
	Loc:= Org;		\start at the origin
	Capture:= Piece;	\(in case there is no move in this direction)

	loop	begin
		Dst:= Loc + Dir(0);
		Sq:= Board(Dst);
		if Sq=Empty ! Sq#Border & (Sq|White)<0 \opponent piece\ then
			begin
			Capture:= Sq;				\(can be Empty)
			StaticValue:= abs(Capture) + StaticValue0;
			Mobility:= Mobility + MobTbl(Dst);

			\keep the queen and king from popping out early in game
			if abs(Piece) >= Queen then
			    begin
			    if MoveCount < 12 then
				 StaticValue:= StaticValue - (120-MoveCount*10);
			    if abs(Piece) = King then	\make mobility neutral
				Mobility:= Mobility - MobTbl(Dst);
			    end
			else		\advancement value is about 10 per rank
			    if White then
				 StaticValue:= StaticValue + Org - Dst
			    else StaticValue:= StaticValue + Dst - Org;

			if Depth=1 & DoInfo=1 then
				begin
				DrawLine(Org, Loc);		\erase old line
				if Loc # Org then DrawSquare(Loc);
				DrawLine(Org, Dst);
				end;

			Board(Dst):= Piece;			\make move
			Board(Loc):= Empty;
			Loc:= Dst;

			if Illegal then			\testing for legality
				begin
				\if Board matches BoardT then clear Illegal flag
				for I:= 0, 77 do
					if Board(I) # BoardT(I) then I:= 32000;
				if I < 32000 then Illegal:= false;
				end;

			BackupValue:= StaticValue;
			if abs(Capture) < King then		\--- RECURSE ---
			    if Capture#0 & Depth<DepthCap ! Depth<DepthMax then
				BackupValue:= -TryMove(~White, -StaticValue, 0);

			if Depth=1 & DoInfo=1 then
				ShowVals(Loc, BackupValue, BestValue);

			if BackupValue > BestValue then
				begin
				if Depth = 1 then
				    begin
				    if DoInfo = 0 then
				    	BestValue:= BestValue + Ran(Random);
				    if BackupValue > BestValue then
					begin
					MoveSrc:= Org; MoveDst:= Dst; \save move
					MovedPiece:= Board(Loc);      \ to make
					end;
				    end;
				BestValue:= BackupValue;	\save best value
				end;

			\quit loop if piece can't continue in current direction
			if Sq \#Empty\ ! OneStep then quit;

			if BackupValue >= -Alpha(Depth-1) then
				[Dir:= Dir2;   quit];		\prune
			end
		else	begin			\reached Border or our piece
			if Sq # Border then 	\points for covering our piece
				Mobility:= Mobility + MobTbl(Dst);
			quit;
			end;
		end;	\loop

	Board(Org):= Piece;		\put piece back to its origin
	Board(Loc):= Capture;		\restore whatever was at the destination

	if Depth=1 & DoInfo=1 then
		begin
		DrawLine(Org, Loc);	\erase line
		DrawSquare(Loc);
		DrawPiece(Loc, Board(Loc));
		end;
	Dir:= Dir + 1;			\step by 1+1 to access words, not bytes
	end;
end;	\MovePiece



proc	MovePawn;		\Trial move for a pawn
begin
\Dir=0 is a straight-ahead move; Dir=1 or 2 is a diagonal capture move
for Dir:= 0, 2 do		\for all the directions in OffTblP...
	begin
	Loc:= Org;
	Capture:= Piece;	\(in case pawn cannot move in current direction)

	loop	begin
		Dst:= Loc + (if White then -OffTblP(Dir) else OffTblP(Dir));
		Sq:= Board(Dst);
		if Dir=0 & Sq=Empty !
		   Dir#0 & Sq#Empty & (Sq|White)<0 & Sq#Border \opponent\ then
			begin
			Capture:= Sq;				\(can be Empty)
			GoDeeper:= Capture;
			StaticValue:= abs(Capture) + StaticValue0;
			Mobility:= Mobility + MobTbl(Dst);

			if White then		\calculate pawn advancement
				StaticValue:= StaticValue - Dst - Dst + 120
			else	StaticValue:= StaticValue + Dst + Dst - 40;

			if Depth=1 & DoInfo=1 then
				begin
				DrawLine(Org, Loc);		\erase old line
				if Loc # Org then DrawSquare(Loc);
				DrawLine(Org, Dst);
				end;

			Board(Dst):= Piece;			\make move
			Board(Loc):= Empty;
			Loc:= Dst;

			\if 8th rank reached then turn pawn into a queen
			if Loc <= 17 then			\white
				begin
				GoDeeper:= true;
				if Loc <= 7 then
					begin
					Board(Loc):= -Queen;
					StaticValue:= StaticValue + (Queen-Pawn);
					end;
				end
			else if Loc >= 60 then			\black
				begin
				GoDeeper:= true;
				if Loc >= 70 then
					begin
					Board(Loc):= Queen;
					StaticValue:= StaticValue + (Queen-Pawn);
					end;
				end;

			if Illegal then			\testing for legality
				begin
				\if Board matches BoardT then clear Illegal flag
				for I:= 0, 77 do
					if Board(I) # BoardT(I) then I:= 32000;
				if I < 32000 then Illegal:= false;
				end;

			BackupValue:= StaticValue;
			if abs(Capture) < King then		\--- RECURSE ---
			    if GoDeeper#0 & Depth<DepthCap ! Depth<DepthMax then
				BackupValue:= -TryMove(~White, -StaticValue,
					if abs(Loc-Org)=20 then Loc else 0);

			if Depth=1 & DoInfo=1 then
				ShowVals(Loc, BackupValue, BestValue);

			if BackupValue > BestValue then
				begin
				if Depth = 1 then
				    begin
				    if DoInfo = 0 then
				    	BestValue:= BestValue + Ran(Random);
				    if BackupValue > BestValue then
					begin
					MoveSrc:= Org; MoveDst:= Dst; \save move
					MovedPiece:= Board(Loc);      \ to make
					end;
				    end;
				BestValue:= BackupValue;	\save best value
				end;

			if Dir \#0\ then quit;		\diagonal (capture) move
			if White then
				[if Loc < 50 then quit]	\not on its 3rd rank
			else	[if Loc > 27 then quit];\(black's move)

			if BackupValue >= -Alpha(Depth-1) then
				[Dir:= 2;   quit];	\prune
			end
		else	begin		\pawn can't move in direction Dir
			if Dir#0 & Sq#Border then	\points for attacking Sq
				begin
				Mobility:= Mobility + MobTbl(Dst);
				if Sq \#Empty\ then	\points for covering
					Mobility:= Mobility + MobTbl(Dst);
				end;
			quit;
			end;
		end;	\loop

	Board(Org):= Piece;		\put piece back to its origin
	Board(Loc):= Capture;		\restore what was at the square where
					\ the piece moved to (can be Empty)
	if Depth=1 & DoInfo=1 then
		begin
		DrawLine(Org, Loc);	\erase line
		DrawSquare(Loc);
		DrawPiece(Loc, Board(Loc));
		end;
	end;
end;	\MovePawn



proc	NullMove;		\Don't move the king but say we did
\This prevents stalemates from being considered checkmates. If the best move is
\ not to move the king then if Matey selects this move as its (Depth=1) move
\ then it's a stalemate. In other words, the only alternative was to move into
\ check. This move is scored very low so that it is only selected as an
\ alternative to a resignation. If this is the best move for a human (Depth=2)
\ then it is scored neutral (=0). The assumption is that Matey is ahead in
\ material, and it should not force a stalemate.
begin
StaticValue:= StaticValue0;
if Depth = 1 then 		\very bad score, but not a resignation
	StaticValue:= StaticValue - King>>2;

if Depth < DepthMax then					\--- RECURSE ---
	BackupValue:= -TryMove(~White, -StaticValue, 0)
else	BackupValue:= StaticValue;

if BackupValue > BestValue then
	begin
	BestValue:= BackupValue;		\save best value
	if Depth = 1 then
		begin
		MoveSrc:= Org; MoveDst:= Org;	\fake move for MakeMove
		MovedPiece:= Board(Org);
		end;
	end;
end;	\NullMove



begin	\TryMove
Depth:= Depth + 1;
BestValue:= -32000;			\assume the worst
Alpha(Depth):= BestValue;
Mobility:= 0;
for Inx:= 0, 64-1 do			\for all the squares...
	begin
	if Depth=1 & DepthMax>1 then	\show progress of computer's thinking
		begin			\ (also keeps horrid Windows XP awake)
		MY:= GetMousePosition(1);
		if MY >= 28*16 then ShowMouse(false);	\avoid flicker
		Cursor(26, 29);
		if 63-Inx < 10 then ChOut(1, ^ );
		IntOut(1, 63-Inx);

		TickTock(White);

		if ChkKey then
			if ChIn(1) = Esc then Exit(" "); \(no echo)
		if MY >= 28*16 then ShowMouse(true);
		end;

	Org:= BoardTbl(Inx);
	Piece:= Board(Org);		\scan board for one of our pieces
	if Piece#Empty & (Piece|White)>=0 then
		begin			\square contains our piece
		case abs(Piece) of	\make tenative move
		  Pawn:		begin
				MovePawn;
				\if opponent's pawn moved 2 squares, en passant?
				if EPDst#0 & (EPDst+1=Org ! EPDst-1=Org) then
					begin	\square number that was skipped
					EPSkip:= if EPDst>=40 then EPDst+10
							      else EPDst-10;
					\move opponent's pawn back a square
					Board(EPSkip):= Board(EPDst);
					Board(EpDst):= Empty;
					MovePawn;
					Board(EpDst):= Board(EPSkip); \put back
					Board(EPSkip):= Empty;
					if Depth=1 & DoInfo=1 then
						DrawSquare(EPSkip);
					end;
				end;
		  Knight:	MovePiece(addr OffTblK(0),addr OffTblK(7), true);
		  Bishop:	MovePiece(addr OffTbl(4), addr OffTbl(7), false);
		  Rook:		MovePiece(addr OffTbl(0), addr OffTbl(3), false);
		  Queen:	MovePiece(addr OffTbl(0), addr OffTbl(7), false);
		  King:	       [MovePiece(addr OffTbl(0), addr OffTbl(7), true);
				NullMove]
		other		Exit("Board is corrupt");	\fatal error

		if BestValue >= -Alpha(Depth-1) then Inx:= 64;	\prune
		end;
	end;

if Depth <= DepthMax then BestValue:= BestValue + Mobility;
Alpha(Depth):= BestValue;
Depth:= Depth - 1;
return BestValue;
end;	\TryMove

\===============================================================================

func	Check(White);	\Returns value of move. Used by InCheck and LegalMove.
int	White;		\set 'true' for white's move; 'false' for black's
begin
Depth:= DepthCap-1;		\only search one level deep
if Depth = 0 then Depth:= 1;	\(Level = 1 is allowed)
Alpha(Depth):= -32000;		\make sure there are no cutoffs (pruning)
if White then
	return TryMove(true,  0, if MovedPiece=+Pawn &
		MoveDst-MoveSrc=+20 then MoveDst else 0)   \en passant stuff
else	return TryMove(false, 0, if MovedPiece=-Pawn &
		MoveDst-MoveSrc=-20 then MoveDst else 0);
end;	\Check



func	InCheck;	\Returns true if a king is in check
return Check(true)>=King/2 ! Check(false)>=King/2;



proc	PrintPiece(OD, Piece);	\Outputs the letter corresponding to Piece
int	OD, Piece;	\output device number
int	Tbl;
char	Tbl2;
int	I;
begin
Tbl:= [Empty, Pawn, Rook, Knight, Bishop, Queen, King];
Tbl2:= "  RNBQK ";
for I:= 0, 6 do
	if abs(Piece) = Tbl(I) then ChOut(OD, Tbl2(I));
end;	\PrintPiece



proc	PrintMove(Src, Dst, Cap); \Print move using (somewhat) standard notation
\ e.g: 1. Rf6- g6  Qb3xNf6
int	Src, Dst, Cap;
int	OD;	\output device number
begin
OD:= 6;		\output to screen
loop	begin
	if Src = Dst then
		Text(OD, "
     Stalemate")
	else if MoveValue <= -King/2 then
		Text(OD, "Resign   ")		\also avoids showing stupid move
	else	begin
		if Board(Dst)=-King & Src=74 & Dst=76 !
		   Board(Dst)=+King & Src=04 & Dst=06 then	\castle
			Text(OD, "O-O    ")
		else if Board(Dst)=-King & Src=74 & Dst=72 !
			Board(Dst)=+King & Src=04 & Dst=02 then
			Text(OD, "O-O-O  ")
		else	begin
			PrintPiece(OD, Board(Dst));
			ChOut(OD, rem(Src/10) + ^a);
			ChOut(OD, ^8 - Src/10);
			ChOut(OD, if Cap=Empty then ^- else ^x);
			PrintPiece(OD, Cap);
			ChOut(OD, rem(Dst/10) + ^a);
			ChOut(OD, ^8 - Dst/10);
			end;
		ChOut(OD, if InCheck then ^+ else ^ );
		ChOut(OD, ^ );
		end;

	if OD = 3 then quit;
	OD:= 3;		\do a second pass, but output to disk file instead
	end;

Text(OD, "(");   IntOut(OD, MoveValue);
Text(OD, ")	");
end;	\PrintMove



proc	PrintMoveCount;		\Print move count at start of line (e.g: "1. ")
int	OD;	\output device number
begin
OD:= 6;
loop	begin
	if MoveCount < 100 then ChOut(OD, ^ ); \right-justify 3 digits
	if MoveCount < 10 then ChOut(OD, ^ );
	IntOut(OD, MoveCount);   Text(OD, ". ");
	if OD = 3 then quit;
	OD:= 3;
	end;
end;	\PrintMoveCount

\-------------------------------------------------------------------------------

proc	MakeMove(White);	\Make actual move of a piece on the board
int	White;		\is 'true' for white's move; 'false' for black's
int	P, S, I, J;
begin
P:= King;   S:= 0;
if White then [P:= -King;   S:= 70];
if MoveValue <= -King/2 then KillKing(P);

MoveCap:= Board(MoveDst);			\make move on the Board
Board(MoveDst):= MovedPiece;
Board(MoveSrc):= Empty;

for I:= 0, 2 do					\flash piece & make rising sound
	begin					\ so human notices it
	DrawPiece(MoveSrc, MovedPiece);
	Sound(false, 1, 1);
	DrawSquare(MoveSrc);
	J:= if White then 2-I else I;
	Sound(DoSound&J, 1, 4000*J);
	end;
DrawSquare(MoveDst);				\make move on the screen
DrawPiece(MoveDst, MovedPiece);

if MovedPiece=P & MoveSrc=S+4 then		\castle move
	begin
	if MoveDst = S+6 then
		begin
		Board(S+5):= Board(S+7);	\king side
		Board(S+7):= Empty;
		DrawSquare(S+4);
		DrawPiece(S+6, P);
		DrawSquare(S+7);
		DrawPiece(S+5, Board(S+5));
		end;
	if MoveDst = S+2 then
		begin
		Board(S+3):= Board(S+0);	\queen side
		Board(S+0):= Empty;
		DrawSquare(S+4);
		DrawPiece(S+2, P);
		DrawSquare(S+0);
		DrawPiece(S+3, Board(S+3));
		end;
	end;

\En passant capture:
P:= Pawn;   S:= -10;
if White then [P:= -P;   S:= -S];
\Pawn moved to an empty square, and it didn't move straight ahead
if MovedPiece=P & MoveCap=Empty & rem(MoveSrc/10) # rem(MoveDst/10) then
	begin
	MoveCap:= -P;
	Board(MoveDst+S):= Empty;
	DrawSquare(MoveDst+S);
	end;

ShowMouse(false);
PrintMove(MoveSrc, MoveDst, MoveCap);

RecordCap(MoveCap);
ShowCaps;
ShowMouse(true);
if InCheck then
	begin
	Sound(false, 7, 1000);
	Sound(DoSound, 1, 2000);
	Sound(false, 1, 1);
	Sound(DoSound, 1, 2000);
	end;
end;	\MakeMove

\===============================================================================

proc	UndoMove;		\Swap Board with (backup) Board2, etc.
int	T;
begin				\only the pointers to the arrays need be swapped
T:= Board;   Board:= Board2;   Board2:= T;
T:= MoveCount;   MoveCount:= MoveCount2;   MoveCount2:= T;
T:= CapWh;   CapWh:= CapWh2;   CapWh2:= T;
T:= CapBl;   CapBl:= CapBl2;   CapBl2:= T;
T:= Resigned;   Resigned:= Resigned2;   Resigned2:= T;

T:= MayCastleBlackKingSide;   MayCastleBlackKingSide:= MayCastle2BlackKingSide;
MayCastle2BlackKingSide2:= T;
T:= MayCastleBlackQueenSide; MayCastleBlackQueenSide:= MayCastle2BlackQueenSide;
MayCastle2BlackQueenSide2:= T;
T:= MayCastleWhiteKingSide;   MayCastleWhiteKingSide:= MayCastle2WhiteKingSide;
MayCastle2WhiteKingSide2:= T;
T:= MayCastleWhiteQueenSide; MayCastleWhiteQueenSide:= MayCastle2WhiteQueenSide;
MayCastle2WhiteQueenSide2:= T;

ShowCaps;
DrawBoard;
end;	\UndoMove



func	LegalMove(White, Src, Dst);	\Returns true if human's move is legal
int	White,		\boolean: human plays white pieces (else black pieces)
	Src, Dst;	\tenative source and destination square numbers
int	I, T,
	MVal,		\move value
	IC;		\in check
begin
if Src<0 ! Src>77 ! Dst<0 ! Dst>77 then return false;
if Board(Src)=Border ! Board(Dst)=Border then return false;

\make human's move on the temporary board
for I:= 0, 77 do BoardT(I):= Board(I);
BoardT(Dst):= BoardT(Src);
BoardT(Src):= Empty;

if White then		\if pawn reached 8th rank then poof! It's a queen.
	[if Dst<=7  & BoardT(Dst)=-Pawn then BoardT(Dst):= -Queen]
else	[if Dst>=70 & BoardT(Dst)=+Pawn then BoardT(Dst):= +Queen];

\En passant capture:
\Pawn moved to an empty square and it didn't move straight ahead
if Board(Dst)=Empty & rem(Src/10)#rem(Dst/10) then
	begin
	if  White & BoardT(Dst)=-Pawn then BoardT(Dst+10):= Empty;
	if ~White & BoardT(Dst)=+Pawn then BoardT(Dst-10):= Empty;
	end;

\see if tenative human move on the temporary board can be duplicated by TryMove
Illegal:= true;
Check(White);
if not Illegal then
	begin		\human's move could be duplicated, but
	T:= Board;	\ see if his king moved into check
	Board:= BoardT;	\use temporary Board
	MVal:= Check(~White);
	Board:= T;	\restore original Board
	if MVal < King/2 \not in check\ then return true;
	end;

\couldn't duplicate move, but see if human made a legal castling move
if White then
	begin
	if BoardT(Dst)=-King & Src=74 then	\moved white king
		begin
		if Dst=76 & MayCastleWhiteKingSide & Board(75)=Empty &
		    Board(76)=Empty then
			begin
			Board(75):= -King;  Board(76):= -King;
			IC:= InCheck;		\the 3 squares can't be attacked
			Board(75):= Empty;  Board(76):= Empty;
			return not IC;
			end;
		if Dst=72 & MayCastleWhiteQueenSide & Board(73)=Empty &
		    Board(72)=Empty & Board(71)=Empty then
			begin
			Board(73):= -King;  Board(72):= -King;
			IC:= InCheck;		\the 3 squares can't be attacked
			Board(73):= Empty;  Board(72):= Empty;
			return not IC;
			end;
		end;
	end
else	begin
	if BoardT(Dst)=+King & Src=04 then	\moved black king
		begin
		if Dst=06 & MayCastleBlackKingSide & Board(05)=Empty &
		    Board(06)=Empty then
			begin
			Board(05):= +King;  Board(06):= +King;
			IC:= InCheck;		\the 3 squares can't be attacked
			Board(05):= Empty;  Board(06):= Empty;
			return not IC;
			end;
		if Dst=02 & MayCastleBlackQueenSide & Board(03)=Empty &
		    Board(02)=Empty & Board(01)=Empty then
			begin
			Board(03):= +King;  Board(02):= +King;
			IC:= InCheck;		\the 3 squares can't be attacked
			Board(03):= Empty;  Board(02):= Empty;
			return not IC;
			end;
		end;
	end;

return false;		\couldn't duplicate the move, so it must be illegal
end;	\LegalMove



proc	GetHumanMove(White);	\Get human's move using the mouse
\Outputs: MoveSrc, MoveDst, MovedPiece and MoveValue
int	White;		\boolean: human plays white pieces (else black pieces)
int	I,
	MX, MY,		\mouse coordinates (pixels and squares)
	Src, Dst;	\tenative source and destination square numbers


	func	OnHumanPiece;	\Return true if one of human's pieces is on Src
	begin
	if Src<0 ! Src>77 then return false;	\check for valid Src square
	if Board(Src)=Border then return false;
	return White & Board(Src)<0 ! ~White & Board(Src)>0;
	end;	\OnHumanPiece


begin	\GetHumanMove
loop	begin				\get human's move
	CallInt($33, 9, 0, 0, PtrArrow, 0, 0, DataSeg);
	ShowMouse(false);		\dither mouse for stupid Windows XP
	ShowMouse(true);
	repeat TickTock(White) until GetMouseButton;	\wait for press
	MX:= (GetMousePosition(0) - OffsetX) / SideLen;
	MY:= (GetMousePosition(1) - OffsetY) / SideLen;
	Src:= MX + MY*10;

	\if mouse clicked on a human's piece then show the hand symbol
	if OnHumanPiece then
		begin
		CallInt($33, 9, 0, 0, PtrHand, 0, 0, DataSeg);
		ShowMouse(false);	\change mouse pointer to a grasping hand
		ShowMouse(true);	\Windows XP is stupid and needs jostling
		if DoInfo = 2 then	\outline squares he may move to
		    begin
		    OutlineSquare(Src);	\outline selected piece
		    for I:= 0, 77 do
			if LegalMove(White, Src, I) then OutlineSquare(I);
		    end;
		end;

	while GetMouseButton do [TickTock(White)];	\wait for release
	MX:= GetMousePosition(0);
	MY:= GetMousePosition(1);

	\check if a soft button was clicked (pressed and released)
	if MX>=ResetX & MX<ResetX+ButtonWidth &
	   MY>=ResetY & MY<ResetY+ButtonHeight then
	   	[FClose(Handle);  Restart];
	if MX>=SoundX & MX<SoundX+ButtonWidth &
	   MY>=SoundY & MY<SoundY+ButtonHeight then
		begin
		DoSound:= ~DoSound;
		DrawRectangle(SoundX, SoundY, ButtonWidth-1, ButtonHeight-1,
			$86, true);	\xor to turn button ghosting on and off
		end;
	if MX>=InfoX & MX<InfoX+ButtonWidth &
	   MY>=InfoY & MY<InfoY+ButtonHeight then
		begin
		DoInfo:= DoInfo+1;  if DoInfo > 2 then DoInfo:= 0;
	   	if DoInfo=1 & Human1st & ~Matey2nd then	\person vs. person
		    [DoInfo:= 2;			\skip Matey's thinking
		    DrawRectangle(InfoX, InfoY, ButtonWidth-1, ButtonHeight-1,
				   $86, true)];		\xor to un-ghost button
		case DoInfo of
		  1: DrawRectangle(InfoX, InfoY, ButtonWidth-1, ButtonHeight-1,
				   $86, true);		\xor to un-ghost button
		  2: [DrawRectangle(InfoX, InfoY, ButtonWidth-1, ButtonHeight-1,
				   $85, false);		\xor to outline button
		      DrawRectangle(InfoX+1, InfoY+1, ButtonWidth-3,
					ButtonHeight-3, $AA85, false)];
		  0: [DrawRectangle(InfoX, InfoY, ButtonWidth-1, ButtonHeight-1,
				   $85, false);		\xor to remove outline
		      DrawRectangle(InfoX+1, InfoY+1, ButtonWidth-3,
					ButtonHeight-3, $AA85, false);
		      DrawRectangle(InfoX, InfoY, ButtonWidth-1, ButtonHeight-1,
				   $86, true)]		\xor to ghost button
		other [];
		end;
	if MX>=UndoX & MX<UndoX+ButtonWidth &
	   MY>=UndoY & MY<UndoY+ButtonHeight then    \can't undo human vs. human
	   	if not Human1st ! Matey2nd then UndoMove;
	if MX>=ExitX & MX<ExitX+ButtonWidth &
	   MY>=ExitY & MY<ExitY+ButtonHeight then Exit(" ");

	MX:= (MX - OffsetX) / SideLen;	\convert pixel coords to square coords
	MY:= (MY - OffsetY) / SideLen;
	Dst:= MX + MY*10;

	\was mouse button released while pointer was still on human's piece?
	if Dst=Src & OnHumanPiece then	\didn't drag piece, so destination needs
		begin			\ a second click
		\wait for 2nd press and release to get destination square
		repeat TickTock(White) until GetMouseButton;
		while GetMouseButton do [TickTock(White)];
		MX:= (GetMousePosition(0) - OffsetX) / SideLen;
		MY:= (GetMousePosition(1) - OffsetY) / SideLen;
		Dst:= MX + MY*10;
		end;

	if LegalMove(White, Src, Dst) then quit;

	if DoInfo=2 & Src<=77 then DrawBoard;	\remove outlined squares
	if Dst <= 77 then Beep;			\don't beep if soft button
	end;	\loop until a piece is moved (not just a soft button pressed)

if DoInfo = 2 then DrawBoard;			\remove outlined squares
MoveSrc:= Src;   MoveDst:= Dst;			\tenative move is now official

CallInt($33, 9, 0, 0, PtrHour, 0, 0, DataSeg);
ShowMouse(false);	\change mouse pointer to an hourglass
ShowMouse(true);	\Windows XP is stupid and needs jostling

for I:= 0, 77 do	\save current state for the Undo command
	Board2(I):= Board(I);
MoveCount2:= MoveCount;
for I:= 0, 5 do
	begin
	CapWh2(I):= CapWh(I);
	CapBl2(I):= CapBl(I);
	end;
Resigned2:= Resigned;
MayCastle2BlackKingSide:= MayCastleBlackKingSide;
MayCastle2BlackQueenSide:= MayCastleBlackQueenSide;
MayCastle2WhiteKingSide:= MayCastleWhiteKingSide;
MayCastle2WhiteQueenSide:= MayCastleWhiteQueenSide;

MovedPiece:= Board(MoveSrc);
if White then		\if pawn reached 8th rank then poof!
	[if MoveDst<= 7 & MovedPiece = -Pawn then MovedPiece:= -Queen]
else	[if MoveDst>=70 & MovedPiece = +Pawn then MovedPiece:= +Queen];

if MovedPiece = King then		\if black moved king then can't castle
	[MayCastleBlackKingSide:= false;  MayCastleBlackQueenSide:= false]
else if MovedPiece = -King then		\if white moved king then can't castle
	[MayCastleWhiteKingSide:= false;  MayCastleWhiteQueenSide:= false];

if White then		\if player moved rook then can't castle on that side
	begin
	if MoveSrc=77 then MayCastleWhiteKingSide:= false;
	if MoveSrc=70 then MayCastleWhiteQueenSide:= false;
	end
else	begin
	if MoveSrc=07 then MayCastleBlackKingSide:= false;
	if MoveSrc=00 then MayCastleBlackQueenSide:= false;
	end;

MoveValue:= 0;		\(unknown, use 0 for output file)
end;	\GetHumanMove

\-------------------------------------------------------------------------------

proc	GetBlackMove;		\Get and make black's move
begin
ElapsedTime(TimeOfDay, TimerBlack, Time0);
SecOld:= -1;			\display timer immediately

if Matey2nd then
	begin
	Delay(10);				\don't respond too quickly
	Depth:= 0;				\ (it unnerves the human)
	MoveValue:= TryMove(false, 0,
	  if MovedPiece=-Pawn & MoveSrc-MoveDst=20 then MoveDst else 0);
	end	\if white pawn advanced two squares then search for en passant
else	GetHumanMove(false);			\human plays black

Cursor(CursorX+14, CursorY);			\list black's move
MakeMove(false);
CrLf(3);

GetTimeOfDay;			\get Black's total time (for complete accuracy)
ElapsedTime(TimeOfDay, Time0, TimerBlack);
end;	\GetBlackMove



proc	GetWhiteMove;		\Get and make white's move
begin
ElapsedTime(TimeOfDay, TimerWhite, Time0);
SecOld:= -1;			\display timer immediately

if Human1st then
	GetHumanMove(true)
else	begin					\computer plays white
	Delay(10);				\don't respond too quickly
	Depth:= 0;
	MoveValue:= TryMove(true, 0,
	  if MovedPiece=Pawn & MoveDst-MoveSrc=20 then MoveDst else 0);
	end;	\if black pawn advanced two squares then search for en passant

CursorY:= CursorY + 1;				\list white's move
if CursorY > LineMax then CursorY:= LineMax;
Cursor(CursorX, CursorY);
MoveCount:= MoveCount + 1;
ShowMouse(false);
PrintMoveCount;
ShowMouse(true);
MakeMove(true);

GetTimeOfDay;			\get White's total time (for complete accuracy)
ElapsedTime(TimeOfDay, Time0, TimerWhite);
end;	\GetWhiteMove

\-------------------------------------------------------------------------------

proc	DrawScreen;		\Draw board with borders and all of its pieces
char	Str;
int	I;
begin
Attrib($5E);			\dark brown on white
Text(6,
"    M A T E Y    1.4                                   00:00:00    00:00:00     ");

DrawBoard;
				\draw top and bottom borders
DrawSprite(OffsetX-24, OffsetY-24, SpBorder, true, false);
DrawSprite(OffsetX+4*SideLen, OffsetY-24, SpBorder, true, false);
DrawSprite(OffsetX-24, OffsetY+8*SideLen, SpBorder, true, false);
DrawSprite(OffsetX+4*SideLen, OffsetY+8*SideLen, SpBorder, true, false);

Attrib($EE);
Cursor(OffsetX/8+3, OffsetY/16-1);
Str:= "a     b     c     d     e     f     g     h";
Text(6, Str);
Cursor(OffsetX/8+3, OffsetY/16+3*8);
Text(6, Str);
				\draw side borders
DrawSprite(OffsetX-24, OffsetY-24, SpBorder, true, true);
DrawSprite(OffsetX-24, OffsetY+4*SideLen, SpBorder, true, true);
DrawSprite(OffsetX+8*SideLen, OffsetY-24, SpBorder, true, true);
DrawSprite(OffsetX+8*SideLen, OffsetY+4*SideLen, SpBorder, true, true);

for I:= 0, 7 do
	begin
	Cursor(OffsetX/8-2, OffsetY/16+3*I+1);
	ChOut(6, ^8-I);
	Cursor(OffsetX/8+48/8*8+1, OffsetY/16+3*I+1);
	ChOut(6, ^8-I);
	end;
				\sex it up
DrawRectangle(OffsetX-1, OffsetY-1, SideLen*8+2, SideLen*8+2, 9, false);
for I:= 0, 4 do
	DrawRectangle(OffsetX-24-I, OffsetY-24-I, SideLen*8+48+I+I,
		SideLen*8+48+I+I, 9+I, false);

Attrib($5E);				\display soft buttons
Cursor(ResetX/8, ResetY/16); Text(6, " Reset ");   ShadeButton(ResetX, ResetY);
Cursor(SoundX/8, SoundY/16); Text(6, " Sound ");   ShadeButton(SoundX, SoundY);
Cursor(UndoX/8,  UndoY/16);  Text(6, " Undo  ");   ShadeButton(UndoX,  UndoY);
Cursor(ExitX/8,  ExitY/16);  Text(6, " Exit  ");   ShadeButton(ExitX,  ExitY);
Cursor(InfoX/8,  InfoY/16);  Text(6, " Info  ");   ShadeButton(InfoX,  InfoY);
DrawRectangle(InfoX, InfoY, ButtonWidth-1, ButtonHeight-1, $85, false);
DrawRectangle(InfoX+1, InfoY+1, ButtonWidth-3, ButtonHeight-3, $AA85, false);
end;	\DrawScreen



proc	Initialize;		\Start a game: set up the board, etc.
int	I, Pieces;
begin
\Move offset table for Rook, Queen, King, and Bishop:
OffTbl:= [+1, +10, -1, -10, +11, +9, -11, -9];
\Move offset table for Knight:
OffTblK:= [+12, +21, +19, +8, -12, -21, -19, -8];
\Move offset table for black's Pawn (use negative values for white's):
OffTblP:= [10, 11, 9];
\Mobility Table: emphasizes control of the center of the board
\		 a  b  c  d  e  f  g  h
MobTbl:= [	M0,M0,M0,M0,M0,M0,M0,M0,0,0,	\8
		M0,M0,M0,M0,M0,M0,M0,M0,0,0,	\7
		M0,M0,M0,M1,M1,M0,M0,M0,0,0,	\6
		M0,M0,M1,M2,M2,M1,M0,M0,0,0,	\5
		M0,M0,M1,M2,M2,M1,M0,M0,0,0,	\4
		M0,M0,M0,M1,M1,M0,M0,M0,0,0,	\3
		M0,M0,M0,M0,M0,M0,M0,M0,0,0,	\2
		M0,M0,M0,M0,M0,M0,M0,M0     ];	\1

\Board square numbers, beginning with most likely move for black (for pruning)
BoardTbl:=     [43, 44, 33, 34, 53, 54, 23, 24,
		52, 55, 42, 45, 32, 35, 22, 25,
		63, 64, 62, 65, 61, 66, 13, 14,
		12, 15, 11, 16, 51, 56, 41, 46,
		31, 36, 21, 26, 73, 74, 72, 75,
		71, 76, 70, 77, 60, 67, 50, 57,
		40, 47, 30, 37, 20, 27, 10, 17,
		03, 04, 02, 05, 01, 06, 00, 07];

\Initialize the Board to Empty with borders
for I:= 0, 10*12-1 do Board(I):= Border;
Board:= Board + 21*IntSize;	\WARNING: Board address is moved to playing area
for I:= 0, 77 do
	if rem(I/10) <= 7 then Board(I):= Empty;

\Set up the pieces in their starting positions:
Pieces:=[Empty, Pawn, Rook, Knight, Bishop, Queen, King, Bishop, Knight, Rook];
for I:= 0, 7 do Board(I):= Pieces(I+IntSize);
for I:= 10, 17 do Board(I):= Pawn;
for I:= 60, 67 do Board(I):= -Pawn;
for I:= 70, 77 do Board(I):= -Pieces(I+IntSize-70);

for I:= 0, 10*12-1 do		\save current state for Undo command
	Board2(I):= Board(I-21);
Board2:= Board2 + 21*IntSize;	\WARNING: moved to playing area

for I:= 0, 10*12-1 do		\save a copy with the top border
	BoardT(I):= Board(I-21);\ the rest doesn't matter for now
BoardT:= BoardT + 21*IntSize;	\WARNING: moved to playing area

for I:= 0, 5 do			\no captures...yet
	begin
	CapWh(I):= 0;
	CapBl(I):= 0;
	CapWh2(I):= 0;
	CapBl2(I):= 0;
	end;

Alpha(0):= -32000;

DoInfo:= 2;
DoSound:= true;
MoveCount:= 0;
MoveCount2:= 0;
MovedPiece:= 0;			\(in case of autoplay)
Illegal:= false;
Resigned:= Empty;
Resigned2:= Empty;
CursorX:= 1;   CursorY:= 7;
MayCastleBlackKingSide:= true;  MayCastleBlackQueenSide:= true;
MayCastle2BlackKingSide:= true;  MayCastle2BlackQueenSide:= true;
MayCastleWhiteKingSide:= true;  MayCastleWhiteQueenSide:= true;
MayCastle2WhiteKingSide:= true;  MayCastle2WhiteQueenSide:= true;

for I:= 0, 3 do [TimerWhite(I):= 0;  TimerBlack(I):= 0];
end;	\Initialize

\-------------------------------------------------------------------------------

begin	\Main
CpuReg:= GetReg;
DataSeg:= CpuReg(12);

if not OpenMouse then Exit("This program requires a mouse");
TrapC(true);		\prevent Ctrl+C from leaving screen in graphics mode

SetVid($12);				\640x480 graphics (clears screen)
LoadSprites;
AlignPalette;
Initialize;
DrawScreen;

Attrib($D7);				\white on brown
SetWind(1, 2, 22, LineMax, 0, true);
DrawRectangle(1*8-1, 2*16-1, 22*8+2, (LineMax+1-2)*16+2, $8, false);
\From this point on, device 6 writes into the window. Devices 0 & 1 are used
\ with the Cursor intrinsic to write anywhere on the screen, but the background
\ will be cyan.

Cursor(1, 3);
Text(6, "  P=Person, M=Matey");
Cursor(1, 4);
Text(6, "  First  (P/M)? ");
Human1st:= (KeyIn!$20) # ^m;			\default is p (=Person)
Cursor(1, 5);
Text(6, "  Second (M/P)? ");
Matey2nd:= (KeyIn!$20) # ^p;			\default is m (=Matey)
if not Human1st ! Matey2nd then			\not person vs. person
	begin
	repeat	Cursor(1, 6);
		Text(6, "  Level  (2-5)?  ");	\advised range
		ChOut(1, BS);			\back up over illegal keystroke
		DepthMax:= KeyIn - ^0;
	until	DepthMax>=1 & DepthMax<=7;	\allowed range
	end
else	begin	\human vs. human; ghost Undo button (because it can't work)
	DrawRectangle(UndoX, UndoY, ButtonWidth-1, ButtonHeight-1, $86, true);
	DepthMax:= 2;				\because of legal move checking
	end;
DepthCap:= DepthMax + DepthMax/2;		\captures cause deeper search

Handle:= FOpen("GAME.TXT", 1);	\save a list of the moves in an output file
FSet(Handle, ^o);
OpenO(3);
Text(3, "Level ");   IntOut(3, DepthMax);   CrLf(3);   CrLf(3);
Text(3, if Human1st then "     Person (white)" else "     Matey (white) ");
Text(3, if Matey2nd then "	Matey (black)" else "	Person (black)");
CrLf(3);

GetTimeOfDay;
if not Human1st & Matey2nd then			\auto-play mode
	begin
	loop	begin
		GetWhiteMove;
		GetBlackMove;
	   	if Resigned \#Empty\ then [FClose(Handle);  Restart];
		end;
	end
else	begin
	CallInt($33, 4, 0, 100, 150);		\start mouse pointer where it
	ShowMouse(true);			\ will be noticed
	loop	begin
		GetWhiteMove;
		GetBlackMove;
		end;
	end;
end;	\Main
