program Trapper;

(* Written by Darren Grey in March 2010.
   Created as part of the 7DRL 2010 competition.
   Some initial code ripped from my previous efforts with Gruesome and
   The Lion King.
   Released without licence - use as you wish, though I would like to be
   informed if you can make anything useful out of it!

   This is the Director's Cut, not the original 7drl version.  This
   release fixes some small bugs, adds a bit of polish, and corrects a
   couple of features that weren't fully implemented because I ending up
   rushing them at the end.  Gameplay wise it's mostly identical to the
   original.  See the history file for more details.
 *)


uses
   crt;  (* Basic display library.  Porting to curses may be faster on some
            systems. *)

const (* Global *)
   xmin = 1;
   xmax = 67;
   ymin = 4;
   ymax = 24; (* Map limits.  With sidebar comes to 80x24 - standard Unix
                 terminal size. *)
   DfTxC = 7;
   DfBgC = 0; (* Default text and background colours. *)


type (* Global *)

	text = ansistring;

	mapsquare = record
      Passable : boolean;
      Connected : boolean;
      Explored : boolean;
	   Undiggable : boolean;
      Visible : boolean;
      Drawn : boolean;
      VisLimit : boolean; (* These all start as false. *)
      cindex : longint; (* Reference to creature array. *)
	   Symbol : char; (* Display symbol *)
	   BGColour : byte; (* 0 to 15 *)
	   ForeColour : byte; (* 0 to 15 *)
      tileid : byte;
	   smell : longint;
   end;

   creaturetype = record
      Symbol : char;
      Colour : byte; (* 0 to 15. *)
      xpos : byte;
      ypos : byte;
	   Named : boolean;
	   Name : text;
	   prefix : text;
	   blindcount : byte;
      turnwait : byte;
      turndelay : byte;
      HP : byte; (* For the giants and Wizard Mode. *)
      jumping : boolean; (* For Jack. *)
      ctype : byte;
	
   end;

   mapcoords = array[xmin..xmax, ymin..ymax] of mapsquare;
   patharray = array of smallint;
   creaturearray = array of creaturetype; (* Used for player and monsters. *)


var (* Global *)
   coord : mapcoords;
   D, CaveColourLight, CaveColourDark : longint;
   move : char;
   i, a, b, visrange, turns, messagex, messagey : longint;
   turncount, wizardmode : boolean;
   activetrap, maxtrap, kills, cleankills, killer, trapslaid, detonated : longint;
   victorytype : longint;
   creatureindex : creaturearray;
   dasherkilled, giantkilled, shadowkilled, jesterkilled, warlockkilled,
      wraithkilled, gotlife, ultraused, lifebroken, playerdead, quit : boolean;



(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)

procedure StatusBar; FORWARD;


procedure DrawTile(x,y : byte);
(* Draws single tile in console.  Cursor must be set to correct location
   before calling this procedure. *)

begin
   if coord[x,y].Explored = false then
   begin
      TextColor(DfTxC);
	   write(' ')
   end
   else
   begin
      TextColor(coord[x,y].ForeColour);
      if ((coord[x,y].Visible = false) AND (coord[x,y].tileid in [1,2,4]))
	      then TextColor(CaveColourDark); (* Darken unseen dungeon *)
      if (coord[x,y].Visible AND (coord[x,y].cindex > 0)) then
	  (* Only draw creatures if you can see them *)
	   begin
	      TextColor(creatureindex[coord[x,y].cindex].Colour);
		   write(creatureindex[coord[x,y].cindex].Symbol)
      end
      else write(coord[x,y].Symbol);
   end;
end;

(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)


procedure DrawMap;
(* Draws map in console using global map array. *)

var
   x, y : byte;

begin
   GoToXY(xmin,ymin); (* Map display starts on the 4th row. *)
   for y := ymin to ymax do  (* Rows *)
   begin
      GoToXY(1,y);
      for x := xmin to xmax do  (* Columns *)
         DrawTile(x,y);
   end;
   TextColor(DfTxC);
   TextBackground(DfBgC); (* Resets to default colours. *)
end; (*** End of Procedure. ***)

	
(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)	


procedure DrawSurround (centrex,centrey,visrange : byte);
(* Instead of refreshing the whole map, this only redraws squares within
   a certain radius of the character.  Covers area just moved from too,
   so no artifacts should be left from previous vision. *)

var
   xa, ya, xb, yb, x, y : byte;

begin
   coord[centrex,centrey].Visible := false; (* Necessary to stop artifacts. *)
   
   if centrey - visrange - 1 < ymin then ya := ymin
      else ya := centrey - visrange - 1;
   if centrex - visrange - 1 < xmin then xa := xmin
      else xa := centrex - visrange - 1;
   if centrey + visrange + 1 > ymax then yb := ymax
      else yb := centrey + visrange + 1;
   if centrex + visrange + 1 > xmax then xb := xmax
      else xb := centrex + visrange + 1;  (* Ensures boundaries. *)
   for y := ya to yb do
   begin
      for x := xa to xb do
      begin
         if (coord[x,y].Drawn = false) and (coord[x,y].Visible = true) then
         begin
            GoToXY(x,y);
            DrawTile(x,y);
            coord[x,y].Drawn := true
         end
         else
         if (coord[x,y].Drawn = true) and (coord[x,y].Visible = false) then
         begin
            GoToXY(x,y);
            DrawTile(x,y);
            coord[x,y].Drawn := false;
         end;
      end;
   end;
   
   GoToXY(centrex,centrey);
   if creatureindex[1].blindcount = 0 then coord[centrex,centrey].Visible := true;
   DrawTile(centrex,centrey); (* Redraw player. *)
   
   TextColor(DfTxC); 
   TextBackground(DfBgC); (* Resets to default colours. *)
end;


(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)



procedure RevealMap;

(* Marks whole map as Explored, using a flood fill starting from the given
   point. *)

var
   a, b, c, d : byte;

begin
   for a := (ymin+1) to (ymax-1) do
   begin
      for b := (xmin+1) to (xmax-1) do
      begin
         if coord[b,a].Passable = true then
         begin
            for c := (a-1) to (a+1) do
               for d := (b-1) to (b+1) do
                  coord[d,c].Explored := true;
         end;
      end;
   end;
   DrawMap;
end; (*** End of procedure. ***)



(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)


procedure WipeMap;

var
   a, b : byte;

begin
   for b := ymin to ymax do
   begin
      for a := xmin to xmax do
      begin
         with coord[a,b] do
         begin
            Explored := false;
            Passable := false;
            Connected := false;
	         Undiggable := false;
            Visible := false;
            Drawn := false;
            cindex := 0;
			   Symbol := ' ';
			   BGColour := 0;
			   ForeColour := 0;
			   tileid := 0;
			   smell := 0;
         end;
      end; (* All squares are reset to null. *)
   end;
end;


procedure ClearMessageBar;
(* Clears the first 3 lines and sets the cursor on 1,1 for new messages. *)

begin
  messagex := 1;
  messagey := 1;
  TextColor(7);
  TextBackground(0);
  GoToXY(1,1);
  ClrEol;
  GoToXY(1,2);
  ClrEol;
  GoToXY(1,3);
  ClrEol;
  GoToXY(1,1);
end;

(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)





(* Now beginning Line of Sight algorithms.  Makes use of line drawing and
   circle drawing, which both produce results in arrays of co-ordinates.
   Arrays must be declared within main LoS procedure, and passed by
   reference to the subprocedures. *)


procedure MarkStep (x1, y1 : longint; yslope : boolean;
                    VAR xpath,ypath : patharray; VAR Li : longint); FORWARD;
procedure StraightLine (x1,y1,x2,y2 : longint; yslope : boolean;
                    VAR xpath,ypath : patharray; VAR Li : longint); FORWARD;
procedure Circle (xstart,ystart,r : longint;
                    VAR circlepathx,circlepathy : patharray); FORWARD;
procedure VisLine (x1,y1,x2,y2,r : longint); FORWARD;
(* Marking for future use. *)


procedure LineofSight2 (centrex, centrey, visrange : longint);
(* Fairly accurate but possibly slow ray-casting algorithm.
   Written from scratch and probably inefficient as hell, but it's mine  :) *)

var
   a, b, xa, xb, ya, yb, res : longint;

begin
   if centrey - visrange - 1 < ymin then ya := ymin
      else ya := centrey - visrange -1;
   if centrex - visrange - 1 < xmin then xa := xmin
      else xa := centrex - visrange -1;
   if centrey + visrange + 1 > ymax then yb := ymax
      else yb := centrey + visrange +1;
   if centrex + visrange + 1 > xmax then xb := xmax
      else xb := centrex + visrange +1;
   for a := ya to yb do
      for b := xa to xb do
         coord[b,a].Visible := false; (* Clears previous visible flags. *)

   res := 10; (* Resolution - affects how many lines are drawn in calculating
                 viewable squares.  Diminishing returns for exponentially
                 increasing CPU time as it rises.
                 This also acts as a maximum sight range that can be set
                 for the player. *)
   b := centrey - res;
   for a := (centrex - res) to (centrex + res) do
      VisLine(centrex,centrey,a,b,visrange);
   for b := (centrey - res)  to (centrey + res) do
      VisLine(centrex,centrey,a,b,visrange);
   for a := (centrex + res) downto (centrex - res) do
      VisLine(centrex,centrey,a,b,visrange);
   for b := (centrey + res) downto (centrey - res) do
      VisLine(centrex,centrey,a,b,visrange);
end;


procedure VisLine (x1,y1,x2,y2,r : longint);

var
   xpath, ypath, circlepathx, circlepathy : patharray;
   yslope, finished : boolean;
   Li, b : longint;

begin
   Li := 0;
   if abs(y2-y1) > abs(x2-x1) then yslope := true
      else yslope := false;  (* Tests for yslope to see if calling values
                                should be reversed for StraightLine. *)
   if yslope then
      StraightLine(y1,x1,y2,x2,yslope,xpath,ypath,Li)
   else StraightLine(x1,y1,x2,y2,yslope,xpath,ypath,Li);
   Circle(x1,y1,r,circlepathx,circlepathy);
   for b := 0 to (r*8-1) do
      coord[circlepathx[b],circlepathy[b]].VisLimit := true;
         (* Limit on sight vision is marked by a circle matching the sight
            radius.  This produces nicer circles than bloody trig. *)
   b := -1;
   finished := false;
   repeat
      b := b + 1;
      coord[xpath[b],ypath[b]].Explored := true;
      coord[xpath[b],ypath[b]].Visible := true;
      if (coord[xpath[b],ypath[b]].Passable = false)
         or (coord[xpath[b],ypath[b]].VisLimit = true) then finished := true;
   until finished;
      (* Each point on line is marked visible and explored.
         Line ended when range reached or wall hit. *)
   for b := 0 to (r*8-1) do
      coord[circlepathx[b],circlepathy[b]].VisLimit := false;
      (* Clears VisLimits for next time. *)
end;



(* Geometrical subroutines: *)

(* NOTE - calling function must include xpath, ypath and yslope variables.
   Must call StraightLine with potentially reversed y values. *)

procedure StraightLine (x1,y1,x2,y2 : longint; yslope : boolean;
                    VAR xpath,ypath : patharray; VAR Li : longint);
(* Creates a straight line between two points.  Works in 360 degrees as long
   as the x and y values are swapped upon calling if the y distance is
   greater. *)

var
   stepsize, remainder, countmax, n, xdiff, ydiff : longint;
   stepcount, xdir, ydir : longint;

begin
   if x2 > x1 then xdir := 1 else xdir := -1;
   if y2 > y1 then ydir := 1 else ydir := -1;
   xdiff := abs(x2 - x1) + 1;
   ydiff := abs(y2 - y1) + 1;
   SetLength(xpath,xdiff);
   SetLength(ypath,xdiff);
   Li := -1;
   if ydiff = 0 then
   begin
      stepsize := 0;
      remainder := 0;
      countmax := 1;
   end
   else
   begin
      stepsize := xdiff div ydiff;
      remainder := xdiff mod ydiff;
      countmax := ydiff div 2;
   end;
   stepcount := 0; (* End of initial set-up. *)

   repeat
      MarkStep(x1,y1,yslope,xpath,ypath,Li);
      n := 1;
      while n < stepsize do
      begin
         x1 := x1 + xdir;
         n := n + 1;
         Markstep(x1,y1,yslope,xpath,ypath,Li);
      end; (* End of while.  This segment completes long straight parts of
              the line. *)
      if x1 <> x2 then (* Necessary check on whether to continue or not. *)
      begin
         stepcount := stepcount + remainder;
         if stepcount > countmax then
         begin
            stepcount := stepcount - ydiff;
            x1 := x1 + xdir;
            Markstep(x1,y1,yslope,xpath,ypath,Li);
         end;
      end; (* End of both ifs.  This section increases the stepcount on each
              loop until it reaches the limit where an additional x step is
              needed.  It reduces itself when this happens so the loop
              carries on.  Can tweak countmax to get different lines.  *)
      if x1 <> x2 then
      begin
         x1 := x1 + xdir;
         y1 := y1 + ydir; (* Diagonal shift after all x shifts done.
                             May be redundant if target reached. *)
         if x1 = x2 then Markstep(x1,y1,yslope,xpath,ypath,Li);
      end;
   until x1 = x2; (* Necessary end condition. *)
end; (*** End of procedure. ***)


procedure Markstep (x1, y1 : longint; yslope : boolean;
                    VAR xpath,ypath : patharray; VAR Li : longint);
(* Notes the co-ordinates achieved as parts of the path being plotted.
   Reverses the co-ordinates if it's a y-slope.  i is a counter for the
   array of points, initially set to zero. *)

begin
   Li := Li + 1;
   if yslope then
   begin
      ypath[Li] := x1;
      xpath[Li] := y1
   end
   else
   begin
      xpath[Li] := x1;
      ypath[Li] := y1;
   end;
end; (*** End of Procedure. ***)



(* NOTE: Calling procedure must include initialisation of circlepathx and
   circlepathy arrays as type "patharray". *)

procedure Circle (xstart,ystart,r : longint;
                  VAR circlepathx, circlepathy : patharray);
(* Creates an array of points matching a circle of the desired radius.
   First creates an anti-clockwise arc from the min y position using a check
   on which direction follows the circle most truly.  Then uses this arc
   as a basis for completing the whole circle.  All 8 arcs are identical
   through reflection and rotation.

   Outputs two path arrays with co-ordinates for each circumference point.
   Arrays must be passed by reference. *)

type
   arctracker = array of boolean;

var
   x, y : longint;
   i, n : longint;
   xmove : arctracker;

begin
   i := -1;
   x := xstart;
   y := ystart + r;
   SetLength(xmove, r);
   SetLength(circlepathx, 8*r);
   SetLength(circlepathy, 8*r);
   for n := 0 to (r - 1) do
   begin
      if (sqrt(sqr(x-xstart+1) + sqr(y-ystart)) - r) < 0.5 then
         (* Checks if moving in the x direction stays within the circle.
            If not moves in y dir instead.  Note that it uses an effective
            circle radius of r - 0.5, to suit the block style of the circle
            better.  Uses simple circle equation: x^2 + y^2 = r^2 *)
      begin
         x := x + 1;
         xmove[n] := true
      end
      else
      begin
         y := y - 1;
         xmove[n] := false;
      end;
      i := i + 1;
      circlepathx[i] := x;
      circlepathy[i] := y;
   end; (* Arc 1 completed. xmove copies arc 1 for all other arcs. *)

   for n := (r - 1) downto 0 do
   begin
      if xmove[n] = true then y := y - 1 (* Arc 2 reflection of Arc 1. *)
         else x := x + 1;
      i := i + 1;
      circlepathx[i] := x;
      circlepathy[i] := y;
   end;
   for n := 0 to (r - 1) do
   begin
      if xmove[n] = true then y := y - 1 (* Arc 3: x-, y- *)
         else x := x - 1;
      i := i + 1;
      circlepathx[i] := x;
      circlepathy[i] := y;
   end;
   for n := (r - 1) downto 0 do
   begin
      if xmove[n] = true then x := x - 1 (* Arc 4: x-, y-, reflected *)
         else y := y - 1;
      i := i + 1;
      circlepathx[i] := x;
      circlepathy[i] := y;
   end;
   for n := 0 to (r - 1) do
   begin
      if xmove[n] = true then x := x - 1 (* Arc 5: x-, y+ *)
         else y := y + 1;
      i := i + 1;
      circlepathx[i] := x;
      circlepathy[i] := y;
   end;
   for n := (r - 1) downto 0 do
   begin
      if xmove[n] = true then y := y + 1 (* Arc 6: x-, y+, reflected *)
         else x := x - 1;
      i := i + 1;
      circlepathx[i] := x;
      circlepathy[i] := y;
   end;
   for n := 0 to (r - 1) do
   begin
      if xmove[n] = true then y := y + 1 (* Arc 7: x+, y+ *)
         else x := x + 1;
      i := i + 1;
      circlepathx[i] := x;
      circlepathy[i] := y;
   end;
   for n := (r - 1) downto 0 do
   begin
      if xmove[n] = true then x := x + 1 (* Arc 8: x+, y+, reflected *)
         else y := y + 1;
      i := i + 1;
      circlepathx[i] := x;
      circlepathy[i] := y;
   end; (* Circle complete!  Co-ords stored in circlepath arrays. *)

   for n := 0 to i do
   begin
      if circlepathx[n] < xmin then circlepathx[n] := xmin;
      if circlepathx[n] > xmax then circlepathx[n] := xmax;
      if circlepathy[n] < ymin then circlepathy[n] := ymin;
      if circlepathy[n] > ymax then circlepathy[n] := ymax;
   end; (* Enforces range limits on the circle. *)
end; (*** End of procedure. ***)


(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)


(* Now beginning terrain generation procedures.
   Note that most use the SpreadTile and PlaceTile functions, a variant of
   Spread and Dig used in Gruesome.  Randomly spreads an individual tile type.
   "rnd" variable affects how much it spreads, and "dropoff" can allow
   for more circular patterns to be produced.
   Need to play around with these more to investigate the wonders they can
   create!  *)



procedure PlaceTile (x,y,rnd,dropoff : longint; tileid : byte); FORWARD;
   (* Set up for recursive calling between Spread and Dig. *)
procedure PlaceOneTile(x,y : longint; tid : byte); FORWARD;


procedure SpreadTile (swx,swy,rnd,dropoff : longint; tileid : byte);
(* Checks squares around target (within map limits) and randomly places.
   Begins recursive spreading for placed squares.
   Note: Procedure should ONLY be called for squares that have a matching
   tileid.  Summoning procedure should have a check that the tileid is the
   same.  (Procedure does this itself when recursing.) *)

begin
   if (swy > (ymin + 1)) and (swy < (ymax - 1)) and (swx > (xmin + 1))
      and (swx < (xmax - 1))  then
   begin
      if (random(rnd) > 100) then PlaceTile(swx+1,swy,rnd,dropoff,tileid);
	   if (random(rnd) > 100) then PlaceTile(swx,swy+1,rnd,dropoff,tileid);
	   if (random(rnd) > 100) then PlaceTile(swx,swy-1,rnd,dropoff,tileid);
	   if (random(rnd) > 100) then PlaceTile(swx-1,swy,rnd,dropoff,tileid);
         (* Small chance of placing tiles in 4 compass directions	(rnd is
            tweakable for different map styles).  This differs from
            Gruesome and The Lion King where diagonal connections	were
            frequent.  PlaceTile procedure then begins recursion.  Random
            value must have chance of fail or recursion will not end.
            Proper investigation and testing needed of various values
            for rnd and dropoff to create interesting cavern types.  See
            CaveGenerator procedure for good examples from Trapper. *)

   end; (* End of min/max check. Note that squares can be placed near the
           edges with Spread, but no Spread is begun for those near-edge
           squares.  This leaves more jagged edges. *)
end; (*** End of procedure. ***)


procedure PlaceTile (x,y,rnd,dropoff : longint; tileid : byte);
(* Places tile if not already placed and calls to recursive SpreadTile
   procedure with dropoff applied. *)

begin
   if (coord[x,y].tileid <> tileid) then
   begin (* Vital check to make recursion eventually stop! *)
      PlaceOneTile(x,y,tileid);
      SpreadTile(x,y,rnd-dropoff,dropoff,tileid);
   end; (* Endif - procedure does nothing without matching the statement. *)
end; (*** End of procedure. ***)


procedure PlaceOneTile(x,y : longint; tid : byte);
(* Places individual tile.  This and WipeMap are the only procedures to
   affect individual mapcoord values relating to display. *)

begin
   coord[x,y].tileid := tid;
   with coord[x,y] do
   begin
	   case tileid of
	      0 : begin (* Void terrain - unnused.  Template for other terrains. *)
	             Passable := false;
	             BGColour := 0;
	             ForeColour := 0;
	             Symbol := ' ';
			    end;
		   1 : begin (* Regular tunnels, traversable. *)
	             Passable := true;
	             BGColour := 0;
	             ForeColour := CaveColourLight;
	             Symbol := '.';
	             Undiggable := false;
			    end;
	      2 : begin (* Walls, not traversable. *)
	             Passable := false;
	             BGColour := 0;
	             ForeColour := CaveColourLight;
	             Symbol := '#';
	             Undiggable := false;
		       end;
		   3 : begin (* Stairs, traversable. *)
	             Passable := true;
	             BGColour := 0;
	             ForeColour := 15;
	             Symbol := '>';
	             Undiggable := false;
			    end;	
		   4 : begin (* Walls, not traversable, not diggable.  Used for edges. *)
	             Passable := false;
	             BGColour := 0;
	             ForeColour := CaveColourLight;
	             Symbol := '#';
	             Undiggable := true;
	          end;
         5 : begin (* Gem of Power. *)
               Passable := true;
               BGColour := 0;
               ForeColour := 9;
               Symbol := '*';
	            Undiggable := false;
			    end;	
         6 : begin (* Gem of Life. *)
               Passable := true;
               BGColour := 0;
               ForeColour := 15;
               Symbol := '*';
	            Undiggable := false;
             end;
         
         (* Traps!  Tileids are deliberately set to be 10 higher than the
            activetrap numbers used for the player interface.  Colours are also
            set the same as the display colour on the sidebar trap list. *)
         11 : begin (* Flasher Trap. *)
               Passable := true;
               BGColour := 0;
               ForeColour := 11;
               Symbol := '^';
	            Undiggable := false;
             end;
         12 : begin (* Banger Trap. *)
               Passable := true;
               BGColour := 0;
               ForeColour := 4;
               Symbol := '^';
	            Undiggable := false;
             end;
         13 : begin (* Boomer Trap. *)
               Passable := true;
               BGColour := 0;
               ForeColour := 12;
               Symbol := '^';
	            Undiggable := false;
             end;
         14 : begin (* Blinder Trap. *)
               Passable := true;
               BGColour := 0;
               ForeColour := 15;
               Symbol := '^';
	            Undiggable := false;
             end;
         15 : begin (* KABOOMer Trap. *)
               Passable := true;
               BGColour := 0;
               ForeColour := 14;
               Symbol := '^';
	            Undiggable := false;
             end;
         16 : begin (* X-Boomer Trap. *)
               Passable := true;
               BGColour := 0;
               ForeColour := 1;
               Symbol := '^';
	            Undiggable := false;
             end;
         17 : begin (* KaoSBoomer Trap. *)
               Passable := true;
               BGColour := 0;
               ForeColour := 5;
               Symbol := '^';
	            Undiggable := false;
             end;
         18 : begin (* ULTRABOOMER Trap. *)
               Passable := true;
               BGColour := 0;
               ForeColour := 2;
               Symbol := '^';
	            Undiggable := false;
             end;
			   (* Insert further terrain types here! *)				
	     (* No otherwise statement - ignores nonsense commands. *)
      end; (* end case *)
   end; (* End of with. *)
end; (*** End of procedure. ***)



(* Connection procedures to make sure map is traversable. *)

procedure ConnectFill (x, y : longint);
(* Beginning with input square it marks passable squares as connected, and
   calls itself on all surrounding squares.  Stops when surrounded by walls or
   already connected squares.  Thus recursively maps out which squares are
   connected to the main area. *)

var
   a, b : longint;

begin
   if (coord[x,y].Passable = true) then (* Ignores walls. *)
   begin
      coord[x,y].Connected := true;
      for b := (y - 1) to (y + 1) do
         for a := (x - 1) to (x + 1) do
         begin
            if (coord[a,b].Connected = false) then ConnectFill(a,b);
               (* Necessary limiter - only calls for unconnected squares. *)
         end;
   end;
end; (*** End of procedure. ***)


procedure ConnectDig (x1, y1, x2, y2 : longint; tileid : byte);
(* Procedure which connects two points with the shortest path.  Since y-axis
   is smaller than x it tends to make horizontal lines.  Slight random
   element added for uneven lines. *)

var
   i : longint;

begin
   while (x2 <> x1) or (y2 <> y1) do (* End condition is reaching target. *)
   begin
      if random(100) > 35 then i := 1 else i := 0; (*random element*)
      (* x movement direction. *)
           if (x2 > x1) then x1 := x1 + i
      else if (x2 < x1) then x1 := x1 - i;
	  if random(100) > 35 then i := 1 else i := 0; (*random element*)
      (* y movement direction. *)
           if (y2 > y1) then y1 := y1 + i
      else if (y2 < y1) then y1 := y1 - i;
      PlaceOneTile(x1,y1,tileid); (* Creates passable tile. *)
      ConnectFill(x1,y1); (* Marks new path as connected. *)
   end;
end; (*** End of procedure. ***)


procedure ConnectMake (x, y : longint; tileid : byte);
(* Searches in circles around the starting point till it finds a square
   marked as connected.  Then sets this as the target destination and makes
   a connecting path to it by calling ConnectDig.
   tileid needed to determine what sort of connection to make. *)

var
   a, b, r, i : longint;
   ConnectFound : boolean;

begin
   ConnectFound := false;
   r := 2; (* Ignores immediate squares, as technically if they were
              connected then the beginning square would be too. *)
   repeat
      a := x + r;
      b := y + r; (* Starts search in square to top-right.
                     Each subsequent repeat moves out in this direction. *)
      i := 0;

      while (ConnectFound = false) and (i < 2*r) do
      begin
         b := b - 1;
         i := i + 1;
         if (b > ymin) and (b < ymax) and (a > xmin) and (a < xmax) then
            if (coord[a,b].Connected = true) then ConnectFound := true;
      end; (* First arm of spiral going down from start of loop. *)

      i := 0;
      while (ConnectFound = false) and (i < 2*r) do
      begin
         a := a - 1;
         i := i + 1;
         if (b > ymin) and (b < ymax) and (a > xmin) and (a < xmax) then
            if (coord[a,b].Connected = true) then ConnectFound := true;
      end; (* Seconds arm goes from right to left. *)

      i := 0;
      while (ConnectFound = false) and (i < 2*r) do
      begin
         b := b + 1;
         i := i + 1;
         if (b > ymin) and (b < ymax) and (a > xmin) and (a < xmax) then
            if (coord[a,b].Connected = true) then ConnectFound := true;
      end; (* Third arm moves from bottom to top. *)

      i := 0;
      while (ConnectFound = false) and (i < 2*r) do
      begin
         a := a + 1;
         i := i + 1;
         if (b > ymin) and (b < ymax) and (a > xmin) and (a < xmax) then
            if (coord[a,b].Connected = true) then ConnectFound := true;
      end; (* Fourth arm loops back to start. *)

      r := r + 1; (* Spiral moves further out. *)
   until ConnectFound;
      (* This scans in a spiral around the starting point until a connected
         point is found.  The values of a,b will end up as the coordinates of
         the closest connected point.  Spiral starts at radius 2 since no
         immediate points can be already connected (in theory). *)
   ConnectDig(x,y,a,b,tileid); (* Calls procedure to clear tunnel between
                                  initial point and connect point found of
								          type tileid. *)
end; (*** End of procedure. ***)



procedure CreaturePopulate; FORWARD;


(* Main caller of above functions *)
procedure CaveGenerator;
(* Creates dungeon with connectivity etc.  Style of dungeon depends on 'D'. *)

var
   a,b,stairsx,stairsy,gemx,gemy,n : longint;
   rnd, dropoff, rooms : longint;
   tooempty : boolean;

begin
   case D of
      12..13 : i := 5; (* Big rooms connected *)
      9..11 : i := 4; (* Small rooms connected *)
      6..8 : i := 3; (* Bigger caves *)
      4..5 : i := 2; (* Narrow caves *)
      1..3 : i := 1;
   end;
   (* Different dungeon types for different depths *)

   case i of (* attribute dungeon type *)
   (* High rnd -> lots of room spreading
      High dropoff -> rooms get cut off quick
	   Rooms must be less than 100.  70 means lots of rooms, 95 means few. *)
      1 : begin
	         rnd := 140;
			   dropoff := 2;
			   rooms := 75;  (* Tight, windy caverns. *)
			   CaveColourLight := 15;
			   CaveColourDark := 8;  (* White + dark grey colours *)
          end;
      2 : begin
	         rnd := 300;
			   dropoff := 30;
			   rooms := 90;  (* More open windy caverns. *)
			   CaveColourLight := 14;
			   CaveColourDark := 6;  (* Yellowish + brown *)
          end;
      3 : begin
	         rnd := 500;
			   dropoff := 40;
			   rooms := 96;  (* Large open caves. *)
			   CaveColourLight := 8;
			   CaveColourDark := 1;  (* Dark grey + dark blue *)
		    end;
      4 : begin
	         rnd := 600;
			   dropoff := 90;
			   rooms := 96;  (* Numerous connected small pockets of space. *)
			   CaveColourLight := 12;
			   CaveColourDark := 4;  (* Red + dark red *)
		    end;
      5 : begin
             rnd := 1000;
             dropoff := 98;
             rooms := 98;  (* Few large pockets connected by tunnels. *)
             CaveColourLight := 11;
             CaveColourDark := 3;  (* Cyan + dark cyan *)
          end;
   end; (* End case *)

   for b := ymin to ymax do (* Initializes all as walls *)
      for a := xmin to xmax do
	      PlaceOneTile(a,b,2);

   for b := (ymin + 2) to (ymax - 2) do
   (* Places random bursts of floor according to dungeon type stats. *)
   begin
      for a := (xmin + 2) to (xmax - 2) do
         if random(100) > rooms then
		      PlaceTile(a,b,rnd,dropoff,1)
   end;

   (* Position player and clears area around starting point. *)
   creatureindex[1].xpos := trunc(random(xmax - 8) + xmin + 3);
   creatureindex[1].ypos := trunc(random(ymax - 8) + ymin + 3);
   coord[creatureindex[1].xpos,creatureindex[1].ypos].cindex := 1;
   for b := (creatureindex[1].ypos - 1) to (creatureindex[1].ypos + 1) do
      for a := (creatureindex[1].xpos - 1) to (creatureindex[1].xpos + 1) do
         PlaceTile(a,b,145,2,1);

   (* Create stairs min distance from starting point and clear area around it *)
   repeat
      stairsx := trunc(random(xmax - 8) + xmin + 3);
      stairsy := trunc(random(ymax - 8) + ymin + 3)
   until sqrt((sqr(stairsx-creatureindex[1].xpos))+(sqr(stairsy-creatureindex[1].ypos))) > 20;
      (* Down stairs always generated a certain min distance from entrance.
         This uses the Pythagorean distance between points, which doesn't
         actually match the game movement distance, but suffices as a min
         distance check. *)
   for b := (stairsy - 1) to (stairsy + 1) do
      for a := (stairsx - 1) to (stairsx + 1) do
         PlaceTile(a,b,145,2,1);
	
   (* Ensures dungeon is fully connected. *)
   ConnectFill(creatureindex[1].xpos,creatureindex[1].ypos);
   for a := xmin to xmax do
      for b := ymin to ymax do
	     if (coord[a,b].Passable = true) and (coord[a,b].Connected = false) then
		     ConnectMake(a,b,1);
   
   gemx := 0;
   gemy := 0;
   
   if D in [1,2,4,6,7,9,11,12] then
   begin (* Place gem of power or life on appropriate levels. *)
      repeat
         gemx := trunc(random(xmax - 8) + xmin + 3);
         gemy := trunc(random(ymax - 8) + ymin + 3)
      until (abs(stairsx - gemx) > 8) and (abs(creatureindex[1].xpos - gemx) > 8)
         and (coord[gemx,gemy].tileid = 1);
         (* Places gem a min distance from the stairs and player and only on
            existing floor. *)
      if D < 12 then PlaceOneTile(gemx,gemy,5) else PlaceOneTile(gemx,gemy,6);
   end;
   
   if D < 12 then PlaceOneTile(stairsx,stairsy,3); (* 3 = down stairs *)
     (* No stairs on D12 or 13. *)
   
   CreaturePopulate;
   
   (* Surround edges by undiggable wall. *)
   b := ymin;
   for a := xmin to xmax do PlaceOneTile(a,b,4); (* Top line. *)
   b := ymax;
   for a := xmin to xmax do PlaceOneTile(a,b,4); (* Last line. *)
   a := xmin;
   for b := ymin to ymax do PlaceOneTile(a,b,4); (* 1st row. *)
   a := xmax;
   for b := ymin to ymax do PlaceOneTile(a,b,4); (* last row. *)	

   (* Check if dungeon edges are too "empty" - redesign level if they are.
      Only applies to i=5 dungeons since these are the only ones that suffer
	   this problem. *)
   if i = 5 then
   begin
      tooempty := false;
      n := 0;
      for a := (xmin + 1) to (xmin + 5) do
         for b := ymin to ymax do
            if coord[a,b].tileid = 1 then n := n + 1;
      if n = 0 then tooempty := true; (* Checks left. *)
      n := 0;
      for a := (xmax - 5) to (xmax - 1) do
         for b := ymin to ymax do
            if coord[a,b].tileid = 1 then n := n + 1;
      if n = 0 then tooempty := true;  (* Checks right. *)
      n := 0;
      for b := (ymin + 1) to (ymin + 4) do
         for a := xmin to xmax do
            if coord[a,b].tileid = 1 then n := n + 1;
      if n = 0 then tooempty := true; (* Checks top. *)
      n := 0;
      for b := (ymax - 4) to (ymax - 1) do
         for a := xmin to xmax do
            if coord[a,b].tileid = 1 then n := n + 1;
      if n = 0 then tooempty := true; (* Checks bottom. *)
      if tooempty then
      begin
         WipeMap;
         CaveGenerator;
      end;
      (* Procedure repeats until correct solution reached. *)
   end;
end;  (*** End of procedure. ***)


(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)
(* Message procedures: *)


procedure WriteWord(word : text);
(* Paired with WriteMessage.
   Produces messages in the message bar without overflowing at the sides or
   spilling into the map area.  All messages should be delivered through this
   function.  No spaces between sentences should be given - this procedure
   sorts that out itself. *)


begin
   TextColor(DfTxC);
   TextBackground(DfBgC);
   GoToXY(messagex,messagey);
   if messagex = 1 then write(word)
   else
   begin
      if messagey = 3 then
      begin (* Extra checks and stuff for third liners. *)
         if (72 - messagex) > Length(word) then
         begin  (* Word fits into remaining space okay. *)
            write(' ');
            write(word)
         end
         else
         begin
            TextColor(15);
            GoToXY(74,3);
            write('[more]');
            TextColor(DfTxC);
            ReadKey; (* Clears key buffer. *)
            ClearMessageBar;
            write(word)
         end (* End of third line exception. *)
      end
      else
      begin
         if (79 - messagex) > Length(word) then
         begin (* Normal message. *)
            if (word[1] <> '.') and (word[1] <> '!') then write(' ');
            write(word)
         end
         else
         begin (* Loop long lines. *)
            messagex := 1;
            messagey := messagey + 1;
            GoToXY(messagex,messagey);
            WriteWord(word); (* Recursive call to catch 3rd liners. *)
         end; (* Long line end. *)
      end; (* Top 2 lines end. *)
   end; (* Include 3rd line exceptions end. *)
   messagex := WhereX;
   messagey := WhereY; (* Sets pointers for next message. *)
end;


procedure WriteMessage(stuff : text);
(* Splits long messages into words before passing them to WriteWord so that
   they're easier to loop around the message bar.
   All messages should be given through this function. *)
   
var
   n,m,z : longint;
   word : text;
   
begin
   if playerdead = false then
   (* Message output stopped upon death. *)
   begin
      n := 1;
      m := 1;
      z := Length(stuff);
      repeat
         repeat
            SetLength(word,n); (* Initiate string references. *)
            word[n] := stuff[m];
            n := n + 1;
            m := m + 1;
         until (m > z) or (stuff[m] = ' ');
         (* Stops at wordbreak or end of line - need to look for end of line first
            or referencing stuff[m] will fail. *)
         WriteWord(word); (* Output through WriteWord function. *)
         n := 1;
         m := m + 1;
      until m > z; (* Stops after all of message is written. *)
   end;
end;


procedure DungeonMessage;
(* Level entry messages. *)

begin
   case D of
      1 : WriteMessage('You enter a cavern of dark, twisting passageways. In the distance you hear the foul grunting of some ugly ogres. Armed only with some gems of power normally used for mining you plod bravely forward...');
      3 : WriteMessage('The airs stirs rapidly around you. You sense a sinister presence about.');
      4 : WriteMessage('The caverns open up into larger earthen caves. The stench of ogres is stronger down here.');
      5 : WriteMessage('*THOMP* The ground shakes slightly and dust stirs from the high-reaching ceilings of this cavern.');
      6 : WriteMessage('The rocky caves here seem darker and ever more sinister, with shadows sticking deep in every crevasse.');
      8 : WriteMessage('You suddenly shudder horribly and your skin starts to crawl. The shadows here seem to be watching you.');
      9 : WriteMessage('It''s hot down here! The red clay caverns look to have been lived in long. Who knows what mad creatures would live down here...');
      10 : WriteMessage('You hear a distant cackle and you suddenly feel goosebumps pop up all over you. You really didn''t like the sound of that...');
      12 : begin
               WriteMessage('You enter a crystal cavern of scintillating beauty. Light is refracted into a miriad of astounding shapes and colours before your eyes. And yet you sense a darkness nearby.');
               WriteMessage('A profound and terrible darkness that threatens to snatch away your very soul. It can mean only one thing - the final battle is afoot...');
           end;
   end;
end;


procedure FullName(c : longint);
(* Used for enemy interactions.  Distinguishes between bosses (named) and
   regular enemies, and outputs a full description as appropriate.
   Regular enemies must be prefixed with the/The/a, bosses need no
   introduction. *)

begin
   with creatureindex[c] do
   begin
      if Named then
      begin
         WriteMessage(prefix);
         WriteMessage('the');
         WriteMessage(Name)
      end
      else
      begin
         WriteMessage(prefix);
         WriteMessage(Name);
      end;
   end;
end;

procedure BumpMessage(c : longint);
(* Procedure called when player bumps into enemy.  Value passed is creatureindex
   reference. *)
   
begin
   if creatureindex[1].blindcount > 0 then
      WriteMessage('Eek, there''s something there!!')
   else
   begin
      WriteMessage('Yikes, it''s');
      if creatureindex[c].Named = false then WriteMessage('a');
      FullName(c);
      case creatureindex[c].ctype of
         1 : WriteMessage('! The big brute is a lot taller than you, but also a lot slower, in both body and mind. It sniffs the air with hunger as you come near. Ick, better get clear quick!');
         2 : WriteMessage('! Half rat, half man, he''s a lot quicker on his feet than the dumb ogres around him. In his hand is a sinister sabre with a mean edge to it. Careful you don''t get too near!');
         3 : WriteMessage('! He towers far over you - even his feet are ten times your height. He moves as slow as a mountain, but looks just as difficult to damage.');
         4 : begin
             WriteMessage('! Wuggy''s personal assassin is an insidious force to deal with. She flicks in and out of the shadows with ease, and steps so lightly she doesn''t set off traps.');
             WriteMessage('Guess you''ll need some smart manouvres to take her down...');
             end;
         5 : WriteMessage('! Wuggy''s right hand man, and a formiddable opponent to anyone. He jumps faster than your traps can explode. To catch him you''ll need to watch where he lands...');
         6 : WriteMessage('!! You can''t help but feel your knees quake as you stand before his tall and terrible cloaked form. Can your little traps really defeat his terrible black magic? Only one way to find out...');
         7 : WriteMessage('!!! Back from the dead and still full of dread, and it looks like his powers have gotten even stronger than before!');
         8 : WriteMessage('! You''ve never seen bones move so fast - better get him off your tail somehow!');
         9 : WriteMessage('! Being undead sure hasn''t made this big brute prettier... You have to dodge to avoid the enormous drips of green pus leaking from its mouth!');
         10 : WriteMessage('! This isn''t one crature you want to see in life or death! You feel a dark mist rising before your vision... get away, get away!!!');
         11 : WriteMessage('! A wicked grin is sliced across his face and his twitching body seems ready to leap right at you!');
         12 : WriteMessage('! The other ogres treat this gal with respect and caution, steering well clear of her big rolling pin. You might want to do the same...');
         13 : WriteMessage('! Swooping above your head this winged wastrel won''t set off any floor traps, and has a rather insistent desire to suck out your bloooood!');
         14 : WriteMessage('! He''s wearing a big trenchcoat and has a creepy grin on his sweaty face. Ugh, you really don''t want to be anywhere near this thing...');
      end; (* End of case *)
   end;
end;


(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)
(* Trap related procedures: *)


procedure Blind(x,y : longint); FORWARD;
procedure Burn(x,y : longint); FORWARD;
procedure KaoS(x,y,rnd : longint); FORWARD;
procedure Ultra(x,y : longint); FORWARD;
procedure UltraCircle(x,y,r : longint); FORWARD;


procedure ActivateTrap (x,y : longint);

var
   a,b, t : longint;

begin
   detonated := detonated + 1;
   t := coord[x,y].tileid;
   PlaceOneTile(x,y,1); (* Changes square to floor. *)
   case t of
      11 : begin
              if coord[x,y].Visible then 
                 WriteMessage('The Flasher flares up!')
              else WriteMessage('You hear a distant click.');
              Blind(x,y);
           end;
      12 : begin
              if coord[x,y].Visible then
                 WriteMessage('The Banger explodes!')
              else WriteMessage('You hear a distant bang.');
              Burn(x,y);
           end;
      13 : begin
               if coord[x,y].Visible then
                  WriteMessage('The boomer explodes into a ball of flame!')
               else WriteMessage('You hear a distant booming sound.');
               for a := x-1 to x+1 do
                  for b := y-1 to y+1 do
                     if coord[a,b].Passable then Burn(a,b);
               (* Burns in a circle around the trap. Doesn't burn through
                  walls. *)
           end;
      14 : begin
               if coord[x,y].Visible then
                  WriteMessage('The Blinder erupts into a huge flash light!')
               else WriteMessage('You hear a distant whooshing noise.');
               for a := x-1 to x+1 do
                  for b := y-1 to y+1 do
                     if coord[a,b].Passable then Blind(a,b);
               if coord[x-1,y].Passable and coord[x-2,y].Passable
                  then Blind(x-2,y);
               if coord[x+1,y].Passable and coord[x+2,y].Passable
                  then Blind(x+2,y);
               if coord[x,y-1].Passable and coord[x,y-2].Passable
                  then Blind(x,y-2);
               if coord[x,y+1].Passable and coord[x,y+2].Passable
                  then Blind(x,y+2);
               (* Blind in a circle extended slightly in the four cardinal
                  directions. Doesn't flash through walls. *)
           end;
      15 : begin
               if coord[x,y].Visible then
                  WriteMessage('The KABOOMer explodes in a huge blazing ball!')
               else WriteMessage('You hear a distant KABOOM.');
               (* Burn in a large circle, but doesn't pass through walls. *)
               for a := x-1 to x+1 do
                  for b := y-1 to y+1 do
                     if coord[a,b].Passable then Burn(a,b);
               a := x;
               b := y; (* Quick patch of a coding error. *)
               
               if coord[a+1,b+1].Passable then
               begin (* Bottom-Right Arm. *)
                  if coord[a+2,b+2].Passable then Burn(a+2,b+2);
                  if coord[a+1,b+2].Passable then Burn(a+1,b+2);
                  if coord[a+2,b+1].Passable then Burn(a+2,b+1);
               end;
               
               if coord[a-1,b+1].Passable then
               begin (* Bottom-Left Arm. *)
                  if coord[a-2,b+2].Passable then Burn(a-2,b+2);
                  if coord[a-1,b+2].Passable then Burn(a-1,b+2);
                  if coord[a-2,b+1].Passable then Burn(a-2,b+1);
               end;
               
               if coord[a-1,b-1].Passable then
               begin (* Top-Left Arm. *)
                  if coord[a-2,b-2].Passable then Burn(a-2,b-2);
                  if coord[a-1,b-2].Passable then Burn(a-1,b-2);
                  if coord[a-2,b-1].Passable then Burn(a-2,b-1);
               end;
               
               if coord[a+1,b-1].Passable then
               begin (* Top-Right Arm. *)
                  if coord[a+2,b-2].Passable then Burn(a+2,b-2);
                  if coord[a+1,b-2].Passable then Burn(a+1,b-2);
                  if coord[a+2,b-1].Passable then Burn(a+2,b-1);
               end;
               
               if coord[a-1,b].Passable and coord[a-2,b].Passable then
               begin (* Left arm. *)
                  Burn(a-2,b);
                  if coord[a-3,b-1].Passable then Burn(a-3,b-1);
                  if coord[a-3,b].Passable then Burn(a-3,b);
                  if coord[a-3,b+1].Passable then Burn(a-3,b+1);
               end;
               
               if coord[a+1,b].Passable and coord[a+2,b].Passable then
               begin (* Right arm. *)
                  Burn(a+2,b);
                  if coord[a+3,b-1].Passable then Burn(a+3,b-1);
                  if coord[a+3,b].Passable then Burn(a+3,b);
                  if coord[a+3,b+1].Passable then Burn(a+3,b+1);
               end;
               
               if coord[a,b-1].Passable and coord[a,b-2].Passable then
               begin (* Top arm. *)
                  Burn(a,b-2);
                  if coord[a-1,b-3].Passable then Burn(a-1,b-3);
                  if coord[a,b-3].Passable then Burn(a,b-3);
                  if coord[a+1,b-3].Passable then Burn(a+1,b-3);
               end;
               
               if coord[a,b+1].Passable and coord[a,b+2].Passable then
               begin (* Bottom arm. *)
                  Burn(a,b+2);
                  if coord[a-1,b+3].Passable then Burn(a-1,b+3);
                  if coord[a,b+3].Passable then Burn(a,b+3);
                  if coord[a+1,b+3].Passable then Burn(a+1,b+3);
               end;
           end;
      16 : begin
               if coord[x,y].Visible then
                  WriteMessage('The X-Boomer shoots out streams of fire!')
               else WriteMessage('You hear a distant explosion.');
               Burn(x,y);
               a := 0;
               repeat (* Bottom-Right stream. *)
                  a := a + 1;
                  if coord[x+a,y+a].Passable then Burn(x+a,y+a)
                  else a := 5;
               until a = 5;
               a := 0;
               repeat (* Bottom-Left stream. *)
                  a := a + 1;
                  if coord[x-a,y+a].Passable then Burn(x-a,y+a)
                  else a := 5;
               until a = 5;
               a := 0;
               repeat (* Top-Left stream. *)
                  a := a + 1;
                  if coord[x-a,y-a].Passable then Burn(x-a,y-a)
                  else a := 5;
               until a = 5;
               a := 0;
               repeat (* Top-Right stream. *)
                  a := a + 1;
                  if coord[x+a,y-a].Passable then Burn(x+a,y-a)
                  else a := 5;
               until a = 5;
           end;
      17 : begin
               if coord[x,y].Visible then
                  WriteMessage('The KaoSBoomer erupts into a terrifying inferno!')
               else WriteMessage('You hear a terrifying crackling sound in the distance.');
               for a := x-1 to x+1 do
                  for b := y-1 to y+1 do
                     if coord[a,b].Passable then KaoS(a,b,125);
                 (* Separate procedure needed to do recursion properly. *)
           end;
      18 : begin
               ultraused := true;
               
               if coord[x,y].Visible then
                  WriteMessage('The ULTRABOOMER is detonated!!!');
               WriteMessage('A deafening blast fills the cavern and black winds howl through every crack and crevasse.');
               
               Ultra(x,y);
               UltraCircle(x,y,1);
               UltraCircle(x,y,2);
               UltraCircle(x,y,3);
               UltraCircle(x,y,5);
               UltraCircle(x,y,7);
               UltraCircle(x,y,11);
               UltraCircle(x,y,13);
               UltraCircle(x,y,17);
               UltraCircle(x,y,19);
               UltraCircle(x,y,23);
               UltraCircle(x,y,29);
               UltraCircle(x,y,31);
               UltraCircle(x,y,37);
               UltraCircle(x,y,41);
               UltraCircle(x,y,43);
               UltraCircle(x,y,47);
               UltraCircle(x,y,53);
               UltraCircle(x,y,59);
               UltraCircle(x,y,61);
               
               ReadKey;
               TextBackground(0);
               if creatureindex[1].blindcount = 0 then
                  LineofSight2(creatureindex[1].xpos,creatureindex[1].ypos,visrange);
               DrawMap;
               
           end;
   end; (* End of case. *)
end;


procedure Blind(x,y : longint);

var
   a,b,r : longint;

begin
   if coord[x,y].Visible then
   begin
      GoToXY(x,y);
      r := trunc(random(10));
      case r of
         0..7 : TextColor(11);
         8..9 : TextColor(15);
      end; (* Random color chosen for graphical effect. *)
      TextBackground(0);
      write('*');
      Delay(25); (* Pauses a moment so trap effects are more visible. *)
   end;
   if coord[x,y].cindex > 0 then
   begin
      if coord[x,y].cindex = 1 then
      begin
         creatureindex[coord[x,y].cindex].blindcount := 7;
         WriteMessage('Ack! You are blinded by the sudden flash of light!');
         for a := xmin to xmax do
            for b := ymin to ymax do
               coord[a,b].Visible := false;
         DrawSurround(a,b,visrange)
      end
      else
      begin
         creatureindex[coord[x,y].cindex].blindcount := 4;
         if coord[x,y].Visible then
         begin
            if creatureindex[coord[x,y].cindex].Named = false then WriteMessage('The');
            FullName(coord[x,y].cindex);
            WriteMessage('is blinded by the sudden flash of light!');
         end;
      end;
   end;
   if coord[x,y].Visible then
   begin
      GoToXY(x,y);
      DrawTile(x,y);
   end;
end;


procedure CreatureKill(c : longint); FORWARD;
procedure Jump(c : longint); FORWARD;

procedure Burn(x,y : longint);

var
   r : longint;

begin
   if coord[x,y].Visible then
   begin
      GoToXY(x,y);
      r := trunc(random(10));
      case r of
         0..4 : TextColor(4);
         5..7 : TextColor(12);
         8..9 : TextColor(14);
      end; (* Random color chosen - looks pretty on big traps. *)
      TextBackground(0);
      write('*');
      Delay(25); (* Pauses a moment so trap effects are more visible. *)
   end;
   if coord[x,y].cindex > 0 then
   begin
      if coord[x,y].cindex = 1 then
      begin
         WriteMessage('Eeeeayahhhh!!! You are engulfed in roaring flames that strip through your flesh in seconds. You die!');
         if creatureindex[1].HP = 0 then
         begin
            playerdead := true;
            killer := 1
         end
         else
         begin (* Wizard mode. *)
            creatureindex[1].HP := creatureindex[1].HP - 1;
            WriteMessage('You get better.')
         end
      end
      else
      begin
         if creatureindex[coord[x,y].cindex].HP = 0 then
         begin
            if (creatureindex[coord[x,y].cindex].Symbol = 'J') 
               and (creatureindex[coord[x,y].cindex].jumping = false) then
            begin
               if coord[x,y].Visible then WriteMessage('The Jester laughs and jumps away!')
                  else WriteMessage('You heard a distant laugh.');
               creatureindex[coord[x,y].cindex].jumping := true;
                 (* Setting jump = true means that future traps in this round won't be avoided. *)
               Jump(coord[x,y].cindex)
            end
            else
            begin
               if coord[x,y].Visible then
               begin
                  if creatureindex[coord[x,y].cindex].Named = false then WriteMessage('The');
                  FullName(coord[x,y].cindex);
                  WriteMessage('is horribly burned by the searing flames!')
               end
               else if (creatureindex[coord[x,y].cindex].Named = false)
                  then WriteMessage('You hear a gurgling scream.')
                     else WriteMessage('A deafening roar of anguish echoes loudly through the caverns.');
               CreatureKill(coord[x,y].cindex);
               kills := kills + 1
            end
         end
         else
         begin (* Here be giants. *)
            creatureindex[coord[x,y].cindex].HP := creatureindex[coord[x,y].cindex].HP - 1;
            if coord[x,y].Visible then
            begin
               if creatureindex[coord[x,y].cindex].Named = false then WriteMessage('The');
               FullName(coord[x,y].cindex);
               WriteMessage('grunts in anger as the flames lick its feet!')
            end
            else WriteMessage('You hear a loud grunt in the distance.');
         end;
      end;
   end;
   if coord[x,y].Visible then
   begin
      GoToXY(x,y);
      DrawTile(x,y);
   end;
   if coord[x,y].tileid > 10 then ActivateTrap(x,y);
   (* Can cause chains of traps going off. *)
end;


procedure KaoS(x,y,rnd : longint);
(* Algorithm stolen from my map generator.  Simple but effective. *)

var
   a, b : longint;

begin
   Burn(x,y); (* Burn chosen square then spread around it. *)
   
   (* Chance to repeat again for 8 surrounding squares. *)
   if coord[x-1,y-1].Passable and (random(rnd) > 100) then
      KaoS(x-1,y-1,rnd-4); (* Reduction in rnd needed to make it peter out. *)
   
   if coord[x,y-1].Passable and (random(rnd) > 100) then
      KaoS(x,y-1,rnd-4);

   if coord[x+1,y-1].Passable and (random(rnd) > 100) then
      KaoS(x+1,y-1,rnd-4);

   if coord[x+1,y].Passable and (random(rnd) > 100) then
      KaoS(x+1,y,rnd-4);

   if coord[x+1,y+1].Passable and (random(rnd) > 100) then
      KaoS(x+1,y+1,rnd-4);

   if coord[x,y+1].Passable and (random(rnd) > 100) then
      KaoS(x,y+1,rnd-4);

   if coord[x-1,y+1].Passable and (random(rnd) > 100) then
      KaoS(x-1,y+1,rnd-4);

   if coord[x-1,y].Passable and (random(rnd) > 100) then
      KaoS(x-1,y,rnd-4);

end;


procedure UltraCircle(x,y,r : longint);
(* Calls ultra procedure in a ring of radius r. *)

var
   circlepathx, circlepathy : patharray;
   i, j : longint;

begin
   Circle(x,y,r,circlepathx,circlepathy);
   j := Length(circlepathx);
   for i := 0 to (j-1) do
   begin
      if (circlepathx[i] <> xmin) and (circlepathx[i] <> xmax)
         and (circlepathy[i] <> ymin) and (circlepathy[i] <> ymax) then
            Ultra(circlepathx[i],circlepathy[i]);
   end;
end;


procedure Ultra(x,y : longint);

begin
   GoToXY(x,y);
   if trunc(random(3)) = 0 then TextColor(10) else TextColor(2);
   TextBackground(0);
   write('*');
   Delay(1); (* Pauses a moment so trap effects are more visible. *)
   
   if coord[x,y].tileid = 6 then (* Crystal of Life tile. *)
   begin
      if coord[x,y].Visible then WriteMessage('The Gem of Life is utterly annihilated!')
      else WriteMessage('You hear a terrible shattering sound in the distance.');
      WriteMessage('A grievous sense of loss utterly overwhelms you!');
      WriteMessage('The gems of power you hold suddenly explode, ripping you to pieces and smearing your body parts over the cavern walls!');
      lifebroken := true;
      playerdead := true;
      killer := -2;
   end;
   
   if coord[x,y].cindex > 0 then (* Occupied spaces. *)
   begin
      if coord[x,y].cindex = 1 then (* Player. *)
      begin
         WriteMessage('The dark energies of the ULTRABOOMER rip through your body, tearing each particle apart. You are utterly annihilated!');
         if playerdead = false then killer := -1;
         playerdead := true
      end
      else (* Enemy. *)
      begin
         if coord[x,y].Visible then
         begin
            if creatureindex[coord[x,y].cindex].Named = false then WriteMessage('The');
            FullName(coord[x,y].cindex);
            WriteMessage('is utterly annihilated!')
         end
         else
         begin
            if creatureindex[coord[x,y].cindex].Named then
            begin
               if (D = 15) then wraithkilled := true;
               warlockkilled := true;
               WriteMessage('A terrible roar of anger echoes through the caverns before being sharply cut off.')
            end
            else WriteMessage('You hear a blood-curdling scream in the distance.');
         end;
         CreatureKill(coord[x,y].cindex);
         kills := kills + 1
      end;
   end;
   
   (* Change regular walls into floor tiles. *)
   if coord[x,y].Symbol <> '>' then PlaceOneTile(x,y,1);
end;


(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)
(* Enemy related procedures: *)


procedure CreatureMake(creature,x,y : byte); FORWARD;

procedure CreaturePopulate;
(* Decides how many creatures will be in the dungeon level and calls on
   procedure to create them all. 
   For boss levels it places the bosses too. *)

var
   i,j,x,y,r : longint;

begin
   (* Makes random number of enemies, increasing as player goes higher up
      in D levels. *)
   SetLength(creatureindex,2); (* Wipes previous entrants, apart from Toby
                                  of course. *)
   case D of  (* Sets number of ogres per level. *)
      1 : j := 13;
      2 : j := 16;
      3 : j := 18;
      4 : j := 26;
      5 : j := 32;
      6 : j := 22;
      7 : j := 23;
      8 : j := 25;
      9 : j := 22;
      10 : j := 24;
      11 : j := 26;
      12 : j := 30;
      13 : j := 30;
   end;
   
   (* Place boss first, so it gets first move in creaturearray. *)
   repeat
      x := random(xmax - xmin - 7) + xmin + 7;
      y := random(ymax - ymin - 1) + ymin + 1;
   until (coord[x,y].Passable = true)
      and (coord[x,y].cindex = 0)
      and (sqrt((sqr(x-creatureindex[1].xpos))+(sqr(y-creatureindex[1].ypos))) > 8);
      (* Generates monster on Passable, empty tile a certain min
         distance from the player. *)
   if D = 3 then CreatureMake(2,x,y); (* Dasher *)
   if D = 5 then CreatureMake(3,x,y); (* Giant *)
   if D = 8 then CreatureMake(4,x,y); (* Shadow *)
   if D = 10 then CreatureMake(5,x,y); (* Jester *)
   if D = 12 then CreatureMake(6,x,y); (* Warlock *)
   if D = 13 then CreatureMake(7,x,y); (* Wraith *)
   
   if D < 5 then r := 4 else r := 3;
   for i := 1 to j do
   begin
      repeat
         x := random(xmax - xmin - 7) + xmin + 7;
         y := random(ymax - ymin - 1) + ymin + 1;
      until (coord[x,y].Passable = true)
         and (coord[x,y].cindex = 0)
         and (sqrt((sqr(x-creatureindex[1].xpos))+(sqr(y-creatureindex[1].ypos))) > r);
         (* Generates monster on Passable, empty tile a certain min
            distance from the player. *)
      CreatureMake(1,x,y); (* 1 = ogre *)
   end;
   
   if D > 3 then
   begin
      j := trunc(random(4) + 3);
      case D of
         4..5   : r := 12; (* momma ogre *)
         6..8   : r := 13; (* vampire bat *)
         9..10  : r := 14; (* perverted ogre *)
         11..13 : r := trunc(random(3) + 12); (* random of above *)
      end; (* end of case *)
      
      for i := 1 to j do
      begin
         repeat
            x := random(xmax - xmin - 7) + xmin + 7;
            y := random(ymax - ymin - 1) + ymin + 1;
         until (coord[x,y].Passable = true)
            and (coord[x,y].cindex = 0)
            and (sqrt((sqr(x-creatureindex[1].xpos))+(sqr(y-creatureindex[1].ypos))) > 6);
            (* Generates monster on Passable, empty tile a certain min
               distance from the player. *)
         CreatureMake(r,x,y); (* 1 = ogre *)
      end;
   end;
end;


procedure CreatureMake(creature,x,y : byte);
(* Initiates a creature record and sets the creature in place on the
   dungeon level.  Input parameter is creature number, which identifies
   type of monster. *)

var
   n,i,r : longint;   
   
begin
   n := Length(creatureindex);
   SetLength(creatureindex,n + 1); (* Makes room for newbie. *)
   with creatureindex[n] do
   begin
      xpos := x;
      ypos := y;
      coord[xpos,ypos].cindex := n;
      case creature of
         1 : begin
                ctype := 1;
                Symbol := 'O';
                Colour := 10;
                Named := false;
                Name := 'ogre';
                r := trunc(random(33) + 1);
                case r of
                   1 : prefix := 'very ugly';
                   2 : prefix := 'grotesque';
                   3 : prefix := 'stinky';
                   4 : prefix := 'sweaty';
                   5 : prefix := 'lazy';
                   6 : prefix := 'grumpy';
                   7 : prefix := 'grouchy';
                   8 : prefix := 'yellow-skinned';
                   9 : prefix := 'thick-hided';
                   10 : prefix := 'bug-eyed';
                   11 : prefix := 'toothless';
                   12 : prefix := 'hungry';
                   13 : prefix := 'horrible';
                   14 : prefix := 'moody';
                   15 : prefix := 'stupid';
                   16 : prefix := 'dumb';
                   17 : prefix := 'dull';
                   18 : prefix := 'red-eyed';
                   19 : prefix := 'grinning';
                   20 : prefix := 'squinting';
                   21 : prefix := 'sleepy';
                   22 : prefix := 'sloppy';
                   23 : prefix := 'messy';
                   24 : prefix := 'manic';
                   25 : prefix := 'raging';
                   26 : prefix := 'tall';
                   27 : prefix := 'skinny';
                   28 : prefix := 'squat';
                   29 : prefix := 'fat';
                   30 : prefix := 'swollen';
                   31 : prefix := 'chubby';
                   32 : prefix := 'grimy';
                   33 : prefix := 'dirty';
                   (* Prefixes must be adjectives starting with consonants. *)
                end; (* End of case. *)                   
                blindcount := 0;
                turnwait := trunc(random(2));
                turndelay := 1;
                jumping := false;
             end;
         2 : begin
                ctype := 2;
                Symbol := 'D';
                Colour := 14;
                Named := true;
                Name := 'Dasher';
                prefix := 'Danny';
                blindcount := 0;
                turnwait := 0;
                turndelay := 0; (* Moves as fast as Toby - faster in Hardcore. *)
                HP := 0;
                jumping := false;
             end;
         3 : begin
                ctype := 3;
                Symbol := 'G';
                Colour := 12;
                Named := true;
                Name := 'Giant';
                prefix := 'Gordon';
                blindcount := 0;
                turnwait := trunc(random(3));
                turndelay := 2; (* Slow bastard. *)
                HP := 4; (* Takes 5 hits to kill. *)
                jumping := false;
             end;
         4 : begin
                ctype := 4;
                Symbol := '.'; (* Looks like a dungeon floor tile. *)
                Colour := CaveColourLight;
                Named := true;
                Name := 'Shadow';
                prefix := 'Sally';
                blindcount := 0;
                turnwait := trunc(random(2));
                turndelay := 1;
                HP := 0;
                jumping := false;
             end;
         5 : begin
                ctype := 5;
                Symbol := 'J';
                Colour := 10;
                Named := true;
                Name := 'Jester';
                prefix := 'Jack';
                blindcount := 0;
                turnwait := trunc(random(2));
                turndelay := 1;
                HP := 0;
                jumping := false;
             end;
         6 : begin
                ctype := 6;
                Symbol := 'W';
                Colour := 13;
                Named := true;
                Name := 'Warlock';
                prefix := 'Wuggy';
                blindcount := 0;
                turnwait := trunc(random(2));
                turndelay := 1;
                HP := 0;
                jumping := false;
             end;
         7 : begin
                ctype := 7;
                Symbol := 'W';
                Colour := 5;
                Named := true;
                Name := 'Wraith';
                prefix := 'Wuggy';
                blindcount := 0;
                turnwait := trunc(random(2));
                turndelay := 1;
                HP := 0;
                jumping := false;
             end;
         8 : begin
                ctype := 8;
                Symbol := 'D';
                Colour := 6;
                Named := false;
                Name := 'dasher';
                prefix := 'skeleton';
                blindcount := 0;
                turnwait := 0;
                turndelay := 0; (* Moves as fast as Toby - faster in Hardcore. *)
                HP := 0;
                jumping := false;
             end;
         9 : begin
                ctype := 9;
                Symbol := 'G';
                Colour := 4;
                Named := false;
                Name := 'giant';
                prefix := 'zombie';
                blindcount := 0;
                turnwait := trunc(random(3));
                turndelay := 2; (* Slow bastard. *)
                HP := 4; (* Takes 5 hits to kill. *)
                jumping := false;
             end;
         10 : begin
                ctype := 10;
                Symbol := '.';
                Colour := CaveColourLight;
                Named := false;
                Name := 'shadow';
                prefix := 'shadow of a';
                blindcount := 0;
                turnwait := trunc(random(2));
                turndelay := 1;
                HP := 0;
                jumping := false;
             end;
         11 : begin
                ctype := 11;
                Symbol := 'J';
                Colour := 2;
                Named := false;
                Name := 'jester';
                prefix := 'ghoulish';
                blindcount := 0;
                turnwait := trunc(random(2));
                turndelay := 1;
                HP := 0;
                jumping := false;
             end;
         12 : begin
                ctype := 12;
                Symbol := 'O';
                Colour := 13;
                Named := false;
                Name := 'ogre';
                prefix := 'momma';
                blindcount := 0;
                turnwait := trunc(random(2));
                turndelay := 1;
                HP := 0;
                jumping := false;
             end;
         13 : begin
                ctype := 13;
                Symbol := 'B';
                Colour := 4;
                Named := false;
                Name := 'bat';
                prefix := 'vampire';
                blindcount := 0;
                turnwait := trunc(random(2));
                turndelay := 1;
                HP := 0;
                jumping := false;
             end;
         14 : begin
                ctype := 14;
                Symbol := 'O';
                Colour := 8;
                Named := false;
                Name := 'ogre';
                prefix := 'perverted';
                blindcount := 0;
                turnwait := trunc(random(2));
                turndelay := 1;
                HP := 0;
                jumping := false;
             end;
         
      end; (* End of case. *)
	end; (* End of with statement. *)
end; (*** End of Procedure. ***)



procedure CreatureMove(c : longint);
(* Moves enemy towards player's scent. *)

var
   a, b, n, m, o, r : longint;
   sarray, barray, rarray : array[1..8] of longint;

begin
   with creatureindex[c] do
   begin
      coord[xpos,ypos].cindex := 0;
      if coord[xpos,ypos].Visible then
      begin
         GoToXY(xpos,ypos);
         DrawTile(xpos,ypos);
      end;
      
      (* Smell array gives a smell rating for each square which is based
         mainly on that square, but also its surrounding squares. *)
      sarray[1] := 2*coord[xpos-1,ypos+1].smell
                   + coord[xpos,ypos+1].smell + coord[xpos-1,ypos].smell;
      sarray[2] := 2*coord[xpos,ypos+1].smell
                   + coord[xpos+1,ypos+1].smell + coord[xpos-1,ypos+1].smell;
      sarray[3] := 2*coord[xpos+1,ypos+1].smell
                   + coord[xpos+1,ypos].smell + coord[xpos,ypos+1].smell;
      sarray[4] := 2*coord[xpos-1,ypos].smell
                   + coord[xpos-1,ypos+1].smell + coord[xpos-1,ypos-1].smell;
      (* Using numpad as direction guidance - pretend 5 is a 9. *)
      sarray[5] := 2*coord[xpos+1,ypos-1].smell
                   + coord[xpos,ypos-1].smell + coord[xpos+1,ypos].smell;
      sarray[6] := 2*coord[xpos+1,ypos].smell
                   + coord[xpos+1,ypos-1].smell + coord[xpos+1,ypos+1].smell;
      sarray[7] := 2*coord[xpos-1,ypos-1].smell
                   + coord[xpos-1,ypos].smell + coord[xpos,ypos-1].smell;
      sarray[8] := 2*coord[xpos,ypos-1].smell
                   + coord[xpos-1,ypos-1].smell + coord[xpos+1,ypos-1].smell;
      
      (* Squares with walls or other enemies are ranked lower. *)
      if ((coord[xpos-1,ypos+1].Passable = false)
         or (coord[xpos-1,ypos+1].cindex > 1)) then sarray[1] := -1;
      if ((coord[xpos,ypos+1].Passable = false)
         or (coord[xpos,ypos+1].cindex > 1)) then sarray[2] := -1;
      if ((coord[xpos+1,ypos+1].Passable = false)
         or (coord[xpos+1,ypos+1].cindex > 1)) then sarray[3] := -1;
      if ((coord[xpos-1,ypos].Passable = false)
         or (coord[xpos-1,ypos].cindex > 1)) then sarray[4] := -1;
      if ((coord[xpos+1,ypos-1].Passable = false)
         or (coord[xpos+1,ypos-1].cindex > 1)) then sarray[5] := -1;
      if ((coord[xpos+1,ypos].Passable = false)
         or (coord[xpos+1,ypos].cindex > 1)) then sarray[6] := -1;
      if ((coord[xpos-1,ypos-1].Passable = false)
         or (coord[xpos-1,ypos-1].cindex > 1)) then sarray[7] := -1;
      if ((coord[xpos,ypos-1].Passable = false)
         or (coord[xpos,ypos-1].cindex > 1)) then sarray[8] := -1;
      
      barray := sarray; (* Backup for later. *)
      
      (* Sort sarray directions into orders of ascendance in rarray. *)
      for n := 1 to 8 do
      begin
         o := 1;
         for m := 2 to 8 do
            if sarray[o] < sarray[m] then o := m;
         rarray[n] := o;
         sarray[o] := -2;
      end;
      
      r := 1;
      for n := 2 to 8 do
         if barray[rarray[n]] = barray[rarray[1]] then
            r := r + 1;
      (* If multiple numbers of the same smell value then choose randomly
         between them. *)
      
      o := trunc(random(r)) + 1;
      
      case rarray[o] of (* Corresponds to numpad directions, 5 = 9. *)
         1 : begin
                a := -1;
                b := 1;
             end;
         2 : begin
                a := 0;
                b := 1;
             end;
         3 : begin
                a := 1;
                b := 1;
             end;
         4 : begin
                a := -1;
                b := 0;
             end;
         5 : begin
                a := 1;
                b := -1;
             end;
         6 : begin
                a := 1;
                b := 0;
             end;
         7 : begin
                a := -1;
                b := -1;
             end;
         8 : begin
                a := 0;
                b := -1;
             end;
      end; (* End of case - direction is chosen and set. *)
      
      if (barray[1] + barray[2] + barray[3] + barray[4] + barray[5] +
          barray[6] + barray[7] + barray[8]) = -8 then
      begin (* Sum of these is only -8 if no moves available. *)
         a := 0;
         b := 0;
      end;
      
      if blindcount > 0 then
      begin (* Random direction for blindies. *)
         blindcount := blindcount - 1;
         repeat
            a := trunc(random(3)) - 1;
            b := trunc(random(3)) - 1;
         until (coord[xpos + a,ypos + b].Passable = true);
         if coord[xpos + a,ypos + b].cindex > 0 then
         begin
            if coord[xpos+a,ypos+b].cindex = 1 then
            begin
               if Named = false then WriteMessage('The');
               FullName(c);
               case ctype of
                 1 : WriteMessage('blindly stumbles into you and clobbers you to death!');
                 2 : WriteMessage('blindly swipes his blade and slices your face off, followed by your limbs and head most of the rest of you.');
                 3 : WriteMessage('blindly stomps his foot down on your head.  You do not survive the encounter...');
                 4 : WriteMessage('blindly thrashes about and accidentally rips your spleen through your buttocks.');
                 5 : WriteMessage('blindly pokes his cane through your stomach. It turns into a duck and flies away, leaving you quivering and helpless in a rapidly expanding pool of blood.');
                 6..7 : WriteMessage('blindly grabs at you! He locks onto your head, screams in rage, and proceeds to torturously suck all mental ability from your mind.  You are left a gibbering, senseless wreck!');
                 8 : WriteMessage('blindly swipes his blade and slices your face off, followed by your limbs and head most of the rest of you.');
                 9 : WriteMessage('blindly stomps his foot down on your head.  You do not survive the encounter...');
                 10 : WriteMessage('blindly thrashes about and accidentally rips your spleen through your buttocks.');
                 11 : WriteMessage('blindly pokes his cane through your stomach. It turns into a duck and flies away, leaving you quivering and helpless in a rapidly expanding pool of blood.');
                 12 : WriteMessage('blindly stumbles into you and yelps in surprise. She cracks you over the skull with her rolling pin and keeps screeching for help as she batters you to death.');
                 13 : WriteMessage('blindly claws at you, ripping you to shreds!');
                 14 : WriteMessage('blindly grabs you and violently thrusts you about! After a short minute of pain and confusion your head is bashed open against the nearest cave wall.');
               end; (* End of case *)
               
               if creatureindex[1].HP = 0 then
               begin
                  playerdead := true;
                  killer := c
               end
               else
               begin (* Wizard mode. *)
                  creatureindex[1].HP := creatureindex[1].HP - 1;
                  WriteMessage('You get better.');
               end;
               
               a := 0;
               b := 0
            end
            else
            begin
               if creatureindex[coord[xpos+a,ypos+b].cindex].Named then
               begin
                  if coord[xpos,ypos].Visible = false then
                     WriteMessage('You hear fighting noises in the distance.') else
                  begin
                     WriteMessage('The');
                     FullName(c);
                     WriteMessage('blindly stumbles into');
                     FullName(coord[xpos+a,ypos+b].cindex);
                     WriteMessage('and tries to attack.');
                     WriteMessage(creatureindex[coord[xpos+a,ypos+b].cindex].prefix);
                     WriteMessage('yells in fury and tears the head off the blind');
                     WriteMessage(creatureindex[c].Name);
                     WriteMessage('.')
                  end;
                  CreatureKill(c);
                  cleankills := cleankills + 1
               end
               else
               begin
                  if coord[xpos,ypos].Visible = false then
                     WriteMessage('You hear some fighting in the distance.') else
                  begin
                     WriteMessage('The');
                     FullName(c);
                     WriteMessage('blindly stumbles into the');
                     FullName(coord[xpos+a,ypos+b].cindex);
                     WriteMessage('and batters it to death!');
                  end;
                  CreatureKill(coord[xpos+a,ypos+b].cindex);
                  cleankills := cleankills + 1;
                  if coord[xpos+a,ypos+b].Visible then
                  begin
                     GoToXY(xpos+a,ypos+b);
                     DrawTile(xpos+a,ypos+b);
                  end;
               end;
            end;
         end;
      end; (* End of blind movement. *)
      
      if ypos > 0 then (* Check in case creature has been killed yet. *)
      begin
         if coord[xpos+a,ypos+b].cindex = 1 then
         begin (* Player is on target square. *)
            if Named = false then WriteMessage('The');
            FullName(c);
            r := trunc(random(6)) + 1; (* For death message variants. *)
            
            if creatureindex[c].ctype = 1 then
            begin
               case r of
                  1 : WriteMessage('grabs you and squeezes the life out of you before thrusting your corpse down its gaping maw!');
                  2 : WriteMessage('clubs you over the head and pounds you into the dirt!');
                  3 : WriteMessage('rips open your stomach and begins devouring your entrails!');
                  4 : WriteMessage('tears you limb from limb and starts popping the pieces down its throat, chewing merrily!');
                  5 : WriteMessage('seizes you suddenly and bites your tiny head off!');
                  6 : WriteMessage('squishes your little head between its enormous fists!');
               end;
            end;
            
            if creatureindex[c].ctype = 12 then
            begin
               case r of
                  1..3 : WriteMessage('thwacks you over the head with her rolling pin. "Dinner''s ready, boys!" she yells as she stuff you into a pot of boiling water.');
                  4..6 : WriteMessage('screams in disgust and hits you with her clothes mangler. You are thoroughly mangled!');
               end;
            end;
            
            if creatureindex[c].ctype = 13 then WriteMessage('swoops down, latches onto you and bites hard into your neck. Within seconds it sucks every ounce of blood from your little body!');
            
            if creatureindex[c].ctype = 14 then
            begin
               case r of
                  1..3 : WriteMessage('leaps at you and lands on you with his fat belly. You are horribly crushed!');
                  4..6 : WriteMessage('snatches you up and stuffs you down his trenchcoat. You last a few brief moments pressed against his horrid sweaty flesh before you mercifully suffocate.');
               end;
            end;
            
            if creatureindex[c].Symbol = 'D' then
            begin
               case r of
                  1..3 : WriteMessage('slices you in two, and then in four, then in eight - it all gets a bit messy after that...');
                  4..6 : WriteMessage('sticks his sword through your ribs and plucks your heart out!');
               end;
            end;
            
            if creatureindex[c].Symbol = 'G' then
            begin
               case r of
                  1..3 : WriteMessage('steps on you. You are squished into gnome soup!');
                  4..6 : WriteMessage('bends down and thumps his huge fist into you. You explode!');
               end;
            end;
            
            if creatureindex[c].Symbol = '.' then
            begin
               case r of
                  1..3 : WriteMessage('creeps into your throat and yanks your lungs out!');
                  4..6 : WriteMessage('reaches into your chest and squeezes on your liver till it implodes!');
               end;
            end;
            
            if creatureindex[c].Symbol = 'J' then
            begin
               case r of
                  1..3 : WriteMessage('laughs merrily as he smashes his cane through your skull!');
                  4..6 : WriteMessage('guffaws in amusement as he plucks out your eyes and starts juggling with them! Then he kills you.');
               end;
            end;
            
            if creatureindex[c].Symbol = 'W' then
            begin
               case r of
                  1..3 : WriteMessage('utters a dark enchantment and rips your soul from its body! With a fiendish cackle he devours it.');
                  4..6 : WriteMessage('zaps you with a dark ray, neatly tearing a large hole through your torso!');
               end;
            end;
            
            
            if creatureindex[1].HP = 0 then
            begin
               playerdead := true;
               killer := c
            end
            else
            begin (* Wizard mode. *)
               creatureindex[1].HP := creatureindex[1].HP - 1;
               WriteMessage('You get better.');
            end;
            
            a := 0;
            b := 0; (* Keeps monster on same tile if player dies. *)
         end; (* End of player target. *)
      
         xpos := xpos + a;
         ypos := ypos + b;
         coord[xpos,ypos].cindex := c; (* Places monster on new square. *)

         if coord[xpos,ypos].Visible then
         begin
            GoToXY(xpos,ypos);
            DrawTile(xpos,ypos);
         end;
         (* Updates display. *)
         
         if coord[xpos,ypos].tileid > 10 then
            if ctype in [1..3,5..9,11,12,14] then ActivateTrap(xpos,ypos);
         (* Sets off any traps on new square, except for shadows. *)
         
         if Symbol = '.' then
         begin
            if Colour = CaveColourLight then Colour := CaveColourDark
               else Colour := CaveColourLight;
            if coord[xpos,ypos].Visible then
            begin
               r := trunc(sqrt((sqr(xpos-creatureindex[1].xpos))+(sqr(ypos-creatureindex[1].ypos))));
               case r of
                  3..4 : WriteMessage('You see something moving in the shadows!');
                     2 : WriteMessage('You feel horribly threatened by the encroaching shadows!');
                     1 : WriteMessage('The looming shadows send a terrifying chill down your spine!');
               end;
            end; (* End shadow move message. *)
         end; (* End of shadow move. *)
         
      end;
   end; (* End of with. *)
end;


procedure WuggySpecials(c : longint);

var
   a,b,i,r,empty : longint;
   trapfound : boolean;
   
begin
   with creatureindex[c] do
   begin
      trapfound := false;
      empty := 0;
      
      (* Search for traps and empty squares. *)
      for a := xpos-2 to xpos+2 do
      begin
         for b := ypos-2 to ypos+2 do
         begin
            if (a > xmin) and (a < xmax) and (b > ymin) and (b < ymax) then
            begin
               if coord[a,b].tileid in [11..17] then trapfound := true;
               if (coord[a,b].Passable = true) and (coord[a,b].cindex = 0) then empty := empty + 1;
            end;
         end;
      end;
      
      if (trapfound = true) then
      begin (* Destroy traps. *)
      if coord[xpos,ypos].Visible = false then
            WriteMessage('You hear a dark chant in the distance.')
         else
         begin
            FullName(c);
            WriteMessage('utters a dark chant.');
         end;
      
         for a := xpos-2 to xpos+2 do
         begin
            for b := ypos-2 to ypos+2 do
            begin
               if (a > xmin) and (a < xmax) and (b > ymin) and (b < ymax) then
               begin
                  if coord[a,b].tileid in [11..17] then
                  begin
                     if coord[a,b].Visible then WriteMessage('A trap is destroyed!')
                     else WriteMessage('You hear a clunking noise.');
                     PlaceOneTile(a,b,1);
                  end;
               end;
            end;
         end
      end
      
      else (* Won't summon if he's destroying traps. *)
      begin
         if empty > 4 then
         begin (* Summon creatures. *)
            if trunc(random(6)) = 0 then
            begin (* Doesn't always summon. *)
               if coord[xpos,ypos].Visible = false then
                  WriteMessage('You hear an unholy evocation in the distance.')
               else
               begin
                  FullName(c);
                  WriteMessage('utters an unholy evocation.');
               end;
            
               (* Enemy summoned. *)
               repeat
                  a := xpos - 2 + trunc(random(5));
                  b := ypos - 2 + trunc(random(5));
               until (a > xmin) and (a < xmax) and (b > ymin) and (b < ymax)
                  and (coord[a,b].Passable = true) and (coord[a,b].cindex = 0);
                
               if warlockkilled = false then CreatureMake(1,a,b) else
               begin (* Wraith summons extra types. *)
                  r := trunc(random(55)) - 40;
                  if r < 8 then r := 1;
                  CreatureMake(r,a,b);
               end;
               if coord[a,b].Visible then
               begin
                  GoToXY(a,b);
                  DrawTile(a,b);
                  WriteMessage('A');
                  FullName(coord[a,b].cindex);
                  WriteMessage('appears!');
               end;
            end;
         end;
      end;
   end; (* End of with *)
end;


procedure MommaSpecial(c : longint);
(* Mommas sometimes summon ogres. *)

var
   empty, a, b : longint;

begin
   empty := 0;
   with creatureindex[c] do
   begin
      (* Search for empty squares. *)
      for a := xpos-1 to xpos+1 do
         for b := ypos-1 to ypos+1 do
            if ((coord[a,b].Passable = true) and (coord[a,b].cindex = 0)) then empty := empty + 1;
      if (empty > 4) and (trunc(random(6)) = 0) then
      begin (* Summon ogre. *)
         if coord[xpos,ypos].Visible = false then
            WriteMessage('You hear a commanding screech in the distance.')
         else
         begin
            WriteMessage('The ');
            FullName(c);
            WriteMessage('shouts out, "Boys, come help your old momma!"')
         end;
         
         repeat
            a := xpos - 1 + trunc(random(3));
            b := ypos - 1 + trunc(random(3));
         until (coord[a,b].Passable = true) and (coord[a,b].cindex = 0);
         
         CreatureMake(1,a,b);
         
         if coord[a,b].Visible then
         begin
            GoToXY(a,b);
            DrawTile(a,b);
            WriteMessage('A');
            FullName(coord[a,b].cindex);
            WriteMessage('appears!');
         end;
      end;
   end;
end;


procedure PervertSpecial(c : longint);

var
   a, b, i : longint;

begin
   with creatureindex[c] do
   begin
      if (sqrt((sqr(xpos-creatureindex[1].xpos))+(sqr(ypos-creatureindex[1].ypos))) < 3) then
      begin
         WriteMessage('The perverted ogre opens up his trenchcoat and flashes you.  You are blinded by the horrific sight!');
         creatureindex[1].blindcount := 7;
         for a := xmin to xmax do
            for b := ymin to ymax do
               coord[a,b].Visible := false;
         DrawSurround(a,b,visrange)
      end;
   end;
end;


procedure Jump(c : longint);
(* Finds a point along a circle of radius three around jester's initial
   position.  Point must not be a wall and must be within map boundaries.
   If square is occupied then jester will kill occupant (including player)
   unless it is another boss (ie Wuggy - he'd kill the jester instead).
   Jester, if still alive, is finally placed on the square, with the
   possibility of setting off a trap there. *)

var
   x, y, r, ctarget : longint;

begin
   with creatureindex[c] do
   begin
      (* Blank current square. *)
      coord[xpos,ypos].cindex := 0;
      if coord[xpos,ypos].Visible then
      begin
         GoToXY(xpos,ypos);
         DrawTile(xpos,ypos);
      end;
      
      repeat
         r := trunc(random(24)) + 1;
         case r of
            1 : begin
                  x := xpos - 2;
                  y := ypos - 3;
                end;
            2 : begin
                  x := xpos - 1;
                  y := ypos - 3;
                end;
            3 : begin
                  x := xpos;
                  y := ypos - 3;
                end;
            4 : begin
                  x := xpos + 1;
                  y := ypos - 3;
                end;
            5 : begin
                  x := xpos + 2;
                  y := ypos - 3;
                end;
            6 : begin
                  x := xpos + 2;
                  y := ypos - 2;
                end;
            7 : begin
                  x := xpos + 3;
                  y := ypos - 2;
                end;
            8 : begin
                  x := xpos + 3;
                  y := ypos - 1;
                end;
            9 : begin
                  x := xpos + 3;
                  y := ypos;
                end;
            10 : begin
                  x := xpos + 3;
                  y := ypos + 1;
                end;
            11 : begin
                  x := xpos + 3;
                  y := ypos + 2;
                end;
            12 : begin
                  x := xpos + 2;
                  y := ypos + 2;
                end;
            13 : begin
                  x := xpos + 2;
                  y := ypos + 3;
                end;
            14 : begin
                  x := xpos + 1;
                  y := ypos + 3;
                end;
            15 : begin
                  x := xpos;
                  y := ypos + 3;
                end;
            16 : begin
                  x := xpos - 1;
                  y := ypos + 3;
                end;
            17 : begin
                  x := xpos - 2;
                  y := ypos + 3;
                end;
            18 : begin
                  x := xpos - 2;
                  y := ypos + 2;
                end;
            19 : begin
                  x := xpos - 3;
                  y := ypos + 2;
                end;
            20 : begin
                  x := xpos - 3;
                  y := ypos + 1;
                end;
            21 : begin
                  x := xpos - 3;
                  y := ypos;
                end;
            22 : begin
                  x := xpos - 3;
                  y := ypos - 1;
                end;
            23 : begin
                  x := xpos - 3;
                  y := ypos - 2;
                end;
            24 : begin
                  x := xpos - 2;
                  y := ypos - 2;
                end;
         end; (* End of case *)
      until (x in [xmin..xmax]) and (y in [ymin..ymax]) and coord[x,y].Passable;
      
      if coord[x,y].cindex > 0 then (* Occupied. *)
      begin
         ctarget := coord[x,y].cindex;
         if ctarget = 1 then
         begin (* Player is on target square - death. *)
            if Named = false then WriteMessage('The');
            FullName(c);
            WriteMessage('guffaws with glee as he lands on your head and drives his cane through your skull.');
            playerdead := true;
            killer := c
         end
         else
         begin
            if creatureindex[ctarget].Named then
            begin (* Jester tries to land on Wuggy. *)
               if coord[x,y].Visible then
                  WriteMessage('The jester lands on Wuggy''s head.  The dark sorceror roars in fury and sends the jester''s screeching soul into the Abyss.')
                  else WriteMessage('You hear a distant roar and a sudden wail of anguish.');
               CreatureKill(c);
               cleankills := cleankills + 1
            end
            else (* Regular enemy - jester kills. *)
            begin
               if coord[x,y].Visible then
               begin
                  if Named = false then WriteMessage('The');
                  FullName(c);
                  WriteMessage('guffaws with glee as he lands on the');
                  FullName(ctarget);
                  WriteMessage('''s head and drives his cane through it''s skull.')
               end
               else
                  WriteMessage('You hear a distant cracking sound and a gleeful laugh.');
               
               CreatureKill(ctarget);
               cleankills := cleankills + 1;
            end;
         end;
      end; (* End of target on square situations. *)
      
      (* Sets Jester on square if still alive. *)
      if ypos > 0 then
      begin
         xpos := x;
         ypos := y;
         coord[xpos,ypos].cindex := c;
         if coord[xpos,ypos].Visible then
         begin
            GoToXY(xpos,ypos);
            DrawTile(xpos,ypos);
         end;
         if coord[xpos,ypos].tileid > 10 then ActivateTrap(xpos,ypos);
      end;
   end;
end;
   
   

procedure CreatureTurn(c : longint);

begin
   with creatureindex[c] do
   begin
      if jumping = true then jumping := false; (* Jester lands every turn. *)
      (* Check if turn has come up. *)
      if turnwait > 0 then turnwait := turnwait - 1 else
      begin
         if (Symbol = 'W') and (blindcount = 0) then WuggySpecials(c);
         if (ctype = 12) and (blindcount = 0) then MommaSpecial(c);
         if (ctype = 14) and (blindcount = 0) and (coord[xpos,ypos].Visible = true) then PervertSpecial(c);
            (* Special actions for some enemies - only if they're not blind. *)
         CreatureMove(c); (* All creatures move. *)
         turnwait := turndelay;
      end; (* End of turn check *)
   end; (* End of with *)
end;


procedure CreatureKill(c : longint);
(* Removes creature from the game.  Also sets flags on bosskills - important
   for game progression.  Calling procedure is responsible for giving message
   to the player and updating killcount. *)

begin
   with creatureindex[c] do
   begin
      if Named then
      begin
         if Name = 'Dasher' then dasherkilled := true;
         if Name = 'Giant' then giantkilled := true;
         if Name = 'Shadow' then shadowkilled := true;
         if Name = 'Jester' then jesterkilled := true;
         if Name = 'Warlock' then warlockkilled := true;
         if Name = 'Wraith' then wraithkilled := true;
      end;
      coord[xpos,ypos].cindex := 0; (* Remove from map. *)
      ypos := 0; (* Remove from event loop. *)
   end;
end;



(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)
(* Player Action procedures: *)


procedure TrapListUpdate; FORWARD;

procedure GetPower(x,y : longint);
(* Procedure enacted when the player steps on a Gem of Power.
   Increases trap availability and removes gem from playing field. *)
   
begin
   PlaceOneTile(x,y,1); (* Turns tile into floor. *)
   if maxtrap < 8 then maxtrap := maxtrap + 1; (* More traps available! *)
   TrapListUpdate;
   case maxtrap of
      2 : WriteMessage('Holy moly, it''s a Gem of Power! Now you can give those ugly ogres a nasty surprise...');
      3 : WriteMessage('Zoiks, another Gem of Power! Time to add some booming sound to the underground!');
      4 : WriteMessage('Why looky here, a Gem of Power! Them ogres won''t even see you coming.');
      5 : WriteMessage('Great googly-moogly, a Gem of Power! This oughtta help heat things up!');
      6 : WriteMessage('Gods below, a Gem of Power! No one will want to cross paths with you now!');
      7 : WriteMessage('By the stars above, another Gem of Power! But such power... how difficult it is to wield...');
      8 : WriteMessage('A Gem of Power! And what immense power too! You feel you can only get one shot out of this... But can you really hope to use such force unscathed...?');
   end; (* End of case. *)
end;


procedure GetLife(x,y : longint);
(* Procedure to signal ending the game - the gem of life is recovered. *)

begin
   PlaceOneTile(x,y,1); (* Turns tile into floor. *)
   WriteMessage('The holy Gem of Life! Hurrah! Your other gems glow with resurgent power.');
   gotlife := true;
   if warlockkilled = false then WriteMessage('But there''s still Wuggy to deal with...')
   else WriteMessage('But... something doesn''t feel quite right...');
end;


procedure Dig(x,y : longint);
(* Activates when player moves into diggable wall square.
   Must set turnwait on player to 1. *)

begin
   if coord[x,y].Undiggable = false then
   begin
      PlaceOneTile(x,y,1);
      WriteMessage('You start to bore your way through the wall.');
      if creatureindex[1].blindcount = 0 then
      begin
         GoToXY(x,y);
         DrawTile(x,y)
      end;
      turncount := true;
      creatureindex[1].turnwait := 1 + 2*creatureindex[1].turndelay;
      if gotlife = true then creatureindex[1].turnwait := creatureindex[1].turnwait - 1
   end
   else
   begin
      WriteMessage('Oof, this stone is too tough to dig through!');
      turncount := false;
   end;
end;


procedure LayTrap(x,y,trap : longint);
(* Lays a trap of a certain type at a certain location.
   1 - Flasher
   2 - Banger
   3 - Boomer
   4 - Blinder
   5 - KaBoomer
   6 - X-Boomer
   7 - KaoSBoomer
   8 - ULTRABOOMER *)

begin
   turncount := true;
   trapslaid := trapslaid + 1;
   PlaceOneTile(x,y,trap+10); (* Tileid of traps is set to be 10 more than
                                 corresponding activetrap value. *)
   case trap of
      1 : creatureindex[1].turnwait := 0 + creatureindex[1].turndelay;
      2 : creatureindex[1].turnwait := 0 + creatureindex[1].turndelay;
          (* Zero default cooldown turns for Flasher, Banger, Boomer, Blinder. *)
          (* Varied cooldown for others. *)
      3 : creatureindex[1].turnwait := 0 + creatureindex[1].turndelay; 
      4 : creatureindex[1].turnwait := 0 + creatureindex[1].turndelay;
      5 : creatureindex[1].turnwait := 1 + 2*creatureindex[1].turndelay;
      6 : creatureindex[1].turnwait := 1 + 2*creatureindex[1].turndelay;
      7 : creatureindex[1].turnwait := 2 + 3*creatureindex[1].turndelay;
      8 : creatureindex[1].turnwait := 4 + 4*creatureindex[1].turndelay;
   end;
   if gotlife = true then creatureindex[1].turnwait := creatureindex[1].turnwait - 1;
   (* Gem of Life makes trap-laying faster. *)
   WriteMessage('You lay a');
   case trap of
      1 : WriteMessage('Flasher');
      2 : WriteMessage('Banger');
      3 : WriteMessage('Boomer');
      4 : WriteMessage('Blinder');
      5 : WriteMessage('KABOOMer');
      6 : WriteMessage('X-Boomer');
      7 : WriteMessage('KaoSBoomer');
      8 : WriteMessage('ULTRABOOMER');
   end;
   WriteMessage('trap.');
   if trap = 8 then
   begin
      WriteMessage('You feel you may have just violated something very sacred...');
      activetrap := 7;
      maxtrap := 7;
      TrapListUpdate;
      GoToXY(xmax+11,ymin+4);
      write(' ');
      GoToXY(xmax+2,ymin+13);
      write('           ');
      GoToXY(8,78);
      write(' ');
   end;
end;


procedure PlayerMove(x,y : longint);
(* Controls movement of the player from input commands. *)

begin
   if coord[x,y].Passable = false then
   (* Obstacle collision - no turns pass unless digging. *)
   begin
      if activetrap = 0 then Dig(x,y) else
      begin
         if creatureindex[1].blindcount = 0 then
            WriteMessage('You give the wall a sharp glance.')
         else WriteMessage('Oof! You walked right into a wall!');
         turncount := false
      end
   end
   else
   begin
	   if coord[x,y].cindex < 2 then
	   begin  (* Not occupied - simple move. *)
         turncount := true;
         creatureindex[1].turnwait := creatureindex[1].turndelay;
         creatureindex[1].xpos := x;
         creatureindex[1].ypos := y;
	      coord[x,y].cindex := 1; (* Sets player on spot. *)
         if coord[x,y].tileid = 5 then GetPower(x,y);
         if coord[x,y].tileid = 6 then GetLife(x,y)
	   end
      else (* Target square occupied - give Description. *)
      begin
         turncount := false;
         BumpMessage(coord[x,y].cindex);
      end;
   end;
end;


procedure DescendStairs;
(* Initiated by attempt to descend a stairs square. *)

var
   stairsopen : boolean;

begin
   stairsopen := false;
   if D in [1,2,4,6,7,9,11] then stairsopen := true
   else
   begin
      case D of
         3 : if dasherkilled then stairsopen := true;
         5 : if giantkilled then stairsopen := true;
         8 : if shadowkilled then stairsopen := true;
         10 : if jesterkilled then stairsopen := true;
      end; (* Test for boss killed on appropriate levels. *)
   end;
   
   if stairsopen then
   begin
      if creatureindex[1].blindcount = 0 then
      begin
         WipeMap;
         D := D + 1;
         CaveGenerator;
         DrawMap;
         DungeonMessage
      end
      else WriteMessage('You trip up on the steps!');
      turncount := true
   end
   else
   begin
      WriteMessage('Some force is blocking the stairs.');
      turncount := false;
   end;
end;


procedure LastLevel;

begin
   TextColor(15);
   GoToXY(74,3);
   write('[more]');
   messagex := 77;
   messagey := 3;
   WriteMessage('You hear a sudden dark cackle. "Foolish little creature, did you not know I have the powers of necromancy?! Come, I think I shall have a little fun with you..." Suddenly a pit opens beneath you. You fall down!');
   WipeMap;
   D := D + 1;
   CaveGenerator;
   DrawMap;
end;


procedure ScentPlusArm(x,y,xmod,ymod : longint);
(* Called by Mark Scent.  Different xmod/ymod value correspond to different
   directions of the scent spreading. *)

begin
   if coord[x+xmod,y+ymod].Passable then
   begin
      coord[x+xmod,y+ymod].smell := coord[x+xmod,y+ymod].smell + 64;
      if coord[x+2*xmod,y+2*ymod].Passable then
      begin
         coord[x+2*xmod,y+2*ymod].smell := coord[x+2*xmod,y+2*ymod].smell + 32;
         if coord[x+3*xmod-ymod,y+3*ymod-xmod].Passable 
            then coord[x+3*xmod-ymod,y+3*ymod-xmod].smell := coord[x+3*xmod-ymod,y+3*ymod-xmod].smell + 16;
         if coord[x+3*xmod,y+3*ymod].Passable 
            then coord[x+3*xmod,y+3*ymod].smell := coord[x+3*xmod,y+3*ymod].smell + 16;
         if coord[x+3*xmod+ymod,y+3*ymod+xmod].Passable 
            then coord[x+3*xmod+ymod,y+3*ymod+xmod].smell := coord[x+3*xmod+ymod,y+3*ymod+xmod].smell + 16;
      end;
   end;
end;

procedure ScentXArm(x,y,xmod,ymod : longint);
(* Called by Mark Scent.  Different xmod/ymod value correspond to different
   directions of the scent spreading. *)
   
begin
   if coord[x+xmod,y+ymod].Passable then
   begin
      coord[x+xmod,y+ymod].smell := coord[x+xmod,y+ymod].smell + 64;
      if coord[x+xmod,y+2*ymod].Passable
         then coord[x+xmod,y+2*ymod].smell := coord[x+xmod,y+2*ymod].smell + 32;
      if coord[x+2*xmod,y+2*ymod].Passable
         then coord[x+2*xmod,y+2*ymod].smell := coord[x+2*xmod,y+2*ymod].smell + 24;
      if coord[x+2*xmod,y+ymod].Passable
         then coord[x+2*xmod,y+ymod].smell := coord[x+2*xmod,y+ymod].smell + 32;
   end;
end;
   
procedure MarkScent(x, y : longint);
(* Increases the smell value of the squares surrounding the player to allow
   nearby enemies to find him.  Also decreases the smell value every 5 turns
   so that running away and hiding is possible. *)
   
var
   xmod, ymod, a, b : longint;
   
begin
   (* Reduces leftover scent every second turn. *)
   if turns mod 2 = 0 then
   begin
      for b := ymin to ymax do
         for a := xmin to xmax do
            if coord[a,b].smell > 0 then coord[a,b].smell := trunc(coord[a,b].smell / 2);
   end;
   (* Doing it every second turn means that there is a little bit of lag in the
      ogres path-finding, which has the effect of making them look quite dumb
      and slow at grabbing the player - a nice side-effect.  This 'feature'
      disappears in hardcore mode. *)

   coord[x,y].smell := coord[x,y].smell + 200; (* Player's position. *)

   (* Area around player is split into 8 arms and smell is spread along them.
      Walls stop the smell from spreading. *)
   ScentPlusArm(x,y,0,1); (* Bottom Arm. *)
   ScentPlusArm(x,y,1,0); (* Right Arm. *)
   ScentPlusArm(x,y,0,-1); (* Top Arm. *)
   ScentPlusArm(x,y,-1,0); (* Left Arm. *)
   ScentXArm(x,y,1,1); (* Bottom-Right Arm. *)
   ScentXArm(x,y,-1,1); (* Bottom-Left Arm. *)
   ScentXArm(x,y,-1,-1); (* Top-Left Arm. *)
   ScentXArm(x,y,1,-1); (* Top-Right Arm. *)
end;


(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)
(* Status Bar Procedures: *)


procedure StatusBarUpdate; FORWARD;

procedure StatusBar;
(* Redraws status bar on right of screen. *)

var
   y : longint;

begin
   (* Draws border around status box. *)
   TextColor(9); (* Bright blue. *)
   GoToXY(xmax+1,ymin);
   write('+-----------+');
   for y := (ymin + 1) to (ymax - 1) do
   begin
      GoToXY(xmax+1,y);
	   write('|');
	   GoToXY(80,y);
	   write('|');
	end;
	GoToXY(xmax+1,ymax);
	write('+-----------+');
   TrapListUpdate;
	StatusBarUpdate;
end;


procedure TrapListUpdate;
(* Updates trap list when new traps discovered and when switching active
   traps for use. *)

var
   i : longint;

begin
   (* Gem symbol line: *)
   i := 0; (*makes copy-paste code easier*)
   GoToXY(xmax+3,ymin+4);
   
   if activetrap = i then TextColor(6) else TextColor(8);
   if i <= maxtrap then write('*');
   i := i + 1;
   
   if activetrap = i then TextColor(11) else TextColor(8);
   if i <= maxtrap then write('*');
   i := i + 1;
   
   if activetrap = i then TextColor(4) else TextColor(8);
   if i <= maxtrap then write('*');
   i := i + 1;
   
   if activetrap = i then TextColor(12) else TextColor(8);
   if i <= maxtrap then write('*');
   i := i + 1;
   
   if activetrap = i then TextColor(15) else TextColor(8);
   if i <= maxtrap then write('*');
   i := i + 1;
   
   if activetrap = i then TextColor(14) else TextColor(8);
   if i <= maxtrap then write('*');
   i := i + 1;
   
   if activetrap = i then TextColor(1) else TextColor(8);
   if i <= maxtrap then write('*');
   i := i + 1;
   
   if activetrap = i then TextColor(5) else TextColor(8);
   if i <= maxtrap then write('*');
   i := i + 1;
   
   if activetrap = i then TextColor(2) else TextColor(8);
   if i <= maxtrap then write('*');

   
   (* Trap list: *)
   i := 0; (*makes copy-paste code easier*)
   GoToXY(xmax+2,ymin+5+i);
   if activetrap = i then TextColor(6) else TextColor(8);
   if i <= maxtrap then write('Digger');
   i := i + 1;

   GoToXY(xmax+2,ymin+5+i);
   if activetrap = i then TextColor(11) else TextColor(8);
   if i <= maxtrap then write('Flasher');
   i := i + 1;

   GoToXY(xmax+2,ymin+5+i);
   if activetrap = i then TextColor(4) else TextColor(8);
   if i <= maxtrap then write('Banger');
   i := i + 1;

   GoToXY(xmax+2,ymin+5+i);
   if activetrap = i then TextColor(12) else TextColor(8);
   if i <= maxtrap then write('Boomer');
   i := i + 1;

   GoToXY(xmax+2,ymin+5+i);
   if activetrap = i then TextColor(15) else TextColor(8);
   if i <= maxtrap then write('Blinder');
   i := i + 1;

   GoToXY(xmax+2,ymin+5+i);
   if activetrap = i then TextColor(14) else TextColor(8);
   if i <= maxtrap then write('KABOOMer');
   i := i + 1;

   GoToXY(xmax+2,ymin+5+i);
   if activetrap = i then TextColor(1) else TextColor(8);
   if i <= maxtrap then write('X-Boomer');
   i := i + 1;

   GoToXY(xmax+2,ymin+5+i);
   if activetrap = i then TextColor(5) else TextColor(8);
   if i <= maxtrap then write('KaoSBoomer');
   i := i + 1;

   GoToXY(xmax+2,ymin+5+i);
   if activetrap = i then TextColor(2) else TextColor(8);
   if i <= maxtrap then write('ULTRABOOMER');
end;  (* End of TrapListUpdate. *)


procedure StatusBarUpdate;
(* Updates turn and level listing. *)

begin
   TextBackground(DfBgC);
   TextColor(9);
   GoToXY(xmax+2,ymin+1);
   write(creatureindex[1].prefix);
   if creatureindex[1].turndelay > 0 then
   begin
      GoToXY(xmax+2,ymin+2);
      TextColor(14);
      write('HARDCORE');
   end;
   TextColor(15);
   GoToXY(xmax+2,ymin+15);
   write('Kills: ',kills);
   GoToXY(xmax+2,ymin+17);
   write('T: ',turns,' ');
   GoToXY(xmax+2,ymin+19);
   write('Cave ',D,' ')
end;


procedure InfoScreen;
(* Gives basic game and trap instructions. *)

begin
   TextColor(9);
   TextBackground(0);
   ClrScr;
   GoToXY(33,2);
   writeln('TOBY THE TRAPPER');
   TextColor(7);
   writeln;
   writeln('Move with the numpad or vi-keys');
   writeln('   (Toby moves twice as fast as most enemies, unless on Hardcore mode)');
   TextColor(15);
   write(' *');
   TextColor(7);
   write(' or ');
   TextColor(15);
   write('t  ');
   TextColor(7);
   writeln(': Lay a trap of the selected type (may take several turns)');
   TextColor(15);
   write(' +');
   TextColor(7);
   write(' and ');
   TextColor(15);
   write('-');
   TextColor(7);
   write(' : Switch trap selection (or ');
   TextColor(15);
   write('[ ');
   TextColor(7);
   write('and ');
   TextColor(15);
   write(']');
   TextColor(7);
   writeln(')');
   TextColor(15);
   write(' >');
   TextColor(7);
   write(' or ');
   TextColor(15);
   write('0');
   TextColor(7);
   writeln('  : Descend stairs');
   TextColor(15);
   write('    Q    ');
   TextColor(7);
   writeln(': Quit');
   writeln;
   writeln('   Digger (2 turns) - burrows through most walls - move into wall to use');
   TextColor(11);
   write(' ^');
   TextColor(7);
   writeln(' Flasher (1 turn) - stuns a single enemy');
   TextColor(4);
   write(' ^');
   TextColor(7);
   writeln(' Banger (1 turn) - burns a single enemy');
   TextColor(12);
   write(' ^');
   TextColor(7);
   writeln(' Boomer (1 turns) - fireballs a small area');
   TextColor(15);
   write(' ^');
   TextColor(7);
   writeln(' Blinder (1 turns) - blinds over a small area');
   TextColor(14);
   write(' ^');
   TextColor(7);
   writeln(' KABOOMer (2 turns) - blazes a large circle');
   TextColor(1);
   write(' ^');
   TextColor(7);
   writeln(' X-Boomer (2 turns) - shoots torrents of flame in 4 directions');
   TextColor(5);
   write(' ^');
   TextColor(7);
   writeln(' KaoSBoomer (3 turns) - incinerates over a large random area');
   TextColor(2);
   write(' ^');
   TextColor(7);
   writeln(' ULTRABOOMER (5 turns) - annihilates *everything* in concentric circles');
   writeln('      of prime distance from the trap.');
   writeln('      One use only.  (Caution: Use with extreme care!)');
   writeln;
   writeln('Note that Toby will not set off traps himself, but can get caught up in');
   write('traps set off around him.');
   ReadKey;
   ClrScr;
   DrawMap;
   StatusBar;
end;


procedure DeathScreen;

begin
   TextColor(12);
   TextBackground(0);
   ClrScr;
   
   GoToXY(34,1);
   writeln('R     I     P');
   TextColor(7);
   writeln;
   
   if killer = 0 then
      writeln('  Toby the Trapper was killed by some unknown force (possibly a bug).');
   
   if killer = 1 then
   begin
      writeln('  Toby the Trapper was horribly incinerated by one of his own traps.');
      writeln('  Clever, eh?');
   end;
      
   if killer > 1 then
   begin
      with creatureindex[killer] do
      begin
         write('  Toby the Trapper was ');
         case Symbol of
            'O' : write('battered and eaten by ');
            'D' : write('sliced to pieces by ');
            'G' : write('squashed into a pulp by ');
            '.' : write('slaughtered by ');
            'J' : write('butchered by ');
            'W' : write('obliterated by ');
            'B' : write('clawed apart by ');
         end;
         if Named then
            writeln(prefix,' the ',Name,'.')
         else
            writeln('a ',prefix,' ',Name,'.');
         if creatureindex[1].turndelay = 1 then
            writeln('  You played on Hardcode mode, what did you expect?');
         if wizardmode = true then writeln('  You died in Wizard Mode? How pathetic...');
      end; (* End of with. *)
   end; (* End of killed by monster. *)
   
   if killer = -1 then
      writeln('  Toby the Trapper was utterly annihilated by the ULTRABOOMER.');
   
   if killer = -2 then
   begin
      writeln('  Toby the Trapper destroyed the Gem of Life with the ULTRABOOMER, dying in');
      writeln('  the process. The shock of the ULTRABOOMER''s explosion sent quakes thoughout');
      writeln('  the nearby region, decimating his home town and killing thousands of his');
      writeln('  people. His name was cursed forever.');
      writeln('  Wow, you really screwed this one up, didn''t you?');
   end;
   
   writeln;
   writeln('  You reached level ',D,' of Wuggy''s Lair.');
   writeln;
   writeln('  You killed ',kills,' monsters with your traps.');
   writeln('  ',cleankills,' other monsters were indirectly killed by you.');
   writeln('  Bosses defeated:');
   if dasherkilled then write('    Danny the Dasher') else write('    None!');
   if giantkilled then writeln('             Gordon the Giant');
   if shadowkilled then write('    Sally the Shadow');
   if jesterkilled then writeln('             Jack the Jester');
   if warlockkilled then write('    Wuggy the Warlock');
   if wraithkilled and (D = 16)
      then write(', and his undead incarnation, Wuggy the Wraith');
   writeln;
   if warlockkilled then writeln('  Pity you''re too dead to celebrate, eh?');
   
   writeln;
   writeln('  Turns taken: ',turns);
   writeln('  Traps laid: ',trapslaid);
   writeln('  Traps detonated: ',detonated);

   GoToXY(3,23);
   writeln('(This information is not written to any file - if you want to keep a record');
   write('   then please copy the text into another program.)');
   
   
end;


procedure VictoryScreen;

begin
   if kills > 250 then victorytype := 1;
   if kills > 500 then victorytype := 2;
   if ((kills < 500) and (cleankills > trunc(kills/2))) then victorytype := 3;
   if kills < 20 then victorytype := 4;
   TextColor(7);
   TextBackground(0);
   ClrScr;
   GoToXY(1,2);
   
   if ultraused then
   begin
      writeln(' You return to Turgylton a hero for your daring and cunning in defeating Wuggy');
      writeln(' the Warlock and retrieving the sacred Gem of Life, but some of the elders');
      writeln(' seem reserved in their praise.');
      writeln;
      writeln(' Celebrations begin as your people rejoice at having their town saved.');
      writeln(' However the festivities are cut short when the ground is suddenly torn open.');
      writeln(' The earthquake devastates your town and other areas, killing thousands.');
      writeln;
      writeln(' The elders investigate the cause and realise that it was your use of the');
      writeln(' ULTRABOOMER that started the quake.  You are put on trial for your actions.');
      writeln(' Some call for your blood, but the elders show restraint in simply sentencing');
      writeln(' you to exile.  You are banished from Turgylton forever.');
      writeln;
      
      case victorytype of
         0 : begin
                writeln(' You wander lost for a long time, living in woods and caves, feeling dejected');
                writeln(' and alone.  Eventually you settle in a disused mine where you try to dig away');
                writeln(' hoping to find old treasures or hidden gems.  You chase away anyone that');
                writeln(' comes near the area, and become known as a mad old hermit.  After a few years');
                writeln(' you become ill and die, alone and forgotten in a dark cave.');
             end;
         1 : begin
                writeln(' Feeling angry and dejected you join a nearby bandit gang and begin raiding');
                writeln(' the local territories for loot.  After some months out of greed you murder');
                writeln(' the leader and take over the gang yourself.  You lead the group to become');
                writeln(' more boisterous and seek bigger rewards.  You gain many riches, and take many');
                writeln(' women by force, but always you want more power.  Eventually you die in battle');
                writeln(' trying to raid a strong city, and your body is hung over the city gates as a');
                writeln(' warning to any who would try the same.  Your name is never remembered.');
             end;
         2 : begin
                writeln(' Furious with rage you storm away from the town with great haste.  You seek');
                writeln(' out new powers, and through deception and force you begin to learn new arcane');
                writeln(' abilities.  In a desperate desire for further strength you turn to the dark');
                writeln(' arts, learning much of necromancy and unholy evocations.  You begin openly');
                writeln(' attacking travellers and villages, and over time build up an undead army with');
                writeln(' which to terrorise the region.  One dark night you return to Turgylton with');
                writeln(' your undead legions and destroy the town, turning all the townspeople into');
                writeln(' your undead slaves.  You turn the area into a place of far more dread and');
                writeln(' evil than Wuggy the Warlock ever managed.  To this day people still shudder');
                writeln(' when they hear the dread legends of Toby the Terrible.');
             end;
         3 : begin
                writeln(' You scamper out of town quick before the elders change their mind and ask for');
                writeln(' your head.  Heading to the nearest city you engage in a life of crime to');
                writeln(' survive.  Ever greedy for new spoils you steal from rich and poor alike,');
                writeln(' building up a large collection of treasure.  One day you are caught trying to');
                writeln(' steal the sacred Gem of Infinity in the Temple to Elyshalia.  You are put in');
                writeln(' prison, where you remain to the end of your days.');
             end;
         4 : begin
                writeln(' You leave town and begin to wander the countryside in grief.  In attempt to');
                writeln(' pay penance for your sins you try to help out travellers and do good deeds in');
                writeln(' towns you pass through.  After many years a reputation builds up about a');
                writeln(' wandering gnome who helps those in need.  You always remain annonymous');
                writeln(' though, and accept no reward for aid given to others.  Eventually, old and');
                writeln(' weary, you pass away on a disused road.  Some travellers find you, and');
                writeln(' knowing the tales of the wandering gnome they bury you on a low hill nearby.');
                writeln(' Flowers bloom over your grave and the place becomes known as a sacred spot');
                writeln(' where people go for blessing.  Your name is never known, but you are always');
                writeln(' remembered.');
             end;
      end (* End of case. *)
   end
   else if maxtrap = 8 then
   begin
      writeln(' You return to Turgylton a glorious hero and are praised by all the elders for');
      writeln(' your daring and cunning in defeating Wuggy the Warlock and retrieving the');
      writeln(' sacred Gem of Life.');
      writeln;
      writeln(' The celebrations go on for days, and it is unanimously agreed that you should');
      writeln(' become the new town leader!  You are inducted in a ceremony of great pomp and');
      writeln(' circumstance, and move into the town hall with many of the prettiest gnome');
      writeln(' maidens.');
      writeln;
      
      case victorytype of
         0 : begin
                writeln(' You help rebuild your town after the ogre attacks and return the sense of');
                writeln(' normality to life.  Soon your town becomes happy again, and you manage to');
                writeln(' hold off future attacks and threats thanks to the gems of power.  After a');
                writeln(' few years you retire and lead a quiet life, always remembered as the');
                writeln(' saviour of Turgylton.');
             end;
         1 : begin
                writeln(' You help rebuild your town greatly after the ogre attacks and secure it');
                writeln(' against any future invasion.  Turgylton becomes a strong and independant');
                writeln(' town, and grows over the years as other gnomes come to live there seeking a');
                writeln(' safe haven in a dangerous world.  Eventually you retire, and are remembered');
                writeln(' ever after as a great warrior and shrewd tactician.');
             end;
         2 : begin
                writeln(' With the gems of power in your possession you quickly unite the nearby');
                writeln(' gnomish tribes under your rule.  You demand a tithe for your protection and');
                writeln(' forge a great army.  You defeat your nearby enemies with ruthless efficiency');
                writeln(' and soon go about securing lands further away.  Under a banner of blood your');
                writeln(' armies march through the continent, conquering all in their path.  Eventually');
                writeln(' you die in your imperial harem, surrounded by riches and spoils from all the');
                writeln(' lands.  Your empire soon collapses after your death, but you are remembered');
                writeln(' ever after as Toby the Glorious.');
             end;
         3 : begin
                writeln(' With clever and tactful diplomacy you gain the help of nearby tribes to');
                writeln(' rebuild your town even greater than it was before.  Soon after you use this');
                writeln(' alliance to improve trade relations and encourage cooperation, though the');
                writeln(' benefit always seems to be more to your own.  Your enemies have a habit of');
                writeln(' disappearing suddenly, but no blame can ever be laid on you.  Your town');
                writeln(' prospers and grows and your influence spreads.  Eventually you retire very');
                writeln(' rich and very happy, and you are remembered as a subtle and cunning leader.');
             end;
         4 : begin
                writeln(' You immediately begin work on repairing your town and healing the grievous');
                writeln(' wounds caused by the recent ogre attacks.  Your compassion and wisdom become');
                writeln(' well-known, and visitors come from all the lands to hear your sayings.  You');
                writeln(' spend the rest of your life promoting peace and cooperation and eventually');
                writeln(' pass away of old age.  Your are enshrined as a saint, and gain followers the');
                writeln(' world over.  People worship your name, and you become known as Toby the');
                writeln(' Messiah.');
             end;
      end (* End of case. *) 
   end
   else
   begin
      writeln(' You return to Turgylton a hero and are praised by the elders for your daring');
      writeln(' and cunning in defeating Wuggy the Warlock and retrieving the sacred Gem of');
      writeln(' Life.');
      writeln; 
      writeln(' The celebrations go on for days, and the people decide to erect a statue in');
      writeln(' your honor!  You are given a new house and many rewards for your endeavours.');
      writeln;
      
      case victorytype of
         0 : begin
                writeln(' You settle down and return to your life of mining.  It''s hard work, but the');
                writeln(' rewards are good, and you become known as one of the best gem-miners in the');
                writeln(' town.  You get married to a good wife and have many children and grand-');
                writeln(' children.  Eventually you pass away in peace, surrounded by many loving');
                writeln(' family and friends.');
             end;
         1 : begin
                writeln(' As champion of the town you decide to dedicate your life to its further');
                writeln(' protection.  You form a small militia with some of the young men of the town');
                writeln(' and help secure the town against future attacks.  You are honoured for your');
                writeln(' continual help in protecting the town, and become known as a strong hero.');
                writeln(' Eventually you die of old age, and are given a great funeral by your people.');
                writeln(' The name of Toby the Trapper is ever remembered as the saviour of Turgylton.');
             end;
         2 : begin
                writeln(' You find it hard to adjust to normal life in town after your adventures.  The');
                writeln(' time you spent battling Wuggy''s forces has left you changed.  The townspeople');
                writeln(' seem to sense this too, and though they''re very respectful of you they also');
                writeln(' keep their distance.  Eventually you leave town one night and never return.');
                writeln(' After a while reports come back of a gnome of immense bravery and power who');
                writeln(' roams the lands looking for adventure.  In time legends begin to grow, and');
                writeln(' all the people come to hear the name of the great adventurer, Toby the');
                writeln(' Trapper.');
             end;
         3 : begin
                writeln(' You decide to start up your own gem merchanting business, and quickly become');
                writeln(' very successful.  Your clever and subtle manipulation allows you to go on to');
                writeln(' take over many of your rivals (some talk of foul play, but your every action');
                writeln(' was legitimate) and build a huge emporium.  You gain a seat on the town');
                writeln(' council and become very influential.  Eventually you die fat and happy at a');
                writeln(' grand old age, very rich and very powerful.');
             end;
         4 : begin
                writeln(' After saving the town you put great effort into helping all those who have');
                writeln(' suffered from the ogres attacks.  You join the town''s temple and help to heal');
                writeln(' the wounded and tend to the griefstricken.  Your unbounded compassion is');
                writeln(' known to all, and though many offer riches in reward for your good work you');
                writeln(' refuse them all.  You eventually pass away and are buried in the town''s');
                writeln(' temple.  In time you become known as Saint Toby of Turgylton.');
             end;
      end; 
   end;
   
   ReadKey;
   TextColor(15);
   ClrScr;
   GoToXY(33,2);
   writeln('CONGRATULATIONS!');
   
   TextColor(7);
   writeln;
   writeln('  Toby the Trapper defeated Wuggy the Wraith and rescued the Gem of Life!');
   
   (* Insert death variances. *)
   if ultraused then
   begin
      writeln('  He was banished from his town for using the dreaded ULTRABOOMER and became');
      case victorytype of
         0 : writeln('  a mad old hermit.');
         1 : writeln('  a cruel bandit.');
         2 : writeln('  the dark wizard, Toby the Terrible.');
         3 : writeln('  a master thief.');
         4 : writeln('  a repentant wanderer of legend.');
      end
   end
   else if maxtrap = 8 then
   begin
      write('  He was declared chief of his town and ');
      case victorytype of
         0 : writeln('ruled in peace and prosperity.');
         1 : writeln('became a strong and just leader.');
         2 : writeln('built an empire of blood.');
         3 : writeln('became a diplomatic mastermind.');
         4 : writeln('became a holy Messiah.');
      end
   end
   else
   begin
      write('  He was declared a hero in his town and ');
      case victorytype of
         0 : writeln('lived a happy life as a miner.');
         1 : writeln('became its champion protector.');
         2 : writeln('became an adventurer of legend.');
         3 : writeln('became a rich merchant.');
         4 : writeln('became a holy priest.');
      end
   end;
   
   writeln;
   if creatureindex[1].turndelay = 1 then
   begin
      writeln('  You won on Hardcore Mode!  Uh, wow!  Please send an e-mail to');
      writeln('  darrenjohngrey@hotmail.com to let me know how - I''m honestly amazed!');
   end;
   if wizardmode = true then 
      writeln('  You won on Wizard Mode, which isn''t a great deal to be proud about.');
   
   writeln;
   writeln('  You killed ',kills,' monsters with your traps.');
   writeln('  ',cleankills,' other monsters were indirectly killed by you.');
   writeln('  Bosses defeated:');
   if dasherkilled then write('    Danny the Dasher') else write('    None!');
   if giantkilled then writeln('             Gordon the Giant');
   if shadowkilled then write('    Sally the Shadow');
   if jesterkilled then writeln('             Jack the Jester');
   if warlockkilled then write('    Wuggy the Warlock');
   if wraithkilled and (D = 16)
      then write(', and his undead incarnation, Wuggy the Wraith');
   writeln;
   
   writeln;
   writeln('  Turns taken: ',turns);
   writeln('  Traps laid: ',trapslaid);
   writeln('  Traps detonated: ',detonated);

   GoToXY(3,22);
   writeln('(This information is not written to any file - if you want to keep a record');
   writeln('   then please copy the text into another program.  Victories should be posted');
   write('   to rec.games.roguelike.misc or http://www.gruesomegames.com/blog/)');
end;



(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)


(*** MAIN PROGRAM ***)

begin
   randomize;
   TextColor(DfTxC);
   TextBackground(DfBgC);
   ClrScr;
   cursoroff;
   GoToXY(29,4);
   TextColor(15);
   write('The Grand Adventures of');
   GoToXY(32,6);
   TextColor(9);
   write('TOBY THE TRAPPER');
   GoToXY(21,8);
   TextColor(15);
   write('vs Wuggy the Warlock and his Ugly Ogres');
   GoToXY(27,17);
   TextColor(8);
   write('a roguelike by Darren Grey');
   GoToXY(22,19);
   write('Created for the 7DRL 2010 competition');
   GoToXY(29,20);
   write('Updated September 2010');
   GoToXY(4,23);
   TextColor(7);
   writeln('Press any key to begin.  Press ''x'' for hardcore mode, ''!'' for wizard mode.');
   write('   Press ? in-game for instructions.');
   repeat until KeyPressed;
   while KeyPressed do move := ReadKey; (* Clears key buffer. *)

   TextColor(DfTxC);
   TextBackground(DfBgC);
   ClrScr;
   D := 1;
   SetLength(creatureindex,2);
   creatureindex[1].prefix := 'Toby';
   creatureindex[1].Named := true;
   creatureindex[1].Name := 'Trapper';
   creatureindex[1].Symbol := '@';
   creatureindex[1].Colour := 9;
   creatureindex[1].turnwait := 0;
   creatureindex[1].turndelay := 0;
   if move = 'x' then creatureindex[1].turndelay := 1;
   
   if move = '!' then wizardmode := true;
   if wizardmode then
   begin (* Extra lives and extra sight for wizards. *)
      visrange := 8;
      creatureindex[1].HP := 9
   end
   else
   begin
      visrange := 5;
      creatureindex[1].HP := 0;
   end;
   
   activetrap := 1;
   maxtrap := 1;
   CaveGenerator;  (* Creates new level. *)
   turns := 0;
   ClearMessageBar;
   ClrScr;

   StatusBar;
   LineofSight2(creatureindex[1].xpos,creatureindex[1].ypos,visrange);
   DrawMap;
   DungeonMessage;
   cursoron;
   GoToXY(creatureindex[1].xpos,creatureindex[1].ypos);
      (* First level is created and drawn, cursor on player. *)

   (* Beginning main game loop. *)
   repeat
      if creatureindex[1].turnwait > 0 then
      begin
         cursoroff;
         creatureindex[1].turnwait := creatureindex[1].turnwait - 1;
         turncount := true
      end (* End of trap cooldown. *)
      else
      begin
         coord[creatureindex[1].xpos,creatureindex[1].ypos].cindex := 0;
            (* Removes player from his spot until replaced. *)
         move := ReadKey; (* Waits until key read before action taken. *)
         cursoroff;
         ClearMessageBar; (* Clears message buffer upon keypress. *)

         case move of
         '1','b' : PlayerMove(creatureindex[1].xpos-1,creatureindex[1].ypos+1);
         '2','j' : PlayerMove(creatureindex[1].xpos,creatureindex[1].ypos+1);
         '3','n' : PlayerMove(creatureindex[1].xpos+1,creatureindex[1].ypos+1);
         '4','h' : PlayerMove(creatureindex[1].xpos-1,creatureindex[1].ypos);
         '5','.' : PlayerMove(creatureindex[1].xpos,creatureindex[1].ypos);
         '6','l' : PlayerMove(creatureindex[1].xpos+1,creatureindex[1].ypos);
         '7','y' : PlayerMove(creatureindex[1].xpos-1,creatureindex[1].ypos-1);
         '8','k' : PlayerMove(creatureindex[1].xpos,creatureindex[1].ypos-1);
         '9','u' : PlayerMove(creatureindex[1].xpos+1,creatureindex[1].ypos-1);
         '>','0' : if coord[creatureindex[1].xpos,creatureindex[1].ypos].Symbol = '>' then
                         DescendStairs;
         '+',']' : begin
                      if activetrap = maxtrap then activetrap := 0
                         else activetrap := activetrap + 1;
                      TrapListUpdate;
                   end;
         '-','[' : begin
                      if activetrap = 0 then activetrap := maxtrap
                         else activetrap := activetrap - 1;
                      TrapListUpdate;
                   end;
         '*','t' : begin (* Lays a trap at current position unless Digger is active. *)
                      if activetrap = 0 then WriteMessage('Dig downwards? Don''t be silly...')
                      else
                      begin (* No traps on stairs. *)
                        if coord[creatureindex[1].xpos,creatureindex[1].ypos].tileid <> 3 then
                        LayTrap(creatureindex[1].xpos,creatureindex[1].ypos,activetrap)
                        else WriteMessage('This is no place to lay a trap!');
                      end;
                   end;
         '?' : InfoScreen;
         'Q' : begin
                  quit := true;
                  WriteMessage('Seeya next time!');
               end;
         
         (* Debug commands: *)
         'N' : begin
                  if wizardmode then
                  begin
                     WipeMap;
                     CaveGenerator;
                     DrawMap;
                     DungeonMessage;
                     turncount := true;
                  end;
               end;
         'R' : if wizardmode then RevealMap;
         'D' : if wizardmode then D := D + 1;
         'T' : if wizardmode then if (maxtrap < 8) then maxtrap := maxtrap + 1;
         'W' : if wizardmode = true then wraithkilled := true;
         'K' : if wizardmode = true then kills := kills + 100;
         'C' : if wizardmode = true then cleankills := cleankills + 100;
         'L' : begin
                  if wizardmode = true then
                  begin
                     lifebroken := true;
                     playerdead := true;
                     killer := -2;
                  end;
               end;
         (* No otherwise statement - ignores nonsense commands. *)
         end; (* End of case. *)
      end; (* End of player's turn. *)

      coord[creatureindex[1].xpos,creatureindex[1].ypos].cindex := 1;
            (* Reaffirms player placement on his spot. *)
      
	   if turncount = true then
      begin (* Monster movement time. *)
         turns := turns + 1;
         MarkScent(creatureindex[1].xpos,creatureindex[1].ypos);
            (* Increases smell level around player each turn. *)
         
         if Length(creatureindex) > 2 then
            for i := 2 to (Length(creatureindex) - 1) do
               if (creatureindex[i].ypos > 0) and (playerdead = false)
                  and (wraithkilled = false) then CreatureTurn(i); 
               (* Enemy movements. ypos > 0 excludes the dead.
                  Also stops processing enemy movements when player dies or wins. *)
               (* Problem: keeps processing when Warlock initially defeated. *)
         
         if (turns mod 30 = 0) and (playerdead = false) then
         begin
            repeat
               a := random(xmax - xmin - 7) + xmin + 7;
               b := random(ymax - ymin - 1) + ymin + 1;
            until (coord[a,b].Passable = true)
               and (coord[a,b].cindex = 0)
               and (sqrt((sqr(a-creatureindex[1].xpos))+(sqr(b-creatureindex[1].ypos))) > 8);
            CreatureMake(1,a,b);
            (* Generates ogre on Passable, empty tile a certain min
               distance from the player every 30 turns. *)
         end;
         
         
         if warlockkilled and gotlife and (wraithkilled = false) and (D = 12)
            and (playerdead = false)
         then LastLevel; (* Generates last level under strict conditions. *)
         
         if creatureindex[1].blindcount > 0 then
         begin
            creatureindex[1].blindcount := creatureindex[1].blindcount - 1;
            if creatureindex[1].blindcount = 0 then
            begin
               WriteMessage('You can see again!');
            end;
         end;
         
         if creatureindex[1].blindcount = 0 then
            LineofSight2(creatureindex[1].xpos,creatureindex[1].ypos,visrange);
            (* Only gives sight to those able to see. *)
         
         DrawSurround(creatureindex[1].xpos,creatureindex[1].ypos,visrange);
         
         StatusBarUpdate;
         turncount := false;
      end;
      
      if creatureindex[1].blindcount = 0 then
      begin
         GoToXY(creatureindex[1].xpos,creatureindex[1].ypos);
         cursoron
      end;
         (* Sets cursor flashing on player location - easier to see. *)

   until quit or playerdead or wraithkilled;
      (* Exits command loop - game over. *)

   cursoroff;
   RevealMap;
   TextColor(15);
   GoToXY(74,3);
   write('[more]');
   GoToXY(creatureindex[1].xpos,creatureindex[1].ypos);
   cursoron;
   ReadKey; (* Wait for key read before exit or Victory/Death screen. *)
   cursoroff;
   
   if playerdead then
   begin
      DeathScreen;
      ReadKey (* Wait for key read before exit. *)
   end
   else
   begin
      if wraithkilled then
      begin 
         VictoryScreen;
         ReadKey; (* Wait for key read before exit. *)
      end;
   end;
   
   ClrScr;
end. (*** END OF PROGRAM. ***)