/**********************************************************************************************************

						LL-INTRO.C - Implementation code for the introduction of Lunar Lander


	DESCRIPTION:	Introduction code for the 



	COMMENTS:		Some of the code and ideas in the LANDER project was adapted from Tricks Of
					The Game Programming Gurus.


	Components: 	LL-INTRO.C	-	Implementation file


	Written By:		Joe Lambert
	Date:			06-23-95

	Revisions:
	06-23-95		So it began, stopped with just a ship moving on the screen.
	03-08-96		Added gravity, boundry checking, active fuel and score, crashes,
					number of lives.  Started looking like a real game instead of
					just sprite stuff.
	03-15-96		Broke code into seperate modules for graphics functions and the like.
	03-20-96		Added hooks for keyboard and timer.  Broke code into IO module, out
					of near heap space.	
	10-23-96		Dropped fonts and did clean up.

***********************************************************************************************************/
#include <io.h>
#include <conio.h>
#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <bios.h>
#include <fcntl.h>
#include <memory.h>
#include <math.h>
#include <string.h>
#include <malloc.h>
#include "lander.h"


	Motion_Sprite sprite[NUM_INTRO_SPRITES];	// Define variable 'sprite' of type Motion_Sprite. 


void main(void)
{

	unsigned char ucRedraw;			// Indicates if we need to do a redraw or not.  0 = No, 1 = Yes

	int nShipSpeed = ZERO_SPEED;	// Current speed of ship.
	int nDone = 0;					// Used to terminate main while loop
	int nLandings = 0;				// Number of successful landings
	int nNumLives = LIVES_3;		// How many lives do we have when we start
	int nTimeLoop = 0;

	char szTempString[20];			// Scratch pad

	unsigned short usScoreValue = 0;// Where score will be stored in decimal
	unsigned short usFuelValue;		// Where fuel will be stored in decimal
	long lLoopCount;				// Used for fade out

	int nCycleKeyCount = CYCLE_KEY;			// How many time through the loop until key service
	int nCyclePlatCount = CYCLE_PLATFORM;	// How many time through the loop until platform service
	int nGameState;							// Current state of the game


// ********************************************
// ********** Define local variables **********
// ********************************************
	Pcx_Picture scape;				// Define variable 'scape' of type Pcx_Picture. Entire PCX picture
	Pcx_Picture Sprites;			// Define variable 'Sprites' of type Pcx_Picture. Entire PCX picture
	Motion_Sprite Ship;				// Define variable 'Ship' of type sprite. One variable with many small images
	Motion_Sprite Lives;			// Define variable 'Lives' of type sprite. One variable with many small images
	Motion_Sprite Platform;			// Define variable 'Platform' of type sprite. One variable with many small images

	Text Score;						// Define variable Score of type Text
	Text Title;						// Define variable Title of type Text
	Text Fuel;						// Define variable Fuel of type Text
	Text Success;					// Define variable Success of type Text
	Text Landings;					// Define variable Landings of type Text


// *****************************************
// ********** Initialize globals  **********
// *****************************************
	video_buffer = (unsigned char *)0xA0000000L;	// Video RAM byte pointer
	video_buffer_w= (unsigned short *)0xA0000000L;	// Video RAM word pointer
	ucfpRomCharSet = (unsigned char *)0xF000FA6EL;	// ROM characters 8x8
	uspCurrentTime = (unsigned short *)0x0000046CL;	// pointer to internal
	
	nKeyTable[0] = 0;			// Clear key 0
	nKeyTable[1] = 0;			// Clear key 1
	nKeyTable[2] = 0;			// Clear key 2
	nKeyTable[3] = 0;			// Clear key 3

	nTimeOut = 0;				// Has the clock timed out

	if (!(Double_Buff = (char far *)malloc(SCREEN_WIDTH * SCREEN_HEIGHT + 1)))	// Allocate memory for double buffer
	{
		printf("\ncouldn't allocate double buffer memory");
	}



// ****************************************************
// ********** Video and Timer initialization **********
// ****************************************************


	Configure_Timer(TIMER_18HZ);				// Configure timer and hook interrupt
	Set_Video_Mode(MODE13);							// Configure Video Mode


	Old_Key_Isr = _dos_getvect(KEYBOARD_INT);	// Save old keyboard interrupt vector
	_dos_setvect(KEYBOARD_INT, New_Key_Int);	// Install new ISR





// ****************************************************
// ********** Initialize text to be blitted  **********
// ****************************************************
// ********** Title information initialization ********
// ****************************************************

	strcpy (Title.szString, TITLE_TITLE);			// Initialize title string 

	Title.nMaxStringLength = strlen(Title.szString);// Title length is static
	AllocateTextBack(&Title);						// Call function to allocate memory for background


	Title.nX_Pos = (SCREEN_WIDTH - (strlen(Title.szString) * 8)) / 2;	// Auto center X posistion of title
	Title.nY_Pos = TITLE_Y_POS;						// Y posistion of title
	Title.nColor = BLUE;							// Color of title

		
// ********************************************************
// ********** Success information initialization **********
// ********************************************************

	strcpy (Success.szString, SUCCESS_TITLE);			// Initialize success 

	Success.nMaxStringLength = strlen(Success.szString);// Success title length is static
	AllocateTextBack(&Success);							// Call function to allocate memory for background

	Success.nX_Pos = (SCREEN_WIDTH - (strlen(Success.szString) * 8)) / 2;	// Auto center X posistion of success
	Success.nY_Pos = SUCCESS_Y_POS;						// Y posistion of success
	Success.nColor = WHITE;								// Color of success


// ******************************************************
// ********** Score information initialization **********
// ******************************************************
	Score.nX_Pos = SCORE_X_POS;					// X posistion of score
	Score.nY_Pos = SCORE_Y_POS;					// Y posistion of score
	Score.nColor = WHITE;						// Color of score

	Score.nMaxStringLength = SCORE_LENGTH;		// Assign max string length for memory allocation
	AllocateTextBack(&Score);					// Call function to allocate memory for background

	strcpy(Score.szString, SCORE_TITLE);		// Initialize score string
	itoa( usScoreValue, szTempString, 10);		// Convert score from decimal value to ASCII string
	strcat(Score.szString, szTempString);		// Build score string

// *****************************************************
// ********** Fuel information initialization **********
// *****************************************************
	usFuelValue = FULL_TANK;					// Start with a full tank of petrol

	Fuel.nX_Pos = FUEL_X_POS;					// X posistion of fuel
	Fuel.nY_Pos = FUEL_Y_POS;					// Y posistion of fuel
	Fuel.nColor = GREEN;						// Color of fuel

	Fuel.nMaxStringLength = FUEL_LENGTH;		// Assign max string length for memory allocation
	AllocateTextBack(&Fuel);					// Call function to allocate memory for background

	strcpy( Fuel.szString, FUEL_TITLE);			// Begin to build fuel string
	itoa( usFuelValue, szTempString, 10);		// Convert fuel from decimal value to ASCII string
	strcat(Fuel.szString, szTempString);		// Build fuel string



// ***************************************************
// ********** Load the landscape imagery **********
// ***************************************************

	PCX_Init(&scape);							// Alocate memory for landscape PCX file
	PCX_Load(SCAPE_FILENAME, &scape,1);			// Load the landscape PCX file and the palette
	PCX_Show_Buffer(&scape);					// Copy landscape to VIDEO memory. DISPLAY LANDSCAPE
	Free_PCX_Mem(&scape);						// Free memory used for landscape PCX file


	Intro();


// ******************************************
// ********** Bit Blit information **********
// ******************************************

	SaveTextBackground(&Title);					// Save background of title
	SaveTextBackground(&Success);				// Save background of success
	SaveTextBackground(&Score);					// Save background of score
	SaveTextBackground(&Fuel);					// Save background of fuel

	Blit_Text(&Score);							// Bit blit score
	Blit_Text(&Fuel);							// Bit blit fuel


// *****************************************
// ********** Load player imagery **********
// *****************************************

	PCX_Init(&Sprites);							// Alocate memory for spaceship PCX file
	PCX_Load(SPRITE_FILENAME, &Sprites,0);		// Load the Sprites PCX file

	Sprite_Init(&Ship,SHIP_START_X,SHIP_START_Y, SPRITE_WIDTH, SPRITE_HEIGHT);	// Initialize 'Ship' sprite
	Sprite_Init(&Lives,LIVES_X,LIVES_Y, SPRITE_WIDTH, SPRITE_HEIGHT);			// Initialize 'Lives' sprite
	Sprite_Init(&Platform,PLATFORM_X,PLATFORM_Y, SPRITE_WIDTH, SPRITE_HEIGHT);	// Initialize 'Platform' sprite


	PCX_Grap_Bitmap(&Sprites,&Ship,0,0,0);		// Ship with no engines fired
	PCX_Grap_Bitmap(&Sprites,&Ship,1,1,0);		// Ship with both engines fired
	PCX_Grap_Bitmap(&Sprites,&Ship,2,2,0);		// Ship with left engine fired
	PCX_Grap_Bitmap(&Sprites,&Ship,3,3,0);		// Ship with right engine fired

	PCX_Grap_Bitmap(&Sprites,&Ship,4,4,0);		// First frame of crash
	PCX_Grap_Bitmap(&Sprites,&Ship,5,5,0);		// Second frame of crash
	PCX_Grap_Bitmap(&Sprites,&Ship,6,6,0);		// Third frame of crash
	PCX_Grap_Bitmap(&Sprites,&Ship,7,7,0);		// Fourth frame of crash

	PCX_Grap_Bitmap(&Sprites,&Lives,0,8,0);		// One life
	PCX_Grap_Bitmap(&Sprites,&Lives,1,9,0);		// Two lives
	PCX_Grap_Bitmap(&Sprites,&Lives,2,10,0);	// Three lives

	PCX_Grap_Bitmap(&Sprites,&Platform,0,0,1);	// Frame one of platform
	PCX_Grap_Bitmap(&Sprites,&Platform,1,1,1);	// Frame two of platform
	PCX_Grap_Bitmap(&Sprites,&Platform,2,2,1);	// Frame three of platform
	PCX_Grap_Bitmap(&Sprites,&Platform,3,3,1);	// Frame four of platform

	Free_PCX_Mem(&Sprites);						// Free the memory used for sprites PCX file

/*************************************************
			Calling conventions

	PCX_Grap_Bitmap(Pcx_Picture_Ptr , Motion_Sprite_Ptr, int, int, int)

	1st param = Pointer to image to grab from
	2nd param = Pointer to sprite where image goes
	3rd param = Current sprite frame
	4th param = X posistion to grab from	
	5th param = Y posistion to grab from	


**************************************************/
	
	Behind_Sprite(&Lives);						// Save the background before we draw a sprite
	Behind_Sprite(&Ship);						// Save the background before we draw a sprite
	Behind_Sprite(&Platform);					// Save the background before we draw a sprite

	Lives.nCurrFrame = nNumLives;				// Set which frame (or sprite) to draw 
	Platform.nDirection = RIGHT;				// What direction is platform moving

	Draw_Sprite(&Ship);							// Draw the shaceship sprite
	Draw_Sprite(&Lives);						// Draw the Lives sprite
	Draw_Sprite(&Platform);						// Draw the Lives sprite


// *******************************
// ********** Main loop **********
// *******************************

	while (!nDone)								// While nDone is not 1
	{

		Get_Input();							// Call function to get keys pressed
	    if (nRawKey == 1)						// End if user is exiting
	       {
		       nDone = 1;						// Set nDone to 1
	       }


// *************************************
// ********** Decode Key hits **********
// *************************************
			
	// ***** Moving LEFT *****
	
	    if (nKeyTable[INDEX_LEFT])				// Was the move LEFT command given
		{
			if (Ship.nX > 0)					// Make sure we're not all the way to the left
			{
				Ship.nX_Old = Ship.nX;			// Save Ship.nX so we can restore the background
				Ship.nX--;						// Decrement the Ship's location to the LEFT
				ucRedraw = YES;					// Set ucRedraw so we will redraw the Ship
		
			}
			Ship.nCurrFrame = SHIP_FRAME_THREE;	// Set which frame (or sprite) to redraw the Ship with
		}
		
	// ***** Moving RIGHT *****
	
	    else if (nKeyTable[INDEX_RIGHT])		// Was the move RIGHT command given
		{
			if (Ship.nX < (SCREEN_WIDTH - SPRITE_WIDTH))	// Have we hit the RIGHT wall
			{
				Ship.nX_Old = Ship.nX;			// Save Ship.nX so we can restore the background
				Ship.nX++;						// Increment the ship's location to the RIGHT
				ucRedraw = YES;					// Set ucRedraw so we will redraw the ship
		
			}
			Ship.nCurrFrame = SHIP_FRAME_TWO;	// Set which frame (or sprite) to redraw the ship with
		}
			
		
	// ***** Moving UP with UP key pressed and without UP key pressed *****
	
		if (nShipSpeed > ZERO_SPEED)							// ** Are we moving UP **
		{
			if (nKeyTable[INDEX_UP] && usFuelValue)				// Is the UP key pressed and do we have gas
			{
		
				if (usFuelValue > 0)							// If we still have fuel
				{
					usFuelValue -= 1;							// Decrease amount of fuel
				}
		
		
				if (nShipSpeed < MAX_SPEED)						// Are we at max speed
				{
					nShipSpeed += GRAVITY;
				}
		
	//					if (Ship.nY > 0)								// Make sure we're not all the way UP
	
				if ((Ship.nY - (int) (nShipSpeed / SPEED_DIVIDE)) > 0)	// Make sure we're not all the way UP
				{
					Ship.nY -= (int) (nShipSpeed / SPEED_DIVIDE);		// Move the ship UP
				}
				else
				{
					Ship.nY = 0;
				}										
				Ship.nCurrFrame = SHIP_FRAME_ONE;				// Which frame (or sprite) to redraw the ship with
				ucRedraw = YES;									// Set ucRedraw so we will redraw the ship
			}
			else												// Going UP but UP key NOT pressed
			{
				if (nShipSpeed > MIN_SPEED)						// Are we above MIN speed
				{
					nShipSpeed -= GRAVITY;
		
				}
		
				if ((Ship.nY - (int) (nShipSpeed / SPEED_DIVIDE)) > 0)	// Make sure we're not all the way UP
				{
					Ship.nY -= (int) (nShipSpeed / SPEED_DIVIDE);						// Move the ship UP
				}
				else
				{
					Ship.nY = 0;
				}
		
				Ship.nCurrFrame = SHIP_FRAME_ZERO;				// Set which frame (or sprite) to redraw the ship with
				ucRedraw = YES;									// Set ucRedraw so we will redraw the ship
			} 
		}
	
	// ***** Moving DOWN with UP key pressed and without UP key pressed *****
	
		else if (nShipSpeed <= ZERO_SPEED)						// ** Are we falling **
		{
			if (nKeyTable[INDEX_UP] && usFuelValue)				// Is the UP key pressed and do we have gas
			{
				if (usFuelValue > 0)							// If we still have fuel
				{
					usFuelValue -= 1;							// Decrease amount of fuel
				}
			
				if (nShipSpeed < MAX_SPEED)						// Are we below MAX speed
				{
					nShipSpeed += (int)(GRAVITY / 2);
				}
							
				if (Ship.nY < (SCREEN_HEIGHT - SPRITE_HEIGHT))	// Make sure we're not all the way down
				{
					Ship.nY -= (int) (nShipSpeed / SPEED_DIVIDE);// Move the ship DOWN
				}
							
				Ship.nCurrFrame = SHIP_FRAME_ONE;				// Which frame (or sprite) to redraw the ship with
				ucRedraw = YES;									// Set ucRedraw so we will redraw the ship
			}
			else												// ** Falling and UP key NOT pressed **
			{
				if (nShipSpeed > MIN_SPEED)						// Are we above MIN speed
				{
					nShipSpeed -= GRAVITY;
				}
		
				if (Ship.nY < (SCREEN_HEIGHT - SPRITE_HEIGHT))	// Make sure we're not all the way down
				{
					Ship.nY -= (int) (nShipSpeed / SPEED_DIVIDE);// Move the ship DOWN
		
				}
				Ship.nCurrFrame = SHIP_FRAME_ZERO;				// Set which frame (or sprite) to redraw the ship with
				ucRedraw = YES;									// Set ucRedraw so we will redraw the ship
			} 
		}
	
	

			
// ******************
// ***** REDRAW *****
// ******************

		if (ucRedraw)								// Do we have to redraw
		{
        	
// Are we at the same height as platform surface AND Is our RIGHT past the LEFT side of platform AND 
// Is our LEFT before the RIGHT side of platform

			if ((Ship.nY + SPRITE_HEIGHT) == Platform.nY &&	(Ship.nX + SPRITE_WIDTH) > Platform.nX &&
			Ship.nX < (Platform.nX + SPRITE_WIDTH) )
			{											// ** LANDED **
				nLandings++;							// Increment number of successful landings
				nShipSpeed = ZERO_SPEED;				// Set Speed to zero
	
				if (usScoreValue < MAX_SCORE)			// Make sure we're not at Max score
				{
					usScoreValue += usFuelValue;		// Increase Score
				}
				strcpy( Score.szString, SCORE_TITLE);	// Begin to build Score string
				itoa(usScoreValue, szTempString, 10);	// Convert Score from decimal value to ASCII string
				strcat(Score.szString, szTempString);	// Build Score string

				usFuelValue = FULL_TANK;				// Fuel up
				strcpy( Fuel.szString, FUEL_TITLE);		// Begin to build fuel string
				itoa( usFuelValue, szTempString, 10);	// Convert fuel from decimal value to ASCII string
				strcat(Fuel.szString, szTempString);	// Build fuel string

				Ship.nCurrFrame = SHIP_FRAME_ZERO;		// No engines fired

				Blit_Text(&Success);					// Blit Success
				Blast_Buffer();							// Copy double buffer

				Timer(36);								// Wait 36 clicks

				EraseText(&Success);					// Erase current Success 
				Blast_Buffer();							// Copy double buffer

				Ship.nX = SHIP_START_X;					// Start sprite back at X beginning
				Ship.nY = SHIP_START_Y;					// Start sprite back at Y beginning

				Erase_Sprite(&Ship);					// Erase sprite at old position
				EraseText(&Score);						// Erase current score value
				EraseText(&Fuel);						// Erase current fuel value
	
				Behind_Sprite(&Ship);					// Save the background at new postition
				Blit_Text(&Fuel);						// Blit new fuel value
				Blit_Text(&Score);						// Blit new score value
				Draw_Sprite(&Ship);						// Draw sprite at new position

				Ship.nX_Old = SHIP_START_X;				// Update old X position indicator
				Ship.nY_Old = SHIP_START_Y;				// Update old Y position indicator
				ucRedraw = 0; 							// Clear ucRedraw

				Wait_For_Vsync();						// Wait for verticel sync so we don't get a flicker
				Blast_Buffer();							// Copy double buffer
//				nGameState = LANDED;
			}


// Have we hit bottom OR Is our RIGHT past the LEFT side of the platform AND Is our LEFT before the RIGHT side of
// platform AND Are we below the platform surface
		
			else if ((Ship.nY + SPRITE_HEIGHT) >= SCREEN_HEIGHT || (Ship.nX + SPRITE_WIDTH) > Platform.nX &&
			Ship.nX < (Platform.nX + SPRITE_WIDTH) && (Ship.nY + SPRITE_HEIGHT) > Platform.nY )
			{											// *** CRASH ***

 				Ship.nCurrFrame = SHIP_FRAME_FOUR;		// First stage of crash
				Erase_Sprite(&Ship);					// Erase sprite at old position
				Ship.nX_Old = Ship.nX;					// Update old X position indicator
				Ship.nY_Old = Ship.nY;					// Update old Y position indicator
				Behind_Sprite(&Ship);					// Save the background at new postition
				Draw_Sprite(&Ship);						// Draw sprite at new position
				Wait_For_Vsync();						// Wait for verticel sync so we don't get a flicker
				Blast_Buffer();							// Copy double buffer
				Timer(2);								// Wait n clicks 
	
				Ship.nCurrFrame = SHIP_FRAME_FIVE;		// Second stage of crash
				Erase_Sprite(&Ship);					// Erase sprite at old position
				Behind_Sprite(&Ship);					// Save the background at new postition
				Draw_Sprite(&Ship);						// Draw sprite at new position
				Wait_For_Vsync();						// Wait for verticel sync so we don't get a flicker
				Blast_Buffer();							// Copy double buffer
				Timer(2);								// Wait n clicks 
	
				Ship.nCurrFrame = SHIP_FRAME_SIX;		// Third stage of crash
				Erase_Sprite(&Ship);					// Erase sprite at old position
				Behind_Sprite(&Ship);					// Save the background at new postition
				Draw_Sprite(&Ship);						// Draw sprite at new position
				Wait_For_Vsync();						// Wait for verticel sync so we don't get a flicker
				Blast_Buffer();							// Copy double buffer
				Timer(2);								// Wait n clicks 
	
				Ship.nCurrFrame = SHIP_FRAME_SEVEN;		// Forth stage of crash
				Erase_Sprite(&Ship);					// Erase sprite at old position
				Behind_Sprite(&Ship);					// Save the background at new postition
				Draw_Sprite(&Ship);						// Draw sprite at new position
				Erase_Sprite(&Ship);					// Erase sprite at old position
				Wait_For_Vsync();						// Wait for verticel sync so we don't get a flicker
				Blast_Buffer();							// Copy double buffer
	
				if ( nNumLives-- > 0)					// ** Not out of lives **
				{
					nShipSpeed = ZERO_SPEED;			// Set Speed to zero
					Timer(30);							// Wait n clicks 
//					Erase_Sprite(&Ship);				// Erase sprite at old position
		
					Ship.nX = SHIP_START_X;				// Start sprite back at beginning
					Ship.nY = SHIP_START_Y;				// Start sprite back at beginning
					Ship.nX_Old = SHIP_START_X;			// Update old X position indicator
					Ship.nY_Old = SHIP_START_Y;			// Update old Y position indicator
		
					Erase_Sprite(&Lives);				// Erase sprite at old position

					Lives.nCurrFrame = nNumLives;		// Set which frame (or sprite) to draw 
					Behind_Sprite(&Lives);				// Save the background before we draw a sprite
					Draw_Sprite(&Lives);				// Draw the sprite
		
					Ship.nCurrFrame = SHIP_FRAME_ZERO;	// Set which frame (or sprite) to draw 
					Behind_Sprite(&Ship);				// Save the background at new postition
					Draw_Sprite(&Ship);					// Draw sprite at new position
		
					Wait_For_Vsync();					// Wait for verticel sync so we don't get a flicker
					Blast_Buffer();						// Copy double buffer
		
					ucRedraw = 0; 						// Clear ucRedraw
				}
				else									// ** Out of lives **
				{
					nDone = 1;							// Set nDone to 1
				}
//				nGameState = CRASH;
			}

			else										// ** No landing, no crash, just flying **
			{

// **** Update Fuel ****

				strcpy( Fuel.szString, FUEL_TITLE);		// Begin to build fuel string
				itoa( usFuelValue, szTempString, 10);	// Convert fuel from decimal value to ASCII string
				strcat(Fuel.szString, szTempString);	// Build fuel string
				if (usFuelValue < 250)					// Is fuel getting low, if not leave it green
				{
					if (usFuelValue < 100)				// If fuel is getting REAL low
					{
						Fuel.nColor = RED;				// Fuel color turns RED
					}
					else
					{
						Fuel.nColor = YELLOW;			// If fuel's just a little low it's YELLOW
					}
				}

// **** Update Platform ****

				if (Platform.nDirection == RIGHT)						// Is platform moving RIGHT
				{
					if (Platform.nX++ == (SCREEN_WIDTH - SPRITE_WIDTH))	// Have we hit the RIGHT wall
					{
						Platform.nDirection = LEFT;						// We hit the RIGHT, change direction
						Platform.nX--;									// Start heading LEFT
					}
					if (Platform.nCurrFrame++ > PLATFORM_FRAME_TWO)		// Are we on the last frame
					{
						Platform.nCurrFrame = PLATFORM_FRAME_ZERO;		// Reset frame
					}
				}
				else													// We're moving LEFT
				{
					if (Platform.nX-- == 0)								// Have we hit the left wall
					{
						Platform.nDirection = RIGHT;					// Start heading RIGHT
						Platform.nX++;									// Move RIGHT one
					}
					if (Platform.nCurrFrame-- == PLATFORM_FRAME_ZERO)	// Are we on the first most frame
					{
						Platform.nCurrFrame = PLATFORM_FRAME_THREE;		// Set frame to last frame, frames go backwards now
					}
				}			
	
				Erase_Sprite(&Platform);				// Erase sprite at old position
				Behind_Sprite(&Platform);				// Save the background at new postition
				Draw_Sprite(&Platform);					// Draw sprite at new position
				Platform.nX_Old = Platform.nX;			// Update old X position indicator
				Platform.nY_Old = Platform.nY;			// Update old Y position indicator				Erase_Sprite(&Ship);					// Erase sprite at old position

				Erase_Sprite(&Ship);				// Erase sprite at old position
				Behind_Sprite(&Ship);					// Save the background at new postition
		
				EraseText(&Fuel);						// Erase current fuel value
				Blit_Text(&Fuel);						// Blit new fuel value
		
				Draw_Sprite(&Ship);						// Draw sprite at new position

				Ship.nX_Old = Ship.nX;					// Update old X position indicator
				Ship.nY_Old = Ship.nY;					// Update old Y position indicator
		
				Wait_For_Vsync();						// Wait for verticel sync so we don't get a flicker
				Blast_Buffer();							// Copy double buffer

				ucRedraw = 0; 							// Clear ucRedraw
//				nGameState = FLYING;					// Set state to FLYING
			}
//			    Delay(6000);							// Wait a bit



/*



			switch (nGameState)
			{
				case FLYING:
					Erase_Sprite(&Ship);					// Erase sprite at old position
					Erase_Sprite(&Platform);				// Erase sprite at old position
					Behind_Sprite(&Platform);				// Save the background at new postition
					Behind_Sprite(&Ship);					// Save the background at new postition
		
					EraseText(&Score);						// Erase current score value
					EraseText(&Fuel);						// Erase current fuel value
		
					Blit_Text(&Score);						// Blit new score value
					Blit_Text(&Fuel);						// Blit new fuel value
		
					Draw_Sprite(&Ship);						// Draw sprite at new position
					Draw_Sprite(&Platform);					// Draw sprite at new position
		
					Platform.nX_Old = Platform.nX;			// Update old X position indicator
					Platform.nY_Old = Platform.nY;			// Update old Y position indicator
		
					Ship.nX_Old = Ship.nX;					// Update old X position indicator
					Ship.nY_Old = Ship.nY;					// Update old Y position indicator
		
//					Blit_Text(&Fuel);						// Blit new fuel value
//					Blit_Text(&Speed);						// Blit new speed value
					ucRedraw = 0; 							// Clear ucRedraw
					break;
				
				case LANDED:
					Erase_Sprite(&Ship);					// Erase sprite at old position
	
					Ship.nX = SHIP_START_X;					// Start sprite back at X beginning
					Ship.nY = SHIP_START_Y;					// Start sprite back at Y beginning
					Ship.nX_Old = SHIP_START_X;				// Update old X position indicator
					Ship.nY_Old = SHIP_START_Y;				// Update old Y position indicator
	
					EraseText(&Score);						// Erase current score value
					EraseText(&Fuel);						// Erase current fuel value
	
					Behind_Sprite(&Ship);					// Save the background at new postition
					Draw_Sprite(&Ship);						// Draw sprite at new position
	
					Blit_Text(&Score);						// Blit new score value
					Blit_Text(&Fuel);						// Blit new fuel value
	
					ucRedraw = 0; 							// Clear ucRedraw
//					Timer(60);								// Wait n clicks (1 click ~= 55 miliseconds)
					break;
				
				case CRASH:
	 				Ship.nCurrFrame = SHIP_FRAME_FIVE;		// First stage of crash
					Erase_Sprite(&Ship);					// Erase sprite at old position
					Behind_Sprite(&Ship);					// Save the background at new postition
					Draw_Sprite(&Ship);						// Draw sprite at new position
					Timer(2);								// Wait n clicks (1 click ~= 55 miliseconds)
	
					Ship.nX_Old = Ship.nX;					// Update old X position indicator
					Ship.nY_Old = Ship.nY;					// Update old Y position indicator
	
					Ship.nCurrFrame = SHIP_FRAME_SIX;		// Second stage of crash
					Erase_Sprite(&Ship);					// Erase sprite at old position
					Behind_Sprite(&Ship);					// Save the background at new postition
					Draw_Sprite(&Ship);						// Draw sprite at new position
					Timer(2);								// Wait n clicks (1 click ~= 55 miliseconds)
	
					Ship.nCurrFrame = SHIP_FRAME_SEVEN;		// Third stage of crash
					Erase_Sprite(&Ship);					// Erase sprite at old position
					Behind_Sprite(&Ship);					// Save the background at new postition
					Draw_Sprite(&Ship);						// Draw sprite at new position
					Timer(2);								// Wait n clicks (1 click ~= 55 miliseconds)
	
					Ship.nCurrFrame = SHIP_FRAME_EIGHT;		// Forth stage of crash
					Erase_Sprite(&Ship);					// Erase sprite at old position
					Behind_Sprite(&Ship);					// Save the background at new postition
					Draw_Sprite(&Ship);						// Draw sprite at new position
					Erase_Sprite(&Ship);					// Erase sprite at old position
	
					if ( --nNumLives >= LIVES_1)			// ** Not out of lives **
					{
						nShipSpeed = ZERO_SPEED;			// Set Speed to zero
						Timer(40);							// Wait n clicks (1 click ~= 55 miliseconds)
						Erase_Sprite(&Ship);				// Erase sprite at old position
		
						Ship.nX = SHIP_START_X;				// Start sprite back at beginning
						Ship.nY = SHIP_START_Y;				// Start sprite back at beginning
						Ship.nX_Old = SHIP_START_X;			// Update old X position indicator
						Ship.nY_Old = SHIP_START_Y;			// Update old Y position indicator
		
						EraseText(&Score);					// Erase current score value
						EraseText(&Fuel);					// Erase current fuel value
						Erase_Sprite(&Lives);				// Erase sprite at old position
	
						Lives.nCurrFrame = nNumLives;		// Set which frame (or sprite) to draw 
	
						Behind_Sprite(&Lives);				// Save the background before we draw a sprite
						Draw_Sprite(&Lives);				// Draw the sprite
		
						Ship.nCurrFrame = SHIP_FRAME_ZERO;	// Set which frame (or sprite) to draw 
						Behind_Sprite(&Ship);				// Save the background at new postition
						Draw_Sprite(&Ship);					// Draw sprite at new position
		
						Blit_Text(&Score);					// Blit new score value
	//					Blit_Text(&Speed);					// Blit new speed value
						Blit_Text(&Fuel);					// Blit new fuel value
		
						ucRedraw = 0; 						// Clear ucRedraw
					}
					else									// ** Out of lives **
					{
						nDone = 1;							// Set nDone to 1
					}
					break;	
	
			}// End switch (nGameState)
		
*/

		}// End if (Redraw)

		while(!nTimeOut);		// Hang out until timer lets us go
		nTimeOut = 0;

	}// End While Loop



// ***** GAME OVER *****


// ***** Blit final scores and info *****

	EraseText(&Score);							// Erase current score value
	EraseText(&Fuel);							// Erase current fuel value
	Erase_Sprite(&Ship);						// Erase ship sprite at old position
	Erase_Sprite(&Lives);						// Erase Lives sprite at old position

	Title.nY_Pos = 60;							// Y posistion of Title
	Title.nColor = BLUE;						// Color of Title
	Blit_Text(&Title);							// Blit Title

	Score.nX_Pos = (SCREEN_WIDTH - (strlen(Score.szString) * 8)) / 2;	// Auto center X posistion of Score
	Score.nY_Pos = 70;							// Y posistion of Score
	Score.nColor = WHITE;						// Color of Score
	Blit_Text(&Score);							// Blit Score

	strcpy( Landings.szString, LANDINGS_TITLE);	// Begin to build Landings string
	itoa( nLandings, szTempString, 10);			// Convert Landings from decimal value to ASCII string
	strcat(Landings.szString, szTempString);	// Build Landings string

	Landings.nX_Pos = (SCREEN_WIDTH - (strlen(Landings.szString) * 8)) / 2;	// Auto center X posistion of Landings
	Landings.nY_Pos = 80;						// Y posistion of Landings
	Landings.nColor = BLUE;						// Color of Landings

	Blit_Text(&Landings);						// Blit new Landings 

	Wait_For_Vsync();							// Wait for verticel sync so we don't get a flicker
	Blast_Buffer();								// Copy double buffer


// ***** Restore interrupt hooks and timer *****

	Reset_Timer();
	_dos_setvect(KEYBOARD_INT, Old_Key_Isr);		// Restore old keyboard ISR


// ***** FREE MEMORY - FREE MEMORY - FREE MEMORY *****

	_ffree(Double_Buff);				// Free memory allocated for double buffer
	Free_Sprite_Mem(&Ship);			// Free memory allocated for ship sprite
	Free_Sprite_Mem(&Lives);		// Free memory allocated for Lives sprite
	Free_Sprite_Mem(&Platform);		// Free memory allocated for platform sprite
	FreeTextMem(&Title);			// Free memory allocated for title text
	FreeTextMem(&Score);			// Free memory allocated for score text
	FreeTextMem(&Success);			// Free memory allocated for success text
	FreeTextMem(&Fuel);				// Free memory allocated for fuel text
	FreeTextMem(&Landings);			// Free memory allocated for landings text

// ***** DID YOU GET IT ALL??? *****



	printf("Hit any key to continue");
	while (!kbhit());
	Set_Video_Mode(TEXT_MODE);			// Set text mode

// Fade out with random pixel placement
	for (lLoopCount=0; lLoopCount<=30000; lLoopCount++,Plot_Pixel_Fast(rand()%320, rand()%200, 0));	// Fade out with random pixel placement

   

} // end main ***** LATER *****






/********************************************************************
	
	Function:
		VOID Configure_Timer(INT)
	
	Description:
		Initialize timer setup and hook timer interrupt.
					
	Returns:
		None
		
	Comments:
  
*********************************************************************/
void Configure_Timer(int nTimer_Setting)
{

	Change_Timer(nTimer_Setting);						// Re-program the timer chip;

	Old_Timer_Isr = _dos_getvect(TIME_KEEPER_INT);	// Save old Timer interrupt vector
	_dos_setvect(TIME_KEEPER_INT, New_Timer_Int);	// Install new ISR


}// End of Blast_Buffer function



/********************************************************************
	
	Function:
		VOID Init_Timer(VOID)
	
	Description:
		Initialize timer setup and hook timer interrupt.
					
	Returns:
		None
		
	Comments:
  
*********************************************************************/
void Reset_Timer(void)
{

	_dos_setvect(TIME_KEEPER_INT, Old_Timer_Isr);	// Restore old Timer ISR

	Change_Timer(TIMER_18HZ);						// Re-program the timer chip;


}// End of Blast_Buffer function






void Intro(void)
{

	int nDone = 0;					// Used to terminate main while loop
	int nTimeLoop;
	int nLoopCount;
	int nRowGrab = 0;
	int nColGrab = 0;
	
	Pcx_Picture Demo_Frames;				// Define variable 'Demo_Frames' of type Pcx_Picture. Entire PCX picture
//	Motion_Sprite sprite[NUM_INTRO_SPRITES];	// Define variable 'sprite' of type Motion_Sprite. 

// ********** Load sprite imagery **********
	PCX_Init(&Demo_Frames);							// Alocate memory for frames PCX file
	PCX_Load(INTRO_SPRITE_FILENAME, &Demo_Frames,0);		// Load the frames PCX file

// ********** Loop through sprite array to initialize ********** 
	for (nLoopCount = 0; nLoopCount < NUM_INTRO_SPRITES; nLoopCount++)
	{
		Sprite_Init(&sprite[nLoopCount],nColGrab * SPRITE_WIDTH ,
		nRowGrab * SPRITE_HEIGHT, INTRO_SPRITE_WIDTH, INTRO_SPRITE_HEIGHT);

		if (nColGrab++ == MAX_COL - 1)
		{
			if (nRowGrab++ == MAX_ROW)
			{
				nRowGrab = 0;
			}
			nColGrab = 0;
		}
	}
// ********** Loop through sprite array to load sprites ********** 
	nRowGrab = 0;
	nColGrab = 0;

	for (nLoopCount = 0; nLoopCount < NUM_INTRO_SPRITES; nLoopCount++)
	{
		PCX_Grap_Bitmap(&Demo_Frames,&sprite[nLoopCount],0,nColGrab,nRowGrab);	// Grab individual sprite images from PCX file.
		if (nColGrab++ == (MAX_COL) - 1)
		{
			if (nRowGrab++ == (MAX_ROW))
			{
				nRowGrab = 0;
			}
			nColGrab = 0;
		}

		 sprite[nLoopCount].nDirection = rand()%8;		// Pick a direction
	}
	Free_PCX_Mem(&Demo_Frames);							// Free the memory used for frames PCX file

// ********** Loop through sprite array to draw sprites ********** 
	for (nLoopCount = 0; nLoopCount < NUM_INTRO_SPRITES; nLoopCount++)
	{
		Behind_Sprite(&sprite[nLoopCount]);		// Save the background before we draw a sprite
		Draw_Sprite(&sprite[nLoopCount]);		// Draw the lives sprite
	}
	Wait_For_Vsync();						// Wait for verticel sync so we don't get a flicker
	Blast_Buffer();							// Copy double buffer

	Timer(20);


	for (nTimeLoop = 0; nTimeLoop < 200; nTimeLoop++)
	{
		Get_Input();							// Call function to get keys pressed
	    if (nRawKey == 1)						// End if user is exiting
	       {
		       nDone = 1;						// Set nDone to 1
	       }
	
		for (nLoopCount = 0; nLoopCount < NUM_INTRO_SPRITES; nLoopCount++)
		{
		
			Erase_Sprite(&sprite[nLoopCount]);	// Erase sprite at old position
		}


		for (nLoopCount = 0; nLoopCount < NUM_INTRO_SPRITES; nLoopCount++)
		{

			switch (sprite[nLoopCount].nDirection)
			{
				case N:	// ** NORTH **
					if (sprite[nLoopCount].nY > 0)
					{
						sprite[nLoopCount].nY--;
					}
					else
					{
						sprite[nLoopCount].nDirection = S;
						sprite[nLoopCount].nY++;
					}
					break;
					
				case NE: // ** NORTH EAST **
					if (sprite[nLoopCount].nY > 0)
					{
						sprite[nLoopCount].nY--;
					}
					else
					{
						sprite[nLoopCount].nDirection = SE;
						sprite[nLoopCount].nY++;
					}

					if (sprite[nLoopCount].nX < SCREEN_WIDTH - SPRITE_WIDTH)
					{
						sprite[nLoopCount].nX++;
					}
					else
					{
						sprite[nLoopCount].nDirection = NW;
						sprite[nLoopCount].nX--;
					}
					break;


				case E: // ** EAST **
					if (sprite[nLoopCount].nX < SCREEN_WIDTH - SPRITE_WIDTH)
					{
						sprite[nLoopCount].nX++;
					}
					else
					{
						sprite[nLoopCount].nDirection = W;
						sprite[nLoopCount].nX--;
					}
					break;

				case SE: // ** SOUTH EAST **
					if (sprite[nLoopCount].nY < SCREEN_HEIGHT - SPRITE_HEIGHT)
					{
						sprite[nLoopCount].nY++;
					}
					else
					{
						sprite[nLoopCount].nDirection = NE;
						sprite[nLoopCount].nY--;
					}

					if (sprite[nLoopCount].nX < SCREEN_WIDTH - SPRITE_WIDTH)
					{
						sprite[nLoopCount].nX++;
					}
					else
					{
						sprite[nLoopCount].nDirection = SW;
						sprite[nLoopCount].nX--;
					}
					break;

				case S: // ** SOUTH **
					if (sprite[nLoopCount].nY < SCREEN_HEIGHT - SPRITE_HEIGHT)
					{
						sprite[nLoopCount].nY++;
					}
					else
					{
						sprite[nLoopCount].nDirection = N;
						sprite[nLoopCount].nY--;
					}
					break;
					
				case SW: // ** SOUTH WEST **
					if (sprite[nLoopCount].nY < SCREEN_HEIGHT - SPRITE_HEIGHT)
					{
						sprite[nLoopCount].nY++;
					}
					else
					{
						sprite[nLoopCount].nDirection = NW;
						sprite[nLoopCount].nY--;
					}

					if (sprite[nLoopCount].nX > 0)
					{
						sprite[nLoopCount].nX--;
					}
					else
					{
						sprite[nLoopCount].nDirection = SE;
						sprite[nLoopCount].nX++;
					}
					break;
					

				case W: // ** WEST **
					if (sprite[nLoopCount].nX > 0)
					{
						sprite[nLoopCount].nX--;
					}
					else
					{
						sprite[nLoopCount].nDirection = E;
						sprite[nLoopCount].nX++;
					}
					break;

				case NW: // ** NORTH WEST **
					if (sprite[nLoopCount].nY > 0)
					{
						sprite[nLoopCount].nY--;
					}
					else
					{
						sprite[nLoopCount].nDirection = SW;
						sprite[nLoopCount].nY++;
					}

					if (sprite[nLoopCount].nX > 0)
					{
						sprite[nLoopCount].nX--;
					}
					else
					{
						sprite[nLoopCount].nDirection = NE;
						sprite[nLoopCount].nX++;
					}
					break;


			}// End of Switch
		}// End of for loop

		for (nLoopCount = 0; nLoopCount < NUM_INTRO_SPRITES; nLoopCount++)
		{
		
			Behind_Sprite(&sprite[nLoopCount]);	// Save the background at new postition
		}

		for (nLoopCount = 0; nLoopCount < NUM_INTRO_SPRITES; nLoopCount++)
		{
		
			Draw_Sprite(&sprite[nLoopCount]);	// Draw sprite at new position
			sprite[nLoopCount].nX_Old = sprite[nLoopCount].nX;					// Update old X position indicator
			sprite[nLoopCount].nY_Old = sprite[nLoopCount].nY;					// Update old Y position indicator
		}

		Wait_For_Vsync();						// Wait for verticel sync so we don't get a flicker
		Blast_Buffer();							// Copy double buffer

	}// End of for loop

	for (nLoopCount = 0; nLoopCount < NUM_INTRO_SPRITES; nLoopCount++)
	{
		
		Free_Sprite_Mem(&sprite[nLoopCount]);	// Free memory allocated for ship sprite
	}


}
