/**********************************************************************************************************

						LL-GRAFX.C - Graphics module for LANDER.C, the Lunar Lander Game


	DESCRIPTION:	Graphics module for LANDER.EXE (well it's called that as of now).



	COMMENTS:		Some of the code and ideas in the LANDER project was adapted from Tricks Of
					The Game Programming Gurus.


	Components: 	LL-GRAFX.C	-	Implementation file


	Written By:		Joe Lambert
	Date:			06-23-95

	Revisions:
	06-23-95		So it began, stopped with just a ship moving on the screen.
	03-08-96		Added gravity, boundary checking, active fuel and score, crashes,
					number of lives.  Started looking like a real game instead of
					just sprite stuff.
	03-15-96		Broke code into separate modules for graphics functions and the like.
	03-20-96		Added hooks for keyboard and timer.  Broke code into IO module, out
					of near heap space.	
	03-24-96		Added sprite animation, messaging system, and started on fonts.
	10-23-96		Dropped fonts and did clean up.
    11-01-96        Lander went to X2FTP. I'm somebody now, I'm somebody now!!
***********************************************************************************************************/
#include <io.h>
#include <conio.h>
#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <bios.h>
#include <fcntl.h>
#include <memory.h>
#include <math.h>
#include <string.h>
#include <malloc.h>
#include "lander.h"


/********************************************************************
	
	Function:
		VOID Blit_Char(INT, INT, CHAR, INT)
	
	Description:
		this function uses the rom 8x8 character set to blit a
		character on the video screen.
					
	Returns:
		None
		
	Comments:
		Notice the trick used to extract bits out of each character
		byte that comprises a line.	
					
*********************************************************************/
void Blit_Char(int nXC,int nYC,char cChar,int nColor)
{
	int nOffset;    // Used for offset
	int nX;
	int nY;
	float fNewColor;                                // Multi color trick.
	unsigned char far *ucfpWorkChar;                // Pointer to character being worked on.  Makes for cleaner looking code
	unsigned char ucBitMask = 0x80;                 // Mask value. MSB set.

	ucfpWorkChar = ucfpRomCharSet + (cChar * CHAR_HEIGHT);	// Compute starting offset in rom character lookup table
	nOffset = (nYC << 8) + (nYC << 6) + nXC;		// Compute offset of character in video buffer
	fNewColor = nColor;                             // Set new color

	for (nY=0; nY < CHAR_HEIGHT; nY++)              // Loop through full height
    {
	    ucBitMask = 0x80;							// Reset bit mask
	    for (nX = 0; nX < CHAR_WIDTH; nX++)         // Loop through full width
        {
	        if ((*ucfpWorkChar & ucBitMask))		// Is it a transparent pixel? If not transparent then draw
	        {
	        	Double_Buff[nOffset + nX] = nColor; // Set value in memory
	        }
	
	        ucBitMask = (ucBitMask>>1);				// Shift back mask
        }	// end for nX
	
	    nOffset += SCREEN_WIDTH;					// Move to next line down in video buffer
	    ucfpWorkChar++;								// Move to next line in rom character data area
		fNewColor += .25F;                          // Goofy little color trick for multi color fonts.
		nColor = (int)fNewColor;                    // More of the same
    }	// end for nY

}// End of Blit_Char function



/********************************************************************
	
	Function:
		VOID Blit_Message(MessagePtrTyp)
	
	Description:
		This function blits an entire string on the screen with fixed spacing
		between each character.
					
	Returns:
		None
		
	Comments:
		This function calls blit_char.
  


*********************************************************************/
void Blit_Message(MessagePtrTyp Message)
{
	int nLoopCount;

	for (nLoopCount=0; Message->szString[nLoopCount] != 0; nLoopCount++)    // Loop through string, blit each character
     {
  	     Blit_Char(Message->nX_Pos + (nLoopCount << 3), Message->nY_Pos , Message->szString[nLoopCount], Message->nColor);
     }
}// End of Blit_Message function




/********************************************************************
	
	Function:
		VOID Blit_String(INT, INT, INT, CHAR *)
	
	Description:
		This function blits an entire string on the screen with fixed spacing
		between each character.
					
	Returns:
		None
		
	Comments:
		This function calls blit_char.
  
*********************************************************************/
void Blit_String(int nX,int nY,int nColor, char *cpString)
{
	int nLoopCount; // Used for loop

	for (nLoopCount=0; cpString[nLoopCount]!=0; nLoopCount++)   // Loop through string and blit each character.
     {
  	     Blit_Char(nX + (nLoopCount << 3), nY , cpString[nLoopCount], nColor);
     }
}// End of Blit_String function


/********************************************************************
	
	Function:
		VOID Set_Palette_Register(INT, RGB_Color_Ptr)
	
	Description:
		This function sets a single color look up table value indexed by index
		with the value in the color structure

					
	Returns:
		None
		
	Comments:
  
*********************************************************************/
void Set_Palette_Register(int nIndex, RGB_Color_Ptr color)
{
	_outp(PALETTE_MASK,0xff);			// Tell VGA card we are going to update a pallete register
	_outp(PALETTE_REGISTER_WR, nIndex);	// Tell vga card which register we will be updating
	_outp(PALETTE_DATA,color->red);		// Update the RGB triple, note the same port is used each time
	_outp(PALETTE_DATA,color->green);	// Update the RGB triple, note the same port is used each time
	_outp(PALETTE_DATA,color->blue);	// Update the RGB triple, note the same port is used each time
}// End of Set_Palette_Color function


/********************************************************************
	
	Function:
		VOID PCX_Init(Pcx_Picture_Ptr)
	
	Description:
		This function allocates the buffer region needed to load
		a pcx file.

					
	Returns:
		None
		
	Comments:
  
*********************************************************************/
void PCX_Init(Pcx_Picture_Ptr image)
{
	if (!(image->buffer = (char far *)malloc(SCREEN_WIDTH * SCREEN_HEIGHT + 1)))
	{
		sprintf(szErrorString, "Couldn't allocate memory for PCX image buffer.\n"); 
//		printf("Couldn't allocate memory for PCX image buffer.\n");
		AbEnd();
	}

}// End of PCX_Init function.


/********************************************************************
	
	Function:
		VOID Plot_Pixel_Fast(INT, INT, UNSIGNED CHAR)
	
	Description:
		Plots the pixel in the desired color a little quicker using
		binary shifting	to accomplish the multiplications.

					
	Returns:
		None
		
	Comments:
  
*********************************************************************/
void Plot_Pixel_Fast(int nX,int nY,unsigned char ucColor)
{
	Double_Buff[( (nY<<8) + (nY<<6) ) + nX] = ucColor;	// Use the fact that 320*y = 256*y + 64*y = y<<8 + y<<6

}// End of Plot_Pixel_Fast function


/********************************************************************
	
	Function:
		VOID Free_PCX_Mem(Pcx_Picture_Ptr)
	
	Description:
		This function de-allocates the buffer region used for the
		pcx file load.

					
	Returns:
		None
		
	Comments:
  
*********************************************************************/
void Free_PCX_Mem(Pcx_Picture_Ptr Image)
{
	_ffree(Image->buffer);  // Free the image buffer memory

}// End of Free_PCX_Mem function




/********************************************************************
	
	Function:
		VOID PCX_Load(CHAR *, Pcx_Picture_Ptr, INT)
	
	Description:
		This function loads a pcx file into a picture structure,
		the actual image data for the pcx file is decompressed and
		expanded into a secondary buffer within the picture structure,
		the separate images can be grabbed from this buffer later.
		Also the header and palette are loaded

					
	Returns:
		None
		
	Comments:
  
*********************************************************************/
void PCX_Load(char *cpFilename, Pcx_Picture_Ptr image,int nEnablePalette)
{
	FILE *fp;
	int nNumBytes;
	int nIndex;
	long lCount;
	unsigned char ucData;
	char far *cfpTempBuffer;

	if (!(fp = fopen(cpFilename,"rb")))		// Open the file
	{
		sprintf(szErrorString, "Unable to open file %s.\n", cpFilename); 
//		printf("Unable to open file %s.\n", cpFilename);
		AbEnd();
	}

	cfpTempBuffer = (char far *)image;		// Assign temp_buffer with address of image?
	for (nIndex=0; nIndex<128; nIndex++)	// Load the header of the PCX image.  Loop through the entire 128 byte header
    {
	    cfpTempBuffer[nIndex] = getc(fp);	// Load one character at a time
    }

	lCount=0;	// Reset count
	while(lCount<=SCREEN_WIDTH * SCREEN_HEIGHT)	// Load the data and decompress into buffer
	{
		ucData = getc(fp);				// Get the first piece of data
		if (ucData>=192 && ucData<=255)	// Is it Run Length Encoded?
		{
			nNumBytes = ucData-192;		// If it is RLE, how many bytes in the run?
			ucData  = getc(fp);			// Grab data from file (next character)
	
			while(nNumBytes-- > 0)		// Replicate data in buffer num_bytes times
			{
				image->buffer[lCount++] = ucData;
			}
		}

		else
		{
			image->buffer[lCount++] = ucData;	// Else if it's actual data, just copy it into buffer at next location
		}
	}

	fseek(fp,-768L,SEEK_END);					// Move to 768 bytes from end of file, to begining of palette
	for (nIndex=0; nIndex < 256; nIndex++)		// Load the pallete.  Each loop grabs three bytes. 768 bytes total.
	{
		image->palette[nIndex].red   = (getc(fp) >> 2);	// Get the RED component and divide by 4
		image->palette[nIndex].green = (getc(fp) >> 2);	// Get the GREEN component and divide by 4
		image->palette[nIndex].blue  = (getc(fp) >> 2);	// Get the BLUE component and divide by 4
	}

	fclose(fp);	// Close file
	if (nEnablePalette)		// Change the palette to newly loaded palette if commanded to do so
	{
		for (nIndex=0; nIndex<256; nIndex++)	// Loop through all 256 locations
		{
			Set_Palette_Register(nIndex,(RGB_Color_Ptr)&image->palette[nIndex]);	// Set palette to new value
		}
	}

}// End of PCX_Load function




/********************************************************************
	
	Function:
		VOID PCX_Show_Buffer(Pcx_Picture_Ptr)
	
	Description:
		Just copy the PCX buffer into the VIDEO buffer

					
	Returns:
		None
		
	Comments:
  
*********************************************************************/
void PCX_Show_Buffer(Pcx_Picture_Ptr image)
{
	_fmemcpy((char far *)Double_Buff, (char far *)image->buffer,SCREEN_WIDTH*SCREEN_HEIGHT);
}// End of PCX_Show_Picture function




/********************************************************************
	
	Function:
		VOID Sprite_Init(Sprite_Ptr, INT, INT, INT, INT)
	
	Description:
		This function initializes a sprite with the sent data.

					
	Returns:
		None
		
	Comments:
		1st param = pointer to sprite
		2nd param = x posistion of sprite
		3rd param = y posistion of sprite
  
*********************************************************************/
void Sprite_Init(Sprite_Ptr sprite,int nX,int nY, int nWidth, int nHeight)
{
	int nIndex;
	
	sprite->nX			= nX;				// X posistion of sprite on screen
	sprite->nY			= nY;				// Y posistion of sprite on screen
	sprite->nX_Old		= nX;				// Old X for replacing background
	sprite->nY_Old		= nY;				// Old Y for replacing background
	sprite->nWidth		= nWidth;			// Sprite width in pixels
	sprite->nHeight		= nHeight;			// Sprite height in pixels
	sprite->nDirection	= RIGHT;
	sprite->nFrameRate	= 0;
	sprite->nFrameClock	= 0;
	sprite->nCurrFrame	= 0;				// Which image 
	sprite->nState		= SPRITE_DEAD;
	sprite->nNumFrames	= 0;

	if (!(sprite->cfpBackground	= (char far *)malloc(nWidth * nHeight + 1)))
	{
		sprintf(szErrorString, "Unable to allocate sprite memory.\n"); 
//		printf("Unable to allocate sprite memory.\n");
		AbEnd();
	}
	
	for (nIndex=0; nIndex < MAX_SPRITE_FRAMES; nIndex++)		// Set all bitmap pointers to null
	{
		sprite->cfpFrames[nIndex] = NULL;
	}
}// End of Sprite_Init function



/********************************************************************
	
	Function:
		VOID Free_Sprite_Mem(Sprite_Ptr)
	
	Description:
		This function frees all the memory associated with a sprite.

					
	Returns:
		None
		
	Comments:
  
*********************************************************************/
void Free_Sprite_Mem(Sprite_Ptr sprite)
{
	int nIndex;
	
	_ffree(sprite->cfpBackground);
	for (nIndex=0; nIndex < MAX_SPRITE_FRAMES; nIndex++)		// Now de-allocate all the animation frames.
	{
		_ffree(sprite->cfpFrames[nIndex]);
	}
}// End of Free_Sprite_Mem function


/********************************************************************
	
	Function:
		VOID PCX_Grap_Bitmap(Pcx_Picture_Ptr, Sprite_Ptr, INT, INT, INT)
	
	Description:
		This function will grap a bitmap from the PCX frame buffer. It uses the
		convention that the 320x200 pixel matrix is sub divided into a smaller
		matrix of 12x8 adjacent squares each being a 24x24 pixel bitmap.
		The caller sends the pcx picture along with the sprite to save the image
		into and the frame of the sprite.  Finally, the position of the bitmap
		that should be grabbed is sent.

					
	Returns:
		None
		
	Comments:
	1st param = pointer to image to grab from
	2nd param = pointer to sprite where image goes
	3rd param = current sprite frame
	4th param = x posistion to grab from	
	5th param = y posistion to grab from	
  
*********************************************************************/
void PCX_Grap_Bitmap(Pcx_Picture_Ptr image, Sprite_Ptr sprite, int nSpriteFrame, int nGrabX, int nGrabY)
{
	int nX_Off;
	int nY_Off;
	int nX;
	int nY;
	char far *cfpSpriteData;
	int nWidth = sprite->nWidth;
	int nHeight = sprite->nHeight;

	sprite->cfpFrames[nSpriteFrame] = (char far *)malloc(sprite->nWidth * sprite->nHeight);// Allocate sprite memory for the sprite in the sprite structure
	cfpSpriteData = sprite->cfpFrames[nSpriteFrame];		// Create an alias to the sprite frame for ease of access

// Load the sprite data into the sprite frame array from the pcx picture.
// We need to find which bitmap to scan, the pcx picture is really a
// 12x8 matrix of bitmaps where each bitmap is 24x24 pixels. 
// Note:0,0 is the upper left bitmap and 11,7 is the lower right bitmap.

	nX_Off = (sprite->nWidth + 1) * nGrabX + 1;			// Find the pixel offset location for the X direction
	nY_Off = (sprite->nHeight + 1) * nGrabY + 1;			// Find the pixel offset location for the Y direction
	nY_Off = nY_Off * SCREEN_WIDTH;				// Compute starting y address
	for (nY=0; nY < nHeight; nY++)		// Loop from top to bottom
	{
	    for (nX=0; nX < nWidth; nX++)	// Loop from left to right per top to bottom loop					
		{												
	        cfpSpriteData[nY * nWidth + nX] = image->buffer[nY_Off + nX_Off + nX];	// Get the next byte of current row and place into
        }	                                                         	// next position in sprite frame data buffer

		nY_Off += SCREEN_WIDTH;						// Move to next line of the picture buffer.
    } 
	sprite->nNumFrames++;				// Increment number of frames
}// End of PCX_Grap_Bitmap function



/********************************************************************
	
	Function:
		VOID Behind_Sprite(Sprite_Ptr)
	
	Description:
		This function copies the background behind a sprite before 
		it's written so when the sprite is drawn, the background
		isn't obliterated.

					
	Returns:
		None
		
	Comments:
  
*********************************************************************/
void Behind_Sprite(Sprite_Ptr sprite)
{
	char far *cfpWorkBack;
	int nWorkOffset=0;
	int nOffset;
	int nY;
	int nWidth = sprite->nWidth;
	int nHeight = sprite->nHeight;

	cfpWorkBack = sprite->cfpBackground;								// Alias a pointer to sprite background for ease of access.
	nOffset = (sprite->nY << 8) + (sprite->nY << 6) + sprite->nX;	// Compute offset of background in video buffer.
	for (nY=0; nY < nHeight; nY++)
	{												// copy the next row out off screen buffer into sprite background buffer
	    _fmemcpy((char far *)&cfpWorkBack[nWorkOffset], (char far *)&Double_Buff[nOffset], nWidth);
	
	    nOffset      += SCREEN_WIDTH;	// Move to next line in the video buffer
	    nWorkOffset += SPRITE_WIDTH;	// Move to next line in the sprite background buffer
    }
}// End of Behind_Sprite function



/********************************************************************
	
	Function:
		VOID Erase_Sprite(Sprite_Ptr)
	
	Description:
		Replace the background that was behind the sprite.
		This function replaces the background that was saved from
		where a sprite was going to be placed.

					
	Returns:
		None
		
	Comments:
  
*********************************************************************/
void Erase_Sprite(Sprite_Ptr sprite)
{
	char far *cfpWorkBack;
	int nWorkOffset=0;
	int nOffset;
	int nY;
	int nWidth = sprite->nWidth;
	int nHeight = sprite->nHeight;

	cfpWorkBack = sprite->cfpBackground;// Alias a pointer to sprite background for ease of access.
	nOffset = (sprite->nY_Old << 8) + (sprite->nY_Old << 6) + sprite->nX_Old;	// Compute offset of background in video buffer
	for (nY=0; nY < nHeight; nY++)
	{
	    // copy the next row out of the screen buffer into sprite background buffer
	    _fmemcpy((char far *)&Double_Buff[nOffset], (char far *)&cfpWorkBack[nWorkOffset], nWidth);
	
	    nOffset      += SCREEN_WIDTH;	// Move to the next line in the video buffer.
	    nWorkOffset += SPRITE_WIDTH;	// Move to the next line in the sprite background buffer.
    }
}// End of Erase_Sprite function


/********************************************************************
	
	Function:
		VOID Draw_Sprite(Sprite_Ptr)
	
	Description:
		This function draws a sprite on the screen row by row very quickly.
		Note the use of shifting to implement multplication.

					
	Returns:
		None
		
	Comments:
  
*********************************************************************/
void Draw_Sprite(Sprite_Ptr sprite)
{
	char far *cfpWorkSprite;
	int nWorkOffset=0;
	int nOffset;
	int nX;
	int nY;
	int nWidth = sprite->nWidth;
	int nHeight = sprite->nHeight;
	unsigned char ucData;

	cfpWorkSprite = sprite->cfpFrames[sprite->nCurrFrame];			// Alias a pointer to the sprite for ease of access.
	nOffset = (sprite->nY << 8) + (sprite->nY << 6) + sprite->nX;	// Compute offset of sprite in video buffer

	for (nY = 0; nY < nHeight; nY++)
    {
	    for (nX = 0; nX < nWidth; nX++)				// Copy the next row into the screen buffer using memcpy for speed
        {
			if ((ucData = cfpWorkSprite[nWorkOffset+nX]))	// Test for transparent pixel i.e. 0, if not transparent then draw
			{
				Double_Buff[nOffset+nX] = ucData;
			}
        }
	
	    nOffset      += SCREEN_WIDTH;		// Move to the next line in the video buffer.
	    nWorkOffset += SPRITE_WIDTH;		// Move to the next line in the sprite bitmap buffer.
    }
}// End of Draw_Sprite function



/********************************************************************
	
	Function:
		VOID Wait_For_Vsync(VOID)
	
	Description:
		This function waits for the start of a vertical retrace,
		if a vertical retrace is in progress then it waits until
		the next one.

					
	Returns:
		None
		
	Comments:
  
*********************************************************************/
void Wait_For_Vsync(void)
{
	_asm
	{
		mov dx,0x3da
	NoRetrace:
		in al,dx
		and al,8
		jz NoRetrace
	Retrace:
		in al,dx
		and al,8
		jz Retrace
	}
}// End of Wait_For_Vsync function


/********************************************************************
	
	Function:
		VOID Set_Video_Mode(INT)
	
	Description:
		This function set the video mode to mode requested.
					
	Returns:
		None
		
	Comments:
  
*********************************************************************/
void Set_Video_Mode(int mode)
{
	_asm 
	{
		mov	ax,mode		/* Load function Mode */
	    int	0x10		/* Call interrupt */
    }
}



/********************************************************************
	
	Function:
		VOID SaveMessageBackground(MessagePtrTyp)	

	Description:
		This function saves the background behind any Message to be 
		blitted so we can update the Message.
		
	Returns:
		None
		
	Comments:

		MessageTyp
		{
		int nX_Pos;					// X posistion of Message
		int nY_Pos;					// Y posistion of Message
		int nColor;					// Color of Message
		char szString[25];			// String containing Message
		char far *cfpBackground;	// Behind Message		
		} Message, *MessagePtrTyp;
  
*********************************************************************/
void SaveMessageBackground(MessagePtrTyp Message)
{
	int nMessageWidth;				// How many pixels wide the Message is
	int nMessageHeight = CHAR_HEIGHT;	// How many pixels high the Message is
	char far *cfpWorkBack;			
	int nWorkOffset=0;
	int nOffset;
	int nY;

// Calculate total width of background
	nMessageWidth = Message->nMaxStringLength * CHAR_WIDTH;
// Create pointer to background for ease of access.
	cfpWorkBack = Message->cfpBackground;
// Compute offset of background in video buffer.
	nOffset = (Message->nY_Pos << 8) + (Message->nY_Pos << 6) + Message->nX_Pos;

	for (nY=0; nY < nMessageHeight; nY++)
	{												// copy the next row out off screen buffer into sprite background buffer
	    _fmemcpy((char far *)&cfpWorkBack[nWorkOffset], (char far *)&Double_Buff[nOffset], nMessageWidth);
	
	    nOffset      += SCREEN_WIDTH;	// Move to next line in the video buffer
	    nWorkOffset += nMessageWidth;	// Move to next line in the score background buffer
    }
}// End of SaveMessageBackground function



/********************************************************************
	
	Function:
		VOID EraseMessage(MessagePtrTyp)	

	Description:
		This function restores the background behind any Message that 
		was blitted so we can update the Message.
		
	Returns:
		None
		
	Comments:
		None
  
*********************************************************************/
void EraseMessage(MessagePtrTyp Message)
{
	int nStringWidth;					// How many pixels wide is the Message
	char far *cfpWorkBack;			
	int nWorkOffset=0;
	int nOffset;
	int nY;

	nStringWidth = Message->nMaxStringLength * CHAR_WIDTH;
// Create pointer to background for ease of access.
	cfpWorkBack = Message->cfpBackground;
// Compute offset of background in video buffer.
	nOffset = (Message->nY_Pos << 8) + (Message->nY_Pos << 6) + Message->nX_Pos;

	for (nY=0; nY < CHAR_HEIGHT; nY++)
	{
	    // copy the next row out of the screen buffer into Message background buffer
	    _fmemcpy((char far *)&Double_Buff[nOffset], (char far *)&cfpWorkBack[nWorkOffset], nStringWidth);
	
	    nOffset      += SCREEN_WIDTH;	// Move to the next line in the video buffer.
	    nWorkOffset += nStringWidth;	// Move to the next line in the Message background buffer.
    }
}// End of EraseMessage function



/********************************************************************
	
	Function:
		VOID FreeMessageMem(MessagePtrTyp)	

	Description:
		This function frees memory allocated for Message that was blitted.
		
	Returns:
		None
		
	Comments:
		None

*********************************************************************/
void FreeMessageMem(MessagePtrTyp Message)
{
	_ffree(Message->cfpBackground);
}// End of FreeMessageMem function



/********************************************************************
	
	Function:
		VOID AllocateMessageBack(MessagePtrTyp)	

	Description:
		This function allocates memory to store the background
		of the Message that will be blitted.  That way we can erase
		the Message be replacing the background.
		
	Returns:
		None
		
	Comments:
		None
  
*********************************************************************/
void AllocateMessageBack(MessagePtrTyp Message)
{
    int nMemSize;

// memory size is number of characters * CHAR_WIDTH * CHAR_HEIGHT pixels due to the
// charaters being 8 * 8 pixels.    
	nMemSize = (Message->nMaxStringLength * CHAR_WIDTH) * CHAR_HEIGHT;	// Allocate enough memory for background
	if ( ! (Message->cfpBackground = (char far *)malloc(nMemSize)))
	{
		sprintf(szErrorString, "Couldn't allocate Message buffer memory.\n"); 
//		printf("Couldn't allocate Message buffer memory.\n");
		AbEnd();
	}
}// End of AllocateMessageBack function.


/********************************************************************
	
	Function:
		VOID Blast_Buffer(VOID)
	
	Description:
		Just copy the PCX buffer into the VIDEO buffer

					
	Returns:
		None
		
	Comments:
		None
		
*********************************************************************/
void Blast_Buffer(void)
{
	_fmemcpy((char far *)video_buffer, (char far *)Double_Buff,SCREEN_WIDTH * SCREEN_HEIGHT);
}// End of Blast_Buffer function


/********************************************************************
	
	Function:
		VOID Init_Double_Buffer(VOID)
	
	Description:

					
	Returns:
		None
		
	Comments:
		None
  
*********************************************************************/
void Init_Double_Buffer(void)
{
	if (!(Double_Buff = (char far *)_fmalloc(SCREEN_WIDTH * SCREEN_HEIGHT + 1)))	// Allocate memory for double buffer
	{
		sprintf(szErrorString, "Couldn't allocate double buffer memory.\n"); 
//		printf("Couldn't allocate double buffer memory.\n");
		AbEnd();
	}
	_fmemset(Double_Buff, 0, SCREEN_WIDTH * SCREEN_HEIGHT + 1);
}// End of Init_Double_Buffer function



/********************************************************************
	
	Function:
		VOID Font_Init(Font_Ptr, INT, INT, INT, INT)
	
	Description:

					
	Returns:
		None
		
	Comments:
		None
  
*********************************************************************/
void Font_Init(Font_Ptr font,int nX,int nY, int nWidth, int nHeight)
{
	font->nWidth		= nWidth;			// font width in pixels
	font->nHeight		= nHeight;			// font height in pixels
	font->cfpImage = NULL;
}// End of Font_Init function


/********************************************************************
	
	Function:
		VOID Free_Font_Mem(Sprite_Ptr)
	
	Description:
		This function frees all the memory associated with a sprite.

					
	Returns:
		None
		
	Comments:
		None
  
*********************************************************************/
void Free_Font_Mem(Font_Ptr font)
{
	_ffree(font->cfpImage);
}// End of Free_Font_Mem function


/********************************************************************
	
	Function:
		VOID PCX_Grap_Font(Pcx_Picture_Ptr, Font_Ptr, INT, INT, INT)
	
	Description:

					
	Returns:
		None
		
	Comments:
		None
  
*********************************************************************/
void PCX_Grap_Font(Pcx_Picture_Ptr image, Font_Ptr font, int nGrabX, int nGrabY)
{
	int nX_Off;
	int nY_Off;
	int nX;
	int nY;
	char far *cfpFontData;
	int nWidth = font->nWidth;
	int nHeight = font->nHeight;

	font->cfpImage = (char far *)malloc(font->nWidth * font->nHeight);// Allocate sprite memory for the sprite in the sprite structure
	cfpFontData = font->cfpImage;		// Create an alias to the sprite frame for ease of access

	nX_Off = (font->nWidth + 1) * nGrabX + 1;			// Find the pixel offset location for the X direction
	nY_Off = (font->nHeight + 1) * nGrabY + 1;			// Find the pixel offset location for the Y direction
	nY_Off = nY_Off * SCREEN_WIDTH;				// Compute starting y address
	
	for (nY=0; nY < nHeight; nY++)		// Loop from top to bottom
	{
	    for (nX=0; nX < nWidth; nX++)	// Loop from left to right per top to bottom loop					
		{												
	        cfpFontData[nY * nWidth + nX] = image->buffer[nY_Off + nX_Off + nX];	// Get the next byte of current row and place into
        }	                                                         	// next position in sprite frame data buffer

		nY_Off += SCREEN_WIDTH;						// Move to next line of the picture buffer.
    } 
}// End of PCX_Grap_Bitmap function


/********************************************************************
	
	Function:
		VOID Draw_Font(Font_Ptr)
	
	Description:

					
	Returns:
		None
		
	Comments:
		None
  
*********************************************************************/
void Draw_Font(Font_Ptr font, int nX, int nY)
{
	char far *cfpWorkSprite;
	int nWorkOffset=0;
	int nOffset;
	int nWidth = font->nWidth;
	int nHeight = font->nHeight;
	unsigned char ucData;

	cfpWorkSprite = font->cfpImage;
	nOffset = (nY << 8) + (nY << 6) + nX;	// Compute offset of font in video buffer
	for (nY = 0; nY < nHeight; nY++)
    {
	    for (nX = 0; nX < nWidth; nX++)				// Copy the next row into the screen buffer using memcpy for speed
        {
			if ((ucData = cfpWorkSprite[nWorkOffset+nX]))	// Test for transparent pixel i.e. 0, if not transparent then draw
			{
				Double_Buff[nOffset+nX] = ucData;
			}
        }
	    nOffset      += SCREEN_WIDTH;		// Move to the next line in the video buffer.
	    nWorkOffset += nWidth;		// Move to the next line in the font bitmap buffer.
    }
}// End of Draw_Font function




