/**********************************************************************************************************

						LANDER.H - Header file for LANDER.C


	DESCRIPTION:	Header module for LANDER.EXE (well it's called that as of now).


                	
	COMMENTS:		Some of the code and ideas in the LANDER project was adapted from Tricks Of
					The Game Programming Gurus.


	Components: 	LANDER.H	-	This file
					LANDER.C	-	Main implementation module
					LL-GRAFX.C	-	Graphics implementation file
					LL-IO.C		-	Input Output implementation file



	Written By:		Joe Lambert
	Date:			06-23-95

	Revisions:
	06-23-95		So it began, stopped with just a ship moving on the screen.
	03-08-96		Added gravity, boundary checking, active fuel and score, crashes,
					number of lives.  Started looking like a real game instead of
					just sprite stuff.
	03-15-96		Broke code into separate modules for graphics functions and the like.
	03-20-96		Added hooks for keyboard and timer.  Broke code into IO module, out
					of near heap space.	
	03-24-96		Added sprite animation, messaging system, and started on fonts.
	10-23-96		Dropped fonts and did clean up.
    11-01-96        Lander went to X2FTP. I'm somebody now, I'm somebody now!!
***********************************************************************************************************/


// ************************************
// ********** Define Globals **********
// ************************************
	unsigned char far *video_buffer;		// Video RAM byte pointer
	unsigned short far *video_buffer_w;		// Video RAM word pointer
	unsigned char far *Double_Buff;			// Pointer to double buffer
	unsigned char far *ucfpRomCharSet;		// ROM characters 8x8
	unsigned short far *uspCurrentTime;		// pointer to internal timer
	int nKeyTable[4];						// The key state table for the motion keys
	int nRawKey;							// The global raw keyboard data aquired from the ISR
	int nTimeOut;							// Has the clock timed out
	char szErrorString[128];	
	void (_interrupt _far *Old_Key_Isr)();	// Holds old keyboard interrupt handler
	void (_interrupt _far *Old_Timer_Isr)();// Holds old com port (Timer?) interrupt handler

// *********************************************
// ********** Define Constant Strings **********
// *********************************************
#define INTRO_FILE "LL-INTRO.PCX"	// Name of intro image file
#define SPRITE_FILE "LL-SPRT.PCX"	// Name of sprite image file
#define SCAPE_FILE "LL-SCAPE.PCX"	// Name of scape image file
#define FONT_FILE "LL-FONT.PCX"		// Name of font image file
#define SCORE_TITLE "Score "		// Assign text to Score which is a dynamic length string
#define LEVEL_TITLE "Level "		// Assign text to Level which is a dynamic length string
#define SPEED_TITLE "Speed "		// Assign text to Speed which is a dynamic length string
#define FUEL_TITLE "Fuel "			// Assign text to Fuel which is a dynamic length string
#define LOW_FUEL_TITLE "Low Fuel"	// Low Fuel message is a static length string
#define TITLE_TITLE "Lunar Lander"	// Title message is a static length string
#define SUCCESS_TITLE "TOUCHDOWN"	// Success message is a static length string
#define LANDINGS_TITLE "Landings "	// Landings text only blitted at end, we don't erase it

// **************************************
// ********** Define Constants **********
// **************************************
#define NUM_MESSAGE 6		// Number of Message strings that will require memory allocation
#define SCORE 0				// ID for Score message
#define FUEL 1				// ID for Fuel message
#define TITLE 2				// ID for Title message
#define SUCCESS 3			// ID for Success message
#define LEVEL 4				// ID for Level message
#define LOW_FUEL 5			// ID for Low Fuel message
#define NUM_SPRITES 3		// Number of sprites used
#define SHIP 0				// ID for ship sprite
#define PLATFORM 1			// ID for platform sprite
#define LIVES 2				// ID for lives sprite
#define FLYING 0			// Game state normal, just flying around
#define LANDED 1			// Sucessful landing
#define CRASH 2				// Crashed
#define GAME_OVER 3			// Out of lives, game over
#define SPRITE_DEAD 0		// Indicates a dead sprite.
#define SPRITE_ALIVE 1		// Indicates a live sprite.
#define SPRITE_DYING 2		// Indicates a dying sprite.
#define STATE_RATE 60		// The time a particular game state will last.
#define NUM_INTRO_SPRITES 21// Number of sprites used in introduction
#define GONE 0				// ID for state of being GONE for message 
#define TEMPERARY 1			// ID for state of being TEMPERARY for message
#define PERMENENT 2			// ID for state of being PERMENENT for message
#define MESSAGE_TIME 120	// How long messages will be displayed

// ***** Sprite stuff *****
#define SPRITE_WIDTH 24		// Sprite width in pixels. Used for ship, platform, and lives.
#define SPRITE_HEIGHT 24	// Sprite height in pixels. Used for ship, platform, and lives.
#define MAX_SPRITE_FRAMES 18// Maximum number of frames the sprite will have, different views
#define NUM_FONT 62			// Number of font characters used
#define ROW_POS 64			// Row of introduction
#define COL_POS 76			// Column of introduction
#define MAX_COL 7			// Max column value
#define MAX_ROW 3			// Max row value
#define NE 0				// North East - introduction sprite direction.
#define SE 1				// South East - introduction sprite direction.
#define SW 2				// South West - introduction sprite direction.
#define NW 3				// North West - introduction sprite direction.
#define N 4					// North - introduction sprite direction.
#define S 5					// South - introduction sprite direction.
#define E 6					// East - introduction sprite direction.
#define W 7					// West - introduction sprite direction.
#define UP 0				// Direction sprite (or whatever) is heading
#define DOWN 1				// Direction sprite (or whatever) is heading
#define LEFT 2				// Direction sprite (or whatever) is heading
#define RIGHT 3				// Direction sprite (or whatever) is heading

// ***** Colors *****
#define RED 30				// Index of color in color palette
#define BLUE 5				// Index of color in color palette
#define GREEN 180			// Index of color in color palette
#define YELLOW 210			// Index of color in color palette
#define BLACK 0				// Index of color in color palette
#define WHITE 215			// Index of color in color palette
#define GRAY 220			// Index of color in color palette
#define LT_BLUE 125			// Index of color in color palette
#define PURPLE 16			// Index of color in color palette
#define FONT_WIDTH 10		// Width of font sprite
#define FONT_HEIGHT 10		// Height of font sprite
#define GRAVITY 32			// Gravity.  Well what did you expect?

// ***** Speed *****
#define SPEED_DIVIDE 50		// Speed division factor
#define MAX_SPEED 400		// Maximum ship speed
#define ZERO_SPEED 0		// Ship standing still
#define MIN_SPEED -400		// Minimum ship speed

// ***** Fuel *****
#define FUEL_LENGTH 10		// Place holder for dynamic length fuel message
#define FULL_TANK 500		// Full tank of gas
#define FUEL_X_POS 238		// X position of fuel
#define FUEL_Y_POS 0		// Y position of fuel
#define LOW_FUEL_Y_POS 100	// Y position of low fuel message
#define LEVEL_Y_POS 100		// Y position level message
#define TITLE_Y_POS 0		// Y position of title message
#define SUCCESS_Y_POS 90	// Y position of success message

// ***** Score *****
#define SCORE_LENGTH 11		// Pace holder for dynamic length score message
#define SCORE_X_POS 0		// X position of score
#define SCORE_Y_POS 0		// Y position of score
#define MAX_SCORE 64000		// Maximum score

// ***** Platform *****
#define PLATFORM_X 0				// X position of lives sprite
#define PLATFORM_Y 184				// Y position of lives sprite
#define PLATFORM_FRAME_ZERO 0		// Ship with no engines fired
#define PLATFORM_FRAME_ONE 1		// Ship with both engines fired
#define PLATFORM_FRAME_TWO 2		// Ship with RIGHT engine fired
#define PLATFORM_FRAME_THREE 3		// Ship with RIGHT engine fired
#define PLATFORM_FRAMERATE 4		// How often to update platform frames

// ***** Lives *****
#define LIVES_X 296			// X position of lives sprite
#define LIVES_Y 176			// Y position of lives sprite
#define LIVES_FRAMERATE 10	// How often to update lives sprite frames
#define LIVES_3_0 0			// First frame 3 lives
#define LIVES_3_5 5			// Last frame 3 lives
#define LIVES_2_6 6			// First frame 2 lives
#define LIVES_2_11 11		// Last frame 2 lives
#define LIVES_1_12 12		// First frame 1 life
#define LIVES_1_17 17		// Last frame 1 life
#define LIVES_1 1			// Number of lives indicated by lives sprite
#define LIVES_2 2			// Number of lives indicated by lives sprite
#define LIVES_3 3			// Number of lives indicated by lives sprite

// ***** Ship *****
#define SHIP_FRAMERATE 4	// How often to update ship sprite frames
#define SHIP_START_X 160	// Starting X position of ship
#define SHIP_START_Y 0		// Starting Y position of ship
#define NO_ENGINE_0 0		// First frame of ship with no engines fired
#define NO_ENGINE_5 5		// Last frame of ship with no engines fired
#define MAIN_ENGINE_6 6		// First frame of ship with main engines fired
#define MAIN_ENGINE_11 11	// Last frame of ship with main engines fired
#define CRASH_1 12			// Frame 1 of crash
#define CRASH_2 13			// Frame 2 of crash
#define CRASH_3 14			// Frame 3 of crash
#define CRASH_4 15			// Frame 4 of crash
#define CRASH_5 16			// Frame 5 of crash
#define CRASH_6 17			// Frame 6 of crash
#define CRASH_RATE 5		// Frame rate of crash sequence

// ***** Keyboard Stuff *****
#define INDEX_UP		0	// Indicates the UP button
#define INDEX_DOWN		1	// Indicates the DOWN button
#define INDEX_RIGHT		2	// Indicates the RIGHT button
#define INDEX_LEFT		3	// Indicates the LEFT button
#define KEYBOARD_INT	0x09// The keyboard interrupt vector
#define KEY_BUFFER		0x60// The keyboard buffer area
#define KEY_CONTROL		0x61// The keyboard control register
#define INT_CONTROL		0x20// Interrupt control register
#define MAKE_RIGHT		77	// Make code for the RIGHT arrow key
#define MAKE_LEFT		75	// Make code for the LEFT arrow key
#define MAKE_UP			72	// Make code for the UP arrow key
#define MAKE_DOWN		80	// Make code for the DOWN arrow key
#define BREAK_RIGHT     205	// Break code for the RIGHT arrow key
#define BREAK_LEFT      203	// Break code for the LEFT arrow key
#define BREAK_UP        200	// Break code for the UP arrow key
#define BREAK_DOWN      208	// Break code for the DOWN arrow key

// ***** Timer Stuff *****
#define TIME_KEEPER_INT 0x1C// Interrupt of timer
#define CONTROL_8253	0x43// The 8253's control register
#define CONTROL_WORD	0x3C// The control word to set mode 2,binary least/most
#define COUNTER_0		0x40// Counter 0 address
// 1.19318MHz
// 1,193,180 Hz
#define TIMER_120HZ	0x2D67	// Value for programming the timer for 120 hz.
#define TIMER_60HZ	0x4DAE	// Value for programming the timer for 60 hz
#define TIMER_18HZ	0xFFFF	// Value for programming the timer for 18.2 hz (the standard count)

// ***** Macros *****
#define LOW_BYTE(n) (n & 0x00ff)	// Macro to mask lower byte, keeps only lower byte
#define HI_BYTE(n) ((n>>8) & 0x00ff)// Macro to mask upper byte, keeps only upper byte

// ***** Misc *****
#define YES 1						// Yes 
#define NO 0						// No
#define TRUE 1						// True
#define FALSE 0						// False

#define ROM_CHAR_SET_SEG 0xF000		// Segment of 8x8 ROM character set
#define ROM_CHAR_SET_OFF 0xFA6E		// Begining offset of 8x8 ROM character set

#define MODE13            0x13		// 320 X 200 256 color Mode
#define TEXT_MODE         0x03		// TEXT Mode

#define PALETTE_MASK         0x3c6	// Mask for color palette.
#define PALETTE_REGISTER_RD  0x3c7	// Read register for color palette.
#define PALETTE_REGISTER_WR  0x3c8	// Write register for color palette.
#define PALETTE_DATA         0x3c9	// Data regiater for color palette.

#define SCREEN_WIDTH      (unsigned short)320	// Width of the graphics mode screen
#define SCREEN_HEIGHT     (unsigned short)200	// Height of the graphics mode screen

#define CHAR_WIDTH        8			// Character width in pixels
#define CHAR_HEIGHT       8			// Character height in pixels

#define VGA_INPUT_STATUS_1   0x3DA	// Vga status reg 1, bit 3 is the vsync
									// When 1 - retrace in progress
									// When 0 - no retrace
#define VGA_VSYNC_MASK 0x08			// Masks off unwanted bit of status reg


// ***************************************
// ********** Define Structures **********
// ***************************************

typedef struct						// This structure holds information on Message to be blitted
		{
		int nX_Pos;					// X position of Message
		int nY_Pos;					// Y position of Message
		int nColor;					// Color of Message
		int nMaxStringLength;		// Maximum length the string will be, so we can erase it later
		int nState;					// Current state of the message
		int nMessageTime;			// How long the message will last
		char szString[15];			// String containing Message
		char far *cfpBackground;	// Pointer to memory for image behind Message		
		} MessageTyp, *MessagePtrTyp;

typedef struct						// This structure holds a RGB triple in three bytes
        {
        unsigned char red;    		// RED component of color 0-63
        unsigned char green;  		// GREEN component of color 0-63
        unsigned char blue;   		// BLUE  component of color 0-63
        } RGB_Color, *RGB_Color_Ptr;

typedef struct						// Structure of PCX file header, just the header
        {
        char manufacturer;			// Who edited or made the PCX file
        char version;				// The version of the file
        char encoding;				// Type of encoding, RLE
        char bits_per_pixel;		// How many bits are used per pixel
        int x;						// X position
        int y;						// Y position
        int width;					// Width of the picture
        int height;					// Height of the picture
        int horz_res;				// Horizontal resolution
        int vert_res;				// Verticle resolution
        char ega_palette[48];		// Pointer to colr palette
        char reserved;				// Just as it says
        char num_color_planes;		// Number of colors per plane?
        int bytes_per_line;			// Number of bytes per line
        int palette_type;			// Type of color palette
        char padding[58];			// Extra padding
        } Pcx_Header, *Pcx_Header_Ptr;

typedef struct						// Structure of a PCX file
        {
        Pcx_Header header;			// A header of a PCX file
        RGB_Color palette[256];		// Array of RGB triple structures
        char far *buffer;			// Pointer to where data will go
        } Pcx_Picture, *Pcx_Picture_Ptr;

typedef struct						// Structure of a sprite
        {
        int nX;	            		// X Position of sprite
        int nY;	            		// Y Position of sprite
        int nX_Old;					// Old X position of sprite
        int nY_Old;					// Old Y position of sprite
        int nWidth;					// Width of sprite in pixels
        int nHeight;				// Height of sprite in pixels
		int nSpeed;					// Speed of the sprite
		int nDirection;				// Direction of the sprite
		int nPower;					// How much power, or fuel left.
        char far *cfpFrames[MAX_SPRITE_FRAMES]; // Array of pointers to the images
		int nFrameRate;				// Rate at which the frames are changed
		int nFrameClock;			// Counter used for rate
        int nCurrFrame;				// Current frame being displayed
        int nNumFrames;				// Total number of frames
        int nState;					// State of sprite, alive, dead...
        char far *cfpBackground;	// Whats under the sprite
        } Sprite, *Sprite_Ptr;

typedef struct						// Structure of a font
        {
        int nWidth;					// Width of font in pixels
        int nHeight;				// Height of font in pixels
        char far *cfpImage;			// Pointer to font image
        } Font, *Font_Ptr;


// ************************************************
// ************* Function Prototypes **************
// ************************************************
          	

// ********** Message Blitting **********

	void AllocateMessageBack(MessagePtrTyp);		// Allocates memory for the background of Message to be blitted
	void SaveMessageBackground(MessagePtrTyp);		// Copies the background of test to be blitted
	void Blit_Message(MessagePtrTyp);				// Blits Message
	void Blit_Char(int, int, char, int);			// Bit blits a character to the screen
	void Blit_String(int, int, int, char *);		// Bit blits an un-erasable string to the screen
	void EraseMessage(MessagePtrTyp);				// Erases Message that has been blitted
	void FreeMessageMem(MessagePtrTyp);				// Frees memory allocated for Message background
	void Plot_Pixel_Fast(int, int, unsigned char);	// Plots a pixel to the screen

// ********** Sprite **********
	void Sprite_Init(Sprite_Ptr, int, int, int, int);	// Allocates memory for sprite loading
	void Erase_Sprite(Sprite_Ptr);					// Erases a sprite by restoring its background
	void Behind_Sprite(Sprite_Ptr);					// Grabs sprite background
	void Draw_Sprite(Sprite_Ptr);					// Draws a sprite on the screen
	void Free_Sprite_Mem(Sprite_Ptr);				// Frees memory used to load sprite and it's background
	void PCX_Grap_Bitmap(Pcx_Picture_Ptr, Sprite_Ptr, int, int, int);	// Grabs images and saves it into sprite mem

// ********** Image stuff **********
	void Set_Palette_Register(int, RGB_Color_Ptr);	// Loads color palette
	void PCX_Init(Pcx_Picture_Ptr);					// Allocates memory for loading PCX image
	void Free_PCX_Mem(Pcx_Picture_Ptr);				// Frees memory used for loading PCX image
	void PCX_Load(char *, Pcx_Picture_Ptr, int);	// Loads PCX image from file
	void PCX_Show_Buffer(Pcx_Picture_Ptr);			// Displays PCX image to screen

// ********** Misc **********
	void Wait_For_Vsync(void);						// Waits for vertical sync
	void Set_Video_Mode(int);						// Sets display mode
	void _interrupt _far New_Key_Int(void);			// Grabs key hits, much better than Get_Scan_Code
	void _interrupt _far New_Timer_Int(void);		// Grabs timer timeouts, re-programmed timer
	int Get_Input(void);							// Checks to see if a key is pressed
	void Timer(int);								// Used for delays
	void Blast_Buffer(void);						// Copy double buffer
	void Change_Timer(unsigned int);				// Change re-program the timer chip
	void Configure_Timer(int);						// Configure timer and hook interrupt
	void Init_Double_Buffer(void);					// Initialize the double buffer
	void Init_Images(void);							// Initialize images
	void Init_Message(void);						// Inializes messages
	void AbEnd(void);								// If error occurs during initialization
	void UpDateShipPos(void);						// Update the ships position
	void Intro(void);								// Perform the introduction
