/**********************************************************************************************************

                        LANDER.C - Source code for the Lunar Lander Game


    DESCRIPTION:    Main implementation module for LANDER.EXE (well it's called that as of now).



    COMMENTS:       Some of the code and ideas in the LANDER project was adapted from Tricks Of
                    The Game Programming Gurus.


    Components:     LANDER.C    -   Main implementation file
                    LANDER.H    -   Header file for LANDER.C
                    LL-GRAFX.C  -   Graphics implementation file
                    LL-IO.C     -   Input Output implementation file


    Written By:		Joe Lambert
    Date:			06-23-95

    Revisions:
    06-23-95		So it began, stopped with just a ship moving on the screen.
    03-08-96		Added gravity, boundary checking, active fuel and score, crashes,
					number of lives.  Started looking like a real game instead of
					just sprite stuff.
    03-15-96		Broke code into separate modules for graphics functions and the like.
    03-20-96		Added hooks for keyboard and timer.  Broke code into IO module, out
					of near heap space.	
    03-24-96		Added sprite animation, messaging system, and started on fonts.
    10-23-96		Dropped fonts and did clean up.
    11-01-96        Lander went to X2FTP. I'm somebody now, I'm somebody now!!
***********************************************************************************************************/
#include <io.h>
#include <conio.h>
#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <bios.h>
#include <fcntl.h>
#include <memory.h>
#include <math.h>
#include <string.h>
#include <malloc.h>
#include "lander.h"

#pragma comment( exestr , "Written By: Joe Lambert")
#pragma comment( exestr , "Compiled on " __DATE__ " at " __TIME__ ) 

// ***** Globals *****

	Sprite sprite[NUM_SPRITES];			// Define array of Sprite structures
	MessageTyp Message[NUM_MESSAGE];	// Define array of MessageTyp structures

void main(int argc, char *argv[ ])
{
// ********** Define local variables **********
	int nDebugMode = 0;					// Used for debug mode.
	int nDone = 0;						// Used to terminate main while loop
	int nLandings = 0;					// Number of successful landings
	int nNumLives = LIVES_3;			// How many lives do we have when we start
	int nMessageTime = MESSAGE_TIME;	// How long a message we appear
	int nGameState = FLYING;			// Current state of the game
	int nLoopCount = 0;					// For loops
	int nColor = 0;						// Used for ending
	int nCrashTimer = CRASH_RATE;		// How long the crash sequence will last
	int nStateTimer = STATE_RATE;		// How long the current game state will last
	char szTempString[20];				// Scratch pad
	unsigned short usScoreValue = 0;	// Where score will be stored in decimal
	Pcx_Picture Scape_PCX;				// Define variable 'Scape_PCX' of type Pcx_Picture to hold scape


	if (argc > 1)						// Figure it out
	{
		if (!strcmp(argv[1], "GOD") || !strcmp(argv[1], "god"))
		{
			nDebugMode = 1;
		}
	}


// ********** Initialize globals  **********
	video_buffer = (unsigned char *)0xA0000000L;	// Video RAM byte pointer
	video_buffer_w= (unsigned short *)0xA0000000L;	// Video RAM word pointer
	ucfpRomCharSet = (unsigned char *)0xF000FA6EL;	// ROM characters 8x8
	uspCurrentTime = (unsigned short *)0x0000046CL;	// pointer to internal
	
	nKeyTable[0] = 0;							// Clear key 0
	nKeyTable[1] = 0;							// Clear key 1
	nKeyTable[2] = 0;							// Clear key 2
	nKeyTable[3] = 0;							// Clear key 3

// ********** Configure and Initialize **********
	Init_Double_Buffer();						// Initialize Double Buffer
	Init_Images();								// Initialize imagery
	Init_Message();								// Initialize message strings

	Configure_Timer(TIMER_60HZ);				// Configure timer and hook interrupt
	Set_Video_Mode(MODE13);						// Configure Video Mode

	Old_Key_Isr = _dos_getvect(KEYBOARD_INT);	// Save old keyboard interrupt vector
	_dos_setvect(KEYBOARD_INT, New_Key_Int);	// Install new ISR (Interrupt Service Routine)

	PCX_Init(&Scape_PCX);						// Allocate memory for PCX file
	PCX_Load(SCAPE_FILE, &Scape_PCX,1);			// Load the landscape PCX file and the palette
	PCX_Show_Buffer(&Scape_PCX);				// Copy landscape to VIDEO memory. Display the landscape
	Free_PCX_Mem(&Scape_PCX);					// Free memory used for landscape PCX file

	Intro();									// Call intro function

	nTimeOut = 0;								// Clear nTimeOut, it will be set via an interrupt hook

// ******************************************
// ********** Bit Blit information **********
// ******************************************
	SaveMessageBackground(&Message[TITLE]);		// Save background of title
	SaveMessageBackground(&Message[SUCCESS]);	// Save background of success
	SaveMessageBackground(&Message[LEVEL]);		// Save background of level
	SaveMessageBackground(&Message[LOW_FUEL]);	// Save background of low fuel
	SaveMessageBackground(&Message[SCORE]);		// Save background of score
	SaveMessageBackground(&Message[FUEL]);		// Save background of fuel

// ***** Setup some more stuff
	sprite[LIVES].nCurrFrame = LIVES_3_0;		// Set which LIVES frame (or sprite) to draw 
	sprite[LIVES].nFrameRate = LIVES_FRAMERATE;	// Set LIVES frame rate
	sprite[LIVES].nFrameClock = LIVES_FRAMERATE;// Set LIVES frame clock
	sprite[LIVES].nState = SPRITE_ALIVE;

	sprite[PLATFORM].nDirection = RIGHT;				// What direction is platform moving
	sprite[PLATFORM].nFrameRate = PLATFORM_FRAMERATE;	// Set PLATFORM frame rate
	sprite[PLATFORM].nFrameClock = PLATFORM_FRAMERATE;	// Set PLATFORM frame clock
	sprite[PLATFORM].nFrameClock = PLATFORM_FRAMERATE;	// Set PLATFORM frame clock
	sprite[PLATFORM].nState = SPRITE_ALIVE;

	sprite[SHIP].nSpeed = ZERO_SPEED;			// Set ship initial speed
	sprite[SHIP].nPower = FULL_TANK;			// Fill-er-up
	sprite[SHIP].nFrameRate = SHIP_FRAMERATE;	// Set SHIP frame rate					
	sprite[SHIP].nFrameClock = SHIP_FRAMERATE;	// Set SHIP frame clock					
	sprite[SHIP].nFrameClock = SHIP_FRAMERATE;	// Set SHIP frame clock					
	sprite[SHIP].nState = SPRITE_ALIVE;

	Behind_Sprite(&sprite[LIVES]);				// Save the background before we draw LIVES sprite
	Behind_Sprite(&sprite[SHIP]);				// Save the background before we draw SHIP sprite
	Behind_Sprite(&sprite[PLATFORM]);			// Save the background before we draw PLATFORM sprite

	Draw_Sprite(&sprite[SHIP]);					// Draw the shaceship sprite
	Draw_Sprite(&sprite[LIVES]);				// Draw the lives sprite
	Draw_Sprite(&sprite[PLATFORM]);				// Draw the platform sprite

	strcpy( Message[LEVEL].szString, LEVEL_TITLE);	// Begin to build fuel string
	itoa( (nLandings + 1), szTempString, 10);		// Convert fuel from decimal value to ASCII string
	strcat(Message[LEVEL].szString, szTempString);	// Build fuel string
	Message[LEVEL].nState = TEMPERARY;				// Turn on the message

// Yes I know we did't blast the double buffer yet

// *******************************
// ********** Main loop **********
// *******************************
	while (!nDone)								// While nDone is not 1
	{
		Get_Input();							// Call function to get keys pressed
	    if (nRawKey == 1)						// End if user is exiting
	       {
		       nDone = 1;						// Set nDone to 1 to end
	       }

// ********** Decode Key hits **********
		if (nGameState == FLYING)				// If we are happy go lucky flying around...
		{
			if (!sprite[SHIP].nFrameClock--)	// Is it time to change the ship sprite frame for animation 
			{
				UpDateShipPos();				// Go check keyhit to change ship pos (really need to change this)
				sprite[SHIP].nFrameClock = SHIP_FRAMERATE;	// Reset frame clock					
			}
	
	// **** Update Fuel ****
			strcpy( Message[FUEL].szString, FUEL_TITLE);	// Begin to build fuel string
			itoa( sprite[SHIP].nPower, szTempString, 10);	// Convert fuel from decimal value to ASCII string
			strcat(Message[FUEL].szString, szTempString);	// Build fuel string
	
			if (sprite[SHIP].nPower == 100)					// If fuel is getting REAL low
			{
				Message[FUEL].nColor = RED;					// Fuel color turns RED
				Message[LOW_FUEL].nState = TEMPERARY;		// Turn on message
			}
			else if (sprite[SHIP].nPower == 250)			// If the fuel is just low
			{
				Message[FUEL].nColor = YELLOW;				// Turn it yellow
			}


	// **** Update Platform posistion ****
			if (!sprite[PLATFORM].nFrameClock--)			// If it's time to change frames...
			{
				if (sprite[PLATFORM].nDirection == RIGHT)						// Is platform moving RIGHT
				{
					if (sprite[PLATFORM].nX++ == (SCREEN_WIDTH - SPRITE_WIDTH))	// Have we hit the RIGHT wall
					{
						sprite[PLATFORM].nDirection = LEFT;						// We hit the RIGHT, change direction
						sprite[PLATFORM].nX--;									// Start heading LEFT
					}
					if (sprite[PLATFORM].nCurrFrame++ > PLATFORM_FRAME_TWO)		// Are we on the last frame
					{
						sprite[PLATFORM].nCurrFrame = PLATFORM_FRAME_ZERO;		// Reset frame
					}
				}
				else															// Is platform moving LEFT
				{
					if (sprite[PLATFORM].nX-- == 0)								// Have we hit the left wall
					{
						sprite[PLATFORM].nDirection = RIGHT;					// Start heading RIGHT
						sprite[PLATFORM].nX++;									// Move RIGHT one
					}
					if (sprite[PLATFORM].nCurrFrame-- == PLATFORM_FRAME_ZERO)	// Are we on the first most frame
					{
						sprite[PLATFORM].nCurrFrame = PLATFORM_FRAME_THREE;		// Set frame to last frame, frames go backwards now
					}
				}			
				sprite[PLATFORM].nFrameClock = sprite[PLATFORM].nFrameRate;		// Reset frame clock
			}

		}// End of If !sprite[PLATFORM]

// **** Update Lives ****
		if (!sprite[LIVES].nFrameClock--)						// If it's time to change LIVES frame
		{
			sprite[LIVES].nCurrFrame++;							// Go to next frame
			switch (nNumLives)
			{
				case LIVES_3:									// Do we currently have three lives
					if (sprite[LIVES].nCurrFrame < LIVES_3_0 || sprite[LIVES].nCurrFrame >= LIVES_3_5)	// Is the current frame within the series
					{
						sprite[LIVES].nCurrFrame = LIVES_3_0;	// Set it to the first frame within the series
					}
					break;
								
				case LIVES_2:									// Do we currently have two lives
					if (sprite[LIVES].nCurrFrame < LIVES_2_6 || sprite[LIVES].nCurrFrame >= LIVES_2_11)	// Is the current frame within the series
					{
						sprite[LIVES].nCurrFrame = LIVES_2_6;	// Set it to the first frame within the series
					}
					break;
							
				case LIVES_1:									// Do we currently have one life
					if (sprite[LIVES].nCurrFrame < LIVES_1_12  || sprite[LIVES].nCurrFrame >= LIVES_1_17)// Is the current frame within the series
					{
						sprite[LIVES].nCurrFrame = LIVES_1_12;	// Set it to the first frame within the series
					}
					break;
			}
			sprite[LIVES].nFrameClock = sprite[LIVES].nFrameRate;// Reset frame clock					
		}

// Collision detect
		// Are we at the same height as platform surface AND Is our RIGHT past the LEFT side of platform AND 
		// Is our LEFT before the RIGHT side of platform

		if ((sprite[SHIP].nY + SPRITE_HEIGHT) == sprite[PLATFORM].nY		// If ship bottom = platform top
			&& (sprite[SHIP].nX + SPRITE_WIDTH) > sprite[PLATFORM].nX		// AND is ship right side > platform left side
			&& sprite[SHIP].nX < (sprite[PLATFORM].nX + SPRITE_WIDTH))		// AND Is ship left side < platform right side
		{																	// ********** LANDED **********
			nGameState = LANDED;		// Set game state to LANDED
		}

// Have we hit bottom OR Is our RIGHT past the LEFT side of the platform AND Is our LEFT before the RIGHT side of
// platform AND Are we below the platform surface
		
		else if ((sprite[SHIP].nY + SPRITE_HEIGHT) >= SCREEN_HEIGHT			// Are we on the bottom of the screen
				|| (sprite[SHIP].nX + SPRITE_WIDTH) > sprite[PLATFORM].nX	// OR is ship right side > platform left side
				&& sprite[SHIP].nX < (sprite[PLATFORM].nX + SPRITE_WIDTH)	// AND Is ship left side < platform right side
				&& (sprite[SHIP].nY + SPRITE_HEIGHT) > sprite[PLATFORM].nY)	// AND are we below the platform top, landed too fast.
		{																	// *** CRASH ***
			if (nDebugMode)									// If we are in debug mode...
			{
				nGameState = LANDED;						// Set game state to LANDED
			}
			else
			{
				nGameState = CRASH;							// Set game state to CRASH
			}
		}
		else	// ********** No collisions, No landing, No crashes, still flying *********
		{
			nGameState = FLYING;							// Set game state to FLYING
		}

		switch (nGameState)									// Test game state
		{
			case FLYING:									// Are we happy go lucky FLYING
				break;										// Then just break
				
			case LANDED:									// Have we landed safely
				Message[SUCCESS].nState = TEMPERARY;		// Let the player know he (oops) he or she landed safely

				if (!nStateTimer--)							// Check to see if our success inform time has run out
				{
					nStateTimer = STATE_RATE;				// Reset state timer
					nLandings++;							// Add to successful landings
					if (usScoreValue < MAX_SCORE)			// Make sure we're not at Max score
					{
						usScoreValue += sprite[SHIP].nPower;// Increase score by fuel left
					}
					strcpy( Message[SCORE].szString, SCORE_TITLE);	// Begin to build Score string
					itoa(usScoreValue, szTempString, 10);			// Convert Score from decimal value to ASCII string
					strcat(Message[SCORE].szString, szTempString);	// Build Score string
		
					strcpy( Message[FUEL].szString, FUEL_TITLE);	// Begin to build fuel string
					itoa( sprite[SHIP].nPower, szTempString, 10);	// Convert fuel from decimal value to ASCII string
					strcat(Message[FUEL].szString, szTempString);	// Build fuel string
					Message[FUEL].nColor = GREEN;
		
					strcpy( Message[LEVEL].szString, LEVEL_TITLE);	// Begin to build fuel string
					itoa( (nLandings + 1), szTempString, 10);		// Convert fuel from decimal value to ASCII string
					strcat(Message[LEVEL].szString, szTempString);	// Build fuel string
					Message[LEVEL].nState = TEMPERARY;

					sprite[SHIP].nCurrFrame = NO_ENGINE_0;	// No engines fired
					sprite[SHIP].nPower = FULL_TANK;		// Fuel up
					sprite[SHIP].nSpeed = ZERO_SPEED;		// Set Speed to zero
					sprite[SHIP].nX = SHIP_START_X;			// Start sprite back at X beginning
					sprite[SHIP].nY = SHIP_START_Y;			// Start sprite back at Y beginning
					nGameState = FLYING;
				}// End if !nStateTimer--
				break;
				
			case CRASH:
				if (!nCrashTimer--)							// If time's up for this frame
				{
					sprite[SHIP].nCurrFrame++;				// Next frame
					if (sprite[SHIP].nCurrFrame < CRASH_1  || sprite[SHIP].nCurrFrame >= CRASH_6)	// Out of frames?
					{
						sprite[SHIP].nCurrFrame = CRASH_1;	// Set frame to first
					}
					nCrashTimer = CRASH_RATE;				// Reset frame timer
				}
				if (!nStateTimer--)							// If time's up for the crash sequence
				{
					nGameState = FLYING;					// Reset game state to flying
					if ( nNumLives-- > 1)					// If we're ** Not out of lives **
					{
						sprite[SHIP].nSpeed = ZERO_SPEED;	// Set Speed to zero
						sprite[SHIP].nX = SHIP_START_X;		// Start sprite back at beginning
						sprite[SHIP].nY = SHIP_START_Y;		// Start sprite back at beginning
						switch (nNumLives)					// Switch on how many lives we have left
						{
							case LIVES_2:					// If we only have two lives
								sprite[LIVES].nCurrFrame = LIVES_2_6;	// Set correct sprite frame
								break;						// Break out
								
							case LIVES_1:					// Only one life left?
								sprite[LIVES].nCurrFrame = LIVES_1_12;	// Set correct frame
								break;						// Break out
						}
						Message[FUEL].nColor = GREEN;		// Fuel is full again, go to green
						sprite[SHIP].nCurrFrame = NO_ENGINE_0;// Set which frame (or sprite) to draw 
						sprite[SHIP].nPower = FULL_TANK;	// Two more lives isn't much good without gas 
						nGameState = FLYING;				// Set game state to flying
					}
					else									// If we're ** Out of lives **
					{
						nDone = 1;							// Set nDone to 1
					}
					nStateTimer = STATE_RATE;				// Reset state timer
				}// End if !nStateTimer--
				break;	
		}// End switch (nGameState)

// ***** Redraw sprites, messages, Message, etc. *****
		for (nLoopCount = 0; nLoopCount < NUM_MESSAGE; nLoopCount++)// Loop through all messages
		{
			if (Message[nLoopCount].nState != GONE)					// If it's not gone...
			{
				EraseMessage(&Message[nLoopCount]);					// Erase messages before we update sprites
				if ( Message[nLoopCount].nState == TEMPERARY && !Message[nLoopCount].nMessageTime--)	// Will this kill it?
				{
					Message[nLoopCount].nState = GONE;				// If this is it's last time through, kill it
					Message[nLoopCount].nMessageTime = MESSAGE_TIME;// Reset time for next time
				}				
			}
		}

// We have to erase them all just in case any of them overlap.
		for (nLoopCount = 0; nLoopCount < NUM_SPRITES; nLoopCount++)// Loop throught all sprites
		{
			if (sprite[nLoopCount].nState == SPRITE_ALIVE)			// If it's alive...
			{
				Erase_Sprite(&sprite[nLoopCount]);					// Erase sprite at old position
			}
		}
// Must have a clean slate before we start grabbing background.
		for (nLoopCount = 0; nLoopCount < NUM_SPRITES; nLoopCount++)// Loop through all sprites again
		{
			if (sprite[nLoopCount].nState == SPRITE_ALIVE)			// If it's alive...
			{
				Behind_Sprite(&sprite[nLoopCount]);					// Save the background at new postition
			}
		}
		for (nLoopCount = 0; nLoopCount < NUM_SPRITES; nLoopCount++)// Loop through all sprites again.
		{
			if (sprite[nLoopCount].nState == SPRITE_ALIVE)			// If it's alive...
			{
				Draw_Sprite(&sprite[nLoopCount]);					// Draw sprite at new position
				sprite[nLoopCount].nX_Old = sprite[nLoopCount].nX;	// Update old X position indicator
				sprite[nLoopCount].nY_Old = sprite[nLoopCount].nY;	// Update old Y position indicator
			}
			else if (sprite[nLoopCount].nState == SPRITE_DYING)		// For future use
			{
			}
			else if (sprite[nLoopCount].nState == SPRITE_DEAD)		// For future use
			{
			}
		}

			for (nLoopCount = 0; nLoopCount < NUM_MESSAGE; nLoopCount++)// Loop through all the messages
			{
				if (Message[nLoopCount].nState != GONE)				// If it's not gone...
				{
					Blit_Message(&Message[nLoopCount]);				// Blit new Message
				}
			}
			
// ********** BLAST IT ALL TO THE SCREEN **********
		Wait_For_Vsync();								// Wait for verticel retrace so we don't get a flicker
		Blast_Buffer();									// Copy double buffer. Blast that sucker
		while(!nTimeOut);								// Hang out until timer lets us go, this is for game speed control.
		nTimeOut = 0;									// Timer provides independent game speed.
	}// ***** End While Loop *****

// *********************
// ***** GAME OVER *****
// *********************

// ***** Blit final scores and info *****
	EraseMessage(&Message[SCORE]);						// Erase current score value
	EraseMessage(&Message[FUEL]);						// Erase current fuel value
	Erase_Sprite(&sprite[SHIP]);						// Erase ship sprite at old position
	Erase_Sprite(&sprite[LIVES]);						// Erase Lives sprite at old position

// NOTE: At this point no more erasing will take place, Blit at will.
	Message[SCORE].nX_Pos = (SCREEN_WIDTH - (strlen(Message[SCORE].szString) * 8)) / 2;	// Auto center X posistion of Score
	Message[SCORE].nY_Pos = 70;							// Y posistion of Score
	Message[SCORE].nColor = WHITE;						// Color of Score
	Blit_Message(&Message[SCORE]);						// Blit Score
	
	sprintf(szTempString, "Successful Landings  %d", nLandings);	// Build landing count string
	Blit_String( ((SCREEN_WIDTH - (strlen(szTempString) * 8)) / 2), 80, BLUE, szTempString); // Blit it center screen

// ********** KLUDGE WARNING **********
// ********** KLUDGE WARNING **********
// ********** KLUDGE WARNING **********
	nColor = 255;					// Color we'll use
	nLoopCount = 0;					// For loops

	while (nDone)					// While nDone is 1
	{
		Get_Input();				// Call function to get keys pressed
	    if (nRawKey == 1)			// If user is still hitting ESC
	       {
		       nDone = 1;			// Set nDone to 1
	       }
		else
		{
			nDone = 0;				// Clear nDone
		}
	}

	while (!nDone)					// While nDone is not 1
	{
		Get_Input();				// Call function to get keys pressed
	    if (nRawKey == 1)			// End if user is exiting
	       {
		       nDone = 1;			// Set nDone to 1
	       }

		if (nLoopCount++ > 4)		// If it's the fourth time through
		{
			if (nColor-- < 245)		// Decrement color, is it less than 245
			{
				nColor = 255;		// Set it to 255
			}
			Blit_String(1, 0, (nColor - 6), "Press Esc to exit...");	// Blit string shadow
			Blit_String(0, 0, nColor, "Press Esc to exit...");			// Blit string
			Wait_For_Vsync();		// Wait for verticel sync so we don't get a flicker
			Blast_Buffer();			// Copy double buffer
			nLoopCount = 0;			// Clear the loop counter

			while(!nTimeOut);		// Hang out until timer lets us go
			nTimeOut = 0;
		}
	}
// ********** END KLUDGE WARNING **********
// ********** END KLUDGE WARNING **********
// ********** END KLUDGE WARNING **********

// ***** Restore interrupt hooks and timer *****

	Change_Timer(TIMER_18HZ);						// Re-program the timer chip;
	_dos_setvect(TIME_KEEPER_INT, Old_Timer_Isr);	// Restore old Timer ISR
	_dos_setvect(KEYBOARD_INT, Old_Key_Isr);		// Restore old keyboard ISR

	_ffree(Double_Buff);							// Free memory allocated for double buffer
	Free_Sprite_Mem(&sprite[SHIP]);					// Free memory allocated for ship sprite
	Free_Sprite_Mem(&sprite[LIVES]);				// Free memory allocated for Lives sprite
	Free_Sprite_Mem(&sprite[PLATFORM]);				// Free memory allocated for platform sprite

	for (nLoopCount = 0; nLoopCount < NUM_MESSAGE; nLoopCount++)
	{
		FreeMessageMem(&Message[nLoopCount]);		// Free memory allocated for Message
	}

	Set_Video_Mode(TEXT_MODE);						// Set text mode

	for (nLoopCount=0; nLoopCount<=30000; nLoopCount++)
	{
		Plot_Pixel_Fast(rand()%320, rand()%200, 0);	// Fade out with random pixel placement
	}

} // end main ***** LATER *****



/********************************************************************
	
	Function:
		VOID Configure_Timer(INT)
	
	Description:
		Initialize timer setup and hook timer interrupt.
					
	Returns:
		None
		
	Comments:
  
*********************************************************************/
void Configure_Timer(int nTimer_Setting)
{
	Change_Timer(nTimer_Setting);					// Re-program the timer chip;
	Old_Timer_Isr = _dos_getvect(TIME_KEEPER_INT);	// Save old Timer interrupt vector
	_dos_setvect(TIME_KEEPER_INT, New_Timer_Int);	// Install new ISR
}// End of Configure_Timer function


/********************************************************************
	
	Function:
		VOID Init_Ship(VOID)
	
	Description:
		Loads ship images from PCX file.
					
	Returns:
		None
		
	Comments:
		PCX file MUST be open.
  
*********************************************************************/
void Init_Images(void)
{
	Pcx_Picture Sprites_PCX;					// Define variable 'Sprites_PCX' of type Pcx_Picture. Entire PCX picture

	PCX_Init(&Sprites_PCX);						// Allocate memory
	PCX_Load(SPRITE_FILE, &Sprites_PCX,0);		// Load the Sprites_PCX PCX file

	Sprite_Init(&sprite[SHIP],SHIP_START_X,SHIP_START_Y, SPRITE_WIDTH, SPRITE_HEIGHT);	// Initialize 'SHIP' sprite
	Sprite_Init(&sprite[LIVES],LIVES_X,LIVES_Y, SPRITE_WIDTH, SPRITE_HEIGHT);			// Initialize 'LIVES' sprite
	Sprite_Init(&sprite[PLATFORM],PLATFORM_X,PLATFORM_Y, SPRITE_WIDTH, SPRITE_HEIGHT);	// Initialize 'PLATFORM' sprite

	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[SHIP],0,0,0);		// Frame one of ship with no engines fired
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[SHIP],1,1,0);		// Frame two of ship with no engines fired
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[SHIP],2,2,0);		// Frame three of ship with no engines fired
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[SHIP],3,3,0);		// Frame four of ship with no engines fired
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[SHIP],4,4,0);		// Frame five of ship with no engines fired
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[SHIP],5,5,0);		// Frame six of ship with no engines fired

	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[SHIP],6,6,0);		// Frame one of ship with main engine fired
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[SHIP],7,7,0);		// Frame two of ship with main engine fired
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[SHIP],8,8,0);		// Frame three of ship with main engine fired
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[SHIP],9,9,0);		// Frame four of ship with main engine fired
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[SHIP],10,10,0);	// Frame five of ship with main engine fired
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[SHIP],11,11,0);	// Frame six of ship with main engine fired

	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[SHIP],12,0,1);		// First frame of crash
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[SHIP],13,1,1);		// Second frame of crash
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[SHIP],14,2,1);		// Third frame of crash
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[SHIP],15,3,1);		// Fourth frame of crash
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[SHIP],16,4,1);		// Fifth frame of crash
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[SHIP],17,5,1);		// Sixth frame of crash

	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[LIVES],0,0,4);		// Frame one of lives
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[LIVES],1,1,4);		// Frame two of lives
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[LIVES],2,2,4);		// Frame three of lives
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[LIVES],3,3,4);		// Frame four of lives
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[LIVES],4,4,4);		// Frame five of lives
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[LIVES],5,5,4);		// Frame six of lives

	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[LIVES],6,6,3);		// Frame one of lives
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[LIVES],7,7,3);		// Frame two of lives
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[LIVES],8,8,3);		// Frame three of lives
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[LIVES],9,9,3);		// Frame four of lives
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[LIVES],10,10,3);	// Frame five of lives
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[LIVES],11,11,3);	// Frame six of lives

	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[LIVES],12,0,3);	// Frame one of lives
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[LIVES],13,1,3);	// Frame two of lives
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[LIVES],14,2,3);	// Frame three of lives
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[LIVES],15,3,3);	// Frame four of lives
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[LIVES],16,4,3);	// Frame five of lives
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[LIVES],17,5,3);	// Frame six of lives

	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[PLATFORM],0,0,2);	// Frame one of platform
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[PLATFORM],1,1,2);	// Frame two of platform
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[PLATFORM],2,2,2);	// Frame three of platform
	PCX_Grap_Bitmap(&Sprites_PCX,&sprite[PLATFORM],3,3,2);	// Frame four of platform

	Free_PCX_Mem(&Sprites_PCX);								// Free the memory used for Sprites_PCX PCX file
}// End of Init_Ship function


/********************************************************************
	
	Function:
		VOID Init_Message(VOID)
	
	Description:
		Allocates memory and initializes Message structures
					
	Returns:
		None
		
	Comments:
  
*********************************************************************/
void Init_Message(void)
{
	int nLoopCount;					// For loops
	char szTempString[20];			// Scratch pad

	strcpy (Message[LOW_FUEL].szString, LOW_FUEL_TITLE);// Initialize title string 

	Message[LOW_FUEL].nMaxStringLength = strlen(Message[LOW_FUEL].szString);		// Title length is static
	Message[LOW_FUEL].nX_Pos = (SCREEN_WIDTH - (strlen(Message[LOW_FUEL].szString) * 8)) / 2;	// Auto center X posistion of title
	Message[LOW_FUEL].nY_Pos = LOW_FUEL_Y_POS;			// Y posistion of title
	Message[LOW_FUEL].nColor = RED;						// Color of title
	Message[LOW_FUEL].nState = GONE;					// Set message state to GONE
	Message[LOW_FUEL].nMessageTime = MESSAGE_TIME;		// Reset timer

	strcpy (Message[LEVEL].szString, "Level ");			// Initialize title string 
	itoa( 9999, szTempString, 10);						// Convert score from decimal value to ASCII string
	strcat(Message[LEVEL].szString, szTempString);		// Build score string

	Message[LEVEL].nMaxStringLength = strlen(Message[LEVEL].szString);				// Title length is static
	Message[LEVEL].nX_Pos = (SCREEN_WIDTH - (strlen(Message[LEVEL].szString) * 8)) / 2;	// Auto center X posistion of title
	Message[LEVEL].nY_Pos = LEVEL_Y_POS;				// Y posistion of title
	Message[LEVEL].nColor = RED;						// Color of title
	Message[LEVEL].nState = GONE;						// Set state to gone
	Message[LEVEL].nMessageTime = MESSAGE_TIME;			// Reset timer

	strcpy (Message[TITLE].szString, TITLE_TITLE);		// Initialize title string 

	Message[TITLE].nMaxStringLength = strlen(Message[TITLE].szString);	// Title length is static
	Message[TITLE].nX_Pos = (SCREEN_WIDTH - (strlen(Message[TITLE].szString) * 8)) / 2;	// Auto center X posistion of title
	Message[TITLE].nY_Pos = TITLE_Y_POS;				// Y posistion of title
	Message[TITLE].nColor = BLUE;						// Color of title
	Message[TITLE].nState = GONE;						// Set state to gone
	Message[TITLE].nMessageTime = MESSAGE_TIME;			// Reset timer

// ********************************************************
// ********** Success information initialization **********
// ********************************************************
	strcpy (Message[SUCCESS].szString, SUCCESS_TITLE);// Initialize success string 

	Message[SUCCESS].nMaxStringLength = strlen(Message[SUCCESS].szString);// Success length is static
	Message[SUCCESS].nX_Pos = (SCREEN_WIDTH - (strlen(Message[SUCCESS].szString) * 8)) / 2;// Auto center X posistion of success
	Message[SUCCESS].nY_Pos = SUCCESS_Y_POS;		// Y posistion of success
	Message[SUCCESS].nColor = WHITE;				// Color of success
	Message[SUCCESS].nState = GONE;					// Set state to gone
	Message[SUCCESS].nMessageTime = MESSAGE_TIME;	// Reset timer

// ******************************************************
// ********** Score information initialization **********
// ******************************************************
	strcpy (Message[SCORE].szString, SCORE_TITLE);	// Initialize score string 
	itoa( 0, szTempString, 10);						// Convert score from decimal value to ASCII string
	strcat(Message[SCORE].szString, szTempString);	// Build score string

	Message[SCORE].nMaxStringLength =  SCORE_LENGTH;// Assign max string length for memory allocation
	Message[SCORE].nX_Pos = SCORE_X_POS;			// Auto center X posistion of score
	Message[SCORE].nY_Pos = SCORE_Y_POS;			// Y posistion of score
	Message[SCORE].nColor = WHITE;					// Color of score
	Message[SCORE].nState = PERMENENT;				// Set state to permenent
	Message[SCORE].nMessageTime = MESSAGE_TIME;		// Reset timer

// *****************************************************
// ********** Fuel information initialization **********
// *****************************************************
	strcpy (Message[FUEL].szString, FUEL_TITLE);	// Initialize fuel string 
	itoa( FULL_TANK, szTempString, 10);				// Convert fuel from decimal value to ASCII string
	strcat(Message[FUEL].szString, szTempString);	// Build fuel string

	Message[FUEL].nMaxStringLength =  FUEL_LENGTH;	// Assign max string length for memory allocation
	Message[FUEL].nX_Pos = FUEL_X_POS;				// Auto center X posistion of fuel
	Message[FUEL].nY_Pos = FUEL_Y_POS;				// Y posistion of fuel
	Message[FUEL].nColor = GREEN;					// Color of fuel
	Message[FUEL].nState = PERMENENT;				// Set state to permenent
	Message[FUEL].nMessageTime = MESSAGE_TIME;		// Reset timer

// ********** Allocate memory **********
// ***** Loop through Message strings allocating memory for thier background *****
	for (nLoopCount = 0; nLoopCount < NUM_MESSAGE; nLoopCount++)	// Loop through all messages
	{
		AllocateMessageBack(&Message[nLoopCount]);	// Allocate the memory
	}

}// End of Init_Message function


/********************************************************************
	
	Function:
		VOID UpDateShipPos(VOID)
	
	Description:
		Check to see if the ship needs to be moved.
					
	Returns:
		None
		
	Comments:
		What happens if the memory was never allocated?
  
*********************************************************************/
void UpDateShipPos(void)
{

// ***** LEFT Key pressed *****
    if (nKeyTable[INDEX_LEFT])						// Was the move LEFT command given
	{
		if (sprite[SHIP].nX > 0)					// Make sure we're not all the way to the left
		{
			sprite[SHIP].nX_Old = sprite[SHIP].nX;	// Save sprite[SHIP].nX so we can restore the background
			sprite[SHIP].nX--;						// Decrement the Ship's location to the LEFT
		}
	}
// ***** RIGHT Key pressed *****
    else if (nKeyTable[INDEX_RIGHT])				// Was the move RIGHT command given
	{
		if (sprite[SHIP].nX < (SCREEN_WIDTH - SPRITE_WIDTH))// Have we hit the RIGHT wall
		{
			sprite[SHIP].nX_Old = sprite[SHIP].nX;	// Save sprite[SHIP].nX so we can restore the background
			sprite[SHIP].nX++;						// Increment the ship's location to the RIGHT
		}
	}

// ***** UP key pressed ****
	if (nKeyTable[INDEX_UP] && sprite[SHIP].nPower)	// If UP key pressed and we have gas
	{
		sprite[SHIP].nPower -= 1;					// Decrease amount of fuel
		sprite[SHIP].nCurrFrame++;					// Next frame
		if (sprite[SHIP].nCurrFrame < MAIN_ENGINE_6  || sprite[SHIP].nCurrFrame >= MAIN_ENGINE_11)	// Out of frames?
		{
			sprite[SHIP].nCurrFrame = MAIN_ENGINE_6;// Set frame to first
		}

		if (sprite[SHIP].nSpeed > ZERO_SPEED)		// **** Are we moving UP engines fired
		{
			if (sprite[SHIP].nSpeed < MAX_SPEED)	// Are we less then max speed
			{
				sprite[SHIP].nSpeed += GRAVITY;		// Speed up
			}
			if ((sprite[SHIP].nY -= (int) (sprite[SHIP].nSpeed / SPEED_DIVIDE)) <= 0)	// Make sure we're not all the way UP
			{
				sprite[SHIP].nY = 0;				// If we're not all the way up, move up
			}										
		}
		else if (sprite[SHIP].nSpeed <= ZERO_SPEED)	// **** Are we falling with engines fired
		{	
			if (sprite[SHIP].nSpeed < MAX_SPEED)	// Are we below MAX speed
			{
				sprite[SHIP].nSpeed += (int)(GRAVITY / 2);	// Increment speed, slow the fall
			}
// Check to see if we are all the way at the bottom of the screen
			if ( (sprite[SHIP].nY -= (int)(sprite[SHIP].nSpeed / SPEED_DIVIDE)) > (SCREEN_HEIGHT - SPRITE_HEIGHT))
			{
				sprite[SHIP].nY = (SCREEN_HEIGHT - SPRITE_HEIGHT);	// If so don't go any lower
			}
		}
	}// End of UP key pressed	

	else	// Either the UP Key not pressed or out of gas
	{
		if (sprite[SHIP].nSpeed > MIN_SPEED)		// Are we above MIN speed
		{
			sprite[SHIP].nSpeed -= GRAVITY;			// If so, decrease speed
		}

		sprite[SHIP].nCurrFrame++;					// Next frame
		if (sprite[SHIP].nCurrFrame < NO_ENGINE_0 || sprite[SHIP].nCurrFrame >= NO_ENGINE_5)	// Out of frames
		{
			sprite[SHIP].nCurrFrame = NO_ENGINE_0;	// Reset frames
		}

		if (sprite[SHIP].nSpeed > ZERO_SPEED)		// ** Are we moving UP without engines
		{
			if ((sprite[SHIP].nY -= (int) (sprite[SHIP].nSpeed / SPEED_DIVIDE)) <= 0)	// Make sure we're not all the way UP
			{
				sprite[SHIP].nY = 0;				// If we're all the way up, don't go any higher
			}
		} 
		else if (sprite[SHIP].nSpeed <= ZERO_SPEED) // ** Are we falling without engines *****
		{
// Make sure we're not at the bottom
			if ( (sprite[SHIP].nY -= (int) (sprite[SHIP].nSpeed / SPEED_DIVIDE)) > (SCREEN_HEIGHT - SPRITE_HEIGHT))
			{
				sprite[SHIP].nY = (SCREEN_HEIGHT - SPRITE_HEIGHT);	// If so then don't go any lower
			}
		} 
	}// End of UP key not pressed	
}

/********************************************************************
	
	Function:
		VOID Intro(VOID)
	
	Description:
		Does the fancy sprite introduction thing.
					
	Returns:
		None
		
	Comments:
		PCX file MUST be open.
  
*********************************************************************/
void Intro()
{
	int nRowGrab = 0;		// Row to grab sprite from
	int nColGrab = 0;		// Column to grab sprite from
	int nOneStillAlive;		// Is there at least one sprite left
	int nLoopCount;			// For loop counting
	int nDone = 0;			// Are we done
	int nAbort = 1;			// Use hit the ESC key

	Pcx_Picture IntroFrames;			// Variable for entire PCX picture
	Sprite sprite[NUM_INTRO_SPRITES];	// Array of sprites

// ********** Load sprite imagery **********
	PCX_Init(&IntroFrames);					// Alocate memory for IntroFrames PCX file
	PCX_Load(INTRO_FILE, &IntroFrames,0);	// Load the IntroFrames PCX file

// ********** Loop through sprite array to initialize ********** 
	nRowGrab = ROW_POS;		// Initialize row position
	nColGrab = COL_POS;		// Initialize cloumn

	for (nLoopCount = 0; nLoopCount < NUM_INTRO_SPRITES; nLoopCount++)	// Loop through all sprites
	{
		Sprite_Init(&sprite[nLoopCount],nColGrab, nRowGrab, SPRITE_WIDTH, SPRITE_HEIGHT);	// Allocate memory
		sprite[nLoopCount].nState = SPRITE_ALIVE;		// Set state to alive
		sprite[nLoopCount].nDirection = rand()%4;		// Set direction

		if ((nColGrab += SPRITE_WIDTH) == 244)			// If we're all the way over
		{
			if ((nRowGrab += SPRITE_HEIGHT) == 136)		// If we're all the way down
			{
				nRowGrab = ROW_POS;						// Go to start of row
			}
			nColGrab = COL_POS;							// Go to top of column
		}
	}
  
// ********** Loop through sprite array to load sprites ********** 
	nRowGrab = 0;
	nColGrab = 0;
	for (nLoopCount = 0; nLoopCount < NUM_INTRO_SPRITES; nLoopCount++)	// Loop through all the sprites
	{
		PCX_Grap_Bitmap(&IntroFrames,&sprite[nLoopCount],0,nColGrab,nRowGrab);	// Grab individual sprite images from PCX file.
		if (nColGrab++ == (MAX_COL) - 1)		// If we're at the end of column
		{
			if (nRowGrab++ == (MAX_ROW))		// If we'er at the end of the row
			{
				nRowGrab = 0;					// Reset row
			}
			nColGrab = 0;						// Reset column
		}
	}
	Free_PCX_Mem(&IntroFrames);					// Free the memory used for IntroFrames PCX file

// ********** Loop through sprite array to draw sprites ********** 
	for (nLoopCount = 0; nLoopCount < NUM_INTRO_SPRITES; nLoopCount++)	// Loop through all sprites
	{
		Behind_Sprite(&sprite[nLoopCount]);		// Save the background before we draw a sprite
		Draw_Sprite(&sprite[nLoopCount]);		// Draw the sprite
	}
	Wait_For_Vsync();						// Wait for verticel sync so we don't get a flicker
	Blast_Buffer();							// Copy double buffer


// Sit and wait for user to hit ESC
	while (!nDone)					// While nDone is not 1
	{
		Get_Input();				// Call function to get keys pressed
	    if (nRawKey == 1)			// End if user is exiting
	       {
		       nDone = 1;			// Set nDone to 1
	       }
		else
		{
			nDone = 0;				// Set nDone to 1
		}
	}

// User hit ESC, now wait until he lets it go
	while (nDone)					// While nDone is not 1
	{
		Get_Input();				// Call function to get keys pressed
	    if (nRawKey == 1)			// End if user is exiting
	       {
		       nDone = 1;			// Set nDone to 1
	       }
		else
		{
			nDone = 0;				// Set nDone to 1
		}
	}

// Okay we're free and clear, do the intro stuff...

// And I ran, I ran so far away, I just ran, I ran all night and day...
// and I ran, I ran so far away, I just ran, I couldn't get away...

// ********** Intro loop **********
	while (!nDone)					// While nDone is not 1
	{
		Get_Input();				// Call function to get keys pressed
	    if (nRawKey == 1)			// End if user is exiting
	       {
		       nDone = 1;			// Set nDone to 1
	       }
	
		for (nLoopCount = 0; nLoopCount < NUM_INTRO_SPRITES; nLoopCount++)	// Loop through all sprites
		{
			if (sprite[nLoopCount].nState == SPRITE_ALIVE)	// If the sprite is alive
			{
				Erase_Sprite(&sprite[nLoopCount]);			// Erase sprite at old position
			}
		}
		nOneStillAlive = 0;									// Clear alive count

		for (nLoopCount = 0; nLoopCount < NUM_INTRO_SPRITES; nLoopCount++)	// Loop through all sprites
		{
			if (sprite[nLoopCount].nState == SPRITE_ALIVE)	// If the current sprite is alive
			{
				nOneStillAlive = 1;							// Set one is still alive

				switch (sprite[nLoopCount].nDirection)		// Switch on it's direction
				{
					case N:									// ** NORTH **
						if (sprite[nLoopCount].nY > 0)		// If we're not all the way to the top
						{
							sprite[nLoopCount].nY--;		// Move up
						}
						else if (sprite[nLoopCount].nY + sprite[nLoopCount].nHeight > 0)	// If there's any sprite left
						{
							sprite[nLoopCount].nHeight--;	// Shrink the height of the sprite
						}
						else
						{
							sprite[nLoopCount].nState = SPRITE_DEAD;	// No sprite left, kill it
						}
						break;
						
					case S: 								// ** SOUTH **
						if (sprite[nLoopCount].nY + SPRITE_HEIGHT < SCREEN_HEIGHT)
						{
							sprite[nLoopCount].nY++;
						}
						else if (sprite[nLoopCount].nY++ < SCREEN_HEIGHT)
						{
							sprite[nLoopCount].nHeight--;
						}
						else
						{
							sprite[nLoopCount].nState = SPRITE_DEAD;
						}
						break;

					case E: 								// ** EAST **
						if (sprite[nLoopCount].nX + SPRITE_WIDTH < SCREEN_WIDTH)
						{
							sprite[nLoopCount].nX++;
						}
						else if (sprite[nLoopCount].nX++ < SCREEN_WIDTH)
						{
							sprite[nLoopCount].nWidth--;
						}
						else
						{
							sprite[nLoopCount].nState = SPRITE_DEAD;
						}
						break;

					case W:								// ** WEST **
						if (sprite[nLoopCount].nX > 0)	// If we're not all the way to the left.
						{
							sprite[nLoopCount].nX--;
						}
						else if ((sprite[nLoopCount].nX + sprite[nLoopCount].nWidth) > 0)
						{
							sprite[nLoopCount].nWidth--;
						}
						else
						{
							sprite[nLoopCount].nState = SPRITE_DEAD;
						}
						break;

					case NE:								// ** NORTH EAST **
						if (sprite[nLoopCount].nY > 0)
						{
							sprite[nLoopCount].nY--;
						}
						else if (sprite[nLoopCount].nY + sprite[nLoopCount].nHeight > 0)
						{
							sprite[nLoopCount].nHeight--;
						}
						else
						{
							sprite[nLoopCount].nState = SPRITE_DEAD;
						}
	
						if (sprite[nLoopCount].nX + SPRITE_WIDTH < SCREEN_WIDTH)
						{
							sprite[nLoopCount].nX++;
						}
						else if (sprite[nLoopCount].nX++ < SCREEN_WIDTH)
						{
							sprite[nLoopCount].nWidth--;
						}
						else
						{
							sprite[nLoopCount].nState = SPRITE_DEAD;
						}
						break;
	
					case SE:								// ** SOUTH EAST **
						if (sprite[nLoopCount].nY + SPRITE_HEIGHT < SCREEN_HEIGHT)
						{
							sprite[nLoopCount].nY++;
						}
						else if (sprite[nLoopCount].nY++ < SCREEN_HEIGHT)
						{
							sprite[nLoopCount].nHeight--;
						}
						else
						{
							sprite[nLoopCount].nState = SPRITE_DEAD;
						}
	
						if (sprite[nLoopCount].nX + SPRITE_WIDTH < SCREEN_WIDTH)
						{
							sprite[nLoopCount].nX++;
						}
						else if (sprite[nLoopCount].nX++ < SCREEN_WIDTH)
						{
							sprite[nLoopCount].nWidth--;
						}
						else
						{
							sprite[nLoopCount].nState = SPRITE_DEAD;
						}
						break;
						
					case SW: 								// ** SOUTH WEST **
						if (sprite[nLoopCount].nY + SPRITE_HEIGHT < SCREEN_HEIGHT)
						{
							sprite[nLoopCount].nY++;
						}
						else if (sprite[nLoopCount].nY++ < SCREEN_HEIGHT)
						{
							sprite[nLoopCount].nHeight--;
						}
						else
						{
							sprite[nLoopCount].nState = SPRITE_DEAD;
						}
	
						if (sprite[nLoopCount].nX > 0)
						{
							sprite[nLoopCount].nX--;
						}
						else if (sprite[nLoopCount].nX + sprite[nLoopCount].nWidth > 0)
						{
							sprite[nLoopCount].nWidth--;
						}
						else
						{
							sprite[nLoopCount].nState = SPRITE_DEAD;
						}
						break;
	
					case NW:								// ** NORTH WEST **
						if (sprite[nLoopCount].nY > 0)
						{
							sprite[nLoopCount].nY--;
						}
						else if (sprite[nLoopCount].nY + sprite[nLoopCount].nHeight > 0)
						{
							sprite[nLoopCount].nHeight--;
						}
						else
						{
							sprite[nLoopCount].nState = SPRITE_DEAD;
						}
	
						if (sprite[nLoopCount].nX > 0)
						{
							sprite[nLoopCount].nX--;
						}
						else if (sprite[nLoopCount].nX + sprite[nLoopCount].nWidth > 0)
						{
							sprite[nLoopCount].nWidth--;
						}
						else
						{
							sprite[nLoopCount].nState = SPRITE_DEAD;
						}
						break;
	
				}// End of Switch
			}// End if alive
		}// End of for loop

// To shy to shy, hush hush I do I...
// To shy to shy, hush hush I do I.

		for (nLoopCount = 0; nLoopCount < NUM_INTRO_SPRITES; nLoopCount++)	// Loop through all the sprites
		{
			Behind_Sprite(&sprite[nLoopCount]);								// Save the background at new postition
		}
		for (nLoopCount = 0; nLoopCount < NUM_INTRO_SPRITES; nLoopCount++)	// Loop through all sprites
		{
			Draw_Sprite(&sprite[nLoopCount]);								// Draw sprite at new position
			sprite[nLoopCount].nX_Old = sprite[nLoopCount].nX;				// Update old X position indicator
			sprite[nLoopCount].nY_Old = sprite[nLoopCount].nY;				// Update old Y position indicator
		}

		Wait_For_Vsync();		// Wait for verticel sync so we don't get a flicker
		Blast_Buffer();			// Copy double buffer

		if (!nOneStillAlive)	// If there are no more sprites left on the screen
		{
			nDone = 1;			// We're out of here
			nAbort = 0;
		}
	}// End of While

// CLEAN UP
	if (nAbort)					// If user aborted
	{
		for (nLoopCount = 0; nLoopCount < NUM_INTRO_SPRITES; nLoopCount++)	// Loop through all the sprites
		{
			if (sprite[nLoopCount].nState == SPRITE_ALIVE)				// If it's still alive
			{
				Erase_Sprite(&sprite[nLoopCount]);						// Erase sprite at old position
			}
		}
	}

	for (nLoopCount = 0; nLoopCount < NUM_INTRO_SPRITES; nLoopCount++)	// Loop through all the sprites
	{
		Free_Sprite_Mem(&sprite[nLoopCount]);							// Free memory allocated for ship sprite
	}

// Make sure ESC not pressed, if so wait until it's released.
	while (nDone)					// While nDone is 1
	{
		Get_Input();				// Call function to get keys pressed
	    if (nRawKey == 1)			// End if user is exiting
	       {
		       nDone = 1;			// Set nDone to 1
	       }
		else
		{
			nDone = 0;						// Set nDone to 1
		}
	}
}


/********************************************************************
	
	Function:
		VOID AbEnd(VOID)
	
	Description:
		Abnormal ending.  Function called when error during game setup.
					
	Returns:
		None
		
	Comments:
		What happens if the memory was never allocated?
  
*********************************************************************/
void AbEnd(void)
{

	int nLoopCount;									// Loop counting

	Change_Timer(TIMER_18HZ);						// Re-program the timer chip;
	_dos_setvect(TIME_KEEPER_INT, Old_Timer_Isr);	// Restore old Timer ISR
	_dos_setvect(KEYBOARD_INT, Old_Key_Isr);		// Restore old keyboard ISR

	_ffree(Double_Buff);							// Free memory allocated for double buffer
	Free_Sprite_Mem(&sprite[SHIP]);					// Free memory allocated for ship sprite
	Free_Sprite_Mem(&sprite[LIVES]);				// Free memory allocated for Lives sprite
	Free_Sprite_Mem(&sprite[PLATFORM]);				// Free memory allocated for platform sprite

	for (nLoopCount = 0; nLoopCount < NUM_MESSAGE; nLoopCount++)
	{
		FreeMessageMem(&Message[nLoopCount]);		// Free memory allocated for Message
	}

	Set_Video_Mode(TEXT_MODE);						// Set text mode

	printf("An error occurred, unable to continue.\n");
	printf("%s", szErrorString);
	exit(0);
}


/********************************************************************
	
	Function:
		VOID Load_Fonts(VOID)
	
	Description:
		Loads font images from PCX file.
					
	Returns:
		None
		
	Comments:
		PCX file MUST be open.
  
*********************************************************************/
/*
void Load_Fonts(void)
{

	int nLoopCount;
	int nRow = 0;
	int nCol = 0;


	Pcx_Picture Fonts_PCX;			// Define variable 'Sprites_PCX' of type Pcx_Picture. Entire PCX picture

	PCX_Init(&Fonts_PCX);
	PCX_Load(FONT_FILE, &Fonts_PCX,0);		// Load the Sprites_PCX PCX file


	for (nLoopCount = 0; nLoopCount < NUM_FONT; nLoopCount++)
	{
		Font_Init(&font[nLoopCount],0,0, FONT_WIDTH, FONT_HEIGHT);	// Initialize fonts
	}
	

	for (nLoopCount = 0; nLoopCount < NUM_FONT; nLoopCount++)
	{
		PCX_Grap_Font(&Fonts_PCX,&font[nLoopCount],nRow,nCol);
	
		if (nCol++ > 25)
		{
			nCol = 0;
			nRow ++;
		}
	}

	Free_PCX_Mem(&Fonts_PCX);								// Free the memory used for Fonts_PCX PCX file


}// End of Init_Font function
*/















                 