/*
  UNIX & MSDOS NON-DISPLAY, AND CHESSTOOL interface for Chess

  Revision: 1-1-90

  Copyright (C) 1986, 1987, 1988, 1989, 1990 Free Software Foundation, Inc.
  Copyright (c) 1988, 1989, 1990  John Stanback

  This file is part of CHESS.

  CHESS is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY.  No author or distributor accepts responsibility to anyone for
  the consequences of using it or for whether it serves any particular
  purpose or works at all, unless he says so in writing.  Refer to the CHESS
  General Public License for full details.

  Everyone is granted permission to copy, modify and redistribute CHESS, but
  only under the conditions described in the CHESS General Public License. A
  copy of this license is supposed to have been given to you along with
  CHESS so you can know your rights and responsibilities.  It should be in a
  file named COPYING.  Among other things, the copyright notice and this
  notice must be preserved on all copies.
*/


#include <stdio.h>
#include <ctype.h>
#ifdef MSDOS
#include <dos.h>
#include <stdlib.h>
#include <time.h>
#else
#include <sys/param.h>
#include <sys/types.h>
#include <sys/file.h>
#include <signal.h>
#include <sys/time.h>
void TerminateSearch (), Die ();
#endif /* MSDOS */

#include "gnuchess.h"

#ifndef BOOK
#define BOOK "/usr/games/lib/gnuchess.boo"
#endif
#define pxx " PNBRQK"
#define qxx " pnbrqk"
#define rxx "12345678"
#define cxx "abcdefgh"
#define printz printf
#define scanz scanf

static char mvstr[5][6];
static long evrate;
static int mycnt1, mycnt2;
static int ahead;

void
Initialize ()
{
  mycnt1 = mycnt2 = 0;
#ifndef MSDOS
#endif
#ifdef CHESSTOOL
  setlinebuf (stdout);
  /* setvbuf(stdout,NULL,_IOLBF,BUFSIZ); */
  printf ("Chess\n");
  if (Level == 0 && !TCflag)
    Level = 15;
#endif /* CHESSTOOL */
}

void
ExitChess ()
{
  ListGame ();
  exit (0);
}

#ifndef MSDOS
void
Die (sig)
     int sig;
{
  char s[80];
  printz ("Abort? ");
  scanz ("%s", s);
  if (strcmp (s, "yes") == 0)
    ExitChess ();
}

void
TerminateSearch (sig)
     int sig;
{
  timeout = true;
  bothsides = false;
}

#endif /* MSDOS */

void
algbr (f, t, flag)
     short int f;
     short int t;
     short int flag;
                      

/*
   Generate move strings in different formats, a hook has been provided for
   underpromotion
*/

{
  int m3p;

  if (f != t)
    {
      /* algebraic notation */
      mvstr[0][0] = cxx[column (f)];
      mvstr[0][1] = rxx[row (f)];
      mvstr[0][2] = cxx[column (t)];
      mvstr[0][3] = rxx[row (t)];
      mvstr[0][4] = '\0';
      mvstr[3][0] = '\0';
      if ((mvstr[1][0] = qxx[board[f]]) == 'p')
	{
	  if (mvstr[0][0] == mvstr[0][2])	/* pawn did not eat */
	    {
	      mvstr[2][0] = mvstr[1][0] = mvstr[0][2];	/* to column */
	      mvstr[2][1] = mvstr[1][1] = mvstr[0][3];	/* to row */
	      m3p = 2;
	    }
	  else
	    /* pawn ate */
	    {
	      mvstr[2][0] = mvstr[1][0] = mvstr[0][0];	/* from column */
	      mvstr[2][1] = mvstr[1][1] = mvstr[0][2];	/* to column */
	      mvstr[2][2] = mvstr[0][3];
	      m3p = 3;		/* to row */
	    }
	  mvstr[2][m3p] = mvstr[1][2] = '\0';
	  if (flag & promote)
	    {
	      mvstr[0][4] = mvstr[1][2] = mvstr[2][m3p] = qxx[flag & pmask];
	      mvstr[1][3] = mvstr[2][m3p + 1] = mvstr[0][5] = '\0';
	    }
	}
      else
	/* not a pawn */
	{
	  mvstr[2][0] = mvstr[1][0];
	  mvstr[2][2] = mvstr[1][1] = mvstr[0][2];	/* to column */
	  mvstr[2][3] = mvstr[1][2] = mvstr[0][3];	/* to row */
	  mvstr[2][4] = mvstr[1][3] = '\0';
	  mvstr[2][1] = mvstr[0][1];
	  strcpy (mvstr[3], mvstr[2]);
	  mvstr[3][1] = mvstr[0][0];
	  if (flag & cstlmask)
	    {
	      if (t > f)
		{
		  strcpy (mvstr[1], "o-o");
		  strcpy (mvstr[2], "O-O");
		}
	      else
		{
		  strcpy (mvstr[1], "o-o-o");
		  strcpy (mvstr[2], "O-O-O");
		}
	    }
	}
    }
  else
    mvstr[0][0] = mvstr[1][0] = mvstr[2][0] = mvstr[3][0] = '\0';
}

void
seteasy ()
{
  easy = !easy;
}


int
VerifyMove (s, iop, mv)
     char *s;
     short int iop;
     short unsigned int *mv;
              
               
                        

/*
   Compare the string 's' to the list of legal moves available for the
   opponent. If a match is found, make the move on the board.
*/

{
  static short pnt, tempb, tempc, tempsf, tempst, cnt;
  static struct leaf xnode;
  struct leaf *node;

  *mv = 0;
  if (iop == 2)
    {
      UnmakeMove (opponent, &xnode, &tempb, &tempc, &tempsf, &tempst);
      return (false);
    }
  cnt = 0;
  MoveList (opponent, 2);
  pnt = TrPnt[2];
  while (pnt < TrPnt[3])
    {
      node = &Tree[pnt++];
      algbr (node->f, node->t, (short) node->flags);
      if (strcmp (s, mvstr[0]) == 0 || strcmp (s, mvstr[1]) == 0 ||
	  strcmp (s, mvstr[2]) == 0 || strcmp (s, mvstr[3]) == 0)
	{
	  cnt++;
	  xnode = *node;
	}
    }
  if (cnt == 1)
    {
      MakeMove (opponent, &xnode, &tempb, &tempc, &tempsf, &tempst);
      if (SqAtakd (PieceList[opponent][0], computer))
	{
	  UnmakeMove (opponent, &xnode, &tempb, &tempc, &tempsf, &tempst);
	  printz ("Illegal move\n");
	  return (false);
	}
      else
	{
	  if (iop == 1)
	    return (true);
	  if (xnode.flags & epmask)
	    UpdateDisplay (0, 0, 1, 0);
	  else
	    UpdateDisplay (xnode.f, xnode.t, 0, xnode.flags & cstlmask);
	  if (xnode.flags & cstlmask)
	    Game50 = GameCnt;
	  else if (board[xnode.t] == pawn || (xnode.flags & capture))
	    Game50 = GameCnt;
	  GameList[GameCnt].depth = GameList[GameCnt].score = 0;
	  GameList[GameCnt].nodes = 0;
	  ElapsedTime (1);
	  GameList[GameCnt].time = (short) et;
	  TimeControl.clock[opponent] -= et;
	  --TimeControl.moves[opponent];
	  *mv = (xnode.f << 8) + xnode.t;
	  algbr (xnode.f, xnode.t, false);
	  return (true);
	}
    }
#ifdef CHESSTOOL
  printz ("Illegal move\n");
#else
  if (cnt > 1)
    ShowMessage ("Ambiguous Move!");
#endif
  return (false);
}

void
help ()
{
  ClrScreen ();
  printz ("CHESS command summary\n");
  printz ("g1f3      move from g1 to f3\n");
  printz ("nf3       move knight to f3\n");
  printz ("o-o       castle king side\n");
  printz ("o-o-o     castle queen side\n");
  printz ("set       edit board\n");
  printz ("switch    sides with computer\n");
  printz ("white     computer plays white\n");
  printz ("black     computer plays black\n");
  printz ("reverse   board display\n");
  printz ("both      computer match\n");
  printz ("random    randomize play\n");
  printz ("undo      undo last move\n");
  printz ("time      change level\n");
  printz ("depth     set search depth\n");
  printz ("post      principle variation\n");
  printz ("hint      suggest a move\n");
  printz ("bd        redraw board\n");
  printz ("clock     set time control\n");
  printz ("force     enter game moves\n");
  printz ("list      game to chess.lst\n");
  printz ("save      game to file\n");
  printz ("get       game from file\n");
  printz ("new       start new game\n");
  printz ("quit      exit CHESS\n");
  printz ("Computer: ");
  if (computer == white)
    printz ("WHITE\n");
  else
    printz ("BLACK\n");
  printz ("Opponent: ");
  if (opponent == white)
    printz ("WHITE\n");
  else
    printz ("BLACK\n");
  printz ("Response time: %ld sec.\n", Level);
  printz ("Easy mode: ");
  if (easy)
    printz ("ON\n");
  else
    printz ("OFF\n");
  printz ("Depth: %d\n", MaxSearchDepth);
  printz ("Random: ");
  if (dither)
    printz ("ON\n");
  else
    printz ("OFF\n");
  printz ("Transposition table: ");
  if (hashflag)
    printz ("ON\n");
  else
    printz ("OFF\n");
  UpdateDisplay (0, 0, 1, 0);
  signal (SIGINT, TerminateSearch);
  signal (SIGQUIT, TerminateSearch);
}

void
EditBoard ()
/*
   Set up a board position. Pieces are entered by typing the piece followed
   by the location. For example, Nf3 will place a knight on square f3.
*/

{
  short a, r, c, sq;
  char s[80];

  ClrScreen ();
  UpdateDisplay (0, 0, 1, 0);
  printz (".   exit to main\n");
  printz ("#   clear board\n");
  printz ("c   change sides\n");
  printz ("enter piece & location: \n");

  a = white;
  do
    {
      scanz ("%s", s);
      if (s[0] == '#')
	{
	  for (sq = 0; sq < 64; sq++)
	    {
	      board[sq] = no_piece;
	      color[sq] = neutral;
	    }
	  UpdateDisplay (0, 0, 1, 0);
	}
      if (s[0] == 'c' || s[0] == 'C')
	a = otherside[a];
      c = s[1] - 'a';
      r = s[2] - '1';
      if ((c >= 0) && (c < 8) && (r >= 0) && (r < 8))
	{
	  sq = locn (r, c);
	  color[sq] = a;
	  if (s[0] == 'p')
	    board[sq] = pawn;
	  else if (s[0] == 'n')
	    board[sq] = knight;
	  else if (s[0] == 'b')
	    board[sq] = bishop;
	  else if (s[0] == 'r')
	    board[sq] = rook;
	  else if (s[0] == 'q')
	    board[sq] = queen;
	  else if (s[0] == 'k')
	    board[sq] = king;
	  else
	    {
	      board[sq] = no_piece;
	      color[sq] = neutral;
	    }
	}
  } while (s[0] != '.');
  for (sq = 0; sq < 64; sq++)
    Mvboard[sq] = (board[sq] != Stboard[sq]) ? 10 : 0;
  GameCnt = -1;
  Game50 = 0;
  Sdepth = 0;
  InitializeStats ();
  ClrScreen ();
  UpdateDisplay (0, 0, 1, 0);
}

void
SetupBoard ()

/*
   Compatibility with Unix chess and the nchesstool.
   Set up a board position. Eight lines of eight characters are used
   to setup the board. a8-h8 is the first line.
   Black pieces are  represented  by  uppercase characters.
*/

{
  short r, c, sq;
  char ch;
  char s[80];

  for (sq = 0; sq < 64; sq++)
    {
      board[sq] = no_piece;
      color[sq] = neutral;
    }

  gets (s);			/* skip "setup" command */
  for (r = 7; r >= 0; r--)
    {
      gets (s);
      for (c = 0; c <= 7; c++)
	{
	  ch = s[c];
	  sq = locn (r, c);
	  if (ch >= 'A' && ch <= 'Z')
	    {
	      color[sq] = black;
	      ch = ch - 'A' + 'a';
	    }
	  else
	    color[sq] = white;
	  if (ch == 'p')
	    board[sq] = pawn;
	  else if (ch == 'n')
	    board[sq] = knight;
	  else if (ch == 'b')
	    board[sq] = bishop;
	  else if (ch == 'r')
	    board[sq] = rook;
	  else if (ch == 'q')
	    board[sq] = queen;
	  else if (ch == 'k')
	    board[sq] = king;
	  else
	    {
	      board[sq] = no_piece;
	      color[sq] = neutral;
	    }
	}
    }
  for (sq = 0; sq < 64; sq++)
    Mvboard[sq] = (board[sq] != Stboard[sq]) ? 10 : 0;
  GameCnt = -1;
  Game50 = 0;
  Sdepth = 0;
  InitializeStats ();
  ClrScreen ();
  UpdateDisplay (0, 0, 1, 0);
  printz ("Setup successful\n");
}

void
ShowDepth (ch)
     char ch;
{
}

void
ShowResults (score, bstline, ch)
     short int score;
     short unsigned int *bstline;
     char ch;
{
  register int i;
  if (post)
    {
      fprintf (stderr, "%2d%c %6d %4ld %8ld  ", Sdepth, ch, score, et, NodeCnt);
      for (i = 1; bstline[i] > 0; i++)
	{
	  algbr ((short) (bstline[i] >> 8), (short) (bstline[i] & 0xFF), false);
	  if (i == 9 || i == 17)
	    fprintf (stderr, "\n                          ");
	  fprintf (stderr, "%5s ", mvstr[0]);
	}
      fprintf (stderr, "\n");
      fflush (stderr);
    }
}

void
SearchStartStuff (side)
     short int side;
{
  signal (SIGINT, TerminateSearch);
  signal (SIGQUIT, TerminateSearch);
  if (post)
    {
      fprintf (stderr, "\nMove# %d    Target= %ld    Clock: %ld\n",
	       TCmoves - TimeControl.moves[side] + 1,
	       ResponseTime, TimeControl.clock[side]);
      fflush (stderr);
    }
}

void
OutputMove ()
{
  printz ("%d. ... %s\n", ++mycnt1, mvstr[0]);
  if (root->flags & draw)
    {
      printz ("Draw\n");
      ListGame ();
      exit (0);
    }
  if (root->score == -9999)
    {
      if (opponent == white)
	printz ("White\n");
      else
	printz ("Black\n");
      ListGame ();
      exit (0);
    }
  if (root->score == 9998)
    {
      if (computer == white)
	printz ("White\n");
      else
	printz ("Black\n");
      ListGame ();
      exit (0);
    }
  if (post)
    {
      fprintf (stderr, "Nodes= %ld  Eval= %ld  Rate= %ld  ",
	       NodeCnt, EvalNodes, evrate);
      fprintf (stderr, "Hash/Coll= %ld/%ld\n",
	       HashCnt, HashCol);
    }

  if (root->flags & epmask)
    UpdateDisplay (0, 0, 1, 0);
  else
    UpdateDisplay (root->f, root->t, 0, root->flags & cstlmask);
  fprintf (stderr, "My move is: %s\n\n", mvstr[0]);
  if (beep)
    printz ("%c", 7);

  if (root->flags & draw)
    fprintf (stderr, "Drawn game!\n");
  else if (root->score == -9999)
    fprintf (stderr, "opponent mates!\n");
  else if (root->score == 9998)
    fprintf (stderr, "computer mates!\n");
  else if (root->score < -9000)
    fprintf (stderr, "opponent will soon mate!\n");
  else if (root->score > 9000)
    fprintf (stderr, "computer will soon mate!\n");
  fflush (stderr);
}

void
ElapsedTime (iop)
     short int iop;
               

/*
   Determine the time that has passed since the search was started. If the
   elapsed time exceeds the target (ResponseTime+ExtraTime) then set timeout
   to true which will terminate the search.
*/

{
#ifdef MSDOS  
  if (ahead)
    {
      long nchar;
      ioctl(0, FIONREAD, &nchar);
      if (nchar) {
	timeout= true;
	bothsides = false;
      }
    }
#endif /* MSDOS */
  et = time ((long *) 0) - time0;
  if (et < 0)
    et = 0;
  ETnodes += 50;
  if (et > et0 || iop == 1)
    {
      if (et > ResponseTime + ExtraTime && Sdepth > 1)
	timeout = true;
      et0 = et;
      if (iop == 1)
	{
	  time0 = time ((long *) 0);
	  et0 = 0;
	}
      if (et > 0)
	/* evrate used to be Nodes / cputime I dont` know why */
	evrate = NodeCnt / (et + ft);
      else
	evrate = 0;
#ifdef MSDOS      
      if (kbhit () && Sdepth > 1)
	{
	  timeout = true;
	  bothsides = false;
	}
#endif /* MSDOS */
      ETnodes = NodeCnt + 50;
    }
}

void
SetTimeControl ()
{
  if (TCflag)
    {
      TimeControl.moves[white] = TimeControl.moves[black] = TCmoves;
      TimeControl.clock[white] = TimeControl.clock[black] = 60 * (long) TCminutes;
    }
  else
    {
      TimeControl.moves[white] = TimeControl.moves[black] = 0;
      TimeControl.clock[white] = TimeControl.clock[black] = 0;
      Level = 60 * (long) TCminutes;
    }
  et = 0;
  ElapsedTime (1);
}

void
ClrScreen ()
{
#ifndef CHESSTOOL
  printz ("\n");
#endif
}

void
UpdateDisplay (f, t, flag, iscastle)
     short int f;
     short int t;
     short int flag;
     short int iscastle;
{
#ifndef CHESSTOOL
  short r, c, l;
  if (flag)
    {
      printz ("\n");
      for (r = 7; r >= 0; r--)
	{
	  for (c = 0; c <= 7; c++)
	    {
	      if (reverse)
		l = locn (7 - r, 7 - c);
	      else
		l = locn (r, c);
	      if (color[l] == neutral)
		printz (" -");
	      else if (color[l] == white)
		printz (" %c", qxx[board[l]]);
	      else
		printz (" %c", pxx[board[l]]);
	    }
	  printz ("\n");
	}
      printz ("\n");
    }
#endif /* CHESSTOOL */
}

void
GetOpenings ()
/*
   Read in the Opening Book file and parse the algebraic notation for a move
   into an unsigned integer format indicating the from and to square. Create
   a linked list of opening lines of play, with entry->next pointing to the
   next line and entry->move pointing to a chunk of memory containing the
   moves. More Opening lines of up to 256 half moves may be added to
   gnuchess.book.
*/

{
  FILE *fd;
  int c, i, j, side;
  /* char buffr[2048]; */
  struct BookEntry *entry;
  unsigned short mv, *mp, tmp[100];

  if (((fd = fopen (BOOK, "r")) != NULL) ||
      ((fd = fopen ("gnuchess.boo", "r")) != NULL))
    {
      /* setvbuf(fd,buffr,_IOFBF,2048); */
      Book = NULL;
      i = 0;
      side = white;
      while ((c = parse (fd, &mv, side)) >= 0)
	if (c == 1)
	  {
	    tmp[++i] = mv;
	    side = otherside[side];
	  }
	else if (c == 0 && i > 0)
	  {
	    entry = (struct BookEntry *) malloc (sizeof (struct BookEntry));
	    mp = (unsigned short *) malloc ((i + 1) * sizeof (unsigned short));
	    entry->mv = mp;
	    entry->next = Book;
	    Book = entry;
	    for (j = 1; j <= i; j++)
	      *(mp++) = tmp[j];
	    *mp = 0;
	    i = 0;
	    side = white;
	  }
      fclose (fd);
    }
}


int
parse (fd, mv, side)
     FILE *fd;
     short unsigned int *mv;
     short int side;
{
  int c, i, r1, r2, c1, c2;
  char s[100];
  while ((c = getc (fd)) == ' ') ;
  i = 0;
  s[0] = c;
  while (c != ' ' && c != '\n' && c != EOF)
    s[++i] = c = getc (fd);
  s[++i] = '\0';
  if (c == EOF)
    return (-1);
  if (s[0] == '!' || s[0] == ';' || i < 3)
    {
      while (c != '\n' && c != EOF)
	c = getc (fd);
      return (0);
    }
  if (s[4] == 'o')
    if (side == black)
      *mv = 0x3C3A;
    else
      *mv = 0x0402;
  else if (s[0] == 'o')
    if (side == black)
      *mv = 0x3C3E;
    else
      *mv = 0x0406;
  else
    {
      c1 = s[0] - 'a';
      r1 = s[1] - '1';
      c2 = s[2] - 'a';
      r2 = s[3] - '1';
      *mv = (locn (r1, c1) << 8) + locn (r2, c2);
    }
  return (1);
}

void
GetGame ()
{
  FILE *fd;
  char fname[40];
  int c;
  short sq;
  unsigned short m;

  printz ("Enter file name: ");
  scanz ("%s", fname);
  if (fname[0] == '\0')
    strcpy (fname, "chess.000");
  if ((fd = fopen (fname, "r")) != NULL)
    {
      fscanf (fd, "%hd%hd%hd", &computer, &opponent, &Game50);
      fscanf (fd, "%hd%hd", &castld[white], &castld[black]);
      fscanf (fd, "%hd%hd", &TCflag, &OperatorTime);
      fscanf (fd, "%ld%ld%hd%hd",
	      &TimeControl.clock[white], &TimeControl.clock[black],
	      &TimeControl.moves[white], &TimeControl.moves[black]);
      for (sq = 0; sq < 64; sq++)
	{
	  fscanf (fd, "%hd%hd", &m, &Mvboard[sq]);
	  board[sq] = (m >> 8);
	  color[sq] = (m & 0xFF);
	  if (color[sq] == 0)
	    color[sq] = neutral;
	  else
	    --color[sq];
	}
      GameCnt = -1;
      c = '?';
      while (c != EOF)
	{
	  ++GameCnt;
	  c = fscanf (fd, "%hd%hd%hd%ld%hd%hd%hd", &GameList[GameCnt].gmove,
		      &GameList[GameCnt].score, &GameList[GameCnt].depth,
		      &GameList[GameCnt].nodes, &GameList[GameCnt].time,
		      &GameList[GameCnt].piece, &GameList[GameCnt].color);
	  if (GameList[GameCnt].color == 0)
	    GameList[GameCnt].color = neutral;
	  else
	    --GameList[GameCnt].color;
	}
      GameCnt--;
      if (TimeControl.clock[white] > 0)
	TCflag = true;
      computer--;
      opponent--;
    }
  fclose (fd);
  InitializeStats ();
  UpdateDisplay (0, 0, 1, 0);
  Sdepth = 0;
}

void
SaveGame ()
{
  FILE *fd;
  char fname[40];
  short sq, i, c;

  printz ("Enter file name: ");
  scanz ("%s", fname);

  if (fname[0] == '\0')
    strcpy (fname, "chess.000");
  if ((fd = fopen (fname, "w")) != NULL)
    {
      fprintf (fd, "%d %d %d\n", computer + 1, opponent + 1, Game50);
      fprintf (fd, "%d %d\n", castld[white], castld[black]);
      fprintf (fd, "%d %d\n", TCflag, OperatorTime);
      fprintf (fd, "%ld %ld %d %d\n",
	       TimeControl.clock[white], TimeControl.clock[black],
	       TimeControl.moves[white], TimeControl.moves[black]);
      for (sq = 0; sq < 64; sq++)
	{
	  if (color[sq] == neutral)
	    c = 0;
	  else
	    c = color[sq] + 1;
	  fprintf (fd, "%d %d\n", 256 * board[sq] + c, Mvboard[sq]);
	}
      for (i = 0; i <= GameCnt; i++)
	{
	  if (GameList[i].color == neutral)
	    c = 0;
	  else
	    c = GameList[i].color + 1;
	  fprintf (fd, "%d %d %d %ld %d %d %d\n",
		   GameList[i].gmove, GameList[i].score, GameList[i].depth,
		   GameList[i].nodes, GameList[i].time,
		   GameList[i].piece, c);
	}
      fclose (fd);
      printz ("Game saved on file: %s", fname);
    }
  else
    printz ("Could not open file: %s", fname);
}

void
ListGame ()
{
  FILE *fd;
  short i, f, t;
  fd = fopen ("chess.lst", "w");
  fprintf (fd, "\n");
  fprintf (fd, "       score  depth   nodes  time         ");
  fprintf (fd, "       score  depth   nodes  time\n");
  for (i = 0; i <= GameCnt; i++)
    {
      f = GameList[i].gmove >> 8;
      t = (GameList[i].gmove & 0xFF);
      algbr (f, t, false);
      if ((i % 2) == 0)
	fprintf (fd, "\n");
      else
	fprintf (fd, "         ");
      fprintf (fd, "%5s  %5d     %2d %7ld %5d", mvstr[0],
	       GameList[i].score, GameList[i].depth,
	       GameList[i].nodes, GameList[i].time);
    }
  fprintf (fd, "\n\n");
  fclose (fd);
}

void
Undo ()
/*
   Undo the most recent half-move.
*/

{
  short f, t;
  f = GameList[GameCnt].gmove >> 8;
  t = GameList[GameCnt].gmove & 0xFF;
  if (board[t] == king && distance (t, f) > 1)
    (void) castle (GameList[GameCnt].color, f, t, 2);
  else
    {
      /* Check for promotion: */
      if ((row (f) == 6 && row (t) == 7) || (row (f) == 1 && row (t) == 0))
	{
	  int g, from = f;
	  for (g = GameCnt - 1; g >= 0; g--)
	    if (GameList[g].gmove & 0xFF == from)
	      from = GameList[g].gmove >> 8;
	  if (row (from) == 1 || row (from) == 6)
	    board[t] = pawn;
	}

      board[f] = board[t];
      color[f] = color[t];
      board[t] = GameList[GameCnt].piece;
      color[t] = GameList[GameCnt].color;
      if (color[t] != neutral)
	Mvboard[t]--;
      Mvboard[f]--;
    }
  if (TCflag)
    ++TimeControl.moves[color[f]];
  GameCnt--;
  computer = otherside[computer];
  opponent = otherside[opponent];
  mate = false;
  Sdepth = 0;
  UpdateDisplay (0, 0, 1, 0);
  InitializeStats ();
}

void
ShowMessage (s)
     char *s;
{
#ifndef CHESSTOOL
  printz ("%s\n");
#endif /* CHESSTOOL */
}

void
ShowSidetomove ()
{
}

void
PromptForMove ()
{
#ifndef CHESSTOOL
  printz ("\nYour move is? ");
#endif /* CHESSTOOL */
}


void
ShowCurrentMove (pnt, f, t)
     short int pnt;
     short int f;
     short int t;
{
}

void
ChangeAlphaWindow ()
{
  printz ("window: ");
  scanz ("%hd", &Awindow);
}

void
ChangeBetaWindow ()
{
  printz ("window: ");
  scanz ("%hd", &Bwindow);
}

void
GiveHint ()
{
  algbr ((short) (hint >> 8), (short) (hint & 0xFF), false);
  printz ("try %s\n", mvstr[0]);
}

void
SelectLevel ()
{
  OperatorTime = 30000;
  printz ("Enter #moves #minutes: ");
  scanz ("%hd %hd", &TCmoves, &TCminutes);
  printz ("Operator time= ");
  scanz ("%hd", &OperatorTime);
  TCflag = (TCmoves > 1);
  SetTimeControl ();
}

void
ChangeSearchDepth ()
{
  printz ("depth= ");
  scanz ("%hd", &MaxSearchDepth);
}

void
SetContempt ()
{
  printz ("contempt= ");
  scanz ("%hd", &contempt);
}

void
ChangeXwindow ()
{
  printz ("xwndw= ");
  scanz ("%hd", &xwndw);
}


void
InputCommand ()
/*
   Process the users command. If easy mode is OFF (the computer is thinking
   on opponents time) and the program is out of book, then make the 'hint'
   move on the board and call SelectMove() to find a response. The user
   terminates the search by entering ^C (quit siqnal) before entering a
   command. If the opponent does not make the hint move, then set Sdepth to
   zero.
*/

{
  int i;
  short ok, tmp;
  long cnt, rate, t1, t2;
  unsigned short mv;
  char s[80];

  ok = quit = false;
  player = opponent;
  ft = 0;
  if (hint > 0 && !easy && Book == NULL)
    {
      fflush (stdout);
      time0 = time ((long *) 0);
      algbr ((short) hint >> 8, (short) hint & 0xFF, false);
      strcpy (s, mvstr[0]);
      tmp = epsquare;
      if (post)
	{
	  fprintf (stderr, "Hint: %s\n", s);
	  fflush (stderr);
	}
      if (VerifyMove (s, 1, &mv))
	{
	  ahead = 1;
	  SelectMove (computer, 2);
	  VerifyMove (mvstr[0], 2, &mv);
	  if (Sdepth > 0)
	    Sdepth--;
	}
      ft = time ((long *) 0) - time0;
      epsquare = tmp;
    }
  ahead = 0;
#ifndef MSDOS
#endif
  while (!(ok || quit))
    {
      PromptForMove ();
      i = scanz ("%s", s);
      if (i == EOF || s[0] == 0)
	ExitChess ();
      player = opponent;
      if (strcmp (s, "bd") == 0)
	{
	  ClrScreen ();
	  UpdateDisplay (0, 0, 1, 0);
	}
      else if (strcmp (s, "alg") == 0) /* noop */ ;
      else if (strcmp (s, "quit") == 0)
	quit = true;
      else if (strcmp (s, "post") == 0)
	post = !post;
      else if (strcmp (s, "set") == 0)
	EditBoard ();
      else if (strcmp (s, "setup") == 0)
	SetupBoard ();
      else if (strcmp (s, "go") == 0)
	ok = true;
      else if (strcmp (s, "help") == 0)
	help ();
      else if (strcmp (s, "force") == 0)
	force = !force;
      else if (strcmp (s, "book") == 0)
	Book = NULL;
      else if (strcmp (s, "new") == 0)
	NewGame ();
      else if (strcmp (s, "list") == 0)
	ListGame ();
      else if (strcmp (s, "level") == 0 || strcmp (s, "clock") == 0)
	SelectLevel ();
      else if (strcmp (s, "hash") == 0)
	hashflag = !hashflag;
      else if (strcmp (s, "beep") == 0)
	beep = !beep;
      else if (strcmp (s, "Awindow") == 0)
	ChangeAlphaWindow ();
      else if (strcmp (s, "Bwindow") == 0)
	ChangeBetaWindow ();
      else if (strcmp (s, "rcptr") == 0)
	rcptr = !rcptr;
      else if (strcmp (s, "hint") == 0)
	GiveHint ();
      else if (strcmp (s, "both") == 0)
	{
	  bothsides = !bothsides;
	  Sdepth = 0;
	  ElapsedTime (1);
	  SelectMove (opponent, 1);
	  ok = true;
	}
      else if (strcmp (s, "reverse") == 0)
	{
	  reverse = !reverse;
	  ClrScreen ();
	  UpdateDisplay (0, 0, 1, 0);
	}
      else if (strcmp (s, "switch") == 0)
	{
	  computer = otherside[computer];
	  opponent = otherside[opponent];
	  force = false;
	  Sdepth = 0;
	  ok = true;
	}
      else if (strcmp (s, "white") == 0 || strcmp (s, "first") == 0)
	{
	  computer = white;
	  opponent = black;
	  ok = true;
	  force = false;
	  Sdepth = 0;
	}
      else if (strcmp (s, "black") == 0)
	{
	  computer = black;
	  opponent = white;
	  ok = true;
	  force = false;
	  Sdepth = 0;
	}
      else if (strcmp (s, "undo") == 0 && GameCnt >= 0)
	Undo ();
      else if (strcmp (s, "remove") == 0 && GameCnt >= 1)
	{
	  Undo ();
	  Undo ();
	}
      else if (strcmp (s, "get") == 0)
	GetGame ();
      else if (strcmp (s, "save") == 0)
	SaveGame ();
      else if (strcmp (s, "depth") == 0)
	ChangeSearchDepth ();
      else if (strcmp (s, "random") == 0)
	dither = 6;
      else if (strcmp (s, "easy") == 0)
	seteasy ();
      else if (strcmp (s, "contempt") == 0)
	SetContempt ();
      else if (strcmp (s, "xwndw") == 0)
	ChangeXwindow ();
      else if (strcmp (s, "test") == 0)
	{
	  t1 = time (0);
	  cnt = 0;
	  for (i = 0; i < 10000; i++)
	    {
	      MoveList (opponent, 2);
	      cnt += TrPnt[3] - TrPnt[2];
	    }
	  t2 = time (0);
	  rate = cnt / (t2 - t1);
	  printz ("cnt= %ld  rate= %ld\n", cnt, rate);
	}
      else
	{
	  ok = VerifyMove (s, 0, &mv);
	  if (ok && mv != hint)
	    {
	      Sdepth = 0;
	      ft = 0;
	    }
	}
    }

  ElapsedTime (1);
  if (force)
    {
      computer = opponent;
      opponent = otherside[computer];
    }
#ifdef CHESSTOOL
  printf ("%d. %s\n", ++mycnt2, s);
#endif /* CHESSTOOL */
  signal (SIGINT, TerminateSearch);
  signal (SIGQUIT, TerminateSearch);
}
