unit ipx;

interface

uses dos;

type
  netAddr  = array[1..4] of byte;    { The address of a network }
  nodeAddr = array[1..6] of byte;    { The address of a node in a network }

  address  = array[0..1] of word;    { A pointer to the data 0=offset 1=seg }

  netAddress = record
                 Net    : netAddr;   { network address }
                 Node   : nodeAddr;  { node address }
                 Socket : word;      { Big endian socket number}
               end;

  localAddrT = record
                 Net    : netAddr;   { my network address }
                 Node   : nodeAddr;  { my node address }
               end;

  ECBType = record
              link      : address;    { Pointer to next ECB? }
              ESR       : address;    { Event Service Routine 00000000h if none }
              inUse     : byte;       { In use flag }
              complete  : byte;       { Completeing flag }
              socket    : word;       { Big endian socket number }
              IPXwork   : array[1..4] of byte;  { IPX work space }
              Dwork     : array[1..12] of byte; { Driver work space }
              immedAddr : nodeAddr;   { Immediate local node address }
              fragCount : word;       { Fragment count }
              fragData  : address;    { Pointer to data fragment }
              fragSize  : word;       { Size of data fragment }
            end;

  IPXheader = record
                check  : word;                { big endian checksum }
                length : word;                { big endian length in bytes }
                tc     : byte;                { transport control }
                pType  : byte;                { packet type }
                dest   : netAddress;          { destination network address }
                src    : netAddress;          { source network address }
              end;

const
  MYSOCKET  : word = $869C;        { This is the DOOM official socket number }
  BROADCAST : nodeAddr = ($ff,$ff,$ff,$ff,$ff,$ff);  { Address for broadcast }

var
  localAddr    : localAddrT;

function  IPXopenSocket(longevity : byte; var socketNumber : word):byte;
procedure IPXcloseSocket(socketNumber : word);
procedure GetLocalAddress;
procedure IPXsendPacket(var E : ECBtype);
function  IPXlistenForPacket(var E : ECBtype):byte;
procedure ImIdle;
procedure InitSendPacket(var ecb : ecbType; var ipx : ipxHeader; size,sock : word);
procedure InitReceivePacket(var ecb : ecbType; var ipx : ipxHeader; size,sock : word);
procedure InitIPX;

implementation

{ Open a socket
  PARAMS:  longevity = $00 for open till close or terminate
                     = $ff for open till close  use for TSR

           socketNumber = 0 for dynamic allocation
                        = anything else

  RETURNS: completion code $00 = success
                           $fe = socket table full
                           $ff = socket already open }
function IPXopenSocket(longevity : byte; var socketNumber : word):byte;
var
  regs : registers;

begin
  regs.bx:=$0000;
  regs.al:=longevity;
  regs.dx:=swap(socketNumber);

  intr($7A,regs);

  if socketNumber=$0000 then
    socketNumber:=swap(regs.dx);

  IPXopenSocket:=regs.al;
end;

{ Close a socket
  PARMS:  socketNumber = a socket to close }
procedure IPXcloseSocket(socketNumber : word);
var
  regs : registers;

begin
  regs.bx:=$0001;
  regs.dx:=swap(socketNumber);

  intr($7A,regs);
end;

{ Get my address and put it into the local address array! }
procedure GetLocalAddress;
var
  regs : registers;

begin
  regs.bx:=$0009;
  regs.es:=seg(localAddr);
  regs.si:=ofs(localAddr);
  intr($7A,regs);
end;

{ Send an IPX packet
PARAMS:  var E = an initialized Event Control Block }
procedure IPXsendPacket(var E : ECBtype);
var
  regs : registers;

begin
  regs.bx:=$0003;
  regs.es:=seg(E);
  regs.SI:=ofs(E);

  intr($7A,regs);
end;

{ Listen for an IPX packet
PARAMS:  var E = an initialize Event Control Block

RETURNS: 0 for OK, nonzero for an error ????}
function IPXlistenForPacket(var E : ECBtype):byte;
var
  regs : registers;

begin
  regs.bx:=$0004;
  regs.es:=seg(E);
  regs.SI:=ofs(E);

  intr($7A,regs);

  IPXlistenForPacket:=regs.al;
end;

{ Tell the IPX driver that we aren't doing anything at the moment }
procedure ImIdle;
var
  regs : registers;

begin
  regs.bx:=$000A;

  intr($7A,regs);
end;

{ Set up the fields in a send IPX record }
procedure InitSendPacket(var ecb : ecbType; var ipx : ipxHeader; size,sock : word);
begin
  fillChar(ecb,sizeOf(ecb),#0);
  fillChar(ipx,sizeOf(ipx),#0);
  with ecb do
    begin
      socket:=swap(sock);               { Big endian socket number }
      fragCount:=1;                     { Fragment count }
      fragData[0]:=ofs(IPX);            { Pointer to data fragment }
      fragData[1]:=seg(IPX);
      fragSize:=sizeof(IPX)+size;       { Size of data fragment }
      immedAddr:=BROADCAST;             { Needs to be BROADCAST?? }
    end;

  with ipx do
    begin
      check:=$ffff;                     { NO CHECKSUM }
      ptype:=0;                         { Packet exchange packet }
      dest.net:=localAddr.net;          { Send to this network }
      dest.node:=BROADCAST;             { Send to everybody! }
      dest.socket:=swap(sock);          { Send to my socket }
      src.net:=localAddr.net;           { From this net }
      src.node:=localAddr.node;         { From ME }
      src.socket:=swap(sock);           { From my socket }
    end;
end;

{ Set up the fields in a recieve IPX record }
procedure InitReceivePacket(var ecb : ecbType; var ipx : ipxHeader; size,sock : word);
begin
  fillChar(ecb,sizeOf(ecb),#0);
  fillChar(ipx,sizeOf(ipx),#0);
  with ecb do
    begin
      inUse:=$1d;                               { ???? }
      socket:=swap(sock);                       { Big endian socket number }
      fragCount:=1;                             { Fragment count }
      fragData[0]:=ofs(IPX);                    { Pointer to data fragment }
      fragData[1]:=seg(IPX);
      fragSize:=sizeof(IPX)+size;               { Size of data fragment }
    end;

  if IPXlistenForPacket(ecb)<>0 then ;          { Tell IPX to listen }
end;

{ Set up IPX and get the local address }
procedure InitIPX;
var
  i    : integer;
  regs : registers;

begin
  regs.ax:=$7A00;
  intr($2f,regs);

  if regs.al<>255 then
    begin
      writeln('ERROR WHILE INITIALIZING IPX!');
      halt(1);
    end;

  getLocalAddress;
end;


begin
end.
