\STDLIB.XPL	25-Jul-2011
\Standard Library Routines for XPL0

\REVISIONS:
\MAR-01-95, V2.2, Modified: WaitVB, TextIn, Box, and StrCmp; added StrNFill,
\ BCD2Str, Str2BCD and CallInt. Eliminate CpuReg names (AX, BX etc.)--use
\ CallInt instead.
\APR-06-95, V2.2.1, Put CpuReg:= GetReg; in CallInt so it doesn't have to be
\ done elsewhere.
\JUL-01-95, V2.3, Added all standard definitions.
\JAN-29-96, Changed "LfArrow" to "LtArrow".
\19-Feb-2006, Cosmetic changes only.
\29-May-2006, Cosmetic changes only.
\2-Sep-2006, Modify calls to CallInt and CpuReg to work with 32-bit XPL.
\28-Jul-2008, Modify StrNMul (so A and B can be the same item) add StrNDiv
\29-Jan-2010, Replace "rem" with "Rem" for compatibility with XPLI.
\25-Jul-2011, Fixed bug in StrCmp and changed comment in ShowMouse.

\WARNING:
\Don't forget to set CpuReg in your main procedure, i.e: CpuReg:= GetReg;

inc	C:\CXPL\CODESI;		\include code definitions for intrinsic routines

def	IntSize= 2,		\number of bytes in an integer (2 or 4)
	MaxForSize=$7FFE;	\largest limit in a for loop

def	Nul=$00, Bel=$07, BS=$08, Tab=$09, LF=$0A, FF=$0C,	\control chars
	CR=$0D, EOF=$1A, Esc=$1B, Sp=$20, Ctrl=$40;

def	UpArrow=$48, DnArrow=$50, LtArrow=$4B, RtArrow=$4D,	\scan codes
	PageUp=$49, PageDn=$51, Home=$47, End=$4F, Insert=$52, Delete=$53,
	Func1=$3B;

def	Black, Blue, Green, Cyan, Red, Magenta, Brown, White,  \attribute colors
	Gray, LBlue, LGreen, LCyan, LRed, LMagenta, Yellow, BWhite; \EGA palette

def	Pi = 3.14159265358979323846;

int	CpuReg,			\address of CPU register array (from GetReg)
	Get100sec,		\100ths of seconds from GetTime function
	HaveMouse;		\a mouse driver is installed and mouse works

\------------------------------ INTEGER ROUTINES -------------------------------

public func Min(A, B);		\Return the smaller of the arguments
int	A, B;
return if A < B then A else B;



public func Max(A, B);		\Return the larger of the arguments
int	A, B;
return if A > B then A else B;



public func Rol(N, C);		\Rotate N left C bits
int	N, C;
def	M = $000F;
return N<<(C&M) ! N>>(16-(C&M));



public func Ror(N, C);		\Rotate N right C bits
int	N, C;
def	M = $000F;
return N>>(C&M) ! N<<(16-(C&M));



public func SwapBits(N);	\Swap the order of the bits in N
int	N;
int	B, M;
begin
B:= 1;   M:= 0;
while N \#0\ do
	begin
	if N < 0 then M:= M + B;\or bit into M if MSB of N is set
	B:= B + B;		\shift left
	N:= N + N;
	end;
return	M;
end;	\SwapBits



public func IntLen(N);		\Return number of digits in N, a decimal integer
int	N;			\Adds 1 for a minus sign if N is negative
int	I;
for I:= if N>=0 then 1 else 2, 20 do
	[N:= N / 10;   if N = 0 then return I];



public func ISqr(N);		\Square an integer N
int	N;
return N * N;



public func ISqrt(N);		\Square root of a positive integer N
int	N;			\Largest error is < 1 over range 0 thru 32767
int	Guess, I;
begin
Guess:= 20;
for I:= 1, 5 do Guess:= ((N/Guess) + Guess) >> 1;
return Guess;
end;	\ISqrt

\------------------------------- REAL ROUTINES ---------------------------------

public func real Deg(X);	\Convert radians to degrees
real	X;
return 57.2957795130823 * X;



public func real Rad(X);	\Convert degrees to radians
real	X;
return X / 57.2957795130823;



public func real Frac(X);	\Get fractional part of real number
real	X;			\e.g: Frac(12.9) =  0.9
return Mod(X, 1.0);		\     Frac(-3.7) = -0.7



public func real Int(X);	\Get integer part of real number
real	X;			\e.g: Int(12.9) = 12.0
return X - Mod(X, 1.0);		\     Int(-3.7) = -3.0



public func real Sqr(N);	\Square a real
real	X;
return X * X;



public func real Power(X, Y);	\X raised to the Y power
real	X, Y;
return Exp(Y * Ln(X));



public func real Hypot(X, Y);	\Hypotenuse of right triangle with sides X & Y
real	X, Y;			\(also is magnitude of polar coordinate)
return Sqrt(X*X + Y*Y);



public func real ATan(Y);	\Arc tangent
real	Y;
return ATan2(Y, 1.0);



public func real ALog(X);	\Antilog (or 10**X)
real	X;			\10**X = e**(X ln(10)) = e**ln(10**X)
return Exp(X * 2.302585092994045684018);	\Ln(10) = 2.3025...



public func real Fact(X);	\Factorial of a positive real
real	X;
int	I;
begin
if X < 1.1 then return 1.0;	\0! = 1
for I:= 2, Fix(X)-1 do X:= Float(I) * X;
return X;
end;	\Fact



public func real FloatUnsign(N); \Convert a 16-bit, unsigned integer to a real
int	N;
return if N < 0 then Float(N-$8000) + 32768.0 else Float(N);

\----------------------------- CHARACTER ROUTINES ------------------------------

public func ToLower(Ch);	\Convert character to lowercase
int	Ch;
return if Ch>=^A & Ch<=^Z then Ch!$20 else Ch;



public func ToUpper(Ch);	\Convert character to uppercase
int	Ch;
return if Ch>=^a & Ch<=^z then Ch&$DF else Ch;



public func IsUpper(Ch);	\Is character uppercase?
int	Ch;
return Ch>=^A & Ch<=^Z;



public func IsLower(Ch);	\Is character lowercase?
int	Ch;
return Ch>=^a & Ch<=^z;



public func IsDigit(Ch);	\Is character a digit?
int	Ch;
return Ch>=^0 & Ch<=^9;



public func IsHexDigit(Ch);	\Is character a hexadecimal digit?
int	Ch;
case of
  Ch>=^0 & Ch<=^9:  return true;
  Ch>=^A & Ch<=^F:  return true;
  Ch>=^a & Ch<=^f:  return true
other return false;



public func IsAlpha(Ch);	\Is character alphabetic?
int	Ch;
case of
  Ch>=^A & Ch<=^Z:  return true;
  Ch>=^a & Ch<=^z:  return true
other return false;



public func IsAlnum(Ch);	\Is character alphanumeric?
int	Ch;
case of
  Ch>=^A & Ch<=^Z:  return true;
  Ch>=^a & Ch<=^z:  return true;
  Ch>=^0 & Ch<=^9:  return true
other return false;

\------------------------------- INPUT ROUTINES --------------------------------

public func CallInt(Int, AX, BX, CX, DX, BP, DS, ES); \Call software interrupt
int	Int, AX, BX, CX, DX, BP, DS, ES; \(unused arguments need not be passed)
begin
CpuReg:= GetReg;
CpuReg(0):= AX;
CpuReg(1):= BX;
CpuReg(2):= CX;
CpuReg(3):= DX;
CpuReg(6):= BP;
CpuReg(9):= DS;
CpuReg(11):= ES;
SoftInt(Int);
return CpuReg(0) & $FFFF;	\return contents of AX register
end;	\CallInt



public func ShiftKey;		\Returns 'true' if a shift key is down
return (CallInt($16, $0200) & $03) # 0;	\BIOS function $02



public func GetKey;		\Get character from keyboard (wait if necessary)
int	SC, Ch;			\This is a low-level routine with no echo,
begin				\ no Ctrl+C, and no cursor.
SC:= CallInt($16, $0000);	\BIOS function $00
Ch:= SC & $FF;
if Ch = 0 then Ch:= -(SC>>8);	\return non-ASCII chars as negative scan code
return Ch;
end;	\GetKey



public proc WaitKey;		\Wait for a keystroke
begin
OpenI(1);
repeat until ChkKey;
end;	\WaitKey



public func LookKey;		\Returns next keystroke without reading it in
int	SC, Ch;			\ so it can be read in later (e.g: IntIn(0))
begin
repeat until ChkKey;
SC:= CallInt($16, $0100);	\BIOS function $01
Ch:= SC & $FF;
if Ch = 0 then Ch:= -(SC>>8);	\return non-ASCII chars as negative scan code
return Ch;
end;	\LookKey



public func GetCurShape;	\Get cursor shape
begin
CallInt($10, $0300);		\BIOS function $03
return CpuReg(2) & $FFFF;
end;	\GetCurShape



public proc SetCurShape(Shape);	\Set cursor shape
\WARNING: $0607 does not work for monochrome mode 7. Use $0B0C instead.
\ $000D gives a solid block for all modes.
int	Shape;
CallInt($10, $0100, 0, Shape);	\BIOS function $01



public proc ShowCursor(On);	\Turn flashing cursor off and on
int	On;	\flag: True = cursor on; False = cursor off
int	CS;	\own-type variable to save original cursor shape
begin
CS:= [$000D];	\default is solid block
if On then SetCurShape(CS(0))
else	[CS(0):= GetCurShape;
	SetCurShape($2000)];
end;	\ShowCursor

\------------------------------- OUTPUT ROUTINES -------------------------------

public proc SpOut(Dev, N);	\Output N spaces to specified device
int	Dev, N;
int	I;
for I:= 1, N do ChOut(Dev, ^ );



public proc Hex1Out(Dev, N);	\Output a hex digit
int	Dev, N;
char	HexDigit;
begin
HexDigit:= "0123456789ABCDEF ";
ChOut(Dev, HexDigit(N & $0F));
end;	\Hex1Out



public proc Hex2Out(Dev, N);	\Output two hex digits (a byte)
int	Dev, N;
begin
Hex1Out(Dev, N>>4);
Hex1Out(Dev, N);
end;	\Hex2Out



public proc JustOut(Dev, N, Places, Fill); \Output a right-justified integer
\Always outputs correct value of N regardless of Places
int	Dev,	\output device number
	N,	\16-bit integer (negative numbers should be filled with spaces)
	Places,	\size of field in characters (right-justifies)
	Fill;	\character to fill background of field. Usually space (Sp) or ^0
int	I;	\ If not a space then beware of negative numbers.
begin
for I:= 1, Places-IntLen(N) do ChOut(Dev, Fill);
IntOut(Dev, N);
end;	\JustOut



public proc RlOutComma(Dev, X);	\Output a real with commas
\WARNINGS: Uses device 8.
\ Some negative numbers have a space after the minus sign (e.g: - 123,456.1234)
int	Dev;	\output device number
real	X;	\integer part of real to output
int	I, J, K, Ch;
begin
OpenO(8);			\convert real to ASCII string in device 8 buffer
RlOut(8, X);
ChOut(8, ^:);			\mark end of string

OpenI(8);			\find index of "." or ":"
I:= 0;
loop	begin
	Ch:= ChIn(8);
	if Ch=^. ! Ch=^: then quit;
	I:= I + 1;
	end;
J:= Rem(I/3);			\places in front of first comma (0=3)

OpenI(8);			\output digits ahead of "." or ":" with commas
K:= 0;
loop	begin
	Ch:= ChIn(8);
	ChOut(Dev, Ch);
	K:= K + 1;
	if K >= I then quit;
	if Rem(K/3)=J \& Ch#^-\ then
		ChOut(Dev, if Ch>=^0 & Ch<=^9 then ^, else ^ );
	end;
loop	begin			\output rest of number
	Ch:= ChIn(8);
	if Ch = ^: then quit;
	ChOut(Dev, Ch);
	end;
end;	\RlOutComma



public proc DrawBox(X0, Y0, X1, Y1, T); \Draw a rectangle using type T box chars
int	X0, Y0, X1, Y1, T;	\ with upper-left corner at X0,Y0 and
int	Tbl, I;			\ lower-right corner at X1,Y1
char	S;			\Note this uses device 6 with attributes
begin
\	 012345   Box Character Type (T):
Tbl:=  ["Ŀ ",	\0
	"͸Ծ ",	\1
	"ķӽ ",	\2
	"ͻȼ "];	\3
S:= Tbl(T&3);				\(mask for safety)

Cursor(X0, Y0);	ChOut(6, S(0));		\draw top line with brackets
for I:= X0+1, X1-1 do ChOut(6, S(1));
ChOut(6, S(2));
for I:= Y0+1, Y1-1 do			\draw sides
	[Cursor(X0, I);   ChOut(6, S(3));
	 Cursor(X1, I);   ChOut(6, S(3))];
Cursor(X0, Y1);   ChOut(6, S(4));	\draw bottom line with brackets
for I:= X0+1, X1-1 do ChOut(6, S(1));
ChOut(6, S(5));
end;	\DrawBox



public proc HLine(X0, Y0, X1, T); \Draw a horizontal line using type T box chars
int	X0, Y0, X1, T;		\ with left end at X0,Y0 and right end at X1,Y0
int	C, I;			\The minimum width (W) is 2
begin
C:= if T&1\#0\ then ^ else ^;
Cursor(X0, Y0);
for I:= X0, X1 do ChOut(6, C);
end;	\HLine



public proc VLine(X0, Y0, Y1, T); \Draw a horizontal line using type T box chars
int	X0, Y0, Y1, T;		\ with top end at X0,Y0 and bottom at X0,Y1
int	C, J;
begin
C:= if T&2\#0\ then ^ else ^;
for J:= Y0, Y1 do [Cursor(X0, J);   ChOut(6, C)];
end;	\VLine

\------------------------------- STRING ROUTINES -------------------------------

public proc StrCopy(A, B);	\Copy string: A --> B
char	A, B;	\Strings: B must already have enough space "Reserved".
int	I;	\ Beware if strings overlap
for I:= 0, MaxForSize do
	[B(I):= A(I);
	if A(I) >= $80 then return];



public proc StrNCopy(A, B, N);	\Copy string that is N bytes long: A --> B
char	A, B;	\strings: B must already have enough space "Reserved"
int	N;	\number of bytes to copy
int	I;
for I:= 0, N-1 do
	B(I):= A(I);



public func StrCmp(A, B);	\Compare string A to string B
\This returns:			 Strings are terminated with MSB set
\	>0 if A > B		 They do not need to be the same length
\	=0 if A = B
\	<0 if A < B
\This provides a general string compare, for example:
\ if StrCmp(A, B) >= 0 then...	(if A >= B then...)
char	A, B;	\Strings to be compared. If these contain numbers they must
int	I, R;	\ be right justified.
begin
for I:= 0, MaxForSize do
	begin
	if A(I) # B(I) then
		begin				\strings are not equal
		if (A(I) & B(I)) >= $80 then	\strings are the same length but
			return A(I) - B(I);	\ the last chars are not equal
		if A(I) >= $80 then
			begin			\string A is shorter than B
			R:= (A(I)&$7F) - B(I);
			return if R#0 then R else -(B(I+1)&$7F);
			end;
		if B(I) >= $80 then
			begin			\string B is shorter than A
			R:= A(I) - (B(I)&$7F);
			return if R#0 then R else (A(I+1)&$7F);
			end;
		return A(I) - B(I);		\not the terminating character
		end;
	if A(I) >= $80 then return 0;		\strings are exactly the same
	end;
end;	\StrCmp



public func StrNCmp(A, B, N);	\Compare string A to string B up to N bytes long
\This returns:
\	>0 if A > B
\	=0 if A = B
\	<0 if A < B
\This provides a general string compare, for example:
\ if StrNCmp(A, B, N) >= 0 then...	(if A >= B then...)
char	A, B;	\strings to be compared (must be right justified)
int	N;	\number of bytes to compare
int	I;
begin
for I:= 0, N-1 do
	if A(I) # B(I) then
		return A(I) - B(I);
return 0;			\they're equal
end;	\StrNCmp



public func StrLen(Str);	\Returns the number of characters in a string
char	Str;			\Str must be <= 32766 characters long
int	I;
for I:= 0, MaxForSize do
	if Str(I) >= $80 then return I+1;



public proc StrCat(S0, S1);	\Append (concatenate) string S1 onto S0
char	S0, S1;	\S0 must have space already reserved for S1
int	L0, L1;	\lengths of strings (bytes)
begin
L0:= StrLen(S0);
L1:= StrLen(S1);
if L0 > 0 then			\remove terminator from S0
	S0(L0-1):= S0(L0-1) & $7F;
StrNCopy(S1, S0+L0, L1);
end;	\StrCat



public func StrChr(Str, Char);	\Return address of Char in Str, 0 if none
char	Str;
int	Char;
int	I;
begin
for I:= 0, MaxForSize do
	begin
	if Char = Str(I) then return Str+I;
	if Str(I) >= $80 then
		[if Char = (Str(I) & $7F) then return Str+I;
		return 0];
	end;
end;	\StrChr



public proc StrNFill(Str, N, Pat);	\Fill string with pattern
char	Str;	\string
int	N,	\number of bytes to fill
	Pat;	\pattern
int	I;
begin
for I:= 0, N-1 do Str(I):= Pat;
end;	\StrNFill



public proc CTxt(X, Y, Str);	\Display a string on device 6 at X, Y
int	X, Y;
char	Str;
begin
Cursor(X, Y);
Text(6, Str);
end;	\CTxt



public proc TextN(Dev, Str, N);	\Output a string N bytes long
int	Dev;	\output device
char	Str;
int	N;
int	I;
begin
for I:= 0, N-1 do
	ChOut(Dev, Str(I));
end;	\TextN



public proc TextZ(Dev, Str);	\Output a zero-terminated string
int	Dev;	\output device
char	Str;
int	I;
begin
loop	begin
	for I:= 0, MaxForSize do
		if Str(I) = 0 then quit
		else ChOut(Dev, Str(I));
	quit;			\for safety
	end;
end;	\TextZ



public proc TextCenter(Dev, Str, Len); \Display a string centered on a line
int	Dev;	\output device		The line is the current cursor position
char	Str;	\string
int	Len;	\length of line
begin
SpOut(Dev, (Len-StrLen(Str))/2);
Text(Dev, Str);
end;	\TextCenter



public proc TextIn(Dev, Str, N); \Input a string ending with a carriage return
int	Dev;	\device number to input from
char	Str;	\address of string to write to (Reserve N bytes)
int	N;	\maximum number of bytes allowed in string
int	I, Ch;
begin
I:= 0;
loop	begin
	if I >= N then quit;
	repeat Ch:= ChIn(Dev) until Ch # LF;	\ignore line feeds
	Str(I):= Ch;
	I:= I + 1;
	if Ch=CR ! Ch=EOF then quit;
	end;
if I > 0 then Str(I-1):= Str(I-1) ! $80
else Str(0):= ^  ! $80;
end;	\TextIn



public proc Terminate(Str, N);	\Terminate string by setting MSB on last char
char	Str;
int	N;	\number of bytes in string
if N > 0 then Str(N-1):= Str(N-1) ! $80;



public func Unterminate(Str);	\Unterminate string by clearing MSB on last char
char	Str;			\ and return its length
int	N;	\number of bytes in string
begin
N:= StrLen(Str);
if N > 0 then Str(N-1):= Str(N-1) & $7F;
return N;
end;	\Unterminate



public func AtoI(Str);		\Convert a string of ASCII digits to an integer
char	Str;			\Must be terminated with MSB set on last digit
begin				\WARNING: This uses device 8
OpenO(8);
Text(8, Str);
Text(8, " 0");			\return 0 if Str does not contain a number
OpenI(8);
return IntIn(8);		\leading garbage is ignored
end;	\AtoI



public proc ItoA(I, Str);	\Convert a number I to a string of ASCII digits
int	I;			\WARNING: This uses device 8
char	Str;
begin
OpenO(8);
IntOut(8, I);
TextIn(8, Str, IntLen(I));	\string is terminated with MSB set
end;	\ItoA



public func real AtoF(Str);	\Convert string of ASCII digits to floating pt
char	Str;			\Must be terminated with MSB set on last digit
begin				\WARNING: This uses device 8
OpenO(8);
Text(8, Str);
Text(8, " 0");			\return 0 if Str does not contain a number
OpenI(8);
return RlIn(8);			\leading garbage is ignored
end;	\AtoF



public proc FtoA(F, Str);	\Convert real number F to string of ASCII digits
real	F;			\WARNINGS: This uses device 8
char	Str;			\Str must be big enough to hold number
begin
OpenO(8);
RlOut(8, F);			\format is determined by Format intrinsic
ChOut(8, CR);
TextIn(8, Str, 80);
Terminate(Str, StrLen(Str)-1);	\string is terminated with MSB set on last digit
end;	\FtoA



public func AHexToI(Str);	\Convert string of hex ASCII digits to integer
char	Str;			\Must be terminated with MSB set on last digit
begin				\WARNING: This uses device 8
OpenO(8);
Text(8, Str);
Text(8, " 0");			\return 0 if Str does not contain a number
OpenI(8);
return HexIn(8);		\leading garbage is ignored
end;	\AHexToI



public proc ItoAHex(I, Str); 	\Convert number I to string of hex ASCII digits
int	I;		 	\WARNING: This uses device 8
char	Str;
begin
OpenO(8);
HexOut(8, I);
TextIn(8, Str, if IntSize=2 then 4 else 8);  \string is terminated with MSB set
end;	\ItoAHex



public proc Str2BCD(From, To, Size); \Convert ASCII string to BCD string
int	From;	\address of ASCII string (MSB first)
char	To;	\address of space to put BCD string
int	Size;	\number of digits
int	I, W;
begin
for I:= 0, (Size+1)>>1-1 do
	[W:= From(I) & $0F0F;
	To(I):= W<<4 ! W>>8];
end;	\Str2BCD



public proc BCD2Str(From, To, Size); \Convert BCD string to ASCII string
char	From;	\address of BCD string
int	To,	\address of space to put ASCII string (MSB first)
	Size;	\number of digits
int	I, B;
begin
for I:= 0, (Size+1)>>1-1 do
	[B:= From(I);
	To(I):= (B>>4 ! B<<8) & $0F0F ! $3030];
end;	\BCD2Str



public func StrNAdd(A, B, N);	\Multiprecision add: A + B --> B
\Note: These number strings must be right justified
\Each byte contains one ASCII digit in the range 0-9
char	A,	\ASCII string containing number to be added
	B;	\ASCII string which acts as the accumulator
int	N;	\number of digits (number of bytes in A & B)
int	Sum, Carry, I;
begin
Carry:= 0;
for I:= -(N-1), 0 do		\for I:= Size-1 downto 0 do
	begin
	Sum:= A(-I) + B(-I) - ^0 + Carry;
	if Sum > ^9 then [Sum:= Sum - 10;   Carry:= 1]
	else Carry:= 0;
	B(-I):= Sum;
	end;
return Carry;
end;	\StrNAdd



public func StrNSub(A, B, N);	\Multiprecision subtract: A - B --> B
\Note: These number strings must be right justified
\Each byte contains one ASCII digit in the range 0-9
char	A,	\ASCII string containing number to be subtracted from
	B;	\ASCII string which acts as the accumulator
int	N;	\number of digits (number of bytes in A & B)
int	Dif, Borrow, I;
begin
Borrow:= 0;
for I:= -(N-1), 0 do		\for I:= Size-1 downto 0 do
	begin
	Dif:= A(-I) - B(-I) - Borrow;
	if Dif < 0 then [Dif:= Dif + 10;   Borrow:= 1]
	else Borrow:= 0;
	B(-I):= Dif + ^0;
	end;
return Borrow;
end;	\StrNSub



public proc StrNMul(A, B, C, N); \Multiprecision multiply A * B --> C
char	A, B, C; \ASCII strings
int	N;	 \number of digits in A and B (C must have 2*N digits)
int	I, J, K, Carry,
	NA, NB;  \indexes to first digit, skipping any leading zeros
begin
NA:= -1;   NB:= -1;
for I:= 0, N-1 do		\convert ASCII to binary
	begin
	A(I):= A(I) & $0F;   if A(I)\#0\ then if NA<0 then NA:= I;
	B(I):= B(I) & $0F;   if B(I)\#0\ then if NB<0 then NB:= I;
	C(I):= 0;		\initialize product to 0000...
	end;
for I:= N, N+N-1 do C(I):= 0;	\initialize rest of product to 0000...

for I:= -(N-1), -NA do		\for I:= N-1 downto 0 do
	begin
	Carry:= 0;
	K:= N - I;
	for J:= -(N-1), -NB do
		begin		\multiply like in grade school
		Carry:= (A(-I) * B(-J) + C(K) + Carry) / 10;
		C(K):= Rem(0);
		K:= K - 1;
		end;
	C(K):= Carry;
	end;

for I:= 0, N-1 do		\convert binary to ASCII
	begin
	A(I):= A(I) ! $30;
	B(I):= B(I) ! $30;
	C(I):= C(I) ! $30;
	end;
for I:= N, N+N-1 do
	C(I):= C(I) ! $30;
end;	\StrNMul



public proc StrNDiv(A, B, C, N); \Multiprecision divide A / B --> C
char	A;	\ASCII string containing dividend
		\Must be right-justified. Fill with leading zeros if necessary.
int	B;	\integer divisor (1..3276) 
char	C;	\ASCII string for quotient (result)
int	N;	\number of ASCII digits in string A
int	D, I;
begin
D:= 0;
for I:= 0, N-1 do
	begin
	D:= D + A(I) - ^0;	\get digit (converted to binary)
	C(I):= D/B + ^0;	\divide and convert to ASCII
	D:= Rem(0)*10;
	end;
end;	\StrNDiv

\-------------------------------- TIME ROUTINES --------------------------------

public proc Delay(T);		\Delay T microseconds (unsigned)
\The actual delay is a integral multiple of 976 microseconds.
\WARNING: This does not work for an XT, nor does it work for Windows XP.
int	T;
CallInt($15, $8600, 0, 0, T);	\BIOS function $86



public proc Wait(T);		\Wait T 18ths of a second or until keystroke
int	T;
int	I;
begin
for I:= 1, T do
	begin
	Sound(0, 1, 1);
	if ChkKey then return;
	end;
end;	\Wait



public proc WaitVB;		 \Wait for beginning of next vertical blank
int	SR;
begin
SR:= Peek($40, $63) + $306;	\port address of status register
if (SR & $39A) # $39A then return;
while PIn(SR, 0) & $08 do;	\wait for no vertical blank
repeat until PIn(SR, 0) & $08;	\wait for vertical blank
end;	\WaitVB;



public func GetTime;		\Returns the current time in DOS packed format
\Bits: 0-4 = second/2 (0-29);  5-A = minute (0-59);  B-F = hour (0-23)
int	S;
begin
CallInt($21, $2C00);		\DOS function $2C
S:= CpuReg(3) & $FFFF;		\save 100ths of seconds in global variable
Get100sec:= S>>8 *100 + (S&$FF);
return (CpuReg(2)&$FF00)<<3 ! (CpuReg(2)&$00FF)<<5 ! S>>9;
end;	\GetTime



public proc TimeOut(Dev, Time);	\Output Time (e.g: 5:43p)
int	Dev, Time;	\time in DOS packed format
int	HH, MM, SS,	\hours, minutes, seconds
	PM;		\Post Meridiem

	proc	NumOut(NN);	\Output 2-digit number with leading zero
	int	NN;
	begin
	if NN <= 9 then ChOut(Dev, ^0);
	IntOut(Dev, NN);
	end;	\NumOut

begin	\TimeOut
SS:= (Time & $001F) * 2;	\extract seconds, minutes, hours
MM:= Time>>5 & $003F;
HH:= Time>>11;

PM:= HH >= 12;
if HH > 12 then HH:= HH - 12;
if HH = 0 then HH:= 12;
if HH <= 9 then ChOut(Dev, Sp);
IntOut(Dev, HH);
ChOut(Dev, ^:);
NumOut(MM);
\ChOut(Dev, ^:);
\NumOut(SS);
ChOut(Dev, if PM then ^p else ^a);
end;	\TimeOut



public proc SecOut(Dev);	\Output seconds from GetTime (:ss.ss)
int	Dev;

	proc	NumOut(NN);	\Output 2-digit number with leading zero
	int	NN;
	begin
	if NN <= 9 then ChOut(Dev, ^0);
	IntOut(Dev, NN);
	end;	\NumOut

begin	\SecOut
ChOut(Dev, BS);			\back up over "a" or "p"
ChOut(Dev, ^:);
NumOut(Get100sec/100);
ChOut(Dev, ^.);
NumOut(Rem(0));
end;	\SecOut

\-------------------------------- DATE ROUTINES --------------------------------

public func GetDate;		\Returns the current date in DOS packed format
\Bits: 0-4 = day (1-31);  5-8 = month (1-12);  9-F = year (relative to 1980)
begin
CallInt($21, $2A00);		\DOS function $2A
return ((CpuReg(2)&$FFFF)-1980)<<9 ! (CpuReg(3)&$FF00)>>3 ! (CpuReg(3)&$00FF);
end;	\GetDate



public proc DateOut(Dev, Date);	\Output Date (MM-DD-YY)
int	Dev, Date;	\date in DOS packed format
int	DD, MM, YY;

	proc	NumOut(NN);	\Output 2-digit number with leading zero
	int	NN;
	begin
	if NN <= 9 then ChOut(Dev, ^0);
	IntOut(Dev, NN);
	end;	\NumOut

begin	\DateOut
DD:= Date & $001F;		\extract day, month, year
MM:= Date>>5 & $000F;
YY:= Date>>9 + 80;
YY:= Rem(YY/100);		\wrap at the year 2000

NumOut(MM);
ChOut(Dev, ^-);
NumOut(DD);
ChOut(Dev, ^-);
NumOut(YY);
end;	\DateOut



public proc WeekDay(Date);	\Return the day of the week (0=Sun 1=Mon..6=Sat)
int	Date;	\date in DOS packed format
int	DD, MM, YY;
begin
DD:= Date & $001F;		\extract day, month, year
MM:= Date>>5 & $000F;
YY:= Date>>9 + 4;		\the year 2000 is a normal leap year

if MM <= 2 then [MM:= MM + 10;   YY:= YY - 1] else MM:= MM - 2;
return Rem(((26*MM - 2)/10 + DD + YY + YY/4 + 61) / 7);
end;	\WeekDay

\------------------------------- VIDEO ROUTINES --------------------------------

public func GetVid;		\Returns the current video mode
return CallInt($10, $0F00) & $FF; \BIOS function $0F



public func GetPage;		\Return the currently active video page
return Peek(0, $462);



public proc SetPage(P);		\Set video page for writing and viewing
int	P;
CallInt($10, $0500 ! P&$FF);



public func ReadChar(X, Y);	\Get character and attribute at X,Y
int	X, Y;
begin
Cursor(X, Y);
return CallInt($10, $0800, GetPage);
end;	\ReadChar



public proc SetColReg(N, R, G, B); \Set VGA color register (DAC) N
int	N,	 \register to set (0..255)
	R, G, B; \red, green, and blue (only low 6 bits are used)
		 \ 2**6 = 64;  64*64*64 = 262144 possible colors
CallInt($10, $1010, N, G<<8 ! B&$FF, R<<8);  \BIOS function $10, subfunction $10



public proc SetPalReg(N, C);	\Set palette register
int	N,	\register to set (0..15)
	C;	\color for EGA bits: 0 0 r g b R G B  (64 colors)
CallInt($10, $1000, C<<8 ! N&$FF);  \0 0 0 I 0 R G B  (16 colors)



public proc AlignPalette;	\Make palette regs correspond to VGA DAC regs
int	N;
for N:= 0, 15 do
	CallInt($10, $1000, N<<8 !N);	\BIOS function $10, subfunction $00



public proc SetBorder(C);	\Set border color register
int	C;	\color for EGA bits: 0 0 r g b R G B  (64 colors)
CallInt($10, $1001, C<<8);	\    0 0 0 I 0 R G B  (16 colors)



public proc Blink(On);		\Turn video blinking off and on
\When blink is off, hi-intensity background colors are available.
\WARNING: Some BIOS's reset the blink bit when loading a font.
int	On;	\flag: True = blink on; False = blink off
int	BiosVal, BlinkBit, Port;
begin
\Try changing the blink bit with a call to BIOS
CallInt($10, $1003, if On then 1 else 0);

\See if it worked
BiosVal:= Peek(0, $465);		\get BIOS data
BlinkBit:= if On then $20 else $00;
if (BiosVal & $20) # BlinkBit then
	begin				\BIOS call didn't work--set it directly
	BiosVal:= BiosVal | $20;	\change blink bit
	Port:= Peek(0, $463) + $304;	\get port address of mode set register
	POut(BiosVal, Port, 0);		\update mode set register
	Poke(0, $465, BiosVal);		\save updated value in BIOS data area
	end;
end;	\Blink

\------------------------------- MOUSE ROUTINES --------------------------------

public func OpenMouse;		\Initializes mouse; returns 'false' if it fails
begin				\Pointer is set to center of screen but hidden
CallInt($21, $3533);		\Make sure mouse vector $33 points to something
HaveMouse:= false;
if ((CpuReg(1) ! CpuReg(11)) & $FFFF) = 0 then return false;
HaveMouse:= CallInt($33, $0000); \reset mouse and get status
if HaveMouse then HaveMouse:= true; \(beware of 'not' operator in 32-bit XPL)
return HaveMouse;		\return 'false' if failure
end;	\OpenMouse



public proc ShowMouse(On);	\Turn mouse pointer on or off
\The video mode should be set before calling this routine.
\This counts the number of times the pointer is turned on or off. If the
\ pointer is turned off twice, it must be turned on twice before it
\ actually goes on. (If it's turned on twice it only needs to be turned off
\ once to make the pointer go off.) The pointer should be turned off before
\ drawing something over it and (if in text mode) before a program exits.
\ Setting the video mode turns off the pointer.
int	On;	\flag: true = pointer on; false = pointer off
if HaveMouse then
	CallInt($33, if On then $0001 else $0002);



public func GetMousePosition(N); \Return position of specified mouse coordinate
int	N;	\0 = X coordinate; 1 = Y coordinate
\For video modes $0-$E and $13 the maximum coordinates are 639x199, minus
\ the size of the pointer. For modes $F-$12 and VESA the coordinates are the
\ same as the pixels. For 80-column text modes divide the mouse coordinates
\ by 8 to get the character cursor position.
begin
if ~HaveMouse then return 0;
CallInt($33, $0003);
return (if N then CpuReg(3) else CpuReg(2)) & $FFFF;
end;	\GetMousePosition



public func GetMouseButton(N);	\Return 'true' if specified mouse button is down
int	N;	\button number: 0 = left; 1 = right (or middle)
begin
if ~HaveMouse then return false;
CallInt($33, $0003);
return if N then (CpuReg(1)&2)=2 else (CpuReg(1)&1)=1;
end;	\GetMouseButton



public proc MoveMouse(X, Y);	\Move mouse pointer to X,Y
int	X, Y;
if HaveMouse then
	CallInt($33, $0004, 0, X, Y);
