\PUZZLE.XPL	MAR-02-95
\Sliding Block Puzzle by Loren Blaney
\Inspired by the earily Macintosh desk-top accessory and Martin Gardner.
\(14+26+22) = 62 = Maximum depth of search tree.

inc	\CXPL\STDDEF.XPL; \Include library routines and standard definitions
eproc	LoadLBM;	\Load 640x480x16 .LBM file
ext	MovImage;	\Move Image (Width, Height, X0, Y0, X1, Y1)
ext	SaveImage, SaveMask, DrawImage, ImageSize;
ext	WaitBeam,	\Wait for video beam
	ReadTime,	\Read the timer 2 time (must be set up by WaitBeam 1st)
	DelayX,		\Time delay in ticks (838 ns/tick)
	Interrupt;	\Set or clear interrupts
ext	CopyLine;	\Copy a scan line to the screen

\Coordinates in pixels:
def	PuzX=208, PuzY=66;		\Upper-left corner of puzzle
def	LEDX= 52, LEDY= 30;		\Reference point for LED window
def	CountX=280, CountY=36;		\Move counter
def	CountLX=512, CountLY=36;	\Best (low) move counter
\Buttons:
def	ResetX=236, ResetY=448, ResetH=23, ResetW=63;
def	ShuffleX=356, ShuffleY=448, ShuffleH=23, ShuffleW=63;
def	ExitX=504, ExitY=443, ExitH=11, ExitW=11;

def	VGA2YMax=338;			\Maximum Y value on 2nd VGA page

def	PuzSize =3;	\Size of the puzzle. Number of blocks on a side.
def	ColorSize =16;	\Number of colors

int	Count,		\Count of the number of moves
	DataSeg,	\Data segment address that this program executes from
	PSPSeg,		\Segment of PSP (for /Mouse switch)
	DoSound,	\Flag: Sound on/off (S key)
	HoleX, HoleY,	\Coordinates of the hole (empty position)
	LEDOff, LEDRed, LEDGrn,	\Segment addresses of LED images
	LEDSize,	\Number of paragraphs needed to hold LED image
	LowCount,	\Lowest count of several games
	Mode,		\Original video mode to restore upon exit
	MousePress,	\Flag: Mouse button has been pressed
	MousePX, MousePY, \Mouse press & release coords
	MouseRX, MouseRY,
	Puz,		\Array (3x3) of blocks, the puzzle
	Puz0,		\Array (3x3) of blocks, the puzzle in starting position
	SelX, SelY,	\Coordinates of the selected block
	Timexout,	\Timeout time for ReadTime
	BlkImage,	\Segment address of block image
	II, JJ;		\Scratch for Main

char	FontTable;	\8x14 font, for 256 characters
def	FontSize=256*16;\Space required for 8x16 font
char	BlkColRegs,	\Array: Copy of color (DAC) registers (R,G,B,R,G,B,...)
	Palette;	\Array: copy of palette registers (0-15)

def \GetCmd\ MovBlk, Reset, Shuffle, BadCmd;

\--------------------------- MOUSE ROUTINES ---------------------------

proc	GetSwitch;	\Set "HaveMouse" true if there is a "/M" on command line
\Outputs HaveMouse
int	I, CH;
char	CmdTail;
begin
CmdTail:= Reserve($80);			\Get command tail from PSP
Blit(PSPSeg, $80, DataSeg, CmdTail, $80);

for I:= 1, CmdTail(0) do
	begin
	Ch:= CmdTail(I);
	if Ch = ^/ then
		begin
		Ch:= CmdTail(I+1);
		if Ch=^M ! Ch=^m then HaveMouse:= false;
		end;
	end;
end;	\GetSwitch



proc	ClearMouse;	\Clear any mouse button pushes
begin
MousePress:= false;
if ~HaveMouse then return;
CallInt($33, 5, 0);	\Clear any press info for left mouse button
CallInt($33, 6, 0);	\Clear any release info for left mouse button
end;	\ClearMouse



proc	SetupMouse;	\Initialize mouse
begin
HaveMouse:= true;
GetSwitch;
if HaveMouse then OpenMouse;
if ~HaveMouse then return;
ShowMouse(true);
MoveMouse(PuzX+(120*PuzSize)/2, PuzY+(120*PuzSize)/2);

\Don't let mouse get into the LEDs
CallInt($33, $0007, 0, PuzX, PuzX+120*PuzSize-1);	\Set horizontal limits
CallInt($33, $0008, 0, PuzY, ExitY+17);			\Set vertical limits

ClearMouse;
end;	\SetupMouse



func	ChkButton;	\Return 'true' if left mouse button was pressed
begin
if ~HaveMouse then return false;
if MousePress then return true;

CallInt($33, 5, 0);	\Get button press info for left mouse button
if Cpureg(1) \#0\ then	\Left mouse button was pressed
	begin
	MousePX:= Cpureg(2);	\Save coordinates
	MousePY:= Cpureg(3);
	MousePress:= true;	\(Cleared in GetCmd and ClearMouse)
	return true;
	end;

return false;
end;	\ChkButton

\------------------------ VGA DISPLAY ROUTINES ------------------------

proc	LoadFont;	\Load the font table
\This loads a 14-byte-high font and kludges it into a standard 16-byte-high font
int	Handle, I, Ch, L;
begin
Handle:= Fopen("PUZZLE.F14", 0);
Fset(Handle, ^I);
Openi(3);

I:= 0;					\Index into font table
for Ch:= 0, 255 do			\For 256 characters...
	begin
	for L:= 0, 14-1 do		\14 bytes per character
		[FontTable(I):= Chin(3);   I:= I +1];
\	FontTable(I):= 0;   I:= I +1;	\\Kludge on two blank bytes
\	FontTable(I):= 0;   I:= I +1;
	I:= I -5;
\	for L:= 10, 14-1 do		\\14 bytes per character
\		Ch:= Chin(3);
	end;
FontTable($DB*16+14):= $FF;		\Fix up block character $DB
FontTable($DB*16+15):= $FF;		\ (required for background colors)

Fclose(Handle);
end;	\LoadFont



proc	GetBlkColRegs(BCR);	\Get block of color registers (DACs)
char	BCR;			\Block of Color Registers (R,G,B,R,G,B,...)
CallInt($10,
	$1017,			\Function $10, subfunction $17
	0,			\First color register
	ColorSize,		\Number of color registers
	BCR,			\Array of values to set registers to
	0, 0, DataSeg);



proc	SetBlkColRegs(BCR);	\Set block of color registers (DACs)
char	BCR;			\Block of Color Registers (R,G,B,R,G,B,...)
CallInt($10,
	$1012,			\Function $10, subfunction $12
	0,			\First color register
	ColorSize,		\Number of color registers
	BCR,			\Array of values to set registers to
	0, 0, DataSeg);



proc	FadeIn(Spd, BCRTarget);	\Fade in the image in display memory
int	Spd;			\Speed of fade (0, 1, 2... slow to fast)
char	BCRTarget;		\Array of colors to fade to
int	I, J;
char	BCR;			\Current colors
begin
BCR:= Reserve(ColorSize*3);

for I:= 1, 64 do		\for 64 intensities...
	begin
	for J:= 0, ColorSize*3-1 do
		BCR(J):= (BCRTarget(J) *I) >>6;
	WaitVB;
	SetBlkColRegs(BCR);
	I:= I +Spd;
	end;
end;	\FadeIn



proc	FadeOut(Spd);		\Fade out an image
int	Spd;			\Speed of fade (0, 1, 2... slow to fast)
int	I, J;
char	BCR0, BCR;
begin
BCR0:= Reserve(ColorSize*3);	\Get initial colors
BCR:= Reserve(ColorSize*3);
GetBlkColRegs(BCR0);

for I:= -63, 0 do		\Turn down the intensities...
	begin
	for J:= 0, ColorSize*3-1 do
		BCR(J):= (BCR0(J) * -I) >>6;
	WaitVB;
	SetBlkColRegs(BCR);
	I:= I +Spd;
	end;
end;	\FadeOut

\---------------------------- EXIT ROUTINES ---------------------------

proc	RollUp;		\Roll up screen image
def	Radius=30, Radius2=Radius*2;	\Radius of roller
int	ScreenImage,	\Array: Image in memory
	Roller,		\Array(Radius2): Array of scan lines used to draw
			\ image of roller
	I;		\Scratch


	proc	DrawRoller(Y);		\Draw the roller
	\Inputs: Radius, Roller
	int	Y;	\Top edge
	int	J, K, Yi;
	begin
	K:= Radius2 + Fix(Pi *Float(Radius));

	WaitBeam(Y+Radius2+10, false);
	for J:= 0, Radius2-1 do
		begin
		Yi:= Y + Roller(J);
		if Yi >= 480 then
			Yi:= Y + (if J <= Radius then J else K-Roller(J))
		else	CopyLine(Yi, Y+J, ScreenImage, 4);
		end;
	end;	\DrawRoller



	proc	UnrollImage;		\Unroll image from screen
	int	Y;
	begin
	WaitBeam(480+Radius2+10, true);
	for Y:= -(480-1), Radius2 do
		begin
		DrawRoller(-Y);
		\Draw horizontal lines under roller to blank out image
		Move(0, Radius2-Y+0);   Line(640-1, Radius2-Y+0, 0);
		Move(0, Radius2-Y+1);   Line(640-1, Radius2-Y+1, 0);
		Move(0, Radius2-Y+2);   Line(640-1, Radius2-Y+2, 0);
		Move(0, Radius2-Y+3);   Line(640-1, Radius2-Y+3, 0);
		Y:= Y +3;
		end;
	end;	\UnrollImage



	proc	MakeRoller;		\Make Roller, an array of scan lines
	\Inputs: Radius; Outputs: Roller
	def	Pi2 = Pi/2.0, Pi32 = Pi2*3.0;
	int	J, K, Yi;
	real	R, Ang;
	begin
	R:= Float(Radius);
	for J:= 0, Radius2-1 do
		begin
		K:= Radius -J;
		Ang:= Acos(Float(K) / R); \Ang steps from -Pi/2 to +Pi/2
		Yi:= Fix((Pi32 + 1.0 - Ang) *R);
		Roller(J):= Yi;
		end;
	end;	\MakeRoller


begin	\RollUp
ScreenImage:= Malloc(ImageSize(0, 0, 640-1, 480-1, $0f));
SaveImage(0, 0, 640-1, 480-1, $0f, ScreenImage);

Roller:= Reserve(Radius2*Intsize);
MakeRoller;

UnrollImage;
Release(ScreenImage);
end;	\RollUp



proc	Exit;		\Make a graceful exit
int	I;
begin
Openi(0);
ShowMouse(false);
\Show Exit pin removed
MovImage(488, 172+480, (ExitX-16)&$fff8, ExitY, 48, 37);
RollUp;

WaitVB;
SetVid(Mode);		\Restore original video mode
exit;
end;	\Exit



proc	Fatal(Str);	\Fatal error handler: Display message and exit program
char	Str;
int	I;
begin
SetVid(Mode);		\Restore original video mode (text mode)
Text(0,"
Oops, ");
Chout(0, Bel);
Text(0, Str);
Crlf(0);

WaitKey;
exit;
end;	\Fatal

\----------------------------------------------------------------------

proc	Click;		\Make a click sound
begin
Sound(false, 1, 3000);	\Sync to system clock so click always sounds the same
Sound(DoSound, 1, 3000);
end;	\Click



proc	ShowMoveCount;		\Display the move counter
begin
Cursor(CountX/8, CountY/9);
Attrib($91);
if Count <10 then Chout(6, Sp);
if Count <100 then Chout(6, Sp);
Intout(6, Count);
end;	\ShowMoveCount



proc	ShowLowCount;		\Display the low move counter
begin
Cursor(CountLX/8, CountLY/9);
Attrib($91);
if LowCount <10 then CHOUT(6, SP);
if LowCount <100 then CHOUT(6, SP);
if LowCount <1000 then INTOUT(6, LowCount);
end;	\ShowLowCount



proc	Won;			\Indicate puzzle is solved by player
int	N;
begin
if Count < LowCount then
	begin
	LowCount:= Count;
	ShowLowCount;
	end;

Sound(DoSound, 2, 3036);	\(Apologies to Beetoven)
Sound(0, 1, 1);
Sound(DoSound, 2, 3036);
Sound(0, 1, 1);
Sound(DoSound, 2, 3036);
Sound(0, 1, 1);
Sound(DoSound, 8, 4052);
end;	\Won



func	PuzSolved;		\Return TRUE if puzzle is solved
int	I, X, Y, TBL;
begin
\Solution:
\ 7  8 (0)
\ 4  5  6
\ 1  2  3
TBL:= [7, 8, 0, 4, 5, 6, 1, 2, 3];

I:= 0;
for Y:= 0, PuzSize-1 do
	for X:= 0, PuzSize-1 do
		begin
		if Puz(X, Y) #TBL(I) then return false;
		I:= I +1;
		end;
return true;
end;	\PuzSolved



proc	WaitX(T);
\Delay T 18ths of a second or until mouse button or keystroke.
int	T;
int	I;
begin
for I:= 1, T do
	begin
	Sound(0, 1, 1);
	if ChkKey then
		if LookKey = Esc then Exit else return;
	if ChkButton then return;
	end;
end;	\WaitX



proc	WaitXS(T);
\Delay T 18ths of a second or until mouse button or keystroke
\ and show the countdown.
int	T;
int	I;
begin
Attrib($91);
T:= T /18;		\Convert T to seconds (approx)
for I:= -T, -1 do
	begin
	Cursor(LEDX/8+5, LEDY/9-1);	\Show count down
	if -I < 10 then SpOut(6, 1);
	Intout(6, -I);

	WaitX(18);

	if ChkKey then
		if LookKey = Esc then Exit else I:= 0;
	if ChkButton then I:= 0;
	end;
Cursor(LEDX/8+2, LEDY/9-1);		\Erase count
SpOut(6, 5);
end;	\WaitXS

\----------------------- SLIDING BLOCK ANIMATION ----------------------

proc	MoveBlock(X0, Y0, X1, Y1, StepSize);	\Do animation to move a block
int	X0, Y0,		\From block [0..2]
	X1, Y1,		\To block [0..2]
	StepSize;	\Determines speed to move block

int	ObjX, ObjY,	\Position of BLOCK
	TarX, TarY,	\Target location of block (graphic coordinates)
	MouseY,		\Current mouse Y position
	MouseOff,	\Flag: Mouse pointer is turned off
	I, J, K,
	Shadow0, Shadow1, \arrays for bottom and right-side shadow
	ShadowSize;
def	StepMax=24;	\Maximum step size (must be even)



proc	SetShadows;		\Save images for the shadows
begin
case of
  ObjY > TarY:		\up
	begin
	ShadowSize:= ImageSize(40, 131+480, 159, 146+480+StepSize, $08);
	Shadow0(0):= Malloc(ShadowSize);
	Interrupt(false);
	SaveImage(40, 131+480, 159, 146+480+StepSize, $08, Shadow0(0));
	Interrupt(true);

	ShadowSize:= ImageSize(144, 27+480, 159, 130+480+StepSize, $08);
	Shadow1(0):= Malloc(ShadowSize);
	Interrupt(false);
	SaveImage(144, 27+480, 159, 130+480+StepSize, $08, Shadow1(0));
	Interrupt(true);
	for J:= 1, 3 do	\kludge for release below
		begin
		Shadow0(J):= Malloc(1);
		Shadow1(J):= Malloc(1);
		end;
	end;
  ObjY < TarY:		\down
	begin
	ShadowSize:= ImageSize(40, 131+480, 159, 146+480+StepSize, $08);
	Shadow0(0):= Malloc(ShadowSize);
	Interrupt(false);
	SaveImage(40, 131+480-StepSize, 159, 146+480, $08, Shadow0(0));
	Interrupt(true);

	ShadowSize:= ImageSize(144, 27+480, 159, 130+480+StepSize, $08);
	Shadow1(0):= Malloc(ShadowSize);
	Interrupt(false);
	SaveImage(144, 27+480-StepSize, 159, 130+480, $08, Shadow1(0));
	Interrupt(true);
	for J:= 1, 3 do	\kludge for release below
		begin
		Shadow0(J):= Malloc(1);
		Shadow1(J):= Malloc(1);
		end;
	end;
  ObjX > TarX:		\left    <--
	begin
	K:= (StepSize+7) & $fff8;
	for J:= 0, 3 do
		begin
		ShadowSize:= ImageSize(40+152*J, 131+480,
			(159+154*J+StepSize &$fff8)+7, 146+480, $08);
		Shadow0(J):= Malloc(ShadowSize);
		Interrupt(false);
		SaveImage(40+152*J, 131+480,
			(159+154*J+StepSize &$fff8)+7, 146+480, $08, Shadow0(J));
		Interrupt(true);

		ShadowSize:= ImageSize(144+152*J, 27+480,
			(159+154*J+StepSize &$fff8)+7, 130+480+K, $08);
		Shadow1(J):= Malloc(ShadowSize);
		Interrupt(false);
		SaveImage(144+152*J, 27+480,
			(159+154*J+StepSize &$fff8)+7, 130+480, $08, Shadow1(J));
		Interrupt(true);
		end;
	end;
  ObjX < TarX:		\right   -->
	begin
	K:= (StepSize+7) & $fff8;
	for J:= 0, 3 do
		begin
		ShadowSize:= ImageSize(40+154*J-StepSize &$fff8, 131+480,
			(159+154*J &$fff8)+7, 146+480, $08);
		Shadow0(J):= Malloc(ShadowSize);
		Interrupt(false);
		SaveImage(40+154*J-StepSize &$fff8, 131+480,
			(159+154*J &$fff8)+7, 146+480, $08, Shadow0(J));
		Interrupt(true);

		ShadowSize:= ImageSize(144+154*J-StepSize &$fff8, 27+480,
			(159+154*J &$fff8)+7, 130+480+K, $08);
		Shadow1(J):= Malloc(ShadowSize);
		Interrupt(false);
		SaveImage(144+154*J-StepSize &$fff8, 27+480,
			(159+154*J &$fff8)+7, 130+480, $08, Shadow1(J));
		Interrupt(true);
		end;
	end
other	[];
end;	\SetShadows



proc	Dispose;		\Dispose of existing shadows
begin
for J:= 0, 3 do
	begin
	Release(Shadow0(J));
	Release(Shadow1(J));
	end;
end;	\Dispose



begin	\MoveBlock
Shadow0:= Reserve(4*Intsize);
Shadow1:= Reserve(4*Intsize);

StepSize:= (StepSize+1) & $fffe;	\Make sure it is even
if StepSize > StepMax then StepSize:= StepMax;

ObjX:= PuzX + X0 *120;		\Get screen coordinates
ObjY:= PuzY + Y0 *120;
TarX:= PuzX + X1 *120;
TarY:= PuzY + Y1 *120;

\Make four copies of the number in the block into non-displayed video RAM.
\The block is moved by cycling through these four images.
MouseY:= GetMousePosition(1);
WaitBeam(MouseY+16+60, true);
ShowMouse(false);
Interrupt(false);
SaveImage(ObjX+32, ObjY+16, ObjX+120-32-1, ObjY+120-32-1, $0f, BlkImage);
Interrupt(true);
ShowMouse(true);
MouseOff:= false;
for I:= 0, 3 do
	begin
	Interrupt(false);
	DrawImage(32+32+I*154, 480+12+16, BlkImage, 0);		\Horizontal
	Interrupt(true);
	end;
for I:= 0, 3 do
	begin
	Interrupt(false);
	DrawImage(8+32+I*120, 480+164+16+I+I, BlkImage, 0);	\Vertical
	Interrupt(true);
	end;

SetShadows;
WaitBeam(ObjY+120+40-60, true);
loop	begin
	if ChkButton ! Chkkey then
		begin
		if Chkkey then
			if LookKey = Esc then Exit;
		StepSize:= StepMax;
		Dispose;
		SetShadows;
		end;

	case of
	  ObjY > TarY:
		begin			\Move up
		ObjY:= ObjY -StepSize;
		if ObjY < TarY then ObjY:= TarY;
		K:= PuzY & $07;
		J:= (ObjY-K)>>1 & $0003;
		WaitBeam(ObjY+120+60, false);

		MouseY:= GetMousePosition(1);
		if MouseY+32>=ObjY & MouseY<=ObjY+120+16+StepSize then
			[ShowMouse(false);  MouseOff:= true];

		Interrupt(false);
		Pout($0702, $3C4, 1);	\Disable bit plane 3
		MovImage(8+120*J, 480+164+J+J, ObjX, ObjY, 120, 120+StepSize);
		Interrupt(true);

		Interrupt(false);
		Pout($0802, $3C4, 1);	\Enable bit plane 3 for the shadow
		DrawImage(ObjX+112, ObjY+15, Shadow1(0), 0);
		DrawImage(ObjX+8, ObjY+119, Shadow0(0), 0);
		Pout($0F02, $3C4, 1);	\Enable all planes
		Interrupt(true);

		if MouseOff then ShowMouse(true);
		MouseOff:= false;
		end;

	  ObjY < TarY:
		begin			\Move down
		ObjY:= ObjY +StepSize;
		if ObjY > TarY then ObjY:= TarY;
		K:= PuzY & $07;
		J:= (ObjY-K)>>1 & $0003;
		WaitBeam(ObjY+120+60, false);

		MouseY:= GetMousePosition(1);
		if MouseY+32>=ObjY-StepSize & MouseY<=ObjY+120+16 then
			[ShowMouse(false);  MouseOff:= true];

		Interrupt(false);
		Pout($0702, $3C4, 1);	\Disable bit plane 3
		MovImage(8+120*J, 480+164+J+J-StepSize, OBJX, OBJY-StepSize,
			120, 120+StepSize);
		Interrupt(true);

		Interrupt(false);
		Pout($0802, $3C4, 1);	\Enable bit plane 3
		DrawImage(ObjX+112, ObjY+15-StepSize, Shadow1(0), 0);
		DrawImage(ObjX+8, ObjY+119-StepSize, Shadow0(0), 0);
		Pout($0F02, $3C4, 1);	\Enable all planes
		Interrupt(true);

		if MouseOff then ShowMouse(true);
		MouseOff:= false;
		end;

	  ObjX > TarX:
		begin			\Move to the left   <--
		ObjX:= ObjX -StepSize;
		if ObjX < TarX then ObjX:= TarX;
		J:= ObjX>>1 & $0003;
		K:= (J+J+StepSize+7) & $fff8;
		WaitBeam(ObjY+120+60, false);

		MouseY:= GetMousePosition(1);
		if MouseY+32>=ObjY & MouseY<=ObjY+120+16 then
			[ShowMouse(false);  MouseOff:= true];

		Interrupt(false);
		Pout($0702, $3C4, 1);	\Disable bit plane 3
		MovImage(32+152*J, 480+12, ObjX&$FFF8, ObjY, 120+K, 120);
		Interrupt(true);

		Interrupt(false);
		Pout($0802, $3C4, 1);	\Enable bit plane 3
		DrawImage((ObjX+112)&$FFF8, ObjY+15, Shadow1(J), 0);
		DrawImage((ObjX+8)&$FFF8, ObjY+119, Shadow0(J), 0);
		Pout($0F02, $3C4, 1);	\Enable all planes
		Interrupt(true);

		if MouseOff then ShowMouse(true);
		MouseOff:= false;
		end;

	  ObjX < TarX:
		begin			\Move to the right   -->
		ObjX:= ObjX +StepSize;
		if ObjX > TarX then ObjX:= TarX;
		J:= ObjX>>1 & $0003;
		K:= (StepSize+7) & $fff8;
		WaitBeam(ObjY+120+60, false);

		MouseY:= GetMousePosition(1);
		if MouseY+32>=ObjY & MouseY<=ObjY+120+16 then
			[ShowMouse(false);  MouseOff:= true];

		Interrupt(false);
		Pout($0702, $3C4, 1);	\Disable bit plane 3
		MovImage((32+154*J-StepSize)&$fff8, 480+12, (ObjX-StepSize)&$fff8,
			ObjY, 120+K, 120);
		Interrupt(true);

		Interrupt(false);
		Pout($0802, $3C4, 1);	\Enable bit plane 3
		DrawImage((ObjX+112-StepSize)&$FFF8, ObjY+15, Shadow1(J), 0);
		DrawImage((ObjX+8-StepSize)&$FFF8, ObjY+119, Shadow0(J), 0);
		Pout($0F02, $3C4, 1);	\Enable all planes
		Interrupt(true);

		if MouseOff then ShowMouse(true);
		MouseOff:= false;
		end

	other	quit;			\Loop until block is at target
	end;	\loop

Dispose;

if Count < 999 then
	begin
	Count:= Count +1;
	ShowMoveCount;
	end;
end;	\MoveBlock

\----------------------------------------------------------------------

proc	DoMove(StepSize);	\Perform selected move (SelX, SelY)
int	StepSize;
int	I, J;
begin
if SelX = HoleX then
	begin
	if HoleY > SelY then
		begin			\Move down
		for J:= -HoleY, -(SelY+1) do	\for J:= HoleY downto SelY+1 do
			[MoveBlock(SelX, -J-1, SelX, -J, StepSize);
			Puz(SelX, -J):= Puz(SelX, -J-1)];
		end
	else				\HoleY < SelY
		begin			\Move up
		for J:= HoleY, SelY-1 do
			[MoveBlock(SelX, J+1, SelX, J, StepSize);
			Puz(SelX, J):= Puz(SelX, J+1)];
		end
	end
else	begin		\SelY = HoleY
	if HoleX > SelX then
		begin			\Move right
		for I:= -HoleX, -(SelX+1) do	\for I:= HoleX downto SelX+1 do
			[MoveBlock(-I-1, SelY, -I, SelY, StepSize);
			Puz(-I, SelY):= Puz(-I-1, SelY)];
		end
	else				\HoleX < SelX
		begin			\Move left
		for I:= HoleX, SelX-1 do
			[MoveBlock(I+1, SelY, I, SelY, StepSize);
			Puz(I, SelY):= Puz(I+1, SelY)];
		end
	end;

HoleX:= SelX;   HoleY:= SelY;			\Hole appears at selected block
Puz(HoleX, HoleY):= 0;
end;	\DoMove



proc	MixBlksUp;	\Randomly mix up the blocks
\Outputs: Count, HoleX, HoleY, SelX, SelY
int	I, T,
	X, Y,
	X0, Y0,
	HoleX0, HoleY0;	\Prevents undoing previous move
begin
Count:= 0;
ShowMoveCount;

\Mix up the blocks by sliding them around
\WARNING: This is tricky as hell and the comments are wrong!
HoleX0:= -1;   HoleY0:= -1;		\Don't move block in the center first
for I:= 1, 40 do
	begin				\Randomly select a legal move
	repeat	SelX:= RAN(PuzSize);	\"Move" selected might actually be 2 moves
		SelY:= RAN(PuzSize);	\Select column or row of hole but not the hole
	until (SelX=HoleX & SelY#HoleY  !  SelY=HoleY & SelX#HoleX) &
		not (SelX=HoleX0 !\&\ SelY=HoleY0); \And don't select block that moved
		\ into hole on last move because it might undo the last move
	HoleX0:= HoleX;   HoleY0:= HoleY;  \Save posn of hole before move
	DoMove(I+I);
	if ChkButton ! Chkkey then I:= MaxForSize;
	end;

Count:= 0;
ShowMoveCount;
end;	\MixBlksUp

\------------------------- SEARCH FOR SOLUTION ------------------------
\
\This routine solves the sliding-block puzzle. It uses a recursive tree
\ search to find a sequence of moves that transforms an arbitrary initial
\ configuration into the winning configuration (show below). The search
\ time is made over a hundred times faster by the use of sub-goals. First
\ the sequence of moves to get the #1 block in the correct position are
\ found; then the moves for blocks 2 and 3 are found. Only then is the
\ sequence of moves to get all blocks in the correct positions searched
\ for. This is much faster, but it does not give the optimum solution which
\ is 30 moves, worst case.
\
\The winning configuration is:
\
\	7 | 8 |
\	---------
\	4 | 5 | 6
\	---------
\	1 | 2 | 3
\
\The puzzle boxes themselves (Box) are numbered as follows:
\
\	7 | 6 | 5
\	---------
\	8 | 0 | 4
\	---------
\	1 | 2 | 3


proc	Search;		\Search for the sequence of moves to solve puzzle & do it
char	Box;		\Array: array of boxes (the puzzle state)
char	Ans;		\Array: answer, sequence of moves to get winning position
int	Tx, Ty,		\Conversion tables (between Box and P)
	Adj,		\Table of locations adjacent to a hole (legal moves)
	N,		\Block number
	H,		\Location of hole in initial configuration
	J,		\Scratch
	Counter,	\Count of the number of moves tried
	OneSolved,	\Flag: 1 is in correct position
	HalfSolved,	\Flag: 1, 2 & 3 are in correct position
	Solved,		\Flag: All blocks are in correct position
	MaxDepth,	\Maximum depth that tree search is limited to
	Depth;		\Current depth of tree search



proc	TryMove(Fr, To); \Make a trial move and check for winning position
int	Fr, To;		\Box numbers (which contain numbered blocks)
begin
case of		\Terminate search if: solution found, mouse button, or keystroke
  ChkButton, Solved, Chkkey: return
other;

repeat until Timexout-ReadTime <= 0;	\Signed compare handles wrap around
Timexout:= ReadTime +2386;		\= 2ms;  2000us / 0.838us/tick

Counter:= Counter +1;
Attrib($91);
\Kludge to reduce flicker (caused by xoring a block char to get background color)
case of
 Counter<10:	J:= 4;
 Counter<100:	J:= 3;
 Counter<1000:	J:= 2;
 Counter<10000:	J:= 1
other		J:= 0;
Cursor(LEDX/8+2+J, LEDY/9-1);
Intout(6, Counter);

Ans(Depth):= FR;			\Record move in answer array

Interrupt(false);
DrawImage(LEDX+FR*7, LEDY+Depth*7, LEDRed, 0);
Interrupt(true);

Depth:= Depth +1;

Box(To):= Box(Fr);   Box(Fr):= 0;	\Move block

case of					\Check for # 1 block in correct position
  OneSolved, Box(1)#1:[]
other	begin
	OneSolved:= true;
	MaxDepth:= Depth +26;		\Change maximum depth for next sub-goal
	Interrupt(false);
	DrawImage(LEDX+FR*7, LEDY+(Depth-1)*7, LEDGrn, 0);
	Interrupt(true);
	end;

case of					\Check for # 1 2 3 in correct position
  HalfSolved, Box(1)#1, Box(2)#2, Box(3)#3:[]
other	begin
	HalfSolved:= true;
	MaxDepth:= Depth +22;		\Change max depth for final goal
	Interrupt(false);
	DrawImage(LEDX+FR*7, LEDY+(Depth-1)*7, LEDGrn, 0);
	Interrupt(true);
	end;

if FR =5 then				\Check for all blocks in correct position
	begin
	case of
	  ~HalfSolved, Box(8)#4, Box(0)#5, 
	  Box(4)#6, Box(7)#7, Box(6)#8: []
	other	begin
		Solved:= true;		\Solution is found
		MaxDepth:= Depth -1;	\Get actual depth
		Interrupt(false);
		DrawImage(LEDX+FR*7, LEDY+(Depth-1)*7, LEDGrn, 0);
		Interrupt(true);
		return;			\Start unnesting the recursion
		end;
	end;

if Depth <MaxDepth then			\Limit search to maximum depth
	begin
	case FR of
	  0:	[if ~HalfSolved then if 2 #TO then TryMove(2, 0); \Don't undo sub-goal
		if 4 #TO then TryMove(4, 0);	\Don't undo previous move
		if 6 #TO then TryMove(6, 0);
		if 8 #TO then TryMove(8, 0)];
	  1:	[if ~HalfSolved then if 2 #TO then TryMove(2, 1);
		if 8 #TO then TryMove(8, 1)];
	  2:	[if ~OneSolved then if 1 #TO then TryMove(1, 2);
		if ~HalfSolved then if 3 #TO then TryMove(3, 2);
		if 0 #TO then TryMove(0, 2)];
	  3:	[if ~HalfSolved then if 2 #TO then TryMove(2, 3);
		if 4 #TO then TryMove(4, 3)];
	  4:	[if ~HalfSolved then if 3 #TO then TryMove(3, 4);
		if 0 #TO then TryMove(0, 4);
		if 5 #TO then TryMove(5, 4)];
	  5:	if 4 #TO then TryMove(4, 5) else TryMove(6, 5);
	  6:	[if 0 #TO then TryMove(0, 6);
		if 5 #TO then TryMove(5, 6);
		if 7 #TO then TryMove(7, 6)];
	  7:	if 6 #TO then TryMove(6, 7) else TryMove(8, 7);
	  8:	[if ~OneSolved then if 1 #TO then TryMove(1, 8);
		if 0 #TO then TryMove(0, 8);
		if 7 #TO then TryMove(7, 8)]
	other	Fatal("TryMove");
	end;

Box(Fr):= Box(To);   Box(To):= 0;	\Move block back
Depth:= Depth -1;
if ~Solved then
	begin
	Interrupt(false);
	DrawImage(LEDX+FR*7, LEDY+Depth*7, LEDOff, 0);
	Interrupt(true);
	end;
end;	\TryMove



begin	\Search
Box:= Reserve(9);
Ans:= Reserve(80);

WaitXS(10 *18);		\Give the humanoid time to try it and show count down
if PuzSolved then
	begin
	ShowMouse(false);		\Press and release the Shuffle button
	MovImage(488, 280+480, ShuffleX&$fff8, ShuffleY, 72, ShuffleH);
	ShowMouse(true);
	WaitX(8);
	ShowMouse(false);
	MovImage(488, 257+480, ShuffleX&$fff8, ShuffleY, 72, ShuffleH);
	ShowMouse(true);
	MixBlksUp;
	WaitX(3 *18);
	end;

if ChkButton ! Chkkey then return;

\Set up Box to reflect the current state of the puzzle (P):
\     0  1  2  3  4  5  6  7  8		   7 | 6 | 5
					\  ---------
Tx:= [1, 0, 1, 2, 2, 2, 1, 0, 0];	\  8 | 0 | 4
					\  ---------
Ty:= [1, 2, 2, 2, 1, 0, 0, 0, 1];	\  1 | 2 | 3

Adj:=[8, 8, 1, 4, 3, 4, 5, 6, 7];	\Adjacency table (legal move)

for J:= 0, 8 do
	begin
	N:= Puz(Tx(J), Ty(J));
	Box(J):= N;
	if N =0 then H:= J;		\Remember where the hole (H) is
	end;
N:= Adj(H);

Counter:= 0;
OneSolved:= false;
HalfSolved:= false;
Solved:= false;
MaxDepth:= 14;				\Limit tree search to find first sub-goal

Depth:= 0;
WaitBeam(10, true);
Timexout:= ReadTime;
TryMove(N, H);

if Solved then
	begin				\Move the blocks
	for J:= 0, MaxDepth do
		begin
		if not (ChkButton ! Chkkey) then
			begin
			SelX:= Tx(Ans(J));
			SelY:= Ty(Ans(J));
			DoMove(2);
			end;
		Interrupt(false);
		DrawImage(LEDX+Ans(J)*7, LEDY+J*7, LEDOff, 0);
		Interrupt(true);
		end;
	end;

Cursor(LEDX/8+2, LEDY/9-1); SpOut(6, 5);	\Erase count
end;	\Search

\-------------------------- GET PLAYER'S MOVE -------------------------

func	GetCmd;		\Get player's selected block (SelX, SelY) et cetra
int	CH, Tx, Ty, I,
	B, ButX, ButY,	\Arrays: Buttons, state and coordinates
	BR, BSh, BE;	\Flags for buttons
def	Buttons= 9;
begin
B:= Reserve(Buttons *IntSize);
\Locations of blocks in puzzle (blocks are buttons):
ButX:= [PuzX, PuzX+120, PuzX+240, PuzX, PuzX+120, PuzX+240, PuzX, PuzX+120, PuzX+240];
ButY:= [PuzY, PuzY, PuzY, PuzY+120, PuzY+120, PuzY+120, PuzY+240, PuzY+240, PuzY+240];

if MousePress then
	begin
	MousePress:= false;
	for I:= 0, Buttons-1 do B(I):= false;	\All graphic buttons are released
	BR:= false;
	BSh:= false;
	BE:= false;

	\Determine which button (keytop) it was
	for I:= 0, Buttons-1 do
	    if MousePX>=ButX(I) & MousePX<ButX(I)+120 then
		if MousePY>=ButY(I) & MousePY<ButY(I)+120 then
		    begin
		    B(I):= true;
		    I:= Buttons;	\Exit 'for' loop
		    end;

	if MousePX>=ResetX & MousePX<=ResetX+ResetW then	\Reset button
	    if MousePY>=ResetY & MousePY<=ResetY+ResetH then
		[BR:= true;
		ShowMouse(false);
		MovImage(488, 234+480, ResetX&$fff8, ResetY, 72, ResetH);
		ShowMouse(true)];

	if MousePX>=ShuffleX & MousePX<=ShuffleX+ShuffleW then	\Shuffle button
	    if MousePY>=ShuffleY & MousePY<=ShuffleY+ShuffleH then
		[BSh:= true;
		ShowMouse(false);
		MovImage(488, 280+480, ShuffleX&$fff8, ShuffleY, 72, ShuffleH);
		ShowMouse(true)];

	if MousePX>=ExitX & MousePX<=ExitX+ExitW then		\Exit button
	    if MousePY>=ExitY & MousePY<=ExitY+ExitH then BE:= true;

	\Wait for mouse button release
	loop	begin
		CallInt($33, 6, 0);	\Get left mouse button release info
		if Cpureg(1) \#0\ then	\Left mouse button was released
			begin
			MouseRX:= Cpureg(2);
			MouseRY:= Cpureg(3);
			quit;
			end;
		end;

	ShowMouse(false);		\Show buttons in up position
	MovImage(488, 211+480, ResetX&$fff8, ResetY, 72, ResetH);
	MovImage(488, 257+480, ShuffleX&$fff8, ShuffleY, 72, ShuffleH);
	ShowMouse(true);

	CallInt($33, 5, 0);	\Clear any press info for left mouse button

	\Make sure button pressed is same one that was released
	for I:= 0, Buttons-1 do
		begin
		if B(I) then
			begin
			if MouseRX<ButX(I) ! MouseRX>=ButX(I)+120 then B(I):= false;
			if MouseRY<ButY(I) ! MouseRY>=ButY(I)+120 then B(I):= false;
			end;
		end;

	if BR then		\Reset button
		begin
		if MouseRX<ResetX ! MouseRX>ResetX+ResetW then BR:= false;
		if MouseRY<ResetY ! MouseRY>ResetY+ResetH then BR:= false;
		end;

	if BSh then		\Shuffle button
		begin
		if MouseRX<ShuffleX ! MouseRX>ShuffleX+ShuffleW then BSh:= false;
		if MouseRY<ShuffleY ! MouseRY>ShuffleY+ShuffleH then BSh:= false;
		end;

	if BE then		\Exit button
		begin
		if MouseRX<ExitX ! MouseRX>ExitX+ExitW then BE:= false;
		if MouseRY<ExitY ! MouseRY>ExitY+ExitH then BE:= false;
		end;

	for I:= 0, Buttons-1 do
		begin
		if B(I) then
			begin
			Tx:= [0, 1, 2, 0, 1, 2, 0, 1, 2];
			Ty:= [0, 0, 0, 1, 1, 1, 2, 2, 2];
			SelX:= Tx(I);
			SelY:= Ty(I);
			return MovBlk;
			end;
		end;

	if BR then return Reset;
	if BSh then return Shuffle;
	if BE then Exit;
	end;

if Chkkey then
	begin
	Ch:= GetKey;
	case of
	  Ch>=^1 & Ch<=^9:
		begin
		Tx:= [0, 1, 2, 0, 1, 2, 0, 1, 2];
		Ty:= [2, 2, 2, 1, 1, 1, 0, 0, 0];
		I:= Ch -^1;
		SelX:= Tx(I);
		SelY:= Ty(I);
		return MovBlk;
		end;
	  Ch=^R ! Ch=^r:
		begin
		ShowMouse(false);	\Press and release the Shuffle button
		MovImage(488, 234+480, ResetX&$fff8, ResetY, 72, ResetH);
		ShowMouse(true);
		WaitX(8);
		ShowMouse(false);
		MovImage(488, 211+480, ResetX&$fff8, ResetY, 72, ResetH);
		ShowMouse(true);
		return Reset;
		end;

	  Ch=^S ! Ch=^s:
		begin
		ShowMouse(false);	\Press and release the Shuffle button
		MovImage(488, 280+480, ShuffleX&$fff8, ShuffleY, 72, ShuffleH);
		ShowMouse(true);
		WaitX(8);
		ShowMouse(false);
		MovImage(488, 257+480, ShuffleX&$fff8, ShuffleY, 72, ShuffleH);
		ShowMouse(true);
		return Shuffle;
		end;

	  Ch=^E ! Ch=^e ! Ch=Esc: Exit
	other	Click
	end;
return (-1);
end;	\GetCmd



proc	DoReset;
int	I, J;
begin
for J:= 0, PuzSize-1 do			\Set up Puz in initial position
    for I:= 0, PuzSize-1 do
	Puz(I,J):= Puz0(I,J);
HoleX:= 2;   HoleY:= 0;			\Hole is in upper-right corner

Count:= 0;
ShowMoveCount;
Openi(0);
ClearMouse;	
end;	\DoReset



proc	LoadNReset;	\Load and reset
begin
ShowMouse(false);
LoadLBM("PUZZLE1.LBM");
ShowLowCount;
ShowMouse(true);
DoReset;
end;	\LoadNReset

\----------------------------------------------------------------------

begin	\Main
Cpureg:= Getreg;
PSPSeg:= Cpureg(11);
DataSeg:= Cpureg(12);

Puz:= Reserve(PuzSize*Intsize);
for II:= 0, PuzSize-1 do
	Puz(II):= Reserve(PuzSize*Intsize);

\Assign numbers to blocks in the starting position (0 = the hole):
Puz0:=[	[2, 5, 8],		\This is twisted; X is the first dimension
	[1, 4, 7],
	[0, 3, 6] ];

FontTable:= Reserve(FontSize);
Palette:= Reserve(16);
BlkColRegs:= Reserve(ColorSize*3);

BlkImage:= Malloc(120/8*120*4/16 +1);

II:= Equip;				\Note WaitVB also requires color display
if (II(2)&$08) = 0 then
	[Text(0, "

This program requires a VGA display.
");   exit];

Text(0, "Loading... ");

Trapc(true);				\Ctrl+C will not exit cleanly

Mode:= GetVid;
LoadFont;		\Set up font immediately after setting display mode
Setvid($12);		\640x480x16

\Set up font; i.e. set up interrupt vector $43
CallInt($10, $1121, 0,
	9,		\Bytes per char
	30,		\(this has no effect)
	FontTable, 0, DataSeg);

CallInt($10, $1201, $36);		\Disable video display
WaitVB;

LoadLBM("PUZZLE2.LBM");			\Load background patterns into 2nd page
MovImage(0, 0, 0, 480, 640, VGA2YMax);

LoadLBM("PUZZLE1.LBM");

AlignPalette;
GetBlkColRegs(BlkColRegs);
for II:= 0, ColorSize-1 do SetColReg(II, 0, 0, 0);    

WaitVB;
CallInt($10, $1200, $36);		\Reenable video display

LEDSize:= ImageSize(0, 0, 6, 6, $0f);	\Set up LEDs
LEDOff:= Malloc(LEDSize);
LEDRed:= Malloc(LEDSize);
LEDGrn:= Malloc(LEDSize);
SaveImage(0, 164+480, 6, 164+6+480, $0f, LEDOff);
SaveImage(0, 171+480, 6, 171+6+480, $0f, LEDRed);
SaveImage(0, 178+480, 6, 178+6+480, $0f, LEDGrn);

SetupMouse;

FadeIn(0, BlkColRegs);

LowCount:= 1000;
DoSound:= true;
DoReset;

loop	begin
	Search;		\Search for solution until player makes move
	if PuzSolved then
		begin
		end
	else	begin
		case GetCmd of
		  Reset:
			begin
			LoadNReset;
			end;
		  Shuffle:
			begin
			MixBlksUp;
			end;
		  MovBlk:
			begin
			if SelX=HoleX & SelY#HoleY ! SelY=HoleY & SelX#HoleX then
				begin
				DoMove(2);
				if PuzSolved then Won;
				end
			else	Click;
			end
		other	Click;
		end;
	end;
end;	\Main
