	page	55,136
;IMAGE.ASM	JUL-20-95
;External Assembly Language Routines to Save and Draw Images
; by Loren Blaney
;Based on code in "Programmer's Guide to PC & PS/2 Video Systems"
; by Richard Wilton (first edition, not the second).
;
;These routines destroy all registers except ds, es and sp.

cseg	segment dword public 'code'
	assume cs:cseg

	public	SaveImage, SaveMask, DrawImage, ImageSize

bytesPerLine	equ	80	;number of bytes in one horizontal scan line

;----------------------------------------------------------------------
;Save an image into memory. The area and bit planes are specified. This
;  works for video modes 0eh-12h. (If mode 10h, EGA card must have at least
; 128K of video RAM.)
; SaveImage(X0, Y0, X1, Y1, Z, BufSeg);
;
;A separate entry point to this routine saves a mask for an image. A mask is
; required to remove the background before writing an image with DrawImage.
; To remove the background, use the AND operation, and to write the image
; use the OR operation. For example:
;	SaveImage(X0, Y0, X1, Y1, Z, ImageBuf);	\Get the image
;	SaveMask(X0, Y0, X1, Y1, Z, MaskBuf);	\Get a mask for the image
;	DrawImage(X, Y, MaskBuf, 1);		\1 = AND operation
;	DrawImage(X, Y, ImageBuf, 2);		\2 = OR operation
;
argX0	equ	word ptr [bp]+14 ;coordinates of upper-left corner of
argY0	equ	word ptr [bp]+12 ; rectangular area to save
argX1	equ	word ptr [bp]+10 ;lower-right corner
argY1	equ	word ptr [bp]+8
argZ	equ	word ptr [bp]+6	 ;bit array indicating planes to save
				 ; ($0f saves all 4 planes: 0, 1, 2 & 3)
bufSeg	equ	word ptr [bp]+4	 ;segment address of RAM location

height	equ	word ptr [bp]-2	 ;height of image in pixels (= y1 -y0 +1)
byteWidth equ	word ptr [bp]-4	 ;approx no. of bytes in a row for image

SaveImage:
	mov	ax,0005h	;5 = index = mode register, set read mode 0
	jmp	short si00

SaveMask:
	mov	ax,0805h	;set read mode 1 (color compare reg = 0)
si00:	mov	dx,3ceh		;graphics controller registers
	out	dx,ax

	mov	bp,sp
	sub	sp,4		;reserve space for local variables
	push	es		;save registers
	push	ds

;compute dimensions of bit block
	mov	ax,argx1
	sub	ax,argx0
	mov	cx,0ff07h	;ch = unshifted bit mask; cl = AND mask for al
	and	cl,al		;cl = num of pixels in left-most byte of image-1
	xor	cl,7		;cl = number of bits to shift
	shl	ch,cl		;ch = bit mask for last byte of row
	mov	cl,ch
	push	cx

	mov	cl,3		;convert width in pixels to width in bytes
	shr	ax,cl		;width/8 +1 = (width+7)/8
	inc	ax
	push	ax

	mov	ax,argy1	;calculate number of pixel rows (height)
	sub	ax,argy0
	inc	ax
	push	ax

;establish addressing
	mov	ax,argy0
	mov	bx,argx0
	call	pixAddr
	xor	cl,7		;number of bits to shift left
	push	es
	pop	ds
	mov	si,bx		;ds:si -> video memory

	mov	ax,bufSeg	;es:di -> buffer in RAM
	mov	es,ax
	sub	di,di

;first 16 bytes of the output image data is the header:
; height in pixels
; width in bytes
; bit mask for last byte of row
; bit array indicating planes used
	pop	ax
	mov	height,ax
	cld
	stosw			;bytes 0-1 = number of pixel rows
	pop	ax
	mov	byteWidth,ax
	stosw			;bytes 2-3 = bytes per pixel row; es:[di++] <- ax
	pop	ax
	mov	ch,al		;bit mask for last byte in row
	stosb			;byte 4 = bit mask for last byte
	inc	di		;align
	mov	ax,argZ		;planes used
	stosw
	add	di,16-8		;skip unused bytes in header

;set up graphics controller
	mov	dx,3ceh		;graphics controller registers
	mov	ax,0304h	;index = 4 = read map select register
				;set up for bit plane 3
;copy from video memory to system RAM
si10:	test	argZ,08h	;is this bit plane used?
	je	si40		;jump if not
	out	dx,ax		;select next memory map to read
	push	ax
	push	height
	push	si		;save offset of x0,y0

si20:	mov	bx,byteWidth
	push	si		;save si at start of pixel row

si30:	lodsw			;al = next byte in video memory; ax <- ds:[si++]
				; ah = (next byte) + 1
	dec	si		;ds:si -> (next byte) +1
	rol	ax,cl		;al = next 4 pixels in row
	stosb			;copy to system RAM; es:[di++] <- al
	dec	bx		;loop across row
	jnz	si30

	and	es:[di]-1,ch	;mask last byte in row
	pop	si		;ds:si -> start of row
	add	si,bytesPerLine	;ds:si -> start of next row

	dec	height
	jnz	si20		;loop down rows

	call	newAd1		;calculate new segment address
	pop	si		;ds:si -> start of bit block
	pop	height		;restore number of pixel rows
	pop	ax		;ah = last map read
				;al = read map select register number
si40:	shl	argZ,1		;next bit plane
	dec	ah
	jns	si10		;loop

	mov	ax,0005h	;5 = index = mode register, set read mode 0
	mov	dx,3ceh		;graphics controller registers
	out	dx,ax

	pop	ds		;restore registers
	pop	es
	mov	sp,bp
	retf	12		;discard arguments and return

;----------------------------------------------------------------------
;Output an image to the screen. Works for video modes 0eh-12h. (If mode 10h,
; EGA card must have at least 128K of video RAM.)
; DrawImage(X, Y, BufSeg, Op);
;
argx	equ	word ptr [bp]+10	;upper-left corner of image is put here
argy	equ	word ptr [bp]+8
bufSeg	equ	word ptr [bp]+6		;segment address of RAM containing image
op	equ	word ptr [bp]+4		;operation: 0=copy, 1=and, 2=or, 3=xor

height		equ	word ptr [bp]-2	;height of image in pixels
byteWidth	equ	word ptr [bp]-4	;width of image in bytes
rowCounter	equ	word ptr [bp]-6
startMask	equ	word ptr [bp]-8
endMaskL	equ	word ptr [bp]-10
endMaskR	equ	word ptr [bp]-12
planes		equ	byte ptr [bp]-14 ;bit array indicating planes used

DrawImage:
	mov	bp,sp
	sub	sp,14		;reserve space for local variables
	push	es		;save registers
	push	ds

;establish addressing
	mov	ax,argy
	mov	bx,argx
	call	pixAddr		;es:bx -> byte offset of x,y
	inc	cl
	and	cl,7		;number of bits to shift left
	mov	di,bx		;es:di -> X,Y in video memory

	mov	ax,bufSeg	;ds:si -> buffer in system RAM
	mov	ds,ax
	sub	si,si

;get dimensions of bit block from header
	cld
	lodsw			;number of pixel rows; ax <- ds:[si++]
	mov	height,ax
	lodsw			;bytes per pixel row
	mov	byteWidth,ax
	lodsb			;bit mask for last byte in row
	mov	ch,al
	inc	si		;align
	lodsw			;planes used
	mov	planes,al
	add	si,16-8		;skip unused bytes

;set up graphics controller registers
	mov	dx,3ceh		;graphics controller registers
	mov	ax,op		;get operation code (copy / and / or / xor)
	and	al,3		;limit to legal range
	xchg	ah,al		;shift bits into positions 4 and 3
	shl	ah,1
	shl	ah,1
	shl	ah,1
	mov	al,3		;3 = data rotate register
	out	dx,ax

	mov	ax,0805h	;5 = mode register; read mode 1, write mode 0
	out	dx,ax

	mov	ax,0007h	;7 = color do(nt) care, reads return 0FFh
	out	dx,ax

	mov	ax,0ff08h	;8 = bit mask (for safety)
	out	dx,ax

	mov	dl,0c4h		;sequencer I/O port
	mov	ax,0802h	;2 = map mask register:= 1000b

	cmp	cx,0ff00h	;if mask # ffh or bits to shift # 0 then jump
	jne	di15

	cmp	op,0		;is operation a simple copy?
	jne	di10		;jump if not

;very fast routine for copy of byte-aligned bit blocks
di00:	test	ah,planes	;is this bit plane used?
	je	di03		;jump if not
	out	dx,ax		;enable one bit plane for writes
	push	ax		;save map mask value
	push	di		;save video memory offset to upper left corner
	mov	bx,height

di01:	push	di		;save video memory offset to left edge
	mov	cx,byteWidth
	rep movsb		;es:[di++] <- ds:[si++]  cx--

	pop	di		;get pointer to left edge of image
	add	di,bytesPerLine	;es:di -> next pixel row in video memory
	dec	bx
	jnz	di01		;loop down pixel rows

	call	newAd2		;calculate new segment address
	pop	di		;get video memory offset to upper left corner
	pop	ax		;ah = current map mask register value
di03:	shr	ah,1		;ah = new map mask value
	jnz	di00
	jmp	di90		;exit

;fast routine for byte-aligned bit blocks
di10:	test	ah,planes	;is this bit plane used?
	je	di13		;jump if not
	out	dx,ax		;enable one bit plane for writes
	push	ax		;save map mask value
	push	di		;save video memory offset to upper left corner
	mov	bx,height

di11:	push	di		;save video memory offset to left edge
	mov	cx,byteWidth
di12:	lodsb			;al <- ds:[si++]
	and	es:[di],al	;read, modify, write bit-plane memory
	inc	di		;di++
	loop	di12		;cx--

	pop	di		;get pointer to left edge of image
	add	di,bytesPerLine	;es:di -> next pixel row in video memory
	dec	bx
	jnz	di11		;loop down pixel rows

	call	newAd2		;calculate new segment address
	pop	di		;get video memory offset to upper left corner
	pop	ax		;ah = current map mask register value
di13:	shr	ah,1		;ah = new map mask value
	jnz	di10
	jmp	di90		;exit

;routine for non-aligned bit blocks
di15:	push	ax		;save map mask value (0802h)
	mov	bx,000ffh	;bh = 00 = mask for first byte in row
	mov	al,ch		;al = mask for last byte in pixel row
	cbw			;ah = 0ffh = mask for last byte -1

	cmp	byteWidth,1
	jne	di16		;jump if more than one byte per row

	mov	bl,ch
	mov	ah,ch		;ah = mask for last-1 byte
	sub	al,al		;al = 0 = mask for last byte

di16:	shl	ax,cl		;shift masks into position
	shl	bx,cl

	mov	bl,al		;save masks
	mov	al,8		;8 = bit mask register
	mov	endMaskL,ax
	mov	ah,bl
	mov	endMaskR,ax
	mov	ah,bh
	mov	startMask,ax

	mov	bx,byteWidth
	pop	ax		;restore map mask register value (0802h)

;set pixels row-by-row in the bit planes
di17:	test	ah,planes	;is this bit plane used?
	je	di30		;jump if not
	out	dx,ax		;enable one bit plane for writes
	push	ax		;save map mask value
	push	di		;save video memory offset
	mov	dl,0ceh		;dx = 3ceh = graphics controller port

	mov	ax,height	;init loop counter
	mov	rowCounter,ax

;set pixels in start of row in currently enabled bit plane
di18:	push	di		;save offset of start of pixel row
	push	si		;save offset of row in bit block
	push	bx		;save bytes per pixel row

	mov	ax,startMask
	out	dx,ax

	lodsw			;two bytes of pixels; ax <- ds:[si++]
	dec	si		;ds:si -> second byte of pixels
	test	cl,cl
	jnz	di19		;jump if not left-aligned

	dec	bx		;bytes per row -1
	jnz	di20		;jump if at least 2 bytes per row
	jmp	short di22	;jump if only one byte per row

di19:	rol	ax,cl		;ah = left part of first byte and
				; right part of second byte
				;al = right part of first byte and
				; left part of second byte
	and	es:[di],ah	;set pixels for left part of first byte
	inc	di
	dec	bx		;bx = bytes per row -2

di20:	push	ax		;save pixels
	mov	ax,0ff08h	;set bit mask register for succeeding bytes
	out	dx,ax
	pop	ax

	dec	bx
	jng	di22		;jump if only 1 or 2 bytes in pixel row

;set pixels in middle of row
di21:	and	es:[di],al	;set pixels in right part of current byte
	inc	di		; and left part of next byte

	lodsw			;next word of pixels; ax <- ds:[si++]
	dec	si
	rol	ax,cl		;ah = left part of next byte and
				; right part of next+1 byte
				;al = right part of next byte and
	dec	bx		; left part of next+1 byte
	jnz	di21		;loop across pixel row

;set pixels at end of row
di22:	mov	bx,ax		;bh = right part of last byte and
				; left part of last-1 byte
				;bl = left part of last byte and
				; right part of last-1 byte
	mov	ax,endMaskL	;ah = mask for last-1 byte; al = bit mask reg
	out	dx,ax		;set bit mask register
	and	es:[di],bl	;set pixels for last-1 byte

	mov	ax,endMaskR	;mask for last byte in pixel row
	out	dx,ax		;last byte in pixel row
	and	es:[di]+1,bh	;set pixels for last byte

	pop	bx		;bytes per pixel row
	pop	si
	add	si,bx		;ds:si -> next row in bit block
	pop	di
	add	di,bytesPerLine	;es:di -> next pixel row in video memory
	dec	rowCounter
	jnz	di18		;loop down pixel rows

	call	newAd2		;calculate new segment address
	pop	di		;es:di -> video memory offset to X,Y
	pop	ax		;current map mask value
	mov	dl,0c4h		;3c4h = sequencer port
di30:	shr	ah,1		;next map mask value
	jnz	di17

;restore graphics controller and sequencer registers to their default states
di90:	mov	ax,0f02h	;default map mask value
	out	dx,ax

	mov	dl,0ceh		;dx = 3ceh = graphics controller registers
	mov	ax,0003h	;default data rotate/function select (copy)
	out	dx,ax

	mov	ax,0005h	;default mode value
	out	dx,ax

	mov	ax,0f07h	;default color compare value
	out	dx,ax

	mov	ax,0ff08h	;default bit mask value
	out	dx,ax

	pop	ds		;restore registers
	pop	es
	mov	sp,bp
	retf	8		;discard arguments and return

;----------------------------------------------------------------------
;Return offset in video memory for pixel at X,Y
; Inputs:
;	ax = Y coordinate
;	bx = X coordinate
; Outputs:
;	ah = unshifted bit mask (= 01h)
;	bx = byte offset in buffer
;	cl = number of bits to shift 01h left
;	es = video buffer segment address
;
;originOffset	equ	0	;byte offset of (0,0)
videoMemSeg	equ	0a000h	;base segment address of video memory

pixAddr:
	mov	cl,bl		;cl = low-order byte of X coordinate

	push	dx
	mov	dx,bytesPerLine	;ax = Y * bytesPerLine
	mul	dx
	pop	dx

	shr	bx,1		;bx = X / 8
	shr	bx,1
	shr	bx,1
	add	bx,ax
;	add	bx,originOffset

	mov	ax,videoMemSeg
	mov	es,ax		;es:bx = byte address of pixel

	and	cl,7		;X & 7
	xor	cl,7		;number of bits to shift 01h left
	mov	ah,01h		;unshifted bit mask
	ret

;----------------------------------------------------------------------
;Returns the number of paragraphs (16-byte blocks) needed to store an image.
; For video modes 0dh-12h.
; ImageSiz(X0, Y0, X1, Y1, Z);
;
argX0	equ	word ptr [bp]+12	;coordinates of upper-left corner
argY0	equ	word ptr [bp]+10	; of rectangular area to save
argX1	equ	word ptr [bp]+8		;lower-right corner
argY1	equ	word ptr [bp]+6
argZ	equ	word ptr [bp]+4		;bit array indicating planes used

ImageSize:
	mov	bp,sp

; ( (X1-X0+1+7)/8 * (Y1-Y0+1) * Planes + 16 + 15 ) /16
; ( ((X1-X0)/8 +1) * (Y1-Y0+1) * Planes + 15) /16 + 1

	mov	ax,argX1	; ((X1-X0)/8 +1)
	sub	ax,argX0
	shr	ax,1
	shr	ax,1
	shr	ax,1
	inc	ax

	mov	dx,argY1	; * (Y1-Y0+1)
	sub	dx,argY0
	inc	dx
	mul	dx

	mov	bx,ax		; * Planes
	mov	cx,dx
	sub	ax,ax
	sub	dx,dx
	jmp	short is20
is10:	shr	argZ,1
	jnc	is20
	add	ax,bx
	adc	dx,cx
is20:	cmp	argZ,0		;loop until no more planes
	jne	is10

	add	ax,15		; + 15) /16 + 1
	adc	dx,0
	mov	cx,4
is30:	shr	dx,1
	rcr	ax,1
	loop	is30
	inc	ax

	mov	argx0,ax	;replace first argument with size in paragraphs
	retf	8		;discard rest of arguments and return

;----------------------------------------------------------------------
;Calculate new segment address for ES:DI pair
;
newAd1:	push	di		;save offset
	shr	di,1		;convert to segment address
	shr	di,1		;di / 16
	shr	di,1
	shr	di,1
	mov	ax,es		;get the segment address
	add	ax,di		;add to segment address
	mov	es,ax		;store in segment register
	pop	di		;restore offset
	and	di,000fh	;remainder is offset
	ret

;----------------------------------------------------------------------
;Calculate new segment address for DS:SI pair
;
newAd2:	push	si		;save offset
	shr	si,1		;convert to segment address
	shr	si,1		;di / 16
	shr	si,1
	shr	si,1
	mov	ax,ds		;get the segment address
	add	ax,si		;add to segment address
	mov	ds,ax		;store in segment register
	pop	si		;restore offset
	and	si,000fh	;remainder is the offset
	ret

cseg	ends
	end
