unit txtGraph;

interface

uses
	jfunc;

const
	videoCard_mda=0;
	videoCard_cga=1;
	videoCard_pcJr=2;
	videoCard_tandy1000=3;
	videoCard_tandySLTL=4;
	videoCard_ega=5;
	videoCard_vga=6;
	videoCardName:packed array[0..videoCard_vga] of string[11]=(
		'MDA','CGA','PCJr','Tandy 1000','Tandy TL/SL','EGA','VGA'
	);

type
	pTileData=^byte;

	pTileSet=^tileSet;
	tileSet=object
		size:word;
		name:dosName;
		dataStart,
		dataEnd:pTileData;
		constructor init(dataFile:dosName);
		destructor term;
	end;

	pKernSet=^kernSet;
	kernSet=packed array[0..255] of byte;

	writeTileProc=procedure(x,y:integer; tileSet:pointer; tile:byte);
	backgroundProc=procedure(x,y:integer; data:pTileData);

	pfontSet=^fontSet;
	fontSet=object(tileSet)
		glyphStart,
		glyphCount,
		glyphSize:word;
		writeTile:writeTileProc;
		kernData:pKernSet;
		constructor init(dataFile:dosName);
		function textLength(st:st40):word;
		procedure outText(x,y:word; st:st40);
		procedure outTextRight(x,y:word; st:st40);
		procedure outTextCentered(x,y:word; st:st40);
		destructor term;
	end;

	pNumberSet=^numberSet;
	numberSet=object(tileSet)
		procedure outDecNonPadded(x,y:word; n:longint; colorSet:byte);
		procedure outDecimal(x,y:word; n:longint; colorSet:byte);
		procedure outHex(x,y:word; n:longint; colorSet:byte);
	end;

	pSprite=^Sprite;
	sprite=object
		{ these values the user can change directly }
		currentX,currentY,currentTile:integer;

		{ from here out this should be left to the code to manage }
		oldX,oldY,sTileSize:integer;
		sTiles:pTileSet;
		storeSize:word;
		store:pTileData;

		{ function handlers, prevents needing a 'case' statement inside the loop }
		writeTile:writeTileProc;
	  getBackground,
	  restoreBackground:backgroundProc;

	  { pointers for list control }
		prev,next:pSprite;

		constructor init(parent:pSprite; x,y,tile,tileSize:integer; tiles:pTileSet);
		procedure show;
		procedure hide;
		destructor term;
	end;

var
	videoCard:word;

procedure tg_init;
procedure tg_term;

procedure tg_clear(color:byte);

procedure tg_putPixel(x,y:word; color:byte);
procedure tgClip_putPixel(x,y:integer; color:byte);

function tg_getPixel(x,y:word):byte;
function tgClip_getPixel(x,y:integer):byte;

procedure tg_bar(sx,sy,ex,ey:integer; color:byte);
procedure tgClip_bar(sx,sy,ex,ey:integer; color:byte);

procedure tg_rectangle(sx,sy,ex,ey:integer; color:byte);
procedure tgClip_rectangle(sx,sy,ex,ey:integer; color:byte);

procedure tg_line(x1,y1,x2,y2:integer; color:byte);

procedure tg_circle(xc,yc,r:integer; color:byte);
procedure tg_ball(xc,yc,r:integer; color:byte);

procedure tg_Negative;
procedure tg_HardWrite(start,value,count:word);

procedure tg_tile3(x,y:integer; tileSet:pointer; tile:byte);
procedure tg_tile5(x,y:integer; tileSet:pointer; tile:byte);
procedure tg_tile7(x,y:integer; tileSet:pointer; tile:byte);

procedure tg_getBackground3(x,y:integer; data:pTileData);
procedure tg_getBackground5(x,y:integer; data:pTileData);
procedure tg_getBackground7(x,y:integer; data:pTileData);

procedure tg_restoreBackground3(x,y:integer; data:pTileData);
procedure tg_restoreBackground5(x,y:integer; data:pTileData);
procedure tg_restoreBackground7(x,y:integer; data:pTileData);

procedure tg_waitRetrace;

function tg_addSprite(x,y,tile,tileSize:integer; tiles:pTileSet):pSprite;
procedure tg_eraseUnderSprite(sx,sy,ex,ey:integer);
procedure tg_showSprites;
procedure tg_hideSprites;
procedure tg_updateSprites;

implementation

type
	regData=packed record
		register:word;
		data:string[26];
	end;

{
	Why store regData.data in a string? Because it gives
	us a size byte BEFORE the content in the data stream,
	and it's the only dynamic size data type you can use
	in a constant.

	Too bad we can't just read this crap from the EGA/VGA
	data tables, but for some reason DosBox doesn't provide
	those... and it would be nice to have this work on real
	hardware AND dosBox.
}

const
	pageSize=$3E80;
	pageSizeWords=pageSize shr 1;

	CGA_Regs:regData=(
		register:$3D4;
		data:#04+#$7F + #06+#$64 + #07+#$70 + #09+#$01
	);

	EGA_Sequencer:regData=(
		register:$3C4;
		data:#03+#01+#03+#00+#03
	);
	EGA_CRTC:regData=(
		register:$3D4;
		data:
			#$70+#$4F+#$5C+#$2F + #$5F+#$07+#$04+#$11 +
			#$00+#$01+#$06+#$07 + #$00+#$00+#$00+#$00 +
			#$E1+#$24+#$C7+#$28 + #$08+#$E0+#$F0+#$A3 +
			#$FF+#$00
	);

	VGA_Sequencer:regData=(
		register:$3C4;
		data:#03+#01+#03+#00+#02
	);
	VGA_CRTC:regData=(
		register:$3D4;
		data:
			#$5F+#$4F+#$50+#$82 + #$55+#$81+#$BF+#$1F +
			#$00+#$43+#$06+#$07 + #$00+#$00+#$00+#$59 +
			#$9C+#$8E+#$8F+#$28 + #$1F+#$96+#$B9+#$A3 +
			#$FF
	);

type
	tgWindow=record
		sx,sy,ex,ey:integer;
	end;

var
	firstSprite,lastSprite:pSprite;
	textSegment:word;
	oldMode:byte;
	oldTgExitProc:pointer;

	textGraphOn:boolean;
	clippingWindow:tgWindow;

	tg_erase:boolean;
	tg_sx,tg_sy,tg_ex,tg_ey:integer;

constructor sprite.init(parent:pSprite; x,y,tile,tileSize:integer; tiles:pTileSet);
begin
	currentX:=x;
	oldX:=x;
	currentY:=y;
	oldY:=y;
	currentTile:=tile;
	sTiles:=tiles;
	sTileSize:=tileSize;
	case sTileSize of
		3:begin
			storeSize:=12;
			writeTile:=tg_tile3;
		  getBackground:=tg_getBackground3;
		  restoreBackground:=tg_restoreBackground3;
		end;
		5:begin
			storeSize:=30;
			writeTile:=tg_tile5;
		  getBackground:=tg_getBackground5;
		  restoreBackground:=tg_restoreBackground5;
		end;
		7:begin
			storesize:=56;
			writeTile:=tg_tile7;
		  getBackground:=tg_getBackground7;
		  restoreBackground:=tg_restoreBackground7;
		end;
	end;
	getmem(store,storeSize);
	prev:=parent;
	next:=nil;
end;

procedure sprite.show;
begin
	getBackground(currentX,currentY,store);
	writeTile(currentX,currentY,sTiles^.dataStart,currentTile);
	oldX:=currentX;
	oldY:=currentY;
end;

procedure sprite.hide;
begin
	restoreBackground(oldX,oldY,store);
end;

destructor sprite.term;
begin
	freemem(store,storeSize);
end;

function tg_addSprite(x,y,tile,tileSize:integer; tiles:pTileSet):pSprite;
begin
	if firstSprite=nil then begin
		new(firstSprite,init(nil,x,y,tile,tileSize,tiles));
		lastSprite:=firstSprite;
	end else begin
		new(lastSprite^.next,init(lastSprite,x,y,tile,tileSize,tiles));
		lastSprite:=lastSprite^.next;
	end;
	tg_addSprite:=lastSprite;
end;

procedure tg_showSprites;
var
	tSprite:pSprite;
begin
	tSprite:=firstSprite;
	while not(tSprite=nil) do begin
		tSprite^.show;
		tSprite:=tSprite^.next;
	end;
end;

procedure tg_hideSprites;
var
	tSprite:pSprite;
begin
	tSprite:=lastSprite;
	while not(tSprite=nil) do begin
		tSprite^.hide;
		tSprite:=tSprite^.prev;
	end;
end;

procedure tg_updateSprites;
begin
	tg_hideSprites;
	if (tg_erase) then tg_bar(tg_sx,tg_sy,tg_ex,tg_ey,0);
	tg_showSprites;
end;

procedure tg_eraseUnderSprite(sx,sy,ex,ey:integer);
begin
	tg_erase:=true;
	tg_sx:=sx;
	tg_sy:=sy;
	tg_ex:=ex;
	tg_ey:=ey;
end;

procedure discardSprites;
begin
	while not(lastSprite=nil) do begin
		firstSprite:=lastSprite^.prev;
		dispose(lastSprite,term);
		lastSprite:=firstSprite;
	end;
end;

constructor tileSet.init(dataFile:dosName);
var
	f:file;
begin
	name:=datafile;
	assign(f,name+'.DAT');
	reset(f,1);
	size:=filesize(f);
	getmem(dataStart,size);
	dataEnd:=dataStart;
	inc(dataEnd,size);
	blockread(f,dataStart^,size);
	close(f);
end;

destructor tileSet.term;
begin
	freemem(dataStart,size);
end;

constructor fontSet.init(dataFile:dosName);
var
	f:file;
	p,e:pTileData;
begin
	tileSet.init(dataFile);
	assign(f,name+'.KRN');
	reset(f,1);
	blockread(f,glyphStart,2);
	blockread(f,glyphCount,2);
	blockread(f,glyphSize,2);
	getmem(kernData,glyphCount);
	blockread(f,kernData^,glyphCount);
	close(f);
	case glyphSize of
		3:writeTile:=tg_tile3;
		5:writeTile:=tg_tile5;
		7:writeTile:=tg_tile7;
	end;
end;

function fontSet.textLength(st:st40):word;
var
	t,sx:word;
	b:byte;
begin
	sx:=0;
	t:=0;
	while (t<length(st)) do begin
		inc(t);
		b:=ord(st[t])-glyphStart;
		{ remap high values into end of table }
		while (b>glyphCount) do b:=b-$20;
		sx:=sx+kernData^[b];
	end;
	textLength:=sx;
end;

procedure fontSet.outText(x,y:word; st:st40);
var
	sx,t:word;
	b:byte;
begin
	sx:=x;
	t:=0;
	while (t<length(st)) do begin
		inc(t);
		b:=ord(st[t])-GlyphStart;
		{ remap high values into end of table }
		while (b>=glyphCount) do b:=b-$20;
		writeTile(sx,y,dataStart,b);
		sx:=sx+kernData^[b];
	end;
end;

procedure fontSet.outTextCentered(x,y:word; st:st40);
var
	sx:word;
begin
	sx:=x-textLength(st) div 2;
	outText(sx,y,st);
end;

procedure fontSet.outTextRight(x,y:word; st:st40);
var
	sx:word;
begin
	sx:=x-textLength(st);
	outText(sx,y,st);
end;

destructor fontSet.term;
begin
	tileSet.term;
	freemem(kernData,glyphCount);
end;

procedure numberSet.outDecNonPadded(x,y:word; n:longint; colorSet:byte);
var
	sx:word;
	d:longint;
	b:byte;
	started:boolean;
begin
	d:=10000000;
	sx:=x;
	started:=false;
	while (d>0) do begin
		b:=((n div d) mod 10);
		d:=d div 10;
		if (b>0) or started then begin
			tg_tile5(sx,y,dataStart,b+colorSet*16);
			inc(sx,4);
			started:=true;
		end;
	end;
end;


procedure numberSet.outDecimal(x,y:word; n:longint; colorSet:byte);
var
	sx:word;
	d:longint;
	b:byte;
begin
	d:=10000000;
	sx:=x;
	while (d>0) do begin
		b:=((n div d) mod 10)+colorSet*16;
		d:=d div 10;
		tg_tile5(sx,y,dataStart,b);
		inc(sx,4);
	end;
end;

procedure numberSet.outHex(x,y:word; n:longint; colorSet:byte);
var
	sx:word;
	d:longint;
	b:byte;
begin
	d:=$10000000;
	sx:=x;
	while (d>0) do begin
		b:=((n div d) and $0F)+colorSet*16;
		d:=d div $10;
		tg_tile5(sx,y,dataStart,b);
		inc(sx,4);
	end;
end;

procedure tg_waitRetrace; assembler;
asm
	mov  dx,$3DA
	mov  ah,$08
@loop:
	in   al,dx
	and  al,ah
	jz   @loop
end;

procedure outCGAPort(data:pointer); assembler;
asm
	les  di,data
	mov  dx,es:[di]
	add  di,2
	xor  ch,ch
	mov  cl,es:[di]
	shr  cl,1
	inc  di
@loopRegData:
	mov  ax,es:[di]
	out  dx,ax
	add  di,2
	loop @loopRegData
end;

procedure outEGAVGAPort(data:pointer); assembler;
asm
	les  di,data
	mov  dx,es:[di]
	add  di,2
	xor  cx,cx
	mov  cl,es:[di]
	inc  di
	xor  al,al
@loopEGAVGA:
	mov  ah,es:[di]
	out  dx,ax
	inc  di
	inc  al
	loop @loopEGAVGA
end;

procedure disableVideoOutputAndInterrupts; assembler;
asm
	{ video off first }
	mov  dx,$03D8
	mov  al,1
	out  dx,al
	{ disable normal interrupts }
	cli
	{ disable NMI too -- can't have CRTC interuppted! }
	in   al,$70
	and  al,$7F
	out  $70,al
end;

procedure enableVideoOutputAndInterrupts; assembler;
asm
	{ enable NMI }
	in   al,$70
	or   al,$80
	out  $70,al
	{ enable normal interrupts }
	sti
	{ video on }
	mov  dx,$03D8
	mov  al,9
	out  dx,al
end;

procedure load8x8FontAndDisableBlink; assembler;
asm
	mov ax,$1102 { load 8x8 font }
	xor bl,bl
	int $10
	mov ax,$1003 { disable Blink }
	xor bl,bl
	int $10
end;

{
	Detecting which video card is present is kinda tricky...
	but thankfully they did something smart with int $10.
	Calls to unknown subfunctions just RET leaving registers
	intact, so if you call a VGA function that you know changes
	a register, and the register doesn't change, it's not a VGA.
	Call a EGA function ditto, ditto... finally check if we're in
	a monochrome display mode, that's MDA.

	Unfortunately there's no known reliable check for a CGA since
	newer cards pretend to be one -- but if we eliminate
	'everything else' from the list, it must be CGA.
}

function detectCard:byte; assembler;
asm
	mov  ax,$1200
	mov  bl,$32       { VGA only enable video }
	int  $10
	cmp  al,$12       { VGA returns $12, all others leave it unmodified! }
	jne  @notVGA      { not a vga, test for EGA }
	mov  al,videoCard_vga
	ret
@notVGA:           { We eliminated VGA, so an EGA/VGA true must be EGA }
	mov  ah,$12
	mov  bl,$10       { EGA/VGA get configuration info }
	int  $10
	and  bl,$03       { EGA/VGA returns a 0..3 value here }
	jz   @notEGA      { not a VGA, test for MDA }
	mov  al,videoCard_ega
	ret
@notEGA:            { MDA all we need to detect is video mode 7 }
	mov  ah,$0F       { get Video mode }
	int  $10
	cmp  al,$07
	jne  @notMDA
	mov  al,videoCard_mda
	ret
@notMDA:            { not MDA, check for Jr. }
	mov  ax,$FFFF
	mov  es,ax
	mov  di,$000E     { second to last byte PCjr/Tandy BIOS info area }
	mov  al,$FD       { ends up $0F only on the Jr. }
	cmp  es:[di],al
	jne  @notJr
	mov  al,videoCard_pcJr
	ret
@notJr:             { not junior, test for tandy }
	mov  al,$FF       { all tandy's return $FF here }
	cmp  es:[di],al
	jne  @notTandy
	mov  ax,$FC00
	mov  es,ax
	xor  di,di
	mov  al,$21
	cmp  es:[di],al
	jne  @notTandy
	mov  ah,$C0       { test for SL/TL }
	int  $15          { Get System Environment }
	jnc  @tandySLTL     { early Tandy's leave the carry bit set, TL/SL does not }
	mov  al,videoCard_tandy1000
	ret
@tandySLTL:
	mov  al,videoCard_tandySLTL
	ret
@notTandy:
	mov  al,videoCard_cga { all other cards eliminated, must be CGA }
end;

function getVideoMode:byte; assembler;
asm
	mov  ax,$0F00;
	int  $10;
end;

procedure setVideoMode(mode:byte); assembler;
asm
	xor ah,ah
	mov al,mode
	int $10;
end;

procedure tg_putPixel(x,y:word; color:byte); assembler;
asm
	mov  ax,y
	mov  di,ax
	mov  cl,7
	shl  di,cl
	mov  cl,5
	shl  ax,cl
	add  di,ax

	mov  dx,x
	add  di,dx
	or   di,1

	mov  al,color
	mov  bx,textSegment
	mov  es,bx
	mov  bl,es:[di]

	and  dx,1
	jnz  @xOdd

	and  bl,$F0
	jmp  @xDone

@xOdd:
	and  bl,$0F
	mov  cl,4
	shl  al,cl

@xDone:
	or   al,bl
	mov  es:[di],al
end;

function tg_getPixel(x,y:word):byte; assembler;
asm
	mov  ax,y
	mov  di,ax
	mov  cl,7
	shl  di,cl
	mov  cl,5
	shl  ax,cl
	add  di,ax

	mov  dx,x
	add  di,dx
	or   di,1

	mov  bx,textSegment
	mov  es,bx
	mov  al,es:[di]

	and  dx,$01
	jz   @xEven

	mov  cl,4
	shl  al,cl

@xEven:
	and  al,$0F
end;

procedure hLine(x,y,disX:integer; color:byte); assembler;
asm
	mov  ax,y
	mov  di,ax
	mov  cl,7
	shl  di,cl
	mov  cl,5
	shl  ax,cl
	add  di,ax

	mov  bx,x
	add  di,bx
	or   di,1

	mov  ax,textSegment
	mov  es,ax

	mov  dx,disX

	mov  bh,color
	mov  al,bh

	mov  cl,4
	shl  al,cl

	and  bl,1
	jz   @hLongRun

	mov  bl,es:[di]
	and  bl,$0F
	or   bl,al
	mov  es:[di],bl
	add  di,2
	dec  dx

@hLongRun:
	cmp  dx,2
	jl   @hFinalPixel
	mov  ah,al
	or   ah,bh
	mov  al,$DD
	mov  cx,dx
	shr  cx,1
	dec  di
	rep  stosw
	inc  di

@hFinalPixel:
	test dx,1
	jz   @hLineDone
	mov  al,es:[di]
	and  al,$F0
	or   al,bh
	mov  es:[di],al
@hLineDone:
end;

procedure vLine(x,y,disY:word; color:byte); assembler;
asm
	mov  ax,y
	mov  bx,160
	mul  bx
	mov  di,ax

	mov  dx,x
	add  di,dx
	or   di,1

	mov  ax,textSegment
	mov  es,ax

	mov  bl,color
	mov  bh,$F0 {
		It's faster to set BH and then change it on condition
		than it is to add all sorts of test and jump logic to
		only set $F0 as needed
	}

	and  dx,1
	jz   @vSetDistance

	mov  cl,4
	shl  bl,cl
	mov  bh,$0F

@vSetDistance:
	mov  cx,disY

@vLineLoop:
	mov  al,es:[di]
	and  al,bh
	or   al,bl
	mov  es:[di],al
	add  di,$A0
	loop @vLineLoop
end;


function inWindow(x,y:integer):boolean;
begin
	inWindow:=(
		(x>=clippingWindow.sx) and
		(x<=clippingWindow.ex) and
		(y>=clippingWindow.sy) and
		(y<=clippingWindow.ey)
	);
end;

procedure tgClip_putPixel(x,y:integer; color:byte);
begin
	if inWindow(x,y) then tg_putPixel(x,y,color);
end;

function tgClip_getPixel(x,y:integer):byte;
begin
	if inWindow(x,y) then begin
		tgClip_getPixel:=tg_getPixel(x,y);
	end else tgClip_getPixel:=0;
end;

procedure tg_bar(sx,sy,ex,ey:integer; color:byte);
var
	y,distance:integer;
begin
	distance:=(ex-sx);
	if (distance=0) then begin
		vLine(sx,sy,(ey-sy)+1,color);
	end else begin
		y:=sy;
		inc(distance);
		while (y<=ey) do begin
			hLine(sx,y,distance,color);
			inc(y);
		end;
	end;
end;

procedure tgClip_bar(sx,sy,ex,ey:integer; color:byte);
var
	csx,cex,csy,cey:integer;
begin
	if (sx<ex) then begin
		csx:=sx;
		cex:=ex;
	end else begin
		csx:=ex;
		cex:=sx;
	end;
	if (csx<=clippingWindow.ex) and (cex>=clippingWindow.sx) then begin
		if (sy<ey) then begin
			csy:=sy;
			cey:=ey;
		end else begin
			csy:=ey;
			cey:=sy;
		end;
		if (csy<=clippingWindow.ey) and (cey>=clippingWindow.sy) then begin
			if (csx<clippingWindow.sx) then csx:=clippingWindow.sx;
			if (cex>clippingWindow.ex) then cex:=clippingWindow.ex;
			if (csy<clippingWindow.sy) then csy:=clippingWindow.sy;
			if (cey>clippingWindow.ey) then cey:=clippingWindow.ey;
			tg_bar(csx,csy,cex,cey,color);
		end;
	end;
end;

procedure tg_rectangle(sx,sy,ex,ey:integer; color:byte);
var
	distance:integer;
begin
	distance:=(ex-sx)+1;
	hLine(sx,sy,distance,color);
	hLine(sx,ey,distance,color);
	distance:=(ey-sy)+1;
	vLine(sx,sy,distance,color);
	vLine(ex,sy,distance,color);
end;

procedure tgClip_Rectangle(sx,sy,ex,ey:integer; color:byte);
var
	csx,cex,csy,cey,distance:integer;
	drawLeft,drawRight,drawTop,drawBottom:boolean;
begin
	if (sx<ex) then begin
		csx:=sx;
		cex:=ex;
	end else begin
		csx:=ex;
		cex:=sx;
	end;
	if (sy<ey) then begin
		csy:=sy;
		cey:=ey;
	end else begin
		csy:=ey;
		cey:=sy;
	end;
	if (csx<clippingWindow.sx) then begin
		csx:=clippingWindow.sx;
		drawLeft:=false;
	end else drawLeft:=true;
	if (cex>=clippingWindow.ex) then begin
		cex:=clippingWindow.ex;
		drawRight:=false;
	end else drawRight:=true;
	if (csy<clippingWindow.sy) then begin
		csy:=clippingWindow.sy;
		drawTop:=false;
	end else drawTop:=true;
	if (cey>=clippingWindow.ey) then begin
		cey:=clippingWindow.ey;
		drawBottom:=false;
	end else drawBottom:=true;
	if (
		(csx<=clippingWindow.ex) and
		(csy<=clippingWindow.ey) and
		(cex>=clippingWindow.sx) and
		(cey>=clippingWindow.sy)
	) then begin
		distance:=(cex-csx)+1;
		if drawTop then hline(csx,csy,distance,color);
		if drawBottom then hline(csx,cey,distance,color);
		distance:=(cey-csy)+1;
		if drawLeft then vline(csx,csy,distance,color);
		if drawRight then vline(cex,csy,distance,color);
	end;
end;

procedure tg_negative; assembler;
asm
	mov  ax,textSegment
	mov  es,ax
	mov  di,1
	mov  cx,pageSize
@loopNegative:
	mov  al,es:[di]
	not  al
	mov  es:[di],al
	add  di,2
	loop @loopNegative;
end;

procedure lineDraw(x1,y1,x2,y2:integer; color:byte);
var
  dx,dy,incr1,incr2,d,x,y,xend,yend,xinc,yinc:integer;
begin
  dx:=abs(x2-x1);
  dy:=abs(y2-y1);
  if (dx>=dy) then begin
    if (x1>x2) then begin
      x:=x2;  y:=y2;  xend:=x1;
      if (y2>y1) then yinc:=-1 else yinc:=1;
    end else begin
      x:=x1;  y:=y1;  xend:=x2;
      if (y2>y1) then yinc:=1 else yinc:=-1;
    end;
    incr1:=2*dy; d:=incr1-dx; incr2:=(dy-dx)*2;
    tg_putpixel(x,y,color);
    while x<xend do begin
      inc(x);
      if d<0 then d:=d+incr1 else begin
        y:=y+yinc;
        d:=d+incr2;
      end;
      tg_putpixel(x,y,color);
    end;
  end else begin
    if (y1>y2) then begin
      x:=x2;  y:=y2;  yend:=y1;
      if x2>x1 then xinc:=-1 else xinc:=1;
    end else begin
      x:=x1;  y:=y1;  yend:=y2;
      if x2>x1 then xinc:=1 else xinc:=-1;
    end;
    incr1:=2*dx; d:=incr1-dy; incr2:=(dx-dy)*2;
    tg_putpixel(x,y,color);
    while y<yend do begin
      inc(y);
      if d<0 then d:=d+incr1 else begin
        x:=x+xinc;
        d:=d+incr2;
      end;
      tg_putpixel(x,y,color);
    end;
  end;
end;

procedure tg_line(x1,y1,x2,y2:integer; color:byte);
begin
  if y1=y2 then begin
  	if (x1<x2) then begin
  		hLine(x1,y1,(x2-x1)+1,color);
  	end else begin
  		hLine(x2,y1,(x1-x2)+1,color);
    end;
  end else if x1=x2 then begin
    if (y1<y2) then begin
    	vLine(x1,y1,(y2-y1)+1,color);
    end else begin
    	vLine(x1,y2,(y1-y2)+1,color);
    end;
  end else begin
    lineDraw(x1,y1,x2,y2,color);
  end;
end;

procedure tg_circle(xc,yc,r:integer; color:byte);
var
  x,
  y,
  d:integer;

  procedure circpnt(x,y,xc,yc:integer);
  var
    xxcp,xxcm,xycp,xycm,
    yxcp,yxcm,yycp,yycm:integer;
  begin
    xxcp:=xc+x; xxcm:=xc-x; xycp:=xc+y; xycm:=xc-y;
    yxcp:=yc+x; yxcm:=yc-x;
    yycp:=yc+y; yycm:=yc-y;
    tg_putpixel(xxcp,yycp,color);
    tg_putpixel(xxcm,yycp,color);
    tg_putpixel(xycp,yxcp,color);
    tg_putpixel(xycm,yxcp,color);
    tg_putpixel(xxcp,yycm,color);
    tg_putpixel(xxcm,yycm,color);
    tg_putpixel(xycp,yxcm,color);
    tg_putpixel(xycm,yxcm,color);
  end;

begin
  x:=0;
  y:=r;
  d:=3-(2*r);
  while x<y do begin
    circpnt(x,y,xc,yc);
    if d<0 then begin
      d:=d+(4*x)+6;
    end else begin
      d:=d+4*(x-y)+10;
      dec(y);
    end;
    inc(x);
  end;
  if x=y then circpnt(x,y,xc,yc);
end;

procedure tg_ball(xc,yc,r:integer; color:byte);
var
  x,
  y,
  d:integer;

  procedure circpnt(x,y,xc,yc:integer);
  var
    xxcp,xxcm,xycp,xycm,
    yxcp,yxcm,yycp,yycm:word;
  begin
    xxcp:=xc+x; xxcm:=xc-x; xycp:=xc+y; xycm:=xc-y;
    yxcp:=yc+x; yxcm:=yc-x;
    yycp:=yc+y; yycm:=yc-y;
    tg_line(xxcm,yycp,xxcp,yycp,color);
    tg_line(xxcm,yycm,xxcp,yycm,color);
    tg_line(xycm,yxcp,xycp,yxcp,color);
    tg_line(xycm,yxcm,xycp,yxcm,color);
  end;

begin
  x:=0;
  y:=r;
  d:=3-(2*r);
  while x<y do begin
    circpnt(x,y,xc,yc);
    if d<0 then begin
      d:=d+(4*x)+6;
    end else begin
      d:=d+4*(x-y)+10;
      dec(y);
    end;
    inc(x);
  end;
  if x=y then circpnt(x,y,xc,yc);
end;

procedure tg_tile3(x,y:integer; tileSet:pointer; tile:byte); assembler;
asm
	{ set video address for STOSW }
	mov  ax,$B800
	mov  es,ax
	mov  ax,y
	mov  bx,160
	mul  bx
	mov  cx,x
	mov  di,ax
	add  di,cx
	or   di,1

	{ calculate tile offset }
	xor  ax,ax
	mov  al,tile
	mov  bx,24
	mul  bx
	and  cx,1
	jnz  @oddtile
	add  ax,12
@oddTile:
	mov  dx,ds
	lds  si,tileSet
	add  si,ax

	mov  cx,3

@tileLoop:

	lodsw
	mov  bx,ax
	mov  ax,es:[di]
	and  al,bh
	or   al,bl
	stosw

	lodsw
	mov  bx,ax
	mov  ax,es:[di]
	and  al,bh
	or   al,bl
	stosw

	add  di,156

	loop @tileLoop

	mov ds,dx
end;

procedure tg_tile5(x,y:integer; tileSet:pointer; tile:byte); assembler;
asm
	{ set video address for STOSW }
	mov  ax,$B800
	mov  es,ax
	mov  ax,y
	mov  bx,160
	mul  bx
	mov  cx,x
	mov  di,ax
	add  di,cx
	or   di,1

	{ calculate tile offset }
	xor  ax,ax
	mov  al,tile
	mov  bx,60
	mul  bx
	and  cx,1
	jnz  @oddtile
	add  ax,30
@oddTile:
	mov  dx,ds
	lds  si,tileSet
	add  si,ax

	mov  cx,5

@tileLoop:

	lodsw
	mov  bx,ax
	mov  ax,es:[di]
	and  al,bh
	or   al,bl
	stosw

	lodsw
	mov  bx,ax
	mov  ax,es:[di]
	and  al,bh
	or   al,bl
	stosw

	lodsw
	mov  bx,ax
	mov  ax,es:[di]
	and  al,bh
	or   al,bl
	stosw

	add  di,154

	loop @tileLoop

	mov ds,dx
end;

procedure tg_tile7(x,y:integer; tileSet:pointer; tile:byte); assembler;
asm
	{ set video address for STOSW }
	mov  ax,$B800
	mov  es,ax
	mov  ax,y
	mov  bx,160
	mul  bx
	mov  cx,x
	mov  di,ax
	add  di,cx
	or   di,1

	{ calculate tile offset }
	xor  ax,ax
	mov  al,tile
	mov  bx,128
	mul  bx
	and  cx,1
	jnz  @oddtile
	add  ax,64
@oddTile:
	mov  dx,ds
	lds  si,tileSet
	add  si,ax

	mov  cx,7

@tileLoop:
	mov  dx,cx
	mov  cx,4

@innerTileLoop:

	lodsw
	mov  bx,ax
	mov  ax,es:[di]
	and  al,bh
	or   al,bl
	stosw

	loop @innerTileLoop

	mov cx,dx
	add  di,152

	loop @tileLoop

	mov ds,dx
end;

procedure tg_getBackground3(x,y:integer; data:pTileData); assembler;
asm
	les  di,data
	mov  ax,y
	mov  bx,160
	mul  bx
	mov  si,x
	add  si,ax
	or   si,1
	mov  ax,$B800
	mov  dx,ds
	mov  ds,ax
	mov  cx,3
	mov  bx,156
@getLoop:
	movsw
	movsw
	add  si,bx
	loop @getLoop
	mov  ds,dx
end;

procedure tg_getBackground5(x,y:integer; data:pTileData); assembler;
asm
	les  di,data
	mov  ax,y
	mov  bx,160
	mul  bx
	mov  si,x
	add  si,ax
	or   si,1
	mov  ax,$B800
	mov  dx,ds
	mov  ds,ax
	mov  cx,5
	mov  bx,154
@getLoop:
	movsw
	movsw
	movsw
	add  si,bx
	loop @getLoop
	mov  ds,dx
end;

procedure tg_getBackground7(x,y:integer; data:pTileData); assembler;
asm
	les  di,data
	mov  ax,y
	mov  bx,160
	mul  bx
	mov  si,x
	add  si,ax
	or   si,1
	mov  ax,$B800
	mov  dx,ds
	mov  ds,ax
	mov  cx,5
	mov  bx,152
@getLoop:
	movsw
	movsw
	movsw
	movsw
	add  si,bx
	loop @getLoop
	mov  ds,dx
end;

procedure tg_restoreBackground3(x,y:integer; data:pTileData); assembler;
asm
	mov  ax,y
	mov  bx,160
	mul  bx
	mov  di,x
	add  di,ax
	or   di,1
	mov  ax,$B800
	mov  es,ax
	mov  cx,3
	mov  bx,156
	mov  dx,ds
	lds  si,data
@putLoop:
	movsw
	movsw
	add  di,bx
	loop @putLoop
	mov  ds,dx
end;

procedure tg_restoreBackground5(x,y:integer; data:pTileData); assembler;
asm
	mov  ax,y
	mov  bx,160
	mul  bx
	mov  di,x
	add  di,ax
	or   di,1
	mov  ax,$B800
	mov  es,ax
	mov  cx,5
	mov  bx,154
	mov  dx,ds
	lds  si,data
@putLoop:
	movsw
	movsw
	movsw
	add  di,bx
	loop @putLoop
	mov  ds,dx
end;

procedure tg_restoreBackground7(x,y:integer; data:pTileData); assembler;
asm
	mov  ax,y
	mov  bx,160
	mul  bx
	mov  di,x
	add  di,ax
	or   di,1
	mov  ax,$B800
	mov  es,ax
	mov  cx,7
	mov  bx,152
	mov  dx,ds
	lds  si,data
@putLoop:
	movsw
	movsw
	movsw
	movsw
	add  di,bx
	loop @putLoop
	mov  ds,dx
end;

procedure tg_clear(color:byte); assembler;
asm { clear video memory }
	mov  ax,textSegment
	mov  es,ax
	xor  di,di
	mov  ah,color
	mov  bl,ah
	mov  cl,4
	shl  bl,cl
	and  ah,$03
	or   ah,bl
	mov  al,$DD
	mov  cx,pageSize
	shr  cx,1
	rep  stosw
	xor  al,al
	mov  tg_erase,al
end;

procedure tg_HardWrite(start,value,count:word); assembler;
asm
	mov  ax,textSegment
	mov  es,ax
	mov  di,start
	mov  ax,value
	mov  cx,count
	rep  stosw
end;

function paramExists(st:st40):boolean;
var
	t:word;
	result:boolean;
begin
	t:=0;
	result:=false;
	while (t<paramCount) do begin
		inc(t);
		if (paramStr(t)=st) then begin
			result:=true;
			t:=paramCount;
		end;
	end;
	paramExists:=result;
end;

procedure tg_init;
var
	t:word;
begin
	oldMode:=getVideoMode;
	setVideoMode($02);
	tg_clear(0);

	disableVideoOutputAndInterrupts;

	case videoCard of
		videoCard_cga,
		videoCard_tandy1000,
		videoCard_tandySLTL,
		videoCard_pcJr:begin
			outCGAPort(@CGA_Regs);
		end;
		videoCard_ega:begin
			load8x8FontAndDisableBlink;
			outEGAVGAPort(@EGA_Sequencer);
			outEGAVGAPort(@EGA_CRTC);
		end;
		videoCard_vga:begin
			if (paramExists('/vgasafe')) then begin
				asm
					mov ax,$1003 { disable Blink }
					xor bl,bl
					int $10
					mov ax,$0309 { set fonts to 4px tall }
					mov dx,$3D4
					out dx,ax
				end;
			end else begin
				load8x8FontAndDisableBlink;
				outEGAVGAPort(@VGA_Sequencer);
				outEGAVGAPort(@VGA_CRTC);
			end;
		end;
	end;

	enableVideoOutputAndInterrupts;

	textGraphOn:=true;

	with clippingWindow do begin
		sx:=0;   sy:=0;
		ex:=159; ey:=99;
	end;

end;

procedure tg_term;
begin
	setVideoMode(oldMode);
	textGraphOn:=false;
end;

{
	adding our own custom exitproc is a safeguard so that
	if someone using this library forgets to call tg_term
	the program will still exit to text mode. I WISH that
	execution breaks or failures would call exitproc.

	A more robust unit would add a custom error handler.
}
procedure tg_exitProc; far;
begin
	if textGraphOn then tg_term;
	discardSprites;
	exitProc:=oldTgExitProc;
end;

begin
	write('Video card detected: ');
	videoCard:=detectCard;
	writeln(videoCardName[videoCard]);
	if (videoCard=videoCard_MDA) then begin
		writeln('This program does not work on a monochrome display adapter!');
		halt;
	end;
	firstSprite:=nil;
	lastSprite:=nil;
	textSegment:=$B800;
	textGraphOn:=false;
	oldTgExitProc:=exitProc;
	exitProc:=@tg_exitProc;
end.