/* graphics.c --- Ascii drawings for hangit */

#include "defs.h"

#include <curses.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/ioctl.h>

#include "graphics.h"
#include "extutil.h"

/* Data for drawings */

extern WINDOW * hang_win;

char draw_gallows[] =
"     __________\n"
"    |__________`\\\n"
"     |         | |\n"
"               | |\n"
"               | |\n"
"               | |\n"
"               | |\n"
"               | |\n"
"               | |\n"
"               | |\n"
"               | |\n"
"               | |\n"
"               | |\n"
"               | |\n"
"              /' `\\\n"
"           _/__/^\\__\\_";

/* Drawing structures */
struct drawing *draw_cur;	/* current drawing */
struct drawing *draw_man;	/* man */
struct drawing *draw_tux;	/* tux the penguin */
struct drawing *draw_bill;	/* bill gate$ */
struct drawing *draw_sco;       /* big-headed SCO person */

void
init_drawings ()
{
  /* Populate the drawing objects.  */
  draw_man = (struct drawing *) xmalloc (sizeof (struct drawing));
  draw_man->desc = "An ordinary man";
  draw_man->hanged_msg = "He was hanged!";
  draw_man->saved_msg = "You saved him!";
  draw_man->start_line[6] = 0;
  draw_man->stages[6] =
    "     /\"\"\"\\\n"
    "    ( o o )\n"
    "     \\._./";
  draw_man->start_line[5] = 2;
  draw_man->stages[5] =
    "    _\\._./_\n"
    "   / .\n"
    "  |_|\n"
    "  @@";
  draw_man->start_line[4] = 2;
  draw_man->stages[4] =
    "    _\\._./_\n"
    "   / .   . \\\n"
    "  |_|     |_|\n"
    "  @@       @@";
  draw_man->start_line[3] = 4;
  draw_man->stages[3] =
    "  |_||   ||_|\n"
    "  @@ |___| @@";
  draw_man->start_line[2] = 6;
  draw_man->stages[2] =
    "     |   |\n"
    "     | | |\n"
    "     | | |";
  draw_man->start_line[1] = 8;
  draw_man->stages[1] =
    "   __|_|_|__\n"
    "  |__/_|_\\__|";
  draw_man->start_line[0] = 0;
  draw_man->stages[0] =
    "     /\"\"\"\\\n"
    "    ( + + )\n"
    "    _\\._./_\n"
    "   / .   . \\\n"
    "  |_||   ||_|\n"
    "  @@ |___| @@\n"
    "     |   |\n"
    "     | | |\n"
    "   __|_|_|__\n"
    "  |__/_|_\\__|";

  draw_tux = (struct drawing *) xmalloc (sizeof (struct drawing));
  draw_tux->desc = "Tux, the Linux penguin";
  draw_tux->hanged_msg = "Aw, poor little guy!";
  draw_tux->saved_msg = "Saved to eat herring another day!";
  draw_tux->start_line[6] = 0;
  draw_tux->stages[6] =
    "     /'~`\\\n"
    "    | 0 0 |\n"
    "    | \\_/ |";
  draw_tux->start_line[5] = 3;
  draw_tux->stages[5] =
    "   /'     `\\\n"
    "    .     .";
  draw_tux->start_line[4] = 5;
  draw_tux->stages[4] =
    "   |  /^\\  |\n"
    "   ( (   ) )\n"
    "      ---";
  draw_tux->start_line[3] = 4;
  draw_tux->stages[3] =
    "./' .     .\n"
    "|_.|  /^\\\n"
    "   ( (   ) )\n"
    "      ---";
  draw_tux->start_line[2] = 4;
  draw_tux->stages[2] =
    "./' .     . `\\.\n"
    "|_.|  /^\\  |._|\n"
    "   ( (   ) )\n"
    "      ---";
  draw_tux->start_line[1] = 7;
  draw_tux->stages[1] =
    "   AAA---AAA\n"
    "   \\A/   \\A/";
  draw_tux->start_line[0] = 0;
  draw_tux->stages[0] =
    "     /'~`\\\n"
    "    | + + |\n"
    "    | \\_/ |\n"
    "   /'     `\\\n"
    "./' .     . `\\.\n"
    "|_.|  /^\\  |._|\n"
    "   ( (   ) )\n"
    "   AAA---AAA\n"
    "   \\A/   \\A/";
  
  draw_bill = (struct drawing *) xmalloc (sizeof (struct drawing));
  draw_bill->desc = "Bill Gate$";
  draw_bill->hanged_msg = "So much for 'charity'.";
  draw_bill->saved_msg = "M$ keeps its visionary.";
  draw_bill->start_line[6] = 0;
  draw_bill->stages[6] =
    "     /\"\"\"\\\n"
    "    (\\o-o/)\n"
    "     \\._./";
  draw_bill->start_line[5] = 2;
  draw_bill->stages[5] =
    "    _\\._./_\n"
    "   / .\n"
    "  |_|\n"
    "  @@";
  draw_bill->start_line[4] = 3;
  draw_bill->stages[4] =
    "   / .   . \\\n"
    "  |_|     |_|\n"
    "  @@       @@";
  draw_bill->start_line[3] = 4;
  draw_bill->stages[3] =
    "  |_||M $||_|\n"
    "  @@ |___| @@";
  draw_bill->start_line[2] = 6;
  draw_bill->stages[2] =
    "     |   |\n"
    "     | | |";
  draw_bill->start_line[1] = 8;
  draw_bill->stages[1] =
    "   __|_|_|__\n"
    "  |__/_|_\\__|";
  draw_bill->start_line[0] = 0;
  draw_bill->stages[0] =
    "     /\"\"\"\\\n"
    "    (\\+-+/)\n"
    "    _\\._./_\n"
    "   / .   . \\\n"
    "  |_||M $||_|\n"
    "  @@ |___| @@\n"
    "     |   |\n"
    "     | | |\n"
    "   __|_|_|__\n"
    "  |__/_|_\\__|";
  
  draw_sco = (struct drawing *) xmalloc (sizeof (struct drawing));
  draw_sco->desc = "Guy from SCO";
  draw_sco->hanged_msg = "Ha, he is unable to harrass IBM now!";
  draw_sco->saved_msg = "Free to terrorize the sane world!";
  draw_sco->start_line[6] = 0;
  draw_sco->stages[6] =
    "    /#####\\\n"
    "   (  O O  )\n"
    "    \\ ._. /";
  draw_sco->start_line[5] = 2;
  draw_sco->stages[5] =
    "    \\ ._. /\n"
    "   / .\n"
    "  |_|\n"
    "  @@";
  draw_sco->start_line[4] = 3;
  draw_sco->stages[4] =
    "   / .   . \\\n"
    "  |_| ANTI |_|\n"
    "  @@ LINUX @@";
  draw_sco->start_line[3] = 4;
  draw_sco->stages[3] =
    "  |_||SCO||_|\n"
    "  @@ |___| @@";
  draw_sco->start_line[2] = 6;
  draw_sco->stages[2] =
    "     |   |\n"
    "     | | |";
  draw_sco->start_line[1] = 8;
  draw_sco->stages[1] =
    "   __|_|_|__\n"
    "  |__/_|_\\__|";
  draw_sco->start_line[0] = 0;
  draw_sco->stages[0] =
    "    /#####\\\n"
    "   (  x x  )\n"
    "    \\ ._. /\n"
    "   / .   . \\\n"
    "  |_||SCO||_|\n"
    "  @@ |___| @@\n"
    "     |   |\n"
    "     | | |\n"
    "   __|_|_|__\n"
    "  |__/_|_\\__|";

  draw_cur = draw_man;		/* default character = man */
}

void
init_curses (void)
{
  /* Initialize curses library.  */
  initscr ();                   /* initialize curses library */
  if (stdscr == 0)
    fatal ("Could not initialize curses");

  cbreak ();                    /* take chars one at a time */
  noecho ();                    /* don't echo chars on input */
  intrflush (stdscr, FALSE);
  keypad (stdscr, TRUE);        /* enable keypad mapping */
}

void
restore_terminal (void)
{
  /* This restores the terminal back to normal settings. */
  clear ();			/* clear screen */
  refresh ();			/* reveal changes */
  endwin ();			/* end the stdwin */
}

#ifdef ALLOW_RESIZING
void
global_resize_screen (void)
{
  /* Handles screen resize for all situations.  Ian Jackson's dpkg
     project was consulted to determine the correct way to do this.
     It is available under the terms of the GNU GPL.  */
  struct winsize size;

  endwin ();
  initscr ();
  
  if (ioctl (fileno (stdout), TIOCGWINSZ, &size) != 0)
    fatal ("Failed to determine new screen coordinates");
  resizeterm (size.ws_row, size.ws_col);

  box (stdscr, 0, 0);           /* draw a box around the window */
  refresh ();
}
#endif /* resizable */

void
print_score (void)
{
  /* Outputs score to stdscr. */
  char buffer[16];

  sprintf (buffer, "Score: %d  ", score);	/* put score in buffer */
  mvaddstr (1, COLS - 20, buffer);
}

void
print_letter_list (char *letter_list)
{
  /* Prints the list of chars at bottom of screen. */
  char buffer[53] = "";		/* 26 * 2 + 1 */
  int i, maxlines, maxcols;

  if (!buffer[0])
    {				/* is buffer unmodified? */
      for (i = 0; i < 26; i++)
	{			/* all the chars, separated by spaces */
	  buffer[i * 2] = letter_list[i];
	  buffer[i * 2 + 1] = ' ';
	}
      buffer[52] = 0;		/* null-terminate */
    }

  getmaxyx (stdscr, maxlines, maxcols);
  mvaddstr (maxlines - 2, 2, buffer); /* put it on screen */
}

void
print_underscores (char *guessed_word)
{
  /* Makes and prints a string of underscores to show the # chars
     in the correct word. */
  char *buffer = (char *) xmalloc (word_length * 2);
  int i, maxlines, maxcols;

  for (i = 0; i < word_length; i++)	/* space out letter list */
    {
      buffer[i * 2] = guessed_word[i];
      buffer[i * 2 + 1] = ' ';
    }

  buffer[word_length * 2 - 1] = 0;	/* null-terminate */
  getmaxyx (stdscr, maxlines, maxcols);
  mvaddstr (maxlines - 6, 2, buffer);	/* put it to screen */
}

void
print_art (WINDOW * win, int number)
{
  /* Select which character to hang, or save coding
     time by printing empty gallows. */
  int maxlines, maxcols;
  char *msg = (char *) xmalloc (25);
  getmaxyx (stdscr, maxlines, maxcols);
  
  if (number == LIVES_AVAIL)
    {
      mvwaddstr (hang_win, 0, 0, draw_gallows);
      /* mvwaddstr (stdscr, 0, 0, "I've printed the gallows!!"); */
      return;
    }

  if (number > LIVES_AVAIL || number < 0)
    {
      sprintf (msg, "Illegal stage: %d!", number);
      mvwaddstr (win, 0, 0, msg);
    }
  
  mvwaddstr (win, draw_cur->start_line[number], 0,
	     draw_cur->stages[number]);
}

void
print_gameover (WINDOW * win)
{
  /* Print "Game Over" message. */
  mvwaddstr (win, 1, 2, "  ___            ____    ___  _ _ ____ ___  _ ");
  mvwaddstr (win, 2, 2, " /,-.\\ /^\\|\\   /| ___|  /   \\| | | ___| D )| |");
  mvwaddstr (win, 3, 2, "((  __| A | `V' | __|  (  O  | | | __||   \\|_|");
  mvwaddstr (win, 4, 2, " \\___/|_|_|_|_|_|____|  \\___/ \\./|____|_|\\_|_|");
}

void
print_welldone (WINDOW * win)
{
  /* Print "Well Done" message. */
  mvwaddstr (win, 1, 2, " _ _ _ ____ _   _     _    ___  _  _ ____ _");
  mvwaddstr (win, 2, 2, "| | | | ___| | | |   |  \\ /   \\| || | ___| |");
  mvwaddstr (win, 3, 2, "|     | __|| |_| |_  | D |  O  |  \\ | __||_|");
  mvwaddstr (win, 4, 2, "|/'^`\\|____|___|___| |__/ \\___/|_|\\_|____|_|");
}
