/* game.c -- Contains a listing of all decision-making parts of
             the game.
*/

#include "defs.h"

#include <curses.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef __LINUX__
# include <signal.h>
#endif /* Linux */

#include "graphics.h"
#include "extutil.h"

/* Globals */

/* Parameter input window */
WINDOW *param_win;

int
is_correct_guess (char guess, char *word)
{
  /* Determine if the guess was correct. */
  int i;

  for (i = 0; i < word_length; i++)
    {				/* process all letters in word */
      if (word[i] == guess)
	return 1;
    }
  return 0;			/* letter not found in word */
}

int
add_correct_letter (char guess, char *guessed_word, char *word)
{
  /* Add guessed letter(s) to replace '_' chars, and
     return the score to be awarded. */
  int i, score = 0;

  for (i = 0; i < word_length; i++)
    {
      if (word[i] == guess)
	{
	  guessed_word[i] = guess;
	  score += 10;		/* 10 points per replaced letter */
	}
    }

  return score;
}

void
process_guess (WINDOW * ascii_win, char guess, char *letter_list,
	       char *guessed_word, char *word)
{
  /* Process the user's guess and take action. */
  if (guess == 27)
    {				/* did they hit ESC? */
      lives = -1;
      return;
    }

  if (!isalpha (guess))
    {				/* is guess an alphabetic char */
      mvaddstr (2, 2, "You used an illegal character!    ");
      guess = 0;
      return;
    }

  guess = toupper (guess);	/* convert to uppercase */
  addch (guess);		/* output character typed */
  if (letter_list[guess - 'A'] == '*')
    {				/* is letter used already? */
      mvaddstr (2, 2, "You've already tried that!        ");
      guess = 0;
      return;
    }

  letter_list[guess - 'A'] = '*';	/* blank out guessed letter */

  if (is_correct_guess (guess, word))
    {				/* is guess correct? */
      mvaddstr (2, 2, "You made a correct guess!         ");

      /* Now increment score and add guessed letters. */
      score += add_correct_letter (guess, guessed_word, word);
      return;
    }

  /* Hmm, the guess must be incorrect. */
  mvaddstr (2, 2, "This letter is not in the word!   ");
  lives--;
  score -= 5;
  print_art (ascii_win, lives);	/* draw the next hanging stage */
  wnoutrefresh (ascii_win);	/* refresh the hanging 'window' */
}

#ifdef ALLOW_RESIZING
void
param_resize_screen ()
{
  /* Handle resizing the parameter window.  */
  int maxlines, maxcols;
  
  global_resize_screen ();

  getmaxyx (stdscr, maxlines, maxcols);
  if (mvwin (param_win, (maxlines - 7) / 2, (maxcols - 15) / 2)
      != OK)
    fatal ("Could not move parameter window");
  
  wmove (param_win, 5, 11);
  wnoutrefresh (param_win);
  doupdate ();
}
#endif /* resizable */

void
get_user_params ()
{
  /* Prompt the user concerning the difficulty level and drawing to
     use.  Set the global vars 'difficulty' and 'draw_cur' as per the
     user's preferences.  For 'difficulty': 1=easy, 2=normal, or
     3=hard. Normal is the default.  */
  char *msg = (char *) xmalloc (30);
  int key, maxlines, maxcols;
  getmaxyx (stdscr, maxlines, maxcols);

  /* Now draw the window with the difficulty question. */
  param_win = subwin (stdscr, 17, 38, (maxlines - 17) / 2,
			   (maxcols - 38) / 2);
  box (param_win, 0, 0);
  mvwaddstr (param_win, 1, 2, "--Wordlist--");
  mvwaddstr (param_win, 2, 2, "1. Easy");
  mvwaddstr (param_win, 3, 2, "2. Normal");
  mvwaddstr (param_win, 4, 2, "3. Star Wars");
  mvwaddstr (param_win, 5, 2, "Choice: ");
  mvwaddstr (param_win, 7, 2, "--Drawing--");
  mvwaddstr (param_win, 8, 2, "A. ");
  waddstr (param_win, draw_man->desc);
  mvwaddstr (param_win, 9, 2, "B. ");
  waddstr (param_win, draw_tux->desc);
  mvwaddstr (param_win, 10, 2, "C. ");
  waddstr (param_win, draw_bill->desc);
  mvwaddstr (param_win, 11, 2, "D. ");
  waddstr (param_win, draw_sco->desc);
  mvwaddstr (param_win, 12, 2, "Choice: ");
  wnoutrefresh (param_win);
  mvwaddstr (param_win, 14, 2, "Choose a wordlist and drawing,");
  mvwaddstr (param_win, 15, 2, "and then press ENTER when done.");
  doupdate ();

#ifdef ALLOW_RESIZING
  signal (SIGWINCH, param_resize_screen);
#endif /* resizable */
  
  do
    {
      key = wgetch (param_win); /* fetch keypress */
      if (key >= '1' && key <= '3')
	{
	  sprintf (msg, "Choice: %c", key);
	  mvwaddstr (param_win, 5, 2, msg);
	  difficulty = key - '0';
	}
      else if ((key >= 'A' && key <= 'D') ||
	       (key >= 'a' && key <= 'd'))
	{
	  key = toupper (key);
	  sprintf (msg, "Choice: %c", key);
	  mvwaddstr (param_win, 12, 2, msg);
	  if (key == 'A')
	    draw_cur = draw_man;
	  else if (key == 'B')
	    draw_cur = draw_tux;
	  else if (key == 'C')
	    draw_cur = draw_bill;
	  else
	    draw_cur = draw_sco;
	}
      else
	{
	  /*sprintf (msg, "Got %d, KEY_ENTER == %d", key, KEY_ENTER);
	    mvwaddstr (param_win, 13, 4, msg);*/
	}
    } while (key != 10 && key != 27);

#ifdef ALLOW_RESIZING
  signal (SIGWINCH, SIG_IGN);
#endif /* resizable */
  
  wclear (param_win);
  wnoutrefresh (param_win);
  delwin (param_win);		/* remove window */

  if (key == 27)
    {				/* ESC pressed */
      restore_terminal ();
      exit (EXIT_FAILURE);
    }
}

int
end_game (char *word)
{
  /* End of game routine. */
  char ch;
  int maxlines, maxcols;
  WINDOW *game_end_win;

  game_end_win = subwin (stdscr, 4, 40, 6, 2);	/* win for "game over" string */
  box (game_end_win, 0, 0);	/* draw box around it */

  if (lives > 0)
    {				/* user won the game */
      print_welldone (stdscr);
      mvwaddstr (game_end_win, 1, 2, draw_cur->saved_msg);
      mvwaddstr (game_end_win, 2, 4, "You win!");
    }
  else
    {				/* they lost or quit */
      print_gameover (stdscr);	/* shows "GameOver" graphic on screen */
      score = score > 0 ? 0 : score;	/* score = 0 if positive */
      print_score ();
      print_underscores (word);	/* display the word */
      
      if (lives == 0)
	{			/* lost */
	  mvwaddstr (game_end_win, 1, 2, draw_cur->hanged_msg);
	  mvwaddstr (game_end_win, 2, 4, "You lose.");
	}
      else
	{			/* quit */
	  mvwaddstr (game_end_win, 1, 2,
		     "You have aborted the game!");
	}
    }

  getmaxyx (stdscr, maxlines, maxcols);
  mvaddstr (maxlines - 4, 2,
	    "Press any key to play again, ESC exits...");
  wnoutrefresh (game_end_win);	/* make sure BOTH of these windows get updated */
  refresh ();			/*   in the correct order */
  ch = wgetch (game_end_win);	/* get a keypress */
  wclear (game_end_win);
  wnoutrefresh (game_end_win);
  delwin (game_end_win);	/* remove end of game window */
  wnoutrefresh (stdscr);	/* refresh the new top window */

  return ch;
}
