\SOKOBAN.XPL	31-MAY-2002
\3D Isometric Sokoban Game
\ by Loren Blaney
\ loren_blaney@idcomm.com
\Compile with XX.BAT
\
\REVISIONS:
\04-MAY-02, Released (Escaped?)
\31-MAY-02, Added Sound Blaster sounds and other minor mods


inc	\cxpl\codesi;		\intrinsic routine declarations

def	Debug = true;		\enable error trapping and diagnostic messages

seg char ScreenBuf(1);		\screen buffer: image that gets copied to screen

char	SpTitle(2+255*64),	\sprites: (byte: width, height, graphic image)
	SpFloor(2+32*32),	\most sprites are 32x32 pixels
	SpDock(2+32*32),
	SpWall(2+32*32),
	SpBox(2+32*32),
	SpManS(2+32*32),	\man facing in four directions
	SpManN(2+32*32),
	SpManW(2+32*32),
	SpManE(2+32*32);

int	FloorPlan(20, 17);	\20 columns by 17 rows, maximum

def	StkMax = 30000;		\stack for saving & undoing 30000/11 = 2727 moves
char	Stack(StkMax),
	StkPtr;			\stack pointer (see Push and Pop procedures)

char	ColorReg(256*3);	\copy of initial color register values for Fade

int	Key,			\keystroke's scan code
	Level,			\level number of floor plan
	LevelHandle,		\input file handle to read LEVELS.TXT
	ManI, ManJ,		\man's location in FloorPlan (tiles)
	Nx, Ny,			\possible new man location (tiles)
	Bx, By,			\possible new box location (tiles)
	MoveCtr,		\move counter (number of arrow keystrokes)
	RefI, RefJ;		\reference tile location; used for scrolling

seg char SoundBuf(1);		\buffer that holds sound sample data
int	BlasterPort,		\Sound Blaster's base port addr (usually = $220)
	BlasterDMA,		\DMA channel used by Sound Blaster (usually = 1)
	HaveBlaster,		\flag: a Sound Blaster is present
	SoundMem,		\extra memory allocated for SoundBuf
	SoundLength,		\sound sample data length (bytes)
	SoundRate;		\sound sample rate (bytes per second)



fproc	CloseSound;		\forward procedure declaration (for neatness)

proc	Exit(Msg);		\Display (error) message and terminate program
char	Msg;
begin
SetVid($03);			\restore normal text mode
Text(0, Msg);
Crlf(0);

CloseSound;			\(keeps Windows happy)
exit;
end;	\Exit



func	CallInt(Int, AX, BX, CX, DX, BP, DS, ES); \Call a software interrupt
int	Int, AX, BX, CX, DX, BP, DS, ES; \(unused arguments need not be passed)
int	Cpureg;
begin
Cpureg:= Getreg;
Cpureg(0):= AX;
Cpureg(1):= BX;
Cpureg(2):= CX;
Cpureg(3):= DX;
Cpureg(6):= BP;
Cpureg(9):= DS;
Cpureg(11):= ES;
Softint(Int);
return Cpureg(0);		\return AX register
end;	\CallInt



proc	Delay(T);	\Delay approximately T milliseconds
int	T;		\maximum T = 64
int	Cpureg;
begin
Cpureg:= Getreg;
Cpureg(0):= $8600;	\delay CX:DX microseconds
Cpureg(2):= 0;		\ (resolution is only 976 microseconds)
Cpureg(3):= T*1000;
Softint($15);
end;	\Delay



func	ToUpper(Ch);	\Convert character to uppercase
int	Ch;
return if Ch>=^a & Ch<=^z then Ch&$DF else Ch;

\=========================== SOUND BLASTER ROUTINES ============================
\Plays .WAV files on Sound Blaster (and work-alikes). Based on the code
\ SFXMIX.C by Erik Lorenzen & Keith Weiner in the book "More Tricks of the
\ Game Programming Gurus".
\ 
\Beware of memory managers (such as EMM386) and sound sample lengths > 16K. The
\ DMA hardware cannot cross a 64K page boundary. Memory managers remap memory,
\ so that an application cannot know where it's physically located. However they
\ do it in 16K chunks, so as long as a chunk is <= 16K, it's guaranteed to not
\ cross a page boundary.



proc	LoadWav(FN);	\Load (.WAV) FileName into SoundBuf
			\Outputs: SoundLength and SoundRate
char	FN;	\string: file name (including ".WAV" extension)
int	ID, Size, Han, I, T;

	proc	ReadChunkHeader;	\Read 4 ID bytes and 4 Size bytes
	int	I, T;
	begin
	ID:= 0;
	for I:= 1, 4 do ID:= ID + Chin(3);	\cheap hash code
	Size:= Chin(3) + Chin(3)<<8;		\0..32764 is sufficient here
	T:= Chin(3);
	T:= Chin(3);
	end;	\ReadChunkHeader

begin
Trap(false);				\don't trap "file not found" error
Han:= FOpen(FN, 0);			\open file for input
if Geterr \#0\ then Exit(".WAV file not found");
Trap(Debug);				\turn traps back on if debug mode
FSet(Han, ^I);				\set device 3 to handle
Openi(3);

SoundRate:= 0;

ReadChunkHeader;
if ID # ^R+^I+^F+^F then Exit("unrecognized .WAV file format");
for I:= 1, 4 do T:= Chin(3);		\skip "WAVE" bytes

loop	begin
	ReadChunkHeader;
	case ID of
	  ^f+^m+^t+^ :
		begin
		for I:= 1, 4 do T:= Chin(3);		\skip format & num chans
		SoundRate:= Chin(3) + Chin(3)<<8;	\0..32767 is sufficient
		for I:= 4+2, Size-1 do T:= Chin(3);	\skip rest of chunk
		end;
	  ^d+^a+^t+^a:
		begin			\load .WAV data into SoundBuf
		for I:= 0, Size-1 do			\32 K maximum
			SoundBuf(0, I):= Chin(3);
		SoundLength:= Size;
		quit;
		end
	other	for I:= 0, Size-1 do T:= Chin(3);	\skip unrecognized chunk
	end;
if SoundRate <= 0 then Exit("illegal .WAV rate");
FClose(Han);				\close handle so it can be used again
end;	\LoadWav

\-------------------------------------------------------------------------------

proc	SetUpDMA(Chan);	\Set up PC's DMA regs to send sound sample to Blaster
int	Chan;		\DMA channel (0..7) (normally = 1)
int	MaskReg,
	ClrFFReg,
	ModeReg,
	AddrReg,
	PageReg,
	CountReg;
begin
\DMA controller port addresses
\ Chan:      0    1    2    3    4    5    6    7
MaskReg:=  [$0A, $0A, $0A, $0A, $D4, $D4, $D4, $D4];
ClrFFReg:= [$0C, $0C, $0C, $0C, $D8, $D8, $D8, $D8];
ModeReg:=  [$0B, $0B, $0B, $0B, $D6, $D6, $D6, $D6];
AddrReg:=  [$00, $02, $04, $06, $C0, $C4, $C8, $CC];
PageReg:=  [$87, $83, $81, $82, $8F, $8B, $89, $8A];
CountReg:= [$01, $03, $05, $07, $C2, $C6, $CA, $CE];

port(MaskReg(Chan)):= Chan&$03 ! $04;	\disable selected DMA channel
port(ClrFFReg(Chan)):= 0;	\clear flipflop: select low byte of 2-byte value
port(ModeReg(Chan)):= Chan&$03 ! $48;	\single mode, read, no auto-init
port(AddrReg(Chan)):= SoundBuf(0) << 4;	\physical address bits A0..A7
port(AddrReg(Chan)):= SoundBuf(0) >> 4;	\physical address bits A8..A15
port(PageReg(Chan)):= SoundBuf(0) >> 12;\physical address bits A16..A19 (or A23)
port(CountReg(Chan)):= SoundLength-1;	\number of bytes to transfer -1
port(CountReg(Chan)):= (SoundLength-1)>>8;
port(MaskReg(Chan)):= Chan & $03;	\enable selected DMA channel
end;	\SetUpDMA

\-------------------------------------------------------------------------------

proc	DmaOff(Chan);	\Disable selected DMA channel
int	Chan;		\DMA channel (0..7) (normally = 1)
int	MaskReg;
begin
\DMA controller port addresses
\ Chan:      0    1    2    3    4    5    6    7
MaskReg:=  [$0A, $0A, $0A, $0A, $D4, $D4, $D4, $D4];
port(MaskReg(Chan)):= Chan&$03 ! $04;
end;	\DmaOff

\-------------------------------------------------------------------------------

func	ResetBlaster;	\Reset Sound Blaster's DSP. Failure returns 'false'.
\DSP, in Creative Labs' parlance, stands for "Digital Sound Processor".
def	TO = \10\2;	\timeout duration (milliseconds)
int	T1, T2;		\timeout timers
begin
port(BlasterPort+6):= $01;	\write a "1" to RESET port
Delay(1);			\wait at least 3 microseconds
port(BlasterPort+6):= $00;	\write a "0" to RESET port

\Poll for a READY BYTE = $AA from the READ DATA port; timeout if it fails
T2:= 0;
repeat	begin
	T2:= T2 + 1;
	if T2 >= TO then return false;	\if timeout then indicate failure
	\Check DATA AVAILABLE status (= $80)
	T1:= 0;
	repeat	if T1 >= TO then return false;
		Delay(1);
		T1:= T1 + 1;
	until port(BlasterPort+$E) & $80 \# 0\;
	end;
until port(BlasterPort+$A) = $AA;
return true;			\indicate success
end;	\ResetBlaster

\-------------------------------------------------------------------------------

proc	WriteDSP(Cmd);	\Write a command to Sound Blaster's DSP
int	Cmd;
def	TO = \10\2;	\timeout duration (milliseconds)
int	T1;		\timeout timer
begin
Delay(1);	\Required for Compaq Presario laptop running Windows 98, but
		\ not necessary for the same computer running under pure DOS
T1:= 0;
while port((BlasterPort+$0C)&$80) # 0  &  T1 < TO do   \wait for DSP to be ready
	[T1:= T1 + 1;   Delay(1)];

\Some computers are always ready and don't send the DSP Ready signal.
\The Compaq Presario laptop running in pure DOS mode does not send the DSP
\ Ready signal, whereas the same computer under Windows 98 does.

port(BlasterPort+$0C):= Cmd;
end;	\WriteDSP

\-------------------------------------------------------------------------------

proc	GetBlasterEnv;	\Get Sound Blaster's base port and DMA Channel
			\Outputs: BlasterPort and BlasterDMA
int	Cpureg, I, Ch;
seg int	PSPSeg(1);	\segment address of Program Segment Prefix
seg char EnvSeg(1);	\segment address of the environment block



func	GetEnv(Str);	\Find string (Str) in environment block (EnvSeg) and
char	Str;		\ return the index to the rest of the line. Returns 0
int	Ch, I, J;	\ if Str is not found.
begin
Ch:= EnvSeg(0,0);	\get first character from environment
J:= 1;			\set index past first character
loop	begin		\for all the strings in the environment
	I:= 0;		\set Str index
	loop	begin	\for each char in a line
		if ToUpper(Ch) = ToUpper(Str(I)&$7F) then
			begin
			if Str(I) >= $80 then	\if match then return index
				return J;
			I:= I + 1;
			Ch:= EnvSeg(0,J);
			J:= J + 1;
			end
		else	begin		\eat rest of line
			while EnvSeg(0,J) # 0 do J:= J + 1;
			J:= J + 1;	\point past 0
			quit;
			end;
		end;
	Ch:= EnvSeg(0,J);
	J:= J + 1;
	if Ch = 0 then quit;		\00 terminates all strings
	end;
return 0;		\Str not found
end;	\GetEnv



begin	\GetBlasterEnv
\Get segment address where PSP starts
Cpureg:= Getreg;
PSPSeg(0):= Cpureg(9);

\Get segment pointer to environment block
EnvSeg(0):= PSPSeg(0, $2C>>1);	\divide by 2 to address words, not bytes

BlasterPort:= $220;		\set default values
BlasterDMA:= 1;
I:= GetEnv("BLASTER");
if I # 0 then
  repeat begin			\e.g: BLASTER=A220 I5 D1
	Ch:= EnvSeg(0,I);
	I:= I + 1;
	case ToUpper(Ch) of
	  ^A:	begin
		BlasterPort:= 0;
		Ch:= EnvSeg(0,I);
		I:= I + 1;
		while Ch>=^0 & Ch<=^9 do  \only 220..260 are allowed
			begin
			BlasterPort:= BlasterPort<<4 ! Ch-^0;
			Ch:= EnvSeg(0,I);
			I:= I + 1;
			end;
		end;
	  ^D:	begin
		Ch:= EnvSeg(0,I);
		I:= I + 1;
		BlasterDMA:= Ch - ^0;
		end
	other	[];
	end;
until Ch = 0;			\ASCIIZ string
end;	\GetBlasterEnv

\-------------------------------------------------------------------------------

proc	OpenSound;	\Initialize Sound Blaster
begin			\Outputs: HaveBlaster, SoundMem and SoundBuf
GetBlasterEnv;
HaveBlaster:= ResetBlaster;
if not HaveBlaster then return;

\Set up a 64K sound buffer on an even page (64K) boundary for DMA
SoundMem:= Malloc($2002);		\ = 131k
SoundBuf(0):= (SoundMem & $F000) + $1000;
end;	\OpenSound

\-------------------------------------------------------------------------------

proc	PlaySound(FileName);	\Play specified .WAV file on Sound Blaster
char	FileName;		\string: file name (including .WAV extension)
begin
if not HaveBlaster then return;
if not ResetBlaster then return;

LoadWav(FileName);
SetUpDMA(BlasterDMA);

WriteDSP($D1);			\enable speaker (rather than recording)
Delay(1);			\wait at least 112 uSec
WriteDSP($40);			\set time constant for sample rate
WriteDSP(256 - Fix(1E6/Float(SoundRate)));
WriteDSP($48);			\set block transfer size
WriteDSP(SoundLength-1);
WriteDSP((SoundLength-1)>>8);
WriteDSP($1C);			\play the sound
end;	\PlaySound

\-------------------------------------------------------------------------------

proc	CloseSound;	\Shut the Sound Blaster down (and keep Windows happy)
begin
if not HaveBlaster then return;
Release(SoundMem);

\Some Windows systems get upset if you don't do the following:
WriteDSP($D3);			\disable speaker
Delay(1);			\wait at least 220 uSec
DmaOff(BlasterDMA);
ResetBlaster;			\leave Sound Blaster in a known, good state
end;	\CloseSound

\===============================================================================

proc	Erase(Im, C);		\Erase image Im
seg int Im;			\write two bytes at a time for speed
int	C;			\color
int	T, I;
begin
C:= C ! C<<8;			\set high & low bytes to same color
T:= (320*8)/2;			\number of words on top 8 scan lines
for I:= 0, T-1 do
	Im(0, I):= $0000;	\black (to match background of mode $13 text)
for I:= T, 32000-1 do
	Im(0, I):= C;
end;	\Erase



proc	Push(N);		\Push byte N onto the Stack
int	N;
begin
if StkPtr < StkMax then
	begin
	Stack(StkPtr):= N;
	StkPtr:= StkPtr + 1;
	end;
end;	\Push



func	Pop;			\Pop item from Stack and return it
begin
if StkPtr > 0 then
	StkPtr:= StkPtr - 1;
return Stack(StkPtr);
end;	\Pop

\-------------------------------------------------------------------------------

proc	Fade(Dn, Spd);		\Fade screen colors up or down
int	Dn,			\flag: fade down to black
	Spd;			\speed: 0=slow, 1=fast, 2=very fast
int	Ints;			\intensity


	proc	Set(I);		\Set intensity of the colors
	int	I;
	int	J;
	begin
	while port($3DA) & $08 do;	\wait for no vertical blank
	repeat until port($3DA) & $08;	\wait for vertical blank

	port($3C8):= 0;			\set color registers
	for J:= 0, 256*3-1 do
		port($3C9):= (ColorReg(J) * I) >>6;
	end;	\Set


begin	\Fade
if Dn then
	for Ints:= -$3F, 0 do [Set(-Ints);   Ints:= Ints +Spd]
else	for Ints:= 1, $40 do [Set(Ints);   Ints:= Ints +Spd];
end;	\Fade

\-------------------------------------------------------------------------------

proc	DrawSprite(X0, Y0, S);	\Draw a sprite onto ScreenBuf
int	X0, Y0;	\coordinates where upper-left corner of sprite is displayed
char	S;	\address of sprite data
int	X, Y, K, Y320, P, W, H;
begin
W:= S(0);	\get width and height (in pixels)
H:= S(1);
K:= 2;
for Y:= Y0, Y0+H-1 do
	begin
	Y320:= Y * 320;
	for X:= X0, X0+W-1 do
		begin
		P:= S(K);			\get pixel's color
		K:= K + 1;
		if P & X>=0 & X<320 then	\background (0) is transparent
		    if Y>=8 & Y<200 then	\clip below text on top line
			ScreenBuf(0, X+Y320):= P;
		end;
	end;
end;	\DrawSprite

\-------------------------------------------------------------------------------

proc	LoadFloorPlan;		\Load next floorplan level into FloorPlan array
int	I, J, Ch;
begin
FSet(LevelHandle, ^i);			\associate device 3 with LEVELS.TXT file
for J:= 0, 17-1 do			\all floor plans are 20x17
    for I:= 0, 20-1 do
	begin
	repeat	Ch:= Chin(3);
		if Ch = $1A\EOF\ then Exit("All levels solved!");  \exit program
	until Ch>=$20 & Ch<=$7E & Ch#^>;	\skip CR, LF, etc.
	case Ch of
	  ^ :	FloorPlan(I,J):= $01;	\floor
	  ^!:	FloorPlan(I,J):= $02;	\loading dock
	  ^":	FloorPlan(I,J):= $00;	\background
	  ^#:	FloorPlan(I,J):= $03;	\wall
	  ^$:	FloorPlan(I,J):= $11;	\box on floor
	  ^%:	FloorPlan(I,J):= $12;	\box on dock
	  ^&:	begin
		FloorPlan(I,J):= $21;	\man on floor
		ManI:= I;		\record man's position
		ManJ:= J;

		RefI:= 0;		\shift floorplan image if necessary to
		RefJ:= 0;		\ center it on the man
		if ManI <  6 then RefI:= RefI + ManI-6;
		if ManI > 13 then RefI:= RefI + ManI-13;
		if ManJ <  6 then RefJ:= RefJ + ManJ-6;
		if ManJ > 13 then RefJ:= RefJ + ManJ-13;
		end
	other	[];			\ignore any other characters
	end;
end;	\LoadFloorPlan

\-------------------------------------------------------------------------------

proc	DrawFloorPlan;	\...for current level, including boxes and man
int	I, J,		\coordinates on floor (tiles)
	X, Y,		\coordinates on screen (pixels)
	T;		\temporary scratch


	func	Tile2PixX(I, J);    \Convert tile position to pixel X coordinate
	int	I, J;
	begin
	I:= I - RefI;
	J:= J - RefJ;
	return J*16 - (19-I)*16 + 160-16;
	end;	\Tile2PixX


	func	Tile2PixY(I, J);    \Convert tile position to pixel Y coordinate
	int	I, J;
	begin
	I:= I - RefI;
	J:= J - RefJ;
	return J*8 - I*8 + 96-8;
	end;	\Tile2PixY


	proc	DrawMan;	\...facing direction of last keystroke
	case Key of
	  $48:	DrawSprite(X, Y, SpManN);	\up
	  $4B:	DrawSprite(X, Y, SpManW);	\left
	  $4D:	DrawSprite(X, Y, SpManE)	\right
	other	DrawSprite(X, Y, SpManS);	\down


begin	\DrawFloorPlan
Erase(ScreenBuf, 2\green grass background\);

\Draw tiles starting from the back (painter's algorithm)
for J:= 0, 17-1 do				\from top to bottom...
    for I:= -19, 0 do				\ from right to left...
	begin
	X:= Tile2PixX(-I, J);
	Y:= Tile2PixY(-I, J);
	case FloorPlan(-I, J) of
	  $01:	DrawSprite(X, Y, SpFloor);
	  $02:	DrawSprite(X, Y, SpDock);
	  $00:	[];				\background
	  $03:	DrawSprite(X, Y, SpWall);
	  $11:	[DrawSprite(X, Y, SpFloor);
		 DrawSprite(X, Y, SpBox)];
	  $12:	[DrawSprite(X, Y, SpDock);
		 DrawSprite(X, Y, SpBox)];
	  $21:	[DrawSprite(X, Y, SpFloor);   DrawMan];
	  $22:	[DrawSprite(X, Y, SpDock);   DrawMan]
	other	[if Debug then Exit("Illegal sprite code")];
	end;

while port($3DA) & $08 do;			\wait for no vertical blank
repeat until port($3DA) & $08;			\wait for vertical blank
Blit(ScreenBuf(0), 0, $A000, 0, 64*1000);	\copy ScreenBuf image to screen

\Display level name in upper-left corner
Attrib($07);			\for mode $13: white on background color (black)
Cursor(0, 0);
Text(6, "LEVEL ");
Intout(6, Level);

\Display move counter in upper-right corner
Cursor(30, 0);
T:= MoveCtr;			\right-justify the number
if T = 0 then T:= 1;
while T < 1000 do [Chout(6, ^ );  T:= T*10];
Intout(6, MoveCtr);
Text(6, " MOVES");
end;	\DrawFloorPlan

\-------------------------------------------------------------------------------

func	Solved;		\Return 'true' if all the docks have boxes on them
int	I, J;		\(Note: Some levels have more boxes than docks)
begin
for J:= 0, 17-1 do
    for I:= 0, 20-1 do
	begin
	if (FloorPlan(I,J)&$0F) = $02\dock\ then
		if (FloorPlan(I,J)&$F0) # $10\box\ then return false;
	end;
return true;
end;	\Solved



proc	Undo;		\Undo a move
begin
if StkPtr > 0 then			\there is a move saved on the stack
	begin
	RefJ:= extend(Pop);		\restore scroll info
	RefI:= extend(Pop);
	By:= Pop;			\restore new location of box
	Bx:= Pop;
	Ny:= Pop;			\restore new location of man
	Nx:= Pop;
	ManJ:= Pop;			\restore location of man
	ManI:= Pop;
	FloorPlan(Bx,By):= Pop;		\restore contents where box might go
	FloorPlan(Nx,Ny):= Pop;		\restore contents at new man location
	FloorPlan(ManI,ManJ):= Pop;	\restore man and his background
	MoveCtr:= MoveCtr - 1;
	end;
end;	\Undo



proc	DoMove;		\The arrow keys move the man and possibly a box


	proc	MoveMan;	\Display man at N and blank out man at ManI,ManJ
	begin
	FloorPlan(Nx,Ny):= (FloorPlan(Nx,Ny) & $0F) ! $20;
	FloorPlan(ManI,ManJ):= FloorPlan(ManI,ManJ) & $0F;
	ManI:= Nx;   ManJ:= Ny;	\set man to new location

	if ManI-RefI <  6 then RefI:= RefI - 1;
	if ManI-RefI > 13 then RefI:= RefI + 1;
	if ManJ-RefJ <  6 then RefJ:= RefJ - 1;
	if ManJ-RefJ > 13 then RefJ:= RefJ + 1;
	end;	\MoveMan


begin	\DoMove
Nx:= ManI;   Ny:= ManJ;
Bx:= ManI;   By:= ManJ;
case Key of	\determine possible new locations for man (N) and box (B)
  $48:	[Ny:= ManJ-1; By:= Ny-1];	\up
  $4B:	[Nx:= ManI-1; Bx:= Nx-1];	\left
  $4D:	[Nx:= ManI+1; Bx:= Nx+1];	\right
  $50:	[Ny:= ManJ+1; By:= Ny+1]	\down
other	[];

\Save state so the move can be undone.
\When moving the man or box, preserve background color of floor or dock.
Push(FloorPlan(ManI,ManJ));		\save man and his background
Push(FloorPlan(Nx,Ny));			\save contents at new man location
Push(FloorPlan(Bx,By));			\save contents where box might go
Push(ManI);				\save location of man
Push(ManJ);
Push(Nx);				\save new location of man
Push(Ny);
Push(Bx);				\save new location of box
Push(By);
Push(RefI);				\save scroll info
Push(RefJ);

MoveCtr:= MoveCtr + 1;	\increment move counter (even if move is illegal)

if FloorPlan(Nx,Ny)=$01\floor\ ! FloorPlan(Nx,Ny)=$02\dock\ then
	MoveMan
else if (FloorPlan(Nx,Ny) & $F0) = $10\box\ then
    begin					\attempting to move a box
    if FloorPlan(Bx,By)=$01\floor\ ! FloorPlan(Bx,By)=$02\dock\ then
	begin					\move a box
	if FloorPlan(Bx,By)=$02\dock\ & FloorPlan(Nx,Ny)=$11\box on floor\ then
		PlaySound("DOCKED.WAV")		\box moved from floor to dock
	else	PlaySound("SLIDE.WAV");
	FloorPlan(Bx,By):= (FloorPlan(Bx,By) & $0F) ! $10;     \display box at B
	MoveMan;
	end
    else PlaySound("GRUNT7.WAV");	\2 boxes or box and wall
    end
else PlaySound("GRUNT2.WAV");		\hit wall
end;	\DoMove

\-------------------------------------------------------------------------------

proc	LoadSprites;		\Load sprite images from .BMP file
int	Han,			\file handle
	Width, Height,		\dimensions of entire image in .BMP file, pixels
	R, G, B,		\red, green, blue
	X, Y,			\screen coordinates (pixels)
	Y320,			\Y * 320
	I, K, T;		\indexes and tempory scratch


	proc	LoadSp(X0, Y0, W, H, Sp);	\Load a sprite
	int	X0, Y0,		\coordinates in ScreenBuf to get sprite from
		W, H;		\width and height (pixels)
	char	Sp;		\sprite to load
	int	K, X, Y;
	begin
	Sp(0):= W;
	Sp(1):= H;
	K:= 2;
	for Y:= Y0, Y0+H-1 do
	    for X:= X0, X0+W-1 do
		begin
		Sp(K):= ScreenBuf(0, X+Y*320);
		K:= K + 1;
		end;
	end;	\LoadSp


begin	\LoadSprites
\Read in a 256-color .BMP file
Trap(false);			\don't trap "file not found" error; we'll do it
Han:= FOpen("SPRITES.BMP", 0);	\open file for input
if Geterr \#0\ then Exit("SPRITES.BMP not found");
Trap(Debug);			\turn traps back on if debug mode
FSet(Han, ^I);			\set device 3 to handle
Openi(3);

for Y:= 0, 17 do X:= Chin(3);	\skip unused header info
Width:= Chin(3) + Chin(3)<<8;	\0..32764 (ample range)
for Y:= 0, 2-1 do X:= Chin(3);	\skip
Height:= Chin(3) + Chin(3)<<8;	\0..32767 (ample range)
for Y:= 24, 53 do X:= Chin(3);	\skip

K:= 0;
port($3C8):= 0;			\set color registers
for I:= 0, 255 do
	begin
	B:= Chin(3)>>2;
	G:= Chin(3)>>2;
	R:= Chin(3)>>2;
	T:= Chin(3);
	port($3C9):= R;
	port($3C9):= G;
	port($3C9):= B;

	ColorReg(K):= R;	\save copy for Fade routine
	ColorReg(K+1):= G;
	ColorReg(K+2):= B;
	K:= K + 3;
	end;

\Load .BMP image into ScreenBuf
for Y:= -(Height-1), 0 do	\.BMP files are upside down
	begin
	Y320:= 320 * -Y;
	for X:= 0, Width-1 do
		ScreenBuf(0, X+Y320):= Chin(3);
	end;
FClose(Han);			\close handle so it can be used again

LoadSp( 0,136,255, 64, SpTitle); \grab individual sprites
LoadSp( 0,  0, 32, 32, SpFloor);
LoadSp(32,  0, 32, 32, SpDock);
LoadSp(64,  0, 32, 32, SpWall);
LoadSp(96,  0, 32, 32, SpBox);
LoadSp( 0, 32, 32, 32, SpManS);
LoadSp(32, 32, 32, 32, SpManN);
LoadSp(64, 32, 32, 32, SpManW);
LoadSp(96, 32, 32, 32, SpManE);
end;	\LoadSprites

\-------------------------------------------------------------------------------

proc	GetStartingLevel;	\Display title screen and get starting level num
int	I, Ch;
begin
Trap(false);			\don't trap "file not found" error; we'll do it
LevelHandle:= FOpen("LEVELS.TXT", 0);	\open file for input
if Geterr \#0\ then Exit("LEVELS.TXT not found");
Trap(Debug);			\restore traps if not debug mode
FSet(LevelHandle, ^i);		\read levels with small buffer to avoid conflict
Openi(3);

Erase(ScreenBuf, 0\black\);	\show title
DrawSprite(32, 32, SpTitle);
Blit(ScreenBuf(0), 0, $A000, 0, 64*1000);	\copy image to screen

Attrib($20);			\bright red
Cursor(7, 17);
Text(6, "Enter starting level: __");
loop	begin
	Attrib($0F);		\bright white
	Cursor(7+22, 17);

	Level:= 0;		\Level:= Intin(6); doesn't do everything we want
	Openi(6);
	repeat	Ch:= Chin(6);
		if Ch>=^0 & Ch<=^9 then
			Level:= Level*10 + Ch - ^0;
		if Ch=$1B\Esc\ ! Ch=$03\Ctrl-C\ ! Ch=0\Break\ then Exit(" ");
	until Ch<^0 ! Ch>^9;

	if Level>=1 & Level<=36 then quit;	\check for valid range
	Cursor(7, 19);
	Attrib($20);
	Text(6, "Please select 1 thru 36");
	repeat until Chkkey;	\wait for keystroke
	Cursor(7+22, 17);	\blank erroneous entry
	Text(6, "__     ");
	end;

\Skip to selected level in the input file
for I:= 1, Level do
	repeat	Ch:= Chin(3);
	until Ch = ^>;

\PlaySound("MENU3.WAV");
Fade(true, 0);			\fade out
end;	\GetStartingLevel

\===============================================================================

begin	\Main
ScreenBuf(0):= Malloc(320/16*200);	\get a 64000-byte buffer
TrapC(true);	\don't allow a Ctrl-C or Ctrl-Breal to bypass the Exit routine
OpenSound;

SetVid($13);				\set 320x200x256 graphic mode
LoadSprites;				\load sprites and set up color registers
GetStartingLevel;

loop	begin	\level loop
	StkPtr:= 0;			\initialize for Undo command
	MoveCtr:= 0;

	LoadFloorPlan;			\from input file
	DrawFloorPlan;
	Fade(false, 0);			\fade in new level's floor plan

	loop	begin	\move loop
		Key:= CallInt($16,0) >> 8;	\read keyboard scan code
		case Key of
		  $00, $01, $2E:		\Ctrl-Break, Esc, and Ctrl-C
			begin			\ (or C) = quit
			Fade(true, 0);		\fade out
			SetVid($03);		\restore 80-column text display
			exit;
			end;
		  $0E:	Undo;			\Backspace = undo previous move
		  $48, $4B, $4D, $50: DoMove
		other	[];			\ignore illegal keystrokes

		DrawFloorPlan;

		if Solved then
			begin
			PlaySound("SOLVED.WAV");
			\Wait for keystroke, read and discard character
			Key:= CallInt($16, 0);
			quit;
			end;
		end;	\move loop

	Fade(true, 0);				\fade out
	Sound(0, 9, 0);				\pause about 1/2 second
	Openi(1);				\clear any keystroke
	Level:= Level + 1;			\next level
	end;	\level loop
end;	\Main
