{****************************************************************************
 * Extract                                                                  *
 * Written by Paul Gilbert                                                  *
 * s9406702@yallara.cs.rmit.edu.au                                          *
 *                                                                          *
 * This program is designed to extract files from an Apple //e disk image   *
 * file. It can extract (1) text files or (2) binary files.                 *
 ****************************************************************************}
Program Extract;

const
  largefile : boolean = false;

type
  sector = array [0..255] of byte;
  position = record
    track: 0..34;
    sector: 0..15;
  end;

  directory_entry = record
    track_num, sector_num: byte;
    file_type: byte;
    filename: array [1..30] of byte;
    sector_count: word;
  end;

  directory_sector = record
    unused: byte;
    next_track, next_sector: byte;
    unused2: array [1..8] of byte;
    files: array [1..7] of directory_entry;
  end;

var
  f: file of sector;
  sectorlist: position;
  list: sector;
  pos: position;

  buffer: sector;

{ Lists the contents of an Apple image file's directory. }
procedure list_contents(filename: string);
const
  buf: ^directory_sector = @buffer;
var
  ctr: byte;
  i, j: integer;
begin
  pos.track := $11; pos.sector := $F; { Set to directory start }
  ctr := 0;
  Assign(f, filename);
  Reset(f);
  repeat
    Seek(f, pos.track*16+pos.sector);
    Read(f, buffer);
    for i := 1 to 7 do begin
      { Print out the seven directory entries for the sector }
      if ((buf^.files[i].track_num = 0) and (buf^.files[i].sector_num = 0)) then
      begin
        { Reached the end of the list, so exit }
        Close(f);
        Exit;
      end;
      Inc(ctr);
      write(ctr, '  ');          { Write file number }
      for j := 1 to 30 do write(Chr(buf^.files[i].filename[j] mod 128));
      writeln;
    end;
    pos.track := buf^.next_track; pos.sector := buf^.next_sector;
  until ((pos.track = 0) and (pos.sector = 0));
end;

procedure extract_file(extype: byte; filename:string; filenum: integer;
                       outfilename: string);
const
  buf: ^directory_sector = @buffer;
var
  ctr, ctr2: integer;
  fout: file of sector;
begin
  Assign(fout, outfilename);
  Rewrite(fout);
  Assign(f, filename);
  Reset(f);
  pos.track := $11; pos.sector := $F;    { Set to directory start }
  Seek(f, pos.track*16+pos.sector);
  Read(f, buffer);

  { Count up to the correct sector in directory of file number }
  ctr := filenum;
  while (ctr > 7) do
  begin
    Dec(ctr, 7);
    pos.track := buf^.next_track; pos.sector := buf^.next_sector;
    Seek(f, pos.track*16+pos.sector);
    Read(f, buffer);
  end;

  { Read in the file's sector list }
  sectorlist.track  := buf^.files[ctr].track_num;
  sectorlist.sector := buf^.files[ctr].sector_num;
  Seek(f, sectorlist.track*16+sectorlist.sector);
  Read(f, list);

  { Begin reading in the sectors and writing out to file }
  ctr := $C;
  repeat
    { Obtain track/sector of next section of file }
    pos.track := list[ctr]; pos.sector := list[ctr+1];
    if ((pos.track = 0) and (pos.sector = 0)) then
    begin
      { Found an empty entry. If the list's next link is 0/0, and the rest of
        the track/sectors in the list are 0/0, then we've reached the end of
        the file. Otherwise, it's a random access file "empty" section, so
        just store a sector of zeroes to output file.                        }
      if ((list[1] = 0) and (list[2] = 0)) then
      begin
        { No next link, so check to see if 0/0 continues to end of list }
        ctr2 := ctr;
        while ((list[ctr2] = 0) and (list[ctr2+1] = 0) and (ctr2 <= $FE)) do
          Inc(ctr2, 2);
        if (ctr2 = $100) then
        begin
          { Zeroes were found to end of list, so is end of file. Exit }
          Close(f); Close(fout);
          Exit;
        end;
      end;

      { If it reaches here, it means that it was just an "empty" sector, }
      { so represent it in output file with a "sector" of zeroes         }
      FillChar(buffer, $100, 0);
      Write(fout, buffer);
    end
    else
    begin
      { A track and sector specified. Read it in }
      Seek(f, pos.track*16+pos.sector);
      Read(f, buffer);
      if (extype = 1) then
      begin
        { Apple uses bit 7 set [it being related to it's method of storing a
          set of normal, flashing, and inverse characters in the one 256
          character set. For text files therefore, we must strip it off.     }
        for ctr2 := 0 to 255 do
          buffer[ctr2] := buffer[ctr2] and $7F;
      end;

      { Write out sector to the output file }
      Write(fout, buffer);
    end;

    Inc(ctr, 2);                { Move to next track/sector pair }
    if (ctr = $100) then
    begin
      { Reached the end of the current list sector }
      ctr := $C;
      sectorlist.track := list[1]; sectorlist.sector := list[2];
      if ((sectorlist.track = 0) and (sectorlist.sector = 0)) then
      begin
        { End of file reached. }
        Close(f); Close(fout);
        Exit;
      end;
      { Still more of file, so read in next sector of file's sector list }
      Seek(f, sectorlist.track*16+sectorlist.sector);
      Read(f, list);
    end;
  until (0=1);
end;

var
  t: byte;
  v1, v2, code: integer;

begin
  if (((ParamCount=1) and (ParamStr(1)='/?')) or (ParamCount=0)) then
  begin
    writeln('Apple // program extractor.');
    writeln('Extracts a file from a 143Kb image file of an Apple diskette.');
    writeln('Format: EXTRACT filename.dsk   will list the contents.');
    writeln('        EXTRACT {1|2} filename.dsk Apple-file-number output-filename');
    writeln('            where 1 extracts a text file, and 2 extracts a binary,');
    writeln('            Apple-file-number is the number printed by the catalogue');
    writeln('            lister, and output-filename is the output file.');
    Halt;
  end;

  if (ParamCount = 1) then
  begin
    list_contents(ParamStr(1));
    Halt;
  end;

  { Extract file }
  Val(ParamStr(1), v1, code);
  Val(ParamStr(3), v2, code);
  extract_file(v1, ParamStr(2), v2, ParamStr(4));
end.