\MJ.XPL		FEB-13-98
\Mah Jongg Solitaire Game by Loren Blaney

\Tiles are initially set up in a pile called a "dragon". Pairs of
\ matching tiles are removed from the sides of the dragon, and you win when
\ you remove all the tiles. The tiles are numbered 1-144. 0 means no tile,
\ and 145 is a blank tile. There are 36 groups each containing four
\ matching tiles (1-4 match, 5-8 match, and so forth). Most of the groups
\ have four identical tiles, but the first two groups, the seasons and
\ flowers, have different although related images.

inc	C:\CXPL\CODESI;	\Include code definitions for intrinsic routines

int	Cpureg,		\Address of CPU register array (Getreg). Must be first.
	HaveMouse;	\Flag: A mouse driver is installed and the mouse works

eproc	LoadLBM;	\External procedure to load a 640x480x16 .LBM file
ext	ImageSize(X0, Y0, X1, Y1, Z),		\Gets size of image (paragraphs)
	DrawImage(X, Y, BufSeg, Op),		\Draw image in BufSeg at X,Y
	SaveImage(X0, Y0, X1, Y1, Z, BufSeg),	\Copy image at X,Y into BufSeg
	SaveMask(X0, Y0, X1, Y1, Z, MaskBuf);	\Save copy of mask for the image
ext	SetPal;		\Set palette registers from array of 16 values

def	Fudge=16;	\Fudge factor to avoid bug in some mouse drivers which
			\ use the top portion of undisplayed video RAM (lines)

def	ParallaxX=6, ParallaxY=7;	\Amount tiles on upper levels shift
def	DragonX=16, DragonY=5*ParallaxY; \Offset to upper-left corner of dragon
def	TileWidth=42, TileHeight=58;	\Overall tile dimensions (pixels)
def	TileFaceWidth=37, TileFaceHeight=53;	\Tile face dimensions (pixels)

def	CounterX=76, CounterY=1;	\Character coordinates of Counter
def	TimerX=73, TimerY=4;		\Character coordinates of Timer
def	GameX=76, GameY=7;		\Character coordinates of Game number
def	GameDigits=3;			\Maximum number of digits in Game
def	MsgX=61, MsgY=10; 		\Character coordinates of messages

def	ButtonWidth=48, ButtonHeight=16;	\Button size in pixels
	\Button locations in pixels:
def	HintX=640-ButtonWidth, HintY=480-11*ButtonHeight;
def	UndoX=640-ButtonWidth, UndoY=480-9*ButtonHeight;
def	RedoX=640-ButtonWidth, RedoY=480-7*ButtonHeight;
def	UsedX=640-ButtonWidth, UsedY=480-5*ButtonHeight;
def	NewX =640-ButtonWidth, NewY =480-3*ButtonHeight;
def	ExitX=640-ButtonWidth, ExitY=480-1*ButtonHeight;

def	SortX=640-ButtonWidth, SortY=480-7*ButtonHeight;
def	PlayX=640-ButtonWidth, PlayY=480-5*ButtonHeight;

def	BS=$08, CR=$0D, Esc=$1B;		\Control character codes
def	Delete=$53;				\Keyboard scan code

char	Dragon(15,8,5),	\Array to hold tiles in dragon formation (exceptions)
	Palette(16),	\Array: Copy of palette registers (0-15)
	SaveMoveX(144),	\Holds moves for Undo and Redo commands
	SaveMoveY(144),
	SaveMoveZ(144),
	SaveMoveN(144);

int	Counter,	\Number of tiles remaining in the dragon
	FaceBuf,	\Buffer holding mask for face of tile image
	Game,		\Game number (0-999); gives repeatable tile patterns
	GameGiven,	\Flag "Array": A game number was specified
	Hilite1, Hilite2, \Tile numbers to be highlighted by ShowTile (none = 0)
	VidMode,	\Initial video mode to be restored upon exit
	TileBuf(43),	\Array of buffers holding tile images (including blank)
	TileMaskBuf,	\Buffer holding mask for tile image
	TileSize,	\Number of paragraphs required to hold a tile image
	Timer,		\Elapsed time in seconds since beginning of game
	Timer0,		\Used to determine when a full second has elapsed
	UndoCount;	\Counter value at last move actually made (for Redo)

\==============================================================================

proc	WaitKey;	\Wait for a keystroke
begin
Openi(1);
repeat until ChkKey;
end;	\WaitKey



proc	WaitVB;		 \Wait for beginning of next vertical blank
int	SR;
begin
SR:= Peek($40, $63) + $306;	\Port address of status register
if (SR & $39A) # $39A then return;
while Pin(SR, 0) & $08 do;	\Wait for no vertical blank
repeat until Pin(SR, 0) & $08;	\Wait for vertical blank
end;	\WaitVB;



proc	ClearPage2;	\Clear second (partial) page of video memory
seg char VideoMem(1);
int	I;
begin
VideoMem(0):= $A000;
I:= 480*80;		\Start beyond bottom of first page
repeat	VideoMem(0, I):= 0;
	I:= I +1;
until I = 0;
end;	\ClearPage2



func	IntLen(N);	\Return number of digits in N, a decimal integer
int	N;		\Adds 1 for a minus sign if N is negative
int	I;
for I:= if N>=0 then 1 else 2, 20 do
	[N:= N /10;   if N = 0 then return I];



proc	Ctxt(X, Y, Str); \Display a string on device 6 at X, Y
int	X, Y;
char	Str;
begin
Cursor(X, Y);
Text(6, Str);
end;	\Ctxt



proc	SpOut(Dev, N);	\Output N spaces to specified device
int	Dev, N;
int	I;
for I:= 1, N do Chout(Dev, ^ );



proc	JustOut(N, Places);	\Output a right-justified integer
\Always outputs correct value of N regardless of Places
int	N,	\16-bit integer (negative numbers should be filled with spaces)
	Places;	\Size of field in characters (right-justifies)
begin
SpOut(6, Places-IntLen(N));
Intout(6, N);
end;	\JustOut



func	CallInt(Int, AX, BX, CX, DX, BP, DS, ES); \Call software interrupt
int	Int, AX, BX, CX, DX, BP, DS, ES; \(Unused arguments need not be passed)
begin
Cpureg:= Getreg;
Cpureg(0):= AX;
Cpureg(1):= BX;
Cpureg(2):= CX;
Cpureg(3):= DX;
Cpureg(6):= BP;
Cpureg(9):= DS;
Cpureg(11):= ES;
Softint(Int);
return Cpureg(0);		\return AX register
end;	\CallInt



func	GetSec;		\Get seconds in BCD from CMOS time/date chip
begin
CallInt($1A, $0200);
return Cpureg(3)>>8;
end;	\GetSec



func	GetVid;		\Return the current video mode
return CallInt($10, $0F00) & $FF; \Function $0F



func	GetKey;		\Get character from keyboard (wait if necessary)
int	SC, Ch;			\This is a low-level routine with no echo,
begin				\ no Ctrl-C, and no cursor.
SC:= CallInt($16, $0000);	\Function $00
Ch:= SC & $FF;
if Ch = 0 then Ch:= -(SC>>8);	\Return non-ascii chars as negative scan code
return Ch;
end;	\GetKey



func	GetInt(N0, Size, X, Y); \Input an integer; scroll calculator-style
int	N0,	\Initial value
	Size,	\Number of digits in field
	X, Y;	\Display position on screen
int	N,	\Number
	M,	\Maximum number +1
	Ch,	\Digit (character) from keyboard
	First;	\Flag: First digit clears display
begin
M:= 1;						\Create maximum number +1
for N:= 1, Size-1 do M:= M *10;

N:= N0;						\Get initial value
First:= true;
loop	begin
	Cursor(X, Y);				\Display our number
	JustOut(N, Size);

	Ch:= GetKey;
	case CH of
	  BS, -Delete: [N:= 0;   First:= true];
	  Esc:	return N0;			\Return original value
	  CR:	return N			\Return new value
	other
	    if Ch>=^0 & Ch<=^9 then		\Ignore other characters
		begin
		if First then N:= 0		\First digit replaces number
		else	if N < M then N:= N * 10;
		N:= N/10*10 + Ch-^0;
		First:= false;
		end
	end;	\loop
end;	\GetInt

\------------------------------- MOUSE ROUTINES -------------------------------

func OpenMouse;		\Initializes mouse; returns 'false' if it fails
begin			\Pointer is set to center of screen but is hidden
CallInt($21, $3533);	\Make sure mouse vector ($33) points to something
HaveMouse:= false;
if Cpureg(1)=0 & Cpureg(11)=0 then return false;
HaveMouse:= CallInt($33, $0000); \Reset mouse and get status
if HaveMouse then HaveMouse:= true; \(Beware of 'not' operator in 32-bit XPL)
return HaveMouse;		\Return 'false' if failure
end;	\OpenMouse



proc ShowMouse(On);	\Turn mouse pointer on or off
\The video mode should be set before calling this routine.
\This counts the number of times the pointer is turned on or off. If the
\ pointer is turned off twice, it must be turned on twice before it
\ actually goes on. The pointer should be turned off before drawing over
\ it and before your program exits. Setting the video mode will also turn
\ off the pointer.
int	On;	\Flag: True = pointer on; False = pointer off
if HaveMouse then
	CallInt($33, if On then $0001 else $0002);



func GetMousePosition(N); \Return position of specified mouse coordinate
int	N;	\0 = X coordinate; 1 = Y coordinate
\For video modes $0-$E and $13 the maximum coordinates are 639x199, minus
\ the size of the pointer. For modes $F-$12 the coordinates are the same as
\ the pixels. For 80-column text modes divide the mouse coordinates by 8 to
\ get the character cursor position.
begin
if ~HaveMouse then return 0;
CallInt($33, $0003);
return if N then Cpureg(3) else Cpureg(2);
end;	\GetMousePosition



func GetMouseButton(N);	\Return 'true' if specified mouse button is down
int	N;	\Button number: 0 = left; 1 = right (or middle)
begin
if ~HaveMouse then return false;
CallInt($33, $0003);
return if N then (Cpureg(1)&2)=2 else (Cpureg(1)&1)=1;
end;	\GetMouseButton



proc MoveMouse(X, Y);	\Move mouse pointer to X,Y
int	X, Y;
if HaveMouse then
	CallInt($33, $0004, 0, X, Y);

\==============================================================================

proc	Fall;		\Falling screen effect
int	T, L;
begin
ClearPage2;		\Clear out any mouse turds
T:= 1;
loop	begin
	WaitVB;				\Synchronize with display
	L:= T*T;			\Distance = acceleration * time squared
	if L > 338 then quit;		\there are only 338 lines on page 2
	T:= T+1;
	L:= -L *80;			\80 bytes per line
	Pout((L&$FF00)+$0C, $3D4, 1);	\VGA Start Address Register, high byte
	Pout(L<<8 + $0D, $3D4, 1);	\Low byte (yes, it's backwards)
	end;

Clear;
L:= 0;					\Restore VGA Start Address Register
Pout((L&$FF00)+$0C, $3D4, 1);		\Start address, high byte
Pout(L<<8 + $0D, $3D4, 1);		\Low byte
for T:= 1, 10 do WaitVB;
end;	\Fall



proc	Exit;		\Make a graceful exit
begin
Fall;
SetVid(VidMode);	\Restore original video mode (clears screen)
exit;
end;	\Exit



proc	ShowCounter;	\Show the Counter, the number of tiles in the dragon
begin
Cursor(CounterX, CounterY);
JustOut(Counter, 3);
end;	\ShowCounter



proc	ShowTimer;	\Show number of seconds since beginning of game. Min:Sec
int	N;
begin
Cursor(TimerX, TimerY);
N:= Timer/60;				\Min
JustOut(N, 3);
Chout(6, ^:);
N:= Rem(Timer/60);			\Sec
if N < 10 then Chout(6, ^0);		\Leading zero
Intout(6, N);
end;	\ShowTimer



proc	ShowGame;	\Show the Game number
begin
Cursor(GameX, GameY);
JustOut(Game, GameDigits);
end;	\ShowGame



proc	ShowStatus;	\Show status
begin
Ctxt(75, 0, "Tiles");
ShowCounter;
Ctxt(75, 3, "Time");
ShowTimer;
Ctxt(75, 6, "Game");
ShowGame;
end;	\ShowStatus



proc	RunTimer;	\Update timer display once per second
begin
if Timer0 # GetSec then
	begin
	Timer0:= GetSec;
	Timer:= Timer +1;
	ShowMouse(false);
	ShowTimer;
	ShowMouse(true);
	end;
end;	\RunTimer



proc	WaitMouseButton; \Wait for left mouse button to be pressed
begin
ShowMouse(true);
repeat  RunTimer;
until not GetMouseButton(0);		\Wait for release
repeat	RunTimer;
	if Chkkey then
	    if GetKey = Esc then Exit;
until GetMouseButton(0);		\Wait for mouse button
ShowMouse(false);
end;	\WaitMouseButton



proc	Wait(T);	\Wait T 18ths of a second or until mouse click
int	T;
begin
loop	begin
	if T <= 0 then return;
	if not GetMouseButton(0) then quit;
	Sound(0, 1, 1);
	T:= T -1;
	end;
loop	begin
	if T <= 0 then return;
	if GetMouseButton(0) then quit;
	Sound(0, 1, 1);
	T:= T -1;
	end;
end;	\Wait



proc	ShowTile(N, X, Y); \Display tile image N at X,Y
int	N,	\Tile number (1-145)
	X, Y;	\Graphic screen coordinates
int	M;
begin
M:= N -1;
if M >= 8 then M:= (M-8)/4 +8;
DrawImage(X, Y, TileMaskBuf, 1\AND\);	\Mask out a hole
DrawImage(X, Y, TileBuf(M), 2\OR\);	\Fill in the image
case N of Hilite1, Hilite2: DrawImage(X, Y, FaceBuf, 3\XOR\) other [];
end;	\ShowTile;

\------------------------------------------------------------------------------

proc	YouWin;		\Display winning screen
def	LogoX=0, LogoY=50;	\Offset to upper-left corner of logo
def	ParallaxX=5;		\(Tweaked to make logo fit on screen)
int	X, Y, Z, X1, Y1;
char	Logo(15,8,5),		\Array to hold tiles in logo formation
	A;
begin
\Set up Logo:

for Z:= 0, 5-1 do			\Initialize Logo array
    for Y:= 0, 8-1 do
	for X:= 0, 15-1 do
	    Logo(X,Y,Z):= 0;

	\012345678901234
A:= [	"*   *  **  *  * ",	\0
	" * *  *  * *  * ",	\1
	"  *   *  * *  * ",	\2
	"  *   *  * *  * ",	\3
	"  *    **   **  "];	\4

for Z:= 0, 4 do
    for X:= 0, 14 do
	if A(Z,X) = ^* then Logo(X,7,4-Z):= 1;

	\012345678901234
A:= [	"*   *  *  *   * ",	\0
	"*   *  *  **  * ",	\1
	"* * *  *  * * * ",	\2
	"** **  *  *  ** ",	\3
	"*   *  *  *   * "];	\4

for Z:= 0, 4 do
    for X:= 0, 14 do
	if A(Z,X) = ^* then Logo(X,3,4-Z):= 1;

\Show Logo:

Clear;					\Erase buttons that no longer work
ShowStatus;				\Restore status (and stop the clock)
\Draw the tiles using the painter's algorithm (start at the back)
for Z:= 0, 5-1 do
    for X:= -14, 0 do
	for Y:= 0, 8-1 do
	    begin
	    if Logo(-X,Y,Z) > 0 then	\Is there a tile at this position?
		begin
		\Convert tile coordinates into graphic coordinates
		X1:= -X*TileFaceWidth + Z*ParallaxX*3 + LogoX;
		Y1:= Y*TileFaceHeight - Z*ParallaxY*6 +LogoY;
		ShowTile(Ran(144)+1, X1, Y1);
		Sound(0,1,1);		\Delay for effect

		X1:= X1 + ParallaxX;
		Y1:= Y1 - 2*ParallaxY;
		ShowTile(Ran(144)+1, X1, Y1);
		Sound(0,1,1);		\Delay

		X1:= X1 + ParallaxX;
		Y1:= Y1 - 2*ParallaxY;
		ShowTile(Ran(144)+1, X1, Y1);
		Sound(0,1,1);		\Delay
		end;
	    end;

\Start a new game:

Attrib($23);				\Display "New" button
Ctxt(NewX/8,  NewY/16,  " New  ");
Attrib($02);

ShowMouse(true);
repeat until not GetMouseButton(0);	\Wait for release
Openi(1);
loop	begin				\Wait for mouse button or keystroke
	if Chkkey then
	    if GetKey = Esc then Exit else quit;
	if GetMouseButton(0) then quit;
	end;
ShowMouse(false);

Fall;
Restart;
end;	\YouWin

\------------------------------------------------------------------------------

proc	ShowUsed;	\Show the tiles that have been removed from the dragon
int	I, X, Y, S, N, G, MX, MY;
def	X0=4, Y0=0;	\Offset to upper left corner of display
char	Occupied(36);	\Flags: is a tile group already occupied?
begin
S:= false;		\Sort button toggles
loop	begin
	Clear;
	ShowStatus;

	if S then
		begin				\Show tiles in sorted order
		for G:= 0, 36-1 do Occupied(G):= false;
		for I:= -(144-1), -Counter do
			begin
			N:= SaveMoveN(-I);
			G:= (N-1) /4;		\Group (0..35)
			Y:= G /6 *80 + Y0;
			X:= Rem(G/6) * 2*48 + X0;
			if Occupied(G) then X:= X + 48;
			Occupied(G):= true;
			ShowTile(N, X, Y);
			I:= I +1;		\Get paired tile
			N:= SaveMoveN(-I);	\Put it on top
			ShowTile(N, X+ParallaxX, Y+2*ParallaxY);
			end;
		end
	else	begin				\Show tiles in order played
		X:= X0;   Y:= Y0;
		for I:= -(144-1), -Counter do
			begin
			ShowTile(SaveMoveN(-I), X, Y);
			I:= I +1;
			ShowTile(SaveMoveN(-I), X+ParallaxX, Y+2*ParallaxY);
			X:= X + 48;
			if X >= 576 then
				[X:= X0;   Y:= Y + 80];
			end;
		end;
	S:= not S;

	Attrib($23);
	Ctxt(SortX/8, SortY/16, " Sort ");	\Make a Sort button
	Ctxt(PlayX/8, PlayY/16, " Play ");	\Make a button to return to play
	Attrib($02);

	WaitMouseButton;
	MX:= GetMousePosition(0);
	MY:= GetMousePosition(1);
	if MY>=SortY & MY<SortY+ButtonHeight & MX>=SortX & MX<SortX+ButtonWidth
		then \loop back\ else quit;
	end;
end;	\ShowUsed

\------------------------------------------------------------------------------

proc	ShowDragon;	\Display all the tiles remaining in the dragon
int	X, Y, Z, X1, Y1, N;
begin
Clear;
ShowStatus;

Attrib($23);				\Display the buttons
Ctxt(NewX/8,  NewY/16,  " New  ");
Ctxt(HintX/8, HintY/16, " Hint ");
Ctxt(UndoX/8, UndoY/16, " Undo ");
Ctxt(RedoX/8, RedoY/16, " Redo ");
Ctxt(UsedX/8, UsedY/16, " Used ");
Ctxt(ExitX/8, ExitY/16, " Exit ");
Attrib($02);

\Draw the tiles using the painter's algorithm (start at the back)
for Z:= 0, 4 do				\Show all the tiles remaining
    for X:= -14, 0 do
	for Y:= 0, 7 do
	    begin
	    N:= Dragon(-X,Y,Z);		\Get tile
	    if N > 0 then		\If there is a tile then display it
		begin
		\Convert tile coordinates into graphic coordinates
		X1:= -X*TileFaceWidth + Z*ParallaxX + DragonX;
		Y1:= Y*TileFaceHeight - Z*ParallaxY +DragonY;

		\Deal with shifted tiles
		if Y = 3 then
		    begin
		    \Tiles on ends are shifted down half a tile
		    if X=0 ! X=-13 ! X=-14 then
			Y1:= Y1 + TileFaceHeight/2;
		    \Top tile is shifted down and to the left
		    if X=-7 & Z=4 then
			begin
			Y1:= Y1 + TileFaceHeight/2;
			X1:= X1 - TileFaceWidth/2;
			end;
		    end;

		\If there is a tile above then don't show tile face
		if Z < 3 then
		    if Dragon(-X,Y,Z+1)\#0\ then
			N:= 145;	\Blank tile

		ShowTile(N, X1, Y1);
		end;
	    end;
end;	\ShowDragon

\------------------------------------------------------------------------------

proc	ShowTilesNear(X0, Y0);	\Display tiles near X0,Y0
int	X0, Y0;
def	OffsetY = 480 + Fudge + 5*ParallaxY;
int	X, Y, X1, Y1, Z, N,
	FX, FY, TX, TY, W, H,	\From and To coordinates, Width and Height
	CopyBuf;
begin
\Clear background in upper-left corner on page 2
for X:= 0, 2 do
    for Y:= 0, 2 do
	begin
	X1:= X*TileFaceWidth;
	Y1:= Y*TileFaceHeight + OffsetY;
	DrawImage(X1, Y1, TileMaskBuf, 1\AND\);
	end;

\Draw the tiles using the painter's algorithm (start at the back).
\First, draw on page 2 then copy portion back to page 1.
for Z:= 0, 4 do			\For all levels
    for X:= -(X0+1), -(X0-1) do	\For X0+1 downto X0-1 do...
	for Y:= Y0-1, Y0+1 do
	    if -X>=0 & -X<=14 & Y>=0 & Y<=7 then	\In bounds?
		begin
		N:= Dragon(-X,Y,Z);	\Get tile
		if N > 0 then		\If there is a tile then display it
		    begin
		    \Convert tile coords (-X,Y) into graphic coords (X1,Y1)
		    X1:= (-X-X0+1)*TileFaceWidth + Z*ParallaxX;
		    Y1:= (Y-Y0+1)*TileFaceHeight - Z*ParallaxY + OffsetY;
		    if Y = 3 then
			begin
			\Tiles on ends are shifted down half a tile
			if -X=0 ! -X=13 ! -X=14 then
			    Y1:= Y1 + TileFaceHeight/2;
			\Top tile is shifted down and to the left
			if -X=7 & Z=4 then
			    begin
			    Y1:= Y1 + TileFaceHeight/2;
			    X1:= X1 - TileFaceWidth/2;
			    end;
			end;

		    \For speed, don't draw tile if it's completely hidden
		    if not( Z<3 & Dragon(-X,Y,Z+1)#0 &    \Above
			    -X>0 & Dragon(-X-1,Y,Z)#0 &	  \On the left
			    Y<7 & Dragon(-X,Y+1,Z+1)#0 &  \In front and above
			    Dragon(-X-1,Y+1,Z+1)#0 ) then \Front, left, above
				ShowTile(N, X1, Y1);
		    end;
		end;

\Copy image to page 1
FX:= TileFaceWidth;
FY:= TileFaceHeight + OffsetY - 5*ParallaxY;
TX:= X0*TileFaceWidth + DragonX;
TY:= Y0*TileFaceHeight + DragonY - 5*ParallaxY;
W:= TileWidth + 4*ParallaxX;	       \Don't glitch when removing leftmost tile
H:= TileHeight*3/2 + 4+4*ParallaxY;    \Allow for tiles shifted down

\Copy a rectangular area from one part of video RAM to another
CopyBuf:= Malloc(ImageSize(FX, FY, FX+W-1, FY+H-1, $0F));
SaveImage(FX, FY, FX+W-1, FY+H-1, $0F, CopyBuf);
DrawImage(TX, TY, CopyBuf, 0\MOV\);	\(masking is unnecessary)
Release(CopyBuf);
end;	\ShowTilesNear

\------------------------------------------------------------------------------

proc	GiveHint;	\Highlight pairs of tiles that can be removed
int	I, J, X, Y, Z, MX, MY;
char	TileX(144), TileY(144), TileZ(144), TileN(144);


	proc	Sort(N);		\Shell sort Tile arrays
	int	N;			\Number of elements in each array (size)
	int	J, Gap, I, JG, T;
	begin
	Gap:= N /2;
	while Gap > 0 do
		begin
		for I:= Gap, N-1 do
			begin
			J:= I -Gap;
			loop  begin
			      if J < 0 then quit;
			      JG:= J +Gap;
			      if TileN(J) <= TileN(JG) then quit;
			      T:= TileN(J); TileN(J):= TileN(JG); TileN(JG):= T;
			      T:= TileX(J); TileX(J):= TileX(JG); TileX(JG):= T;
			      T:= TileY(J); TileY(J):= TileY(JG); TileY(JG):= T;
			      T:= TileZ(J); TileZ(J):= TileZ(JG); TileZ(JG):= T;
			      J:= J -Gap;
			      end;
			end;
			Gap:= Gap /2;
		end;
	end;	\Sort


begin	\GiveHint
\Scan for and make a list of the unblocked tiles
I:= 0;
for Z:= 0, 4 do
    for Y:= 0, 7 do
	for X:= 0, 14 do
	    begin
	    if Dragon(X,Y,Z)\#0\ then			\Tile exists
		if not ( Z<4 & Dragon(X,Y,Z+1)#0 !	\Is there a tile above?
		    X>0 & Dragon(X-1,Y,Z)#0 & 		\Tile on the left and
		    X<14 & Dragon(X+1,Y,Z)#0 !		\ tile on the right?
		    Z=3 & Dragon(7,3,4)#0 !		\Top tile blocks 4 tiles
		    X=1 & Y=4 & Dragon(0,3,0)#0 !	\Shifted tiles block 2
		    X=12 & Y=4 & Dragon(13,3,0)#0 ) then 
			begin				\Tile is not blocked
			TileX(I):= X;			\Put it in the list
			TileY(I):= Y;
			TileZ(I):= Z;
			TileN(I):= Dragon(X,Y,Z);
			I:= I +1;
			end;
	    end;

Sort(I);						\Sort the list

\Show doubles
for J:= 0, I-2 do
	begin
	if (TileN(J)-1)/4 = (TileN(J+1)-1)/4 then
		begin
		Hilite1:= TileN(J);   Hilite2:= TileN(J+1);	\Highlight
		ShowTilesNear(TileX(J), TileY(J));
		ShowTilesNear(TileX(J+1), TileY(J+1));
		WaitMouseButton;
		Hilite1:= 0;   Hilite2:= 0;
		ShowTilesNear(TileX(J), TileY(J));		\Unhighlight
		ShowTilesNear(TileX(J+1), TileY(J+1));

		\If not Hint button then return
		MX:= GetMousePosition(0);
		MY:= GetMousePosition(1);
		if not( MY>=HintY & MY<HintY+ButtonHeight &
			MX>=HintX & MX<HintX+ButtonWidth ) then return;
		end;
	end;
Ctxt(MsgX, MsgY, "      No more moves");
Wait(30);
end;	\GiveHint

\------------------------------------------------------------------------------

proc	UndoMove;	\Undo previous moves
int	I, X, Y, Z;
begin
if Counter < 144 then		\If not all the tiles are in the dragon
	begin
	for I:= 0, 1 do		\Put back the last two tiles
		begin
		X:= SaveMoveX(Counter);
		Y:= SaveMoveY(Counter);
		Z:= SaveMoveZ(Counter);
		Dragon(X,Y,Z):= SaveMoveN(Counter);
		Counter:= Counter +1;
		ShowTilesNear(X,Y);
		end;
	ShowCounter;
	end;
end;	\UndoMove



proc	RedoMove;	\Redo "undoed" moves
int	I, X, Y, Z;
begin
if Counter > UndoCount then	\If the move has been undone
	begin
	for I:= 0, 1 do		\Remove the last two tiles
		begin
		Counter:= Counter -1;
		X:= SaveMoveX(Counter);
		Y:= SaveMoveY(Counter);
		Z:= SaveMoveZ(Counter);
		Dragon(X,Y,Z):= 0;
		ShowTilesNear(X,Y);
		end;
	ShowCounter;
	end;
end;	\RedoMove



proc	GetGame;	\Get Game number
int	I;
begin
Attrib($23);			\Highlight
I:= GetInt(Game, GameDigits, GameX, GameY);
Attrib($02);			\Unhighlight
ShowGame;

if I # Game then		\If player wants a different game
	begin
	Game:= I -1;		\Because of increment in Main
	if Game < 0 then Game:= 999;	\Handle wrap around
	GameGiven(0):= true;
	Fall;
	Restart;
	end;
end;	\GetGame

\------------------------------------------------------------------------------

func	Confirm(X, Y, Lab);	\Confirm a button press
\Returns 'true' if a button was pressed a second time
int	X, Y;		\Coordinates of button (in pixels)
char	Lab;		\Button label
int	MX, MY;
begin
Attrib($23);
Ctxt(X/8, Y/16, "  OK  ");	\Change button label to "OK"
Attrib($02);

WaitMouseButton;
MX:= GetMousePosition(0);
MY:= GetMousePosition(1);

Attrib($23);			\Restore button label
Ctxt(X/8, Y/16, Lab);
Attrib($02);

return MX>=X & MX<X+ButtonWidth & MY>=Y & MY<Y+ButtonHeight;
end;	\Confirm



proc	GetMove;	\Player selects two matching tiles or presses a button
int	X, Y, Z, X1, Y1, Z1;


	func	Select;		\Select a tile
	int	MX, MY;
	begin
	loop	begin
		Ctxt(MsgX, MsgY, "                   ");
		\Select tile with mouse
		WaitMouseButton;
		MX:= GetMousePosition(0);
		MY:= GetMousePosition(1);

		case of					\Was a button pressed?
		  MX>=HintX & MX<HintX+ButtonWidth &
		  MY>=HintY & MY<HintY+ButtonHeight:
			GiveHint;
		  MX>=UndoX & MX<UndoX+ButtonWidth &
		  MY>=UndoY & MY<UndoY+ButtonHeight:
			UndoMove;
		  MX>=RedoX & MX<RedoX+ButtonWidth &
		  MY>=RedoY & MY<RedoY+ButtonHeight:
			RedoMove;
		  MX>=UsedX & MX<UsedX+ButtonWidth &
		  MY>=UsedY & MY<UsedY+ButtonHeight:
			[ShowUsed;   ShowDragon];
		  MX>=NewX & MX<NewX+ButtonWidth &
		  MY>=NewY & MY<NewY+ButtonHeight:
			if Confirm(NewX, NewY, " New  ") then [Fall;   Restart];
		  MX>=ExitX & MX<ExitX+ButtonWidth &
		  MY>=ExitY & MY<ExitY+ButtonHeight:
			if Confirm(ExitX, ExitY, " Exit ") then Exit;
		  MX/8>=GameX & MX/8<GameX+GameDigits &
		  MY/16>=GameY & MY/16<GameY+1:
			GetGame
		other	quit;
		end;

	\Is there a tile here?
	Z:= 4;
	loop	begin
		\Get tile coordinates (X,Y) from mouse position (MX,MY)
		X:= (MX - DragonX - 6 - Z*ParallaxX) / TileFaceWidth;
		if X < 0 then X:= 0;
		if X > 14 then X:= 14;
		Y:= (MY - DragonY + Z*ParallaxY) / TileFaceHeight;
		if Y < 0 then Y:= 0;
		if Y > 7 then Y:= 7;

		\Handle shifted tiles
		if Y = 4 then
			case X of 0, 13, 14: Y:= 3 other [];
		if Z = 4 then
			begin
			if X = 6 then X:= 7;
			if Y = 4 then Y:= 3;
			end;

		if Dragon(X,Y,Z) # 0 then quit;
		if Z = 0 then return false;
		Z:= Z -1;
		end;

	if X>0 & X<14 then
	    if Dragon(X-1,Y,Z)#0 & Dragon(X+1,Y,Z)#0 !
		Z=3 & Dragon(7,3,4)#0 !
		X=1 & Y=4 & Dragon(0,3,0)#0 !
		X=12 & Y=4 & Dragon(13,3,0)#0 then 
		begin
		Ctxt(MsgX, MsgY, "    Tile is blocked");
		Wait(30);
		return false;
		end;
	return true;
	end;	\Select


begin	\GetMove
Hilite1:= 0;   Hilite2:= 0;
repeat until Select;			\Select first tile
Hilite1:= Dragon(X,Y,Z);
ShowTilesNear(X,Y);			\Highlight selected tile
X1:= X;   Y1:= Y;   Z1:= Z;

repeat until Select;			\Select second tile
if X=X1 & Y=Y1 & Z=Z1 then		\Same as the first tile?
	begin
	Hilite1:= 0;
	ShowTilesNear(X1,Y1);		\Unhighlight deselected tile
	return;
	end;

if (Dragon(X,Y,Z)-1)/4 # (Dragon(X1,Y1,Z1)-1)/4 then
	begin
	Ctxt(MsgX, MsgY, "   Tiles must match");
	Wait(30);
	Hilite1:= 0;
	ShowTilesNear(X1,Y1);		\Unhighlight deselected tile
	return;
	end;

Counter:= Counter -1;			\Remove selected tiles
SaveMoveX(Counter):= X;			\Save move for Undo command
SaveMoveY(Counter):= Y;
SaveMoveZ(Counter):= Z;
SaveMoveN(Counter):= Dragon(X,Y,Z);
Dragon(X,Y,Z):= 0;

Counter:= Counter -1;
SaveMoveX(Counter):= X1;
SaveMoveY(Counter):= Y1;
SaveMoveZ(Counter):= Z1;
SaveMoveN(Counter):= Dragon(X1,Y1,Z1);
Dragon(X1,Y1,Z1):= 0;

Hilite1:= 0;   Hilite2:= 0;
ShowTilesNear(X,Y);			\Show result
ShowTilesNear(X1,Y1);
ShowCounter;

UndoCount:= Counter;			\Don't redo beyond this point
end;	\GetMove

\------------------------------------------------------------------------------

proc	SetUp;		\Set up tiles in the initial dragon configuration
int	X, Y, Z;
char	A,
	Tile(144);	\Array of flags to indicate which tiles have been used


	func	GetTile;	\Randomly select a tile (1-144)
	int	N;
	begin
	loop	begin
		N:= Ran(144);
		if Tile(N) then quit;
		end;
	Tile(N):= false;
	return N+1;
	end;	\GetTile


begin	\SetUp
Counter:= 144;				\Initialize
UndoCount:= Counter;

for X:= 0, 144-1 do Tile(X):= true;
for Z:= 0, 5-1 do
    for Y:= 0, 8-1 do
	for X:= 0, 15-1 do
	    Dragon(X,Y,Z):= 0;

for Z:= 0, 3 do				\Set up main tower (center of dragon)
    for Y:= 0+Z, 7-Z do
	for X:= 3+Z, 10-Z do
	    Dragon(X,Y,Z):= GetTile;

	\012345678901234		 Set up peripheral tiles
A:= [	".11........11.. ",	\0
	"............... ",	\1
	"..1........1... ",	\2
	"111........1111 ",	\3
	".11........11.. ",	\4
	"..1........1... ",	\5
	"............... ",	\6
	".11........11.. "];	\7
for Y:= 0, 7 do
    for X:= 0, 14 do
	if A(Y,X) = ^1 then Dragon(X,Y,0):= GetTile;

Dragon(7,3,4):= GetTile;		\Top tile
end;	\SetUp

\------------------------------------------------------------------------------

proc	StartUp;	\Start program with 640x480 graphic image and mouse
int	I, X, Y;
begin
I:= Equip;
if (I(2)&$08) = 0 then
	[Text(0, "

This program requires a VGA display.
");   exit];

Trapc(true);				\Ignore Ctrl+C if struck on keyboard
VidMode:= GetVid;
Setvid($12);				\640x480x16

for I:= 0, 16-1 do Palette(I):= 0;	\Hide images while they are loading
SetPal(Palette);			\Turn colors off
LoadLBM("MJ.LBM");			\Load graphic images

\Save tile images:
TileSize:= ImageSize(1, 1, TileWidth, TileHeight, $0F);
I:= 0;
for Y:= 0, 4-1 do
    for X:= 0, 11-1 do
	begin
	TileBuf(I):= Malloc(TileSize);
	SaveImage(X*48+1, Y*64+1, X*48+TileWidth, Y*64+TileHeight, $0F,
		TileBuf(I));
	I:= I +1;
	if I >= 43 then X:= 11;
	end;

\Use blank tile for mask (beware: dark green images are transparent)
TileMaskBuf:= Malloc(TileSize);
SaveMask(9*48+1, 3*64+1, 9*48+TileWidth, 3*64+TileHeight, $0F, TileMaskBuf);

FaceBuf:= Malloc(TileSize);
SaveImage(10*48+1, 3*64+1, 10*48+TileWidth, 3*64+TileHeight, $0F, FaceBuf);

Clear;
for I:= 0, 16-1 do Palette(I):= I;	\Turn colors on (and align palette)
SetPal(Palette);
Attrib($02);
Setwind(0,0,79,29,$12,false);		\Turn off screen scroll mode

\Set up mouse
if not OpenMouse then
	[Text(0, "This program requires a mouse.

Press Enter to continue...");
	WaitKey;   Exit];		\(Changing video modes erases screen)
MoveMouse(640/2, 480/2);		\Show mouse pointer in center of screen
end;	\StartUp

\==============================================================================

begin	\Main
GameGiven:= [false];			\(Own/static-type variable)
if GameGiven(0) then			\If a game number was given then use it
	begin
	Game:= Game +1;			\Increment games from now on
	if Game >= 1000 then Game:= 0;	\Wrap
	end
else	Game:= Ran(1000);		\Randomly select a game
RanSeed(Game);				\Set seed for repeatable dragon layout

if not Rerun then StartUp;		\"New" button restarts/reruns program
Setup;
Timer:= 0;
Hilite1:= 0;   Hilite2:= 0;
ShowDragon;

Timer0:= GetSec;			\Start the timer
loop	begin
	GetMove;
	if Counter <= 0 then YouWin;	\If all tiles are taken then You Win!
	end;
end;	\Main
