
// Header file GRAPHICS.C

// D E F I N E S /////////////////////////////////////////////////////////////

#define SCREEN_WIDTH      (unsigned int)320 // mode 13h screen dimensions
#define SCREEN_HEIGHT     (unsigned int)200

// screen transition commands

#define SCREEN_DARKNESS  0         // fade to black
#define SCREEN_WHITENESS 1         // fade to white
#define SCREEN_WARP      2         // warp the screen image
#define SCREEN_SWIPE_X   3         // do a horizontal swipe
#define SCREEN_SWIPE_Y   4         // do a vertical swipe
#define SCREEN_DISOLVE   5         // a pixel disolve

#define VGA_INPUT_STATUS_1   0x3DA // VGA input status register number 1
                                   // D3 is vertical retrace bit
                                   // 1 - retrace in progress
                                   // 0 - no retrace

#define VGA_VRETRACE_MASK 0x08     // masks off unwanted bit of status reg

#define SPRITE_WIDTH      16       // default width of a sprite
#define SPRITE_HEIGHT     16       // default height of a sprite

#define MAX_SPRITE_FRAMES 4

// sprite states

#define SPRITE_DEAD       0
#define SPRITE_ALIVE      1
#define SPRITE_DYING      2

// mode Z page stuff

#define PAGE_0            0
#define PAGE_1            1

// these are used to change the visual page of the VGA

#define CRT_ADDR_LOW      0x0D // the index of the low byte of the start address
#define CRT_ADDR_HI       0x0C // the index of the hi byte of the start address


#define GRAPHICS_MODE13     0x13  // 320x200x256
#define TEXT_MODE           0x03  // 80x25 text mode

#define COLOR_MASK        0x3C6 // the bit mask register
#define COLOR_REGISTER_RD 0x3C7 // set read index at this I/O
#define COLOR_REGISTER_WR 0x3C8 // set write index at this I/O
#define COLOR_DATA        0x3C9 // the R/W data is here

// size of the "tech" font used in intro

#define TECH_FONT_WIDTH         4   // width of high tech font
#define TECH_FONT_HEIGHT        7   // height of high tech font
#define NUM_TECH_FONT           64  // number of characters in tech font

// ROM character set stuff

#define ROM_CHAR_SET_SEG 0xF000   // segment of 8x8 ROM character set
#define ROM_CHAR_SET_OFF 0xFA6E   // begining offset of 8x8 ROM character set

#define ROM_CHAR_WIDTH      8     // width of ROM 8x8 character set in pixels
#define ROM_CHAR_HEIGHT     8     // height of ROM 8x8 character set in pixels

#define MODE13_WIDTH   (unsigned int)320 // mode 13h screen dimensions
#define MODE13_HEIGHT  (unsigned int)200

// the VGA card controller ports

#define SEQUENCER             0x3C4    // the sequencer's index port
#define CRT_CONTROLLER        0x3D4    // the crt controller's index port
#define GFX_CONTROLLER        0x3CE    // the graphics controller's index port
#define ATTR_CONTROLLER_FF    0x3DA    // the attribute controller's Flip Flop
#define ATTR_CONTROLLER_DATA  0x3C0    // the attribute controller's data port

// defines for the CRT controller registers of interest

#define CRT_MAX_SCANLINE      0x09     // the maximum scanline register
                                       // used to select how many scanlines
                                       // per row

#define CRT_ADDR_MODE         0x14     // the address mode register
                                       // used to select byte addressing
                                       // for VGA
                                       // also known as the underline register

#define CRT_MODE_CONTROL      0x17     // the mode control register
                                       // used to select single byte addressing


// defines for the GFX controller registers of interest

#define GFX_WRITE_MODE       0x05      // the memory write mode register
                                       // used to deselect even/odd plane
                                       // addressing

#define GFX_MISC             0x06      // the miscellaneous register
                                       // used to deselect the chaining
                                       // of memory


// defines for the SEQUENCER registers of interest

#define SEQ_PLANE_ENABLE     0x02      // plane enable register, used to select
                                       // which planes are written to by a
                                       // CPU write
#define SEQ_MEMORY_MODE      0x04      // the memory mode register
                                       // used to deselect memory chain mode
                                       // and odd/even memory addressing

// M A C R O S ///////////////////////////////////////////////////////////////

#define SET_BITS(x,bits)   (x | bits)
#define RESET_BITS(x,bits) (x & ~bits)
#define SET_SPRITE_SIZE(w,h) {sprite_width=w; sprite_height=h;}

// S T R U C T U R E S ///////////////////////////////////////////////////////

typedef struct grid_typ
{
    int beenShotAt;
    int coveredByShip;
    int * coveredByWhat;
} gridSquare, *gridSquare_ptr;


typedef struct shipSquare_typ
{
    int x;
    int y;
    int hit;
} shipSquare, *shipSquare_ptr;


// this structure holds a RGB triple in three unsigned bytes

typedef struct RGB_color_typ
        {

        unsigned char red;    // red   component of color 0-63
        unsigned char green;  // green component of color 0-63
        unsigned char blue;   // blue  component of color 0-63

        } RGB_color, *RGB_color_ptr;

// this structure holds an entire color palette

typedef struct RGB_palette_typ
        {
        int start_reg;   // index of the starting register that is save
        int end_reg;     // index of the ending registe that is saved

        RGB_color colors[256];   // the storage area for the palette

        } RGB_palette, *RGB_palette_ptr;

// the PCX file structure

typedef struct pcx_header_typ
        {
        char manufacturer;          // the manufacturer of the file
        char version;               // the file format version
        char encoding;              // type of compression
        char bits_per_pixel;        // number of bits per pixel
        int x,y;                    // starting location of image
        int width,height;           // size of image
        int horz_res;               // resolution in DPI (dots per inch)
        int vert_res;
        char ega_palette[48];       // the old EGA palette (usually ignored)
        char reserved;              // don't care
        char num_color_planes;      // number of color planes
        int bytes_per_line;         // number of bytes per line of the image
        int palette_type;           // 1 for color, 2 for grey scale palette
        char padding[58];           // extra bytes

        } pcx_header, *pcx_header_ptr;

// this holds the PCX header and the actual image

typedef struct pcx_picture_typ
        {
        pcx_header header;          // the header of the PCX file
        RGB_color palette[256];     // the palette data
        unsigned char far *buffer;  // a pointer to the 64,000 byte buffer                            // holding the decompressed image
        } pcx_picture, *pcx_picture_ptr;


// this is a sprite structure

typedef struct sprite_typ
        {
        int x,y, x_old, y_old;            // position of sprite
        int width,height;   // dimensions of sprite in pixels

        int counter_1;      // some counters for timing and animation
        int counter_2;
        int counter_3;

        int howManyTimesHit;

        shipSquare coveredSquare1;
        shipSquare coveredSquare2;
        shipSquare coveredSquare3;

/*
        int threshold_1;    // thresholds for the counters (if needed)
        int threshold_2;
        int threshold_3;
*/
        unsigned char far *frames[MAX_SPRITE_FRAMES]; // array of pointers to
                                                       // the images

        int curr_frame;                // current frame being displayed
        int num_frames;                // total number of frames
        int state;                     // state of sprite, alive, dead...
        unsigned char far *background; // image under the sprite

        int x_clip,y_clip;             // clipped position of sprite
        int width_clip,height_clip;    // clipped size of sprite
        int visible;                   // used by sprite engine to flag
                                       // if a sprite was invisible last
                                       // time it was drawn hence the background
                                       // need not be replaced
        int verticalOrHorizantal;
        } sprite, *sprite_ptr;

// this is the typedef for a bitmap

typedef struct bitmap_typ
        {
        int x,y;                    // position of bitmap
        int width,height;           // size of bitmap
        unsigned char far *buffer;  // buffer holding image

        } bitmap, *bitmap_ptr;

// this is a typdef used for the layers in parallax scrolling
// note it is identical to a bitmap, but we'll make a separate typedef
// in the event we later need to add fields to it

typedef struct layer_typ
        {
        int x,y;           // used to hold position information
                           // no specific function

        int width,height;  // size of layer,note:width must be divisible by 2

        unsigned char far *buffer;  // the layer buffer

        } layer, *layer_ptr;



// P R O T O T Y P E S ///////////////////////////////////////////////////////

void Print_Char(int xc,int yc,char c,int color,int transparent);

void Print_String(int x,int y,int color, char *string,int transparent);

void Write_Pixel(int x,int y,int color);

int Read_Pixel(int x,int y);

void Set_Graphics_Mode(int mode);

void Time_Delay(int clicks);

void Line_H(int x1,int x2,int y,int color);

void Line_V(int y1,int y2,int x,int color);

void Write_Color_Reg(int index, RGB_color_ptr color);

void Read_Palette(int start_reg,int end_reg, RGB_palette_ptr the_palette);

void Write_Palette(int start_reg,int end_reg, RGB_palette_ptr the_palette);

RGB_color_ptr Read_Color_Reg(int index, RGB_color_ptr color);

void Draw_Rectangle(int x1,int y1,int x2,int y2,int color);

void Fill_Screen(int color);

void Fill_Screen_Z(int color);

void Write_Pixel_Z(int x,int y,int color);

void Set_Mode_Z(void);

extern void Fill_da_screen(void);

int  PCX_Init(pcx_picture_ptr image);

int PCX_Load(char *filename, pcx_picture_ptr image, int load_palette);

void PCX_Delete(pcx_picture_ptr image);

void PCX_Show_Buffer(pcx_picture_ptr image, int appear_from_black);

void PCX_Copy_To_Buffer(pcx_picture_ptr image,unsigned char far *buffer);

void PCX_Get_Sprite(pcx_picture_ptr image,
                     sprite_ptr sprite,
                     int sprite_frame,
                     int cell_x, int cell_y);

void Sprite_Init(sprite_ptr sprite,int x,int y, int x_old, int y_old, int width, int height,
                                   int c1,int c2,int c3,
                                   int xGrid, int yGrid);

void Sprite_Delete(sprite_ptr sprite);

void Sprite_Under(sprite_ptr sprite, unsigned char far *buffer);

void Sprite_Erase(sprite_ptr sprite, unsigned char far *buffer);

void Sprite_Draw(sprite_ptr sprite, unsigned char far *buffer,int transparent);


void Sprite_Under_Clip(sprite_ptr sprite, unsigned char far *buffer);

void Sprite_Erase_Clip(sprite_ptr sprite, unsigned char far *buffer);

void Sprite_Draw_Clip(sprite_ptr sprite, unsigned char far *buffer,int transparent);

int  Sprite_Collide(sprite_ptr sprite_1, sprite_ptr sprite_2);

void Display_Double_Buffer(unsigned char far *buffer,int y);

int  Create_Double_Buffer(int num_lines);

void Fill_Double_Buffer(int color);

void Delete_Double_Buffer(void);

void Screen_Transition(int effect);

void Wait_For_Vertical_Retrace(void);

void fwordcpy(void far *destination, void far *source,int num_words);

void Bitmap_Get(bitmap_ptr image, unsigned char far *source);

void Bitmap_Put(bitmap_ptr image, unsigned char far *destination,int transparent);

int Bitmap_Allocate(bitmap_ptr image, int width, int height);

void Bitmap_Delete(bitmap_ptr the_bitmap);

int Layer_Create(layer_ptr dest_layer, int width, int height);

void Layer_Build(layer_ptr dest_layer,int dest_x, int dest_y,
                unsigned char far *source_buffer,int source_x,int source_y,
                int width,int height);

void Layer_Draw(layer_ptr source_layer, int source_x, int source_y,
                unsigned char far *dest_buffer,int dest_y,int dest_height,
                int transparent);

void Layer_Delete(layer_ptr the_layer);

void Print_Char_DB(int xc,int yc,char c,int color,int transparent);

void Print_String_DB(int x,int y,int color, char *string,int transparent);

void Write_Pixel_DB(int x,int y,int color);

int Read_Pixel_DB(int x,int y);

void Set_Working_Page_Mode_Z(int page);

void Set_Visual_Page_Mode_Z(int page);

extern void PCX_show_da_buffer(void);

extern void Display_da_double_buffer(void);

extern void Fill_da_double_buffer(void);

extern void copy_da_fword(void);

void Tech_Print(int x,int y,char *string,unsigned char far *destination);

void Font_Engine_1(int x,int y,
                   int font,int color,
                   char *string,unsigned char far *destination);


// E X T E R N A L S /////////////////////////////////////////////////////////////

extern unsigned char far *video_buffer;   // video ram byte ptr
extern unsigned char far *rom_char_set;   // rom characters 8x8
extern unsigned char far *double_buffer;     // the double buffer

extern unsigned int double_buffer_height;    // height of double buffer

extern unsigned int double_buffer_size;      // total size of buffer in bytes

extern unsigned int sprite_width;            // width of a sprite

extern unsigned int sprite_height;           // of a sprite

extern unsigned char far *page_0_buffer;     // pointer to mode z page 0

extern unsigned char far *page_1_buffer;     // pointer to mode z page 0

extern int mode_z_page;                      // current mode z page

extern int raw_key;                  // the global raw keyboard data aquired from the ISR
extern int key_table[4]; // the key state table for the motion keys
extern unsigned int far *clock; // pointer to internal

extern sprite shell;
extern pcx_picture background, shell_back, ships_back;
extern bitmap tech_font[NUM_TECH_FONT];

