// ****************************************************************************
// * ProgressBar.java                                                         *
// *                                                                          *
// * (c) 1997-2002 by Eric Tucker.                                            *
// *                                                                          *
// * This program is free software; you can redistribute it and/or modify it  *
// * under the terms of version 2 of the GNU General Public License (GPL) as  *
// * published by the Free Software Foundation.  This program is distributed  *
// * in the hope that it will be useful but WITHOUT ANY WARRANTY; without     *
// * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR *
// * PURPOSE.  See the GPL for more details.  You should have received a copy *
// * of the GPL along this program.  If not, write to the Free Software       *
// * Foundation, Inc. at 59 Temple Place, Suite 330, Boston, MA  02111  USA   *
// *                                                                          *
// * Please contact Eric Tucker at erictucker2000@yahoo.com with any          *
// * questions, comments, suggestions, or bug reports.                        *
// ****************************************************************************

import java.awt.*;

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------
class ProgressBar
{
  Canvas canvas;
  Rectangle rect;
  boolean horiz;
  PieceColor color;
  double percentFull;

  // --------------------------------------------------------------------------
  // Create a new progress bar.  Horizontal ones fill from left to right, and
  // vertical ones fill from bottom to top.  For the width and height
  // parameters, 0 indicates that they should be left the way they are;
  // anything else will cause the canvas to be forced to that size.
  //
  // All percentages are in [0..1], NOT [0..100]
  // --------------------------------------------------------------------------
  ProgressBar(Canvas canvas, Rectangle rect, boolean horiz,
	      PieceColor color, double percentFull)
  {
    this.canvas = canvas;
    this.rect = rect;
    this.horiz = horiz;
    this.color = color;
    this.percentFull = percentFull;
    Draw();
  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  void SetPercentageComplete(double percentFull)
  {
    Graphics g;

    percentFull = MyUtils.Clamp(percentFull, 0, 1);
    if (percentFull == this.percentFull)
      return;

    g = canvas.getGraphics();
    if (g == null)
      return;

    if (percentFull > this.percentFull) {
      g.setColor(color.r);
      FillRegion(g, 0, percentFull);
    }

    if (percentFull < this.percentFull) {
      g.setColor(Color.black);
      FillRegion(g, percentFull, 1);
    }

    this.percentFull = percentFull;
    g.dispose(); // if we don't do this, drawing may not occur immediately
  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  void Draw() { paint(canvas.getGraphics()); }
  public void paint(Graphics g)
  {
    if (g == null)
      return;
    g.setColor(color.r);
    g.drawRect(rect.x, rect.y, rect.width, rect.height);
    g.setColor(color.r);
    FillRegion(g, 0, percentFull);
  }

  // --------------------------------------------------------------------------
  // Fills the part of the progress bar (in the preset color) between the
  // specified percentages.
  // --------------------------------------------------------------------------
  void FillRegion(Graphics g, double startPercent, double endPercent)
  {
    Rectangle fill;
    Shape clip;

    fill = new Rectangle(rect);
    startPercent = MyUtils.Clamp(startPercent, 0, 1);
    endPercent = MyUtils.Clamp(endPercent, 0, 1);
    ++fill.x; ++fill.y; --fill.width;
    if (horiz) {
      fill.x += (int) (fill.width * startPercent);
      fill.width = (int) (fill.width * (endPercent-startPercent));
    }
    else {
      fill.y += (int) (fill.height * (1-endPercent));
      fill.height = (int) (fill.height * (endPercent-startPercent));
    }

    clip = g.getClip();
    g.setClip(rect.x, rect.y, rect.width, rect.height);
    g.fillRect(fill.x, fill.y, fill.width, fill.height);
    g.setClip(clip);
  }

  // --------------------------------------------------------------------------
  // --------------------------------------------------------------------------
  void SetRect(int x, int y, int width, int height)
  { SetRect(new Rectangle(x, y, width, height)); }

  void SetRect(Rectangle rect)
  {
    Graphics g;

    g = canvas.getGraphics();
    if (g != null) {
      g.setColor(Color.black);
      FillRegion(g, 0, 1);
    }
    this.rect = rect;
    paint(g);
  }
}
